﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Xml;
using nw.g3d.nw4f_3dif;
using Opal.Security.Cryptography;
using Nintendo.ToolFoundation.Collections;
using Nintendo.ToolFoundation.Contracts;
using Nintendo.G3dTool.Entities.Internal;

namespace Nintendo.G3dTool.Entities
{
    public class BufferVar : ObservableEntity<buffer_varType>, IUiElementHolder, IDeepCopyable<BufferVar>, IDeepCopyFrom<BufferVar>
    {
        private readonly VertexSymbol @vertexSymbol = new VertexSymbol();
        private readonly GeometrySymbol @geometrySymbol = new GeometrySymbol();
        private readonly FragmentSymbol @fragmentSymbol = new FragmentSymbol();
        private readonly ComputeSymbol @computeSymbol = new ComputeSymbol();
        private string @id = string.Empty;
        private bool @nostrip;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public BufferVar()
        {
            this.@vertexSymbol.PropertyChanged += this.OnPropertyChanged;
            this.@geometrySymbol.PropertyChanged += this.OnPropertyChanged;
            this.@fragmentSymbol.PropertyChanged += this.OnPropertyChanged;
            this.@computeSymbol.PropertyChanged += this.OnPropertyChanged;
        }

        /// <summary>
        /// コピーコンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public BufferVar(BufferVar source)
            : this()
        {
            this.DeepCopyFrom(source);
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public BufferVar(buffer_varType source)
        {
            if (source.@vertex_symbol != null)
            {
                this.@vertexSymbol = new VertexSymbol(source.@vertex_symbol);
            }
            else
            {
                this.@vertexSymbol = new VertexSymbol();
            }
            if (source.@geometry_symbol != null)
            {
                this.@geometrySymbol = new GeometrySymbol(source.@geometry_symbol);
            }
            else
            {
                this.@geometrySymbol = new GeometrySymbol();
            }
            if (source.@fragment_symbol != null)
            {
                this.@fragmentSymbol = new FragmentSymbol(source.@fragment_symbol);
            }
            else
            {
                this.@fragmentSymbol = new FragmentSymbol();
            }
            if (source.@compute_symbol != null)
            {
                this.@computeSymbol = new ComputeSymbol(source.@compute_symbol);
            }
            else
            {
                this.@computeSymbol = new ComputeSymbol();
            }
            this.@id = source.@id;
            this.@nostrip = source.@nostrip;
            this.@vertexSymbol.PropertyChanged += this.OnPropertyChanged;
            this.@geometrySymbol.PropertyChanged += this.OnPropertyChanged;
            this.@fragmentSymbol.PropertyChanged += this.OnPropertyChanged;
            this.@computeSymbol.PropertyChanged += this.OnPropertyChanged;
        }

        public VertexSymbol VertexSymbol
        {
            get
            {
                return this.@vertexSymbol;
            }
        }

        public GeometrySymbol GeometrySymbol
        {
            get
            {
                return this.@geometrySymbol;
            }
        }

        public FragmentSymbol FragmentSymbol
        {
            get
            {
                return this.@fragmentSymbol;
            }
        }

        public ComputeSymbol ComputeSymbol
        {
            get
            {
                return this.@computeSymbol;
            }
        }

        public string Id
        {
            get
            {
                return this.@id;
            }

            set
            {
                if (this.@id == value)
                {
                    return;
                }

                Ensure.Argument.NotNull(value);
                this.SetProperty(ref this.@id, value, () => this.CalcCRC());
            }
        }

        public bool Nostrip
        {
            get
            {
                return this.@nostrip;
            }

            set
            {
                if (this.@nostrip == value)
                {
                    return;
                }

                this.SetProperty(ref this.@nostrip, value, () => this.CalcCRC());
            }
        }

        /// <summary>
        /// 出力データを作成します。
        /// </summary>
        /// <returns>出力データのインスタンスを返します。</returns>
        public override buffer_varType CreateSerializableData()
        {
            Ensure.Operation.True(
                !string.IsNullOrEmpty(this.vertexSymbol.Name) ||
                !string.IsNullOrEmpty(this.fragmentSymbol.Name) ||
                !string.IsNullOrEmpty(this.geometrySymbol.Name) ||
                !string.IsNullOrEmpty(this.computeSymbol.Name));
            var writeData = new buffer_varType();
            writeData.@vertex_symbol = this.@vertexSymbol.CreateSerializableData() as vertex_symbolType;
            writeData.@geometry_symbol = this.@geometrySymbol.CreateSerializableData() as geometry_symbolType;
            writeData.@fragment_symbol = this.@fragmentSymbol.CreateSerializableData() as fragment_symbolType;
            writeData.@compute_symbol = this.@computeSymbol.CreateSerializableData() as compute_symbolType;
            writeData.@id = this.Id;
            writeData.@nostrip = this.Nostrip;
            return writeData;
        }

        /// <summary>
        /// 現在のインスタンスをディープコピーで複製した新規インスタンスを返します。
        /// </summary>
        BufferVar IDeepCopyable<BufferVar>.DeepCopy()
        {
            return new BufferVar(this);
        }

        /// <summary>
        /// 入力ファイルからディープコピーします。
        /// </summary>
        /// <param name="source">コピー元となる入力ファイルです。</param>
        public void DeepCopyFrom(BufferVar source)
        {
            this.@vertexSymbol.DeepCopyFrom(source.@vertexSymbol);
            this.@geometrySymbol.DeepCopyFrom(source.@geometrySymbol);
            this.@fragmentSymbol.DeepCopyFrom(source.@fragmentSymbol);
            this.@computeSymbol.DeepCopyFrom(source.@computeSymbol);
            this.@id = source.@id;
            this.@nostrip = source.@nostrip;
        }

        /// <summary>
        /// エンティティの CRC を作成します。（内部処理用）
        /// </summary>
        /// <returns>CRC の値を返します。</returns>
        protected override uint CreateCRCInternal()
        {
            CRC32 crc = new CRC32();
            List<byte> buffers = new List<byte>();
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.VertexSymbol.HashValue)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.GeometrySymbol.HashValue)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.FragmentSymbol.HashValue)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.ComputeSymbol.HashValue)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.Id)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.Nostrip)));
            return crc.ComputeHashUInt32(buffers.ToArray());
        }

        /// <summary>
        /// 自動計算フラグを設定します。(内部処理用）
        /// </summary>
        protected override void SetAutoCalcFlagInternal()
        {
            this.@vertexSymbol.AutoCalc = this.AutoCalc;
            this.@geometrySymbol.AutoCalc = this.AutoCalc;
            this.@fragmentSymbol.AutoCalc = this.AutoCalc;
            this.@computeSymbol.AutoCalc = this.AutoCalc;
        }

        /// <summary>
        /// エンティティの状態をリセットします。(内部処理用）
        /// </summary>
        protected override void ResetInternal()
        {
            this.@vertexSymbol.Reset();
            this.@geometrySymbol.Reset();
            this.@fragmentSymbol.Reset();
            this.@computeSymbol.Reset();
        }

        /// <summary>
        /// エンティティの状態を更新します。(内部処理用）
        /// </summary>
        protected override void RefreshInternal()
        {
            this.@vertexSymbol.Refresh();
            this.@geometrySymbol.Refresh();
            this.@fragmentSymbol.Refresh();
            this.@computeSymbol.Refresh();
        }

        private void OnPropertyChanged(object sender, PropertyChangedEventArgs e)
        {
            if (e.PropertyName == "HashValue")
            {
                this.CalcCRC();
            }
        }
    }
}
