﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Xml;
using nw.g3d.nw4f_3dif;
using Opal.Security.Cryptography;
using Nintendo.ToolFoundation.Collections;
using Nintendo.ToolFoundation.Contracts;
using Nintendo.G3dTool.Entities.Internal;

namespace Nintendo.G3dTool.Entities
{
    public class BoneVisBoneAnim : ObservableEntity<bone_vis_bone_animType>, IAnimTarget, IDeepCopyable<BoneVisBoneAnim>, IDeepCopyFrom<BoneVisBoneAnim>, IDisposable
    {
        private string @name = string.Empty;
        private ICurve @curve = null;
        private string @boneName = string.Empty;
        private string @parentName = string.Empty;
        private bool @rigidBody = true;
        private bool @binarizeVisibility = true;
        private bool @compressEnable = true;
        private float @baseValue;
        private int @smoothSkinningMatrixIndex = -1;
        private int @rigidSkinningMatrixIndex = -1;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public BoneVisBoneAnim()
        {
        }

        /// <summary>
        /// コピーコンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public BoneVisBoneAnim(BoneVisBoneAnim source)
            : this()
        {
            this.DeepCopyFrom(source);
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public BoneVisBoneAnim(bone_vis_bone_animType source)
        {
            this.@name = source.@name;
            if (source.step_curve != null)
            {
                this.Curve = DataModelConverter.ConvertCurve(source.step_curve) as ICurve;
            }
            else
            {
                this.Curve = null;
            }
            this.@boneName = source.@bone_name;
            this.@parentName = source.@parent_name;
            this.@rigidBody = source.@rigid_body;
            this.@binarizeVisibility = source.@binarize_visibility;
            this.@compressEnable = source.@compress_enable;
            this.@baseValue = source.@base_value;
            this.@smoothSkinningMatrixIndex = source.matrix_index[0];
            this.@rigidSkinningMatrixIndex = source.matrix_index[1];
        }

        public void Dispose()
        {
            if (this.@curve != null)
            {
                this.@curve.PropertyChanged -= this.OnPropertyChanged;
            }
        }

        public string Name
        {
            get
            {
                return this.@name;
            }

            set
            {
                if (this.@name == value)
                {
                    return;
                }

                Ensure.Argument.NotNull(value);
                this.SetProperty(ref this.@name, value, () => this.CalcCRC());
            }
        }

        public ICurve Curve
        {
            get
            {
                return this.@curve;
            }

            set
            {
                if (this.@curve == value)
                {
                    return;
                }

                Nintendo.ToolFoundation.Contracts.Ensure.Argument.True(value != null ? value.GetType() == typeof(StepCurve) : true);
                if (this.@curve != null)
                {
                    this.@curve.PropertyChanged -= this.OnPropertyChanged;
                }

                if (value != null)
                {
                    value.PropertyChanged += this.OnPropertyChanged;
                }

                this.SetProperty(ref this.@curve, value, () => this.CalcCRC());
            }
        }

        public string BoneName
        {
            get
            {
                return this.@boneName;
            }

            set
            {
                if (this.@boneName == value)
                {
                    return;
                }

                Ensure.Argument.NotNull(value);
                this.SetProperty(ref this.@boneName, value, () => this.CalcCRC());
            }
        }

        public string ParentName
        {
            get
            {
                return this.@parentName;
            }

            set
            {
                if (this.@parentName == value)
                {
                    return;
                }

                Ensure.Argument.NotNull(value);
                this.SetProperty(ref this.@parentName, value, () => this.CalcCRC());
            }
        }

        public bool RigidBody
        {
            get
            {
                return this.@rigidBody;
            }

            set
            {
                if (this.@rigidBody == value)
                {
                    return;
                }

                this.SetProperty(ref this.@rigidBody, value, () => this.CalcCRC());
            }
        }

        public bool BinarizeVisibility
        {
            get
            {
                return this.@binarizeVisibility;
            }

            set
            {
                if (this.@binarizeVisibility == value)
                {
                    return;
                }

                this.SetProperty(ref this.@binarizeVisibility, value, () => this.CalcCRC());
            }
        }

        public bool CompressEnable
        {
            get
            {
                return this.@compressEnable;
            }

            set
            {
                if (this.@compressEnable == value)
                {
                    return;
                }

                this.SetProperty(ref this.@compressEnable, value, () => this.CalcCRC());
            }
        }

        public float BaseValue
        {
            get
            {
                return this.@baseValue;
            }

            set
            {
                if (this.@baseValue == value)
                {
                    return;
                }

                this.SetProperty(ref this.@baseValue, value, () => this.CalcCRC());
            }
        }

        public int SmoothSkinningMatrixIndex
        {
            get
            {
                return this.@smoothSkinningMatrixIndex;
            }

            set
            {
                if (this.@smoothSkinningMatrixIndex == value)
                {
                    return;
                }

                this.SetProperty(ref this.@smoothSkinningMatrixIndex, value, () => this.CalcCRC());
            }
        }

        public int RigidSkinningMatrixIndex
        {
            get
            {
                return this.@rigidSkinningMatrixIndex;
            }

            set
            {
                if (this.@rigidSkinningMatrixIndex == value)
                {
                    return;
                }

                this.SetProperty(ref this.@rigidSkinningMatrixIndex, value, () => this.CalcCRC());
            }
        }

        /// <summary>
        /// 出力データを作成します。
        /// </summary>
        /// <returns>出力データのインスタンスを返します。</returns>
        public override bone_vis_bone_animType CreateSerializableData()
        {
            var writeData = new bone_vis_bone_animType();
            writeData.@name = this.Name;
            if (this.@curve != null)
            {
                writeData.@step_curve = this.Curve.CreateSerializableData() as step_curveType;
            }
            writeData.@bone_name = this.BoneName;
            writeData.@parent_name = this.ParentName;
            writeData.@rigid_body = this.RigidBody;
            writeData.@binarize_visibility = this.BinarizeVisibility;
            writeData.@compress_enable = this.CompressEnable;
            writeData.@base_value = this.BaseValue;
            writeData.matrix_index = new int[2];
            writeData.matrix_index[0] = this.SmoothSkinningMatrixIndex;
            writeData.matrix_index[1] = this.RigidSkinningMatrixIndex;
            return writeData;
        }

        /// <summary>
        /// 現在のインスタンスをディープコピーで複製した新規インスタンスを返します。
        /// </summary>
        BoneVisBoneAnim IDeepCopyable<BoneVisBoneAnim>.DeepCopy()
        {
            return new BoneVisBoneAnim(this);
        }

        /// <summary>
        /// 入力ファイルからディープコピーします。
        /// </summary>
        /// <param name="source">コピー元となる入力ファイルです。</param>
        public void DeepCopyFrom(BoneVisBoneAnim source)
        {
            this.@name = source.@name;
            if (source.@curve == null)
            {
                this.Curve = null;
            }
            else
            {
                this.Curve.DeepCopyFrom(source.@curve);
            }
            this.@boneName = source.@boneName;
            this.@parentName = source.@parentName;
            this.@rigidBody = source.@rigidBody;
            this.@binarizeVisibility = source.@binarizeVisibility;
            this.@compressEnable = source.@compressEnable;
            this.@baseValue = source.@baseValue;
            this.@smoothSkinningMatrixIndex = source.@smoothSkinningMatrixIndex;
            this.@rigidSkinningMatrixIndex = source.@rigidSkinningMatrixIndex;
        }

        /// <summary>
        /// エンティティの CRC を作成します。（内部処理用）
        /// </summary>
        /// <returns>CRC の値を返します。</returns>
        protected override uint CreateCRCInternal()
        {
            CRC32 crc = new CRC32();
            List<byte> buffers = new List<byte>();
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.Name)));
            if (this.Curve != null)
            {
                buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.Curve.HashValue)));
            }
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.BoneName)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.ParentName)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.RigidBody)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.BinarizeVisibility)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.CompressEnable)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.BaseValue)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.SmoothSkinningMatrixIndex)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.RigidSkinningMatrixIndex)));
            return crc.ComputeHashUInt32(buffers.ToArray());
        }

        /// <summary>
        /// 自動計算フラグを設定します。(内部処理用）
        /// </summary>
        protected override void SetAutoCalcFlagInternal()
        {
            if (this.@curve != null)
            {
                this.@curve.AutoCalc = this.AutoCalc;
            }
        }

        /// <summary>
        /// エンティティの状態をリセットします。(内部処理用）
        /// </summary>
        protected override void ResetInternal()
        {
            if (this.@curve != null)
            {
                this.@curve.Reset();
            }
        }

        /// <summary>
        /// エンティティの状態を更新します。(内部処理用）
        /// </summary>
        protected override void RefreshInternal()
        {
            if (this.@curve != null)
            {
                this.@curve.Refresh();
            }
        }

        private void OnPropertyChanged(object sender, PropertyChangedEventArgs e)
        {
            if (e.PropertyName == "HashValue")
            {
                this.CalcCRC();
            }
        }
    }
}
