﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Xml;
using nw.g3d.nw4f_3dif;
using Opal.Security.Cryptography;
using Nintendo.ToolFoundation.Collections;
using Nintendo.ToolFoundation.Contracts;
using Nintendo.G3dTool.Entities.Internal;

namespace Nintendo.G3dTool.Entities
{
    public class Bone : ObservableEntity<boneType>, IChildEntity, IDeepCopyable<Bone>, IDeepCopyFrom<Bone>
    {
        private Float4x3 @invModelMatrix = new Float4x3();
        private readonly ObservableList<UserData> @userDatas = new ObservableList<UserData>();
        private readonly Comment @comment = new Comment();
        private readonly ToolData @toolData = new ToolData();
        private readonly UserToolData @userToolData = new UserToolData();
        private string @name = string.Empty;
        private string @parentName = string.Empty;
        private bool @rigidBody = true;
        private readonly Float3 @scale = new Float3(1, 1, 1);
        private readonly Float4 @rotate = new Float4(0, 0, 0, 1);
        private readonly Float3 @translate = new Float3(0, 0, 0);
        private bool @scaleCompensate = false;
        private bone_billboardType @billboard = bone_billboardType.none;
        private bool @visibility = true;
        private bool @compressEnable = true;
        private bone_sideType @side;
        private string @type = string.Empty;
        private Skeleton parent = null;
        private int @smoothSkinningMatrixIndex = -1;
        private int @rigidSkinningMatrixIndex = -1;
        private Bone @parentBone = null;
        private readonly ObservableList<Bone> @childBones = new ObservableList<Bone>();
        private readonly ReadOnlyObservableCollection<Shape> @shapes = null;
        private readonly ObservableList<Shape> sourceShapes = new ObservableList<Shape>();

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public Bone()
        {
            this.@userDatas.CollectionChanged += OnCollectionChanged<UserData>;
            this.@comment.PropertyChanged += this.OnPropertyChanged;
            this.@toolData.PropertyChanged += this.OnPropertyChanged;
            this.@userToolData.PropertyChanged += this.OnPropertyChanged;
            this.@scale.PropertyChanged += this.OnPropertyChanged;
            this.@rotate.PropertyChanged += this.OnPropertyChanged;
            this.@translate.PropertyChanged += this.OnPropertyChanged;
            this.@childBones.CollectionChanged += OnCollectionChanged<Bone>;
            this.@childBones.CollectionChanged += OnChildBonesCollectionChanged;
            this.sourceShapes.CollectionChanged += OnCollectionChanged<Shape>;
            this.@shapes = new ReadOnlyObservableCollection<Shape>(this.sourceShapes);
        }

        /// <summary>
        /// コピーコンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public Bone(Bone source)
            : this()
        {
            this.DeepCopyFrom(source);
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public Bone(boneType source)
        {
            this.@userDatas.CollectionChanged += OnCollectionChanged<UserData>;
            this.@childBones.CollectionChanged += OnCollectionChanged<Bone>;
            this.@childBones.CollectionChanged += OnChildBonesCollectionChanged;
            if (source.inv_model_matrix != null)
            {
                this.@invModelMatrix = new Float4x3();
                Utility.CopyArray(StringUtility.ConvertTextToValueArray<float>(source.inv_model_matrix), this.@invModelMatrix.Values);
            }
            if (source.@user_data_array != null)
            {
                this.@userDatas.Clear();
                foreach (var elem in source.@user_data_array.Items)
                {
                    this.@userDatas.Add(DataModelConverter.Convert(elem));
                }
            }
            if (source.@comment != null)
            {
                this.@comment = new Comment(source.@comment);
            }
            else
            {
                this.@comment = new Comment();
            }
            if (source.@tool_data != null)
            {
                this.@toolData = new ToolData(source.@tool_data);
            }
            else
            {
                this.@toolData = new ToolData();
            }
            if (source.@user_tool_data != null)
            {
                this.@userToolData = new UserToolData(source.@user_tool_data);
            }
            else
            {
                this.@userToolData = new UserToolData();
            }
            this.@name = source.@name;
            this.@parentName = source.@parent_name;
            this.@rigidBody = source.@rigid_body;
            Utility.CopyArray(source.scale, this.@scale.Values);
            Utility.CopyArray(source.rotate, this.@rotate.Values);
            Utility.CopyArray(source.translate, this.@translate.Values);
            this.@scaleCompensate = source.@scale_compensate;
            this.@billboard = source.@billboard;
            this.@visibility = source.@visibility;
            this.@compressEnable = source.@compress_enable;
            this.@side = source.@side;
            this.@type = source.@type;
            this.@smoothSkinningMatrixIndex = source.matrix_index[0];
            this.@rigidSkinningMatrixIndex = source.matrix_index[1];
            this.@comment.PropertyChanged += this.OnPropertyChanged;
            this.@toolData.PropertyChanged += this.OnPropertyChanged;
            this.@userToolData.PropertyChanged += this.OnPropertyChanged;
            this.@scale.PropertyChanged += this.OnPropertyChanged;
            this.@rotate.PropertyChanged += this.OnPropertyChanged;
            this.@translate.PropertyChanged += this.OnPropertyChanged;
            this.sourceShapes.CollectionChanged += OnCollectionChanged<Shape>;
            this.@shapes = new ReadOnlyObservableCollection<Shape>(this.sourceShapes);
        }

        private Float4x3 InvModelMatrix
        {
            get
            {
                return this.@invModelMatrix;
            }

            set
            {
                if (this.@invModelMatrix == value)
                {
                    return;
                }

                this.SetProperty(ref this.@invModelMatrix, value, () => this.CalcCRC());
            }
        }

        public ObservableList<UserData> UserDatas
        {
            get
            {
                return this.@userDatas;
            }
        }

        public Comment Comment
        {
            get
            {
                return this.@comment;
            }
        }

        public ToolData ToolData
        {
            get
            {
                return this.@toolData;
            }
        }

        public UserToolData UserToolData
        {
            get
            {
                return this.@userToolData;
            }
        }

        public string Name
        {
            get
            {
                return this.@name;
            }

            set
            {
                if (this.@name == value)
                {
                    return;
                }

                Ensure.Argument.NotNull(value);
                this.SetProperty(ref this.@name, value, () => this.CalcCRC());
            }
        }

        internal string ParentName
        {
            get
            {
                return this.@parentName;
            }
        }

        public bool RigidBody
        {
            get
            {
                return this.@rigidBody;
            }

            set
            {
                if (this.@rigidBody == value)
                {
                    return;
                }

                this.SetProperty(ref this.@rigidBody, value, () => this.CalcCRC());
            }
        }

        public Float3 Scale
        {
            get
            {
                return this.@scale;
            }
        }

        public Float4 Rotate
        {
            get
            {
                return this.@rotate;
            }
        }

        public Float3 Translate
        {
            get
            {
                return this.@translate;
            }
        }

        public bool ScaleCompensate
        {
            get
            {
                return this.@scaleCompensate;
            }

            set
            {
                if (this.@scaleCompensate == value)
                {
                    return;
                }

                this.SetProperty(ref this.@scaleCompensate, value, () => this.CalcCRC());
            }
        }

        public bone_billboardType Billboard
        {
            get
            {
                return this.@billboard;
            }

            set
            {
                if (this.@billboard == value)
                {
                    return;
                }

                this.SetProperty(ref this.@billboard, value, () => this.CalcCRC());
            }
        }

        public bool Visibility
        {
            get
            {
                return this.@visibility;
            }

            set
            {
                if (this.@visibility == value)
                {
                    return;
                }

                this.SetProperty(ref this.@visibility, value, () => this.CalcCRC());
            }
        }

        public bool CompressEnable
        {
            get
            {
                return this.@compressEnable;
            }

            set
            {
                if (this.@compressEnable == value)
                {
                    return;
                }

                this.SetProperty(ref this.@compressEnable, value, () => this.CalcCRC());
            }
        }

        public bone_sideType Side
        {
            get
            {
                return this.@side;
            }

            set
            {
                if (this.@side == value)
                {
                    return;
                }

                this.SetProperty(ref this.@side, value, () => this.CalcCRC());
            }
        }

        public string Type
        {
            get
            {
                return this.@type;
            }

            set
            {
                if (this.@type == value)
                {
                    return;
                }

                Ensure.Argument.NotNull(value);
                this.SetProperty(ref this.@type, value, () => this.CalcCRC());
            }
        }

        IEntity IChildEntity.Parent
        {
            get
            {
                return this.parent as IEntity;
            }

            set
            {
                if (this.parent == value)
                {
                    return;
                }

                Ensure.Argument.True(value != null ? value is Skeleton : true);
                this.SetProperty(ref this.parent, value as Skeleton, () => this.CalcCRC());

                // 所属する親が変化したのでシェイプの関連付けし直し
                this.UpdateShapeReferences();

            }
        }

        public Skeleton Parent
        {
            get
            {
                // ルートボーンまで辿る
                Bone bone = this;
                List<Bone> circularRefCheckList = new List<Bone>();
                while (bone.ParentBone != null)
                {
                    circularRefCheckList.Add(bone);
                    bone = bone.ParentBone;
                    Ensure.Operation.False(circularRefCheckList.Contains(bone), $"Bone circular reference was detected. {bone.Name}");
                }

                if (bone.parent != null)
                {
                    return bone.parent;
                }
                else
                {
                    // スケルトンに紐づいていない
                    return null;
                }
            }
        }

        Type IChildEntity.ParentType
        {
            get
            {
                return typeof(Skeleton);
            }
        }

        public int SmoothSkinningMatrixIndex
        {
            get
            {
                return this.@smoothSkinningMatrixIndex;
            }

            set
            {
                if (this.@smoothSkinningMatrixIndex == value)
                {
                    return;
                }

                this.SetProperty(ref this.@smoothSkinningMatrixIndex, value, () => this.CalcCRC());
                this.RaisePropertyChanged("IsUsedForSkinning");
            }
        }

        public int RigidSkinningMatrixIndex
        {
            get
            {
                return this.@rigidSkinningMatrixIndex;
            }

            set
            {
                if (this.@rigidSkinningMatrixIndex == value)
                {
                    return;
                }

                this.SetProperty(ref this.@rigidSkinningMatrixIndex, value, () => this.CalcCRC());
                this.RaisePropertyChanged("IsUsedForSkinning");
            }
        }

        public Bone ParentBone
        {
            get
            {
                return this.@parentBone;
            }
        }

        /// <summary>
        /// このボーンの子階層のボーンです。
        /// </summary>
        public ObservableList<Bone> ChildBones
        {
            get
            {
                return this.@childBones;
            }
        }

        /// <summary>
        /// このボーンを描画やビジビリティーアニメーションに利用するシェイプです。
        /// </summary>
        public ReadOnlyObservableCollection<Shape> Shapes
        {
            get
            {
                return this.@shapes;
            }
        }

        public bool IsUsedForSkinning
        {
            get
            {
                return (this.SmoothSkinningMatrixIndex >= 0) || (this.RigidSkinningMatrixIndex >= 0);
            }
        }

        public Float4x4 CalculateTransformMatrix()
        {
            Ensure.Operation.NotNull(this.Parent, $"Parent skeleton must not be null. {this.Name}");
            switch (this.Parent.SkeletonInfo.ScaleMode)
            {
                case skeleton_info_scale_modeType.standard:
                    return this.CalculateTransformStandard();
                case skeleton_info_scale_modeType.maya:
                    return this.CalculateTransformMaya();
                case skeleton_info_scale_modeType.softimage:
                    throw new NotImplementedException();
                default:
                    throw new Exception("<skeleton_info> has unexpected value of 'scale_mode'.");
            }
        }

        internal void UpdateShapeReferences()
        {
            if (this.Parent == null)
            {
                // ボーンがモデル構造から外れるのでシェイプ参照をクリアする
                List<Shape> oldShapes = new List<Shape>(this.Shapes);
                this.SourceShapes.Clear();
                foreach (var shape in oldShapes)
                {
                    shape.ShapeInfo.SetBoneWithoutPrePostProcess(null);
                }
            }
            else
            {
                this.sourceShapes.Clear();
                var model = this.Parent.Parent;
                if (model != null)
                {
                    foreach (var shape in model.Shapes.Where(x => x.ShapeInfo.BoneName == this.Name))
                    {
                        shape.ShapeInfo.SetBoneWithoutPrePostProcess(this);
                        this.SourceShapes.Add(shape);
                    }
                }
            }
        }

        internal ObservableList<Shape> SourceShapes
        {
            get
            {
                return this.sourceShapes;
            }
        }

        private void OnChildBonesCollectionChanged(object sender, System.Collections.Specialized.NotifyCollectionChangedEventArgs e)
        {
            if (e.NewItems != null)
            {
                foreach (var item in e.NewItems)
                {
                    var bone = item as Bone;
                    bone.parentBone = this;
                    (bone as IChildEntity).Parent = null;
                }
            }

            if (e.OldItems != null)
            {
                foreach (var item in e.OldItems)
                {
                    var bone = item as Bone;
                    bone.parentBone = null;
                    (bone as IChildEntity).Parent = null;
                }
            }
        }


        /// <summary>
        /// 出力データを作成します。
        /// </summary>
        /// <returns>出力データのインスタンスを返します。</returns>
        public override boneType CreateSerializableData()
        {
            if (this.IsUsedForSkinning)
            {
                var invTransformMat = this.CalculateInverseTransformMatrix();
                Ensure.Operation.NotNull(invTransformMat, $"Can not calculate inverse transform matrix of bone {this.Name}");
                this.InvModelMatrix.DeepCopyFrom(this.CalculateInverseTransformMatrix());
            }
            var writeData = new boneType();
            if (this.IsUsedForSkinning)
            {
                writeData.inv_model_matrix = this.@invModelMatrix.ToString();
            }
            if (this.@UserDatas.Count > 0)
            {
                writeData.@user_data_array = new user_data_arrayType();
                writeData.@user_data_array.Items = this.@userDatas.Select(x => x.CreateSerializableData()).Where(x => x != null).ToArray();
                if (writeData.@user_data_array.Items.Length == 0)
                {
                    writeData.@user_data_array = null;
                }
            }
            writeData.@comment = this.@comment.CreateSerializableData() as commentType;
            writeData.@tool_data = this.@toolData.CreateSerializableData() as tool_dataType;
            writeData.@user_tool_data = this.@userToolData.CreateSerializableData() as user_tool_dataType;
            writeData.@name = this.Name;
            {
                if (this.ParentBone != null)
                {
                    writeData.parent_name = this.ParentBone.Name;
                }
                else
                {
                    writeData.parent_name = string.Empty;
                }
            }
            writeData.@rigid_body = this.RigidBody;
            writeData.scale = new float[3];
            Utility.CopyArray(this.Scale.Values, writeData.scale);
            writeData.rotate = new float[4];
            Utility.CopyArray(this.Rotate.Values, writeData.rotate);
            writeData.translate = new float[3];
            Utility.CopyArray(this.Translate.Values, writeData.translate);
            writeData.@scale_compensate = this.ScaleCompensate;
            writeData.@billboard = this.Billboard;
            writeData.@visibility = this.Visibility;
            writeData.@compress_enable = this.CompressEnable;
            writeData.@side = this.Side;
            writeData.@type = this.Type;
            writeData.matrix_index = new int[2];
            writeData.matrix_index[0] = this.SmoothSkinningMatrixIndex;
            writeData.matrix_index[1] = this.RigidSkinningMatrixIndex;
            return writeData;
        }

        /// <summary>
        /// 現在のインスタンスをディープコピーで複製した新規インスタンスを返します。
        /// </summary>
        Bone IDeepCopyable<Bone>.DeepCopy()
        {
            return new Bone(this);
        }

        /// <summary>
        /// 入力ファイルからディープコピーします。
        /// </summary>
        /// <param name="source">コピー元となる入力ファイルです。</param>
        public void DeepCopyFrom(Bone source)
        {
            this.@invModelMatrix = source.@invModelMatrix;
            CopyElements(source.@userDatas, this.@userDatas);
            this.@comment.DeepCopyFrom(source.@comment);
            this.@toolData.DeepCopyFrom(source.@toolData);
            this.@userToolData.DeepCopyFrom(source.@userToolData);
            this.@name = source.@name;
            this.@parentName = source.@parentName;
            this.@rigidBody = source.@rigidBody;
            this.@scale.DeepCopyFrom(source.@scale);
            this.@rotate.DeepCopyFrom(source.@rotate);
            this.@translate.DeepCopyFrom(source.@translate);
            this.@scaleCompensate = source.@scaleCompensate;
            this.@billboard = source.@billboard;
            this.@visibility = source.@visibility;
            this.@compressEnable = source.@compressEnable;
            this.@side = source.@side;
            this.@type = source.@type;
            this.@smoothSkinningMatrixIndex = source.@smoothSkinningMatrixIndex;
            this.@rigidSkinningMatrixIndex = source.@rigidSkinningMatrixIndex;
            CopyElements(source.@childBones, this.@childBones);
        }

        /// <summary>
        /// エンティティの CRC を作成します。（内部処理用）
        /// </summary>
        /// <returns>CRC の値を返します。</returns>
        protected override uint CreateCRCInternal()
        {
            CRC32 crc = new CRC32();
            List<byte> buffers = new List<byte>();
            buffers.AddRange(BitConverter.GetBytes(this.@invModelMatrix.GetHashCode()));
            foreach (var elem in this.UserDatas)
            {
                buffers.AddRange(BitConverter.GetBytes(elem.HashValue));
            }
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.Comment.HashValue)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.ToolData.HashValue)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.UserToolData.HashValue)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.Name)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.ParentName)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.RigidBody)));
            buffers.AddRange(BitConverter.GetBytes(this.Scale.HashValue));
            buffers.AddRange(BitConverter.GetBytes(this.Rotate.HashValue));
            buffers.AddRange(BitConverter.GetBytes(this.Translate.HashValue));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.ScaleCompensate)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.Billboard)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.Visibility)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.CompressEnable)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.Side)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.Type)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.SmoothSkinningMatrixIndex)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.RigidSkinningMatrixIndex)));
            foreach (var elem in this.ChildBones)
            {
                buffers.AddRange(BitConverter.GetBytes(elem.HashValue));
            }
            return crc.ComputeHashUInt32(buffers.ToArray());
        }

        /// <summary>
        /// 自動計算フラグを設定します。(内部処理用）
        /// </summary>
        protected override void SetAutoCalcFlagInternal()
        {
            foreach (var elem in this.@userDatas)
            {
                elem.AutoCalc = this.AutoCalc;
            }
            this.@comment.AutoCalc = this.AutoCalc;
            this.@toolData.AutoCalc = this.AutoCalc;
            this.@userToolData.AutoCalc = this.AutoCalc;
            this.@scale.AutoCalc = this.AutoCalc;
            this.@rotate.AutoCalc = this.AutoCalc;
            this.@translate.AutoCalc = this.AutoCalc;
            foreach (var elem in this.@childBones)
            {
                elem.AutoCalc = this.AutoCalc;
            }
        }

        /// <summary>
        /// エンティティの状態をリセットします。(内部処理用）
        /// </summary>
        protected override void ResetInternal()
        {
            foreach (var elem in this.@userDatas)
            {
                elem.Reset();
            }
            this.@comment.Reset();
            this.@toolData.Reset();
            this.@userToolData.Reset();
            this.@scale.Reset();
            this.@rotate.Reset();
            this.@translate.Reset();
            foreach (var elem in this.@childBones)
            {
                elem.Reset();
            }
        }

        /// <summary>
        /// エンティティの状態を更新します。(内部処理用）
        /// </summary>
        protected override void RefreshInternal()
        {
            foreach (var elem in this.@userDatas)
            {
                elem.Refresh();
            }
            this.@comment.Refresh();
            this.@toolData.Refresh();
            this.@userToolData.Refresh();
            this.@scale.Refresh();
            this.@rotate.Refresh();
            this.@translate.Refresh();
            foreach (var elem in this.@childBones)
            {
                elem.Refresh();
            }
        }

        private void OnPropertyChanged(object sender, PropertyChangedEventArgs e)
        {
            if (e.PropertyName == "HashValue")
            {
                this.CalcCRC();
            }
        }
    }
}
