﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Drawing.Imaging;
using System.Linq;
using System.IO;
using System.Runtime.Remoting.Messaging;
using System.Windows.Forms;
using System.Diagnostics;
using App.Controls;
using App.Data;
using App.Utility;
using App.res;
using nw.g3d.nw4f_3dif;

namespace App
{
    public partial class MainForm : Form
    {
        public static MainForm This{ get; set; }

        private Texture                        texture_            = null;
        private System.IO.FileSystemWatcher    watcher_            = new System.IO.FileSystemWatcher();

        private string                        lastLoadedFileName_        = null;    // 最後に読み込まれたファイル名
        private string                        requestLoadFileName_    = null;    // 読み込み要求があるファイル名

        private int    loadingCount_ = 0;
        private volatile bool isLoading_ = false;

        private float Magnify
        {
            get
            {
                switch(tsiMagnify.SelectedIndex)
                {
                    case 0:    return 16.0f;
                    case 1:    return 8.0f;
                    case 2:    return 4.0f;
                    case 3:    return 2.0f;
                    case 4:    return 1.5f;
                    case 5:    return 1.25f;
                    case 6:    return 1.0f;
                    case 7:    return 0.5f;
                    case 8:    return 0.25f;
                    case 9:    return 0.125f;
                }

                Debug.Assert(false);
                return 1.0f;
            }
        }

        private int TextureWidth  { get {    return texture_.Width;                                                        } }
        private int TextureHeight { get {    return tsiShowMipmap.Checked ? texture_.HeightWithMipmap : texture_.Height;    } }

        private PointF beforeCenterPoint_;
        private Size beforeSize_;
        private Size beforeClientSize_;
        private Point beforeLocation_;

        private const int minFormWidth    = 636;
        private const int minFormHeight    = 400;
        private const int frameWidth    = 1;            // 枠線の幅

        public MainForm(string[] args)
        {
            This = this;

            InitializeComponent();
            InitializeToolbar();

//            pnlBaseView.MouseEnter += (s, e) => pnlBaseView.Focus();
            pnlBaseView.Click += (s, e) => pnlBaseView.Focus();
            pnlView.Click     += (s, e) => pnlBaseView.Focus();

            if (args.Any())
            {
                LoadFromFile(args[0]);
            }
        }

        private void InitializeToolbar()
        {
            tsiMagnify.SelectedIndex = 6;

            tsiChannelRGB.Checked = true;
            tsiChannelR.Checked = true;
            tsiChannelG.Checked = true;
            tsiChannelB.Checked = true;

            UpdateToolbar();
        }

        private void UpdateToolbar()
        {
            // 倍率の虫眼鏡
            {
                tsiMagnifyMinus.Enabled = tsiMagnify.SelectedIndex < tsiMagnify.Items.Count - 1;
                tsiMagnifyPlus.Enabled  = tsiMagnify.SelectedIndex > 0;
            }

            // チャンネル・ミップマップ
            {
                tsiChannelRGB.Enabled    = (texture_ != null) && texture_.HasColor;
                tsiChannelR.Enabled      = (texture_ != null) && texture_.HasColor;
                tsiChannelG.Enabled      = (texture_ != null) && texture_.HasColor;
                tsiChannelB.Enabled      = (texture_ != null) && texture_.HasColor;
                tsiChannelA.Enabled      = (texture_ != null) && texture_.HasAlpha;
                tsiShowMipmap.Enabled    = (texture_ != null) && texture_.HasMipmap;
            }
        }

        private void SetViewPosition()
        {
            // 新しいテクスチャサイズ取得
            var size = (texture_ != null) && (texture_.Data!=null) ? new Size((int)(TextureWidth * Magnify) + frameWidth * 2, (int)(TextureHeight * Magnify) + frameWidth * 2) : new Size(0, 0);

            // すでにViewPosition設定済みなら戻る
            if (size == beforeSize_ && beforeClientSize_ == pnlBaseView.Size)
            {
                return;
            }

            var location = new Point();
            var scrollValue = new Point();
            var centerPoint = new PointF();
            if (pnlBaseView.ClientSize.Width <= size.Width)
            {
                // 表示領域に収まらない場合はscroll量を求める
                var sizeMagnifi = (float)size.Width / (float)beforeSize_.Width;
                centerPoint.X = (beforeCenterPoint_.X + (beforeLocation_.X - pnlView.Location.X)) * sizeMagnifi;

                scrollValue.X = (int)(((float)pnlBaseView.ClientSize.Width / 2.0) - centerPoint.X);
            }
            else
            {
                // 表示領域に収まる場合はlocation量を求める
                location.X = Math.Max(0, (pnlBaseView.ClientSize.Width - size.Width) / 2);

                centerPoint.X = (float)size.Width / 2.0f;
            }
            if (pnlBaseView.ClientSize.Height <= size.Height)
            {
                // 表示領域に収まらない場合はscroll量を求める
                var sizeMagnifi = (float)size.Height / (float)beforeSize_.Height;
                centerPoint.Y = (beforeCenterPoint_.Y + (beforeLocation_.Y - pnlView.Location.Y)) * sizeMagnifi;

                scrollValue.Y = (int)(((float)pnlBaseView.ClientSize.Height / 2.0) - centerPoint.Y);
            }
            else
            {
                // 表示領域に収まる場合はlocation量を求める
                location.Y = Math.Max(0, (pnlBaseView.ClientSize.Height - size.Height) / 2);

                centerPoint.Y = (float)size.Height / 2.0f;
            }

            // DisplayRectangleより大きな値はscrollできない、色々なタイミングで同じイベントが呼ばれるようなので、DisplayRectangleが新しい値に更新された後のイベントまで待つ
            if ((scrollValue.X >= 0 || size.Width <= pnlBaseView.DisplayRectangle.Width) && (scrollValue.Y >= 0 || size.Height <= pnlBaseView.DisplayRectangle.Height))
            {
                pnlBaseView.AutoScrollPosition = new Point(-scrollValue.X, -scrollValue.Y);
                beforeSize_ = size;
                beforeClientSize_ = pnlBaseView.Size;
                beforeCenterPoint_ = centerPoint;

                pnlView.Location = new Point(scrollValue.X >= 0 ? location.X : location.X + scrollValue.X,
                    scrollValue.Y >= 0 ? location.Y : location.Y + scrollValue.Y);

                beforeLocation_ = pnlView.Location;
            }
        }
        private void SetViewSize()
        {
            pnlView.Size = texture_ != null ? new Size((int)(TextureWidth  * Magnify) + frameWidth*2, (int)(TextureHeight * Magnify) + frameWidth*2) : new Size(0, 0);
        }

        public void UpdateLoadEnable()
        {
            bool isEnabled = ! isLoading_;

            MenuStrip.AllowDrop      = isEnabled;
            label1.AllowDrop         = isEnabled;
            label2.AllowDrop         = isEnabled;
            pnlInfo.AllowDrop        = isEnabled;
            label3.AllowDrop         = isEnabled;
            pnlBaseView.AllowDrop    = isEnabled;
            pnlView.AllowDrop        = isEnabled;
            tspMain.AllowDrop        = isEnabled;
            tsmOpen.Enabled          = isEnabled;
            tsmFileQuit.Enabled      = isEnabled;
            tsiShowMipmap.Enabled    = isEnabled;
            tsiMagnify.Enabled       = isEnabled;
            tsiMagnifyMinus.Enabled  = isEnabled;
            tsiMagnifyPlus.Enabled   = isEnabled;
            tsiChannelRGB.Enabled    = isEnabled;
            tsiChannelR.Enabled      = isEnabled;
            tsiChannelG.Enabled      = isEnabled;
            tsiChannelB.Enabled      = isEnabled;
            tsiChannelA.Enabled      = isEnabled;
        }

        public void LoadFromFile(string fileName)
        {
            Debug.Assert(fileName != null);

            if (string.IsNullOrEmpty(fileName))
                return;

            string ext = Path.GetExtension(fileName).ToLower();
            if ((ext != ".ftxa") &&
                (ext != ".ftxb"))
            {
                MessageBox.Show(string.Format(Strings.IO_NotSupportExtension, fileName));
                return;
            }

            if (isLoading_)
            {
//                MessageBox.Show(Strings.NowLoading);
                requestLoadFileName_ = fileName;
                return;
            }

            // 読み込み開始
            isLoading_ = true;
            UpdateLoadEnable();

            loadingCount_ = 0;
            texture_ = null;
            UpdateInfo();
            SetViewSize();
            Timer.Enabled = true;

            // 画面を更新する
            pnlBaseView.Invalidate();
            pnlView.Invalidate();

            BackgroundTaskManager.AddTask(
                () =>{
                    int formWidth = Width;
                    int formHeigth = Height;

                    texture_ = new Texture();
                    bool bOk = texture_.LoadFromFile(fileName);
                    if (!bOk)
                        texture_ = null;

                    // 情報だけ先に出す
                    BeginInvoke(new MethodInvoker(UpdateInfo));

                    // イメージを作る
                    if (bOk)
                        texture_.CreateImages();

                    // ひと通り更新する
                    BeginInvoke(
                        new MethodInvoker(
                            () =>
                            {
                                if (bOk)
                                {
                                    formWidth = (int) (TextureWidth*Magnify) + Width - pnlBaseView.Width;
                                    formHeigth = (int) (TextureHeight*Magnify) + Height - pnlBaseView.Height;
                                }

                                beforeCenterPoint_ = new Point(4, 4);
                                beforeSize_ = new Size(8, 8);

                                Size = new Size(Math.Max(formWidth  + frameWidth*2, minFormWidth), Math.Max(formHeigth + frameWidth*2, minFormHeight));

                                SetViewSize();

                                isLoading_ = false;
                                UpdateLoadEnable();

                                UpdateToolbar();
                                pnlBaseView.Invalidate();
                                pnlView.Invalidate();

                                Timer.Enabled = false;
                                lastLoadedFileName_ = fileName;

                                // 監視する
                                {
                                    String dirName = Path.GetDirectoryName(fileName);
                                    if (string.IsNullOrEmpty(dirName))
                                        dirName = Directory.GetCurrentDirectory();

                                    watcher_.Dispose();
                                    watcher_ = new FileSystemWatcher
                                        {
                                        Path = dirName,
                                        SynchronizingObject = this,
                                        NotifyFilter = NotifyFilters.LastWrite
                                    };

                                    watcher_.Changed += (s, e) =>
                                    {
                                        if (isLoading_)
                                            return;

                                        // 複数回変更イベントが通知されるのでなるべくひとつにまとめる
                                        if (_timerChangeLoadedFile == null)
                                        {
                                            _timerChangeLoadedFile = new Timer(components);
                                            _timerChangeLoadedFile.Tick += ChangeLoadedFile;
                                            _timerChangeLoadedFile.Interval = 50;        // 注意：この数値に根拠はないです。
                                            _timerChangeLoadedFile.Enabled = true;
                                        }
                                    };

                                    watcher_.EnableRaisingEvents = true;
                                }

                                // 読み込み要求があれば読み込む
                                if (requestLoadFileName_ != null)
                                {
                                    LoadFromFile(requestLoadFileName_);
                                    requestLoadFileName_ = null;
                                }
                            }
                        )
                    );
                }
            );
        }

        private Timer _timerChangeLoadedFile;

        private void ChangeLoadedFile(object sender, EventArgs e)
        {
            if (lastLoadedFileName_ != null)
            {
                LoadFromFile(lastLoadedFileName_);
            }

            if (_timerChangeLoadedFile != null)
            {
                _timerChangeLoadedFile.Dispose();
                _timerChangeLoadedFile = null;
            }
        }

        private void UpdateInfo()
        {
            if ((texture_ == null) || (texture_.Data==null))
            {
                ltbDimension.Text        = string.Empty;
                ltbFormat.Text            = string.Empty;
                ltbSize.Text            = string.Empty;
                ltbDepth.Text            = string.Empty;
                ltbVramSize.Text        = string.Empty;
                ltbMipmap.Text            = string.Empty;
                ltbMipmapMinSize.Text    = string.Empty;
                ltbOriginalImageHash.Text = string.Empty;

                Text                    = VersionInformation.Name;
            }
            else
            {
                ltbDimension.Text        = UIText.EnumValue(texture_.Data.texture_info.dimension);
                ltbFormat.Text            = UIText.EnumValue(texture_.Data.texture_info.quantize_type);
                ltbSize.Text            = string.Format("{0} x {1}", texture_.Data.texture_info.width, texture_.Data.texture_info.height);
                ltbDepth.Text            = texture_.Data.texture_info.depth.ToString();
                ltbVramSize.Text        = DataSize.XBytesText(texture_.Data.texture_info.size);
                ltbMipmap.Text            = UIText.MipLevel(texture_.Data.texture_info.mip_level);
                ltbMipmapMinSize.Text    = UIText.MipmapMinSize(texture_.Data.texture_info);
                ltbOriginalImageHash.Text = texture_.Data.texture_info.original_image_hash;

                Text                    = string.Format("{0} - {1}", texture_.FileName, VersionInformation.Name);
            }
        }

        private void tsmFileQuit_Click(object sender, EventArgs e)
        {
            Close();
        }

        private void tsmHelpAbout_Click(object sender, EventArgs e)
        {
            using (var dialog = new AboutForm())
            {
                dialog.ShowDialog();
            }
        }

        private void MainForm_KeyDown(object sender, KeyEventArgs e)
        {
            if (e.KeyCode == Keys.Escape)
            {
                Close();
            }
            else
            {
                if (isLoading_)
                    return;

                if (e.Control)
                {
                    if (e.KeyCode == Keys.Add)
                    {
                        tsiMagnifyPlus_Click(sender, null);
                    }
                    else if (e.KeyCode == Keys.Subtract)
                    {
                        tsiMagnifyMinus_Click(sender, null);
                    }
                    // ＲＧＢチャンネル
                    else if (e.KeyCode == Keys.D1)
                    {
                        if (tsiChannelRGB.Enabled)
                        {
                            tsiChannelRGB_Click(sender, null);
                        }
                    }
                    // Ｒチャンネル
                    else if (e.KeyCode == Keys.D2)
                    {
                        if (tsiChannelR.Enabled)
                        {
                            tsiChannelR_Click(sender, null);
                        }
                    }
                    // Ｇチャンネル
                    else if (e.KeyCode == Keys.D3)
                    {
                        if (tsiChannelG.Enabled)
                        {
                            tsiChannelG_Click(sender, null);
                        }
                    }
                    // Ｂチャンネル
                    else if (e.KeyCode == Keys.D4)
                    {
                        if (tsiChannelB.Enabled)
                        {
                            tsiChannelB_Click(sender, null);
                        }
                    }
                    // Ａチャンネル
                    else if (e.KeyCode == Keys.D5)
                    {
                        if (tsiChannelA.Enabled)
                        {
                            tsiChannelA_Click(sender, null);
                        }
                    }
                    // ミップマップ表示
                    else if (e.KeyCode == Keys.M)
                    {
                        if (tsiShowMipmap.Enabled)
                        {
                            tsiShowMipmap_Click(sender, null);
                        }
                    }
                }
            }
        }

        private void tsmOpen_Click(object sender, EventArgs e)
        {
            if (OpenFileDialog.ShowDialog() == DialogResult.OK)
            {
                LoadFromFile(OpenFileDialog.FileName);
            }
        }

        private void pnlView_Paint(object sender, PaintEventArgs e)
        {
            e.Graphics.Clear(Color.Silver);

            if (texture_ != null)
            {
                DrawImage(e);
            }
        }

        private void Timer_Tick(object sender, EventArgs e)
        {
            loadingCount_ = (loadingCount_ + 1) & 3;
            pnlBaseView.Invalidate();
        }

        private void pnlView_DragEnter(object sender, DragEventArgs e)
        {
            e.Effect =
                (e.Data.GetDataPresent(DataFormats.FileDrop)) ?
                    DragDropEffects.Copy :
                    DragDropEffects.None;
        }

        private void pnlView_DragDrop(object sender, DragEventArgs e)
        {
            if (e.Data != null)
            {
                var dataObj = e.Data;
                var fileNames = dataObj.GetData(DataFormats.FileDrop, false) as string[];

                Debug.Assert(fileNames != null, "fileNames != null");
                foreach(var fileName in fileNames)
                {
                    string ext = Path.GetExtension(fileName).ToLower();
                    if ((ext == G3dPath.TextureTextExtension) || (ext == G3dPath.TextureBinaryExtension))
                    {
                        LoadFromFile(fileName);
                        break;
                    }
                }
            }
        }

        private void tsiMagnify_SelectedIndexChanged(object sender, EventArgs e)
        {
            UpdateToolbar();

            SetViewSize();

            pnlView.Invalidate();
            pnlBaseView.Invalidate();
        }

        private void tsiMagnifyMinus_Click(object sender, EventArgs e)
        {
            if (tsiMagnify.SelectedIndex < tsiMagnify.Items.Count - 1)
            {
                ++ tsiMagnify.SelectedIndex;
            }
        }

        private void tsiMagnifyPlus_Click(object sender, EventArgs e)
        {
            if (tsiMagnify.SelectedIndex > 0)
            {
                -- tsiMagnify.SelectedIndex;
            }
        }

        private void tsiChannelRGB_Click(object sender, EventArgs e)
        {
            using (var block = new UIControlEventSuppressBlock())
            {
                tsiChannelRGB.Checked = ! tsiChannelRGB.Checked;

                tsiChannelR.Checked    = tsiChannelRGB.Checked;
                tsiChannelG.Checked    = tsiChannelRGB.Checked;
                tsiChannelB.Checked    = tsiChannelRGB.Checked;
            }

            pnlBaseView.Invalidate();
            pnlView.Invalidate();
        }

        private void tsiChannelR_Click(object sender, EventArgs e)
        {
            using (var block = new UIControlEventSuppressBlock())
            {
                tsiChannelR.Checked   = ! tsiChannelR.Checked;
                tsiChannelRGB.Checked = tsiChannelR.Checked && tsiChannelG.Checked && tsiChannelB.Checked;
            }

            pnlBaseView.Invalidate();
            pnlView.Invalidate();
        }

        private void tsiChannelG_Click(object sender, EventArgs e)
        {
            using (var block = new UIControlEventSuppressBlock())
            {
                tsiChannelG.Checked   = ! tsiChannelG.Checked;
                tsiChannelRGB.Checked = tsiChannelR.Checked && tsiChannelG.Checked && tsiChannelB.Checked;
            }

            pnlBaseView.Invalidate();
            pnlView.Invalidate();
        }

        private void tsiChannelB_Click(object sender, EventArgs e)
        {
            using (var block = new UIControlEventSuppressBlock())
            {
                tsiChannelB.Checked   = ! tsiChannelB.Checked;
                tsiChannelRGB.Checked = tsiChannelR.Checked && tsiChannelG.Checked && tsiChannelB.Checked;
            }

            pnlBaseView.Invalidate();
            pnlView.Invalidate();
        }

        private void tsiChannelA_Click(object sender, EventArgs e)
        {
            using (var block = new UIControlEventSuppressBlock())
            {
                tsiChannelA.Checked = ! tsiChannelA.Checked;
            }

            pnlBaseView.Invalidate();
            pnlView.Invalidate();
        }

        private void tsiShowMipmap_Click(object sender, EventArgs e)
        {
            using (var block = new UIControlEventSuppressBlock())
            {
                tsiShowMipmap.Checked = ! tsiShowMipmap.Checked;
            }

            SetViewSize();
            SetViewPosition();

            pnlView.Invalidate();
            pnlBaseView.Invalidate();
        }

        private void pnlBaseView_Layout(object sender, LayoutEventArgs e)
        {
            SetViewPosition();
        }

        private void pnlBaseView_Paint(object sender, PaintEventArgs e)
        {
            e.Graphics.Clear(Color.Silver);

            if (isLoading_)
            {
                DrawLoading(e);
            }
        }

        private readonly string[] loading_ =
        {
            "Now Loading",
            "Now Loading.",
            "Now Loading..",
            "Now Loading...",
        };

        private void DrawLoading(PaintEventArgs e)
        {
            e.Graphics.DrawString(loading_[loadingCount_], pnlView.Font, Brushes.Black, 16, 16);
        }

        private void DrawImage(PaintEventArgs e)
        {
            using(var gib = new GraphicsInterpolationModeBlock(e.Graphics, InterpolationMode.NearestNeighbor))
            {
                var allRect = new Rectangle(0, 0, (int)(TextureWidth * Magnify) + frameWidth, (int)(TextureHeight * Magnify) + frameWidth);

                // 格子模様
//                if (tsiShowMipmap.Checked)
                {
                    using (var brush = new HatchBrush(HatchStyle.LargeCheckerBoard, Color.White, Color.LightGray))
                    {
                        e.Graphics.FillRectangle(brush, allRect);
                    }
                }

                // 枠
                {
                    e.Graphics.DrawRectangle(Pens.Black, allRect);
                }

                var imageRect = new Rectangle(frameWidth, frameWidth, (int)(TextureWidth * Magnify), (int)(TextureHeight * Magnify));

                var ptSampling = new PointF(0.0f, 0.0f);
                if (Magnify >= 2.0f)
                {
                    ptSampling.X = -0.5f;
                    ptSampling.Y = -0.5f;
                }

                // アルファのみ
                if(    (tsiChannelR.Checked == false) &&
                    (tsiChannelG.Checked == false) &&
                    (tsiChannelB.Checked == false) &&
                    tsiChannelA.Checked)
                {
                    if (texture_.HasAlpha)
                    {
                        e.Graphics.DrawImage(
                            tsiShowMipmap.Checked ?
                                texture_.AlphaMipmapImage :
                                texture_.AlphaImage,
                            imageRect,
                            ptSampling.X, ptSampling.Y,
                            TextureWidth, TextureHeight,
                            GraphicsUnit.Pixel
                        );
                    }
                }
                else
                // カラー
                {
                    // カラー要素別に行列作成
                    using (var ia = new ImageAttributes())
                    {
                        ia.SetColorMatrix(GetColorChannelMatrix(tsiChannelR.Checked, tsiChannelG.Checked, tsiChannelB.Checked));

                        e.Graphics.DrawImage(
                            tsiShowMipmap.Checked ?
                                texture_.ColorMipmapImage :
                                texture_.ColorImage,
                            imageRect,
                            ptSampling.X, ptSampling.Y,
                            TextureWidth, TextureHeight,
                            GraphicsUnit.Pixel,
                            ia
                        );
                    }

                    // アルファ合成
                    if (tsiChannelA.Checked)
                    {
                        if (texture_.HasAlpha)
                        {
                            using (var ia = new ImageAttributes())
                            {
                                ia.SetColorMatrix(
                                    new ColorMatrix
                                        {
                                        Matrix00 =  0.0f,    Matrix10 = 0.0f,    Matrix20 = 0.0f,    Matrix30 = 0.0f,    Matrix40 = 1.0f,
                                        Matrix01 =  0.0f,    Matrix11 = 0.0f,    Matrix21 = 0.0f,    Matrix31 = 0.0f,    Matrix41 = 0.0f,
                                        Matrix02 =  0.0f,    Matrix12 = 0.0f,    Matrix22 = 0.0f,    Matrix32 = 0.0f,    Matrix42 = 0.0f,
                                        Matrix03 = -0.5f,    Matrix13 = 0.0f,    Matrix23 = 0.0f,    Matrix33 = 0.5f,    Matrix43 = 0.0f,
                                    }
                                );

                                e.Graphics.DrawImage(
                                    tsiShowMipmap.Checked ?
                                        texture_.AlphaMipmapImage :
                                        texture_.AlphaImage,
                                    imageRect,
                                    ptSampling.X, ptSampling.Y,
                                    TextureWidth, TextureHeight,
                                    GraphicsUnit.Pixel,
                                    ia
                                );
                            }
                        }
                    }
                }
            }
        }

        /// <summary>
        /// カラーチャンネル行列を取得。
        /// </summary>
        private ColorMatrix GetColorChannelMatrix(bool isR, bool isG, bool isB)
        {
            var cm = new ColorMatrix();

            // ＲＧＢ合成
            if (isR && isG && isB)
            {
                // 単位行列
                cm.Matrix00 = 1.0f;    cm.Matrix10 = 0.0f;    cm.Matrix20 = 0.0f;    cm.Matrix30 = 0.0f;    cm.Matrix40 = 0.0f;
                cm.Matrix01 = 0.0f;    cm.Matrix11 = 1.0f;    cm.Matrix21 = 0.0f;    cm.Matrix31 = 0.0f;    cm.Matrix41 = 0.0f;
                cm.Matrix02 = 0.0f;    cm.Matrix12 = 0.0f;    cm.Matrix22 = 1.0f;    cm.Matrix32 = 0.0f;    cm.Matrix42 = 0.0f;
            }
            // 各成分の組み合わせ
            else
            {
                // 一旦ゼロ行列にする
                cm.Matrix00 = 0.0f;    cm.Matrix10 = 0.0f;    cm.Matrix20 = 0.0f;    cm.Matrix30 = 0.0f;    cm.Matrix40 = 0.0f;
                cm.Matrix01 = 0.0f;    cm.Matrix11 = 0.0f;    cm.Matrix21 = 0.0f;    cm.Matrix31 = 0.0f;    cm.Matrix41 = 0.0f;
                cm.Matrix02 = 0.0f;    cm.Matrix12 = 0.0f;    cm.Matrix22 = 0.0f;    cm.Matrix32 = 0.0f;    cm.Matrix42 = 0.0f;

                // Ｒ成分
                if (isR)
                {
                    cm.Matrix00 = 1.0f;    cm.Matrix10 = 0.0f;    cm.Matrix20 = 0.0f;    cm.Matrix30 = 0.0f;    cm.Matrix40 = 0.0f;

                    // Ｒ成分のみ
                    if ((isG == false) && (isB == false))
                    {
                        cm.Matrix01 = 1.0f;    cm.Matrix11 = 0.0f;    cm.Matrix21 = 0.0f;    cm.Matrix31 = 0.0f;    cm.Matrix41 = 0.0f;
                        cm.Matrix02 = 1.0f;    cm.Matrix12 = 0.0f;    cm.Matrix22 = 0.0f;    cm.Matrix32 = 0.0f;    cm.Matrix42 = 0.0f;
                    }
                }
                // Ｇ成分
                if (isG)
                {
                    cm.Matrix01 = 0.0f;    cm.Matrix11 = 1.0f;    cm.Matrix21 = 0.0f;    cm.Matrix31 = 0.0f;    cm.Matrix41 = 0.0f;

                    // Ｇ成分のみ
                    if ((isR == false) && (isB == false))
                    {
                        cm.Matrix00 = 0.0f;    cm.Matrix10 = 1.0f;    cm.Matrix20 = 0.0f;    cm.Matrix30 = 0.0f;    cm.Matrix40 = 0.0f;
                        cm.Matrix02 = 0.0f;    cm.Matrix12 = 1.0f;    cm.Matrix22 = 0.0f;    cm.Matrix32 = 0.0f;    cm.Matrix42 = 0.0f;
                    }
                }
                // Ｂ成分
                if (isB)
                {
                    cm.Matrix02 = 0.0f;    cm.Matrix12 = 0.0f;    cm.Matrix22 = 1.0f;    cm.Matrix32 = 0.0f;    cm.Matrix42 = 0.0f;

                    // Ｂ成分のみ
                    if ((isR == false) && (isG == false))
                    {
                        cm.Matrix00 = 0.0f;    cm.Matrix10 = 0.0f;    cm.Matrix20 = 1.0f;    cm.Matrix30 = 0.0f;    cm.Matrix40 = 0.0f;
                        cm.Matrix01 = 0.0f;    cm.Matrix11 = 0.0f;    cm.Matrix21 = 1.0f;    cm.Matrix31 = 0.0f;    cm.Matrix41 = 0.0f;
                    }
                }
            }
            return cm;
        }

        private void pnlView_MouseMove(object sender, MouseEventArgs e)
        {
            UpdatePixelInfo(e.X, e.Y, true);
        }

        private void pnlView_MouseLeave(object sender, EventArgs e)
        {
            UpdatePixelInfo(0, 0, false);
        }

        private void UpdatePixelInfo(int mouseX, int mouseY, bool isEnable)
        {
            var info = MakePixelInfo(mouseX, mouseY);

            if (isEnable && info.Enable)
            {
                tslPosition.Text    = string.Format("{0} : ( {1}, {2} )", Strings.Position, info.Position.X, info.Position.Y);
                tslRGBA.Text        = texture_.HasAlpha ?
                                        string.Format("RGBA : ( {0}, {1}, {2}, {3} )", info.Color.R, info.Color.G, info.Color.B, info.Color.A) :
                                        string.Format("RGB : ( {0}, {1}, {2} )",       info.Color.R, info.Color.G, info.Color.B);
            }
            else
            {
                tslPosition.Text    = string.Empty;
                tslRGBA.Text        = string.Empty;
            }
        }

        private class PixelInfo
        {
            public PixelInfo()
            {
                Enable = false;
            }

            public bool  Enable{        get; set; }
            public int   MipmapLevel{    get; set; }
            public Point Position{        get; set; }
            public Color Color{            get; set; }
        }

        private PixelInfo MakePixelInfo(int mouseX, int mouseY)
        {
            var info = new PixelInfo();
            {
                if (texture_ != null)
                {
                     // 枠線部分を考慮する
                    mouseX -= frameWidth;
                    mouseY -= frameWidth;

                    if (mouseX < 0 || mouseY < 0)
                    {
                        info.Enable = false;
                        return info;
                    }

                    // 拡大率を考慮する
                    mouseX = (int)(mouseX / Magnify);
                    mouseY = (int)(mouseY / Magnify);

                    var isEnable   = false;
                    var mipmapLevel = 0;
                    var imageW = texture_.Width;
                    var imageH = texture_.Height;

                    int heightPrev = 0;
                    int heightNext = imageH;

                    // ミップマップを追う
                    for (int i = 0; i < texture_.MipmapCount; i++)
                    {
                        // 縦方向にイメージ内
                        if (mouseY < heightNext)
                        {
                            // 横方向にイメージ内
                            if (mouseX < imageW)
                            {
                                isEnable = true;
                                mipmapLevel = i;
                                mouseY = mouseY - heightPrev;
                            }

                            // 終了
                            break;
                        }

                        // 次のレベルへ
                        heightPrev += imageH;
                        {
                            var nextIndex = Math.Min(i + 1, texture_.ColorImages.Length - 1);

                            imageW = texture_.ColorImages[nextIndex].Width;
                            imageH = texture_.ColorImages[nextIndex].Height;
                        }
                        heightNext += imageH;
                    }

                    // ここまででできている情報を保存する
                    info.Enable         = isEnable;
                    info.MipmapLevel    = mipmapLevel;
                    info.Position       = new Point(mouseX, mouseY);

                    // 色をつくる
                    if (isEnable)
                    {
                        info.Color = texture_.ColorImages[mipmapLevel].GetPixel(mouseX, mouseY);

                        if (texture_.HasAlpha)
                        {
                            // アルファはグレースケールで入っている
                            Color alpha = texture_.AlphaImages[mipmapLevel].GetPixel(mouseX, mouseY);

                            info.Color = Color.FromArgb(alpha.R, info.Color.R, info.Color.G, info.Color.B);
                        }
                    }
                }
            }
            return info;
        }

        private void OnHelp(object sender, EventArgs e)
        {
            String dirName = System.IO.Path.GetDirectoryName(System.Reflection.Assembly.GetEntryAssembly().Location);
            String pathToHelpFile = dirName + "\\TextureViewer.html";

            if (System.IO.File.Exists(pathToHelpFile))
            {
                System.Diagnostics.Process p = new System.Diagnostics.Process();
                p.StartInfo.FileName = pathToHelpFile;
                p.Start();
            }
        }
    }
}
