﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Drawing.Imaging;
using System.IO;
using System.Runtime.InteropServices;
using System.Reflection;
using System.Threading.Tasks;
using System.Windows.Forms;
using System.Xml.Serialization;
using System.ComponentModel;
using System.Linq;
using nw.g3d.nw4f_3dif;
using nw.g3d.iflib;

namespace App.Data
{
    // テクスチャクラス
    // 注意：3DEditorとは別実装です。
    public class Texture
    {
        public Bitmap ColorImage { get { return HasColor ? ColorImages[0] : null; } }
        public Bitmap AlphaImage { get { return HasAlpha ? AlphaImages[0] : null; } }

        private Bitmap[] SourceImages_;       // 元のイメージ ほぼFtx
        private Bitmap[] ColorImages_;        // コンポーネントセレクタを通したあとのRGB1
        private Bitmap[] AlphaImages_;        // コンポーネントセレクタを通したあとのAAA1

        public bool HasColor { get { return true; } }
        public bool HasAlpha { get; private set; }

        public Bitmap[] ColorImages
        {
            get
            {
                if ((ColorImages_ == null) || (AlphaImages_ == null))
                {
                    CreateColorAlphaImages();
                }

                return ColorImages_;
            }
        }

        public Bitmap[] AlphaImages
        {
            get
            {
                if ((ColorImages_ == null) || (AlphaImages_ == null))
                {
                    CreateColorAlphaImages();
                }

                return AlphaImages_;
            }
        }

        private Bitmap ColorMipmapImage_;
        public Bitmap ColorMipmapImage
        {
            get
            {
                if (ColorMipmapImage_ == null)
                {
                    ColorMipmapImage_ = new Bitmap(Width, HeightWithMipmap);

                    using (var image = Graphics.FromImage(ColorMipmapImage_))
                    {
                        image.Clear(Color.Transparent);

                        int y = 0;
                        foreach (var src in ColorImages)
                        {
                            image.DrawImage(src, 0, y);
                            y += src.Height;
                        }
                    }
                }

                return ColorMipmapImage_;
            }
        }

        private Bitmap AlphaMipmapImage_;
        public Bitmap AlphaMipmapImage
        {
            get
            {
                if (AlphaMipmapImage_ == null)
                {
                    AlphaMipmapImage_ = new Bitmap(Width, HeightWithMipmap);

                    using (var image = Graphics.FromImage(AlphaMipmapImage_))
                    {
                        image.Clear(Color.Transparent);

                        int y = 0;
                        foreach (var src in AlphaImages)
                        {
                            image.DrawImage(src, 0, y);
                            y += src.Height;
                        }
                    }
                }

                return AlphaMipmapImage_;
            }
        }

        public bool HasMipmap { get { return ColorImages.Length >= 2; } }
        public int MipmapCount { get { return ColorImages.Length; } }

        public int Width { get { return ColorImage.Width; } }
        public int Height { get { return ColorImage.Height; } }
        public int HeightWithMipmap { get { return ColorImages_.Sum(x => x.Height); } }

        public string FileName { get; private set; }

        public textureType Data { get; private set; }

        private List<G3dStream> BinaryStreams_;
        public List<G3dStream> BinaryStreams
        {
            get
            {
                if (BinaryStreams_ == null)
                {
                    var nw4f_3dif = new nw4f_3difType();
                    nw4f_3dif.Item = Data; ;
                    BinaryStreams_ = new List<G3dStream>();

                    // ストリームがあれば保持する
                    if (G3dStreamUtility.HasStreamArray(nw4f_3dif))
                    {
                        G3dStreamUtility.ToStreams(
                            BinaryStreams_, nw4f_3dif.RootElement.stream_array);
                    }
                }

                return BinaryStreams_;
            }
            protected set
            {
                BinaryStreams_ = value;
            }
        }

        /// <summary>
        /// 静的コンストラクタです。
        /// </summary>
        static Texture()
        {
            // exeのパスを取得
            Assembly mainAssembly = Assembly.GetEntryAssembly();
            string executablePath = Path.GetDirectoryName(mainAssembly.Location);

            // スキーマのベースパスを設定
            const string PATH_FOR_NINTENDO_SDK = @"..\..\..\..\3dTools\3dIntermediateFileXsd\";
            const string PATH_FOR_UNIT_PACKAGE = @"3dTools\3dIntermediateFileXsd\";

            string path_nsdk = Path.Combine(executablePath, PATH_FOR_NINTENDO_SDK);  // Nintendo SDKのパス
            string path_unit = Path.Combine(executablePath, PATH_FOR_UNIT_PACKAGE);  // 単体パッケージのパス

            if (Directory.Exists(path_unit))
            {
                XsdBasePath = path_unit;
            }
            else if (Directory.Exists(path_nsdk))
            {
                XsdBasePath = path_nsdk;
            }
        }

        /// <summary>
        /// スキーマのベースパスを取得または設定します。
        /// </summary>
        public static string XsdBasePath { get; private set; }

        // ファイルから読み込む
        // errorMsg メインスレッドとは別スレッドから呼び出される場合、言語を切り替えても切り替わらないため、
        //          関数内部でリソースから文字列を取得しない。
        public bool LoadFromFile(string fileName)
        {
            // TODO:パスのチェック
            ;

            nw4f_3difType nw4f_3dif = null;
            List<G3dStream> streamArray = null;
            {
                try
                {
                    if (G3dPath.IsStreamBinaryPath(fileName))
                    {
                        streamArray = new List<G3dStream>();
                        nw4f_3dif = IfBinaryReadUtility.Read(streamArray, fileName, XsdBasePath);
                    }
                    else
                    {
                        nw4f_3dif = IfReadUtility.Read(fileName, XsdBasePath);
                    }
                }
                catch (Exception exception)
                {
                    // FTXファイルが古いバージョンで、かつXsdBaseBathが未指定のときにロードが失敗することがある
                    // そのときはエラーメッセージを出す
                    if (string.IsNullOrEmpty(XsdBasePath))
                    {
                        string errMsg = App.res.Strings.No_g3dxsd_Folder;
                        MessageBox.Show(errMsg + "\n" + (exception.Message ?? string.Empty), "Error", MessageBoxButtons.OK);
                    }
                    else
                    {
                        MessageBox.Show(exception.Message ?? string.Empty, "Error", MessageBoxButtons.OK);
                    }

                    //                error = new IO_Error(fileName, IO_ErrorType.SerializeFailed);
                    return false;
                }
            }

            Debug.Assert(nw4f_3dif.Item is textureType);

            Data = nw4f_3dif.Item as textureType;
            BinaryStreams = streamArray;

            //            // ロード時にイメージを作成しておく
            //            CreateImages();

            FileName = fileName;

            return true;
        }

        public void CreateImages()
        {
#if false
//            ColorImages_ = new Bitmap[Data.texture_info.mip_level];
//            AlphaImages_ = new Bitmap[Data.texture_info.mip_level];

            var images = new Bitmap[Data.texture_info.mip_level];

            // キューブテクスチャ
            if (Data.texture_info.dimension == texture_info_dimensionType.cube)
            {
                var bitmaps = TexUtilsProxy.ConvertToCubeBitmap(Data, BinaryStreams);

                for(int i = 0;i != Data.texture_info.mip_level;++ i)
                {
                    int width  = Math.Max(1, Data.texture_info.width  >> i);
                    int height = Math.Max(1, Data.texture_info.height >> i);

                    images[i] = new Bitmap(width * 3, height * 2, PixelFormat.Format32bppArgb);
                    {
                        using (var color = Graphics.FromImage(images[i]))
                        {
                            color.Clear(Color.Transparent);
                            color.DrawImage(bitmaps[0][i], width * 0, height * 0);
                            color.DrawImage(bitmaps[1][i], width * 0, height * 1);
                            color.DrawImage(bitmaps[2][i], width * 1, height * 0);
                            color.DrawImage(bitmaps[3][i], width * 1, height * 1);
                            color.DrawImage(bitmaps[4][i], width * 2, height * 0);
                            color.DrawImage(bitmaps[5][i], width * 2, height * 1);
                        }
                    }
                }
            }
            // 1Dテクスチャ、2Dテクスチャ
            else
            {
                var bitmaps = TexUtilsProxy.ConvertToBitmap(Data, BinaryStreams);

                for(int i = 0;i != Data.texture_info.mip_level;++ i)
                {
                    images[i] = bitmaps[i];
                }
            }

            SourceImages_     = images;
#endif

            var images = new Bitmap[Data.texture_info.mip_level];

            switch (Data.texture_info.dimension)
            {
                case texture_info_dimensionType.Item1d:
                case texture_info_dimensionType.Item2d:
                    {
                        var bitmaps = TexUtilsProxy.ConvertTo1d2dBitmap(Data, BinaryStreams);

                        for (int i = 0; i != Data.texture_info.mip_level; ++i)
                        {
                            images[i] = bitmaps[i];
                        }

                        break;
                    }

                case texture_info_dimensionType.Item3d:
                case texture_info_dimensionType.Item1d_array:
                case texture_info_dimensionType.Item2d_array:
                    {
                        var bitmaps =
                            (Data.texture_info.dimension == texture_info_dimensionType.Item3d) ? TexUtilsProxy.ConvertTo3dBitmap(Data, BinaryStreams) :
                            (Data.texture_info.dimension == texture_info_dimensionType.Item1d_array) ? TexUtilsProxy.ConvertTo1dArrayBitmap(Data, BinaryStreams) :
                            (Data.texture_info.dimension == texture_info_dimensionType.Item2d_array) ? TexUtilsProxy.ConvertTo2dArrayBitmap(Data, BinaryStreams) :
                                                                                                       null;

                        Debug.Assert(bitmaps != null);

                        int size = (int)Math.Ceiling(Math.Sqrt(Data.texture_info.depth));
                        int depthCount = Data.texture_info.depth;

                        int allWidth = size;
                        int allHeight = size;

                        // ミップマップ最小サイズを保存する
                        // 縦方向に並ぶ１次元テクスチャ最小サイズを制限する
                        //                    MipmapMinSize = new Size(size, size);

                        for (int i = 0; i != Data.texture_info.mip_level; ++i)
                        {
                            int width = Math.Max(1, Data.texture_info.width >> i);
                            int height = Math.Max(1, Data.texture_info.height >> i);

                            images[i] = new Bitmap(width * allWidth, height * allHeight, PixelFormat.Format32bppArgb);
                            {
                                using (var color = Graphics.FromImage(images[i]))
                                {
                                    color.Clear(Color.Transparent);

                                    for (int j = 0; j != depthCount; ++j)
                                    {
                                        int x = j % size;
                                        int y = j / size;

                                        color.DrawImage(bitmaps[j][i], width * x, height * y);
                                    }
                                }
                            }

                            if (Data.texture_info.dimension == texture_info_dimensionType.Item3d)
                            {
                                if (allHeight > 1)
                                {
                                    allHeight >>= 1;
                                }
                                else if (allWidth > 1)
                                {
                                    allWidth >>= 1;
                                }

                                depthCount >>= 1;
                            }
                        }

                        break;
                    }

                case texture_info_dimensionType.cube:
                    {
                        var bitmaps = TexUtilsProxy.ConvertToCubeBitmap(Data, BinaryStreams);

                        for (int i = 0; i != Data.texture_info.mip_level; ++i)
                        {
                            int width = Math.Max(1, Data.texture_info.width >> i);
                            int height = Math.Max(1, Data.texture_info.height >> i);

                            images[i] = bitmaps[0][i];

                            images[i] = new Bitmap(width * 3, height * 2, PixelFormat.Format32bppArgb);
                            {
                                using (var color = Graphics.FromImage(images[i]))
                                {
                                    var colorMatrix = new ColorMatrix
                                    {
                                        Matrix00 = 1.0f,
                                        Matrix01 = 0.0f,
                                        Matrix02 = 0.0f,
                                        Matrix03 = 0.0f,
                                        Matrix04 = 0.0f,
                                        Matrix10 = 0.0f,
                                        Matrix11 = 1.0f,
                                        Matrix12 = 0.0f,
                                        Matrix13 = 0.0f,
                                        Matrix14 = 0.0f,
                                        Matrix20 = 0.0f,
                                        Matrix21 = 0.0f,
                                        Matrix22 = 1.0f,
                                        Matrix23 = 0.0f,
                                        Matrix24 = 0.0f,
                                        Matrix30 = 0.0f,
                                        Matrix31 = 0.0f,
                                        Matrix32 = 0.0f,
                                        Matrix33 = 1.0f,
                                        Matrix34 = 0.0f,
                                        Matrix40 = 0.0f,
                                        Matrix41 = 0.0f,
                                        Matrix42 = 0.0f,
                                        Matrix43 = 0.0f,
                                        Matrix44 = 0.0f
                                    };

                                    var imageAttributes = new ImageAttributes();
                                    imageAttributes.SetColorMatrix(colorMatrix);
                                    color.CompositingMode = CompositingMode.SourceCopy;
                                    color.InterpolationMode = InterpolationMode.NearestNeighbor;
                                    color.CompositingQuality = CompositingQuality.HighQuality;
                                    color.Clear(Color.Transparent);
                                    color.DrawImage(bitmaps[0][i], new Rectangle(width * 0, height * 0, bitmaps[0][i].Width, bitmaps[0][i].Height), 0, 0, bitmaps[0][i].Width, bitmaps[0][i].Height, GraphicsUnit.Pixel, imageAttributes);
                                    color.DrawImage(bitmaps[1][i], new Rectangle(width * 0, height * 1, bitmaps[1][i].Width, bitmaps[1][i].Height), 0, 0, bitmaps[1][i].Width, bitmaps[1][i].Height, GraphicsUnit.Pixel, imageAttributes);
                                    color.DrawImage(bitmaps[2][i], new Rectangle(width * 1, height * 0, bitmaps[2][i].Width, bitmaps[2][i].Height), 0, 0, bitmaps[2][i].Width, bitmaps[2][i].Height, GraphicsUnit.Pixel, imageAttributes);
                                    color.DrawImage(bitmaps[3][i], new Rectangle(width * 1, height * 1, bitmaps[3][i].Width, bitmaps[3][i].Height), 0, 0, bitmaps[3][i].Width, bitmaps[3][i].Height, GraphicsUnit.Pixel, imageAttributes);
                                    color.DrawImage(bitmaps[4][i], new Rectangle(width * 2, height * 0, bitmaps[4][i].Width, bitmaps[4][i].Height), 0, 0, bitmaps[4][i].Width, bitmaps[4][i].Height, GraphicsUnit.Pixel, imageAttributes);
                                    color.DrawImage(bitmaps[5][i], new Rectangle(width * 2, height * 1, bitmaps[5][i].Width, bitmaps[5][i].Height), 0, 0, bitmaps[5][i].Width, bitmaps[5][i].Height, GraphicsUnit.Pixel, imageAttributes);
                                }
                            }
                        }

                        break;
                    }

                case texture_info_dimensionType.cube_array:
                    {
                        var bitmaps = TexUtilsProxy.ConvertToCubeArrayBitmap(Data, BinaryStreams);

                        int depthCount = Data.texture_info.depth / 6;
                        int size = (int)Math.Ceiling(Math.Sqrt(depthCount));

                        int allWidth = size;
                        int allHeight = size;

                        for (int m = 0; m != Data.texture_info.mip_level; ++m)
                        {
                            int width = Math.Max(1, Data.texture_info.width >> m);
                            int height = Math.Max(1, Data.texture_info.height >> m);

                            images[m] = new Bitmap(width * 3 * allWidth, height * 2 * allHeight, PixelFormat.Format32bppArgb);
                            {
                                using (var color = Graphics.FromImage(images[m]))
                                {
                                    color.Clear(Color.Transparent);

                                    for (int d = 0; d != depthCount; ++d)
                                    {
                                        int x = (d % size) * width * 3;
                                        int y = (d / size) * height * 2;

                                        color.DrawImage(bitmaps[d * 6 + 0][m], x + width * 0, y + height * 0);
                                        color.DrawImage(bitmaps[d * 6 + 1][m], x + width * 0, y + height * 1);
                                        color.DrawImage(bitmaps[d * 6 + 2][m], x + width * 1, y + height * 0);
                                        color.DrawImage(bitmaps[d * 6 + 3][m], x + width * 1, y + height * 1);
                                        color.DrawImage(bitmaps[d * 6 + 4][m], x + width * 2, y + height * 0);
                                        color.DrawImage(bitmaps[d * 6 + 5][m], x + width * 2, y + height * 1);
                                    }
                                }
                            }
                        }

                        break;
                    }
            }

            SourceImages_ = images;
        }

        private void CreateColorAlphaImages()
        {
            if (Data == null)
                return;

            if (Data.texture_info == null)
                return;

            var isHintNormal = Data.texture_info.hint == "normal";
            bool replaceGwithA = false;
            var isAstcFormat = (Data.texture_info.quantize_type >= texture_info_quantize_typeType.unorm_astc_4x4 &&
                          Data.texture_info.quantize_type <= texture_info_quantize_typeType.srgb_astc_12x12);

            {
                if (isHintNormal)
                {
                    replaceGwithA = isAstcFormat;
                }
                else
                {
                    if (Data.original_image_array != null)
                    {
                        var originalFormat = Data.original_image_array.original_image[0].format;
                        replaceGwithA = (originalFormat == original_image_formatType.rgba8 || originalFormat == original_image_formatType.rgba32f) &&
                                       (Data.texture_info.quantize_type == texture_info_quantize_typeType.unorm_4_4 ||
                                        Data.texture_info.quantize_type == texture_info_quantize_typeType.unorm_8_8 ||
                                        Data.texture_info.quantize_type == texture_info_quantize_typeType.unorm_bc5 ||
                                        Data.texture_info.quantize_type == texture_info_quantize_typeType.unorm_eac_11_11);
                    }

                }
            }

            ColorMatrix colorMatrix = null;
            ColorMatrix alphaMatrix = null;
            {
                var compSelR = MakeCompSelMatrixCol_Color(Data.texture_info.comp_sel[0], replaceGwithA);
                var compSelG = MakeCompSelMatrixCol_Color(Data.texture_info.comp_sel[1], replaceGwithA);
                var compSelB = MakeCompSelMatrixCol_Color(Data.texture_info.comp_sel[2], replaceGwithA);
                var compSelA = MakeCompSelMatrixCol_Color(Data.texture_info.comp_sel[3], replaceGwithA);

                {
                    colorMatrix = new ColorMatrix(
                        new[]
                        {
                            new []{compSelR[0], compSelG[0], compSelB[0], 0.0f, 0.0f},
                            new []{compSelR[1], compSelG[1], compSelB[1], 0.0f, 0.0f},
                            new []{compSelR[2], compSelG[2], compSelB[2], 0.0f, 0.0f},
                            new []{compSelR[3], compSelG[3], compSelB[3], 0.0f, 0.0f},
                            new []{compSelR[4], compSelG[4], compSelB[4], 1.0f, 0.0f},
                        }
                    );

                    alphaMatrix = new ColorMatrix(
                        new[]
                        {
                            new []{compSelA[0], compSelA[0], compSelA[0], 0.0f, 0.0f},
                            new []{compSelA[1], compSelA[1], compSelA[1], 0.0f, 0.0f},
                            new []{compSelA[2], compSelA[2], compSelA[2], 0.0f, 0.0f},
                            new []{compSelA[3], compSelA[3], compSelA[3], 0.0f, 0.0f},
                            new []{compSelA[4], compSelA[4], compSelA[4], 1.0f, 0.0f},
                        }
                    );
                }
            }

            var mipmapSize = SourceImages_.Length;

            ColorImages_ = new Bitmap[mipmapSize];
            AlphaImages_ = new Bitmap[mipmapSize];

            for (int i = 0; i != mipmapSize; ++i)
            {
                var src = SourceImages_[i];
                var srcW = src.Width;
                var srcH = src.Height;

                ColorImages_[i] = new Bitmap(srcW, srcH, PixelFormat.Format32bppArgb);
                AlphaImages_[i] = new Bitmap(srcW, srcH, PixelFormat.Format32bppArgb);

                {
                    using (var ia = new ImageAttributes())
                    using (var g = Graphics.FromImage(ColorImages_[i]))
                    {
                        ia.SetColorMatrix(colorMatrix);

                        g.DrawImage(
                            src,
                            new Rectangle(0, 0, srcW, srcH),
                            0, 0, srcW, srcH,
                            GraphicsUnit.Pixel,
                            ia
                        );
                    }
                }

                {
                    using (var ia = new ImageAttributes())
                    using (var g = Graphics.FromImage(AlphaImages_[i]))
                    {
                        ia.SetColorMatrix(alphaMatrix);

                        g.DrawImage(
                            src,
                            new Rectangle(0, 0, srcW, srcH),
                            0, 0, srcW, srcH,
                            GraphicsUnit.Pixel,
                            ia
                        );
                    }
                }

                if (isHintNormal)
                {
                    if (Data.texture_info.quantize_type == texture_info_quantize_typeType.snorm_8_8 ||
                        Data.texture_info.quantize_type == texture_info_quantize_typeType.sint_8_8)
                    {
                        var bitmap = ColorImages_[i];
                        var bitmapData = bitmap.LockBits(new Rectangle(0, 0, bitmap.Width, bitmap.Height),
                            ImageLockMode.ReadWrite, PixelFormat.Format32bppArgb);

                        unsafe
                        {
                            var startPtr = (UInt32*)bitmapData.Scan0;
                            var height = bitmap.Height;
                            var width = bitmap.Width;
                            var stride = bitmapData.Stride;

                            Parallel.For((long)0, height, i1 =>
                            {
                                var linePtr = startPtr + i1 * stride / sizeof(UInt32);

                                Parallel.For((long)0, width, i2 =>
                                {
                                    var currentPtr = linePtr + i2;

                                    var colorPtr = (Byte*)currentPtr;

                                    var sr = colorPtr[2] / 255.0f;
                                    var sg = colorPtr[1] / 255.0f;
                                    var sb = Math.Sqrt(1.0f - Math.Min(sr * sr + sg * sg, 1.0f));

                                    colorPtr[2] = (byte)(128 + (int)(127.0f * sr));
                                    colorPtr[1] = (byte)(128 + (int)(127.0f * sg));
                                    colorPtr[0] = (byte)(128 + (int)(127.0f * sb));
                                });
                            });
                        }

                        bitmap.UnlockBits(bitmapData);
                    }
                    else if (Data.texture_info.quantize_type == texture_info_quantize_typeType.snorm_bc5)
                    {
                        var bitmap = ColorImages_[i];
                        var bitmapData = bitmap.LockBits(new Rectangle(0, 0, bitmap.Width, bitmap.Height),
                            ImageLockMode.ReadWrite, PixelFormat.Format32bppArgb);

                        unsafe
                        {
                            var startPtr = (UInt32*)bitmapData.Scan0;
                            var height = bitmap.Height;
                            var width = bitmap.Width;
                            var stride = bitmapData.Stride;

                            Parallel.For((long)0, height, i1 =>
                            {
                                var linePtr = startPtr + i1 * stride / sizeof(UInt32);

                                Parallel.For((long)0, width, i2 =>
                                {
                                    var currentPtr = linePtr + i2;

                                    var colorPtr = (Byte*)currentPtr;

                                    var sr = Math.Min(Math.Max((((int)colorPtr[2]) - 128) / 127.0f, -1.0f), +1.0f);
                                    var sg = Math.Min(Math.Max((((int)colorPtr[1]) - 128) / 127.0f, -1.0f), +1.0f);
                                    var sb = Math.Sqrt(1.0f - Math.Min(sr * sr + sg * sg, 1.0f));

                                    colorPtr[0] = (byte)Math.Min(Math.Max(128 + (int)(127.0f * sb), 0), 255);
                                });
                            });
                        }

                        bitmap.UnlockBits(bitmapData);
                    }
                }

                // キューブマップであれば展開図にする
                if (Data.texture_info.dimension == texture_info_dimensionType.cube)
                {
                    ColorImages_[i] = MakeOpenedCubeImage(ColorImages_[i]);
                    AlphaImages_[i] = MakeOpenedCubeImage(AlphaImages_[i]);
                }
            }

            MakeHasAlpha();
        }

        // コンポーネントセレクタの色変換行列の列を求める
        private static float[] MakeCompSelMatrixCol_Color(texture_info_comp_selValue compSel, bool replaceGwithA)
        {
            var s = new float[5] { 0.0f, 0.0f, 0.0f, 0.0f, 0.0f };
            {
                switch (compSel)
                {
                    case texture_info_comp_selValue.r: s[0] = 1.0f; break;
                    case texture_info_comp_selValue.g: s[1] = 1.0f; break;
                    case texture_info_comp_selValue.b: s[2] = 1.0f; break;
                    case texture_info_comp_selValue.a: s[3] = 1.0f; break;
                    case texture_info_comp_selValue.Item0: s[4] = 0.0f; break;
                    case texture_info_comp_selValue.Item1: s[4] = 1.0f; break;
                }
            }
            if (replaceGwithA)
            {
                var temp = s[1];
                s[1] = s[3];
                s[3] = temp;
            }
            return s;
        }

        private static Bitmap MakeOpenedCubeImage(Bitmap src)
        {
            int w = src.Width / 3;
            int h = src.Height / 2;

            var image = new Bitmap(w * 4, h * 3, PixelFormat.Format32bppArgb);
            {
                using (var g = Graphics.FromImage(image))
                {
                    g.Clear(Color.Transparent);
                    g.DrawImage(src, w * 2, h * 1, new Rectangle(w * 0, h * 0, w, h), GraphicsUnit.Pixel);
                    g.DrawImage(src, w * 0, h * 1, new Rectangle(w * 0, h * 1, w, h), GraphicsUnit.Pixel);
                    g.DrawImage(src, w * 1, h * 0, new Rectangle(w * 1, h * 0, w, h), GraphicsUnit.Pixel);
                    g.DrawImage(src, w * 1, h * 2, new Rectangle(w * 1, h * 1, w, h), GraphicsUnit.Pixel);
                    g.DrawImage(src, w * 3, h * 1, new Rectangle(w * 2, h * 0, w, h), GraphicsUnit.Pixel);
                    g.DrawImage(src, w * 1, h * 1, new Rectangle(w * 2, h * 1, w, h), GraphicsUnit.Pixel);
                }
            }
            return image;
        }

        private void MakeHasAlpha()
        {
            if (
                (Data.texture_info.quantize_type == texture_info_quantize_typeType.unorm_1_5_5_5) ||
                (Data.texture_info.quantize_type == texture_info_quantize_typeType.unorm_5_5_5_1) ||
                (Data.texture_info.quantize_type == texture_info_quantize_typeType.unorm_4_4_4_4) ||

                (Data.texture_info.quantize_type == texture_info_quantize_typeType.unorm_2_10_10_10) ||

                (Data.texture_info.quantize_type == texture_info_quantize_typeType.unorm_8_8_8_8) ||
                (Data.texture_info.quantize_type == texture_info_quantize_typeType.uint_8_8_8_8) ||
                (Data.texture_info.quantize_type == texture_info_quantize_typeType.snorm_8_8_8_8) ||
                (Data.texture_info.quantize_type == texture_info_quantize_typeType.sint_8_8_8_8) ||
                (Data.texture_info.quantize_type == texture_info_quantize_typeType.srgb_8_8_8_8) ||

                (Data.texture_info.quantize_type == texture_info_quantize_typeType.unorm_10_10_10_2) ||
                (Data.texture_info.quantize_type == texture_info_quantize_typeType.uint_10_10_10_2) ||

                (Data.texture_info.quantize_type == texture_info_quantize_typeType.unorm_16_16_16_16) ||
                (Data.texture_info.quantize_type == texture_info_quantize_typeType.uint_16_16_16_16) ||
                (Data.texture_info.quantize_type == texture_info_quantize_typeType.snorm_16_16_16_16) ||
                (Data.texture_info.quantize_type == texture_info_quantize_typeType.sint_16_16_16_16) ||
                (Data.texture_info.quantize_type == texture_info_quantize_typeType.float_16_16_16_16) ||
                (Data.texture_info.quantize_type == texture_info_quantize_typeType.uint_32_32_32_32) ||
                (Data.texture_info.quantize_type == texture_info_quantize_typeType.sint_32_32_32_32) ||
                (Data.texture_info.quantize_type == texture_info_quantize_typeType.float_32_32_32_32) ||

                (Data.texture_info.quantize_type == texture_info_quantize_typeType.unorm_bc1) ||
                (Data.texture_info.quantize_type == texture_info_quantize_typeType.srgb_bc1) ||
                (Data.texture_info.quantize_type == texture_info_quantize_typeType.unorm_bc2) ||
                (Data.texture_info.quantize_type == texture_info_quantize_typeType.srgb_bc2) ||
                (Data.texture_info.quantize_type == texture_info_quantize_typeType.unorm_bc3) ||
                (Data.texture_info.quantize_type == texture_info_quantize_typeType.srgb_bc3) ||

                (Data.texture_info.quantize_type == texture_info_quantize_typeType.unorm_etc2_mask) ||
                (Data.texture_info.quantize_type == texture_info_quantize_typeType.unorm_etc2_alpha) ||
                (Data.texture_info.quantize_type == texture_info_quantize_typeType.srgb_etc2_mask) ||
                (Data.texture_info.quantize_type == texture_info_quantize_typeType.srgb_etc2_alpha) ||
                (Data.texture_info.quantize_type == texture_info_quantize_typeType.unorm_pvrtc1_alpha_2bpp) ||
                (Data.texture_info.quantize_type == texture_info_quantize_typeType.unorm_pvrtc1_alpha_4bpp) ||
                (Data.texture_info.quantize_type == texture_info_quantize_typeType.srgb_pvrtc1_alpha_2bpp) ||
                (Data.texture_info.quantize_type == texture_info_quantize_typeType.srgb_pvrtc1_alpha_4bpp) ||
                (Data.texture_info.quantize_type == texture_info_quantize_typeType.unorm_pvrtc2_alpha_2bpp) ||
                (Data.texture_info.quantize_type == texture_info_quantize_typeType.unorm_pvrtc2_alpha_4bpp) ||
                (Data.texture_info.quantize_type == texture_info_quantize_typeType.srgb_pvrtc2_alpha_2bpp) ||
                (Data.texture_info.quantize_type == texture_info_quantize_typeType.srgb_pvrtc2_alpha_4bpp))
            {
                HasAlpha = true;
            }
            else
            {
                HasAlpha = Data.texture_info.comp_sel[3] != texture_info_comp_selValue.Item1;
            }
        }
    }
}
