﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef __ShellExtension_Utility_H__
#define __ShellExtension_Utility_H__

#include "ShellExtension_Type.h"
#include <memory>
//==============================================================================
//
// Helper macro to shut up compiler for not using parameter
//
//==============================================================================
#define NW_USE_VAR(x) { if ((x)==0) {} }

#define ARRAY_SIZE(arr) (sizeof(arr) / sizeof(*(arr)))

//==============================================================================
//
// Macro to simplify interface releasing
//
//==============================================================================
#define NW_RELEASE_INTERFACE(x)             \
    if ( (x)!=NULL )                          \
    {                                         \
        (x)->Release();                       \
        (x) = NULL;                           \
    };                                        \

//==============================================================================
//
// GUID to string in format of xxxxxxxx-xxxx-xxxx-xxxx-xxxxxxxxxxxx
//
//==============================================================================
std::wstring NWCreateUUIDString( const GUID &id );

//==============================================================================
//
// Compare GUID
//
//==============================================================================
bool NWIsSameGUID( const GUID &id1, const GUID &id2  );

//==============================================================================
//
// Find child node from xml
//
//==============================================================================
MSXML2::IXMLDOMNodePtr NWXMLFindChildNode( const MSXML2::IXMLDOMNodePtr node,
                                             const char* name );

//==============================================================================
//
// Get attribute from XML node
//
//==============================================================================
std::string NWXMLGetAttrValue( const MSXML2::IXMLDOMNodePtr node,
                                const char* attrName );

//==============================================================================
//
// Get encoder CLSID for saving image for debugging
//
//==============================================================================
int NWGetEncoderClsid( const WCHAR* format, CLSID* pClsid );

//==============================================================================
//
// Get format from string
//
//==============================================================================
NW_IMG_FMT NWGetTextureFormatFromString( const char* szStr );

//==============================================================================
//
// Get format string from format
//
//==============================================================================
const char* NWGetTextureFormatName( NW_IMG_FMT format );

//==============================================================================
//
// Get format string from format for unicode
//
//==============================================================================
const WCHAR* NWGetTextureFormatNameW( NW_IMG_FMT format );

//==============================================================================
//
// Get texture type name
//
//==============================================================================
const char* NWGetTextureTypeName( NW_IMG_TYPE texType );

//==============================================================================
//
// Get texture type name for unicode
//
//==============================================================================
const WCHAR* NWGetTextureTypeNameW( NW_IMG_TYPE texType );

//==============================================================================
//
// Get number of bits per pixel
//
//==============================================================================
int NWGetTextureFormatBitsPerPixel( NW_IMG_FMT format );

//==============================================================================
//
// Check if texture format is linear
//
//==============================================================================
int NWTextureFormatIsLinear( NW_IMG_FMT format );

//==============================================================================
//
// Check if texture format has alpha channel
//
//==============================================================================
bool NWTextureFormatHasAlpha( NW_IMG_FMT format, NW_IMG_FMT paletteFormat );

//==============================================================================
//
// String to linear flag array
//
//==============================================================================
bool NWStringToLinearFlagArray( const std::string &str, USHORT* linearFlags, int flagCount );

//==============================================================================
//
// String to component selector
//
//==============================================================================
NW_COMP_SEL NWStringToCompSel( const std::string &compSelStr );

//==============================================================================
//
// String to component selector
//
//==============================================================================
NW_COMP_SEL NWStringToCompSelW(const std::wstring &compSelStr);

//==============================================================================
//
// Component selector to string
//
//==============================================================================
std::wstring NWCompSelToString( const NW_COMP_SEL& compSel );

//==============================================================================
//
// Unicode string from ASCII string
//
//==============================================================================
std::wstring NWStringToWString( const std::string& s );

//==============================================================================
//
// ASCII string from Unicode string
// ( We can not do perfect conversion, just truncate to ASCII )
//
//==============================================================================
std::string NWWStringToString( const std::wstring& s );

//==============================================================================
//
// Convert UTF8 to W String
//
//==============================================================================
std::wstring NWUTF8toWString( /*[in]*/ const unsigned char* szSrc );

//==============================================================================
//
// Convert UTF8 to W String
//
//==============================================================================
std::wstring NWMultiByteStringToWString( /*[in]*/ const char* szSrc);

//==============================================================================
//
// Swap byte order of word ( 16 bits )
//
//==============================================================================
inline unsigned short NWSwapByteOrderWord ( unsigned short value )
{
    unsigned int val = value;
    unsigned short val1,val2;
    val1 =(unsigned short)(value&0x00FF);
    val2 =(unsigned short)(value>>8);

    val =(unsigned short)((val1<<8) + val2);

    return (unsigned short)val;
} // End of NWSwapByteOrderWord

//==============================================================================
//
// Swap byte order of dword ( 32 bits )
//
//==============================================================================
inline unsigned int NWSwapByteOrderDWord( unsigned int value )
{
    unsigned int val = value;
    unsigned int val1,val2;
    val1 =(unsigned int)(value&0xFFFF);
    val2 =(unsigned int)((value>>16));

    val1 =(unsigned int)NWSwapByteOrderWord((unsigned short)val1 );
    val2 =(unsigned int)NWSwapByteOrderWord((unsigned short)val2 );

    val =(unsigned int)((val1<<16) + val2);

    return val;
} // End of NWSwapByteOrderDWord

//==============================================================================
//
// Get extension of file path
//
//==============================================================================
std::wstring NWGetExtension( const WCHAR* szPath );

//==============================================================================
//
// Get maximum mipmap counts with given size and format
//
//==============================================================================
int NWGetMaxMipmapLevels( NW_IMG_FMT format, int width, UINT height );

//==============================================================================
///
/// Format big numbers by inserting comma
///
//==============================================================================
void NW4FormatNumberStringWithComma( WCHAR *szTempStr, int maxStrLen, __int64 number );

//==============================================================================
///
/// Get file name from path
///
//==============================================================================
std::wstring NWGetFileNameFromPath( const WCHAR *szPath );

//==============================================================================
///
/// Get directory name from path
///
//==============================================================================
std::wstring NWGetDirectoryNameFromPath( const WCHAR *szPath );

//==============================================================================
///
/// Get resource content as buffer.
// ( Caller is responsible for freeing the memory returned by this function )
///
//==============================================================================
unsigned char* NWGetResourceFileContent( HMODULE hModule, int resourceID,
                                         const WCHAR* szResourceType,
                                         int *pSizeOfFile );

//==============================================================================
//
// Find next power of 2
//
//==============================================================================
int NWFindNextPowerOf2( int val );


//==============================================================================
//
// Compute image data size
//
//==============================================================================
__int64 NWComputeImageDataSize( NW_IMG_FMT format, NW_IMG_TYPE type,
                            int width, int height, int mipMap );


//==============================================================================
//
// Get directory name from path
//
//==============================================================================
std::wstring NWGetDirectoryName( const std::wstring& pathName );

//==============================================================================
//
// Get procedure name
//
//==============================================================================
void* NWGetProcAddress( HMODULE hModule, const char *szProcName );

//==============================================================================
///
/// NWLogOutput
///
//==============================================================================
void NWLogOutput( const WCHAR* szMessage,
                    ... );

#define NW_LOG NWLogOutput

//==============================================================================
//
// Determine if the file is a known ascii intermediate file
//    It determines it by checking file extension
//
//==============================================================================
bool NWIsIntermediateAsciiFile( const WCHAR* filename );

//==============================================================================
//
// Determine if the file is a known binary intermediate file
//    It determines it by checking file extension
//
//==============================================================================
bool NWIsIntermediateBinaryFile( const WCHAR* filename );

//==============================================================================
//
// Determine if the file is a known intermediate file
//    It determines it by checking file extension
//
//==============================================================================
bool NWIsIntermediateFile( const WCHAR* filename, const WCHAR** extensionArray, int extensionArrayLength);

//==============================================================================
//
// Determine the number of channels from the image format
//
//==============================================================================
int NWGetChannelCountFromImageFormat(NW_IMG_FMT format);

//==============================================================================
//
// Lock DLL from unloading
//
//==============================================================================
void NWShellLockModule();

//==============================================================================
//
// Release Lock DLL from unloading
//
//==============================================================================
void NWShellUnlockModule();

//==============================================================================
//
// Wait until lock is free
//
//==============================================================================
void NWShellWaitUnlockModule();

//==============================================================================
//
// Base64 文字列からバイナリへ変換
//
//==============================================================================
std::unique_ptr<unsigned char[]> NWDecodeBase64(const unsigned char* srcBuf, int srcSize, int& dstSize);

#endif // End of __ShellExtension_Utility_H__

