﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "ShellExtension_PCH.h"
#include "ShellExtension_Utility.h"
#include "ShellExtension_Type.h"
#include "ShellExtension_Texture.h"
#include "ShellExtension_Manager.h"
#include "ShellExtension_SyncObject.h"
#include "ImageLoader/ShellExtension_ImageLoader.h"
#include "ImageLoader/ShellExtension_CTexLoader.h"
#include "ImageLoader/ShellExtension_FTXLoader.h"
#include "ImageLoader/ShellExtension_TGALoader.h"
#include "ImageLoader/ShellExtension_PSDLoader.h"
#include "ImageLoader/ShellExtension_AILoader.h"

bool global_GammaLookupTableInitialized = false;
unsigned char global_GammaLookupTable[256] = { 0 };
unsigned char global_PassThroughLookupTable[256] = { 0 };

//==============================================================================
//
// Implementation of CNWTexture
//
//==============================================================================
CNWTexture::CNWTexture() :
    m_width(0),
    m_height(0),
    m_depth(0),
    m_numMipmaps(0),
    m_bUseAllMipmaps(false),
    m_bHasAlpha(false),
    m_textureType(NW_IMG_TYPE_NONE),
    m_format(NW_IMG_FMT_NONE),
    m_previewWidth(0),
    m_previewHeight(0),
    m_textureDataSize(0),
    m_bTGA(false),
    m_TGAType(NW_TGA_NONE),
    m_bPSD(false),
    m_bHasExportSetting(false),
    m_bLoadPreviewFailed(false),
    m_bForceNormalMap(false),
    m_bForceUseComputedZ(false),
    m_bHintNormalMap(false),
    m_layerComps(NULL),
    m_layerCompsSize(0),
    m_pPreviewBitmap(NULL),
    m_pPreviewColorBitmap(NULL),
    m_pPreviewAlphaBitmap(NULL),
    m_pPreviewSmallBitmap(NULL),
    m_pPreviewColorSmallBitmap(NULL),
    m_pPreviewAlphaSmallBitmap(NULL),
    m_bForNW4F(false),
    m_bForNW4C(false),
    m_bShowMipmapCount(false),
    m_pOriginalImageFieldInfo(NULL)
{
    InitializeGammaLookupTable();

    m_linearFlags[0] = false;
    m_linearFlags[1] = false;
    m_linearFlags[2] = false;
    m_linearFlags[3] = false;

    m_UILinearFlags[0] = false;
    m_UILinearFlags[1] = false;
    m_UILinearFlags[2] = false;
    m_UILinearFlags[3] = false;
} // End of Constructor for CNWTexture


//------------------------------------------------------------------------------
CNWTexture::~CNWTexture()
{
    m_width           = 0;
    m_height          = 0;
    m_numMipmaps      = 0;
    m_previewWidth    = 0;
    m_previewHeight   = 0;
    m_textureDataSize = 0;

    Destroy();
} // End of Destructor for CNWTexture


//------------------------------------------------------------------------------
// Initializes the gamma correction lookup table.
//
// - This approach is better than hard-coded values because the power can
//   be changed much more easily than with offline generated values array.
//------------------------------------------------------------------------------
void CNWTexture::InitializeGammaLookupTable()
{
    // ensure initialization is done only once
    if (global_GammaLookupTableInitialized)
        return;

    for (int i = 0; i < sizeof(global_GammaLookupTable) / sizeof(global_GammaLookupTable[0]); i++)
    {
        // GAMMA_CORRECTION_POWER define is located in ShellExtension_Texture.h file
        global_GammaLookupTable[i] = (unsigned char)(pow((float)i / 255.0f, 1.0f / GAMMA_CORRECTION_POWER) * 255.0f);
        global_PassThroughLookupTable[i] = (unsigned char)i;
    }

    global_GammaLookupTableInitialized = true;
}

//------------------------------------------------------------------------------
// Set texture description
//
// - Width and Height are each face's width and height ( if cubemap )
//------------------------------------------------------------------------------
void CNWTexture::SetDescription( NW_IMG_TYPE texType,
                                 NW_IMG_FMT format,
                                 NW_IMG_FMT paletteFormat,
                                 int width, int height, int depth,
                                 int numMipmaps )
{
    SetDescription(texType, format, paletteFormat,
                   width, height, width, height,
                   depth, numMipmaps);
}


//------------------------------------------------------------------------------
// Set texture description
//
// - Width and Height are each face's width and height ( if cubemap )
//------------------------------------------------------------------------------
void CNWTexture::SetDescription( NW_IMG_TYPE texType,
                                 NW_IMG_FMT format,
                                 NW_IMG_FMT paletteFormat,
                                 int width, int height,
                                 int previewWidth, int previewHeight,
                                 int depth, int numMipmaps )
{
    Destroy();

    m_textureType = texType;
    m_typeName    = NWGetTextureTypeNameW(m_textureType);
    m_format      = format;
    m_formatName  = NWGetTextureFormatNameW(m_format);
    m_paletteFormat = paletteFormat;
    if (paletteFormat != NW_IMG_FMT_NONE)
    {
        m_paletteFormatName = NWGetTextureFormatNameW(m_paletteFormat);
    }
    switch(texType)
    {
    default:
        m_width       = width;
        m_height      = height;
        m_depth       = depth;
        m_count       = 1;
        break;
    case NW_IMG_TYPE_1D_ARRAY:
        m_width       = width;
        m_height      = m_depth =  1;
        m_count       = depth;
        break;
    case NW_IMG_TYPE_2D_ARRAY:
        m_width       = width;
        m_height      = height;
        m_depth       = 1;
        m_count       = depth;
        break;
    case NW_IMG_TYPE_CUBE_ARRAY:
        m_width = width;
        m_height = height;
        m_depth = 6;
        m_count = depth / 6;
        break;
    }

    m_numMipmaps  = numMipmaps;

    int maxMipCount = NWGetMaxMipmapLevels(m_format,m_width,m_height);
    if (maxMipCount<=m_numMipmaps)
        m_bUseAllMipmaps = true;
    else
        m_bUseAllMipmaps = false;

    m_bHasAlpha  = NWTextureFormatHasAlpha(m_format, paletteFormat);

    int wd = previewWidth;
    int ht = previewHeight;

    // Some format requires multiple of 4
    // PSD ファイルの場合、エクスポートプラグインで圧縮フォーマットが指定されても
    // 元データは未圧縮の jpeg データであるため、ここでサイズが変更されると
    // プレビュー画像のコピーで失敗してしまう。
    // その不具合を回避するため、PSD ファイルの場合のみ圧縮フォーマットでも
    // 画像サイズの調節を行わないようにする。
    if (!m_bPSD)
    {
        if ( (m_format==NW_IMG_FMT_R_CMPR) )
        {
            wd = ((wd+3) / 4) * 4;
            ht = ((ht+3) / 4) * 4;
        } // End if
    } // End if

    if ( (m_textureType==NW_IMG_TYPE_CUBE) ||
         (m_textureType==NW_IMG_TYPE_CUBE_ARRAY) )
    {
        m_previewWidth  = wd * 4;
        m_previewHeight = ht * 3;
    } // End if
    else
    {
        m_previewWidth  = wd;
        m_previewHeight = ht;
    } // End else

    // プレビュー用の表示領域なので大きい方のサイズで正方形でもつ
    if (m_previewWidth>m_previewHeight)
        m_previewHeight = m_previewWidth;
    if (m_previewHeight>m_previewWidth)
        m_previewWidth = m_previewHeight;

} // End of SetDescription for CNWTexture


//------------------------------------------------------------------------------
// CompressType
//------------------------------------------------------------------------------
void CNWTexture::SetCompressTypeText( const WCHAR* compressType )
{
    m_compressTypeText = (compressType != NULL) ? compressType : L"";

    if (_wcsicmp(m_compressTypeText.c_str(),L"Fast")==0)
    {
        m_compressTypeTextShort = L"F";
    }
    else if (_wcsicmp(m_compressTypeText.c_str(),L"Medium")==0)
    {
        m_compressTypeTextShort = L"M";
    }
    else if (_wcsicmp(m_compressTypeText.c_str(),L"Slow")==0)
    {
        m_compressTypeTextShort = L"S";
    }
    else if (_wcsicmp(m_compressTypeText.c_str(),L"Fast Perceptual")==0)
    {
        m_compressTypeTextShort = L"FP";
    }
    else if (_wcsicmp(m_compressTypeText.c_str(),L"Medium Perceptual")==0)
    {
        m_compressTypeTextShort = L"MP";
    }
    else if (_wcsicmp(m_compressTypeText.c_str(),L"Slow Perceptual")==0)
    {
        m_compressTypeTextShort = L"SP";
    }
    else if (_wcsicmp(m_compressTypeText.c_str(),L"Fast Improved")==0)
    {
        m_compressTypeTextShort = L"FI";
    }
    else if (_wcsicmp(m_compressTypeText.c_str(),L"Medium Improved")==0)
    {
        m_compressTypeTextShort = L"MI";
    }
} // End of SetCompressTypeText for CNWTexture


//------------------------------------------------------------------------------
// Create preview
//------------------------------------------------------------------------------
bool CNWTexture::CreatePreviewImage()
{
    if (m_previewWidth<=0)
        return false;

    if (m_previewHeight<=0)
        return false;

    m_pPreviewBitmap = new Gdiplus::Bitmap(m_previewWidth,m_previewHeight,PixelFormat32bppARGB);
    if (m_pPreviewBitmap==NULL)
    {
        return false;
    } // End if

    //If the image type is array, create preview bitmap for small icon
    switch( m_textureType ){
    case NW_IMG_TYPE_1D_ARRAY:
    case NW_IMG_TYPE_2D_ARRAY:
    case NW_IMG_TYPE_CUBE_ARRAY:
        m_pPreviewSmallBitmap = new Gdiplus::Bitmap(m_previewWidth,m_previewHeight,PixelFormat32bppARGB);

        if(m_pPreviewSmallBitmap == NULL)
            return false;
    }

    return true;
} // End of CreatePreviewImage for CNWTexture


//------------------------------------------------------------------------------
void CNWTexture::Destroy()
{
    if (m_layerComps!=NULL)
    {
        delete[] m_layerComps;
        m_layerComps = NULL;
        m_layerCompsSize = 0;
    } // End if

    if (m_pPreviewBitmap!=NULL)
    {
        delete m_pPreviewBitmap;
        m_pPreviewBitmap = NULL;
    } // End if

    if (m_pPreviewColorBitmap!=NULL)
    {
        delete m_pPreviewColorBitmap;
        m_pPreviewColorBitmap = NULL;
    } // End if

    if (m_pPreviewAlphaBitmap!=NULL)
    {
        delete m_pPreviewAlphaBitmap;
        m_pPreviewAlphaBitmap = NULL;
    } // End if
    if (m_pPreviewSmallBitmap!=NULL)
    {
        delete m_pPreviewSmallBitmap;
        m_pPreviewSmallBitmap = NULL;
    } // End if
    if (m_pPreviewColorSmallBitmap!=NULL)
    {
        delete m_pPreviewColorSmallBitmap;
        m_pPreviewColorSmallBitmap = NULL;
    } // End if
    if (m_pPreviewAlphaSmallBitmap!=NULL)
    {
        delete m_pPreviewAlphaSmallBitmap;
        m_pPreviewAlphaSmallBitmap = NULL;
    } // End if

} // End of Destroy for CNWTexture

bool CNWTexture::CreateColorMap( Gdiplus::BitmapData &srcBmpData, OriginalImageFormat originalFmt, Gdiplus::Rect drawRect )
{
    return CreateColorMapImpl(&m_pPreviewColorBitmap,srcBmpData, drawRect, originalFmt);
}
bool CNWTexture::CreateAlphaMap( Gdiplus::BitmapData &srcBmpData, OriginalImageFormat originalFmt, Gdiplus::Rect drawRect )
{
    return CreateAlphaMapImpl(&m_pPreviewAlphaBitmap,srcBmpData, drawRect, originalFmt);
}
bool CNWTexture::CreateColorAndAlphaMap( Gdiplus::BitmapData &srcBmpData, OriginalImageFormat originalFmt )
{
    return CreateColorAndAlphaMapImpl(&m_pPreviewBitmap,srcBmpData, originalFmt);
}
bool CNWTexture::CreateColorSmallMap( Gdiplus::BitmapData &srcBmpData, OriginalImageFormat originalFmt, Gdiplus::Rect drawRect )
{
    return CreateColorMapImpl(&m_pPreviewColorSmallBitmap,srcBmpData, drawRect, originalFmt);
}
bool CNWTexture::CreateAlphaSmallMap( Gdiplus::BitmapData &srcBmpData, OriginalImageFormat originalFmt, Gdiplus::Rect drawRect )
{
    return CreateAlphaMapImpl(&m_pPreviewAlphaSmallBitmap,srcBmpData, drawRect, originalFmt);
}
bool CNWTexture::CreateColorAndAlphaSmallMap( Gdiplus::BitmapData &srcBmpData, OriginalImageFormat originalFmt )
{
    return CreateColorAndAlphaMapImpl(&m_pPreviewSmallBitmap,srcBmpData, originalFmt);
}

//------------------------------------------------------------------------------
// 正方形サムネ領域にpSrcImgを描画。正方形以外の画像も真ん中合わせで描画
// 必要に応じてpSrcImgのアルファチャンネルの影響を切る
//------------------------------------------------------------------------------
bool DrawImageOnBmp(IN Gdiplus::Bitmap* pSrcImg,const Gdiplus::ImageAttributes* pImageAtt,
                    IN Gdiplus::Rect drawRect, OUT Gdiplus::Bitmap* pDstBmp)
{
    std::unique_ptr<Gdiplus::Graphics> graphicsColor(Gdiplus::Graphics::FromImage( pDstBmp ));

    graphicsColor->SetCompositingMode(Gdiplus::CompositingModeSourceCopy);
    graphicsColor->DrawImage(pSrcImg,drawRect,0,0,pSrcImg->GetWidth(),pSrcImg->GetHeight(),Gdiplus::UnitPixel,pImageAtt,NULL,NULL);

    return true;
}

int ElementIndex(NW_COMP_SEL_ELEMENT element, bool swapGA)
{
    switch (element)
    {
    case NW_COMP_SEL_ELEMENT_R:
        return 0;
    case NW_COMP_SEL_ELEMENT_G:
        return swapGA?3:1;
    case NW_COMP_SEL_ELEMENT_B:
        return 2;
    case NW_COMP_SEL_ELEMENT_A:
        return swapGA?1:3;
    case NW_COMP_SEL_ELEMENT_0:
        return -1;
    case NW_COMP_SEL_ELEMENT_1:
        return 4;
    }
    return -1;
}

void SetBitmap(Gdiplus::Bitmap** ppBitmap, Gdiplus::Bitmap* value)
{
    if (*ppBitmap != nullptr)
    {
        delete *ppBitmap;
    }
    *ppBitmap = value;
}

bool CNWTexture::CreatePreviewBitmap(Gdiplus::Bitmap* srcBmp, Gdiplus::Rect drawRect, OriginalImageFormat originalFmt, bool isSmall, bool isColor, bool isAlpha)
{
    assert(isColor || isAlpha);

    bool onlyAlpha = !isColor && isAlpha;
    if (onlyAlpha && !HasAlpha())
    {
        // 何もしない
        return true;
    }

    auto swapGA = false;
    auto format = GetFormat();
    auto isAstc = false;
    if (NW_IMG_FMT_UNORM_ASTC_4x4 <= format &&
        format <= NW_IMG_FMT_SRGB_ASTC_12x12)
    {
        if (GetHint() == L"normal")
        {
            isAstc = true;
            swapGA = true;
        }
    }
    else
    {
        switch (format)
        {
        case NW_IMG_FMT_GX2_UNORM_4_4:
        case NW_IMG_FMT_GX2_UNORM_8_8:
        case NW_IMG_FMT_GX2_UNORM_BC5:
        case NW_IMG_FMT_UNORM_EAC_11:
            if (GetHint() != L"normal")
            {
                // オリジナルにアルファがある
                switch (originalFmt){
                case ORIGINAL_FORMAT_RGBA8:
                case ORIGINAL_FORMAT_RGBA32F:
                    swapGA = true;
                    break;
                }
            }
            break;
        }
    }

    Gdiplus::ColorMatrix colorMatrix = {};
    for (int i = 0; i < 4; i++)
    {
        int j = (i < 3 || (isColor && isAlpha)) ?
            ElementIndex(m_compSel.sel[onlyAlpha ? 3: i], swapGA) :
            ElementIndex(NW_COMP_SEL_ELEMENT_1, swapGA);
        if (0 <= j)
        {
            colorMatrix.m[j][i] = 1.0f;
        }
    }

    Gdiplus::ImageAttributes imageAtt;
    imageAtt.SetColorMatrix(&colorMatrix);

    std::unique_ptr<Gdiplus::Bitmap> pTargetBitmap(new Gdiplus::Bitmap(GetPreviewWidth(), GetPreviewHeight(), PixelFormat32bppARGB));
    if (!DrawImageOnBmp(srcBmp, &imageAtt,
        drawRect,
        pTargetBitmap.get()))
    {
        return false;
    }

    if (GetHint() == L"normal" && !onlyAlpha)
    {
        if (format == NW_IMG_FMT_GX2_SNORM_8_8 ||
            format == NW_IMG_FMT_GX2_SINT_8_8)
        {
            Gdiplus::Rect rect(0,0,pTargetBitmap->GetWidth(), pTargetBitmap->GetHeight());
            Gdiplus::BitmapData bitmapData;
            pTargetBitmap->LockBits(&rect, Gdiplus::ImageLockModeRead | Gdiplus::ImageLockModeWrite, PixelFormat32bppARGB, &bitmapData);
            auto startPtr = (UINT32*)bitmapData.Scan0;
            auto height = bitmapData.Height;
            auto width = bitmapData.Width;
            auto stride = bitmapData.Stride;

            for (unsigned int i = 0; i < height; i++)
            {
                auto linePtr = startPtr + i*stride/sizeof(UINT32);

                for (unsigned int j = 0; j < width; j++)
                {
                    auto currentPtr = linePtr + j;
                    auto colorPtr = (byte*) currentPtr;

                    auto sr = colorPtr[2]/255.0f;
                    auto sg = colorPtr[1]/255.0f;
                    auto sb = sqrt(1.0f - (std::min)(sr*sr + sg*sg, 1.0f));

                    colorPtr[2] = (byte) (128 + (int) (127.0f*sr));
                    colorPtr[1] = (byte) (128 + (int) (127.0f*sg));
                    colorPtr[0] = (byte) (128 + (int) (127.0f*sb));
                }
            }

            pTargetBitmap->UnlockBits(&bitmapData);
        }
        else if (format == NW_IMG_FMT_GX2_SNORM_BC5 || isAstc)
        {
            Gdiplus::Rect rect(0,0,pTargetBitmap->GetWidth(), pTargetBitmap->GetHeight());
            Gdiplus::BitmapData bitmapData;
            pTargetBitmap->LockBits(&rect, Gdiplus::ImageLockModeRead | Gdiplus::ImageLockModeWrite, PixelFormat32bppARGB, &bitmapData);
            auto startPtr = (UINT32*)bitmapData.Scan0;
            auto height = bitmapData.Height;
            auto width = bitmapData.Width;
            auto stride = bitmapData.Stride;

            for (unsigned int i = 0; i< height; i++)
            {
                auto linePtr = startPtr + i*stride/sizeof(UINT32);

                for (unsigned int j = 0; j < width; j++)
                {
                    auto currentPtr = linePtr + j;

                    auto colorPtr = (byte*) currentPtr;

                    auto sr = (std::min)((std::max)((((int) colorPtr[2]) - 128)/127.0f, -1.0f), +1.0f);
                    auto sg = (std::min)((std::max)((((int) colorPtr[1]) - 128)/127.0f, -1.0f), +1.0f);
                    auto sb = sqrt(1.0f - (std::min)(sr*sr + sg*sg, 1.0f));

                    colorPtr[0] = (byte) (std::min)((std::max)(128 + (int) (127.0f*sb), 0), 255);
                }
            }

            pTargetBitmap->UnlockBits(&bitmapData);
        }

    }

    if (isColor && isAlpha)
    {
        if (isSmall)
        {
            SetBitmap(&m_pPreviewSmallBitmap, pTargetBitmap.release());
        }
        else
        {
            SetBitmap(&m_pPreviewBitmap, pTargetBitmap.release());
        }
    }
    else if (isColor)
    {
        if (isSmall)
        {
            SetBitmap(&m_pPreviewColorSmallBitmap, pTargetBitmap.release());
        }
        else
        {
            SetBitmap(&m_pPreviewColorBitmap, pTargetBitmap.release());
        }
    }
    else if (isAlpha)
    {
        if (isSmall)
        {
            SetBitmap(&m_pPreviewAlphaSmallBitmap, pTargetBitmap.release());
        }
        else
        {
            SetBitmap(&m_pPreviewAlphaBitmap, pTargetBitmap.release());
        }
    }

    return true;
}

//------------------------------------------------------------------------------
// Load from .ctex file
//
// - If bLoadPreview is true, create preview image for icon
//   If bLoadPreview is false, just load information ( format, size etc )
//------------------------------------------------------------------------------
bool CNWTexture::Load( const WCHAR *szFilePath,
                       bool bLoadPreview )
{
    bool bResult = false;

    if (m_bLoadPreviewFailed)
    {
        return false;
    } // End if

    // Load texture loading critical section
    CNWSingleLock lock(CShellExtensionManager::Instance()->GetTextureLoadingCS());

    try
    {
        std::wstring extension = NWGetExtension(szFilePath);
#if defined NW_FOR_CTEX
        if (_wcsicmp(extension.c_str(),L".ctex")==0)
        {
            CNWCTexLoader loader;
            bResult = loader.Load(this,szFilePath,bLoadPreview);
            m_bForNW4C = true;
            m_bForNW4F = false;

            m_bShowMipmapCount = true;
        } // End if
#elif defined NW_FOR_FTX
        if ( (_wcsicmp(extension.c_str(),L".ftxa")==0) ||
                  (_wcsicmp(extension.c_str(),L".ftxb")==0) )
        {
                CNWFTXLoader loader;
                bResult = loader.Load(this,szFilePath,bLoadPreview);
                m_bForNW4C = false;
                m_bForNW4F = true;

                m_bShowMipmapCount = true;
        } // End if
#elif defined NW_FOR_TGA
        if (_wcsicmp(extension.c_str(),L".tga")==0)
        {
            CNWTGALoader loader;
            bResult = loader.Load(this,szFilePath,bLoadPreview);
        } // End if
#elif defined NW_FOR_PSD
        if (_wcsicmp(extension.c_str(),L".psd")==0)
        {
            CNWPSDLoader loader;
            bResult = loader.Load(this,szFilePath,bLoadPreview);
        } // End if
#elif defined NW_FOR_AI
        if (_wcsicmp(extension.c_str(), L".ai") == 0)
        {
            CNWAILoader loader;
            bResult = loader.Load(this, szFilePath, bLoadPreview);
        } // End if
#elif defined NW_FOR_EPS
        if (_wcsicmp(extension.c_str(), L".eps") == 0)
        {
            // AI と同じようにサムネイルが埋め込まれている
            CNWAILoader loader;
            bResult = loader.Load(this, szFilePath, bLoadPreview);
        } // End if
#elif defined NW_FOR_ESET || defined NW_FOR_FLYT || defined NW_FOR_THUMBS
        // 拡張子のチェックは不要
        {
            Destroy();

            auto exts = {L".png", L".jpg"};
            for (auto ext : exts)
            {
                std::wstring thumbnailFilePath = NWGetDirectoryNameFromPath(szFilePath) + L"\\.NwThumbs\\" + NWGetFileNameFromPath(szFilePath) + ext;

                auto bitmap = Gdiplus::Bitmap::FromFile(thumbnailFilePath.c_str());
                if (bitmap != nullptr)
                {
                    // FromFile したものをそのまま使うとファイルが開かれたままになるので、DrawImage を使ってコピーする。
                    // FromFile の代わりに IStream を使う方法もあるが、IStream の開放のタイミングが難しい。
                    if (bitmap->GetLastStatus() == Gdiplus::Status::Ok)
                    {
                        m_pPreviewBitmap = new Gdiplus::Bitmap(bitmap->GetWidth(), bitmap->GetHeight(), PixelFormat32bppARGB);
                        if (m_pPreviewBitmap != nullptr && m_pPreviewBitmap->GetLastStatus() == Gdiplus::Status::Ok)
                        {
                            Gdiplus::Graphics g(m_pPreviewBitmap);
                            g.DrawImage(bitmap, 0, 0, m_pPreviewBitmap->GetWidth(), m_pPreviewBitmap->GetHeight());
                        }
                        else if (m_pPreviewBitmap != nullptr)
                        {
                            delete m_pPreviewBitmap;
                            m_pPreviewBitmap = nullptr;
                        }
                    }

                    delete bitmap;
                }

                if (m_pPreviewBitmap != nullptr && m_pPreviewBitmap->GetLastStatus() == 0)
                {
                    bResult = true;
                    break;
                }
                else if (m_pPreviewBitmap != nullptr)
                {
                    delete m_pPreviewBitmap;
                    m_pPreviewBitmap = nullptr;
                }
            }
        } // End if
#endif
    }
    catch (...)
    {
        bResult = false;
    }

    if (bResult==false)
    {
        if (bLoadPreview)
        {
            m_bLoadPreviewFailed = false;
        } // End if
    } // End if

    return bResult;
} // End of Load for CNWTexture

//------------------------------------------------------------------------------
// Create Color Preview Map
//------------------------------------------------------------------------------
bool CNWTexture::CreateColorMapImpl( Gdiplus::Bitmap** ppPreviewColorBitmap, Gdiplus::BitmapData &srcBmpData,
                                     Gdiplus::Rect drawRect, OriginalImageFormat originalFmt )const
{
    if (*ppPreviewColorBitmap!=NULL)
    {
        delete *ppPreviewColorBitmap;
        *ppPreviewColorBitmap = NULL;
    } // End if

    // To be implemented, do actual loading from file
    // but for now, just create simple image for testing
    *ppPreviewColorBitmap = new Gdiplus::Bitmap(m_previewWidth,m_previewHeight,PixelFormat32bppARGB);
    if (*ppPreviewColorBitmap==NULL)
    {
        return false;
    } // End if

    // Lock bits to prepare copy
    Gdiplus::Rect lockRect(0,0,m_previewWidth,m_previewHeight);
    Gdiplus::BitmapData destBmpData;
    if ((*ppPreviewColorBitmap)->LockBits(&lockRect,Gdiplus::ImageLockModeWrite,
                                        (*ppPreviewColorBitmap)->GetPixelFormat(),
                                        &destBmpData)!=Gdiplus::Ok)
    {
        return false;
    } // End if

    unsigned char *pSrcData  = (unsigned char*)srcBmpData.Scan0;
    unsigned char *pDestData = (unsigned char*)destBmpData.Scan0;

    int numLines       = m_previewHeight;
    int numPixelsInRow = m_previewWidth;
    int numPixels      = numLines * numPixelsInRow;

    int i;

    unsigned char colorZERO = 0;

    NW_IMG_FMT imageFormat = GetFormat();

    bool isSRGB =
        imageFormat == NW_IMG_FMT_GX2_SRGB_8_8_8_8 ||
        imageFormat == NW_IMG_FMT_GX2_SRGB_BC1 ||
        imageFormat == NW_IMG_FMT_GX2_SRGB_BC2 ||
        imageFormat == NW_IMG_FMT_GX2_SRGB_BC3;

    NW_COMP_SEL compSel = m_compSel;

    if ( imageFormat == NW_IMG_FMT_A4 ||
         imageFormat == NW_IMG_FMT_A8 )
    {
        // If texture has only alpha, fill RGB as white
        memset(pDestData,0xFFFFFFFF,numPixels*4);
    } // End if
    else if (IsForceNormalMapOn())
    {
        // 2015/03/18時点のコードでは、ここに処理が入ってくることはない

        //#pragma omp parallel for
        for (i=0;i<numPixels;i++)
        {
            int x = i % numPixelsInRow;
            int y = i / numPixelsInRow;

            unsigned char *pSrcPixel  = pSrcData  + y * srcBmpData.Stride  + x * 4;
            unsigned char *pDestPixel = pDestData + y * destBmpData.Stride + x * 4;

            float nx = ((float)pSrcPixel[2] / 255.0f) * 2.0f - 0.5f;
            float ny = ((float)pSrcPixel[1] / 255.0f) * 2.0f - 0.5f;
            float nz = sqrtf(max(0.0f,1.0f-(nx*nx+ny*ny)));

            // Copy alpha to all rgb ( note that for GDI, format is BGRA, so swap R and B )
            unsigned char colSel[6] =
            {
                pSrcPixel[2],
                pSrcPixel[1],
                (unsigned char)((nz+1.0f) * 0.5f * 255.0f),
                pSrcPixel[3],
                colorZERO,
                0xFF
            };

            pDestPixel[2] = colSel[compSel.sel[0]];
            pDestPixel[1] = colSel[compSel.sel[1]];

            if (m_bForceUseComputedZ)
                pDestPixel[0] = colSel[2];
            else
                pDestPixel[0] = colSel[compSel.sel[2]];

            pDestPixel[3] = 0xFF;

            Sleep(0);
        } // End for
    } // End else if
    else
    {

        bool linear[4];
        for (i = 0; i < 4; i++)
        {
            linear[i] = m_linearFlags[i];
        }

        // オリジナルイメージはそのまま表示する
        if (originalFmt == ORIGINAL_FORMAT_RGB8  ||
            originalFmt == ORIGINAL_FORMAT_RGBA8 ||
            originalFmt == ORIGINAL_FORMAT_RGB32F||
            originalFmt == ORIGINAL_FORMAT_RGBA32F)
        {
            linear[0] = linear[1] = linear[2] = false;
            linear[3] = false;
        }

        unsigned char* compSelChannelLookupTable[3] =
        {
            linear[compSel.sel[2]] ? global_GammaLookupTable : global_PassThroughLookupTable,
            linear[compSel.sel[1]] ? global_GammaLookupTable : global_PassThroughLookupTable,
            linear[compSel.sel[0]] ? global_GammaLookupTable : global_PassThroughLookupTable,
        };

        //#pragma omp parallel for
        for (i=0;i<numPixels;i++)
        {
            int x = i % numPixelsInRow;
            int y = i / numPixelsInRow;

            unsigned char *pSrcPixel  = pSrcData  + y * srcBmpData.Stride  + x * 4;
            unsigned char *pDestPixel = pDestData + y * destBmpData.Stride + x * 4;

            pDestPixel[3] = pSrcPixel[3];
            pDestPixel[2] = pSrcPixel[2];
            pDestPixel[1] = pSrcPixel[1];
            pDestPixel[0] = pSrcPixel[0];

            if (isSRGB == false)
            {
                pDestPixel[2] = compSelChannelLookupTable[2][pDestPixel[2]];
                pDestPixel[1] = compSelChannelLookupTable[1][pDestPixel[1]];
                pDestPixel[0] = compSelChannelLookupTable[0][pDestPixel[0]];
            }

            // Copy alpha to all rgb ( note that for GDI, format is BGRA, so swap R and B )
            unsigned char colSel[6] =
            {
                pDestPixel[2],
                pDestPixel[1],
                pDestPixel[0],
                pDestPixel[3],
                colorZERO,
                0xFF
            };
            pDestPixel[2] = colSel[compSel.sel[0]];
            pDestPixel[1] = colSel[compSel.sel[1]];
            pDestPixel[0] = colSel[compSel.sel[2]];
            pDestPixel[3] = colSel[compSel.sel[3]];

            if (m_bHintNormalMap && (m_format==NW_IMG_FMT_GX2_SNORM_8_8 ||
                                     m_format==NW_IMG_FMT_GX2_SINT_8_8) )
            {
                float r = pDestPixel[2] / 255.0f;
                float g = pDestPixel[1] / 255.0f;
                float b = std::sqrtf(1.0f - min(r * r + g * g, 1.0f));

                pDestPixel[0] = (unsigned char)(128 + int(127.0f * b));
                pDestPixel[1] = (unsigned char)(128 + int(127.0f * g));
                pDestPixel[2] = (unsigned char)(128 + int(127.0f * r));
            }// End if
            else if (m_bHintNormalMap && m_format==NW_IMG_FMT_GX2_SNORM_BC5)
            {
                float r = min(max((((int)pDestPixel[2]) - 128) / 127.0f, -1.0f), +1.0f);
                float g = min(max((((int)pDestPixel[1]) - 128) / 127.0f, -1.0f), +1.0f);
                float b = std::sqrtf(1.0f - min(r * r + g * g, 1.0f));

                pDestPixel[0] = (unsigned char)min(max(128 + int(127.0f * b),   0), 255);
            }// End else if

            Sleep(0);
        } // End for
    } // End else

    if(drawRect.Width != drawRect.Height)
    {
        int stRectW = (drawRect.X == 0)?drawRect.Width:drawRect.X;
        int stRectH = (drawRect.Y == 0)?drawRect.Height:drawRect.Y;
        int endRectX = (drawRect.X == 0)?0:drawRect.X + drawRect.Width;
        int endRectY = (drawRect.Y == 0)?0:drawRect.Y + drawRect.Height;

        DrawFillRect(destBmpData, Gdiplus::Color(128,128,128,128), 0, 0, stRectW, stRectH);
        DrawFillRect(destBmpData, Gdiplus::Color(128,128,128,128), endRectX, endRectY, m_previewWidth, m_previewHeight);
    }


    (*ppPreviewColorBitmap)->UnlockBits(&destBmpData);

    return true;
} // End of CreateColorMap for CNWTexture

//------------------------------------------------------------------------------
// Create Alpha Preview Map
//------------------------------------------------------------------------------
bool CNWTexture::CreateAlphaMapImpl( Gdiplus::Bitmap** ppPreviewAlphaBitmap,Gdiplus::BitmapData &srcBmpData,
                                     Gdiplus::Rect drawRect, OriginalImageFormat originalFmt )const
{
    if (*ppPreviewAlphaBitmap!=NULL)
    {
        delete *ppPreviewAlphaBitmap;
        *ppPreviewAlphaBitmap = NULL;
    } // End if

    if (HasAlpha()==false)
        return false;

    // To be implemented, do actual loading from file
    // but for now, just create simple image for testing
    *ppPreviewAlphaBitmap = new Gdiplus::Bitmap(m_previewWidth,m_previewHeight,PixelFormat32bppARGB);
    if (*ppPreviewAlphaBitmap==NULL)
    {
        return false;
    } // End if

    // Lock bits to prepare copy
    Gdiplus::Rect lockRect(0,0, m_previewWidth, m_previewHeight);
    Gdiplus::BitmapData destBmpData;
    if ((*ppPreviewAlphaBitmap)->LockBits(&lockRect,Gdiplus::ImageLockModeWrite,
                                        (*ppPreviewAlphaBitmap)->GetPixelFormat(),
                                        &destBmpData)!=Gdiplus::Ok)
    {
        return false;
    } // End if

    unsigned char *pSrcData  = (unsigned char*)srcBmpData.Scan0;
    unsigned char *pDestData = (unsigned char*)destBmpData.Scan0;

    int numLines       = m_previewHeight;
    int numPixelsInRow = m_previewWidth;
    int numPixels      = numLines * numPixelsInRow;

    int srcAlphaIndex = 3;

    NW_IMG_FMT imageFormat = GetFormat();

    bool isSRGB =
        imageFormat == NW_IMG_FMT_GX2_SRGB_8_8_8_8 ||
        imageFormat == NW_IMG_FMT_GX2_SRGB_BC1 ||
        imageFormat == NW_IMG_FMT_GX2_SRGB_BC2 ||
        imageFormat == NW_IMG_FMT_GX2_SRGB_BC3;

    switch (imageFormat)
    {
        case NW_IMG_FMT_A4 :
        case NW_IMG_FMT_A8 :
            srcAlphaIndex = 0;
            break;
    } // End switch

    NW_COMP_SEL compSel = m_compSel;

    unsigned char colorZERO = 0;

    #if 0
        if (IsSignedFormat())
            colorZERO = 0x80;
    #endif

    int i;
    bool linear[4];
    for (i = 0; i < 4; i++)
    {
        linear[i] = m_linearFlags[i];
    }

    // オリジナルイメージはそのまま表示する
    if (originalFmt == ORIGINAL_FORMAT_RGB8  ||
        originalFmt == ORIGINAL_FORMAT_RGBA8 ||
        originalFmt == ORIGINAL_FORMAT_RGB32F||
        originalFmt == ORIGINAL_FORMAT_RGBA32F)
    {
        linear[0] = linear[1] = linear[2] = false;
        linear[3] = false;
    }

    unsigned char* compSelChannelLookupTable = global_PassThroughLookupTable;
    if (isSRGB == false && linear[compSel.sel[3]])
        compSelChannelLookupTable = global_GammaLookupTable;

    //#pragma omp parallel for
    for (i=0;i<numPixels;i++)
    {
        int x = i % numPixelsInRow;
        int y = i / numPixelsInRow;

        unsigned char *pSrcPixel  = pSrcData  + y * srcBmpData.Stride  + x * 4;
        unsigned char *pDestPixel = pDestData + y * destBmpData.Stride + x * 4;

        // Copy alpha to all rgb ( note that for GDI, format is BGRA, so swap R and B )
        unsigned char colSel[6] =
        {
            pSrcPixel[2],
            pSrcPixel[1],
            pSrcPixel[0],
            pSrcPixel[3],
            colorZERO,
            0xFF
        };

        // Copy alpha to all rgb
        unsigned char alpha = compSelChannelLookupTable[colSel[compSel.sel[3]]];
        pDestPixel[0] = alpha;
        pDestPixel[1] = alpha;
        pDestPixel[2] = alpha;
        pDestPixel[3] = 255;

        Sleep(0);
    } // End for


    if(drawRect.Width != drawRect.Height)
    {
        int stRectW = (drawRect.X == 0)?drawRect.Width:drawRect.X;
        int stRectH = (drawRect.Y == 0)?drawRect.Height:drawRect.Y;
        int endRectX = (drawRect.X == 0)?0:drawRect.X + drawRect.Width;
        int endRectY = (drawRect.Y == 0)?0:drawRect.Y + drawRect.Height;

        DrawFillRect(destBmpData, Gdiplus::Color(128,128,128,128), 0, 0, stRectW, stRectH);
        DrawFillRect(destBmpData, Gdiplus::Color(128,128,128,128), endRectX, endRectY, m_previewWidth, m_previewHeight);
    }


    (*ppPreviewAlphaBitmap)->UnlockBits(&destBmpData);

    return true;
} // End of CreateAlphaMap for CNWTexture


//------------------------------------------------------------------------------
// Create Color and Alpha Preview Map
// (Use Conponet Selector Setting)
//------------------------------------------------------------------------------
bool CNWTexture::CreateColorAndAlphaMapImpl( Gdiplus::Bitmap** ppPreviewColorAndAlphaBitmap,Gdiplus::BitmapData &srcBmpData, OriginalImageFormat originalFmt )const
{

    if (*ppPreviewColorAndAlphaBitmap!=NULL)
    {
        delete *ppPreviewColorAndAlphaBitmap;
        *ppPreviewColorAndAlphaBitmap = NULL;
    } // End if

    // To be implemented, do actual loading from file
    // but for now, just create simple image for testing
    *ppPreviewColorAndAlphaBitmap = new Gdiplus::Bitmap(m_previewWidth,m_previewHeight,PixelFormat32bppARGB);
    if (*ppPreviewColorAndAlphaBitmap==NULL)
    {
        return false;
    } // End if

    // Lock bits to prepare copy
    Gdiplus::Rect lockRect(0,0, m_previewWidth, m_previewHeight);
    Gdiplus::BitmapData destBmpData;
    if ((*ppPreviewColorAndAlphaBitmap)->LockBits(&lockRect,Gdiplus::ImageLockModeWrite,
                                        (*ppPreviewColorAndAlphaBitmap)->GetPixelFormat(),
                                        &destBmpData)!=Gdiplus::Ok)
    {
        return false;
    } // End if

    unsigned char *pSrcData  = (unsigned char*)srcBmpData.Scan0;
    unsigned char *pDestData = (unsigned char*)destBmpData.Scan0;

    int numLines       = m_previewHeight;
    int numPixelsInRow = m_previewWidth;
    int numPixels      = numLines * numPixelsInRow;

    int i;

    unsigned char colorZERO = 0;

    NW_IMG_FMT imageFormat = GetFormat();

    bool isSRGB =
        imageFormat == NW_IMG_FMT_GX2_SRGB_8_8_8_8 ||
        imageFormat == NW_IMG_FMT_GX2_SRGB_BC1 ||
        imageFormat == NW_IMG_FMT_GX2_SRGB_BC2 ||
        imageFormat == NW_IMG_FMT_GX2_SRGB_BC3;

    NW_COMP_SEL compSel = m_compSel;

    if ( imageFormat == NW_IMG_FMT_A4 ||
         imageFormat == NW_IMG_FMT_A8 )
    {
        // If texture has only alpha, fill RGB as white
        memset(pDestData,0xFFFFFFFF,numPixels*4);
    } // End if
    else if (IsForceNormalMapOn())
    {
        // 2015/03/18時点のコードでは、ここに処理が入ってくることはない

        //#pragma omp parallel for
        for (i=0;i<numPixels;i++)
        {
            int x = i % numPixelsInRow;
            int y = i / numPixelsInRow;

            unsigned char *pSrcPixel  = pSrcData  + y * srcBmpData.Stride  + x * 4;
            unsigned char *pDestPixel = pDestData + y * destBmpData.Stride + x * 4;

            float nx = ((float)pSrcPixel[2] / 255.0f) * 2.0f - 0.5f;
            float ny = ((float)pSrcPixel[1] / 255.0f) * 2.0f - 0.5f;
            float nz = sqrtf(max(0.0f,1.0f-(nx*nx+ny*ny)));

            // Copy alpha to all rgb ( note that for GDI, format is BGRA, so swap R and B )
            unsigned char colSel[6] =
            {
                pSrcPixel[2],
                pSrcPixel[1],
                (unsigned char)((nz+1.0f) * 0.5f * 255.0f),
                pSrcPixel[3],
                colorZERO,
                0xFF
            };

            pDestPixel[2] = colSel[compSel.sel[0]];
            pDestPixel[1] = colSel[compSel.sel[1]];

            if (m_bForceUseComputedZ)
                pDestPixel[0] = colSel[2];
            else
                pDestPixel[0] = colSel[compSel.sel[2]];

            pDestPixel[3] = 0xFF;

            Sleep(0);
        } // End for
    } // End else if
    else
    {
        bool linear[4];
        for (i = 0; i < 4; i++)
        {
            linear[i] = m_linearFlags[i];
        }

        // オリジナルイメージはそのまま表示する
        if (originalFmt == ORIGINAL_FORMAT_RGB8  ||
            originalFmt == ORIGINAL_FORMAT_RGBA8 ||
            originalFmt == ORIGINAL_FORMAT_RGB32F||
            originalFmt == ORIGINAL_FORMAT_RGBA32F)
        {
            linear[0] = linear[1] = linear[2] = false;
            linear[3] = false;
        }

        unsigned char* compSelChannelLookupTable[4] =
        {
            linear[compSel.sel[2]] ? global_GammaLookupTable : global_PassThroughLookupTable,
            linear[compSel.sel[1]] ? global_GammaLookupTable : global_PassThroughLookupTable,
            linear[compSel.sel[0]] ? global_GammaLookupTable : global_PassThroughLookupTable,
            linear[compSel.sel[3]] ? global_GammaLookupTable : global_PassThroughLookupTable,
        };


        //#pragma omp parallel for
        for (i=0;i<numPixels;i++)
        {
            int x = i % numPixelsInRow;
            int y = i / numPixelsInRow;

            unsigned char *pSrcPixel  = pSrcData  + y * srcBmpData.Stride  + x * 4;
            unsigned char *pDestPixel = pDestData + y * destBmpData.Stride + x * 4;

            pDestPixel[3] = pSrcPixel[3];
            pDestPixel[2] = pSrcPixel[2];
            pDestPixel[1] = pSrcPixel[1];
            pDestPixel[0] = pSrcPixel[0];

            if (isSRGB == false)
            {
                pDestPixel[2] = compSelChannelLookupTable[2][pDestPixel[2]];
                pDestPixel[1] = compSelChannelLookupTable[1][pDestPixel[1]];
                pDestPixel[0] = compSelChannelLookupTable[0][pDestPixel[0]];
                pDestPixel[3] = compSelChannelLookupTable[3][pDestPixel[3]];
            }

            // Copy alpha to all rgb ( note that for GDI, format is BGRA, so swap R and B )
            unsigned char colSel[6] =
            {
                pDestPixel[2],
                pDestPixel[1],
                pDestPixel[0],
                pDestPixel[3],
                colorZERO,
                0xFF
            };
            pDestPixel[2] = colSel[compSel.sel[0]];
            pDestPixel[1] = colSel[compSel.sel[1]];
            pDestPixel[0] = colSel[compSel.sel[2]];
            pDestPixel[3] = colSel[compSel.sel[3]];

            if (m_bHintNormalMap && (m_format==NW_IMG_FMT_GX2_SNORM_8_8 ||
                                     m_format==NW_IMG_FMT_GX2_SINT_8_8) )
            {
                float r = pDestPixel[2] / 255.0f;
                float g = pDestPixel[1] / 255.0f;
                float b = std::sqrtf(1.0f - min(r * r + g * g, 1.0f));

                pDestPixel[0] = (unsigned char)(128 + int(127.0f * b));
                pDestPixel[1] = (unsigned char)(128 + int(127.0f * g));
                pDestPixel[2] = (unsigned char)(128 + int(127.0f * r));
            }// End if
            else if (m_bHintNormalMap && m_format==NW_IMG_FMT_GX2_SNORM_BC5)
            {
                float r = min(max((((int)pDestPixel[2]) - 128) / 127.0f, -1.0f), +1.0f);
                float g = min(max((((int)pDestPixel[1]) - 128) / 127.0f, -1.0f), +1.0f);
                float b = std::sqrtf(1.0f - min(r * r + g * g, 1.0f));

                pDestPixel[0] = (unsigned char)min(max(128 + int(127.0f * b),   0), 255);
            }// End else if

            Sleep(0);
        } // End for
    } // End else

    (*ppPreviewColorAndAlphaBitmap)->UnlockBits(&destBmpData);

    return true;
} // End of CreateColorMap for CNWTexture

bool CNWTexture::DrawFillRect(Gdiplus::BitmapData bmpData, Gdiplus::Color color, int stX, int stY, int edX, int edY)const
{
    unsigned char *pData  = (unsigned char*)bmpData.Scan0;

    for (int x=stX; x<edX; x++)
    {
        for (int y=stY ;y<edY; y++)
        {
            unsigned char *pDestPixel = pData + y * bmpData.Stride + x * 4;

            // Copy alpha to all rgb
            pDestPixel[0] = color.GetR();
            pDestPixel[1] = color.GetG();
            pDestPixel[2] = color.GetB();
            pDestPixel[3] = 255;

        } // End for
    }

    return true;
}


//------------------------------------------------------------------------------
// Mipmap
//------------------------------------------------------------------------------
bool CNWTexture::HasMipmaps() const
{
    bool bHasMipmap = true;

    if ( (IsTGA()) && (TGAType()==NW_TGA_NONE) )
    {
        bHasMipmap = false;
    } // End if
    else if ( (IsPSD()) && (!HasPSDExportSetting()) )
    {
        if (m_numMipmaps<=1)
            bHasMipmap = false;
    } // End if

    return bHasMipmap;
} // End of HasMipmaps for CNWTexture


//------------------------------------------------------------------------------
// MakeHasAlphaFromAlphaImageWhenBC1
//------------------------------------------------------------------------------
bool CNWTexture::MakeHasAlphaFromAlphaImageWhenBC1(void)
{
    // BC1の場合アルファが全ピクセル255ならアルファなし
    if (
        (GetFormat() == NW_IMG_FMT_GX2_UNORM_BC1) ||
        (GetFormat() == NW_IMG_FMT_GX2_SRGB_BC1) )
    {
        Gdiplus::Bitmap *pPreviewBitmap = GetPreviewAlphaBitmap();
        if (pPreviewBitmap==NULL)
            return false;

        // Lock bits to prepare preview image
        Gdiplus::Rect lockRect(0,0,pPreviewBitmap->GetWidth(),pPreviewBitmap->GetHeight());
        Gdiplus::BitmapData bmpData;
        if (pPreviewBitmap->LockBits(&lockRect,Gdiplus::ImageLockModeWrite,
            pPreviewBitmap->GetPixelFormat(),
            &bmpData)!=Gdiplus::Ok)
        {
            return false;
        } // End if

        m_bHasAlpha = false;

        unsigned int *pDestBuffer = (unsigned int*)bmpData.Scan0;

        int wd = GetWidth();
        int ht = GetHeight();
        for (int y = 0; y < ht; ++y)
        {
            unsigned int* line = pDestBuffer + y * bmpData.Stride / sizeof(unsigned int);

            for (int x = 0; x < wd; ++x, ++line)
            {
                if (*line != 0xFFFFFFFF)
                {
                    m_bHasAlpha = true;
                    break;
                }
            }

            if(m_bHasAlpha)
            {
                break;
            }

            Sleep(0);
        }

        pPreviewBitmap->UnlockBits(&bmpData);

        if(m_bHasAlpha == false)
        {
            if (m_pPreviewAlphaBitmap!=NULL)
            {
                delete m_pPreviewAlphaBitmap;
                m_pPreviewAlphaBitmap = NULL;
            } // End if
        }
    } // End if

    return true;
}


//------------------------------------------------------------------------------
// Compute Each face size
//------------------------------------------------------------------------------
int CNWTexture::ComputeTextureFaceDataSize( int mipLevel )
{
    int textureDataSize = 0;

    if ( (GetFormat()==NW_IMG_FMT_ETC1) ||
         (GetFormat()==NW_IMG_FMT_ETCA) )
    {
        int bytesPerBlock = 8;
        if (GetFormat()==NW_IMG_FMT_ETCA)
            bytesPerBlock = 16;

        int wd = max(4,(m_width >> mipLevel));
        int ht = max(4,(m_height >> mipLevel));

        int numBlocksX    = max(2,wd / 4);
        int numBlocksY    = max(2,ht / 4);

        textureDataSize = (bytesPerBlock * numBlocksX * numBlocksY);
    } // End if
    else if ( (GetFormat()==NW_IMG_FMT_R_CMPR) )
    {
        int bytesPerBlock = 8;
        if (GetFormat()==NW_IMG_FMT_ETCA)
            bytesPerBlock = 16;

        int wd = max(4,(m_width >> mipLevel));
        int ht = max(4,(m_height >> mipLevel));

        int numBlocksX    = max(1,wd / 4);
        int numBlocksY    = max(1,ht / 4);

        textureDataSize = (bytesPerBlock * numBlocksX * numBlocksY);
    } // End else if
    else
    {
        int wd = max(4,(m_width >> mipLevel));
        int ht = max(4,(m_height >> mipLevel));

        int numPixelsX    = max(1,wd);
        int numPixelsY    = max(1,ht);

        int bitsPerPixel = NWGetTextureFormatBitsPerPixel(GetFormat());

        textureDataSize += ((bitsPerPixel * numPixelsX * numPixelsY) / 8);
    } // End else

    return textureDataSize;
} // End of ComputeTextureFaceDataSize for CNWTexture


//------------------------------------------------------------------------------
// Compute ImageDataSize
//------------------------------------------------------------------------------
void CNWTexture::ComputeImageDataSize()
{
    m_textureDataSize = NWComputeImageDataSize(GetFormat(),
                                               m_textureType,
                                               m_width,
                                               m_height,
                                               m_numMipmaps);
} // End of ComputeImageDataSize for CNWTexture


//------------------------------------------------------------------------------
// TGA file
//------------------------------------------------------------------------------
void CNWTexture::SetAsTGAFile( bool bTGA, NW_TGA_Type TGAType )
{
    m_bTGA    = bTGA;
    m_TGAType = TGAType;
} // End of SetAsTGAFile for CNWTexture


//------------------------------------------------------------------------------
// PSD file
//------------------------------------------------------------------------------
void CNWTexture::SetAsPSDFile( bool bPSD, bool hasExpSet )
{
    m_bPSD = bPSD;
    m_bHasExportSetting = hasExpSet;
} // End of SetAsTGAFile for CNWTexture


//------------------------------------------------------------------------------
// Linear flags
//------------------------------------------------------------------------------
void CNWTexture::SetLinearFlags( bool redCh, bool greenCh, bool blueCh, bool alphaCh, bool twoChannelsAlphaFlag )
{
    m_linearFlags[NW_COMP_SEL_ELEMENT_R] = redCh;
    m_linearFlags[NW_COMP_SEL_ELEMENT_G] = twoChannelsAlphaFlag ? alphaCh : greenCh;
    m_linearFlags[NW_COMP_SEL_ELEMENT_B] = blueCh;
    m_linearFlags[NW_COMP_SEL_ELEMENT_A] = alphaCh;

    m_UILinearFlags[NW_COMP_SEL_ELEMENT_R] = redCh;
    m_UILinearFlags[NW_COMP_SEL_ELEMENT_G] = greenCh;
    m_UILinearFlags[NW_COMP_SEL_ELEMENT_B] = blueCh;
    m_UILinearFlags[NW_COMP_SEL_ELEMENT_A] = alphaCh;
}


//------------------------------------------------------------------------------
const bool CNWTexture::GetLinearFlag( NW_COMP_SEL_ELEMENT channel )
{
    if (channel < 0 || channel > 3)
        return false;
    return m_linearFlags[channel];
} // End of GetLinearFlag for CNWTexture


//------------------------------------------------------------------------------
const bool CNWTexture::GetUILinearFlag( NW_COMP_SEL_ELEMENT channel )
{
    if (channel < 0 || channel > 3)
        return false;
    return m_UILinearFlags[channel];
} // End of GetUILinearFlag for CNWTexture


//------------------------------------------------------------------------------
// Comment
//------------------------------------------------------------------------------
void CNWTexture::SetComment( const WCHAR* comment )
{
    if (comment==NULL)
    {
        m_commentText = L"";
        return;
    }

    // Check if this is empty string ( Only contains spaces, tab etc )
    bool bEmpty = true;
    int i = 0;
    while (comment[i]!=(WCHAR)0)
    {
        if ( (comment[i]!=(WCHAR)' ')  &&
             (comment[i]!=(WCHAR)'\n') &&
             (comment[i]!=(WCHAR)'\t') &&
             (comment[i]!=(WCHAR)'\a') )
        {
            bEmpty = false;
            break;
        }

        i++;
    }

    if (bEmpty)
    {
        m_commentText = L"";
        return;
    }

    m_commentText = comment;
}

//------------------------------------------------------------------------------
// Component selector
//------------------------------------------------------------------------------
void CNWTexture::SetCompSel( const NW_COMP_SEL &sel )
{
    m_compSel = sel;

    // Check if we should display as normal map
    m_bForceNormalMap    = false;
    m_bForceUseComputedZ = false;

    if ( (m_format==NW_IMG_FMT_GX2_SNORM_BC5) ||
         (m_format==NW_IMG_FMT_GX2_SNORM_8_8) )
    {
        // We should just listen to Hint, no longer force normal map mode
        // m_bForceNormalMap = true;
        if ( (m_compSel.sel[0]==NW_COMP_SEL_ELEMENT_R) &&
             (m_compSel.sel[1]==NW_COMP_SEL_ELEMENT_G) &&
             (m_compSel.sel[2]==NW_COMP_SEL_ELEMENT_0) &&
             (m_compSel.sel[3]==NW_COMP_SEL_ELEMENT_1) )
        {
            m_bForceUseComputedZ = true;
        } // End if
    } // End if
    if (m_bHasAlpha)
    {
        return;
    }
    else
    {
        m_bHasAlpha = m_compSel.sel[3] != NW_COMP_SEL_ELEMENT_1;
    } // End else
} // End of SetCompSel for CNWTexture


//------------------------------------------------------------------------------
// Component selector
//------------------------------------------------------------------------------
bool CNWTexture::HasCompSel() const
{
    bool bHasCompSel = true;

    if ( (IsPSD()) && (!HasPSDExportSetting()) )
    {
        bHasCompSel = false;
    } // End if

    return bHasCompSel;
} // End of HasCompSel for CNWTexture


//------------------------------------------------------------------------------
// Component selector
//------------------------------------------------------------------------------
void CNWTexture::SetHintNormalMap( bool bHint )
{
    m_bHintNormalMap = bHint;

    if (bHint)
    {
    } // End if
    else
    {
        if (m_format==NW_IMG_FMT_GX2_SNORM_BC5)
        {
            m_bForceNormalMap    = false;
            m_bForceUseComputedZ = false;
        } // End if
    } // End else
} // End of SetHintNormalMap for CNWTexture


//------------------------------------------------------------------------------
// Layer Comps
//------------------------------------------------------------------------------
void CNWTexture::SetLayerComps( const PSD_LAYER_COMPS* layerComps, int size)
{
    if (m_layerComps != NULL)
    {
        delete[] m_layerComps;
        m_layerComps = NULL;
        m_layerCompsSize = 0;
    }

    if (layerComps == NULL) { return; }
    if (size <= 0) { return; }

    m_layerCompsSize = size;
    m_layerComps = new PSD_LAYER_COMPS[size];
    for (int i = 0; i < size; i++)
    {
        m_layerComps[i] = layerComps[i];
    } // End for

} // End of SetLayerComps for CNWTexture

void CNWTexture::SetOriginalImageFieldInfo(std::vector<OriginalImageField>& info)
{
    m_pOriginalImageFieldInfo.swap(info);
    info.clear();
};
