﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef __ShellExtension_Manager_H__
#define __ShellExtension_Manager_H__

//==============================================================================
//
// Forward declaration
//
//==============================================================================
class CShellExtensionFileData;
class CNWCriticalSection;

//==============================================================================
//
// CShellExtensionManager
//
// - Manager for shell extension.
//   Keeps record of FileData. Since FileData can be accessed from different thread,
//   CriticalSection is used to prevent access collision.
//
//==============================================================================
class CShellExtensionManager
{
public:
    CShellExtensionManager();
    virtual ~CShellExtensionManager();

    //--------------------------------------------------------------------------
    // Init and Deinit
    //--------------------------------------------------------------------------
    bool Init( HINSTANCE hInstance );
    void Deinit();

    //--------------------------------------------------------------------------
    // Get Module Handle
    //--------------------------------------------------------------------------
    HINSTANCE GetModule() const                    { return m_hInstance; }
    const std::wstring& GetModuleName() const      { return m_moduleName; };
    const std::wstring& GetModuleDirectory() const { return m_moduleDirectory; };
    const std::wstring& GetAppName() const         { return m_appName; };

    //--------------------------------------------------------------------------
    // Check calling app name is equal to given
    //--------------------------------------------------------------------------
    bool IsAppName( const WCHAR* szAppName ) const;

    //--------------------------------------------------------------------------
    // Get file data with given file path
    //--------------------------------------------------------------------------
    CShellExtensionFileData* GetFileData( const WCHAR* szFileName, const WCHAR *szCaller );

    //--------------------------------------------------------------------------
    // Remove filedata from map
    //--------------------------------------------------------------------------
    void RemoveFileData( CShellExtensionFileData *pData );

    //--------------------------------------------------------------------------
    // FileData critical section
    //--------------------------------------------------------------------------
    CNWCriticalSection* GetFileDataCS() { return m_pFileDataCS; };

    //--------------------------------------------------------------------------
    // Texture loading critical section
    //--------------------------------------------------------------------------
    CNWCriticalSection* GetTextureLoadingCS() { return m_pTextureLoadingCS; };

private :
    HINSTANCE    m_hInstance;
    std::wstring m_moduleName;
    std::wstring m_moduleDirectory;
    std::wstring m_appName;

private : // GDIPlus
    //--------------------------------------------------------------------------
    // Initialize and Uninitialize GDI
    //--------------------------------------------------------------------------
    ULONG_PTR  m_GDIToken;

    bool InitGDIPlus();
    void DeinitGDIPlus();

private : // Map for FileData
    struct PathCompare : public std::less<const WCHAR*>
    {
        bool operator() ( const WCHAR* str1, const WCHAR* str2 ) const
        {
            int cmpResult = _wcsicmp(str1,str2);
            if (cmpResult<0)
                return true;

            return false;
        } // End of operator
    }; // End of PathCompare

    typedef std::map<const WCHAR*,CShellExtensionFileData*,PathCompare> FileDataMap;
    FileDataMap m_fileDataMap;

    CNWCriticalSection* m_pFileDataCS;
    CNWCriticalSection* m_pTextureLoadingCS;

    void ClearAllFileData();

public : // Singleton
    //--------------------------------------------------------------------------
    // Accessor for Singleton Instance
    //--------------------------------------------------------------------------
    static CShellExtensionManager* Instance();
}; // End of CShellExtensionManager

#endif // End of __ShellExtension_Manager_H__
