﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "ShellExtension_PCH.h"
#include "ShellExtension_Resource.h"

#include <initguid.h>

#include "ShellExtension_Type.h"

#ifdef NW_FOR_CTEX
    #include "ShellExtension_CTEX_i.c"
#elif defined NW_FOR_FTX
    #include "ShellExtension_FTX_i.c"
#elif defined NW_FOR_TGA
    #include "ShellExtension_TGA_i.c"
#elif defined NW_FOR_PSD
    #include "ShellExtension_PSD_i.c"
#elif defined NW_FOR_AI
    #include "ShellExtension_AI_i.c"
#elif defined NW_FOR_EPS
    #include "ShellExtension_EPS_i.c"
#elif defined NW_FOR_FXX
    #include "ShellExtension_FXX_i.c"
#elif defined NW_FOR_ESET
    #include "ShellExtension_ESET_i.c"
#elif defined NW_FOR_FLYT
    #include "ShellExtension_FLYT_i.c"
#elif defined NW_FOR_THUMBS
    #include "ShellExtension_THUMBS_i.c"
#endif

#include "ShellExtension_SyncObject.h"
#include "ShellExtension_Manager.h"
#include "ShellExtension_Utility.h"
#include "ShellExtension_Main.h"

#ifndef NW_FOR_FXX
    #include "Service/Icon/ShellExtension_Icon.h"
    #include "Service/InfoTip/ShellExtension_InfoTip.h"
    #include "Service/Preview/ShellExtension_Preview.h"
    #include "Service/Thumbnail/ShellExtension_Thumbnail.h"
    #include "Service/ExtractImage/ShellExtension_ExtractImage.h"
#endif // NW_FOR_FXX

extern const WCHAR* global_IntermediateAsciiFileExtentions[];
extern const WCHAR* global_IntermediateBinaryFileExtentions[];


//==============================================================================
//
// DLL Module
//
//==============================================================================
CComModule g_Module;

BEGIN_OBJECT_MAP(ObjectMap)
#if !defined NW_FOR_FXX

    OBJECT_ENTRY(CLSID_ShellExtensionIcon,           CShellExtensionIcon)
    #if !(defined NW_FOR_ESET || defined NW_FOR_FLYT || defined NW_FOR_THUMBS || defined NW_FOR_AI || defined NW_FOR_EPS)
    OBJECT_ENTRY(CLSID_ShellExtensionInfoTip,        CShellExtensionInfoTip)
    #endif
    OBJECT_ENTRY(CLSID_ShellExtensionPreview,        CShellExtensionPreview)

    #ifdef NW_USE_EXTRACTIMAGE
        OBJECT_ENTRY(CLSID_ShellExtensionExtractImage,   CShellExtensionExtractImage)
    #endif

    #ifdef NW_USE_THUMBNAIL
        OBJECT_ENTRY(CLSID_ShellExtensionThumbnail,      CShellExtensionThumbnail)
    #endif

#endif // NW_FOR_FXX

END_OBJECT_MAP()

extern "C"
{

//==============================================================================
//
// DLLMain
//
//==============================================================================
BOOL WINAPI DllMain(HINSTANCE hInstance, DWORD dwReason, LPVOID lpReserved)
{
    NW_USE_VAR(hInstance);
    NW_USE_VAR(dwReason);
    NW_USE_VAR(lpReserved);

    if (hInstance==NULL)
        return FALSE;

    if (dwReason == DLL_PROCESS_ATTACH)
    {
        #ifdef NW_FOR_CTEX
            g_Module.Init(ObjectMap, hInstance, &LIBID_ShellExtensionLibCTEX);
        #elif defined NW_FOR_FTX
            g_Module.Init(ObjectMap, hInstance, &LIBID_ShellExtensionLibFTX);
        #elif defined NW_FOR_FXX
            g_Module.Init(ObjectMap, hInstance, &LIBID_ShellExtensionLibFXX);
        #elif defined NW_FOR_TGA
            g_Module.Init(ObjectMap, hInstance, &LIBID_ShellExtensionLibTGA);
        #elif defined NW_FOR_PSD
            g_Module.Init(ObjectMap, hInstance, &LIBID_ShellExtensionLibPSD);
        #elif defined NW_FOR_AI
            g_Module.Init(ObjectMap, hInstance, &LIBID_ShellExtensionLibAI);
        #elif defined NW_FOR_EPS
            g_Module.Init(ObjectMap, hInstance, &LIBID_ShellExtensionLibEPS);
        #elif defined NW_FOR_ESET
            g_Module.Init(ObjectMap, hInstance, &LIBID_ShellExtensionLibESET);
        #elif defined NW_FOR_FLYT
            g_Module.Init(ObjectMap, hInstance, &LIBID_ShellExtensionLibFLYT);
        #elif defined NW_FOR_THUMBS
            g_Module.Init(ObjectMap, hInstance, &LIBID_ShellExtensionLibTHUMBS);
        #endif

        g_Module.Lock();

        CShellExtensionManager::Instance()->Init(hInstance);

        DisableThreadLibraryCalls(hInstance);

        NW_LOG(L"ShellExtension Attached");
    } // End if
    else if (dwReason == DLL_PROCESS_DETACH)
    {
        CShellExtensionManager::Instance()->Deinit();
        g_Module.Term();

        NW_LOG(L"ShellExtension Detached");
    } // End else if

    return TRUE;
} // End of DLLMain


//==============================================================================
//
// Used to determine whether the DLL can be unloaded by OLE
//
//==============================================================================
STDAPI DllCanUnloadNow()
{
    if (g_Module.GetLockCount()>0)
        return S_FALSE;

    return S_OK;
} // End of DLLCanUnloadNow


//==============================================================================
//
// Returns a class factory to create an object of the requested type
//
//==============================================================================
STDAPI DllGetClassObject(REFCLSID rclsid, REFIID riid, LPVOID* ppv)
{
    return g_Module.GetClassObject(rclsid, riid, ppv);
} // End of DllGetClassObject


//==============================================================================
//
// Prepare Registration param
//
//==============================================================================
void PrepareRegistrationParam( REGISTRATION_PARAM &param, const TCHAR *szModuleName )
{
    param.moduleName = szModuleName;

    #ifdef NW_FOR_CTEX
        param.fileTypeName = L"CTEX";
        param.progIDNames.push_back(L"NintendoWare.CTEX.1");
        param.supportedExtensions.push_back(L"ctex");
    #elif defined (NW_FOR_FTX)
        param.fileTypeName = L"FTX";
        param.progIDNames.push_back(L"NintendoWare.FTXA.1");
        param.supportedExtensions.push_back(L"ftxa");

        param.progIDNames.push_back(L"NintendoWare.FTXB.1");
        param.supportedExtensions.push_back(L"ftxb");
    #elif defined (NW_FOR_TGA)
        param.fileTypeName = L"TGA";
        param.progIDNames.push_back(L"NintendoWare.TGA.1");
        param.supportedExtensions.push_back(L"tga");
    #elif defined (NW_FOR_PSD)
        param.fileTypeName = L"PSD";
        param.progIDNames.push_back(L"NintendoWare.PSD.1");
        param.supportedExtensions.push_back(L"psd");
    #elif defined (NW_FOR_AI)
        param.fileTypeName = L"AI";
        param.progIDNames.push_back(L"NintendoWare.AI.1");
        param.supportedExtensions.push_back(L"ai");
    #elif defined (NW_FOR_EPS)
        param.fileTypeName = L"EPS";
        param.progIDNames.push_back(L"NintendoWare.EPS.1");
        param.supportedExtensions.push_back(L"eps");
    #elif defined (NW_FOR_ESET)
        param.fileTypeName = L"ESET";
        param.progIDNames.push_back(L"NintendoWare.ESET.1");
        param.supportedExtensions.push_back(L"eset");
    #elif defined (NW_FOR_FLYT)
        param.fileTypeName = L"FLYT";
        param.progIDNames.push_back(L"NintendoWare.FLYT.1");
        param.supportedExtensions.push_back(L"flyt");
    #elif defined (NW_FOR_THUMBS)
        param.fileTypeName = L"FMD";
        param.progIDNames.push_back(L"NintendoWare.FMDA.1");
        param.supportedExtensions.push_back(L"fmda");

        param.progIDNames.push_back(L"NintendoWare.FMDB.1");
        param.supportedExtensions.push_back(L"fmdb");
    #elif defined (NW_FOR_FXX)
        param.fileTypeName = L"FXX";
        WCHAR buffer[32];
        WCHAR temp[8];


        for (int i = 0; global_IntermediateAsciiFileExtentions[i] != NULL; i++)
        {
            memset(buffer, 0, sizeof(buffer));

            memset(temp, 0, sizeof(temp));
            int extensionLength = (int)wcslen(global_IntermediateAsciiFileExtentions[i]);
            for (int j = 0; j < extensionLength && j < ARRAY_SIZE(temp) - 1; j++)
                temp[j] = (WCHAR)towupper(global_IntermediateAsciiFileExtentions[i][j]);

            _snwprintf_s(buffer, ARRAY_SIZE(buffer), _TRUNCATE,L"NintendoWare%s.1", temp);
            param.progIDNames.push_back(buffer);
            param.supportedExtensions.push_back(global_IntermediateAsciiFileExtentions[i] + 1); // +1 to skip the leading period
        }
        for (int i = 0; global_IntermediateBinaryFileExtentions[i] != NULL; i++)
        {
            memset(buffer, 0, sizeof(buffer));

            memset(temp, 0, sizeof(temp));
            int extensionLength = (int)wcslen(global_IntermediateBinaryFileExtentions[i]);
            for (int j = 0; j < extensionLength && j < ARRAY_SIZE(temp) - 1; j++)
                temp[j] = (WCHAR)towupper(global_IntermediateBinaryFileExtentions[i][j]);

            _snwprintf_s(buffer, ARRAY_SIZE(buffer), _TRUNCATE,L"NintendoWare%s.1", temp);
            param.progIDNames.push_back(buffer);
            param.supportedExtensions.push_back(global_IntermediateBinaryFileExtentions[i] + 1); // +1 to skip the leading period
        }
    #endif
} // End of PrepareRegistrationParam


//==============================================================================
//
// CloneKeys
//
//==============================================================================
bool CloneKeys( CRegKey &destKey, CRegKey &srcKey )
{
    // Enumerate values
    DWORD index = 0;
    bool bOk = true;
    TCHAR valName[256];
    DWORD valNameSize;
    DWORD dwType;
    BYTE  pData[256*2];
    DWORD dwDataSize;

    while (bOk)
    {
        valNameSize = 256;
        dwDataSize  = sizeof(pData);
        if (::RegEnumValue(srcKey.m_hKey,index,valName,&valNameSize,NULL,
                           &dwType,pData,&dwDataSize)!=ERROR_SUCCESS)
        {
            bOk = false;
            break;
        } // End if

        destKey.SetValue(valName,dwType,pData,dwDataSize);

        index++;
    } // End while

    // Enumerate keys
    index = 0;
    bOk   = true;
    while (bOk)
    {
        valNameSize = 256;
        dwDataSize  = sizeof(pData);
        if (::RegEnumKeyEx(srcKey.m_hKey,index,valName,&valNameSize,NULL,
                           NULL,NULL,NULL)!=ERROR_SUCCESS)
        {
            bOk = false;
            break;
        } // End if

        CRegKey srcChildKey;
        if (srcChildKey.Open(srcKey.m_hKey,valName,KEY_READ)==ERROR_SUCCESS)
        {
            CRegKey destChildKey;
            if (destChildKey.Create(destKey.m_hKey,valName)==ERROR_SUCCESS)
            {
                CloneKeys(destChildKey,srcChildKey);
                destChildKey.Close();
            } // End if

            srcChildKey.Close();
        } // End if

        index++;
    } // End while

    return true;
} // End of CloneKeys


//==============================================================================
//
// Copy Shell Keys
//
//==============================================================================
bool CopyProgrIDShellKeys( CRegKey &destProgKey, CRegKey &srcProgKey )
{
    if (srcProgKey.m_hKey==NULL)
        return true;

    if (destProgKey.m_hKey==NULL)
        return true;

    CRegKey srcShellKey;
    if (srcShellKey.Open(srcProgKey.m_hKey,
                         L"Shell",
                         KEY_READ)!=ERROR_SUCCESS)
    {
        // Nothing to do
        return true;
    } // End if

    CRegKey destShellKey;
    if (destShellKey.Create(destProgKey.m_hKey,L"Shell")==ERROR_SUCCESS)
    {
        CloneKeys(destShellKey,srcShellKey);
        destShellKey.Close();
    } // End if

    srcShellKey.Close();

    return true;
} // End CopyProgrIDShellKeys


//==============================================================================
//
// Register Program ID
//
//==============================================================================
bool RegisterProgramID( REGISTRATION_PARAM &param )
{
    WCHAR szExtStr[128];

    int i;
    for (i=0;i<(int)param.supportedExtensions.size();i++)
    {
        // Open Extension
        _snwprintf_s(szExtStr,127,_TRUNCATE,L".%s", param.supportedExtensions[i].c_str());

        CRegKey extensionKey;
        if (extensionKey.Create(HKEY_CLASSES_ROOT,
                                szExtStr)!=ERROR_SUCCESS)
        {
            return false;
        } // End if

        // Get previously registered program
        WCHAR szValStr[256];
        ULONG numChars = 256;
        extensionKey.QueryStringValue(NULL,szValStr,&numChars);

        std::wstring prevProgID;
        if (numChars>1)
        {
            prevProgID = szValStr;
        } // End if

        // Create New program ID
        CRegKey progIDKey;
        if (progIDKey.Create( HKEY_CLASSES_ROOT,
                              param.progIDNames[i].c_str() )==ERROR_SUCCESS)
        {
            // If previously registered program is not same as this one
            if (prevProgID.size()>0)
            {
                if (prevProgID!=param.progIDNames[i])
                {
                    CRegKey srcProgIDKey;
                    if (srcProgIDKey.Open(HKEY_CLASSES_ROOT,
                                          prevProgID.c_str(),
                                          KEY_READ)==ERROR_SUCCESS)
                    {
                        CopyProgrIDShellKeys(progIDKey,srcProgIDKey);

                        srcProgIDKey.Close();
                    } // End if
                } // End if
            } // End if

            // Register new prog ID
            extensionKey.SetStringValue(NULL,param.progIDNames[i].c_str());

            // Create ShellEX
            CRegKey shellExKey;
            shellExKey.Create(progIDKey.m_hKey,L"ShellEx");
            shellExKey.Close();

            progIDKey.Close();
        } // End if

        //----------------------------------------------------------------------
        // If we have user choice choice app associated with copy the shell command
        // then remove this user choice
        //----------------------------------------------------------------------
        WCHAR szUserFileExtStr[256];
        _snwprintf_s(szUserFileExtStr,255,_TRUNCATE,L"SOFTWARE\\Microsoft\\Windows\\CurrentVersion\\Explorer\\FileExts\\.%s", param.supportedExtensions[i].c_str());

        CRegKey userFileExtKey;
        if (userFileExtKey.Open(HKEY_CURRENT_USER,
                                szUserFileExtStr,
                                KEY_SET_VALUE)==ERROR_SUCCESS)
        {
            CRegKey userChoiceKey;
            if (userChoiceKey.Open(userFileExtKey.m_hKey,
                                   L"UserChoice",
                                   KEY_READ)==ERROR_SUCCESS)
            {
                WCHAR szUserAppStr[256];
                ULONG dwUserAppCount = 256;
                if (userChoiceKey.QueryStringValue(L"Progid",szUserAppStr,&dwUserAppCount)==ERROR_SUCCESS)
                {
                    CRegKey userChoiceAppKey;
                    if (userChoiceAppKey.Open(HKEY_CLASSES_ROOT,
                                              szUserAppStr,
                                              KEY_READ)==ERROR_SUCCESS)
                    {
                        CRegKey myProgIDKey;
                        if (myProgIDKey.Open( HKEY_CLASSES_ROOT,
                                              param.progIDNames[i].c_str(),
                                              KEY_SET_VALUE)==ERROR_SUCCESS)
                        {
                            CopyProgrIDShellKeys(myProgIDKey,userChoiceAppKey);
                            myProgIDKey.Close();
                        } // End if

                        userChoiceAppKey.Close();
                    } // End if
                } // End if

                userChoiceKey.Close();
            } // End if

            userFileExtKey.DeleteSubKey(L"UserChoice");
            userFileExtKey.Close();
        } // End if
    } // End for

    return true;
} // End of RegisterProgramID


//==============================================================================
//
// Unregister Program ID
//
//==============================================================================
bool UnregisterProgramID( REGISTRATION_PARAM &param )
{
    CRegKey classRootKey(HKEY_CLASSES_ROOT);
    if (classRootKey.m_hKey==NULL)
        return true;

    int i;
    for (i=0;i<(int)param.supportedExtensions.size();i++)
    {
        CRegKey progIDKey;
        if (progIDKey.Open(classRootKey.m_hKey,
                           param.progIDNames[i].c_str(),
                           KEY_SET_VALUE)==ERROR_SUCCESS)
        {
            progIDKey.DeleteValue(L"FullDetails");
            progIDKey.RecurseDeleteKey(L"ShellEx");
        } // End if
    } // End for

    return true;
} // End of UnregisterProgramID


//==============================================================================
//
// DllRegisterServer - Adds entries to the system registry
//
//==============================================================================
STDAPI DllRegisterServer()
{
    CRegKey key;
    LONG lRet;

    // NT/2K 系かどうかの判定はもはやいらないはずなのでチェックしない
    // On NT/2K, put our extension in the "approved" list.
    //if ( (GetVersion() & 0x80000000) == 0 )
    {
        lRet = key.Open ( HKEY_LOCAL_MACHINE,
                          _T("SOFTWARE\\Microsoft\\Windows\\CurrentVersion\\Shell Extensions\\Approved"),
                          KEY_SET_VALUE );

        if ( lRet == ERROR_SUCCESS )
        {
            WCHAR szKeyStr[128];
            WCHAR szExtStr[8];
            WCHAR szNameStr[128];

            #ifdef NW_FOR_CTEX
                wcscpy_s(szExtStr,8,L"CTEX");
            #elif defined (NW_FOR_FTX)
                wcscpy_s(szExtStr,8,L"FTX");
            #elif defined (NW_FOR_TGA)
                wcscpy_s(szExtStr,8,L"TGA");
            #elif defined (NW_FOR_PSD)
                wcscpy_s(szExtStr,8,L"PSD");
            #elif defined (NW_FOR_AI)
                wcscpy_s(szExtStr, 8, L"AI");
            #elif defined (NW_FOR_EPS)
                wcscpy_s(szExtStr, 8, L"EPS");
            #elif defined (NW_FOR_ESET)
                wcscpy_s(szExtStr, 8, L"ESET");
            #elif defined (NW_FOR_FLYT)
                wcscpy_s(szExtStr, 8, L"FLYT");
            #elif defined (NW_FOR_THUMBS)
                wcscpy_s(szExtStr, 8, L"FMD");
            #elif defined (NW_FOR_FXX)
                wcscpy_s(szExtStr,8,L"FXX");
            #endif

            #if !defined NW_FOR_FXX

                _snwprintf_s(szKeyStr,127,_TRUNCATE,L"{%s}", NWCreateUUIDString(CLSID_ShellExtensionIcon).c_str());
                _snwprintf_s(szNameStr,127,_TRUNCATE,_T("NintendoWare %s Icon Shell Extension"), szExtStr );
                lRet = key.SetStringValue ( szKeyStr, szNameStr );

                #if !(defined NW_FOR_ESET || defined NW_FOR_FLYT || defined NW_FOR_THUMBS || defined NW_FOR_AI || defined NW_FOR_EPS)
                _snwprintf_s(szKeyStr,127,_TRUNCATE,L"{%s}", NWCreateUUIDString(CLSID_ShellExtensionInfoTip).c_str());
                _snwprintf_s(szNameStr,127,_T("NintendoWare %s Infotip Shell Extension"), szExtStr );
                lRet = key.SetStringValue ( szKeyStr, szNameStr );
                #endif

                _snwprintf_s(szKeyStr,127,_TRUNCATE,L"{%s}", NWCreateUUIDString(CLSID_ShellExtensionPreview).c_str());
                _snwprintf_s(szNameStr,127,_TRUNCATE,_T("NintendoWare %s Preview Shell Extension"), szExtStr );
                lRet = key.SetStringValue ( szKeyStr, szNameStr );

                #ifdef NW_USE_EXTRACTIMAGE
                    _snwprintf_s(szKeyStr,127,_TRUNCATE,L"{%s}", NWCreateUUIDString(CLSID_ShellExtensionExtractImage).c_str());
                    _snwprintf_s(szNameStr,127,_TRUNCATE,_T("NintendoWare %s ExtractImage Shell Extension"), szExtStr );
                    lRet = key.SetStringValue ( szKeyStr, szNameStr );
                #endif

                #ifdef NW_USE_THUMBNAIL
                    _snwprintf_s(szKeyStr,127,_TRUNCATE,L"{%s}", NWCreateUUIDString(CLSID_ShellExtensionThumbnail).c_str());
                    _snwprintf_s(szNameStr,127,_T("NintendoWare %s Thumbnail Shell Extension"), szExtStr );
                    lRet = key.SetStringValue ( szKeyStr, szNameStr );
                #endif

            #endif // NW_FOR_FXX

            key.Close();
        } // End if

        if ( lRet != ERROR_SUCCESS )
            return HRESULT_FROM_WIN32(lRet);
    } // End if

    // Registers object, typelib and all interfaces in typelib
    //HRESULT hr = g_Module.RegisterServer(false);
    //if (FAILED(hr))
    //    return hr;
    TCHAR szModulePath[256];
    ::GetModuleFileName(g_Module.GetModuleInstance(), szModulePath, sizeof(szModulePath) / sizeof(TCHAR));

    REGISTRATION_PARAM param;
    PrepareRegistrationParam(param,szModulePath);

    if (RegisterProgramID(param)==false)
        return E_ATL_BAD_HKEY;

    #ifndef NW_FOR_FXX
        CShellExtensionIcon::RegisterService(param);
        #if !(defined NW_FOR_ESET || defined NW_FOR_FLYT || defined NW_FOR_THUMBS || defined NW_FOR_AI)
        CShellExtensionInfoTip::RegisterService(param);
        #endif
        CShellExtensionPreview::RegisterService(param);

        #ifdef NW_USE_EXTRACTIMAGE
            CShellExtensionExtractImage::RegisterService(param);
        #endif

        #ifdef NW_USE_THUMBNAIL
            CShellExtensionThumbnail::RegisterService(param);
        #endif

    #endif // NW_FOR_FXX

    return S_OK;
} // End of DllRegisterServer


//==============================================================================
//
// DllUnregisterServer - Removes entries from the system registry
//
//==============================================================================
STDAPI DllUnregisterServer()
{

    CRegKey key;
    LONG lRet;

    // NT/2K 系かどうかの判定はもはやいらないはずなのでチェックしない
    // On NT/2K, remove our extension from the "approved" list.
    //if ( (GetVersion() & 0x80000000) == 0 )
    {
        lRet = key.Open ( HKEY_LOCAL_MACHINE,
                         _T("SOFTWARE\\Microsoft\\Windows\\CurrentVersion\\Shell Extensions\\Approved"),
                         KEY_SET_VALUE );

        if ( lRet == ERROR_SUCCESS )
        {
            WCHAR szKeyStr[1024];

            #ifndef NW_FOR_FXX

                _snwprintf_s(szKeyStr,1023,_TRUNCATE,L"{%s}", NWCreateUUIDString(CLSID_ShellExtensionIcon).c_str());
                key.DeleteValue ( szKeyStr );

            #if !(defined NW_FOR_ESET || defined NW_FOR_FLYT || defined NW_FOR_THUMBS || defined NW_FOR_AI || defined NW_FOR_EPS)
                _snwprintf_s(szKeyStr,1023,_TRUNCATE,L"{%s}", NWCreateUUIDString(CLSID_ShellExtensionInfoTip).c_str());
                key.DeleteValue ( szKeyStr );
            #endif
                _snwprintf_s(szKeyStr,1023,_TRUNCATE,L"{%s}", NWCreateUUIDString(CLSID_ShellExtensionPreview).c_str());
                key.DeleteValue ( szKeyStr );

                #ifdef NW_USE_EXTRACTIMAGE
                    _snwprintf_s(szKeyStr,1023,_TRUNCATE,L"{%s}", NWCreateUUIDString(CLSID_ShellExtensionExtractImage).c_str());
                    key.DeleteValue ( szKeyStr );
                #endif

                #ifdef NW_USE_THUMBNAIL
                    _snwprintf_s(szKeyStr,1023,_TRUNCATE,L"{%s}", NWCreateUUIDString(CLSID_ShellExtensionThumbnail).c_str());
                    key.DeleteValue ( szKeyStr );
                #endif

            #endif // NW_FOR_FXX

            key.Close();
        } // End if
    } // End if

    //return g_Module.UnregisterServer(false);
    TCHAR szModulePath[256];
    ::GetModuleFileName(g_Module.GetModuleInstance(), szModulePath, sizeof(szModulePath) / sizeof(TCHAR));

    REGISTRATION_PARAM param;
    PrepareRegistrationParam(param,szModulePath);
    UnregisterProgramID(param);

    #ifndef NW_FOR_FXX
        CShellExtensionIcon::UnregisterService(param);
        CShellExtensionInfoTip::UnregisterService(param);
        CShellExtensionPreview::UnregisterService(param);

        CShellExtensionExtractImage::UnregisterService(param);

        CShellExtensionThumbnail::UnregisterService(param);
    #endif // NW_FOR_FXX

    return S_OK;
} // End of DllUnregisterServer


} // End of extern "C"

