﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "ShellExtension_PCH.h"

#include "ShellExtension_Resource.h"
#include "ShellExtension_Type.h"
#include "ShellExtension_SyncObject.h"
#include "ShellExtension_Texture.h"
#include "ShellExtension_FileData.h"
#include "ShellExtension_Manager.h"
#include "ShellExtension_Utility.h"

#include "Service/Preview/ShellExtension_Preview.h"

//==============================================================================
//
// CShellExtensionPreview Static
//
//==============================================================================
std::vector<CShellExtensionPreview*> CShellExtensionPreview::s_previewInterfaces = std::vector<CShellExtensionPreview*>();
CNWCriticalSection CShellExtensionPreview::s_previewCS = CNWCriticalSection();

#ifdef NW_FOR_CTEX
    static const WCHAR* s_szPreviewWindowClassName = L"NWPreviewWindow_CTEX";
#elif defined NW_FOR_FTX
    static const WCHAR* s_szPreviewWindowClassName = L"NWPreviewWindow_FTX";
#elif defined NW_FOR_PSD
    static const WCHAR* s_szPreviewWindowClassName = L"NWPreviewWindow_PSD";
#elif defined NW_FOR_TGA
    static const WCHAR* s_szPreviewWindowClassName = L"NWPreviewWindow_TGA";
#elif defined NW_FOR_AI
    static const WCHAR* s_szPreviewWindowClassName = L"NWPreviewWindow_AI";
#elif defined NW_FOR_EPS
    static const WCHAR* s_szPreviewWindowClassName = L"NWPreviewWindow_EPS";
#elif defined NW_FOR_ESET
    static const WCHAR* s_szPreviewWindowClassName = L"NWPreviewWindow_ESET";
#elif defined NW_FOR_FLYT
    static const WCHAR* s_szPreviewWindowClassName = L"NWPreviewWindow_FLYT";
#elif defined NW_FOR_THUMBS
    static const WCHAR* s_szPreviewWindowClassName = L"NWPreviewWindow_THUMBS";

#endif


//==============================================================================
//
// CShellExtensionPreview Implementation
//
//==============================================================================
CShellExtensionPreview::CShellExtensionPreview() :
    m_pFileData(NULL),
    m_hParentWindow(NULL),
    m_hPreviewWindow(NULL),
    m_pSite(NULL),
    m_pFrame(NULL),
    m_pFont(NULL),
    m_drawingStringMaxSize(0)
{
    m_rect.left   = 0;
    m_rect.right  = 0;
    m_rect.top    = 0;
    m_rect.bottom = 0;

    m_scrollV.cbSize = sizeof(SCROLLINFO);
    m_scrollV.nPos = m_scrollV.nTrackPos = 0;
    m_scrollH.cbSize = sizeof(SCROLLINFO);
    m_scrollH.nPos = m_scrollH.nTrackPos = 0;

    m_pFont = new Gdiplus::Font(L"Courier", 12);

    // Register myself to table of preview interfaces
    CNWSingleLock lock(&s_previewCS);
    s_previewInterfaces.push_back(this);

    NWShellLockModule();
} // End of Constructor for CShellExtensionPreview


//------------------------------------------------------------------------------
CShellExtensionPreview::~CShellExtensionPreview()
{
    if (m_pFileData!=NULL)
    {
        m_pFileData->Release(L"Preview");
        m_pFileData = NULL;
    } // End if

    // Remove from Preview interface table
    CNWSingleLock lock(&s_previewCS);

    std::vector<CShellExtensionPreview*>::iterator itr = s_previewInterfaces.begin();
    while (itr!=s_previewInterfaces.end())
    {
        if ((*itr)==this)
        {
            s_previewInterfaces.erase(itr);
            break;
        } // End if

        ++itr;
    } // End while

    if (m_pFont!=NULL)
    {
        delete m_pFont;
        m_pFont = NULL;
    } // End if

    NWShellUnlockModule();
} // End of Destructor for CShellExtensionPreview


//------------------------------------------------------------------------------
STDMETHODIMP CShellExtensionPreview::SetWindow(HWND hwnd, const RECT *prc)
{
    m_hParentWindow = hwnd;
    m_rect = *prc;

    if (m_hPreviewWindow != NULL)
    {
        ::SetParent(m_hPreviewWindow, m_hParentWindow);
        ::SetWindowPos(m_hPreviewWindow, NULL,
                       m_rect.left, m_rect.top, m_rect.right - m_rect.left, m_rect.bottom - m_rect.top,
                       SWP_NOMOVE | SWP_NOZORDER | SWP_NOACTIVATE);
    } // End if

    return S_OK;
} // End of SetWindow for CShellExtensionPreview


//------------------------------------------------------------------------------
STDMETHODIMP CShellExtensionPreview::SetRect(const RECT *prc)
{
    m_rect = *prc;

    if (m_hPreviewWindow != NULL)
    {
        ::SetWindowPos(m_hPreviewWindow, NULL,
                       m_rect.left, m_rect.top, m_rect.right - m_rect.left, m_rect.bottom - m_rect.top,
                       SWP_NOMOVE | SWP_NOZORDER | SWP_NOACTIVATE);
    } // End if

    return S_OK;
} // End of SetRect for CShellExtensionPreview


//------------------------------------------------------------------------------
STDMETHODIMP CShellExtensionPreview::DoPreview(VOID)
{
    if (m_hPreviewWindow != NULL)
        return E_FAIL;



    DWORD style = WS_CHILD | WS_VISIBLE | ES_MULTILINE | ES_WANTRETURN | WS_VSCROLL | WS_HSCROLL;
    CNWTexture *pTexImage = m_pFileData->GetTexture();
    if (pTexImage != NULL && pTexImage->OnlyShowImage())
    {
        style = WS_CHILD | WS_VISIBLE | ES_MULTILINE | ES_WANTRETURN;
    }

    m_hPreviewWindow = CreateWindowEx(0, s_szPreviewWindowClassName, TEXT("NW Preview Data"),
                                      style,
                                      m_rect.left, m_rect.top, m_rect.right - m_rect.left, m_rect.bottom - m_rect.top,
                                      m_hParentWindow, 0, 0, NULL);
    if (m_hPreviewWindow == NULL)
        return HRESULT_FROM_WIN32(GetLastError());

    // スクロールバーのセットアップ
    if (pTexImage==NULL || pTexImage->GetLayerCompsSize() == 0)
    {
        m_scrollV.fMask = SIF_PAGE | SIF_RANGE | SIF_POS | SIF_DISABLENOSCROLL;
        m_scrollV.nPos = 0;
        m_scrollV.nMin = 0;    m_scrollV.nMax = 0;
        m_scrollV.nPage = 0;
    }
    else
    {
        m_scrollV.fMask = SIF_PAGE | SIF_RANGE | SIF_POS;
        m_scrollV.nPos = 0;
        m_scrollV.nMin = 0;
        // エクスポート設定の表示に 8 行
        m_scrollV.nPage = 8;
        // レイヤーごとの情報は空白行を含めて 9 行
        m_scrollV.nMax = pTexImage->GetLayerCompsSize()*9 + m_scrollV.nPage - 1;
    }// End if

    ::SetScrollInfo(m_hPreviewWindow , SB_VERT , &m_scrollV , TRUE);
    m_scrollV.fMask = SIF_POS;

    {
        m_scrollH.fMask = SIF_PAGE | SIF_RANGE | SIF_POS;
        m_scrollH.nPos = 0;
        m_scrollH.nMin = 0;
        m_scrollH.nPage = 1;
        m_scrollH.nMax = 10;
    }
    ::SetScrollInfo(m_hPreviewWindow , SB_HORZ , &m_scrollH , TRUE);
    m_scrollH.fMask = SIF_POS;

    return S_OK;
} // End of DoPreview for CShellExtensionPreview


//------------------------------------------------------------------------------
STDMETHODIMP CShellExtensionPreview::Unload(VOID)
{
    if (m_hPreviewWindow != NULL)
    {
        ::DestroyWindow(m_hPreviewWindow);
        m_hPreviewWindow = NULL;
    } // End if

    NW_RELEASE_INTERFACE(m_pFrame);
    NW_RELEASE_INTERFACE(m_pSite);

    if (m_pFileData!=NULL)
    {
        m_pFileData->Release(L"Preview::Unload");
        m_pFileData = NULL;
    } // End if

    return S_OK;
} // End of Unload for CShellExtensionPreview

//------------------------------------------------------------------------------
STDMETHODIMP CShellExtensionPreview::SetFocus(VOID)
{
    if (m_hPreviewWindow!=NULL)
    {
        ::SetFocus(m_hPreviewWindow);
    } // End if

    return S_OK;
} // End of SetFocus for CShellExtensionPreview


//------------------------------------------------------------------------------
STDMETHODIMP CShellExtensionPreview::QueryFocus(HWND *phwnd)
{
    HRESULT hr = E_INVALIDARG;

    if (phwnd != NULL)
    {
        *phwnd = ::GetFocus();
        if (*phwnd != NULL)
            hr = S_OK;
        else
            hr = HRESULT_FROM_WIN32(GetLastError());
    } // End if

    return hr;
} // End of QueryFocus for CShellExtensionPreview


//------------------------------------------------------------------------------
STDMETHODIMP CShellExtensionPreview::TranslateAccelerator(MSG *pmsg)
{
    HRESULT hr = S_FALSE;

    if (m_pFrame != NULL)
        hr = m_pFrame->TranslateAccelerator(pmsg);

    return hr;
} // End of TranslateAccelerator for CShellExtensionPreview


//------------------------------------------------------------------------------
STDMETHODIMP CShellExtensionPreview::SetSite(IUnknown *pUnkSite)
{
    HRESULT hr = S_OK;

    NW_RELEASE_INTERFACE(m_pFrame);
    NW_RELEASE_INTERFACE(m_pSite);

    m_pSite = pUnkSite;
    if (m_pSite!=NULL)
    {
        m_pSite->AddRef();
        hr = m_pSite->QueryInterface(IID_PPV_ARGS(&m_pFrame));
    } // End if

    return hr;
} // End of SetSite for CShellExtensionPreview


//------------------------------------------------------------------------------
STDMETHODIMP CShellExtensionPreview::GetSite(REFIID riid, void **ppvSite)
{
    NW_USE_VAR(riid.Data1);

    *ppvSite = m_pSite;

    return S_OK;
} // End of GetSite for CShellExtensionPreview


//------------------------------------------------------------------------------
STDMETHODIMP CShellExtensionPreview::GetWindow(HWND *phwnd)
{
    if (phwnd!=NULL)
    {
        *phwnd = m_hParentWindow; // Return what's been set by SetWindow
        if ((*phwnd)!=NULL)
            return S_OK;

        return S_OK;
    } // End if

    return E_FAIL;
} // End of GetWindow for CShellExtensionPreview


//------------------------------------------------------------------------------
STDMETHODIMP CShellExtensionPreview::ContextSensitiveHelp(BOOL fEnterMode)
{
    NW_USE_VAR(fEnterMode);
    return E_NOTIMPL;
} // End of ContextSensitiveHelp for CShellExtensionPreview


//------------------------------------------------------------------------------
// IInitializeWithFile
//------------------------------------------------------------------------------
STDMETHODIMP CShellExtensionPreview::Initialize( LPCWSTR pszFilePath,
                                             DWORD grfMode )
{
    NW_USE_VAR(grfMode);

    if (m_pFileData!=NULL)
    {
        m_pFileData->Release(L"Preview::Initialize");
        m_pFileData = NULL;
    } // End if

    m_pFileData = CShellExtensionManager::Instance()->GetFileData(pszFilePath,L"Preview");
    if (m_pFileData==NULL)
        return E_FAIL;

    if ( m_pFileData->GetTexture()==NULL )
    {
        if (m_pFileData->UpdateImage(true)==false)
            return E_FAIL;
    } // End if

    return S_OK;
} // End of Initialize for CShellExtensionPreview


//------------------------------------------------------------------------------
// IInitializeWithStream
//------------------------------------------------------------------------------
STDMETHODIMP CShellExtensionPreview::Initialize(IStream *pstream, DWORD grfMode)
{
    NW_USE_VAR(pstream);
    NW_USE_VAR(grfMode);

    // Not implemented
    return E_NOTIMPL;
} // End of Initialize for CShellExtensionPreview


//------------------------------------------------------------------------------
// Process Message from Preview Window
//------------------------------------------------------------------------------
LRESULT CALLBACK CShellExtensionPreview::ProcessMessage( HWND hWnd,
                                                     UINT uMsg,
                                                     WPARAM wParam,
                                                     LPARAM lParam )
{
    switch (uMsg)
    {
        case WM_PAINT :
            {
                RECT rect;
                ::GetClientRect(m_hPreviewWindow,&rect);

                LONG textWd = rect.right - rect.left;

                if (m_drawingStringMaxSize <= textWd)
                {
                    m_scrollH.fMask = SIF_PAGE | SIF_RANGE;
                    m_scrollH.nMin = 0;    m_scrollH.nMax = 0;
                    m_scrollH.nPage = 0;
                    ::SetScrollInfo(m_hPreviewWindow , SB_HORZ , &m_scrollH , TRUE);
                }
                else
                {
                    m_scrollH.fMask = SIF_PAGE | SIF_RANGE;
                    m_scrollH.nMin = 0;
                    m_scrollH.nPage = 1;
                    m_scrollH.nMax = 10;
                    ::SetScrollInfo(m_hPreviewWindow , SB_HORZ , &m_scrollH , TRUE);
                }
                m_scrollH.fMask = SIF_POS;

                PAINTSTRUCT ps;
                ::BeginPaint(hWnd,&ps);
                    OnPaint(ps);
                ::EndPaint(hWnd,&ps);

                return 0;
            } // End case

        case WM_SIZE :
            ::InvalidateRect(hWnd,NULL,TRUE);
            break;

        case WM_MOUSEWHEEL :
            {
                // レイヤーカンプがない時にスクロールさせると必要ないのに画面が
                // 動いてしまうので、レイヤーカンプが存在しないとスクロールさせない。
                CNWTexture *pTexImage = m_pFileData->GetTexture();
                if (pTexImage->GetLayerCompsSize() > 0)
                {
                    int delta = GET_WHEEL_DELTA_WPARAM(wParam);

                    m_scrollV.nPos -= delta / 100;
                    if (m_scrollV.nPos < 0)
                    {
                        m_scrollV.nPos = 0;
                    }
                    if (m_scrollV.nPos > (m_scrollV.nMax - (int)m_scrollV.nPage + 1))
                    {
                        m_scrollV.nPos = (m_scrollV.nMax - m_scrollV.nPage + 1);
                    }

                    ::SetScrollInfo(hWnd , SB_VERT , &m_scrollV , TRUE);
                    ::InvalidateRect(hWnd , NULL , TRUE);
                }

                return 0;
            } // End case

        case WM_LBUTTONDOWN :
        case WM_LBUTTONDBLCLK :
        case WM_RBUTTONDOWN :
        case WM_RBUTTONDBLCLK :
        case WM_MBUTTONDOWN :
        case WM_MBUTTONDBLCLK :
            {
                ::SetFocus(hWnd);
            }
            break;

        case WM_VSCROLL :
            {
                int nScrollCode = (int) LOWORD(wParam); // scroll bar value
                int nPos = (short int) HIWORD(wParam);  // scroll box position

                switch(nScrollCode)
                {
                    case SB_TOP:
                        m_scrollV.nPos = m_scrollV.nMin;
                        break;
                    case SB_BOTTOM:
                        m_scrollV.nPos = m_scrollV.nMax;
                        break;
                    case SB_LINEUP:
                        m_scrollV.nPos--;
                        break;
                    case SB_LINEDOWN:
                        m_scrollV.nPos++;
                        break;
                    case SB_PAGEUP:
                        m_scrollV.nPos -= m_scrollV.nPage;
                        break;
                    case SB_PAGEDOWN:
                        m_scrollV.nPos += m_scrollV.nPage;
                        break;
                    case SB_THUMBPOSITION:
                    case SB_THUMBTRACK:
                        m_scrollV.nPos = nPos;
                        break;
                }

                if (m_scrollV.nPos < 0)
                {
                    m_scrollV.nPos = 0;
                }
                if (m_scrollV.nPos > (m_scrollV.nMax - (int)m_scrollV.nPage + 1))
                {
                    m_scrollV.nPos = (m_scrollV.nMax - m_scrollV.nPage + 1);
                }

                ::SetScrollInfo(hWnd , SB_VERT , &m_scrollV , TRUE);
                ::InvalidateRect(hWnd , NULL , TRUE);
            }
            return 0;
        case WM_HSCROLL :
            {
                int nScrollCode = (int) LOWORD(wParam); // scroll bar value
                int nPos = (short int) HIWORD(wParam);  // scroll box position

                switch(nScrollCode)
                {
                case SB_TOP:
                    m_scrollH.nPos = m_scrollH.nMin;
                    break;
                case SB_BOTTOM:
                    m_scrollH.nPos = m_scrollH.nMax;
                    break;
                case SB_LINEUP:
                    m_scrollH.nPos--;
                    break;
                case SB_LINEDOWN:
                    m_scrollH.nPos++;
                    break;
                case SB_PAGEUP:
                    m_scrollH.nPos -= m_scrollH.nPage;
                    break;
                case SB_PAGEDOWN:
                    m_scrollH.nPos += m_scrollH.nPage;
                    break;
                case SB_THUMBPOSITION:
                case SB_THUMBTRACK:
                    m_scrollH.nPos = nPos;
                    break;
                }

                if (m_scrollH.nPos < 0)
                {
                    m_scrollH.nPos = 0;
                }
                if (m_scrollH.nPos > (m_scrollH.nMax - (int)m_scrollH.nPage + 1))
                {
                    m_scrollH.nPos = (m_scrollH.nMax - m_scrollH.nPage + 1);
                }

                ::SetScrollInfo(hWnd , SB_HORZ , &m_scrollH , TRUE);
                ::InvalidateRect(hWnd , NULL , TRUE);
            }
            return 0;
    } // End switch

    return ::DefWindowProc(m_hPreviewWindow,uMsg,wParam,lParam);
} // End of ProcessMessage for CShellExtensionPreview


//------------------------------------------------------------------------------
// Handle Paint Message
//------------------------------------------------------------------------------
void CShellExtensionPreview::OnPaint( PAINTSTRUCT &ps )
{
    if (m_hPreviewWindow==NULL)
        return;

    if (m_pFileData==NULL)
    {
        return;
    } // End if

    CNWSingleLock lock(m_pFileData->GetCriticalSection());

    CNWTexture *pTexImage = m_pFileData->GetTexture();
    if (pTexImage==NULL)
    {
        return;
    } // End if

    if (pTexImage->OnlyShowImage())
    {
        PaintOnlyImage(ps);
        return;
    }

    RECT rect;
    ::GetClientRect(m_hPreviewWindow,&rect);

    const bool bHasMipmaps = pTexImage->HasMipmaps();
    const bool bHasCompSel = pTexImage->HasCompSel();
    const bool bHasHint = !pTexImage->GetHint().empty();
    const bool bHasComment = !pTexImage->GetComment().empty();
    const bool bHasLayerComps = (pTexImage->GetLayerCompsSize() > 0);

    const float spaceMargin = 10.0f;
    const float spaceInfo   =
        + ((bHasHint) ? 20.0f : 0.0f)
        + 20.0f // size
        + ((bHasMipmaps) ? 20.0f : 0.0f)
        + 20.0f // data size
        + ((bHasCompSel) ? 20.0f : 0.0f)
        + 20.0f // Linear flags
        + ((bHasComment) ? 20.0f : 0.0f)
        + ((bHasLayerComps) ? 20.0f : 0.0f);

    Gdiplus::RectF drawInfoRect;
    drawInfoRect.X      = (float)rect.left;
    drawInfoRect.Y      = (float)rect.bottom - spaceInfo - spaceMargin;
    drawInfoRect.Width  = (float)(rect.right - rect.left);
    drawInfoRect.Height = (float)spaceInfo;

    float windowWd = (float)(rect.right  - rect.left);
    float windowHt = (float)(rect.bottom - rect.top);
    windowHt -= (drawInfoRect.Height + spaceMargin);
    if (windowHt<=0)
        return;

    // We are going to split with RGB and Alpha
    float imageWd = windowWd;
    float imageHt = ( windowHt / 2.0f );

    if ( (imageWd<=0) || (imageHt<=0) )
        return;

    float wdRatio = (float)imageWd / (float)pTexImage->GetPreviewWidth();
    float htRatio = (float)imageHt / (float)pTexImage->GetPreviewHeight();

    if (wdRatio<htRatio)
    {
        // Match the size scaling with width
        imageHt = ((float)imageHt*wdRatio/htRatio);
    } // End if
    else
    {
        // Match the size scaling with height
        imageWd = ((float)imageWd*htRatio/wdRatio);
    } // End else

    imageWd -= (spaceMargin*2.0f);
    imageHt -= (spaceMargin*2.0f);

    imageWd = max(2,imageWd);
    imageHt = max(2,imageHt);

    float totalHt = ( imageHt * 2.0f ) + spaceMargin;

    const int columnSize =
        + 1 // Format
        + ((bHasHint) ? 1 : 0)
        + 1 // Dimension
        + ((bHasMipmaps) ? 1 : 0)
        + 1 // ByteSize
        + ((bHasCompSel) ? 1 : 0)
        + 1 // Linear flags
        + ((bHasComment) ? 1 : 0)
        + ((bHasLayerComps) ? 1 : 0);

    int compsAllColumnSize = 0;
    if (bHasLayerComps)
    {
        int layerCompsSize = pTexImage->GetLayerCompsSize();
        const PSD_LAYER_COMPS* layerCompsData = pTexImage->GetLayerComps();
        for (int iLayerComps = 0; iLayerComps < layerCompsSize; iLayerComps++)
        {
            const PSD_LAYER_COMPS& layerComps = layerCompsData[iLayerComps];
            const NW_EXPORT_SETTING& layerCompsSetting = layerComps.setting;

            const bool bHasMipmaps2 = layerCompsSetting.numMipmaps > 1;
            const bool bHasHint2 = layerCompsSetting.hint.empty() == false;
            const bool bHasComment2 = layerCompsSetting.comment.empty() == false;

            compsAllColumnSize +=
                + 1 // space
                + 1 // Name
                + 1 // Format
                + ((bHasHint2) ? 1 : 0)
                + 1 // Dimension
                + ((bHasMipmaps2) ? 1 : 0)
                + 1 // ByteSize
                + 1 // CompSel
                + ((bHasComment2) ? 1 : 0);
        }
    }

    float texInfoHt  = (float)(drawInfoRect.Height);
    float textHt = (((float)texInfoHt-2.0f) / (float)columnSize) + 0.5f;
    textHt = (float)((int)textHt);  // Round to integer

    const float textWd = (float)(drawInfoRect.Width);

    const float scrollVPos = (float)(m_scrollV.nPos - m_scrollV.nMin);
    const float scrollVPosSize = (float)(m_scrollV.nMax - m_scrollV.nMin);
    const float possibleScrollVSize =
        + ( windowHt / 2.0f ) - ( totalHt / 2.0f )           // スペース
        + totalHt              // 画像
        + ( windowHt / 2.0f ) - ( totalHt / 2.0f )           // スペース
        + (textHt * columnSize) // 情報
        + (bHasLayerComps ? textHt * compsAllColumnSize : 0) // LayerComps情報
        + textHt                // スペース
        - windowHt;
    //const float possibleScrollVSizePerPos = textHt + ((totalHt - (textHt * (8 - columnSize + 1)) * (pTexImage->GetLayerCompsSize()) + (spaceMargin*3.0f) + textHt) / scrollVPosSize);
    const bool isDisableScrollV = scrollVPosSize == 0;
    const float scrollVPosPixel = isDisableScrollV ? 0 : (scrollVPos / scrollVPosSize) * possibleScrollVSize;
    drawInfoRect.Y -= scrollVPosPixel;

    const float scrollHPos = (float)(m_scrollH.nPos - m_scrollH.nMin);
    const float scrollHPosSize = (float)(m_scrollH.nMax - m_scrollH.nMin);
    const float possibleScrollHSize = (m_drawingStringMaxSize - windowWd);
    const bool isDisableScrollH = scrollHPosSize == 0;
    const float scrollHPosPixel = isDisableScrollH ? 0 : (scrollHPos / scrollHPosSize) * possibleScrollHSize;

    float startY  = ( windowHt / 2.0f ) - ( totalHt / 2.0f );
    float rgbLocX = ( windowWd / 2.0f ) - ( imageWd / 2.0f ); // Put Images to center of preview window
    float rgbLocY = startY - scrollVPosPixel;

    float alphaLocX = rgbLocX;
    float alphaLocY = rgbLocY + imageHt + spaceMargin;

    std::unique_ptr<Gdiplus::Graphics> pGraphics(Gdiplus::Graphics::FromHDC(ps.hdc));
    if (pGraphics==NULL)
        return;

    Gdiplus::Font *pFont = m_pFont;
    if (pFont==NULL)
    {
        return;
    } // End if

    pGraphics->SetCompositingMode(Gdiplus::CompositingModeSourceCopy);
    pGraphics->SetInterpolationMode(Gdiplus::InterpolationModeNearestNeighbor);
    pGraphics->SetCompositingQuality(Gdiplus::CompositingQualityHighQuality);

    pGraphics->Clear(Gdiplus::Color(255,255,255,255));

    Gdiplus::Pen blackPen(Gdiplus::Color(255,0,0,0));
    Gdiplus::Pen grayPen(Gdiplus::Color(255,128,128,128));

    Gdiplus::RectF drawRGBRect(rgbLocX,
                               rgbLocY,
                               imageWd,
                               imageHt);

    Gdiplus::ImageAttributes imgAttr;
    Gdiplus::ColorMatrix colorMat;
    colorMat.m[0][0] = 1.0f; colorMat.m[0][1] = 0.0f; colorMat.m[0][2] = 0.0f; colorMat.m[0][3] = 0.0f; colorMat.m[0][4] = 0.0f;
    colorMat.m[1][0] = 0.0f; colorMat.m[1][1] = 1.0f; colorMat.m[1][2] = 0.0f; colorMat.m[1][3] = 0.0f; colorMat.m[1][4] = 0.0f;
    colorMat.m[2][0] = 0.0f; colorMat.m[2][1] = 0.0f; colorMat.m[2][2] = 1.0f; colorMat.m[2][3] = 0.0f; colorMat.m[2][4] = 0.0f;
    colorMat.m[3][0] = 0.0f; colorMat.m[3][1] = 0.0f; colorMat.m[3][2] = 0.0f; colorMat.m[3][3] = 0.0f; colorMat.m[3][4] = 0.0f;
    colorMat.m[4][0] = 0.0f; colorMat.m[4][1] = 0.0f; colorMat.m[4][2] = 0.0f; colorMat.m[4][3] = 1.0f; colorMat.m[4][4] = 0.0f;

    imgAttr.SetColorMatrix(&colorMat);

    if ( pTexImage->GetPreviewColorBitmap() == NULL ||
         (pTexImage->GetFormat()==NW_IMG_FMT_A4) ||
         (pTexImage->GetFormat()==NW_IMG_FMT_A8) )
    {
        // If just alpha texture, show that color is not available
        pGraphics->DrawRectangle(&blackPen,drawRGBRect);
        pGraphics->DrawLine(&blackPen,drawRGBRect.GetLeft(),drawRGBRect.GetTop(),
                            drawRGBRect.GetRight(),drawRGBRect.GetBottom());
    } // End if
    else
    {
        pGraphics->DrawImage(pTexImage->GetPreviewColorBitmap(),
                             drawRGBRect, -0.5f, -0.5f,
                             (float)pTexImage->GetPreviewWidth(),(float)pTexImage->GetPreviewHeight(),
                             Gdiplus::UnitPixel,
                             &imgAttr);
        pGraphics->DrawRectangle(&blackPen,drawRGBRect);
    } // End else

    if (pTexImage->GetPreviewAlphaBitmap()!=NULL)
    {
        Gdiplus::RectF drawAlphaRect(alphaLocX,alphaLocY,imageWd,imageHt);
        pGraphics->DrawImage(pTexImage->GetPreviewAlphaBitmap(),
                             drawAlphaRect,
                             -0.5f, -0.5f,
                             (float)pTexImage->GetPreviewWidth(),(float)pTexImage->GetPreviewHeight(),
                             Gdiplus::UnitPixel);
        pGraphics->DrawRectangle(&blackPen,drawAlphaRect);
    } // End if
    else
    {
        Gdiplus::RectF drawAlphaRect(alphaLocX,alphaLocY,imageWd,imageHt);
        pGraphics->DrawRectangle(&blackPen,drawAlphaRect);
        pGraphics->DrawLine(&blackPen,drawAlphaRect.GetLeft(),drawAlphaRect.GetTop(),
                            drawAlphaRect.GetRight(),drawAlphaRect.GetBottom());
    } // End else

    // Draw Info
    Gdiplus::SolidBrush textBrush(Gdiplus::Color(255,0,0,0));
    Gdiplus::SolidBrush mipmapWarningBrush(Gdiplus::Color(255,96,0,0));

    const float rightMargin = 4.0f;
    const float topMargin   = 1.0f;

    Gdiplus::StringFormat strFormat(0);

    strFormat.SetAlignment(Gdiplus::StringAlignmentCenter);
    strFormat.SetLineAlignment(Gdiplus::StringAlignmentCenter);

    WCHAR szTempStr[256];
    WCHAR szTempStr2[256];

    pGraphics->SetCompositingMode(Gdiplus::CompositingModeSourceOver);

    float y = drawInfoRect.Y+topMargin;

    m_drawingStringMaxSize = textWd;

    { // Format
        const WCHAR* szFormatName = NWGetTextureFormatNameW(pTexImage->GetFormat());

        if ( ( (pTexImage->GetFormat()==NW_IMG_FMT_ETC1) ||
                (pTexImage->GetFormat()==NW_IMG_FMT_ETCA) ) &&
             (pTexImage->GetCompressTypeText().size()>0) )
        {
            _snwprintf_s(szTempStr,255,_TRUNCATE,L"%s(%s)", szFormatName, pTexImage->GetCompressTypeText().c_str());

            {
                Gdiplus::RectF drawTextRect;
                pGraphics->MeasureString(szTempStr, (int)wcslen(szTempStr), pFont, Gdiplus::PointF(0.0, 0.0), &drawTextRect);
                m_drawingStringMaxSize = m_drawingStringMaxSize < drawTextRect.Width ? drawTextRect.Width : m_drawingStringMaxSize;
            }
            Gdiplus::RectF textRect(-scrollHPosPixel,(float)y,textWd-rightMargin+scrollHPosPixel,textHt);
            pGraphics->DrawString(szTempStr,(int)wcslen(szTempStr),pFont,
                                  textRect,&strFormat,&textBrush);

        } // End if
        else
        {
            {
                Gdiplus::RectF drawTextRect;
                pGraphics->MeasureString(szFormatName, (int)wcslen(szFormatName), pFont, Gdiplus::PointF(0.0, 0.0), &drawTextRect);
                m_drawingStringMaxSize = m_drawingStringMaxSize < drawTextRect.Width ? drawTextRect.Width : m_drawingStringMaxSize;
            }
            Gdiplus::RectF textRect(-scrollHPosPixel,(float)y,textWd-rightMargin+scrollHPosPixel,textHt);
            pGraphics->DrawString(szFormatName,(int)wcslen(szFormatName),pFont,
                                  textRect,&strFormat,&textBrush);
        } // End else

        y += textHt;
    } // End

    // Hint
    if (bHasHint)
    {
        _snwprintf_s(szTempStr,255,_TRUNCATE,L"Hint : %s", pTexImage->GetHint().c_str());

        {
            Gdiplus::RectF drawTextRect;
            pGraphics->MeasureString(szTempStr, (int)wcslen(szTempStr), pFont, Gdiplus::PointF(0.0, 0.0), &drawTextRect);
            m_drawingStringMaxSize = m_drawingStringMaxSize < drawTextRect.Width ? drawTextRect.Width : m_drawingStringMaxSize;
        }
        Gdiplus::RectF textRect(-scrollHPosPixel,(float)y,textWd-rightMargin+scrollHPosPixel,textHt);
        pGraphics->DrawString(szTempStr,(int)wcslen(szTempStr),pFont,
            textRect,&strFormat,&textBrush);

        y += textHt;
    } // End if

    { // Dimension
        switch(pTexImage->GetTextureType())
        {
        default:
            _snwprintf_s(szTempStr,255,_TRUNCATE,L"%d x %d", pTexImage->GetWidth(), pTexImage->GetHeight());
            break;
        case NW_IMG_TYPE_1D_ARRAY:
        case NW_IMG_TYPE_2D_ARRAY:
        case NW_IMG_TYPE_CUBE_ARRAY:
            _snwprintf_s(szTempStr,255,_TRUNCATE,L"%d x %d x %d", pTexImage->GetWidth(), pTexImage->GetHeight(), pTexImage->GetCount());
            break;
        }

        {
            Gdiplus::RectF drawTextRect;
            pGraphics->MeasureString(szTempStr, (int)wcslen(szTempStr), pFont, Gdiplus::PointF(0.0, 0.0), &drawTextRect);
            m_drawingStringMaxSize = m_drawingStringMaxSize < drawTextRect.Width ? drawTextRect.Width : m_drawingStringMaxSize;
        }
        Gdiplus::RectF textRect(-scrollHPosPixel,(float)y,textWd-rightMargin+scrollHPosPixel,textHt);
        pGraphics->DrawString(szTempStr,(int)wcslen(szTempStr),pFont,
                              textRect,&strFormat,&textBrush);
        y += textHt;
    } // End

    // Mipmaps
    if (bHasMipmaps)
    {
        _snwprintf_s(szTempStr,255,_TRUNCATE,L"Mipmap : %d ", pTexImage->GetNumMipmaps());

        {
            Gdiplus::RectF drawTextRect;
            pGraphics->MeasureString(szTempStr, (int)wcslen(szTempStr), pFont, Gdiplus::PointF(0.0, 0.0), &drawTextRect);
            m_drawingStringMaxSize = m_drawingStringMaxSize < drawTextRect.Width ? drawTextRect.Width : m_drawingStringMaxSize;
        }
        Gdiplus::RectF textRect(-scrollHPosPixel,(float)y,textWd-rightMargin+scrollHPosPixel,textHt);
        if (pTexImage->IsUsingAllMipmaps())
        {
            pGraphics->DrawString(szTempStr,(int)wcslen(szTempStr),pFont,
                                    textRect,&strFormat,&textBrush);
        } // End if
        else
        {
            pGraphics->DrawString(szTempStr,(int)wcslen(szTempStr),pFont,
                                    textRect,&strFormat,&mipmapWarningBrush);
        } // End if

        y += textHt;
    } // End if

    { // ByteSize
        NW4FormatNumberStringWithComma(szTempStr,256,pTexImage->GetTextureDataSize());
        _snwprintf_s(szTempStr2,255,_TRUNCATE,L"%s B", szTempStr);

        {
            Gdiplus::RectF drawTextRect;
            pGraphics->MeasureString(szTempStr2, (int)wcslen(szTempStr2), pFont, Gdiplus::PointF(0.0, 0.0), &drawTextRect);
            m_drawingStringMaxSize = m_drawingStringMaxSize < drawTextRect.Width ? drawTextRect.Width : m_drawingStringMaxSize;
        }
        Gdiplus::RectF textRect(-scrollHPosPixel,(float)y,textWd-rightMargin+scrollHPosPixel,textHt);
        pGraphics->DrawString(szTempStr2,(int)wcslen(szTempStr2),pFont,
                              textRect,&strFormat,&textBrush);

        y += textHt;
    } // End


    // Component Selector
    if (bHasCompSel)
    {
        const NW_COMP_SEL &compSel = pTexImage->GetCompSel();
        std::wstring compSelStr = NWCompSelToString(compSel);

        {
            Gdiplus::RectF drawTextRect;
            pGraphics->MeasureString(szTempStr, (int)wcslen(szTempStr), pFont, Gdiplus::PointF(0.0, 0.0), &drawTextRect);
            m_drawingStringMaxSize = m_drawingStringMaxSize < drawTextRect.Width ? drawTextRect.Width : m_drawingStringMaxSize;
        }
        Gdiplus::RectF textRect(-scrollHPosPixel,(float)y,textWd-rightMargin+scrollHPosPixel,textHt);
        pGraphics->DrawString(compSelStr.c_str(),(int)compSelStr.length(),pFont,
                              textRect,&strFormat,&textBrush);

        y += textHt;
    } // End

    // Linear
    {
        int linearCharIdx    = 0;
        WCHAR szLinearStr[5] = { (WCHAR)0 };

        if (pTexImage->GetUILinearFlag(NW_COMP_SEL_ELEMENT_R))
            szLinearStr[linearCharIdx++] = L'R';
        if (pTexImage->GetUILinearFlag(NW_COMP_SEL_ELEMENT_G))
            szLinearStr[linearCharIdx++] = L'G';
        if (pTexImage->GetUILinearFlag(NW_COMP_SEL_ELEMENT_B))
            szLinearStr[linearCharIdx++] = L'B';
        if (pTexImage->GetUILinearFlag(NW_COMP_SEL_ELEMENT_A))
            szLinearStr[linearCharIdx++] = L'A';

        if (linearCharIdx>0)
        {
            _snwprintf_s(szTempStr,255,_TRUNCATE,L"Linear : %s", szLinearStr);
        }
        else
        {
            _snwprintf_s(szTempStr,255,_TRUNCATE,L"Linear : ");
        }

        {
            Gdiplus::RectF drawTextRect;
            pGraphics->MeasureString(szTempStr, (int)wcslen(szTempStr), pFont, Gdiplus::PointF(0.0, 0.0), &drawTextRect);
            m_drawingStringMaxSize = m_drawingStringMaxSize < drawTextRect.Width ? drawTextRect.Width : m_drawingStringMaxSize;
        }
        Gdiplus::RectF textRect(-scrollHPosPixel,(float)y,textWd-rightMargin+scrollHPosPixel,textHt);
        pGraphics->DrawString(szTempStr,(int)wcslen(szTempStr),pFont,
                              textRect,&strFormat,&textBrush);

        y += textHt;
    }

    // Comment
    if (bHasComment)
    {
        _snwprintf_s(szTempStr,255,_TRUNCATE,L"Comment : %s", pTexImage->GetComment().c_str());

        {
            Gdiplus::RectF drawTextRect;
            pGraphics->MeasureString(szTempStr, (int)wcslen(szTempStr), pFont, Gdiplus::PointF(0.0, 0.0), &drawTextRect);
            m_drawingStringMaxSize = m_drawingStringMaxSize < drawTextRect.Width ? drawTextRect.Width : m_drawingStringMaxSize;
        }
        Gdiplus::RectF textRect(-scrollHPosPixel,(float)y,textWd-rightMargin+scrollHPosPixel,textHt);
        pGraphics->DrawString(szTempStr,(int)wcslen(szTempStr),pFont,
                              textRect,&strFormat,&textBrush);

        y += textHt;
    } // End If

    // Layer
    if (bHasLayerComps)
    {
        int layerCompsSize = pTexImage->GetLayerCompsSize();

        { // layer comps
            _snwprintf_s(szTempStr,255,_TRUNCATE,L"%d Layer Comps", layerCompsSize);

            {
                Gdiplus::RectF drawTextRect;
                pGraphics->MeasureString(szTempStr, (int)wcslen(szTempStr), pFont, Gdiplus::PointF(0.0, 0.0), &drawTextRect);
                m_drawingStringMaxSize = m_drawingStringMaxSize < drawTextRect.Width ? drawTextRect.Width : m_drawingStringMaxSize;
            }
            Gdiplus::RectF textRect(-scrollHPosPixel,(float)y,textWd-rightMargin+scrollHPosPixel,textHt);
            pGraphics->DrawString(szTempStr,(int)wcslen(szTempStr),pFont,
                                  textRect,&strFormat,&textBrush);

            y += textHt;
        }

        const PSD_LAYER_COMPS* layerCompsData = pTexImage->GetLayerComps();
        for (int iLayerComps = 0; iLayerComps < layerCompsSize; iLayerComps++)
        {
            const PSD_LAYER_COMPS& layerComps = layerCompsData[iLayerComps];
            const NW_EXPORT_SETTING& layerCompsSetting = layerComps.setting;

            // レイヤーカンプ間で一行空ける
            y += textHt;

            { // Name
                _snwprintf_s(szTempStr,255,_TRUNCATE,L"Name : %s", layerComps.name.c_str());

                {
                    Gdiplus::RectF drawTextRect;
                    pGraphics->MeasureString(szTempStr, (int)wcslen(szTempStr), pFont, Gdiplus::PointF(0.0, 0.0), &drawTextRect);
                    m_drawingStringMaxSize = m_drawingStringMaxSize < drawTextRect.Width ? drawTextRect.Width : m_drawingStringMaxSize;
                }
                Gdiplus::RectF textRect(-scrollHPosPixel,(float)y,textWd-rightMargin+scrollHPosPixel,textHt);
                pGraphics->DrawString(szTempStr,(int)wcslen(szTempStr),pFont,
                                      textRect,&strFormat,&textBrush);

                y += textHt;
            } // End

            { // Format
                const WCHAR* szFormatName = NWGetTextureFormatNameW(layerCompsSetting.format);
                if (szFormatName == NULL)
                {
                    szFormatName = L"NONE";
                } // End if

                {
                    Gdiplus::RectF drawTextRect;
                    pGraphics->MeasureString(szFormatName, (int)wcslen(szFormatName), pFont, Gdiplus::PointF(0.0, 0.0), &drawTextRect);
                    m_drawingStringMaxSize = m_drawingStringMaxSize < drawTextRect.Width ? drawTextRect.Width : m_drawingStringMaxSize;
                }
                Gdiplus::RectF textRect(-scrollHPosPixel,(float)y,textWd-rightMargin+scrollHPosPixel,textHt);
                pGraphics->DrawString(szFormatName,(int)wcslen(szFormatName),pFont,
                                      textRect,&strFormat,&textBrush);

                y += textHt;
            } // End

            // Hint
            if (!layerCompsSetting.hint.empty())
            {
                _snwprintf_s(szTempStr,255,_TRUNCATE,L"Hint : %s", layerCompsSetting.hint.c_str());

                {
                    Gdiplus::RectF drawTextRect;
                    pGraphics->MeasureString(szTempStr, (int)wcslen(szTempStr), pFont, Gdiplus::PointF(0.0, 0.0), &drawTextRect);
                    m_drawingStringMaxSize = m_drawingStringMaxSize < drawTextRect.Width ? drawTextRect.Width : m_drawingStringMaxSize;
                }
                Gdiplus::RectF textRect(-scrollHPosPixel,(float)y,textWd-rightMargin+scrollHPosPixel,textHt);
                pGraphics->DrawString(szTempStr,(int)wcslen(szTempStr),pFont,
                    textRect,&strFormat,&textBrush);
                y += textHt;
            } // End If

            { // Dimension
                switch(pTexImage->GetTextureType())
                {
                default:
                    _snwprintf_s(szTempStr,255,_TRUNCATE,L"%d x %d", pTexImage->GetWidth(), pTexImage->GetHeight());
                    break;
                case NW_IMG_TYPE_1D_ARRAY:
                case NW_IMG_TYPE_2D_ARRAY:
                case NW_IMG_TYPE_CUBE_ARRAY:
                    _snwprintf_s(szTempStr,255,_TRUNCATE,L"%d x %d x %d", pTexImage->GetWidth(), pTexImage->GetHeight(), pTexImage->GetCount());
                    break;
                }

                {
                    Gdiplus::RectF drawTextRect;
                    pGraphics->MeasureString(szTempStr, (int)wcslen(szTempStr), pFont, Gdiplus::PointF(0.0, 0.0), &drawTextRect);
                    m_drawingStringMaxSize = m_drawingStringMaxSize < drawTextRect.Width ? drawTextRect.Width : m_drawingStringMaxSize;
                }
                Gdiplus::RectF textRect(-scrollHPosPixel,(float)y,textWd-rightMargin+scrollHPosPixel,textHt);
                pGraphics->DrawString(szTempStr,(int)wcslen(szTempStr),pFont,
                                        textRect,&strFormat,&textBrush);
                y += textHt;
            } // End

            // Mipmaps
            if (layerCompsSetting.numMipmaps > 1)
            {
                _snwprintf_s(szTempStr,255,_TRUNCATE,L"Mipmap : %d ", layerCompsSetting.numMipmaps);

                {
                    Gdiplus::RectF drawTextRect;
                    pGraphics->MeasureString(szTempStr, (int)wcslen(szTempStr), pFont, Gdiplus::PointF(0.0, 0.0), &drawTextRect);
                    m_drawingStringMaxSize = m_drawingStringMaxSize < drawTextRect.Width ? drawTextRect.Width : m_drawingStringMaxSize;
                }
                Gdiplus::RectF textRect(-scrollHPosPixel,(float)y,textWd-rightMargin+scrollHPosPixel,textHt);
                pGraphics->DrawString(szTempStr,(int)wcslen(szTempStr),pFont,
                                      textRect,&strFormat,&textBrush);
                y += textHt;
            } // End

            { // ByteSize
                __int64 byteSize = NWComputeImageDataSize(layerCompsSetting.format, pTexImage->GetTextureType(),
                                                          layerCompsSetting.width, layerCompsSetting.height,
                                                          layerCompsSetting.numMipmaps);
                NW4FormatNumberStringWithComma(szTempStr,256,byteSize);
                _snwprintf_s(szTempStr2,255,_TRUNCATE,L"%s B", szTempStr);

                {
                    Gdiplus::RectF drawTextRect;
                    pGraphics->MeasureString(szTempStr2, (int)wcslen(szTempStr2), pFont, Gdiplus::PointF(0.0, 0.0), &drawTextRect);
                    m_drawingStringMaxSize = m_drawingStringMaxSize < drawTextRect.Width ? drawTextRect.Width : m_drawingStringMaxSize;
                }
                Gdiplus::RectF textRect(-scrollHPosPixel,(float)y,textWd-rightMargin+scrollHPosPixel,textHt);
                pGraphics->DrawString(szTempStr2,(int)wcslen(szTempStr2),pFont,
                                      textRect,&strFormat,&textBrush);
                y += textHt;
            } // End

            { // Component Selector
                const NW_COMP_SEL &compSel = layerCompsSetting.compSel;
                std::wstring compSelStr = NWCompSelToString(compSel);

                {
                    Gdiplus::RectF drawTextRect;
                    pGraphics->MeasureString(compSelStr.c_str(), (int)wcslen(compSelStr.c_str()), pFont, Gdiplus::PointF(0.0, 0.0), &drawTextRect);
                    m_drawingStringMaxSize = m_drawingStringMaxSize < drawTextRect.Width ? drawTextRect.Width : m_drawingStringMaxSize;
                }
                Gdiplus::RectF textRect(-scrollHPosPixel,(float)y,textWd-rightMargin+scrollHPosPixel,textHt);
                pGraphics->DrawString(compSelStr.c_str(),(int)compSelStr.length(),pFont,
                                      textRect,&strFormat,&textBrush);
                y += textHt;
            } // End

            if (!layerCompsSetting.comment.empty())
            {
                _snwprintf_s(szTempStr,255,_TRUNCATE,L"Comment : %s", layerCompsSetting.comment.c_str());

                {
                    Gdiplus::RectF drawTextRect;
                    pGraphics->MeasureString(szTempStr, (int)wcslen(szTempStr), pFont, Gdiplus::PointF(0.0, 0.0), &drawTextRect);
                    m_drawingStringMaxSize = m_drawingStringMaxSize < drawTextRect.Width ? drawTextRect.Width : m_drawingStringMaxSize;
                }
                Gdiplus::RectF textRect(-scrollHPosPixel,(float)y,textWd-rightMargin+scrollHPosPixel,textHt);
                pGraphics->DrawString(szTempStr,(int)wcslen(szTempStr),pFont,
                                         textRect,&strFormat,&textBrush);
                y += textHt;
            } // End If
        } // End for
    } // End if


    pGraphics->Flush(Gdiplus::FlushIntentionSync);
} // End of OnPaint for CShellExtensionPreview

void CShellExtensionPreview::PaintOnlyImage(PAINTSTRUCT &ps)
{
    CNWSingleLock lock(m_pFileData->GetCriticalSection());

    RECT rect;
    ::GetClientRect(m_hPreviewWindow, &rect);

    CNWTexture *pTexImage = m_pFileData->GetTexture();
    Gdiplus::Bitmap* pBitmap(pTexImage->GetPreviewBitmap());

    std::unique_ptr<Gdiplus::Graphics> pGraphics(Gdiplus::Graphics::FromHDC(ps.hdc));
    if (pGraphics == nullptr)
    {
        return;
    }

    // 画像の描画
    float ratio = (std::min)((std::min)((rect.right - rect.left - 64)/(float)pBitmap->GetWidth(), (rect.bottom - rect.top - 64)/(float)pBitmap->GetHeight()), 1.0f);
    ratio = (std::max)(0.0f, ratio);
    int width = (int)floor(pBitmap->GetWidth()*ratio);
    int height = (int)floor(pBitmap->GetHeight()*ratio);
    Gdiplus::Rect imageRect((rect.left + rect.right - width) / 2, (rect.top + rect.bottom - height) / 2, width, height);
    pGraphics->DrawImage(pBitmap, imageRect);

    // 枠の描画
    Gdiplus::Pen blackPen(Gdiplus::Color(255, 0, 0, 0));
    imageRect.X -= 1;
    imageRect.Y -= 1;

    // +2 だと隙間ができる
    imageRect.Width += 1;
    imageRect.Height += 1;
    pGraphics->DrawRectangle(&blackPen, imageRect);

    pGraphics->Flush(Gdiplus::FlushIntentionSync);
}

//==============================================================================
//
// PeviewWindow handling proc
//
//==============================================================================
LRESULT CALLBACK NWPreviewWindowProc( HWND hWnd,
                                      UINT uMsg,
                                      WPARAM wParam,
                                      LPARAM lParam )
{
    return CShellExtensionPreview::HandleWindowProc(hWnd,uMsg,wParam,lParam);
} // End of NWPreviewWindowProc


//------------------------------------------------------------------------------
// Static function called on Manager init
//------------------------------------------------------------------------------
bool CShellExtensionPreview::InitPreview()
{
    WNDCLASS wndClass;

    wndClass.style         = CS_CLASSDC | CS_DBLCLKS;
    wndClass.lpfnWndProc   = NWPreviewWindowProc;
    wndClass.cbClsExtra    = 0;
    wndClass.cbWndExtra    = 0;
    wndClass.hInstance     = CShellExtensionManager::Instance()->GetModule();
    wndClass.hIcon         = NULL;
    wndClass.hCursor       = NULL;
    wndClass.hbrBackground = (HBRUSH)::GetStockObject(DC_BRUSH);
    wndClass.lpszMenuName  = NULL;
    wndClass.lpszClassName = s_szPreviewWindowClassName;

    ::RegisterClass(&wndClass);

    return true;
} // End of InitPreview for CShellExtensionPreview


//------------------------------------------------------------------------------
// Find corresponding preview interface from hWnd and let that interface handle
// the messages
//------------------------------------------------------------------------------
LRESULT CALLBACK CShellExtensionPreview::HandleWindowProc( HWND hWnd,
                                                         UINT uMsg,
                                                         WPARAM wParam,
                                                         LPARAM lParam )
{
    CShellExtensionPreview* pPreview = NULL;

    auto deleter = [](CShellExtensionPreview* x) { x->Release(); };
    std::unique_ptr<CShellExtensionPreview, decltype(deleter)> p(nullptr, deleter);

    { // Find preview interface for that window
        CNWSingleLock lock(&s_previewCS);

        std::vector<CShellExtensionPreview*>::iterator itr = s_previewInterfaces.begin();
        while (itr!=s_previewInterfaces.end())
        {
            if ((*itr)->m_hPreviewWindow==hWnd)
            {
                pPreview = (*itr);
                pPreview->AddRef();
                p.reset(pPreview);
                break;
            } // End if

            ++itr;
        } // End while
    }

    if (pPreview!=NULL)
    {
        LRESULT result = pPreview->ProcessMessage(hWnd,uMsg,wParam,lParam);
        return result;
    } // End if

    return ::DefWindowProc(hWnd,uMsg,wParam,lParam);
} // End of HandleWindowProc for CShellExtensionPreview


//------------------------------------------------------------------------------
// Register
//------------------------------------------------------------------------------
bool CShellExtensionPreview::RegisterService( REGISTRATION_PARAM &param )
{
    WCHAR szGUIDStr[128];
    WCHAR szValStr[128];

    _snwprintf_s(szGUIDStr,127,_TRUNCATE,L"{%s}", NWCreateUUIDString(CLSID_ShellExtensionPreview).c_str());

    //--------------------------------------------------------------------------
    // CLSID
    //--------------------------------------------------------------------------
    CRegKey CLSIDKey;
    if (CLSIDKey.Open( HKEY_CLASSES_ROOT,
                       _T("CLSID"),
                       KEY_SET_VALUE )==ERROR_SUCCESS)
    {
        //----------------------------------------------------------------------
        // My Interface ID
        //----------------------------------------------------------------------
        _snwprintf_s(szValStr,127,_TRUNCATE,_T("NintendoWare %s Preview Shell Extension"), param.fileTypeName.c_str() );

        CRegKey GUIDKey;
        if (GUIDKey.Create(CLSIDKey.m_hKey,szGUIDStr)==ERROR_SUCCESS)
        {
            GUIDKey.SetStringValue(NULL,szValStr);
            GUIDKey.SetStringValue(L"AppId",L"{6d2b5079-2f0b-48dd-ab7f-97cec514d30b}");

            //------------------------------------------------------------------
            // Inproc server
            //------------------------------------------------------------------
            CRegKey InpProc32Key;
            if (InpProc32Key.Create(GUIDKey.m_hKey,L"InprocServer32")==ERROR_SUCCESS)
            {
                InpProc32Key.SetStringValue(NULL,param.moduleName.c_str());
                InpProc32Key.SetStringValue(L"ThreadingModel", L"Apartment");
                InpProc32Key.Close();
            } // End if

            GUIDKey.Close();
        } // End if

        CLSIDKey.Close();
    } // End if

    //--------------------------------------------------------------------------
    // ProgID
    //--------------------------------------------------------------------------
    int i;
    for (i=0;i<(int)param.supportedExtensions.size();i++)
    {
        CRegKey progIDKey;
        if (progIDKey.Open( HKEY_CLASSES_ROOT,
                            param.progIDNames[i].c_str(),
                            KEY_SET_VALUE )==ERROR_SUCCESS)
        {
            //------------------------------------------------------------------
            // ShellEx Icon
            //------------------------------------------------------------------
            CRegKey shellExKey;
            if (shellExKey.Open(progIDKey.m_hKey,
                                L"ShellEx",
                                KEY_SET_VALUE)==ERROR_SUCCESS)
            {
                //--------------------------------------------------------------
                // ShellEx Icon
                //--------------------------------------------------------------
                CRegKey handlerKey;
                if (handlerKey.Create(shellExKey.m_hKey,L"{8895b1c6-b41f-4c1c-a562-0d564250836f}")==ERROR_SUCCESS)
                {
                    handlerKey.SetStringValue(NULL,szGUIDStr);
                    handlerKey.Close();
                } // End if

                shellExKey.Close();
            } // End if

            progIDKey.Close();
        } // End if
    } // End for

    //--------------------------------------------------------------------------
    // Add to preview list
    //--------------------------------------------------------------------------
    CRegKey previewHandlerKey;
    if (previewHandlerKey.Open(HKEY_LOCAL_MACHINE,
                               L"SOFTWARE\\Microsoft\\Windows\\CurrentVersion\\PreviewHandlers",
                               KEY_SET_VALUE)==ERROR_SUCCESS)
    {
        _snwprintf_s(szValStr,127,_TRUNCATE,_T("NintendoWare %s Preview Shell Extension"), param.fileTypeName.c_str() );
        previewHandlerKey.SetStringValue(szGUIDStr,szValStr);
        previewHandlerKey.Close();
    } // End if

    return true;
} // End of RegisterService for CShellExtensionPreview


//------------------------------------------------------------------------------
// Unregister
//------------------------------------------------------------------------------
bool CShellExtensionPreview::UnregisterService( REGISTRATION_PARAM &param )
{
    NW_USE_VAR(param.supportedExtensions.size());

    WCHAR szGUIDStr[128];
    _snwprintf_s(szGUIDStr,127,_TRUNCATE,L"{%s}", NWCreateUUIDString(CLSID_ShellExtensionPreview).c_str());

    //--------------------------------------------------------------------------
    // CLSID
    //--------------------------------------------------------------------------
    CRegKey CLSIDKey;
    if (CLSIDKey.Open ( HKEY_CLASSES_ROOT,
                        _T("CLSID"),
                        KEY_SET_VALUE )==ERROR_SUCCESS)
    {
        //----------------------------------------------------------------------
        // My Interface ID
        //----------------------------------------------------------------------
        CLSIDKey.RecurseDeleteKey(szGUIDStr);

        CLSIDKey.Close();
    } // End if

    //--------------------------------------------------------------------------
    // Remove from preview list
    //--------------------------------------------------------------------------
    CRegKey previewHandlerKey;
    if (previewHandlerKey.Open(HKEY_LOCAL_MACHINE,
                               L"SOFTWARE\\Microsoft\\Windows\\CurrentVersion\\PreviewHandlers",
                               KEY_SET_VALUE)==ERROR_SUCCESS)
    {
        previewHandlerKey.DeleteValue(szGUIDStr);
        previewHandlerKey.Close();
    } // End if

    return true;
} // End of UnregisterService for CShellExtensionPreview
