﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "ShellExtension_PCH.h"
#include "ShellExtension_Utility.h"
#include "ShellExtension_Type.h"
#include "ShellExtension_Texture.h"
#include "ImageLoader/ShellExtension_ImageLoader.h"
#include "ImageLoader/ShellExtension_TGALoader.h"
#include "ImageLoader/ShellExtension_CTexLoader.h"

//==============================================================================
//
// Implementation of CNWTGALoader
//
//==============================================================================
CNWTGALoader::CNWTGALoader() :
    CNWImageLoader()
{
} // End of Constructor for CNWTGALoader


//------------------------------------------------------------------------------
CNWTGALoader::~CNWTGALoader()
{
} // End of Destructor for CNWTGALoader


//------------------------------------------------------------------------------
// Load from image file onto pTexture
//
// - If bLoadPreview is true, create preview image for icon
//   If bLoadPreview is false, just load information ( format, size etc )
//------------------------------------------------------------------------------
bool CNWTGALoader::Load( CNWTexture *pTexture,
                         const WCHAR *szFilePath,
                         bool bLoadPreview )
{
    // Get to extension area
    FILE *pFile = NULL;
    WCHAR szModeStr[8];
    szModeStr[0] = (WCHAR)'r';
    szModeStr[1] = (WCHAR)'b';
    szModeStr[2] = (WCHAR)0;

    _wfopen_s(&pFile,szFilePath,szModeStr);
    if (pFile==NULL)
        return false;

    std::unique_ptr<std::FILE, decltype(&std::fclose)> fp(pFile, &std::fclose);

    fseek(pFile,0,SEEK_END);
    int fileSize = ftell(pFile);
    fseek(pFile,0,SEEK_SET);

    std::unique_ptr<unsigned char[]> pTGABuffer(new unsigned char[fileSize]);
    if (pTGABuffer==NULL)
    {
        return false;
    } // End if

    if (fread(pTGABuffer.get(),fileSize,1,pFile)!=1)
    {
        return false;
    } // End if

    // Close file
    fp.reset();


    // Offset in memory
    int iCurBufferPos = 0;

    // Read TGA Header
    TGAHeader fileHeader;
    memset(&fileHeader,0,sizeof(fileHeader));
    if (ReadTGAHeader(pTexture,pTGABuffer.get(),fileHeader,fileSize,iCurBufferPos)==false)
    {
        return false;
    } // End if

    // Read ID
    NW_TGA_Type TGACustomType  = NW_TGA_NONE;
    int additionalInfoOffset = 0;
    if (ReadTGAIdField(pTexture,pTGABuffer.get(),fileHeader,fileSize,iCurBufferPos,&additionalInfoOffset,
                       TGACustomType)==false)
    {
        return false;
    } // End if

    pTexture->SetAsTGAFile(true, TGACustomType);

    // Read TGA
    if (TGACustomType != NW_TGA_NONE)
    {
        iCurBufferPos = additionalInfoOffset;

        if (TGACustomType == NW_TGA_4C) // CTR
        {
            pTexture->SetShowMipmapCount(true);
            if (ReadNW4CTGA(pTexture,pTGABuffer.get(),fileHeader,fileSize,iCurBufferPos,bLoadPreview)==false)
            {
                return false;
            } // End if
        }
        else if (TGACustomType == NW_TGA_4R) // Revolution
        {
            pTexture->SetShowMipmapCount(true);
            if (ReadNW4RTGA(pTexture,pTGABuffer.get(),fileHeader,fileSize,iCurBufferPos,bLoadPreview)==false)
            {
                return false;
            } // End if
        } // End if
        else
        {
            return false;
        }
    } // End if
    else
    {
        pTexture->SetShowMipmapCount(false);

        if (ReadNormalTGA(pTexture,pTGABuffer.get(),fileHeader,fileSize,iCurBufferPos,bLoadPreview)==false)
        {
            return false;
        } // End if
    } // End else

    pTexture->ComputeImageDataSize();

    return true;
} // End of Load for CNWTGALoader


//------------------------------------------------------------------------------
// Read TGA
//------------------------------------------------------------------------------
bool CNWTGALoader::ReadTGAHeader( CNWTexture *pTexture,
                                  unsigned char *pTGABuffer, TGAHeader &fileHeader,
                                  int fileSize, int &iCurBufferPos )
{
    NW_USE_VAR(pTexture);

    if (fileSize<(iCurBufferPos+18))
        return false;

    unsigned char *pBuffer = pTGABuffer + iCurBufferPos;

    fileHeader.idSize          = *((unsigned char*)(pBuffer+0));
    fileHeader.colourMapType   = *((unsigned char*)(pBuffer+1));
    fileHeader.imageType       = *((unsigned char*)(pBuffer+2));
    fileHeader.colourMapStart  = *((short*)(pBuffer+3));
    fileHeader.colourMapLength = *((short*)(pBuffer+5));
    fileHeader.colourMapBits   = *((unsigned char*)(pBuffer+7));
    fileHeader.xStart          = *((short*)(pBuffer+8));
    fileHeader.yStart          = *((short*)(pBuffer+10));
    fileHeader.width           = *((short*)(pBuffer+12));
    fileHeader.height          = *((short*)(pBuffer+14));
    fileHeader.bits            = *((unsigned char*)(pBuffer+16));
    fileHeader.descriptor      = *((unsigned char*)(pBuffer+17));

    iCurBufferPos += 18;

    // Checking if it is valid TGA
    if ( (fileHeader.imageType!=TGA_IMAGE_TYPE_NONE) &&
         (fileHeader.imageType!=TGA_IMAGE_TYPE_UNCOMPRESSED_COLORMAP) &&
         (fileHeader.imageType!=TGA_IMAGE_TYPE_UNCOMPRESSED_RGB) &&
         (fileHeader.imageType!=TGA_IMAGE_TYPE_UNCOMPRESSED_BW) &&
         (fileHeader.imageType!=TGA_IMAGE_TYPE_RLE_COLORMAP) &&
         (fileHeader.imageType!=TGA_IMAGE_TYPE_RLE_RGB) &&
         (fileHeader.imageType!=TGA_IMAGE_TYPE_COMPRESSED_BW) &&
         (fileHeader.imageType!=TGA_IMAGE_TYPE_COMPRESSED_COLORMAP) &&
         (fileHeader.imageType!=TGA_IMAGE_TYPE_COMPRESSED_COLORMAP_4PASS) )
    {
        return false;
    } // End if

    if ( (fileHeader.bits!=8) &&
         (fileHeader.bits!=16) &&
         (fileHeader.bits!=24) &&
         (fileHeader.bits!=32) )
    {
        return false;
    } // End if

    return true;
} // End of ReadTGAHeader for CNWTGALoader


//------------------------------------------------------------------------------
bool CNWTGALoader::ReadTGAIdField( CNWTexture *pTexture,
                                   unsigned char *pTGABuffer, const TGAHeader &fileHeader,
                                   int fileSize, int &iCurBufferPos,
                                   int *pAdditionalInfoOfset,
                                   NW_TGA_Type &TGACustomType )
{
    NW_USE_VAR(pTexture);

    // Set default return values
    TGACustomType = NW_TGA_NONE;
    *pAdditionalInfoOfset = 0;

    if (fileHeader.idSize<=0)
    {
        return true;
    } // End if

    if (fileSize<(iCurBufferPos+fileHeader.idSize))
    {
        return false;
    } // End if

    if (fileHeader.idSize<=(16)) // Make sure we have enough id filed size to do string comparison below
    {
        // This extra info is not for NWTGA
        iCurBufferPos += fileHeader.idSize;
        return true;
    } // End if

    unsigned char *pBuffer = pTGABuffer + iCurBufferPos;
    if (strncmp((char*)pBuffer,"NW4C_Tga Ver1.0",16)==0)
    {
        TGACustomType = NW_TGA_4C;

        int *pOffset = (int*)(pBuffer + 16);
        *pAdditionalInfoOfset = *pOffset;
    }
    else if (strncmp((char*)pBuffer,"NW4R_Tga Ver1.0",16)==0)
    {
        TGACustomType = NW_TGA_4R;

        unsigned int *pOffset = (unsigned int*)(pBuffer + 16);
        *pAdditionalInfoOfset = (int)NWSwapByteOrderDWord(*pOffset);
    }
    else
    {
        // This extra info is not for NWTGA
        iCurBufferPos += fileHeader.idSize;
        return true;
    } // End if

    iCurBufferPos += fileHeader.idSize;

    return true;
} // End of ReadTGAIdField for CNWTGALoader


//------------------------------------------------------------------------------
bool CNWTGALoader::ReadNWTGADataBlockHeader( CNWTexture *pTexture,
                                             unsigned char *pTGABuffer, const TGAHeader &fileHeader,
                                             int fileSize, int &iCurBufferPos,
                                             NWTGADataBlockHeader &dataBlockHeader )
{
    NW_USE_VAR(fileHeader.idSize);

    if (pTexture==NULL)
        return false;

    if (fileSize<(iCurBufferPos+12))
        return false;

    unsigned char *pBuffer = pTGABuffer + iCurBufferPos;

    memcpy(dataBlockHeader.dataBlockID,pBuffer,8);
    if (pTexture->TGAType() == NW_TGA_4R)
    {
        dataBlockHeader.dataBlockSize  = NWSwapByteOrderDWord(*((unsigned int*)(pBuffer+8)));
    }
    else
    {
        dataBlockHeader.dataBlockSize  = *((unsigned int*)(pBuffer+8));
    }

    iCurBufferPos += 12;

    return true;
} // End of ReadNWTGADataBlockHeader for CNWTGALoader


//------------------------------------------------------------------------------
bool CNWTGALoader::ReadNW4CTGA( CNWTexture *pTexture,
                                unsigned char *pTGABuffer, const TGAHeader &fileHeader,
                                int fileSize, int &iCurBufferPos,
                                bool bLoadPreview )
{
    unsigned char *pBuffer = pTGABuffer + iCurBufferPos;

    NWTGADataBlockHeader dataBlockHeader;

    // Read format. Format block must come first
    if (ReadNWTGADataBlockHeader(pTexture,pTGABuffer,fileHeader,fileSize,
                                 iCurBufferPos,dataBlockHeader)==false)
    {
        return false;
    } // End if

    if (strncmp(dataBlockHeader.dataBlockID,"nw4c_tfm",8)!=0)
    {
        // Format block must come here
        return false;
    } // End if

    int dataSize = (dataBlockHeader.dataBlockSize - 12);
    iCurBufferPos += dataSize;

    if (dataSize>16) // Format string should not be that big
        return false;

    char szFormatName[16];
    memset(szFormatName,0,16);
    memcpy(szFormatName,pBuffer+12,dataSize);

    NW_IMG_FMT format     = NWGetTextureFormatFromString(szFormatName);
    if (format==NW_IMG_FMT_NONE)
        return false;

    // Set default values
    int width    = (int)fileHeader.width;
    int height   = (int)fileHeader.height;

    int numMipmaps = 1;

    NW_IMG_TYPE textureType = NW_IMG_TYPE_2D;

    int etcEncoding = -1;

    // Parse NWTGA
    bool bDone = false;
    while (!bDone)
    {
        if (ReadNWTGADataBlockHeader(pTexture,pTGABuffer,fileHeader,fileSize,
                                     iCurBufferPos,dataBlockHeader)==false)
        {
            return false;
        } // End if

        dataSize = (dataBlockHeader.dataBlockSize - 12);

        if (strncmp(dataBlockHeader.dataBlockID,"nw4c_end",8)==0)
        {
            bDone = true;
        } // End if
        else if (strncmp(dataBlockHeader.dataBlockID,"nw4c_cub",8)==0)
        {
            textureType = NW_IMG_TYPE_CUBE;
            width    = (int)fileHeader.width  / 4;
            height   = (int)fileHeader.height / 3;
        } // End else if
        else if (strncmp(dataBlockHeader.dataBlockID,"nw4c_mps",8)==0)
        {
            char mipCountStr[8];
            memset(mipCountStr,0,8);

            if (dataSize<4)
            {
                memcpy(mipCountStr,pTGABuffer+iCurBufferPos,dataSize);
                numMipmaps = atoi(mipCountStr);
            } // End if

            if (numMipmaps<=0)
                numMipmaps = 1;
        } // End else if
        else if (strncmp(dataBlockHeader.dataBlockID,"nw4c_txd",8)==0)
        {
            int dataOffset = iCurBufferPos;

            // Size MAY not be power of 2, so fix it
            width  = NWFindNextPowerOf2(width);
            height = NWFindNextPowerOf2(height);

            pTexture->SetDescription(textureType,format,NW_IMG_FMT_NONE,width,height,0,numMipmaps);

            if (bLoadPreview)
            {
                if (pTexture->CreatePreviewImage()==false)
                    return false;

                Gdiplus::Bitmap *pPreviewBitmap = pTexture->GetPreviewBitmap();
                if (pPreviewBitmap==NULL)
                {
                    return false;
                } // End if

                // Lock bits to prepare preview image
                Gdiplus::Rect lockRect(0,0,pPreviewBitmap->GetWidth(),pPreviewBitmap->GetHeight());
                Gdiplus::BitmapData bmpData;
                if (pPreviewBitmap->LockBits(&lockRect,Gdiplus::ImageLockModeWrite,
                                              pPreviewBitmap->GetPixelFormat(),
                                              &bmpData)!=Gdiplus::Ok)
                {
                    return false;
                } // End if

                int i;
                if (textureType==NW_IMG_TYPE_2D)
                {
                    for (i=0;i<pTexture->GetNumMipmaps();i++)
                    {
                        int imgSize = pTexture->ComputeTextureFaceDataSize(i);
                        unsigned char *pSrcBuffer = pTGABuffer+dataOffset;
                        if (i==0) // Load only top level
                        {
                            CNWCTexLoader::ConvertToPreviewImage(pTexture,
                                                                 pSrcBuffer,
                                                                 imgSize,
                                                                 NULL,
                                                                 NW_CUBE_FACE_POS_X,
                                                                 bmpData);
                        } // End if

                        dataOffset += imgSize;
                    } // End for
                } // End if
                else
                {
                    NW_CUBE_FACE cubeFaces[6] =
                    {
                        NW_CUBE_FACE_POS_X,
                        NW_CUBE_FACE_NEG_X,
                        NW_CUBE_FACE_POS_Y,
                        NW_CUBE_FACE_NEG_Y,
                        NW_CUBE_FACE_POS_Z,
                        NW_CUBE_FACE_NEG_Z
                    }; // End of cubeFaces

                    int cubeFaceIndex;
                    for (cubeFaceIndex=0;cubeFaceIndex<6;cubeFaceIndex++)
                    {
                        NW_CUBE_FACE cubeFace = cubeFaces[cubeFaceIndex];

                        for (i=0;i<pTexture->GetNumMipmaps();i++)
                        {
                            int imgSize = pTexture->ComputeTextureFaceDataSize(i);
                            unsigned char *pSrcBuffer = pTGABuffer+dataOffset;
                            if (i==0) // Load only top level
                            {
                                CNWCTexLoader::ConvertToPreviewImage(pTexture,
                                                                     pSrcBuffer,
                                                                     imgSize,
                                                                     NULL,
                                                                     cubeFace,
                                                                     bmpData);
                            } // End if

                            dataOffset += imgSize;
                        } // End for
                    } // End for
                } // End else

                Gdiplus::Rect drawRect(0,0,pPreviewBitmap->GetWidth(),pPreviewBitmap->GetHeight());

                pTexture->CreateColorMap(bmpData, ORIGINAL_FORMAT_NOT_USED, drawRect);

                if (pTexture->HasAlpha())
                {
                    pTexture->CreateAlphaMap(bmpData, ORIGINAL_FORMAT_NOT_USED, drawRect);
                } // End if

                // Finish locking
                pPreviewBitmap->UnlockBits(&bmpData);
            } // End if
        } // End else if
        else if (strncmp(dataBlockHeader.dataBlockID,"nw4c_udt",8)==0)
        {
        } // End else if
        else if (strncmp(dataBlockHeader.dataBlockID,"nw4c_mpi",8)==0)
        {
        } // End else if
        else if (strncmp(dataBlockHeader.dataBlockID,"nw4c_gnm",8)==0)
        {
        } // End else if
        else if (strncmp(dataBlockHeader.dataBlockID,"nw4c_gvr",8)==0)
        {
        } // End else if
        else if (strncmp(dataBlockHeader.dataBlockID,"nw4c_psh",8)==0)
        {
            NWTGADataBlockHeader pshDataBlockHeader;

            // Read format. Format block must come first
            bool bDonePsh   = false;
            int pshCurPos   = iCurBufferPos;
            int pshDataSize = 0;

            while (!bDonePsh)
            {
                if (ReadNWTGADataBlockHeader(pTexture,pTGABuffer,fileHeader,fileSize,
                                             pshCurPos,pshDataBlockHeader)==false)
                {
                    return false;
                } // End if

                pshDataSize = (pshDataBlockHeader.dataBlockSize - 12);

                if (strncmp(pshDataBlockHeader.dataBlockID,"psh_etco",8)==0)
                {
                    etcEncoding = (int)*((unsigned char*)(pTGABuffer+pshCurPos));

                    switch (etcEncoding)
                    {
                        case 0 :
                            pTexture->SetCompressTypeText(L"Fast");
                            break;

                        case 1 :
                            pTexture->SetCompressTypeText(L"Medium");
                            break;

                        case 2 :
                            pTexture->SetCompressTypeText(L"Slow");
                            break;

                        case 3 :
                            pTexture->SetCompressTypeText(L"Fast Perceptual");
                            break;

                        case 4 :
                            pTexture->SetCompressTypeText(L"Medium Perceptual");
                            break;

                        case 5 :
                            pTexture->SetCompressTypeText(L"Slow Perceptual");
                            break;

                        case 6 :
                            pTexture->SetCompressTypeText(L"Fast Improved");
                            break;

                        case 7 :
                            pTexture->SetCompressTypeText(L"Medium Improved");
                            break;
                    } // End switch
                } // End if
                else if (strncmp(pshDataBlockHeader.dataBlockID,"nw4c_end",8)==0)
                {
                    bDonePsh = true;
                } // End else if

                pshCurPos += pshDataSize;
            } // End while
        } // End else if

        iCurBufferPos += dataSize;
    } // End while

    return true;
} // End of ReadNWTGA for CNWTGALoader

//------------------------------------------------------------------------------
bool CNWTGALoader::ReadNW4RTGA( CNWTexture *pTexture,
                                unsigned char *pTGABuffer, const TGAHeader &fileHeader,
                                int fileSize, int &iCurBufferPos,
                                bool bLoadPreview )
{
    unsigned char *pBuffer = pTGABuffer + iCurBufferPos;

    NWTGADataBlockHeader dataBlockHeader;

    // Read format. Format block must come first
    if (ReadNWTGADataBlockHeader(pTexture,pTGABuffer,fileHeader,fileSize,
                                 iCurBufferPos,dataBlockHeader)==false)
    {
        return false;
    } // End if

    if (strncmp(dataBlockHeader.dataBlockID,"nw4r_tfm",8)!=0) // Texture format
    {
        // Format block must come here
        return false;
    } // End if

    int dataSize = (dataBlockHeader.dataBlockSize - 12);
    iCurBufferPos += dataSize;

    if (dataSize>16) // Format string should not be that big
        return false;

    char szFormatName[16];
    memset(szFormatName,0,16);
    memcpy(szFormatName,pBuffer+12,dataSize);

    int dataOffset  = 0;
    int imgDataSize = 0;
    NW_USE_VAR(imgDataSize);

    NW_IMG_FMT format = NWGetTextureFormatFromString(szFormatName);
    if (format==NW_IMG_FMT_NONE)
        return false;

    NW_IMG_FMT paletteFormat = NW_IMG_FMT_NONE;
    std::unique_ptr<unsigned short[]> paletteData(new unsigned short[16384 * 4]); // 14 bits index * 32 bits colors
    memset(paletteData.get(),0,16384 * 4);

    // Set default values
    int width    = (int)fileHeader.width;
    int height   = (int)fileHeader.height;

    int numMipmaps = 1;

    NW_IMG_TYPE textureType = NW_IMG_TYPE_2D;

    // Parse NWTGA
    bool bDone = false;
    while (!bDone)
    {
        if (ReadNWTGADataBlockHeader(pTexture,pTGABuffer,fileHeader,fileSize,
                                     iCurBufferPos,dataBlockHeader)==false)
        {
            return false;
        } // End if

        dataSize = (dataBlockHeader.dataBlockSize - 12);

        if (strncmp(dataBlockHeader.dataBlockID,"nw4r_end",8)==0) // End
        {
            bDone = true;

            // Size MAY not be power of 2, so fix it
            /*
            width  = NWFindNextPowerOf2(width);
            height = NWFindNextPowerOf2(height);
            */

            pTexture->SetDescription(textureType,format,paletteFormat,width,height,0,numMipmaps);

            if (bLoadPreview)
            {
                if (pTexture->CreatePreviewImage()==false)
                {
                    return false;
                } // End if

                Gdiplus::Bitmap *pPreviewBitmap = pTexture->GetPreviewBitmap();
                if (pPreviewBitmap==NULL)
                {
                    return false;
                } // End if

                // Lock bits to prepare preview image
                Gdiplus::Rect lockRect(0,0,
                                       pPreviewBitmap->GetWidth(),
                                       pPreviewBitmap->GetHeight());
                Gdiplus::BitmapData bmpData;
                if (Gdiplus::Ok != pPreviewBitmap->LockBits(&lockRect, Gdiplus::ImageLockModeWrite,
                                                            pPreviewBitmap->GetPixelFormat(),
                                                            &bmpData))
                {
                    return false;
                } // End if

                for (int i=0;i<pTexture->GetNumMipmaps();i++)
                {
                    int imgSize = pTexture->ComputeTextureFaceDataSize(i);
                    unsigned char *pSrcBuffer = pTGABuffer+dataOffset;
                    if (i==0) // Load only top level
                    {
                        CNWCTexLoader::ConvertToPreviewImage(pTexture,
                                                             pSrcBuffer,
                                                             imgSize,
                                                             paletteData.get(),
                                                             NW_CUBE_FACE_POS_X,
                                                             bmpData);
                    }
                    dataOffset += imgSize;
                }

                Gdiplus::Rect drawRect(0,0,pPreviewBitmap->GetWidth(),pPreviewBitmap->GetHeight());

                pTexture->CreateColorMap(bmpData, ORIGINAL_FORMAT_NOT_USED, drawRect);

                if (pTexture->HasAlpha())
                {
                    pTexture->CreateAlphaMap(bmpData, ORIGINAL_FORMAT_NOT_USED, drawRect);
                }

                pPreviewBitmap->UnlockBits(&bmpData);
            }
        }
        else if (strncmp(dataBlockHeader.dataBlockID,"nw4r_txd",8)==0) // Texel data
        {
            dataOffset  = iCurBufferPos;
            imgDataSize = dataSize;
        }
        else if (strncmp(dataBlockHeader.dataBlockID,"nw4r_pnm",8)==0) // Palette name
        {
            char paletteName[128];
            if (dataSize > 0 && dataSize < 128)
            {
                memcpy(paletteName, pTGABuffer+iCurBufferPos, dataSize);
                paletteName[dataBlockHeader.dataBlockSize] = '\0';
            }
        }
        else if (strncmp(dataBlockHeader.dataBlockID,"nw4r_pfm",8)==0) // Palette format
        {
            char paletteFormatName[16];
            if (dataSize > 0 && dataSize < 16)
            {
                memcpy(paletteFormatName, pTGABuffer+iCurBufferPos, dataSize);
                paletteFormatName[dataSize] = '\0';
                paletteFormat = NWGetTextureFormatFromString(paletteFormatName);
            }

            if (paletteFormat==NW_IMG_FMT_NONE)
            {
                return false;
            } // End if
        }
        else if (strncmp(dataBlockHeader.dataBlockID,"nw4r_pld",8)==0) // Palette data
        {
            if (dataSize > 0 && dataSize < 16384 * 4)
            {
                memcpy(paletteData.get(), pTGABuffer+iCurBufferPos, dataSize);
            }
            else
            {
                return false;
            }
        }
        else if (strncmp(dataBlockHeader.dataBlockID,"nw4r_mpl",8)==0) // Mipmap levels
        {
            char mipCountStr[8];
            memset(mipCountStr,0,8);

            if (dataSize<4)
            {
                memcpy(mipCountStr,pTGABuffer+iCurBufferPos,dataSize);
                numMipmaps = atoi(mipCountStr);
            } // End if

            if (numMipmaps<=0)
                numMipmaps = 1;
        }
        else if (strncmp(dataBlockHeader.dataBlockID,"nw4r_gnm",8)==0) // Tool name
        {
        }
        else if (strncmp(dataBlockHeader.dataBlockID,"nw4r_gvr",8)==0) // Tool version
        {
        }
        else if (strncmp(dataBlockHeader.dataBlockID,"nw4r_psh",8)==0) // Photoshop private data
        {
        }
        else if (strncmp(dataBlockHeader.dataBlockID,"nw4r_eff",8)==0) // Effect maker private data
        {
        }

        iCurBufferPos += dataSize;
    } // End while


    return true;
} // End of ReadNW4RTGA for CNWTGALoader

//------------------------------------------------------------------------------
// Load regular TGA
//------------------------------------------------------------------------------
bool CNWTGALoader::ReadNormalTGA( CNWTexture *pTexture,
                                  unsigned char *pTGABuffer, const TGAHeader &fileHeader,
                                  int fileSize, int &iCurBufferPos,
                                  bool bLoadPreview )
{
    NW_USE_VAR(pTGABuffer);
    NW_USE_VAR(fileSize);
    NW_USE_VAR(iCurBufferPos);
    NW_USE_VAR(bLoadPreview);
    NW_USE_VAR(fileHeader.bits);

    if (pTexture==NULL)
        return false;

    int width  = (int)fileHeader.width;
    int height = (int)fileHeader.height;

    NW_IMG_TYPE textureType = NW_IMG_TYPE_2D;

    NW_IMG_FMT format = NW_IMG_FMT_NONE;

    switch (fileHeader.bits)
    {
        case 8 :
            format = NW_IMG_FMT_L8;
            break;

        case 16 :
            format = NW_IMG_FMT_RGB5_A1;
            break;

        case 24 :
            format = NW_IMG_FMT_RGB8;
            break;

        case 32 :
            format = NW_IMG_FMT_RGBA8;
            break;
    } // End switch

    if (format==NW_IMG_FMT_NONE) // Unknown format?
    {
        return false;
    } // End if

    pTexture->SetDescription(textureType,format,NW_IMG_FMT_NONE,width,height,0,1);

    if (bLoadPreview)
    {
        /*
        TGA_IMAGE_TYPE_NONE
        TGA_IMAGE_TYPE_UNCOMPRESSED_COLORMAP
        TGA_IMAGE_TYPE_UNCOMPRESSED_RGB
        TGA_IMAGE_TYPE_UNCOMPRESSED_BW
        TGA_IMAGE_TYPE_RLE_COLORMAP
        TGA_IMAGE_TYPE_RLE_RGB
        TGA_IMAGE_TYPE_COMPRESSED_BW
        TGA_IMAGE_TYPE_COMPRESSED_COLORMAP
        TGA_IMAGE_TYPE_COMPRESSED_COLORMAP_4PASS
        */

        switch (fileHeader.imageType)
        {
            case TGA_IMAGE_TYPE_UNCOMPRESSED_COLORMAP:
                ReadNormalTGA_UncompressedColormap(pTexture,pTGABuffer,fileHeader,fileSize,iCurBufferPos);
                break;
            case TGA_IMAGE_TYPE_UNCOMPRESSED_BW:
                ReadNormalTGA_UncompressedBW(pTexture,pTGABuffer,fileHeader,fileSize,iCurBufferPos);
                break;
            case TGA_IMAGE_TYPE_COMPRESSED_BW:
                ReadNormalTGA_RLEBW(pTexture,pTGABuffer,fileHeader,fileSize,iCurBufferPos);
                break;
            case TGA_IMAGE_TYPE_UNCOMPRESSED_RGB:
                ReadNormalTGA_UncompressedRGB(pTexture,pTGABuffer,fileHeader,fileSize,iCurBufferPos);
                break;
            case TGA_IMAGE_TYPE_RLE_RGB:
                ReadNormalTGA_RLERGB(pTexture,pTGABuffer,fileHeader,fileSize,iCurBufferPos);
                break;
            default:
                // Format is not managed :-(
                return false;
        }
    } // End if

    pTexture->ComputeImageDataSize();

    return true;
} // End of ReadNormalTGA for CNWTGALoader


//------------------------------------------------------------------------------
bool CNWTGALoader::ReadNormalTGA_UncompressedColormap( CNWTexture *pTexture,
                                                       unsigned char *pTGABuffer, const TGAHeader &fileHeader,
                                                       int fileSize, int &iCurBufferPos )
{
    // Skip over colormap
    int colorMapPos = iCurBufferPos;
    int colorMapBitSize = 0;
    int colorMapTableSize = 0;
    if (fileHeader.colourMapType==1)
    {
        colorMapBitSize = ((int)fileHeader.colourMapBits/8);
        colorMapTableSize = colorMapBitSize * fileHeader.colourMapLength;
    } // End if
    else if (fileHeader.colourMapType!=0)
    {
        // If not 1 or 0? then skip entire image
        return false;
    } // End else if

    iCurBufferPos += colorMapTableSize;
    if (iCurBufferPos>=fileSize) // Overflow?
        return false;

    // Create Texture
    if (pTexture->CreatePreviewImage()==false)
        return false;

    Gdiplus::Bitmap *pPreviewBitmap = pTexture->GetPreviewBitmap();
    if (pPreviewBitmap==NULL)
        return false;

    // Lock bits to prepare preview image
    Gdiplus::Rect lockRect(0,0,pPreviewBitmap->GetWidth(),pPreviewBitmap->GetHeight());
    Gdiplus::BitmapData bmpData;
    if (pPreviewBitmap->LockBits(&lockRect,Gdiplus::ImageLockModeWrite,
                                 pPreviewBitmap->GetPixelFormat(),
                                 &bmpData)!=Gdiplus::Ok)
    {
        return false;
    } // End if

    // Read pixels
    int numPixles = pTexture->GetWidth() * pTexture->GetHeight();

    unsigned char *pDestBuffer = (unsigned char*)bmpData.Scan0;

    // Note : Destination pixels are organized in B,G,R,A order ( from low to high bytes )
    //
    // pDestPixel[0] is Blue
    // pDestPixel[1] is Green
    // pDestPixel[2] is Red
    // pDestPixel[3] is Alpha
    //

    // Scan line must be flipped
    // Up-side down

    int width = pTexture->GetWidth();
    int height = pTexture->GetHeight();
    int stX = (pPreviewBitmap->GetWidth() - width) / 2;
    int stY = (pPreviewBitmap->GetHeight() - height) / 2;

    bool bFlipH = ((fileHeader.descriptor&(0x01<<4))!=0);
    bool bFlipV = ((fileHeader.descriptor&(0x01<<5))!=0);

    if (fileHeader.bits==8)
    {
        unsigned char *pSrcPixels   = (unsigned char*)(pTGABuffer+iCurBufferPos);
        unsigned char *pSrcColorMaps = (unsigned char*)(pTGABuffer+colorMapPos);

        int i=0;
        for(int v=0;v<height;v++)
        {
            for(int u=0;u<width;u++)
            {
                i++;
                int x=u;
                int y=v;

                if (bFlipH)
                    x = width - x -1;
                if (bFlipV==false) // In normal case, bottom-up
                    y = height - y - 1;

                unsigned char srcPixel   = (unsigned char)(pSrcPixels[i]);
                unsigned char *pDestPixel = pDestBuffer+
                    (stX + x) * 4 +
                    (stY + y) * bmpData.Stride;

                pDestPixel[0] = (unsigned char)pSrcColorMaps[(srcPixel * colorMapBitSize) + 0];
                pDestPixel[1] = (unsigned char)pSrcColorMaps[(srcPixel * colorMapBitSize) + 1];
                pDestPixel[2] = (unsigned char)pSrcColorMaps[(srcPixel * colorMapBitSize) + 2];
                pDestPixel[3] = (unsigned char)255;
            }

            // CPU の占有を防ぐ
            Sleep(0);
        } // End for

        iCurBufferPos += (numPixles*1);
    } // End if

    Gdiplus::Rect drawRect(stX,stY,width,height);

    // Finalize
    pTexture->CreateColorMap(bmpData, ORIGINAL_FORMAT_NOT_USED, drawRect);
    if (pTexture->HasAlpha())
    {
        pTexture->CreateAlphaMap(bmpData, ORIGINAL_FORMAT_NOT_USED, drawRect);
    } // End if

    pPreviewBitmap->UnlockBits(&bmpData);

    return true;
}

//------------------------------------------------------------------------------
bool CNWTGALoader::ReadNormalTGA_UncompressedBW( CNWTexture *pTexture,
                                                 unsigned char *pTGABuffer, const TGAHeader &fileHeader,
                                                 int fileSize, int &iCurBufferPos )
{
    // Skip over colormap
    int colorMapTableSize = 0;
    if (fileHeader.colourMapType==1)
    {
        colorMapTableSize = ((int)fileHeader.colourMapBits/8) * fileHeader.colourMapLength;
    } // End if
    else if (fileHeader.colourMapType!=0)
    {
        // If not 1 or 0? then skip entire image
        return false;
    } // End else if

    iCurBufferPos += colorMapTableSize;
    if (iCurBufferPos>=fileSize) // Overflow?
        return false;

    // Create Texture
    if (pTexture->CreatePreviewImage()==false)
        return false;

    Gdiplus::Bitmap *pPreviewBitmap = pTexture->GetPreviewBitmap();
    if (pPreviewBitmap==NULL)
        return false;

    // Lock bits to prepare preview image
    Gdiplus::Rect lockRect(0,0,pPreviewBitmap->GetWidth(),pPreviewBitmap->GetHeight());
    Gdiplus::BitmapData bmpData;
    if (pPreviewBitmap->LockBits(&lockRect,Gdiplus::ImageLockModeWrite,
                                 pPreviewBitmap->GetPixelFormat(),
                                 &bmpData)!=Gdiplus::Ok)
    {
        return false;
    } // End if

    // Read pixels
    int numPixles = pTexture->GetWidth() * pTexture->GetHeight();

    unsigned char *pDestBuffer = (unsigned char*)bmpData.Scan0;

    // Note : Destination pixels are organized in B,G,R,A order ( from low to high bytes )
    //
    // pDestPixel[0] is Blue
    // pDestPixel[1] is Green
    // pDestPixel[2] is Red
    // pDestPixel[3] is Alpha
    //

    // Scan line must be flipped
    // Up-side down

    int width = pTexture->GetWidth();
    int height = pTexture->GetHeight();
    int stX = (pPreviewBitmap->GetWidth() - width) / 2;
    int stY = (pPreviewBitmap->GetHeight() - height) / 2;

    bool bFlipH = ((fileHeader.descriptor&(0x01<<4))!=0);
    bool bFlipV = ((fileHeader.descriptor&(0x01<<5))!=0);

    if (fileHeader.bits==8)
    {
        unsigned char *pSrcPixels   = (unsigned char*)(pTGABuffer+iCurBufferPos);

        int i=0;
        for(int v=0;v<height;v++)
        {
            for(int u=0;u<width;u++)
            {
                i++;
                int x=u;
                int y=v;

                if (bFlipH)
                    x = width - x -1;
                if (bFlipV==false) // In normal case, bottom-up
                    y = height - y - 1;

                unsigned char srcPixel   = (unsigned char)(pSrcPixels[i]);
                unsigned char *pDestPixel = pDestBuffer+
                    (stX + x) * 4 +
                    (stY + y) * bmpData.Stride;

                pDestPixel[0] = (unsigned char)srcPixel;
                pDestPixel[1] = (unsigned char)srcPixel;
                pDestPixel[2] = (unsigned char)srcPixel;
                pDestPixel[3] = (unsigned char)0xFF;
            }

            // CPU の占有を防ぐ
            Sleep(0);
        } // End for

        iCurBufferPos += (numPixles*1);
    } // End if

    Gdiplus::Rect drawRect(stX,stY,width,height);

    // Finalize
    pTexture->CreateColorMap(bmpData, ORIGINAL_FORMAT_NOT_USED, drawRect);
    if (pTexture->HasAlpha())
    {
        pTexture->CreateAlphaMap(bmpData, ORIGINAL_FORMAT_NOT_USED, drawRect);
    } // End if

    pPreviewBitmap->UnlockBits(&bmpData);

    return true;
} // End of ReadNormalTGA_UncompressedBW for CNWTGALoader


//------------------------------------------------------------------------------
bool CNWTGALoader::ReadNormalTGA_RLEBW( CNWTexture *pTexture,
                                        unsigned char *pTGABuffer, const TGAHeader &fileHeader,
                                        int fileSize, int &iCurBufferPos )
{
    // Skip over colormap
    int colorMapTableSize = 0;
    if (fileHeader.colourMapType==1)
    {
        colorMapTableSize = ((int)fileHeader.colourMapBits/8) * fileHeader.colourMapLength;
    } // End if
    else if (fileHeader.colourMapType!=0)
    {
        // If not 1 or 0? then skip entire image
        return false;
    } // End else if

    iCurBufferPos += colorMapTableSize;
    if (iCurBufferPos>=fileSize) // Overflow?
        return false;

    // Create Texture
    if (pTexture->CreatePreviewImage()==false)
        return false;

    Gdiplus::Bitmap *pPreviewBitmap = pTexture->GetPreviewBitmap();
    if (pPreviewBitmap==NULL)
        return false;

    // Lock bits to prepare preview image
    Gdiplus::Rect lockRect(0,0,pPreviewBitmap->GetWidth(),pPreviewBitmap->GetHeight());
    Gdiplus::BitmapData bmpData;
    if (pPreviewBitmap->LockBits(&lockRect,Gdiplus::ImageLockModeWrite,
                                 pPreviewBitmap->GetPixelFormat(),
                                 &bmpData)!=Gdiplus::Ok)
    {
        return false;
    } // End if

    // Read pixels
    int numPixles = pTexture->GetWidth() * pTexture->GetHeight();
    int i = 0;

    unsigned char *pDestBuffer = (unsigned char*)bmpData.Scan0;

    // Note : Destination pixels are organized in B,G,R,A order ( from low to high bytes )
    //
    // pDestPixel[0] is Blue
    // pDestPixel[1] is Green
    // pDestPixel[2] is Red
    // pDestPixel[3] is Alpha
    //

    int width = pTexture->GetWidth();
    int height = pTexture->GetHeight();
    int stX = (pPreviewBitmap->GetWidth() - width) / 2;
    int stY = (pPreviewBitmap->GetHeight() - height) / 2;

    bool bFlipH = ((fileHeader.descriptor&(0x01<<4))!=0);
    bool bFlipV = ((fileHeader.descriptor&(0x01<<5))!=0);

    if (fileHeader.bits==8)
    {
        unsigned char *pSrcPixels   = (unsigned char*)(pTGABuffer+iCurBufferPos);
        while (i<numPixles)
        {
            unsigned char pixelHeader = (*pSrcPixels); pSrcPixels++;
            if (pixelHeader&0x80)
            {
                int repeatCount = (pixelHeader&0x7F) + 1;

                unsigned char srcPixel = (unsigned char)(*((unsigned char*)pSrcPixels)); pSrcPixels++;

                while (repeatCount>0)
                {
                    int x = i % width;
                    int y = i / width;

                    if (bFlipH)
                        x = width - x -1;
                    if (bFlipV==false) // In normal case, bottom-up
                        y = height - y - 1;

                    unsigned char *pDestPixel = pDestBuffer+
                                                (stX + x) * 4 +
                                                (stY + y) * bmpData.Stride;

                    pDestPixel[0] = (unsigned char)srcPixel;
                    pDestPixel[1] = (unsigned char)srcPixel;
                    pDestPixel[2] = (unsigned char)srcPixel;
                    pDestPixel[3] = (unsigned char)0xFF;

                    i++;
                    repeatCount--;
                } // End while
            } // End if
            else
            {
                int repeatCount = (pixelHeader&0x7F) + 1;

                while (repeatCount>0)
                {
                    int x = i % width;
                    int y = i / width;

                    if (bFlipH)
                        x = width - x -1;
                    if (bFlipV==false) // In normal case, bottom-up
                        y = height - y - 1;

                    unsigned char srcPixel = (unsigned char)(*((unsigned char*)pSrcPixels)); pSrcPixels++;
                    unsigned char *pDestPixel = pDestBuffer+
                                                (stX + x) * 4 +
                                                (stY + y) * bmpData.Stride;

                    pDestPixel[0] = (unsigned char)srcPixel;
                    pDestPixel[1] = (unsigned char)srcPixel;
                    pDestPixel[2] = (unsigned char)srcPixel;
                    pDestPixel[3] = (unsigned char)0xFF;

                    i++;
                    repeatCount--;
                } // End while
            } // End else

            int offset = (int)(pSrcPixels - (unsigned char*)(pTGABuffer+iCurBufferPos));
            iCurBufferPos += offset;
        } // End while

        Sleep(0);
    } // End if

    Gdiplus::Rect drawRect(stX,stY,width,height);

    // Finalize
    pTexture->CreateColorMap(bmpData, ORIGINAL_FORMAT_NOT_USED, drawRect);
    if (pTexture->HasAlpha())
    {
        pTexture->CreateAlphaMap(bmpData, ORIGINAL_FORMAT_NOT_USED, drawRect);
    } // End if

    pPreviewBitmap->UnlockBits(&bmpData);

    return true;
} // End of ReadNormalTGA_RLEBW for CNWTGALoader


//------------------------------------------------------------------------------
bool CNWTGALoader::ReadNormalTGA_UncompressedRGB( CNWTexture *pTexture,
                                                  unsigned char *pTGABuffer, const TGAHeader &fileHeader,
                                                  int fileSize, int &iCurBufferPos )
{
    // Skip over colormap
    int colorMapTableSize = 0;
    if (fileHeader.colourMapType==1)
    {
        colorMapTableSize = ((int)fileHeader.colourMapBits/8) * fileHeader.colourMapLength;
    } // End if
    else if (fileHeader.colourMapType!=0)
    {
        // If not 1 or 0? then skip entire image
        return false;
    } // End else if

    iCurBufferPos += colorMapTableSize;
    if (iCurBufferPos>=fileSize) // Overflow?
        return false;

    // Create Texture
    if (pTexture->CreatePreviewImage()==false)
        return false;

    Gdiplus::Bitmap *pPreviewBitmap = pTexture->GetPreviewBitmap();
    if (pPreviewBitmap==NULL)
        return false;

    // Lock bits to prepare preview image
    Gdiplus::Rect lockRect(0,0,pPreviewBitmap->GetWidth(),pPreviewBitmap->GetHeight());
    Gdiplus::BitmapData bmpData;
    if (pPreviewBitmap->LockBits(&lockRect,Gdiplus::ImageLockModeWrite,
                                 pPreviewBitmap->GetPixelFormat(),
                                 &bmpData)!=Gdiplus::Ok)
    {
        return false;
    } // End if

    // Read pixels
    int numPixles = pTexture->GetWidth() * pTexture->GetHeight();

    unsigned char *pDestBuffer = (unsigned char*)bmpData.Scan0;

    // Note : Destination pixels are organized in B,G,R,A order ( from low to high bytes )
    //
    // pDestPixel[0] is Blue
    // pDestPixel[1] is Green
    // pDestPixel[2] is Red
    // pDestPixel[3] is Alpha
    //

    // Scan line must be flipped
    // Up-side down

    int width = pTexture->GetWidth();
    int height = pTexture->GetHeight();
    int stX = (pPreviewBitmap->GetWidth() - width) / 2;
    int stY = (pPreviewBitmap->GetHeight() - height) / 2;

    bool bFlipH = ((fileHeader.descriptor&(0x01<<4))!=0);
    bool bFlipV = ((fileHeader.descriptor&(0x01<<5))!=0);

    if (fileHeader.bits==8)
    {
        unsigned short *pSrcPixels   = (unsigned short*)(pTGABuffer+iCurBufferPos);

        int i=0;
        for(int v=0;v<height;v++)
        {
            for(int u=0;u<width;u++)
            {
                i++;
                int x=u;
                int y=v;

                if (bFlipH)
                    x = width - x -1;
                if (bFlipV==false) // In normal case, bottom-up
                    y = height - y - 1;

                unsigned char srcPixel   = (unsigned char)(pSrcPixels[i]);
                unsigned char *pDestPixel = pDestBuffer+
                    (stX + x) * 4 +
                    (stY + y) * bmpData.Stride;

                pDestPixel[0] = (unsigned char)srcPixel;
                pDestPixel[1] = (unsigned char)srcPixel;
                pDestPixel[2] = (unsigned char)srcPixel;
                pDestPixel[3] = 0xFF;
            }

            // CPU の占有を防ぐ
            Sleep(0);
        } // End for

        iCurBufferPos += (numPixles*1);
    } // End if
    else if (fileHeader.bits==16)
    {
        unsigned short *pSrcPixels   = (unsigned short*)(pTGABuffer+iCurBufferPos);

        int i=0;
        for(int v=0;v<height;v++)
        {
            for(int u=0;u<width;u++)
            {
                i++;
                int x=u;
                int y=v;

                if (bFlipH)
                    x = width - x -1;
                if (bFlipV==false) // In normal case, bottom-up
                    y = height - y - 1;

                unsigned int srcPixel   = (int)(pSrcPixels[i]);
                unsigned char *pDestPixel = pDestBuffer+
                    (stX + x) * 4 +
                    (stY + y) * bmpData.Stride;

                unsigned int r = (srcPixel>>10) & 0x1F;
                unsigned int g = (srcPixel>>5)  & 0x1F;
                unsigned int b = (srcPixel)     & 0x1F;
                unsigned int a = (srcPixel>>15);
                if (a!=0)
                    a = 0xFF;

                float fR = (float)r / (float)0x1F;
                float fG = (float)g / (float)0x1F;
                float fB = (float)b / (float)0x1F;

                r = (unsigned int)(fR * 255.0f);
                g = (unsigned int)(fG * 255.0f);
                b = (unsigned int)(fB * 255.0f);

                pDestPixel[0] = (unsigned char)b;
                pDestPixel[1] = (unsigned char)g;
                pDestPixel[2] = (unsigned char)r;
                pDestPixel[3] = (unsigned char)a;
            }

            // CPU の占有を防ぐ
            Sleep(0);
        } // End for

        iCurBufferPos += (numPixles*2);
    } // End if
    else if (fileHeader.bits==24)
    {
        unsigned char *pSrcPixels   = (unsigned char*)(pTGABuffer+iCurBufferPos);
        int i=0;
        for(int v=0;v<height;v++)
        {
            for(int u=0;u<width;u++)
            {
                i++;
                int x=u;
                int y=v;

                if (bFlipH)
                    x = width - x -1;
                if (bFlipV==false) // In normal case, bottom-up
                    y = height - y - 1;

                unsigned char *pSrcPixel  = pSrcPixels + (i * 3);
                unsigned char *pDestPixel = pDestBuffer+
                    (stX + x) * 4 +
                    (stY + y) * bmpData.Stride;

                // Src is stored as blue, green, red
                pDestPixel[0] = pSrcPixel[0];
                pDestPixel[1] = pSrcPixel[1];
                pDestPixel[2] = pSrcPixel[2];
                pDestPixel[3] = 0xFF;
            }

            // CPU の占有を防ぐ
            Sleep(0);
        } // End for

        iCurBufferPos += (numPixles*3);
    } // End else if
    else if (fileHeader.bits==32)
    {
        unsigned char *pSrcPixels   = (unsigned char*)(pTGABuffer+iCurBufferPos);
        int i=0;
        for(int v=0;v<height;v++)
        {
            for(int u=0;u<width;u++)
            {
                i++;
                int x=u;
                int y=v;

                if (bFlipH)
                    x = width - x -1;

                if (bFlipV==false) // In normal case, bottom-up
                    y = height - y - 1;

                unsigned char *pSrcPixel  = pSrcPixels + (i * 4);
                unsigned char *pDestPixel = pDestBuffer+
                    (stX + x) * 4 +
                    (stY + y) * bmpData.Stride;

                // Src is stored as blue, green, red
                pDestPixel[0] = pSrcPixel[0];
                pDestPixel[1] = pSrcPixel[1];
                pDestPixel[2] = pSrcPixel[2];
                pDestPixel[3] = pSrcPixel[3];
            }

            // CPU の占有を防ぐ
            Sleep(0);
        } // End for

        iCurBufferPos += (numPixles*4);
    } // End else if

    Gdiplus::Rect drawRect(stX,stY,width,height);

    // Finalize
    pTexture->CreateColorMap(bmpData, ORIGINAL_FORMAT_NOT_USED, drawRect);
    if (pTexture->HasAlpha())
    {
        pTexture->CreateAlphaMap(bmpData, ORIGINAL_FORMAT_NOT_USED, drawRect);
    } // End if

    pPreviewBitmap->UnlockBits(&bmpData);

    return true;
} // End of ReadNormalTGA_UncompressedRGB for CNWTGALoader


//------------------------------------------------------------------------------
bool CNWTGALoader::ReadNormalTGA_RLERGB( CNWTexture *pTexture,
                                         unsigned char *pTGABuffer, const TGAHeader &fileHeader,
                                         int fileSize, int &iCurBufferPos )
{
    // Skip over colormap
    int colorMapTableSize = 0;
    if (fileHeader.colourMapType==1)
    {
        colorMapTableSize = ((int)fileHeader.colourMapBits/8) * fileHeader.colourMapLength;
    } // End if
    else if (fileHeader.colourMapType!=0)
    {
        // If not 1 or 0? then skip entire image
        return false;
    } // End else if

    iCurBufferPos += colorMapTableSize;
    if (iCurBufferPos>=fileSize) // Overflow?
        return false;

    // Create Texture
    if (pTexture->CreatePreviewImage()==false)
        return false;

    Gdiplus::Bitmap *pPreviewBitmap = pTexture->GetPreviewBitmap();
    if (pPreviewBitmap==NULL)
        return false;

    // Lock bits to prepare preview image
    Gdiplus::Rect lockRect(0,0,pPreviewBitmap->GetWidth(),pPreviewBitmap->GetHeight());
    Gdiplus::BitmapData bmpData;
    if (pPreviewBitmap->LockBits(&lockRect,Gdiplus::ImageLockModeWrite,
                                 pPreviewBitmap->GetPixelFormat(),
                                 &bmpData)!=Gdiplus::Ok)
    {
        return false;
    } // End if

    // Read pixels
    int numPixles = pTexture->GetWidth() * pTexture->GetHeight();
    int i = 0;

    unsigned char *pDestBuffer = (unsigned char*)bmpData.Scan0;

    // Note : Destination pixels are organized in B,G,R,A order ( from low to high bytes )
    //
    // pDestPixel[0] is Blue
    // pDestPixel[1] is Green
    // pDestPixel[2] is Red
    // pDestPixel[3] is Alpha
    //

    int width = pTexture->GetWidth();
    int height = pTexture->GetHeight();
    int stX = (pPreviewBitmap->GetWidth() - width) / 2;
    int stY = (pPreviewBitmap->GetHeight() - height) / 2;

    bool bFlipH = ((fileHeader.descriptor&(0x01<<4))!=0);
    bool bFlipV = ((fileHeader.descriptor&(0x01<<5))!=0);

    if (fileHeader.bits==16)
    {
        unsigned char *pSrcPixels   = (unsigned char*)(pTGABuffer+iCurBufferPos);
        while (i<numPixles)
        {
            unsigned char pixelHeader = (*pSrcPixels); pSrcPixels++;
            if (pixelHeader&0x80)
            {
                int repeatCount = (pixelHeader&0x7F) + 1;

                unsigned int srcPixel = (int)(*((unsigned short*)pSrcPixels)); pSrcPixels += 2;
                unsigned int r = (srcPixel>>10) & 0x1F;
                unsigned int g = (srcPixel>>5)  & 0x1F;
                unsigned int b = (srcPixel)     & 0x1F;
                unsigned int a = (srcPixel>>15);
                if (a!=0)
                    a = 0xFF;

                float fR = (float)r / (float)0x1F;
                float fG = (float)g / (float)0x1F;
                float fB = (float)b / (float)0x1F;

                r = (unsigned int)(fR * 255.0f);
                g = (unsigned int)(fG * 255.0f);
                b = (unsigned int)(fB * 255.0f);

                while (repeatCount>0)
                {
                    int x = i % width;
                    int y = i / width;

                    if (bFlipH)
                        x = width - x -1;
                    if (bFlipV==false) // In normal case, bottom-up
                        y = height - y - 1;

                    unsigned char *pDestPixel = pDestBuffer+
                                                (stX + x) * 4 +
                                                (stY + y) * bmpData.Stride;

                    pDestPixel[0] = (unsigned char)b;
                    pDestPixel[1] = (unsigned char)g;
                    pDestPixel[2] = (unsigned char)r;
                    pDestPixel[3] = (unsigned char)a;

                    i++;
                    repeatCount--;
                } // End while
            } // End if
            else
            {
                int repeatCount = (pixelHeader&0x7F) + 1;

                while (repeatCount>0)
                {
                    int x = i % width;
                    int y = i / width;

                    if (bFlipH)
                        x = width - x -1;
                    if (bFlipV==false) // In normal case, bottom-up
                        y = height - y - 1;

                    unsigned int srcPixel = (int)(*((unsigned short*)pSrcPixels)); pSrcPixels += 2;
                    unsigned char *pDestPixel = pDestBuffer+
                                                (stX + x) * 4 +
                                                (stY + y) * bmpData.Stride;

                    unsigned int r = (srcPixel>>10) & 0x1F;
                    unsigned int g = (srcPixel>>5)  & 0x1F;
                    unsigned int b = (srcPixel)     & 0x1F;
                    unsigned int a = (srcPixel>>15);
                    if (a!=0)
                        a = 0xFF;

                    float fR = (float)r / (float)0x1F;
                    float fG = (float)g / (float)0x1F;
                    float fB = (float)b / (float)0x1F;

                    r = (unsigned int)(fR * 255.0f);
                    g = (unsigned int)(fG * 255.0f);
                    b = (unsigned int)(fB * 255.0f);

                    pDestPixel[0] = (unsigned char)b;
                    pDestPixel[1] = (unsigned char)g;
                    pDestPixel[2] = (unsigned char)r;
                    pDestPixel[3] = (unsigned char)a;

                    i++;
                    repeatCount--;
                } // End while
            } // End else

            int offset = (int)(pSrcPixels - (unsigned char*)(pTGABuffer+iCurBufferPos));
            iCurBufferPos += offset;

            Sleep(0);
        } // End while
    } // End if

    else if (fileHeader.bits==24)
    {
        unsigned char *pSrcPixels   = (unsigned char*)(pTGABuffer+iCurBufferPos);
        while (i<numPixles)
        {
            unsigned char pixelHeader = (*pSrcPixels); pSrcPixels++;
            if (pixelHeader&0x80)
            {
                int repeatCount = (pixelHeader&0x7F) + 1;

                unsigned char *pSrcPixel = pSrcPixels; pSrcPixels += 3;

                while (repeatCount>0)
                {
                    int x = i % width;
                    int y = i / width;

                    if (bFlipH)
                        x = width - x -1;
                    if (bFlipV==false) // In normal case, bottom-up
                        y = height - y - 1;

                    unsigned char *pDestPixel = pDestBuffer+
                                                (stX + x) * 4 +
                                                (stY + y) * bmpData.Stride;

                    // Src is stored as blue, green, red
                    pDestPixel[0] = pSrcPixel[0];
                    pDestPixel[1] = pSrcPixel[1];
                    pDestPixel[2] = pSrcPixel[2];
                    pDestPixel[3] = 0xFF;

                    i++;
                    repeatCount--;
                } // End while
            } // End if
            else
            {
                int repeatCount = (pixelHeader&0x7F) + 1;

                while (repeatCount>0)
                {
                    int x = i % width;
                    int y = i / width;

                    if (bFlipH)
                        x = width - x -1;
                    if (bFlipV==false) // In normal case, bottom-up
                        y = height - y - 1;

                    unsigned char *pSrcPixel = pSrcPixels; pSrcPixels += 3;
                    unsigned char *pDestPixel = pDestBuffer+
                                                (stX + x) * 4 +
                                                (stY + y) * bmpData.Stride;

                    // Src is stored as blue, green, red
                    pDestPixel[0] = pSrcPixel[0];
                    pDestPixel[1] = pSrcPixel[1];
                    pDestPixel[2] = pSrcPixel[2];
                    pDestPixel[3] = 0xFF;

                    i++;
                    repeatCount--;
                } // End while
            } // End else

            Sleep(0);
        } // End while

        int offset = (int)(pSrcPixels - (unsigned char*)(pTGABuffer+iCurBufferPos));
        iCurBufferPos += offset;
    } // End else if

    else if (fileHeader.bits==32)
    {
        unsigned char *pSrcPixels   = (unsigned char*)(pTGABuffer+iCurBufferPos);
        while (i<numPixles)
        {
            unsigned char pixelHeader = (*pSrcPixels); pSrcPixels++;
            if (pixelHeader&0x80)
            {
                int repeatCount = (pixelHeader&0x7F) + 1;

                unsigned char *pSrcPixel = pSrcPixels; pSrcPixels += 4;

                while (repeatCount>0)
                {
                    int x = i % width;
                    int y = i / width;

                    if (bFlipH)
                        x = width - x -1;
                    if (bFlipV==false) // In normal case, bottom-up
                        y = height - y - 1;

                    unsigned char *pDestPixel = pDestBuffer+
                                                (stX + x) * 4 +
                                                (stY + y) * bmpData.Stride;

                    // Src is stored as blue, green, red
                    pDestPixel[0] = pSrcPixel[0];
                    pDestPixel[1] = pSrcPixel[1];
                    pDestPixel[2] = pSrcPixel[2];
                    pDestPixel[3] = pSrcPixel[3];

                    i++;
                    repeatCount--;
                } // End while
            } // End if
            else
            {
                int repeatCount = (pixelHeader&0x7F) + 1;

                while (repeatCount>0)
                {
                    int x = i % width;
                    int y = i / width;

                    if (bFlipH)
                        x = width - x -1;
                    if (bFlipV==false) // In normal case, bottom-up
                        y = height - y - 1;

                    unsigned char *pSrcPixel = pSrcPixels; pSrcPixels += 4;
                    unsigned char *pDestPixel = pDestBuffer+
                                                (stX + x) * 4 +
                                                (stY + y) * bmpData.Stride;

                    // Src is stored as blue, green, red
                    pDestPixel[0] = pSrcPixel[0];
                    pDestPixel[1] = pSrcPixel[1];
                    pDestPixel[2] = pSrcPixel[2];
                    pDestPixel[3] = pSrcPixel[3];

                    i++;
                    repeatCount--;
                } // End while
            } // End else

            Sleep(0);
        } // End while

        int offset = (int)(pSrcPixels - (unsigned char*)(pTGABuffer+iCurBufferPos));
        iCurBufferPos += offset;
    } // End else if

    Gdiplus::Rect drawRect(stX,stY,width,height);

    // Finalize
    pTexture->CreateColorMap(bmpData, ORIGINAL_FORMAT_NOT_USED, drawRect);
    if (pTexture->HasAlpha())
    {
        pTexture->CreateAlphaMap(bmpData, ORIGINAL_FORMAT_NOT_USED, drawRect);
    } // End if

    pPreviewBitmap->UnlockBits(&bmpData);

    return true;
} // End of ReadNormalTGA_RLERGB for CNWTGALoader
