﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "ShellExtension_PCH.h"

#ifdef NW_FOR_FTX

#include "ShellExtension_Utility.h"
#include "ShellExtension_Type.h"
#include "ShellExtension_Texture.h"
#include "ImageLoader/ShellExtension_ImageLoader.h"
#include "ImageLoader/ShellExtension_FTXLoader.h"
#include "ImageLoader/Decoder/Encoder.h"

//==============================================================================
//
// Implementation of CNWFTXLoader
//
//==============================================================================
CNWFTXLoader::CNWFTXLoader() :
    CNWImageLoader()
{
} // End of Constructor for CNWFTXLoader


//------------------------------------------------------------------------------
CNWFTXLoader::~CNWFTXLoader()
{
} // End of Destructor for CNWFTXLoader


//------------------------------------------------------------------------------
// Decode text into images BYTE format
//------------------------------------------------------------------------------
std::unique_ptr<unsigned char[]> CNWFTXLoader::DecodeStreamTextToBuffer( const unsigned char* srcBuf, int srcSize,
                                                       int& dstSize)
{
    //--------------------------------------------------------------------------
    // alloc dst buffer
    //--------------------------------------------------------------------------
    int allocSize = srcSize;
    if (allocSize<=0)
        return NULL;

    std::unique_ptr<unsigned char[]> dstBuf(new unsigned char[allocSize]);
    if (dstBuf==NULL)
        return NULL;

    //--------------------------------------------------------------------------
    // decode
    //--------------------------------------------------------------------------
    const unsigned char* src = srcBuf;
    const unsigned char* srcEnd = srcBuf + srcSize;
    unsigned char* dst = dstBuf.get();
    dstSize = 0;

    int iCharCount = 0;
    int readByte   = 0;

    while (src < srcEnd)
    {
        char c = *src++;
        if ( (c==' ') ||
             (c=='\t') ||
             (c=='\a') ||
             (c=='\n') )
        {
            if (iCharCount==1)
            {
                (*dst) = (unsigned char)readByte;
                dst++;
            } // End if

            iCharCount = 0;
            readByte   = 0;
            continue;
        } // End while

        if ( (c>='a') && (c<='f') )
            c = c - (char)'a' + 10;
        else if ( (c>='A') && (c<='F') )
            c = c - (char)'A' + 10;
        else
            c = c - (char)'0';

        if (iCharCount==0)
        {
            readByte = (int)c;
            iCharCount = 1;
        } // End if
        else
        {
            readByte   <<= 4;
            readByte   |= (int)c;
            iCharCount = 1;

            (*dst) = (unsigned char)readByte;
            dst++;

            iCharCount = 0;
        } // End if
    } // End while

    dstSize = (INT)(dst - dstBuf.get());
    return dstBuf;
} // End of DecodeStreamTextToBuffer for CNWFTXLoader

//------------------------------------------------------------------------------
// Decode text into images float format
//------------------------------------------------------------------------------
std::unique_ptr<unsigned char[]> CNWFTXLoader::DecodeStreamTextToFloatBuffer( const unsigned char* srcBuf,
                                                            int srcSize,
                                                            int format,
                                                            int& dstSize)
{
    //--------------------------------------------------------------------------
    // alloc dst buffer
    //--------------------------------------------------------------------------
    int allocSize = srcSize;
    int sizeofFloat = allocSize / 4;
    if (allocSize<=0)
        return NULL;

    std::unique_ptr<unsigned char[]> dstBuf(new unsigned char[allocSize]);
    if (dstBuf==NULL)
        return NULL;

    //--------------------------------------------------------------------------
    // decode
    //--------------------------------------------------------------------------
    char* src = (char*)srcBuf;
    //const unsigned char* srcEnd = srcBuf + srcSize;
    float* dst = (float*)dstBuf.get();
    dstSize = 0;
    //int sum=0;
    char *next_token = NULL;

    src = strtok_s(src, "\n", &next_token);
    if (format==3)
    {
        while (dstSize < sizeofFloat)
        {

            int read = sscanf_s(src, "%f %f %f\n", &dst[0], &dst[1], &dst[2]);
            if (read!=3)
            {
                // データが壊れている？
                return NULL;
            }
            dst = dst + read;
            dstSize += read;
            src = strtok_s(NULL,"\n", &next_token);
        } // End while
    }
    else
    {
        while (dstSize < sizeofFloat)
        {

            int read = sscanf_s(src, "%f %f %f %f\n", &dst[0], &dst[1], &dst[2], &dst[3]);
            if (read!=4)
            {
                // データが壊れている？
                return NULL;
            }
            dst = dst + read;
            dstSize += read;
            src = strtok_s(NULL,"\n", &next_token);
        } // End while
    }


    dstSize = (INT)((const unsigned char*)dst - dstBuf.get());
    return dstBuf;
} // End of DecodeStreamTextToBuffer for CNWFTXLoader



//------------------------------------------------------------------------------
// Read XML section to buffer for parsing
//------------------------------------------------------------------------------
bool CNWFTXLoader::ReadXMLSectionToBuffer( const unsigned char *pRawBuffer,
                                           int iBufferSize,
                                           VARIANT *pXMLBuffer )
{
    const unsigned char *pSrc = pRawBuffer;

    int iSize = 0;

    // Skip the BOM bytes
    if ( pSrc[0]==0xEF &&
         pSrc[1]==0xBB &&
         pSrc[2]==0xBF )
    {
        iSize += 3;
        pSrc  += 3;
    } // End if

    // Count the buffer size
    while ( pSrc<(pRawBuffer + iBufferSize) )
    {
        // Is this byte NULL?
        if ( *pSrc==0x00 )
        {
            break;
        } // End if

        if ( (*pSrc & 0x80)==0x00 ) // 0x0xxxxxxx : 1 Byte
        {
            ++iSize;
            ++pSrc;
        } // End if
        else if ( (*pSrc & 0xE0)==0xC0 ) // 0x110xxxxx : 2 Bytes
        {
            iSize += 2;
            pSrc  += 2;
        } // End else if
        else if ( (*pSrc & 0xF0)==0xE0 ) // 0x1110xxxx : 3 Bytes
        {
            iSize += 3;
            pSrc  += 3;
        } // End else if
        else if ( (*pSrc & 0xF8)==0xF0 ) // 0x11110xxx : 4 Bytes
        {
            iSize += 4;
            pSrc  += 4;
        } // End else if
        else if ( (*pSrc & 0xFC)==0xF8 ) // 0x111110xx : 5 Bytes
        {
            iSize += 5;
            pSrc  += 5;
        } // End else if
        else if ( (*pSrc & 0xFE)==0xFC ) // 0x1111110x : 6 Bytes
        {
            iSize += 6;
            pSrc  += 6;
        } // End else if
        else
        {
            // Invalid file
            return false;
        } // End else
    }

    if ( iSize<=0 )
        return false;

    SAFEARRAYBOUND bound;
    bound.cElements = iSize;
    bound.lLbound   = 0;

    // Create the buffer.
    pXMLBuffer->vt     = VT_ARRAY | VT_UI1;
    pXMLBuffer->parray = SafeArrayCreate( VT_UI1, 1, &bound );

    unsigned char *pBuffer = NULL;
    SafeArrayAccessData( pXMLBuffer->parray, (void**)&pBuffer );

    memcpy_s( pBuffer, iSize, pRawBuffer, iSize );

    SafeArrayUnaccessData( pXMLBuffer->parray );

    return true;
} // End of ReadXMLSectionToBuffer for CNWFTXLoader


//------------------------------------------------------------------------------
// Load from image file onto pTexture
//
// - If bLoadPreview is true, create preview image for icon
//   If bLoadPreview is false, just load information ( format, size etc )
//------------------------------------------------------------------------------
bool CNWFTXLoader::Load( CNWTexture *pTexture,
                         const WCHAR *szFilePath,
                         bool bLoadPreview )
{

    pTexture->SetAsTGAFile(false,NW_TGA_NONE);

    //--------------------------------------------------------------------------
    // Load the whole file to our buffer
    //--------------------------------------------------------------------------
    FILE *pFile = NULL;
    WCHAR szModeStr[8];
    szModeStr[0] = (WCHAR)'r';
    szModeStr[1] = (WCHAR)'b';
    szModeStr[2] = (WCHAR)0;

    _wfopen_s(&pFile,szFilePath,szModeStr);
    if (pFile==NULL)
        return false;

    std::unique_ptr<std::FILE, decltype(&std::fclose)> fp(pFile, &std::fclose);

    fseek(pFile,0,SEEK_END);
    int fileSize = ftell(pFile);
    fseek(pFile,0,SEEK_SET);

    std::unique_ptr<unsigned char[]> pBuffer(new unsigned char[fileSize]);
    if (pBuffer==NULL)
    {
        return false;
    } // End if

    if (fread(pBuffer.get(),fileSize,1,pFile)!=1)
    {
        return false;
    } // End if

    // Close file
    fp.reset();

    VARIANT xmlChunk;
    if ( ReadXMLSectionToBuffer( pBuffer.get(), fileSize, &xmlChunk )==false )
    {
        return false;
    }

    auto destroy = [](SAFEARRAY* p) { SafeArrayDestroy(p); };
    std::unique_ptr<SAFEARRAY, decltype(destroy)> p(xmlChunk.parray, destroy);

    //--------------------------------------------------------------------------
    // Create document
    //--------------------------------------------------------------------------
    MSXML2::IXMLDOMDocument2Ptr pDoc;
    HRESULT hr = pDoc.CreateInstance(__uuidof(MSXML2::DOMDocument30));
    if (FAILED(hr))
    {
        return false;
    } // End if

    pDoc->async = VARIANT_FALSE; // (default = TRUE)
    pDoc->validateOnParse = VARIANT_FALSE; // disable schema check (default = TRUE)
    pDoc->resolveExternals = VARIANT_FALSE; // (default = TRUE)
    VARIANT_BOOL result = pDoc->load(xmlChunk);
    if (result != VARIANT_TRUE)
    {
        return false;
    } // End if

    //--------------------------------------------------------------------------
    // Get texture element
    //--------------------------------------------------------------------------
    MSXML2::IXMLDOMNodePtr rootNode = NWXMLFindChildNode(pDoc, "nw4f_3dif");
    if (rootNode == NULL)
    {
        return false;
    }

    MSXML2::IXMLDOMNodePtr textureNode = NWXMLFindChildNode(rootNode, "texture");
    if (textureNode == NULL)
    {
        return false;
    }

    MSXML2::IXMLDOMNodePtr textureInfoNode = NWXMLFindChildNode(textureNode, "texture_info");
    if (textureInfoNode == NULL)
    {
        return false;
    }

    MSXML2::IXMLDOMNodePtr originalImageNode = NULL;
    MSXML2::IXMLDOMNodePtr originalImageArrayNode = NWXMLFindChildNode(textureNode, "original_image_array");
    if (originalImageArrayNode != NULL)
        originalImageNode = NWXMLFindChildNode(originalImageArrayNode, "original_image");

    //--------------------------------------------------------------------------
    // Get information
    //--------------------------------------------------------------------------
    NW_IMG_TYPE textureType = NW_IMG_TYPE_NONE;

    std::string linearFlagsStr = NWXMLGetAttrValue(textureInfoNode, "linear");
    USHORT linearFlags[4];
    NWStringToLinearFlagArray(linearFlagsStr, linearFlags, sizeof(linearFlags) / sizeof(linearFlags[0]));

    std::string textureTypeStr = NWXMLGetAttrValue(textureInfoNode,"dimension");
    if (_strcmpi(textureTypeStr.c_str(),"1d")==0)
    {
        textureType = NW_IMG_TYPE_1D;
    } // End if
    else if (_strcmpi(textureTypeStr.c_str(),"1d_array")==0)
    {
        textureType = NW_IMG_TYPE_1D_ARRAY;
    } // End if
    else if (_strcmpi(textureTypeStr.c_str(),"2d")==0)
    {
        textureType = NW_IMG_TYPE_2D;
    } // End if
    else if (_strcmpi(textureTypeStr.c_str(),"2d_array")==0)
    {
        textureType = NW_IMG_TYPE_2D_ARRAY;
    } // End if
    else if (_strcmpi(textureTypeStr.c_str(),"3d")==0)
    {
        textureType = NW_IMG_TYPE_3D;
    } // End if
    else if (_strcmpi(textureTypeStr.c_str(),"3d_array")==0)
    {
        textureType = NW_IMG_TYPE_3D_ARRAY;
    } // End if
    else if (_strcmpi(textureTypeStr.c_str(),"cube")==0)
    {
        textureType = NW_IMG_TYPE_CUBE;
    } // End if
    else if (_strcmpi(textureTypeStr.c_str(),"cube_array")==0)
    {
        textureType = NW_IMG_TYPE_CUBE_ARRAY;
    } // End if

    // Format
    std::string formatTypeStr = NWXMLGetAttrValue(textureInfoNode,"quantize_type");
    NW_IMG_FMT format = NWGetTextureFormatFromString(formatTypeStr.c_str());

    // Dimension
    int faceW = atol(NWXMLGetAttrValue(textureInfoNode, "width").c_str());
    int faceH = atol(NWXMLGetAttrValue(textureInfoNode, "height").c_str());
    int faceD = atol(NWXMLGetAttrValue(textureInfoNode, "depth").c_str());

    if ( (textureType==NW_IMG_TYPE_1D) && (faceH>1) )
    {
        textureType = NW_IMG_TYPE_2D;
    } // End if

    int mipLevels     = atol(NWXMLGetAttrValue(textureInfoNode, "mip_level").c_str());
    __int64 byteSize  = atol(NWXMLGetAttrValue(textureInfoNode, "size").c_str());

    pTexture->SetDescription(textureType,format,NW_IMG_FMT_NONE,faceW,faceH,faceD,mipLevels);
    pTexture->SetTextureDataSize(byteSize);

    bool twoChannelsAlphaFlag = false;
    if (originalImageNode != NULL)
    {
        std::string originalImageFormatStr = NWXMLGetAttrValue(originalImageNode, "format");

        if (_strnicmp(originalImageFormatStr.c_str(), "rgba", 4) == 0 && NWGetChannelCountFromImageFormat(format) == 2)
            twoChannelsAlphaFlag = true;
    }

    // Linear flags
    pTexture->SetLinearFlags( linearFlags[NW_COMP_SEL_ELEMENT_R] != 0,
                              linearFlags[NW_COMP_SEL_ELEMENT_G] != 0,
                              linearFlags[NW_COMP_SEL_ELEMENT_B] != 0,
                              linearFlags[NW_COMP_SEL_ELEMENT_A] != 0,
                              twoChannelsAlphaFlag);

    // Component Selector
    std::string compSelStr = NWXMLGetAttrValue(textureInfoNode,"comp_sel");
    NW_COMP_SEL compSel    = NWStringToCompSel(compSelStr);
    pTexture->SetCompSel(compSel);

    // Hint
    {
        std::string hintText = NWXMLGetAttrValue(textureInfoNode, "hint").c_str();
        if (_strcmpi(hintText.c_str(), "normal") == 0)
        {
            pTexture->SetHintNormalMap(true);
        } // End if
        else
        {
            pTexture->SetHintNormalMap(false);
        } // End else

        WCHAR wStrHint[128];
        size_t numConverted;
        mbstowcs_s(&numConverted, wStrHint, hintText.c_str(), 127);
        pTexture->SetHint(wStrHint);
    }

    // Comment
    MSXML2::IXMLDOMNodePtr commentNode = NWXMLFindChildNode(textureNode, "comment");
    if (commentNode!=NULL)
    {
        std::string commentText = NWXMLGetAttrValue(commentNode, "text").c_str();

        WCHAR wStrComment[256];
        size_t numConverted;
        mbstowcs_s(&numConverted,wStrComment,commentText.c_str(),255);

        pTexture->SetComment(wStrComment);
    }

    if (bLoadPreview==false)
    {
        return true;
    } // End if

    // For now, we don't support 3D texture, so skip loading contents
    if (//textureType==NW_IMG_TYPE_3D ||
        textureType==NW_IMG_TYPE_3D_ARRAY)
    {
        return true;
    } // End if

    // If 2D or 1D, then depth ( texture array size ) is not 1,
    // Then it is invalid file
    if (textureType==NW_IMG_TYPE_1D ||
        textureType==NW_IMG_TYPE_2D)
    {
        if (faceD>1)
        {
            return true;
        } // End if
    } // End if

    // If cube map, but depth is not 6, then this is invalid cube
    if (textureType==NW_IMG_TYPE_CUBE)
    {
        if (faceD!=6)
        {
            return true;
        } // End if
    } // End if

    std::wstring extension = NWGetExtension(szFilePath);

    bool loaded = false;
    if (nn::gfx::tool::texenc::IsSupported(formatTypeStr))
    {

        int iDataSize = 0;

        // Find the information about the original image
        int iStreamIndex = atol(NWXMLGetAttrValue(textureInfoNode, "stream_index").c_str());

        if (_wcsicmp(extension.c_str(), L".ftxa") == 0)
        {
            // Load ASCII Tiling Stream
            auto pStream = LoadASCIIImageStream(iStreamIndex,
                textureNode,
                iDataSize);

            if (pStream == nullptr)
            {
                return false;
            }

            // Make thumbnail bitmap from tiling stream.
            loaded = ConvertImageStreamToPreviewImage(pTexture,
                pStream.get(),
                formatTypeStr);
        } // End if
        else if (_wcsicmp(extension.c_str(), L".ftxb") == 0)
        {
            const unsigned char *pStream = NULL;

            // XML section size
            int iXMLSectionSize = xmlChunk.parray->rgsabound->cElements;

            // Load Binary Tiling Stream
            pStream = LoadBinaryImageStream(pBuffer.get() + iXMLSectionSize,
                iStreamIndex,
                iDataSize);

            if (pStream == nullptr)
            {
                return false;
            }

                // Make thumbnail bitmap from tiling stream.
            loaded = ConvertImageStreamToPreviewImage(pTexture,
                pStream,
                formatTypeStr);
        }
    }

    if (!loaded && originalImageNode != nullptr)
    {
        int iStreamNum;

        if (_wcsicmp(extension.c_str(),L".ftxa")==0)
        {
            // Load ASCII Original Streams
            std::unique_ptr<std::unique_ptr<const unsigned char[]>[]> pStreamArray = LoadASCIIOriginalStream(pTexture,
                                                    textureNode,
                                                    iStreamNum);

            if (pStreamArray == nullptr)
            {
                return false;
            }   // End if

            std::vector<const unsigned char*> data(iStreamNum);
            for (size_t i = 0; i < data.size(); i++) {
                data[i] = pStreamArray[i].get();
            }

            // Make thumbnail bitmap from original stream.
            ConvertOriginalStreamToPreviewImage( pTexture, data.data(), iStreamNum );
        } // End if
        else if (_wcsicmp(extension.c_str(),L".ftxb")==0)
        {


            // XML section size
            int iXMLSectionSize = xmlChunk.parray->rgsabound->cElements;

            // Load Binary Original Streams
            std::vector<const unsigned char *> pStreamArray = LoadBinaryOriginalStream(pTexture,
                pBuffer.get() + iXMLSectionSize,
                textureNode,
                iStreamNum);

            if (pStreamArray.empty())
            {
                return false;
            }   // End if

            // Make thumbnail bitmap from original stream.
            ConvertOriginalStreamToPreviewImage( pTexture, pStreamArray.data(), iStreamNum );
        } // End else if
    }

    return true;
} // End of Load for CNWFTXLoader

//--------------------------------------------------------------------------
// Read binary chunk header
//--------------------------------------------------------------------------
const unsigned char *CNWFTXLoader::ReadBinaryChunkHeader( const unsigned char *pByte,
                                                          std::unique_ptr<int[]> *streamOffsets,
                                                          std::unique_ptr<int[]> *streamSizes)
{
    // Skip the zero-padding bytes
    while ( *pByte==0x00 )
    {
        ++pByte;
    } // End while

    // Remember the location.
    const unsigned char *pDataHead = pByte;

    //--------------------------------------------------------------------------
    // Read binary chunk header
    //--------------------------------------------------------------------------
    // Binary chunk header identifier.
    const unsigned char chunkHeader[] = { 0x67, 0x33, 0x64, 0x73, 0x74, 0x72, 0x6D, 0x61 };

    // Make sure the chunk header matches.
    int i;
    for ( i=0;i<8;++i )
    {
        if ( pByte[i]!=chunkHeader[i] )
            return NULL;
    } // End for

    // Advance the buffer pointer
    pByte += 8;

    // Read the number of streams
    int iNumStreams = *( (int*)pByte );
    pByte += 4;

    streamOffsets->reset(new int[iNumStreams]);
    if (streamOffsets==NULL)
        return NULL;

    streamSizes->reset(new int[iNumStreams]);
    if (streamSizes==NULL)
    {
       return NULL;
    }

    // Read stream offset and size
    for ( i=0;i<iNumStreams;++i )
    {
        (*streamOffsets)[i] = *( (int*)pByte );
        (*streamSizes)[i]   = *( (int*)(pByte+4) );
        pByte += 8;
    } // End for

    return pDataHead;
}

const unsigned char *CNWFTXLoader::SkipToStreamHead( const unsigned char *pDataHead,
                                                     int *streamOffsets,
                                                     int stream_idx,
                                                     int &iDataSize)
{
    // Stream header identifier.
    const unsigned char streamHeader[] = { 0x73, 0x74, 0x72, 0x65, 0x61, 0x6D, 0x20, 0x20 };

    // Locate the stream
    const unsigned char *pByte = pDataHead + streamOffsets[stream_idx];

    // Make sure the stream header matches.
    int i;
    for ( i=0;i<8;++i )
    {
        if ( pByte[i]!=streamHeader[i] )
        {
            return NULL;
        }
    } // End for

    // Data type : 0: float, 1: int, 2: byte, 3: string, 4: wstring
    int iDataType = *( (int*)(pByte+8) );
    NW_USE_VAR(iDataType);

    // Data count
    int iDataCount = *( (int*)(pByte+12) );
    NW_USE_VAR(iDataCount);

    // Data size ( in bytes )
    iDataSize = *( (int*)(pByte+20) );

    // Advance the buffer pointer to the stream data chunk
    pByte += 32;

    return pByte;
}

//------------------------------------------------------------------------------
// Load Binary Tiling Stream
//------------------------------------------------------------------------------
const unsigned char *CNWFTXLoader::LoadBinaryImageStream( const unsigned char *pBinarySection,
                                                           int iStreamIndex,
                                                           int &iDataSize )
{
    const unsigned char *pByte = pBinarySection;

    std::unique_ptr<int[]> streamOffsets = NULL;
    std::unique_ptr<int[]> streamSizes = NULL;
    //--------------------------------------------------------------------------
    // Read binary chunk header
    //--------------------------------------------------------------------------
    const unsigned char *pDataHead = ReadBinaryChunkHeader(pByte, &streamOffsets, &streamSizes);
    if (pDataHead == NULL)
    {
        return NULL;
    }

    //--------------------------------------------------------------------------
    // Read stream header
    //--------------------------------------------------------------------------
    iDataSize=0;
    pByte = SkipToStreamHead(pDataHead, streamOffsets.get(), iStreamIndex, iDataSize);
    if (pByte == NULL)
    {
        return NULL;
    }

    return pByte;
} // End of LoadBinaryTilingStream for CNWFTXLoader


//------------------------------------------------------------------------------
// Load Binary Original Stream
//------------------------------------------------------------------------------
std::vector<const unsigned char*> CNWFTXLoader::LoadBinaryOriginalStream( CNWTexture *pTexture,
                                                             const unsigned char *pBinarySection,
                                                             MSXML2::IXMLDOMNodePtr textureNode,
                                                             int &iStreamNum)
{
    const unsigned char *pByte = pBinarySection;

    std::vector<const unsigned char*> pStreamArray;
    //--------------------------------------------------------------------------
    // オリジナル画像についてのヘッダー情報を取得
    //--------------------------------------------------------------------------
    MSXML2::IXMLDOMNodePtr originalImageArrayNode = NWXMLFindChildNode(textureNode, "original_image_array");
    if (originalImageArrayNode == NULL)
    {
        return pStreamArray;
    }

    iStreamNum = atol(NWXMLGetAttrValue(originalImageArrayNode, "length").c_str() );
    std::vector<OriginalImageField> originalImageInfo(iStreamNum);

    MSXML2::IXMLDOMNodeListPtr children = originalImageArrayNode->childNodes;

    int i;
    for (i=0; i<iStreamNum; i++)
    {
        MSXML2::IXMLDOMNodePtr originalImageNode = children->nextNode();
        int index = atol(NWXMLGetAttrValue(originalImageNode, "index").c_str() );
        if (strcmp(originalImageNode->nodeName,"original_image")==0 && (i==index))
        {
            originalImageInfo[i].index = index;
            originalImageInfo[i].slice_index = atol(NWXMLGetAttrValue(originalImageNode, "slice_index").c_str() );
            originalImageInfo[i].face = NWXMLGetAttrValue(originalImageNode, "face");
            originalImageInfo[i].format = NWXMLGetAttrValue(originalImageNode, "format");
            originalImageInfo[i].width  = atol(NWXMLGetAttrValue(originalImageNode, "width").c_str() );
            originalImageInfo[i].height = atol(NWXMLGetAttrValue(originalImageNode, "height").c_str() );
            originalImageInfo[i].size = atol(NWXMLGetAttrValue(originalImageNode, "size").c_str() );
            originalImageInfo[i].stream_idx = atol(NWXMLGetAttrValue(originalImageNode, "stream_index").c_str() );
        }
    }

    pStreamArray.resize(iStreamNum);

    std::unique_ptr<int[]> streamOffsets = NULL;
    std::unique_ptr<int[]> streamSizes = NULL;
    //--------------------------------------------------------------------------
    // Read binary chunk header
    //--------------------------------------------------------------------------
    const unsigned char *pDataHead = ReadBinaryChunkHeader(pByte, &streamOffsets, &streamSizes);
    if (pDataHead == NULL)
    {
        pStreamArray.clear();
        return pStreamArray;
    }

    //--------------------------------------------------------------------------
    // Read stream header
    //--------------------------------------------------------------------------
    for (i=0; i<iStreamNum; i++)
    {
        int iDataSize;
        pByte = SkipToStreamHead(pDataHead, streamOffsets.get(), originalImageInfo[i].stream_idx, iDataSize);
        if (pByte == NULL)
        {
            pStreamArray.clear();
            return pStreamArray;
        }
        pStreamArray[i] = pByte;
    }

    // オリジナル画像のヘッダー情報をセット、
    // ここではoriginalImageInfoの解放を行わず、CNWTextureに任せる
    pTexture->SetOriginalImageFieldInfo(originalImageInfo);

    return pStreamArray;
} // End of LoadBinaryOriginalStream for CNWFTXLoader

//--------------------------------------------------------------------------
// Load ASCII Tiling Stream
//--------------------------------------------------------------------------
std::unique_ptr<const unsigned char[]> CNWFTXLoader::LoadASCIIImageStream( int iStreamIndex,
                                                          MSXML2::IXMLDOMNodePtr textureNode,
                                                          int &iDataSize)
{
    //--------------------------------------------------------------------------
    // Read stream array
    //--------------------------------------------------------------------------
    MSXML2::IXMLDOMNodePtr streamArrayNode = NWXMLFindChildNode( textureNode, "stream_array" );
    if (streamArrayNode == NULL)
    {
        return NULL;
    }

    //--------------------------------------------------------------------------
    // Read stream node
    //--------------------------------------------------------------------------
    MSXML2::IXMLDOMNodePtr streamNode = GetStreamByIndex( streamArrayNode, iStreamIndex );
    if (streamNode==NULL)
    {
        return NULL;
    }

    std::string imageText = streamNode->text;
    iDataSize = 0;
    int imageSize = (int)imageText.size();

    std::unique_ptr<const unsigned char[]> decodedBuf = DecodeStreamTextToBuffer( (const unsigned char*)imageText.c_str(),
                                                                 imageSize,
                                                                 iDataSize );

    return decodedBuf;
} // End of LoadASCIITilingStream for CNWFTXLoader

//------------------------------------------------------------------------------
// Load ASCII Original Streams
//------------------------------------------------------------------------------
std::unique_ptr<std::unique_ptr<const unsigned char[]>[]> CNWFTXLoader::LoadASCIIOriginalStream( CNWTexture *pTexture,
                                                             MSXML2::IXMLDOMNodePtr textureNode,
                                                             int &iStreamNum)
{

    //--------------------------------------------------------------------------
    // オリジナル画像についてのヘッダー情報を取得
    //--------------------------------------------------------------------------
    MSXML2::IXMLDOMNodePtr originalImageArrayNode = NWXMLFindChildNode(textureNode, "original_image_array");
    iStreamNum = atol(NWXMLGetAttrValue(originalImageArrayNode, "length").c_str() );

    if (originalImageArrayNode == NULL)
    {
        return nullptr;
    }   // End if

    std::vector<OriginalImageField> originalImageInfo(iStreamNum);

    MSXML2::IXMLDOMNodeListPtr children = originalImageArrayNode->childNodes;
    int i;
    for (i=0; i<iStreamNum; i++)
    {
        MSXML2::IXMLDOMNodePtr originalImageNode = children->nextNode();
        int index = atol(NWXMLGetAttrValue(originalImageNode, "index").c_str() );
        if (strcmp(originalImageNode->nodeName,"original_image")==0 && (i==index))
        {
            originalImageInfo[i].index = index;
            originalImageInfo[i].slice_index = atol(NWXMLGetAttrValue(originalImageNode, "slice_index").c_str() );
            originalImageInfo[i].face = NWXMLGetAttrValue(originalImageNode, "face");
            originalImageInfo[i].format = NWXMLGetAttrValue(originalImageNode, "format");
            originalImageInfo[i].width  = atol(NWXMLGetAttrValue(originalImageNode, "width").c_str() );
            originalImageInfo[i].height = atol(NWXMLGetAttrValue(originalImageNode, "height").c_str() );
            originalImageInfo[i].size = atol(NWXMLGetAttrValue(originalImageNode, "size").c_str() );
            originalImageInfo[i].stream_idx = atol(NWXMLGetAttrValue(originalImageNode, "stream_index").c_str() );
        }   // End if
    }   // End for

    std::unique_ptr<std::unique_ptr<const unsigned char[]>[]> pStreamArray(new std::unique_ptr<const unsigned char[]>[iStreamNum]);

    //--------------------------------------------------------------------------
    // Read stream array
    //--------------------------------------------------------------------------
    MSXML2::IXMLDOMNodePtr streamArrayNode = NWXMLFindChildNode( textureNode, "stream_array" );
    if (streamArrayNode == NULL)
    {
        return nullptr;
    }   // End if
    const char* strFormat = originalImageInfo[0].format.c_str();

    //--------------------------------------------------------------------------
    // Read stream node
    //--------------------------------------------------------------------------
    for (i=0; i<iStreamNum; i++)
    {
        MSXML2::IXMLDOMNodePtr streamNode = GetStreamByIndex( streamArrayNode, originalImageInfo[i].stream_idx );
        if (streamNode==NULL)
        {
            return nullptr;
        }   // End if

        std::string imageText = streamNode->text;
        int decodedSize = 0;

        std::unique_ptr<const unsigned char[]> decodedBuf = NULL;
        if (_stricmp(strFormat, "rgb32f") ==0 || _stricmp(strFormat, "rgba32f") ==0)
        {
            int format=3;
            if (_stricmp(strFormat, "rgba32f") == 0) format = 4;

            // floatデータの読み込み
            int imageSize = originalImageInfo[i].size;
            decodedBuf = DecodeStreamTextToFloatBuffer( (const unsigned char*)imageText.c_str(),
                                                        imageSize,
                                                        format,
                                                        decodedSize );
        }   // End if
        else
        {
            // byteデータの読み込み
            int imageSize = (int)imageText.size();
            decodedBuf = DecodeStreamTextToBuffer( (const unsigned char*)imageText.c_str(),
                                                   imageSize,
                                                   decodedSize );
        }   // End if

        if (decodedBuf==NULL)
        {
            return nullptr;
        }   // End if
        pStreamArray[i].swap(decodedBuf);

        // 念のためサイズをデコード後のサイズに変更する
        originalImageInfo[i].size = decodedSize;
    }   // End for

    // オリジナル画像のヘッダー情報をセット、
    // originalImageInfoの解放はここでは行わず、CNWTextureに任せる
    pTexture->SetOriginalImageFieldInfo(originalImageInfo);

    return pStreamArray;
} // End of const LoadASCIIOriginalStream for CNWFTXLoader

namespace comp_sel
{
    enum value
    {
        r = 0,
        g = 1,
        b = 2,
        a = 3,
        Item0 = 4,
        Item1 = 5
    };

    value GetValue(char c)
    {
        switch( c )
        {
        case 'r':
            return r;
        case 'g':
            return g;
        case 'b':
            return b;
        case 'a':
            return a;
        case '0':
            return Item0;
        case '1':
            return Item1;
        }
        return r;
    }
}

//==============================================================================
//
// Utility used internally
//
//==============================================================================
OriginalImageFormat GetOriginalImageFormatFromString( std::string formatTypeStr );

namespace TexUtils
{
void Decode_original_rgb8(unsigned char *dstColor, const unsigned char *src, int size);
void Decode_original_rgba8(unsigned char *dstColor, const unsigned char *src, int size);
void Decode_original_rgb32f(unsigned char *dstColor, const unsigned char *src, int size);
void Decode_original_rgba32f(unsigned char *dstColor, const unsigned char *src, int size);

}    // namespace TexUtils


//------------------------------------------------------------------------------
// RSurface
//------------------------------------------------------------------------------
class RSurface
{
  public:
    std::string m_Name; //!< 名前です。
    void* imagePtr;
    void* mipPtr;
    bool isSnormOrSint;

  public:
    //! コンストラクタです。
    RSurface(const std::string& name)
    : m_Name(name),
      imagePtr(nullptr),
      mipPtr(nullptr)
    {
    }

    //! デストラクタです。
    virtual ~RSurface()
    {
        if (imagePtr != NULL)
        {
            delete[] imagePtr;
        }

        if (mipPtr != NULL)
        {
            delete[] mipPtr;
        }
    }
};

#if USE_CARE_TEXTURE
bool MakeAllSourceSurface( RSurface &dstSurface,
                           CNWTexture *pTexture,
                           const unsigned char* streams,
                           int srcBufSize,
                           MSXML2::IXMLDOMNodePtr textureNode,
                           comp_sel::value* comp_sel)
{
    MSXML2::IXMLDOMNodePtr textureInfoNode = NWXMLFindChildNode(textureNode, "texture_info");
    if (textureInfoNode == NULL)
        return false;

    std::string dimensionStr    = NWXMLGetAttrValue(textureInfoNode,"dimension");
    std::string quantizeTypeStr = NWXMLGetAttrValue(textureInfoNode,"quantize_type");
    std::string mipOffsetStr    = NWXMLGetAttrValue(textureInfoNode,"mip_offset");
    std::string compSelStr      = NWXMLGetAttrValue(textureInfoNode,"comp_sel");

    if(dimensionStr.empty())
        return false;
    if(quantizeTypeStr.empty())
        return false;
    if(mipOffsetStr.empty())
        return false;

    int mipOffset[13];
    sscanf_s(mipOffsetStr.c_str(), "%d %d %d %d %d %d %d %d %d %d %d %d %d",
                                &mipOffset[0], &mipOffset[1], &mipOffset[2],
                                &mipOffset[3], &mipOffset[4], &mipOffset[5],
                                &mipOffset[6], &mipOffset[7], &mipOffset[8],
                                &mipOffset[9], &mipOffset[10],&mipOffset[11], &mipOffset[12]);

    dstSurface.m_Surface.dim        = GX2GetTextureDimensionFromString( dimensionStr );
    dstSurface.m_Surface.width      = pTexture->GetWidth();
    dstSurface.m_Surface.height     = pTexture->GetHeight();

    if ( (dstSurface.m_Surface.dim==GX2_SURFACE_DIM_1D) && (pTexture->GetHeight()>1) )
    {
        // Fix invalid texture type
        dstSurface.m_Surface.dim = GX2_SURFACE_DIM_2D;
    } // End if

    switch(pTexture->GetTextureType())
    {
    default:
        dstSurface.m_Surface.depth      = pTexture->GetDepth();
        break;
    case NW_IMG_TYPE_1D_ARRAY:
    case NW_IMG_TYPE_2D_ARRAY:
    case NW_IMG_TYPE_CUBE_ARRAY:
        dstSurface.m_Surface.depth      = pTexture->GetCount();
        break;
    }
    dstSurface.m_Surface.numMips    = pTexture->GetNumMipmaps();
    dstSurface.m_Surface.format     = GX2GetTextureFormatFromString( quantizeTypeStr );
    dstSurface.m_Surface.aa         = GX2_AA_MODE_1X;
    dstSurface.m_Surface.use        = GX2_SURFACE_USE_TEXTURE;
    dstSurface.m_Surface.imageSize  = (dstSurface.m_Surface.numMips >= 2) ? mipOffset[0] : srcBufSize;
    dstSurface.m_Surface.imagePtr   = new char[dstSurface.m_Surface.imageSize];
    if (dstSurface.m_Surface.imagePtr==NULL)
        return false;

    dstSurface.m_Surface.mipSize    = (int)max((__int64)pTexture->GetTextureDataSize() - (__int64)dstSurface.m_Surface.imageSize, 0);
    dstSurface.m_Surface.mipPtr        = new char[dstSurface.m_Surface.mipSize];
    if (dstSurface.m_Surface.mipPtr==NULL)
    {
        delete[] dstSurface.m_Surface.imagePtr;
        dstSurface.m_Surface.imagePtr = NULL;
        return false;
    } // End if

    for(int i=0; i!=13; ++i)
    {
        dstSurface.m_Surface.mipOffset[i] = mipOffset[i];
    } // End for

    dstSurface.m_Surface.tileMode     = GX2GetTileModeFromString( NWXMLGetAttrValue(textureInfoNode, "tile_mode") );
    dstSurface.m_Surface.swizzle      = atol(NWXMLGetAttrValue(textureInfoNode, "swizzle").c_str() );
    dstSurface.m_Surface.alignment    = atol(NWXMLGetAttrValue(textureInfoNode, "alignment").c_str() );
    dstSurface.m_Surface.pitch        = atol(NWXMLGetAttrValue(textureInfoNode, "pitch").c_str() );

    // copy image
    memcpy(dstSurface.m_Surface.imagePtr, streams, dstSurface.m_Surface.imageSize);

    const int offset = dstSurface.m_Surface.imageSize;
    const unsigned char *src = streams;
    unsigned char *dst = reinterpret_cast<unsigned char *>(dstSurface.m_Surface.mipPtr);
    for(u32 i = 0;i != dstSurface.m_Surface.mipSize;++ i)
    {
        dst[i] = src[i + offset];
    }

    // set comp sel
    NW_USE_VAR(comp_sel);
    /* We do not change comp_sel here, comp_sel will be applied inside Texture->CreateColorMap
    if(compSelStr.size() >= 1) comp_sel[0] = comp_sel::GetValue(compSelStr[0]);
    if(compSelStr.size() >= 3) comp_sel[1] = comp_sel::GetValue(compSelStr[2]);
    if(compSelStr.size() >= 5) comp_sel[2] = comp_sel::GetValue(compSelStr[4]);
    if(compSelStr.size() >= 7) comp_sel[3] = comp_sel::GetValue(compSelStr[6]);
    */

    return true;
}
#endif
static bool
ImageToPreview1DArrayImage( CNWTexture *pTexture,
                       RSurface &detiledSurface,
                       OriginalImageFormat originalFmt
                         );

static bool
ImageToPreview2DImage( CNWTexture *pTexture,
                       RSurface &detiledSurface,
                       OriginalImageFormat originalFmt
                         );

static bool
ImageToPreviewCubeImage( CNWTexture *pTexture,
                         RSurface &detiledSurface,
                         OriginalImageFormat originalFmt
                         );

static bool
ImageToPreview2DArrayImage( CNWTexture *pTexture,
                         RSurface &detiledSurface,
                         OriginalImageFormat originalFmt
                         );

static bool
ImageToPreviewCubeArrayImage( CNWTexture *pTexture,
                         RSurface &detiledSurface,
                         OriginalImageFormat originalFmt
                         );

static bool
ImageToPreview3DImage( CNWTexture *pTexture,
                         RSurface &detiledSurface,
                         OriginalImageFormat originalFmt
                         );

//------------------------------------------------------------------------------
// Original image to preview image
//------------------------------------------------------------------------------
bool ConvertSurfaceToPreviewImage( CNWTexture *pTexture, RSurface &detiledSurface, OriginalImageFormat originalFmt )
{
    bool result = false;

    switch (pTexture->GetTextureType())
    {
        case NW_IMG_TYPE_1D:
        case NW_IMG_TYPE_2D:
            result = ImageToPreview2DImage(pTexture,
                                           detiledSurface,
                                           originalFmt
                                           );
            break;

        case NW_IMG_TYPE_CUBE:
            result = ImageToPreviewCubeImage(pTexture,
                                             detiledSurface,
                                             originalFmt
                                             );
            break;

        case NW_IMG_TYPE_3D:
            result = ImageToPreview3DImage(pTexture,
                                           detiledSurface,
                                           originalFmt
                                           );
            break;

        case NW_IMG_TYPE_1D_ARRAY:
            result = ImageToPreview1DArrayImage(pTexture,
                                                detiledSurface,
                                                originalFmt
                                                );

            break;
        case NW_IMG_TYPE_2D_ARRAY:
            result = ImageToPreview2DArrayImage(pTexture,
                                                detiledSurface,
                                                originalFmt
                                                );
            break;

        case NW_IMG_TYPE_CUBE_ARRAY:
            result = ImageToPreviewCubeArrayImage(pTexture,
                                                  detiledSurface,
                                                  originalFmt
                                                  );
            break;

        default:
            break;
    } // End switch

    return result;
} // End of HardwareImageToPreviewImage for CNWFTXLoader

  //------------------------------------------------------------------------------
  // Convert Image stream to Thumbnail bitmap
  //------------------------------------------------------------------------------
bool CNWFTXLoader::ConvertImageStreamToPreviewImage(CNWTexture *pTexture,
    const unsigned char* srcBuf,
    const std::string& format)
{
    RSurface detailedSurface("detailed");
    auto isSnormOrSint = format.find("snorm_", 0) != std::wstring::npos
        || format.find("sint_", 0) != std::wstring::npos;

    auto dstFormat = isSnormOrSint ? "snorm_8_8_8_8" : "unorm_8_8_8_8";
    detailedSurface.isSnormOrSint = isSnormOrSint;

    auto dstSize = nn::gfx::tool::texenc::GetDataSize(dstFormat, pTexture->GetWidth(), pTexture->GetHeight(), pTexture->GetDepth() * pTexture->GetCount());
    detailedSurface.imagePtr = new unsigned char[dstSize];

    if (!nn::gfx::tool::texenc::ConvertFormat(
        detailedSurface.imagePtr,
        srcBuf,
        dstFormat,
        format,
        "",
        nn::gfx::tool::texenc::EncodeFlag_ReverseRgba,
        pTexture->GetWidth(),
        pTexture->GetHeight(),
        pTexture->GetDepth() * pTexture->GetCount()))
    {
        return false;
    }

    ConvertSurfaceToPreviewImage(pTexture, detailedSurface, ORIGINAL_FORMAT_INVALID);

    return true;
}

//------------------------------------------------------------------------------
// Convert Original Stream to Thumbnail bitmap
//------------------------------------------------------------------------------
bool CNWFTXLoader::ConvertOriginalStreamToPreviewImage( CNWTexture *pTexture,
                                                        const unsigned char** srcBuf,
                                                        int iStreamNum)
{

    RSurface detiledSurface("detiled");
    const OriginalImageField *pOriginalImageInfo = pTexture->GetOriginalImageFieldInfo();

    // streamごとにフォーマットが変わることは想定していないので、フォーマット一番最初のフォーマットのみを確認
    OriginalImageFormat format = GetOriginalImageFormatFromString(pOriginalImageInfo[0].format);

    int bytesPerPixel;
    switch (format)
    {
    case ORIGINAL_FORMAT_RGB8:   bytesPerPixel = 3;     break;
    case ORIGINAL_FORMAT_RGBA8:  bytesPerPixel = 4;     break;
    case ORIGINAL_FORMAT_RGB32F:  bytesPerPixel = 12;    break;
    case ORIGINAL_FORMAT_RGBA32F: bytesPerPixel = 16;    break;
    default: bytesPerPixel = 0; break;
    }   // End switch

    //------------------------------------------------------------------------
    // ConvertSurfaceToPreviewImageの扱いに合わせるため、複数のstreamを一つのsurfaceにまとめる
    //------------------------------------------------------------------------
    int dataSize=0;
    for (int i=0; i<iStreamNum; i++)
    {
        switch (pTexture->GetTextureType())
        {
        case NW_IMG_TYPE_1D:
        case NW_IMG_TYPE_1D_ARRAY:
            dataSize += pOriginalImageInfo[i].width * bytesPerPixel;
            break;
        default:
            dataSize += pOriginalImageInfo[i].size;
        }
    }   // End for

    detiledSurface.imagePtr = new unsigned char[dataSize];

    int offset = 0;
    for (int i=0; i<iStreamNum; i++)
    {
        const unsigned char *ptr = reinterpret_cast<const unsigned char *>(detiledSurface.imagePtr) + offset;

        int size;
        switch (pTexture->GetTextureType())
        {
        case NW_IMG_TYPE_1D:
        case NW_IMG_TYPE_1D_ARRAY:
            size = pOriginalImageInfo[i].width * bytesPerPixel;
            break;
        default:
            size = pOriginalImageInfo[i].size;
        }
        memcpy((void*)ptr, (void*)srcBuf[i],  size);
        offset += size;
    }   // End for

    //------------------------------------------------------------------------
    // Convert Surface to Thumbnail bitmap
    //------------------------------------------------------------------------
    ConvertSurfaceToPreviewImage(pTexture, detiledSurface, format);

    return true;
} // End of OriginalImageToPreviewImage for CNWFTXLoader

//------------------------------------------------------------------------------
// Get stream XML node with given index
//------------------------------------------------------------------------------
MSXML2::IXMLDOMNodePtr CNWFTXLoader::GetStreamByIndex( MSXML2::IXMLDOMNodePtr streamArrayNode,
                                                       int index )
{
    MSXML2::IXMLDOMNodeListPtr childs = streamArrayNode->childNodes;
    for (;;)
    {
        MSXML2::IXMLDOMNodePtr child = childs->nextNode();
        if (child == NULL)
        {
            break;
        } // End if

        if (strcmp(child->nodeName,"stream")==0)
        {
            int streamIndex = atol(NWXMLGetAttrValue(child, "stream_index").c_str());
            if (streamIndex==index)
                return child;
        } // End if
    } // End for

    return NULL;
} // End of GetStreamByIndex for CNWFTXLoader

static int
MipmapSize(int srcSize, int mipmapLevel)
{
    return max(srcSize >> mipmapLevel, 1);
}

static int
MipmapImageOffset(int width, int height, int depthLevel, float bpp)
{
    int offset = 0;
    {
        width    = MipmapSize(width,  0);
        height    = MipmapSize(height, 0);

        int size = (int)(width * height * bpp);
        offset += size * depthLevel;
    }
    return offset;
}

bool ConvertOriginalImageToBitmap(  Gdiplus::Bitmap *pDestBitmap,
                                    RSurface &detiledSurface,
                                    OriginalImageFormat originalFmt,
                                    int depthLevel )
{
    if (pDestBitmap==NULL)
    {
        return false;
    }   // End if


    int bpp=0;
    switch(originalFmt)
    {
        case ORIGINAL_FORMAT_RGB8:   bpp=3;     break;
        case ORIGINAL_FORMAT_RGBA8:  bpp=4;     break;
        case ORIGINAL_FORMAT_RGB32F:  bpp=12;    break;
        case ORIGINAL_FORMAT_RGBA32F: bpp=16;    break;
        default: bpp=0; break;
    }   // End switch

    int   width    = pDestBitmap->GetWidth();
    int   height   = pDestBitmap->GetHeight();
    int   lineSize = (int)(width * bpp);

    // Lock bits to prepare preview image
    Gdiplus::Rect lockRect(0, 0, width, height);
    Gdiplus::BitmapData bmpData;
    if (pDestBitmap->LockBits(&lockRect,
                              Gdiplus::ImageLockModeWrite,
                              pDestBitmap->GetPixelFormat(),
                              &bmpData)!=Gdiplus::Ok)
    {
        return false;
    }   // End if

    const unsigned char *src = reinterpret_cast<const unsigned char *>
            (detiledSurface.imagePtr) + MipmapImageOffset( width, height, depthLevel, (float)bpp);

    unsigned char* dst = reinterpret_cast<unsigned char *>(bmpData.Scan0);

    for(int y = 0;y != height;++ y)
    {
        switch(originalFmt)
        {
            case ORIGINAL_FORMAT_RGB8:   TexUtils::Decode_original_rgb8(  dst, src, lineSize);    break;
            case ORIGINAL_FORMAT_RGBA8:  TexUtils::Decode_original_rgba8( dst, src, lineSize);    break;
            case ORIGINAL_FORMAT_RGB32F:  TexUtils::Decode_original_rgb32f( dst, src, lineSize);    break;
            case ORIGINAL_FORMAT_RGBA32F: TexUtils::Decode_original_rgba32f(dst, src, lineSize);    break;
        }   // End switch
        src += lineSize;
        dst += bmpData.Stride;

    }   // End for

    pDestBitmap->UnlockBits(&bmpData);

    return true;
}

bool ConvertImageToBitmap(Gdiplus::Bitmap *pDestBitmap,
    RSurface &detailedSurface,
    int width,
    int height,
    int depthLevel,
    int step,
    bool isSnorm
)
{
    Gdiplus::Rect lockRect(0, 0, width, height);
    Gdiplus::BitmapData bmpData;
    if (pDestBitmap->LockBits(&lockRect,
        Gdiplus::ImageLockModeWrite,
        pDestBitmap->GetPixelFormat(),
        &bmpData) != Gdiplus::Ok)
    {
        return false;
    }


    auto dst = reinterpret_cast<unsigned char *>(bmpData.Scan0);
    auto dstStride = bmpData.Stride;
    if (isSnorm)
    {
        auto src = reinterpret_cast<const signed char*>(detailedSurface.imagePtr) + width * height * step * depthLevel;
        for (int y = 0; y < height; y++)
        {
            auto dstIdx = y * dstStride;
            auto srcIdx = y * width * step;
            for (int x = 0; x < width; x++)
            {
                dst[dstIdx + 2] = (src[srcIdx + 0] + 128) & 0xff;
                dst[dstIdx + 1] = (src[srcIdx + 1] + 128) & 0xff;
                dst[dstIdx + 0] = (src[srcIdx + 2] + 128) & 0xff;
                dst[dstIdx + 3] = (src[srcIdx + 3] + 128) & 0xff;
                dstIdx += 4;
                srcIdx += step;
            }
        }
    }
    else
    {
        auto src = reinterpret_cast<const unsigned char*>(detailedSurface.imagePtr) + width * height * step * depthLevel;
        for (int y = 0; y < height; y++)
        {
            auto dstIdx = y * dstStride;
            auto srcIdx = y * width * step;
            for (int x = 0; x < width; x++)
            {
                dst[dstIdx + 2] = src[srcIdx + 0];
                dst[dstIdx + 1] = src[srcIdx + 1];
                dst[dstIdx + 0] = src[srcIdx + 2];
                dst[dstIdx + 3] = src[srcIdx + 3];
                dstIdx += 4;
                srcIdx += step;
            }
        }
    }

    pDestBitmap->UnlockBits(&bmpData);
    return true;
}

//------------------------------------------------------------------------------
// Get image with given depth
//------------------------------------------------------------------------------
bool GetImageByDepth( Gdiplus::Bitmap *pDestBitmap,
                      RSurface &detiledSurface,
                      OriginalImageFormat originalFmt,
                      int depthLevel
                      )
{
    if (originalFmt == ORIGINAL_FORMAT_INVALID)
    {
        return ConvertImageToBitmap(pDestBitmap,
            detiledSurface,
            pDestBitmap->GetWidth(),
            pDestBitmap->GetHeight(),
            depthLevel,
            4,
            detiledSurface.isSnormOrSint);
    }
    else
    {
        return ConvertOriginalImageToBitmap(pDestBitmap,
            detiledSurface,
            originalFmt,
            depthLevel);
    }
}


//------------------------------------------------------------------------------
// Textureからアイコン内への描画位置およびサイズを計算して取得する
//------------------------------------------------------------------------------
void GetImageRect(IN CNWTexture *pTexture,OUT float* pDrawImgX,OUT float* pDrawImgY,OUT float* pDrawImgWd,OUT float* pDrawImgHt)
{
    // Now draw actual image
    *pDrawImgX  = 0.0f;
    *pDrawImgY  = 0.0f;
    *pDrawImgWd = (float)pTexture->GetPreviewWidth();
    *pDrawImgHt = (float)pTexture->GetPreviewHeight();

    if(pTexture->GetWidth() != pTexture->GetHeight())
    {
        float drawImgWdRatio = (float)pTexture->GetPreviewWidth() / (float)pTexture->GetWidth();
        float drawImgHtRatio = (float)pTexture->GetPreviewHeight() / (float)pTexture->GetHeight();

        if (drawImgWdRatio<drawImgHtRatio)
        {
            // Match size scaling with width
            *pDrawImgHt = ((float)pTexture->GetHeight()*drawImgWdRatio);
        } // End if
        else
        {
            // Match size scaling with height
            *pDrawImgWd = ((float)pTexture->GetWidth() *drawImgHtRatio);
        } // End if

        if (*pDrawImgWd<1)
            *pDrawImgWd = 1;

        if (*pDrawImgHt<1)
            *pDrawImgHt = 1;

        *pDrawImgX = ( pTexture->GetPreviewWidth() / 2.0f ) - ( *pDrawImgWd / 2.0f );
        *pDrawImgY = ( pTexture->GetPreviewHeight() / 2.0f ) - ( *pDrawImgHt / 2.0f );
    } // End if
}



bool CreateColorAndAlphaMap(CNWTexture *pTexture,Gdiplus::Bitmap* pImg,OriginalImageFormat originalFmt,
                            IN Gdiplus::Rect drawRect, bool bSmall = false)
{
    // Color + Alpha
    if (!pTexture->CreatePreviewBitmap(pImg, drawRect, originalFmt, bSmall, true, true))
    {
        return false;
    }

    // Color
    if (!pTexture->CreatePreviewBitmap(pImg, drawRect, originalFmt, bSmall, true, false))
    {
        return false;
    }

    // Alpha
    if (!pTexture->CreatePreviewBitmap(pImg, drawRect, originalFmt, bSmall, false, true))
    {
        return false;
    }

    return true;
}


//------------------------------------------------------------------------------
// 1D image to preview image
//------------------------------------------------------------------------------
bool ImageToPreview1DArrayImage( CNWTexture *pTexture,
                            RSurface &detiledSurface,
                            OriginalImageFormat originalFmt
                            )
{
    if (pTexture==NULL)
    {
        return false;
    } // End if

    Gdiplus::Bitmap img(pTexture->GetPreviewWidth(), pTexture->GetPreviewHeight(), PixelFormat32bppARGB);
    //for large icon
    {
        Gdiplus::Bitmap *pPreviewBitmap = &img;
        if (pPreviewBitmap==NULL)
        {
            return false;
        }

        Gdiplus::Rect lockResultRect(0,0,pPreviewBitmap->GetWidth(),pPreviewBitmap->GetHeight());
        Gdiplus::BitmapData bmpResult;
        if (pPreviewBitmap->LockBits(&lockResultRect,Gdiplus::ImageLockModeWrite,
            pPreviewBitmap->GetPixelFormat(),
            &bmpResult)!=Gdiplus::Ok)
        {
            return false;
        }

        unsigned char *pDestData = (unsigned char*)bmpResult.Scan0;

        //Fill the rectangle with Glay
        std::unique_ptr<Gdiplus::Graphics> graphicsColor(Gdiplus::Graphics::FromImage( pPreviewBitmap ));
        Gdiplus::SolidBrush grayBrush(Gdiplus::Color(128,128,128,128));
        graphicsColor->FillRectangle(&grayBrush,0,0,pPreviewBitmap->GetWidth(),pPreviewBitmap->GetHeight());

        std::unique_ptr<Gdiplus::Bitmap> pTmpColorBitmap(new Gdiplus::Bitmap(pTexture->GetWidth(),pTexture->GetHeight(),PixelFormat32bppARGB));
        if (pTmpColorBitmap==NULL)
        {
            // pPreviewBitmap Finish locking
            pPreviewBitmap->UnlockBits(&bmpResult);
            return false;
        } // End if

        int nCountMax = (2 <= pTexture->GetCount() - 1) ? 2 : pTexture->GetCount() - 1;
        for(int count = nCountMax;count >= 0;count--){
            Gdiplus::BitmapData bmpData;
            if (!GetImageByDepth(pTmpColorBitmap.get(),
                detiledSurface,
                originalFmt,
                count
                ))
            {
                // pPreviewBitmap Finish locking
                pPreviewBitmap->UnlockBits(&bmpResult);
                return false;
            }

            // Lock bits to prepare preview image
            Gdiplus::Rect lockRect(0,0,pTmpColorBitmap->GetWidth(),pTmpColorBitmap->GetHeight());
            if (pTmpColorBitmap->LockBits(&lockRect,Gdiplus::ImageLockModeRead,
                pPreviewBitmap->GetPixelFormat(),
                &bmpData)!=Gdiplus::Ok)
            {
                // pPreviewBitmap Finish locking
                pPreviewBitmap->UnlockBits(&bmpResult);
                return false;
            }

            unsigned char *pSrcData  = (unsigned char*)bmpData.Scan0;

            for(int x = 0;x < (int)bmpData.Width;x++)
            {
                int nSrcIndex = x * 4;
                int yy = (int)((count + 3) * bmpResult.Height / 8.0);
                int xx = (int)(count * bmpResult.Width / 8.0 + x * 3.0 / 4.0);
                int nDestIndex = (int)(yy * bmpResult.Stride + xx * 4);

                unsigned char *pSrcPixel = pSrcData + nSrcIndex;
                unsigned char *pDestPixel = pDestData + nDestIndex;

                float alpha = pSrcPixel[3]/255.f;
                for(int i = 0;i < 3;i++)
                    pDestPixel[i] = (unsigned char)((1.f-alpha) * pDestPixel[i] + alpha * pSrcPixel[i]);
                pDestPixel[3] = pSrcPixel[3];
            }

            // Finish locking
            pTmpColorBitmap->UnlockBits(&bmpData);
        }

        // pPreviewBitmap Finish locking
        pPreviewBitmap->UnlockBits(&bmpResult);

        Gdiplus::Rect drawRect(0, 0, pPreviewBitmap->GetWidth(), pPreviewBitmap->GetHeight());

        CreateColorAndAlphaMap(pTexture,pPreviewBitmap,originalFmt,drawRect);
    }

    //for small icon
    {
        Gdiplus::Bitmap *pPreviewSmallBitmap = &img;
        if (pPreviewSmallBitmap==NULL)
        {
            return false;
        }

        Gdiplus::Rect lockResultRect(0,0,pPreviewSmallBitmap->GetWidth(),pPreviewSmallBitmap->GetHeight());
        Gdiplus::BitmapData bmpResult;
        if (pPreviewSmallBitmap->LockBits(&lockResultRect,Gdiplus::ImageLockModeWrite,
            pPreviewSmallBitmap->GetPixelFormat(),
            &bmpResult)!=Gdiplus::Ok)
        {
            return false;
        }

        unsigned char *pDestData = (unsigned char*)bmpResult.Scan0;

        //Fill the rectangle with Glay
        std::unique_ptr<Gdiplus::Graphics> graphicsColor(Gdiplus::Graphics::FromImage( pPreviewSmallBitmap ));
        Gdiplus::SolidBrush grayBrush(Gdiplus::Color(128,128,128,128));
        graphicsColor->FillRectangle(&grayBrush,0,0,pPreviewSmallBitmap->GetWidth(),pPreviewSmallBitmap->GetHeight());

        std::unique_ptr<Gdiplus::Bitmap> pTmpColorBitmap(new Gdiplus::Bitmap(pTexture->GetWidth(),pTexture->GetHeight(),PixelFormat32bppARGB));
        if (pTmpColorBitmap==NULL)
        {
            // pPreviewSmallBitmap Finish locking
            pPreviewSmallBitmap->UnlockBits(&bmpResult);
            return false;
        } // End if

        Gdiplus::BitmapData bmpData;
        if (!GetImageByDepth(pTmpColorBitmap.get(),
            detiledSurface,
            originalFmt,
            0
            ))
        {
            // pPreviewSmallBitmap Finish locking
            pPreviewSmallBitmap->UnlockBits(&bmpResult);
            return false;
        }

        // Lock bits to prepare preview image
        Gdiplus::Rect lockRect(0,0,pTmpColorBitmap->GetWidth(),pTmpColorBitmap->GetHeight());
        if (pTmpColorBitmap->LockBits(&lockRect,Gdiplus::ImageLockModeRead,
            pPreviewSmallBitmap->GetPixelFormat(),
            &bmpData)!=Gdiplus::Ok)
        {
            // pPreviewSmallBitmap Finish locking
            pPreviewSmallBitmap->UnlockBits(&bmpResult);
            return false;
        }

        unsigned char *pSrcData  = (unsigned char*)bmpData.Scan0;

        for(int x = 0;x < (int)bmpData.Width;x++)
        {
            int nSrcIndex = x * 4;
            int yy = (int)(4 * bmpResult.Height / 8.0);
            int xx = (int)(1 * bmpResult.Width / 8.0 + x * 3.0 / 4.0);
            int nDestIndex = (int)(yy * bmpResult.Stride + xx * 4);

            unsigned char *pSrcPixel = pSrcData + nSrcIndex;
            unsigned char *pDestPixel = pDestData + nDestIndex;

            float alpha = pSrcPixel[3]/255.f;
            for(int i = 0;i < 3;i++)
                pDestPixel[i] = (unsigned char)((1.f-alpha) * pDestPixel[i] + alpha * pSrcPixel[i]);
            pDestPixel[3] = pSrcPixel[3];
        }

        // Finish locking
        pTmpColorBitmap->UnlockBits(&bmpData);

        // pPreviewSmallBitmap Finish locking
        pPreviewSmallBitmap->UnlockBits(&bmpResult);

        //Output the final result
        Gdiplus::Rect drawRect(0, 0, pPreviewSmallBitmap->GetWidth(), pPreviewSmallBitmap->GetHeight());

        CreateColorAndAlphaMap(pTexture,pPreviewSmallBitmap,originalFmt,drawRect,true);
    }

    return true;
}

bool ImageToPreview2DImage( CNWTexture *pTexture,
                            RSurface &detiledSurface,
                            OriginalImageFormat originalFmt
                            )
{
    // 2DImage の場合は GetWidth と GetHeight を使う
    Gdiplus::Bitmap img(pTexture->GetWidth(), pTexture->GetHeight(), PixelFormat32bppARGB);

    const int depthLevel = 0;
    if (!GetImageByDepth(&img,
                         detiledSurface,
                         originalFmt,
                         depthLevel
                         ))
    {
        return false;
    }   // End if

    float drawImgX  = 0.0f, drawImgY  = 0.0f,drawImgWd = 0.0f,drawImgHt = 0.0f;
    GetImageRect(pTexture,&drawImgX,&drawImgY,&drawImgWd,&drawImgHt);

    Gdiplus::Rect drawRect((int)drawImgX, (int)drawImgY, (int)drawImgWd, (int)drawImgHt);

    return CreateColorAndAlphaMap(pTexture,&img,originalFmt,drawRect);
}

struct CubeInfo
{
    int xStartOffset;
    int yStartOffset;
    bool fFlipUV;
    bool bTopNeighbor;
    bool bRightNeighbor;
    bool bBottomNeighbor;
    bool bLeftNeighbor;
};

#define CUBE_FACE_NUM 6

const CubeInfo s_cubeInfo[CUBE_FACE_NUM] = {
                                {2, 1, false, false, true, false, true},  // positive x
                                {0, 1, false, false, true, false, false}, // negative x
                                {1, 0, true , false, false, true, false}, // positive y
                                {1, 2, true , true, false, false, false}, // negative y
                                {3, 1, false, false, false, false, true}, // positive z
                                {1, 1, false, true, true, true, true} };  // negative z

bool IsInsideOfCubeFace( int faceWidth,int faceHeight,int x,int y)
{
    for(int i = 0;i < CUBE_FACE_NUM;i++)
    {
        int nXIndex = (int)floor(x / faceWidth);
        int nYIndex = (int)floor(y / faceHeight);
        if(s_cubeInfo[i].xStartOffset == nXIndex &&
            s_cubeInfo[i].yStartOffset == nYIndex)
            return true;
    }
    return false;
}

//------------------------------------------------------------------------------
// Cube image to preview image
//------------------------------------------------------------------------------
bool ImageToPreviewCubeImage( CNWTexture *pTexture,
                              RSurface &detiledSurface,
                              OriginalImageFormat originalFmt
                              )
{
    if (pTexture==NULL)
    {
        return false;
    } // End if

    Gdiplus::Bitmap img(pTexture->GetPreviewWidth(), pTexture->GetPreviewHeight(), PixelFormat32bppARGB);
    Gdiplus::Bitmap *pPreviewBitmap = &img;
    if (pPreviewBitmap==NULL)
    {
        return false;
    } // End if

    //Fill the rectangle with Glay
    std::unique_ptr<Gdiplus::Graphics> graphicsColor(Gdiplus::Graphics::FromImage( pPreviewBitmap ));
    Gdiplus::SolidBrush grayBrush(Gdiplus::Color(128,128,128,128));
    graphicsColor->FillRectangle(&grayBrush,0,0,pPreviewBitmap->GetWidth(),pPreviewBitmap->GetHeight());

    int faceWidth = pTexture->GetWidth();
    int faceHeight = pTexture->GetHeight();
    if (faceWidth != faceHeight)
    {
        return false;
    }

    Gdiplus::Bitmap faceBitmap(faceWidth, faceHeight, PixelFormat32bppARGB);

    std::unique_ptr<Gdiplus::Graphics> graphics(Gdiplus::Graphics::FromImage( pPreviewBitmap ));
    Gdiplus::Image* image = &faceBitmap;

    for(int i=0; i<6; ++i)
    {
        auto index = i;
        if (originalFmt == ORIGINAL_FORMAT_INVALID)
        {
            // キューブマップのテクスチャーデータは右手座標系で +X、-X、+Y、-Y、-Z、+Z の順に格納されます。
            // オリジナルイメージの順番にあわせて +X、-X、+Y、-Y、+Z、-Z の順にして読み込みます。
            switch (i)
            {
            case 4:
                index = 5;
                break;
            case 5:
                index = 4;
                break;
            }
        }

        if (!GetImageByDepth(&faceBitmap,
                             detiledSurface,
                             originalFmt,
                             index
                             ))
        {
            return false;
        }   // End if
        graphics->SetCompositingMode(Gdiplus::CompositingModeSourceCopy);
        graphics->DrawImage(image,
                            faceWidth * s_cubeInfo[i].xStartOffset,
                            faceHeight * s_cubeInfo[i].yStartOffset);
    }   // End for

    Gdiplus::Rect drawRect(0, 0, pPreviewBitmap->GetWidth(), pPreviewBitmap->GetHeight());

    return CreateColorAndAlphaMap(pTexture,pPreviewBitmap,originalFmt,drawRect);
}

//------------------------------------------------------------------------------
// 2D Array images to preview image
//------------------------------------------------------------------------------
bool ImageToPreview2DArrayImage( CNWTexture *pTexture,
                         RSurface &detiledSurface,
                         OriginalImageFormat originalFmt
                         )
{
    if (pTexture==NULL)
    {
        return false;
    } // End if

    Gdiplus::Bitmap img(pTexture->GetPreviewWidth(), pTexture->GetPreviewHeight(), PixelFormat32bppARGB);
    // for large icon
    {
        Gdiplus::Bitmap *pPreviewBitmap = &img;
        if (pPreviewBitmap==NULL)
        {
            return false;
        }

        //Prepare bitmap to write the result of thumbnail.
        Gdiplus::Rect lockResultRect(0,0,pPreviewBitmap->GetWidth(),pPreviewBitmap->GetHeight());
        Gdiplus::BitmapData bmpResult;
        if (pPreviewBitmap->LockBits(&lockResultRect,Gdiplus::ImageLockModeWrite,
            pPreviewBitmap->GetPixelFormat(),
            &bmpResult)!=Gdiplus::Ok)
        {
            return false;
        }

        //Fill the rectangle with Glay.
        std::unique_ptr<Gdiplus::Graphics> graphicsColor(Gdiplus::Graphics::FromImage( pPreviewBitmap ));
        Gdiplus::SolidBrush grayBrush(Gdiplus::Color(128,128,128,128));
        graphicsColor->FillRectangle(&grayBrush,0,0,pPreviewBitmap->GetWidth(),pPreviewBitmap->GetHeight());

        unsigned char *pDestData = (unsigned char*)bmpResult.Scan0;

        //The scratching BMP to aquire texture in texture array.
        std::unique_ptr<Gdiplus::Bitmap> pTmpColorBitmap(new Gdiplus::Bitmap(pPreviewBitmap->GetWidth(),pPreviewBitmap->GetHeight(),PixelFormat32bppARGB));
        if (pTmpColorBitmap==NULL)
        {
           // pPreviewBitmap Finish locking
            pPreviewBitmap->UnlockBits(&bmpResult);
            return false;
        } // End if

        Gdiplus::Pen pen(Gdiplus::Color(0,0,0),30.0f);

        int nCountMax = (2 <= pTexture->GetCount() - 1) ? 2 : pTexture->GetCount() - 1;
        for(int count = nCountMax;count >= 0;count--)
        {
            Gdiplus::BitmapData bmpData;
            if (!GetImageByDepth(pTmpColorBitmap.get(),
                detiledSurface,
                originalFmt,
                count
                ))
            {
                // pPreviewBitmap Finish locking
                pPreviewBitmap->UnlockBits(&bmpResult);
                return false;
            }   // End if

            // Lock bits to prepare preview image
            Gdiplus::Rect lockRect(0,0,pTmpColorBitmap->GetWidth(),pTmpColorBitmap->GetHeight());
            if (pTmpColorBitmap->LockBits(&lockRect,Gdiplus::ImageLockModeRead,
                pPreviewBitmap->GetPixelFormat(),
                &bmpData)!=Gdiplus::Ok)
            {
                // pPreviewBitmap Finish locking
                pPreviewBitmap->UnlockBits(&bmpResult);
                return false;
            }   // End if

            unsigned char *pSrcData  = (unsigned char*)bmpData.Scan0;

            for(int x = 0;x < (int)bmpData.Width;x++)
            {
                for(int y = 0;y < (int)bmpData.Height;y++)
                {
                    int nSrcIndex = y * bmpData.Stride  + x * 4;
                    int yy = (int)(count * bmpResult.Height / 8.0 + y * 3.0 / 4.0);
                    int xx = (int)(count * bmpResult.Width / 8.0 + x * 3.0 / 4.0);
                    int nDestIndex = (int)(yy * bmpResult.Stride + xx * 4);

                    unsigned char *pSrcPixel = pSrcData + nSrcIndex;
                    unsigned char *pDestPixel = pDestData + nDestIndex;

                    if(xx == (int)(count * bmpResult.Width / 8.0)
                        || xx == (int)(count * bmpResult.Width / 8.0 + (bmpData.Width - 1) * 3.0 / 4.0)
                        || yy == (int)(count * bmpResult.Height / 8.0)
                        || yy == (int)(count * bmpResult.Height / 8.0 + (bmpData.Height - 1) * 3.0 / 4.0))
                    {
                        for(int i = 0;i < 3;i++)
                            pDestPixel[i] = 0;
                        pDestPixel[3] = 255;
                        continue;
                    }   // End if

                    float alpha = pSrcPixel[3]/255.f;
                    for(int i = 0;i < 3;i++)
                        pDestPixel[i] = (unsigned char)((1.f-alpha) * pDestPixel[i] + alpha * pSrcPixel[i]);
                    pDestPixel[3] = pSrcPixel[3];
                }   // End for
            }   // End for

            // Finish locking
            pTmpColorBitmap->UnlockBits(&bmpData);
        }   // End for

        // pPreviewBitmap Finish locking
        pPreviewBitmap->UnlockBits(&bmpResult);

        Gdiplus::Rect drawRect(0, 0, pPreviewBitmap->GetWidth(), pPreviewBitmap->GetHeight());

        CreateColorAndAlphaMap(pTexture,pPreviewBitmap,originalFmt,drawRect);
    }

    //for small icon
    {
        Gdiplus::Bitmap *pPreviewSmallBitmap = &img;
        if (pPreviewSmallBitmap==NULL)
        {
            return false;
        } // End if

        const int depthLevel = 0;
        if (!GetImageByDepth(pPreviewSmallBitmap,
                             detiledSurface,
                             originalFmt,
                             depthLevel
                             ))
        {
            return false;
        }   // End if

        float drawImgX  = 0.0f, drawImgY  = 0.0f,drawImgWd = 0.0f,drawImgHt = 0.0f;
        GetImageRect(pTexture,&drawImgX,&drawImgY,&drawImgWd,&drawImgHt);

        Gdiplus::Rect drawRect((int)drawImgX, (int)drawImgY, (int)drawImgWd, (int)drawImgHt);

        CreateColorAndAlphaMap(pTexture,pPreviewSmallBitmap,originalFmt, drawRect,true);
    }
    return true;
}

//------------------------------------------------------------------------------
// Cube Array images to preview image
//------------------------------------------------------------------------------
bool ImageToPreviewCubeArrayImage( CNWTexture *pTexture,
                         RSurface &detiledSurface,
                         OriginalImageFormat originalFmt
                         )
{
    //for normal and large icon
    if (pTexture==NULL)
    {
        return false;
    } // End if

    int faceWidth = pTexture->GetWidth();
    int faceHeight = pTexture->GetHeight();
    if (faceWidth != faceHeight)
    {
        return false;
    }

    Gdiplus::Bitmap img(pTexture->GetPreviewWidth(), pTexture->GetPreviewHeight(), PixelFormat32bppARGB);
    {
        Gdiplus::Bitmap *pPreviewBitmap = &img;
        if (pPreviewBitmap==NULL)
        {
            return false;
        } // End if

        Gdiplus::Bitmap faceBitmap(faceWidth, faceHeight, PixelFormat32bppARGB);

        //Prepare bitmap to write the result of thumbnail
        Gdiplus::Rect lockResultRect(0,0,pPreviewBitmap->GetWidth(),pPreviewBitmap->GetHeight());
        Gdiplus::BitmapData bmpResult;
        if (pPreviewBitmap->LockBits(&lockResultRect,Gdiplus::ImageLockModeWrite,
                                         pPreviewBitmap->GetPixelFormat(),
                                         &bmpResult)!=Gdiplus::Ok)
        {
            return false;
        }   // End if

        unsigned char *pDestData = (unsigned char*)bmpResult.Scan0;

        //Fill the rectangle with Glay
        std::unique_ptr<Gdiplus::Graphics> graphicsColor(Gdiplus::Graphics::FromImage( pPreviewBitmap ));
        Gdiplus::SolidBrush grayBrush(Gdiplus::Color(128,128,128,128));
        graphicsColor->FillRectangle(&grayBrush,0,0,pPreviewBitmap->GetWidth(),pPreviewBitmap->GetHeight());

        //The scratching BMP to aquire texture in texture array.
        std::unique_ptr<Gdiplus::Bitmap> pTmpColorBitmap(new Gdiplus::Bitmap(bmpResult.Width,bmpResult.Height,PixelFormat32bppARGB));
        if (pTmpColorBitmap==NULL)
        {
            // pPreviewBitmap Finish locking
            pPreviewBitmap->UnlockBits(&bmpResult);
            return false;
        }   // End if

        Gdiplus::Pen pen(Gdiplus::Color(0,0,0),5.0f);

        int nCountMax = (2 <= pTexture->GetCount() - 1) ? 2 : pTexture->GetCount() - 1;
        for(int countLevel = nCountMax;countLevel >= 0;--countLevel)
        {
            std::unique_ptr<Gdiplus::Graphics> graphics(Gdiplus::Graphics::FromImage( pTmpColorBitmap.get() ));
            Gdiplus::Image* image = &faceBitmap;

            for(int i=0; i<6; ++i)
            {
                auto index = i;
                if (originalFmt == ORIGINAL_FORMAT_INVALID)
                {
                    // キューブマップのテクスチャーデータは右手座標系で +X、-X、+Y、-Y、-Z、+Z の順に格納されます。
                    // オリジナルイメージの順番にあわせて +X、-X、+Y、-Y、+Z、-Z の順にして読み込みます。
                    switch (i)
                    {
                    case 4:
                        index = 5;
                        break;
                    case 5:
                        index = 4;
                        break;
                    }
                }

                if (!GetImageByDepth(&faceBitmap,
                                     detiledSurface,
                                     originalFmt,
                                     6 * countLevel + index
                                     ))
                {
                    // pPreviewBitmap Finish locking
                    pPreviewBitmap->UnlockBits(&bmpResult);
                    return false;
                }   // End if
                graphics->SetCompositingMode(Gdiplus::CompositingModeSourceCopy);
                graphics->DrawImage(image,
                                    faceWidth * s_cubeInfo[i].xStartOffset,
                                    faceHeight * s_cubeInfo[i].yStartOffset);

                ////Draw Lines on the edges of the faces
                //Top-Left
                Gdiplus::Point point1(faceWidth * s_cubeInfo[i].xStartOffset
                    ,faceHeight * s_cubeInfo[i].yStartOffset);
                //Top-Right
                Gdiplus::Point point2(faceWidth * (s_cubeInfo[i].xStartOffset+1) - 1
                    ,faceHeight * s_cubeInfo[i].yStartOffset);
                //Bottom-Left
                Gdiplus::Point point3(faceWidth * s_cubeInfo[i].xStartOffset
                    ,faceHeight * (s_cubeInfo[i].yStartOffset + 1) - 1);
                //Bottom-Right
                Gdiplus::Point point4(faceWidth * (s_cubeInfo[i].xStartOffset+1) - 1
                    ,faceHeight * (s_cubeInfo[i].yStartOffset + 1) - 1);

                if(!s_cubeInfo[i].bTopNeighbor)
                    graphics->DrawLine(&pen,point1,point2);	//top
                if(!s_cubeInfo[i].bRightNeighbor)
                    graphics->DrawLine(&pen,point2,point4);	//right
                if(!s_cubeInfo[i].bBottomNeighbor)
                    graphics->DrawLine(&pen,point4,point3); //bottom
                if(!s_cubeInfo[i].bLeftNeighbor)
                    graphics->DrawLine(&pen,point1,point3); //left
            }   // End for

            Gdiplus::BitmapData bmpData;
            // Lock bits to prepare preview image
            Gdiplus::Rect lockRect(0,0,pTmpColorBitmap->GetWidth(),pTmpColorBitmap->GetHeight());
            if (pTmpColorBitmap->LockBits(&lockRect,Gdiplus::ImageLockModeRead,
                                         pPreviewBitmap->GetPixelFormat(),
                                         &bmpData)!=Gdiplus::Ok)
            {
                // pPreviewBitmap Finish locking
                pPreviewBitmap->UnlockBits(&bmpResult);
                return false;
            }   // End if

            // Draw cubemap on the result bmp
            unsigned char *pSrcData  = (unsigned char*)bmpData.Scan0;

            for(int x = 0;x < (int)bmpData.Width;x++)
            {
                for(int y = 0;y < (int)bmpData.Height;y++)
                {
                    if(!IsInsideOfCubeFace(faceWidth,faceHeight,x,y))
                        continue;

                    int nSrcIndex = y * bmpData.Stride  + x * 4;
                    int yy = (int)(countLevel * bmpResult.Height / 8.0 + y * 3.0 / 4.0);
                    int xx = (int)(countLevel * bmpResult.Width / 8.0 + x * 3.0 / 4.0);
                    int nDestIndex = (int)(yy * bmpResult.Stride + xx * 4);

                    unsigned char *pSrcPixel = pSrcData + nSrcIndex;
                    unsigned char *pDestPixel = pDestData + nDestIndex;

                    //float alpha = pSrcPixel[3]/255.f;
                    for(int i = 0;i < 3;i++)
                        pDestPixel[i] = pSrcPixel[i];
                    pDestPixel[3] = pSrcPixel[3];
                }   // End for
            }   // End for

            // Finish locking
            pTmpColorBitmap->UnlockBits(&bmpData);
        }   // End for

        // pPreviewBitmap Finish locking
        pPreviewBitmap->UnlockBits(&bmpResult);

        Gdiplus::Rect drawRect(0, 0, pPreviewBitmap->GetWidth(), pPreviewBitmap->GetHeight());

        if(!CreateColorAndAlphaMap(pTexture,pPreviewBitmap,originalFmt,drawRect))
            return false;
    }

    //-------------------------------------------------------------------------------
    //for small icon
    {
        Gdiplus::Bitmap *pPreviewSmallBitmap = &img;
        if (pPreviewSmallBitmap==NULL)
        {
            return false;
        } // End if

        //Fill the rectangle with Glay
        std::unique_ptr<Gdiplus::Graphics> graphicsColor(Gdiplus::Graphics::FromImage( pPreviewSmallBitmap ));
        Gdiplus::SolidBrush grayBrush(Gdiplus::Color(128,128,128,128));
        graphicsColor->FillRectangle(&grayBrush,0,0,pPreviewSmallBitmap->GetWidth(),pPreviewSmallBitmap->GetHeight());

        faceWidth = pTexture->GetWidth();
        faceHeight = pTexture->GetHeight();
        if (faceWidth != faceHeight)
        {
            return false;
        }   // End if

        Gdiplus::Bitmap faceBitmap(faceWidth, faceHeight, PixelFormat32bppARGB);

        std::unique_ptr<Gdiplus::Graphics> graphics(Gdiplus::Graphics::FromImage( pPreviewSmallBitmap ));
        Gdiplus::Image* image = &faceBitmap;

        for(int i=0; i<6; ++i)
        {
            auto index = i;
            if (originalFmt == ORIGINAL_FORMAT_INVALID)
            {
                // キューブマップのテクスチャーデータは右手座標系で +X、-X、+Y、-Y、-Z、+Z の順に格納されます。
                // オリジナルイメージの順番にあわせて +X、-X、+Y、-Y、+Z、-Z の順にして読み込みます。
                switch (i)
                {
                case 4:
                    index = 5;
                    break;
                case 5:
                    index = 4;
                    break;
                }
            }

            if (!GetImageByDepth(&faceBitmap,
                                 detiledSurface,
                                 originalFmt,
                                 index
                                 ))
            {
                return false;
            }   // End if
            graphics->SetCompositingMode(Gdiplus::CompositingModeSourceCopy);
            graphics->DrawImage(image,
                                faceWidth * s_cubeInfo[i].xStartOffset,
                                faceHeight * s_cubeInfo[i].yStartOffset);
        }   // End for

        Gdiplus::Rect drawRect(0, 0, pPreviewSmallBitmap->GetWidth(), pPreviewSmallBitmap->GetHeight());

        if(!CreateColorAndAlphaMap(pTexture,pPreviewSmallBitmap,originalFmt,drawRect,true))
            return false;
    }

    return true;
}

bool ImageToPreview3DImage( CNWTexture *pTexture,
                            RSurface &detiledSurface,
                            OriginalImageFormat originalFmt
                            )
{
    if (pTexture==NULL)
    {
        return false;
    } // End if

    Gdiplus::Bitmap img(pTexture->GetPreviewWidth(), pTexture->GetPreviewHeight(), PixelFormat32bppARGB);
    Gdiplus::Bitmap *pPreviewBitmap = &img;
    if (pPreviewBitmap==NULL)
    {
        return false;
    }   // End if

    const int depthLevel = pTexture->GetDepth() / 2;
    if (!GetImageByDepth(pPreviewBitmap,
                         detiledSurface,
                         originalFmt,
                         depthLevel
                         ))
    {
        return false;
    }   // End if

    Gdiplus::Rect drawRect(0, 0, pPreviewBitmap->GetWidth(), pPreviewBitmap->GetHeight());

    if(!CreateColorAndAlphaMap(pTexture,pPreviewBitmap,originalFmt, drawRect))
        return false;

    return true;
}

OriginalImageFormat GetOriginalImageFormatFromString( std::string formatTypeStr )
{
    const char* str = formatTypeStr.c_str();
    if      (_stricmp(str, "rgb8")      ==0) return ORIGINAL_FORMAT_RGB8;
    else if (_stricmp(str, "rgba8")     ==0) return ORIGINAL_FORMAT_RGBA8;
    else if (_stricmp(str, "rgb32f")    ==0) return ORIGINAL_FORMAT_RGB32F;
    else if (_stricmp(str, "rgba32f")   ==0) return ORIGINAL_FORMAT_RGBA32F;

    return ORIGINAL_FORMAT_INVALID;
} // End of GetOriginalImageFormatFromString


namespace TexUtils{

namespace
{

const unsigned char Table0F[0x10] =
{
    (unsigned char)(float(0xFF) * float(0x00) / float(0x0F)),
    (unsigned char)(float(0xFF) * float(0x01) / float(0x0F)),
    (unsigned char)(float(0xFF) * float(0x02) / float(0x0F)),
    (unsigned char)(float(0xFF) * float(0x03) / float(0x0F)),
    (unsigned char)(float(0xFF) * float(0x04) / float(0x0F)),
    (unsigned char)(float(0xFF) * float(0x05) / float(0x0F)),
    (unsigned char)(float(0xFF) * float(0x06) / float(0x0F)),
    (unsigned char)(float(0xFF) * float(0x07) / float(0x0F)),
    (unsigned char)(float(0xFF) * float(0x08) / float(0x0F)),
    (unsigned char)(float(0xFF) * float(0x09) / float(0x0F)),
    (unsigned char)(float(0xFF) * float(0x0A) / float(0x0F)),
    (unsigned char)(float(0xFF) * float(0x0B) / float(0x0F)),
    (unsigned char)(float(0xFF) * float(0x0C) / float(0x0F)),
    (unsigned char)(float(0xFF) * float(0x0D) / float(0x0F)),
    (unsigned char)(float(0xFF) * float(0x0E) / float(0x0F)),
    (unsigned char)(float(0xFF) * float(0x0F) / float(0x0F)),
};

const unsigned char Table1F[0x20] =
{
    (unsigned char)(float(0xFF) * float(0x00) / float(0x1F)),
    (unsigned char)(float(0xFF) * float(0x01) / float(0x1F)),
    (unsigned char)(float(0xFF) * float(0x02) / float(0x1F)),
    (unsigned char)(float(0xFF) * float(0x03) / float(0x1F)),
    (unsigned char)(float(0xFF) * float(0x04) / float(0x1F)),
    (unsigned char)(float(0xFF) * float(0x05) / float(0x1F)),
    (unsigned char)(float(0xFF) * float(0x06) / float(0x1F)),
    (unsigned char)(float(0xFF) * float(0x07) / float(0x1F)),
    (unsigned char)(float(0xFF) * float(0x08) / float(0x1F)),
    (unsigned char)(float(0xFF) * float(0x09) / float(0x1F)),
    (unsigned char)(float(0xFF) * float(0x0A) / float(0x1F)),
    (unsigned char)(float(0xFF) * float(0x0B) / float(0x1F)),
    (unsigned char)(float(0xFF) * float(0x0C) / float(0x1F)),
    (unsigned char)(float(0xFF) * float(0x0D) / float(0x1F)),
    (unsigned char)(float(0xFF) * float(0x0E) / float(0x1F)),
    (unsigned char)(float(0xFF) * float(0x0F) / float(0x1F)),
    (unsigned char)(float(0xFF) * float(0x10) / float(0x1F)),
    (unsigned char)(float(0xFF) * float(0x11) / float(0x1F)),
    (unsigned char)(float(0xFF) * float(0x12) / float(0x1F)),
    (unsigned char)(float(0xFF) * float(0x13) / float(0x1F)),
    (unsigned char)(float(0xFF) * float(0x14) / float(0x1F)),
    (unsigned char)(float(0xFF) * float(0x15) / float(0x1F)),
    (unsigned char)(float(0xFF) * float(0x16) / float(0x1F)),
    (unsigned char)(float(0xFF) * float(0x17) / float(0x1F)),
    (unsigned char)(float(0xFF) * float(0x18) / float(0x1F)),
    (unsigned char)(float(0xFF) * float(0x19) / float(0x1F)),
    (unsigned char)(float(0xFF) * float(0x1A) / float(0x1F)),
    (unsigned char)(float(0xFF) * float(0x1B) / float(0x1F)),
    (unsigned char)(float(0xFF) * float(0x1C) / float(0x1F)),
    (unsigned char)(float(0xFF) * float(0x1D) / float(0x1F)),
    (unsigned char)(float(0xFF) * float(0x1E) / float(0x1F)),
    (unsigned char)(float(0xFF) * float(0x1F) / float(0x1F)),
};

const unsigned char Table3F[0x40] =
{
    (unsigned char)(float(0xFF) * float(0x00) / float(0x3F)),
    (unsigned char)(float(0xFF) * float(0x01) / float(0x3F)),
    (unsigned char)(float(0xFF) * float(0x02) / float(0x3F)),
    (unsigned char)(float(0xFF) * float(0x03) / float(0x3F)),
    (unsigned char)(float(0xFF) * float(0x04) / float(0x3F)),
    (unsigned char)(float(0xFF) * float(0x05) / float(0x3F)),
    (unsigned char)(float(0xFF) * float(0x06) / float(0x3F)),
    (unsigned char)(float(0xFF) * float(0x07) / float(0x3F)),
    (unsigned char)(float(0xFF) * float(0x08) / float(0x3F)),
    (unsigned char)(float(0xFF) * float(0x09) / float(0x3F)),
    (unsigned char)(float(0xFF) * float(0x0A) / float(0x3F)),
    (unsigned char)(float(0xFF) * float(0x0B) / float(0x3F)),
    (unsigned char)(float(0xFF) * float(0x0C) / float(0x3F)),
    (unsigned char)(float(0xFF) * float(0x0D) / float(0x3F)),
    (unsigned char)(float(0xFF) * float(0x0E) / float(0x3F)),
    (unsigned char)(float(0xFF) * float(0x0F) / float(0x3F)),
    (unsigned char)(float(0xFF) * float(0x10) / float(0x3F)),
    (unsigned char)(float(0xFF) * float(0x11) / float(0x3F)),
    (unsigned char)(float(0xFF) * float(0x12) / float(0x3F)),
    (unsigned char)(float(0xFF) * float(0x13) / float(0x3F)),
    (unsigned char)(float(0xFF) * float(0x14) / float(0x3F)),
    (unsigned char)(float(0xFF) * float(0x15) / float(0x3F)),
    (unsigned char)(float(0xFF) * float(0x16) / float(0x3F)),
    (unsigned char)(float(0xFF) * float(0x17) / float(0x3F)),
    (unsigned char)(float(0xFF) * float(0x18) / float(0x3F)),
    (unsigned char)(float(0xFF) * float(0x19) / float(0x3F)),
    (unsigned char)(float(0xFF) * float(0x1A) / float(0x3F)),
    (unsigned char)(float(0xFF) * float(0x1B) / float(0x3F)),
    (unsigned char)(float(0xFF) * float(0x1C) / float(0x3F)),
    (unsigned char)(float(0xFF) * float(0x1D) / float(0x3F)),
    (unsigned char)(float(0xFF) * float(0x1E) / float(0x3F)),
    (unsigned char)(float(0xFF) * float(0x1F) / float(0x3F)),
    (unsigned char)(float(0xFF) * float(0x20) / float(0x3F)),
    (unsigned char)(float(0xFF) * float(0x21) / float(0x3F)),
    (unsigned char)(float(0xFF) * float(0x22) / float(0x3F)),
    (unsigned char)(float(0xFF) * float(0x23) / float(0x3F)),
    (unsigned char)(float(0xFF) * float(0x24) / float(0x3F)),
    (unsigned char)(float(0xFF) * float(0x25) / float(0x3F)),
    (unsigned char)(float(0xFF) * float(0x26) / float(0x3F)),
    (unsigned char)(float(0xFF) * float(0x27) / float(0x3F)),
    (unsigned char)(float(0xFF) * float(0x28) / float(0x3F)),
    (unsigned char)(float(0xFF) * float(0x29) / float(0x3F)),
    (unsigned char)(float(0xFF) * float(0x2A) / float(0x3F)),
    (unsigned char)(float(0xFF) * float(0x2B) / float(0x3F)),
    (unsigned char)(float(0xFF) * float(0x2C) / float(0x3F)),
    (unsigned char)(float(0xFF) * float(0x2D) / float(0x3F)),
    (unsigned char)(float(0xFF) * float(0x2E) / float(0x3F)),
    (unsigned char)(float(0xFF) * float(0x2F) / float(0x3F)),
    (unsigned char)(float(0xFF) * float(0x30) / float(0x3F)),
    (unsigned char)(float(0xFF) * float(0x31) / float(0x3F)),
    (unsigned char)(float(0xFF) * float(0x32) / float(0x3F)),
    (unsigned char)(float(0xFF) * float(0x33) / float(0x3F)),
    (unsigned char)(float(0xFF) * float(0x34) / float(0x3F)),
    (unsigned char)(float(0xFF) * float(0x35) / float(0x3F)),
    (unsigned char)(float(0xFF) * float(0x36) / float(0x3F)),
    (unsigned char)(float(0xFF) * float(0x37) / float(0x3F)),
    (unsigned char)(float(0xFF) * float(0x38) / float(0x3F)),
    (unsigned char)(float(0xFF) * float(0x39) / float(0x3F)),
    (unsigned char)(float(0xFF) * float(0x3A) / float(0x3F)),
    (unsigned char)(float(0xFF) * float(0x3B) / float(0x3F)),
    (unsigned char)(float(0xFF) * float(0x3C) / float(0x3F)),
    (unsigned char)(float(0xFF) * float(0x3D) / float(0x3F)),
    (unsigned char)(float(0xFF) * float(0x3E) / float(0x3F)),
    (unsigned char)(float(0xFF) * float(0x3F) / float(0x3F)),
};

template<bool, typename T>	inline T Select(T trueValue, T falseValue);
template<>					inline unsigned int Select<true>(unsigned int trueValue, unsigned int falseValue){
    NW_USE_VAR(falseValue);
    return trueValue;
}
template<>					inline unsigned int Select<false>(unsigned int trueValue, unsigned int falseValue){
    NW_USE_VAR(trueValue);
    return falseValue;
}

template<bool HasSign, unsigned int ExpSize, unsigned int FraSize>
inline float ToFloat32(unsigned int src)
{
    // 注意：NaN,無限大は無視してます。

    // ■浮動小数点数
    // http://ja.wikipedia.org/wiki/%E6%B5%AE%E5%8B%95%E5%B0%8F%E6%95%B0%E7%82%B9%E6%95%B0

    // ■11 ビットおよび 10 ビット浮動小数点の規則
    // http://msdn.microsoft.com/ja-jp/library/cc308050(v=vs.85)

    enum
    {
        // IEEE 754 単精度浮動小数点
        IeeeExpSize = 8,
        IeeeFlaSize = 23,
        IeeeBias    = (1 << (IeeeExpSize - 1)) - 1,

        // 入力値
        SrcSgnMask = 1 << (ExpSize + FraSize),
        SrcExpMask = ((1 << ExpSize) - 1) << FraSize,
        SrcFraMask = (1 << FraSize) - 1,
        SrcBias    = (1 << (ExpSize - 1)) - 1
    };

    //
    const unsigned int srcSgn = (src & SrcSgnMask) >> (ExpSize + FraSize);
    const unsigned int srcExp = (src & SrcExpMask) >> FraSize;
    const unsigned int srcFra = (src & SrcFraMask) >> 0;

    if ((srcExp == 0) && (srcFra == 0))
    {
        return 0.0f;
    }
    else
    {
        const unsigned int ansSgn = Select<HasSign, unsigned int>(srcSgn, 0UL);
        const unsigned int ansExp = (srcExp - SrcBias + IeeeBias) & ((1 << IeeeExpSize) - 1);

        unsigned int ansFra = 0;
        {
            int fraDiff = FraSize - IeeeFlaSize;
            if (fraDiff > 0)
            {
                ansFra = (srcFra >> (+fraDiff)) & ((1 << IeeeFlaSize) - 1);
            }
            else if (fraDiff < 0)
            {
                ansFra = (srcFra << (-fraDiff)) & ((1 << IeeeFlaSize) - 1);
            }
            else
            {
                ansFra = srcFra & ((1 << IeeeFlaSize) - 1);
            }
        }

        unsigned int ans =
            (ansSgn << (IeeeExpSize + IeeeFlaSize)) |
            (ansExp << (IeeeFlaSize              )) |
            (ansFra << (0                        ));

        return reinterpret_cast<float &>(ans);
    }
}

inline float Float16ToFloat32(unsigned int src)
{
    return ToFloat32<true, 5, 10>(src);
}

inline float Float11ToFloat32(unsigned int src)
{
    return ToFloat32<false, 5, 6>(src);
}

inline float Float10ToFloat32(unsigned int src)
{
    return ToFloat32<false, 5, 5>(src);
}

}    // namespace

void Decode_original_rgb8(unsigned char *dstColor, const unsigned char *src, int size)
{
    for(int srcIdx = 0, dstIdx = 0;srcIdx != size;srcIdx += 3, dstIdx += 4)
    {
        dstColor[dstIdx + 2] = src[srcIdx + 0];
        dstColor[dstIdx + 1] = src[srcIdx + 1];
        dstColor[dstIdx + 0] = src[srcIdx + 2];
        dstColor[dstIdx + 3] = 0xFF;
    }
}

void Decode_original_rgba8(unsigned char *dstColor, const unsigned char *src, int size)
{
    for(int srcIdx = 0, dstIdx = 0;srcIdx != size;srcIdx += 4, dstIdx += 4)
    {
        dstColor[dstIdx + 2] = src[srcIdx + 0];
        dstColor[dstIdx + 1] = src[srcIdx + 1];
        dstColor[dstIdx + 0] = src[srcIdx + 2];
        dstColor[dstIdx + 3] = src[srcIdx + 3];
    }
}
//==============================================================================
//
// リニア空間のfloatをsRGB空間のbyteに変換
//
//==============================================================================
unsigned char LinearFloatToSrgbByte(float v)
{
    float r = v <= 0.0031308f ? v * 12.92f : pow(v, 1.0f / 2.4f) * 1.055f - 0.055f;
    return (unsigned char)(min(max(0.0, r), 1.0f) * 255.0f);
}

void Decode_original_rgb32f(unsigned char *dstColor, const unsigned char *src, int size)
{
    const float *float32Src = reinterpret_cast<const float *>(src);

    for(int srcIdx = 0, dstIdx = 0;srcIdx != size / 4;srcIdx += 3, dstIdx += 4)
    {
        dstColor[dstIdx + 2] = LinearFloatToSrgbByte(float32Src[srcIdx + 0]);
        dstColor[dstIdx + 1] = LinearFloatToSrgbByte(float32Src[srcIdx + 1]);
        dstColor[dstIdx + 0] = LinearFloatToSrgbByte(float32Src[srcIdx + 2]);
        dstColor[dstIdx + 3] = 0xFF;
    }
}

void Decode_original_rgba32f(unsigned char *dstColor, const unsigned char *src, int size)
{
    const float *float32Src = reinterpret_cast<const float *>(src);

    for(int srcIdx = 0, dstIdx = 0;srcIdx != size / 4;srcIdx += 4, dstIdx += 4)
    {
        dstColor[dstIdx + 2] = LinearFloatToSrgbByte(float32Src[srcIdx + 0]);
        dstColor[dstIdx + 1] = LinearFloatToSrgbByte(float32Src[srcIdx + 1]);
        dstColor[dstIdx + 0] = LinearFloatToSrgbByte(float32Src[srcIdx + 2]);
        dstColor[dstIdx + 3] = (unsigned char)(min(max(float32Src[srcIdx + 3], 0.0f), 1.0f) * 255.0f);
    }
}
#undef min
#undef max

namespace
{

struct RGBA
{
    unsigned char r_;
    unsigned char g_;
    unsigned char b_;
    unsigned char a_;
};

struct IntRGBA
{
    int r_;
    int g_;
    int b_;
    int a_;
};

}    // namespace RGBA

}    // namespace TexUtils

#endif // End of NW_FOR_FTX
