﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <string>
namespace nn {
namespace gfx {
namespace tool {
namespace texenc {

/**
* @brief イメージの次元です。
*/
enum ImageDimension
{
    ImageDimension_1d, //!< 一次元です。
    ImageDimension_2d, //!< 二次元です。
    ImageDimension_3d, //!< 三次元です。
    ImageDimension_CubeMap, //!< キューブマップイメージです。
    ImageDimension_1dArray, //!< 一次元配列です。
    ImageDimension_2dArray, //!< 二次元配列です。
    ImageDimension_2dMultisample, //<! マルチサンプルイメージです。
    ImageDimension_2dMultisampleArray, //!< マルチサンプルイメージ配列です。
    ImageDimension_CubeMapArray, //!< キューブマップイメージ配列です。

    ImageDimension_End
};

/**
* @brief エンコードフラグです。
*/
enum EncodeFlag
{
    EncodeFlag_Default = 0, //!< デフォルトのエンコード方法です。
    EncodeFlag_ReverseRgba   = 0x1 << 0, //!< 16bit RGB(A) / float_11_11_10 フォーマットの場合に下位 bit から RGB(A) の順に格納します。
    EncodeFlag_NormalMapLa   = 0x1 << 1, //!< 法線マップの X 成分が RGB 成分、Y 成分が A 成分となるようにエンコードします。ASTC フォーマットでのみ有効です。
    EncodeFlag_Dither        = 0x1 << 2, //!< RGB 成分にディザを使用します（エンコーダが対応していれば）。
    EncodeFlag_AutoDither    = 0x1 << 3, //!< ゆるやかにカラーが変化する画像なら RGB 成分にディザを使用します（エンコーダが対応していれば）。
    EncodeFlag_GpuEncoding   = 0x1 << 4, //!< 対象フォーマットであれば GPU でエンコードします。
    EncodeFlag_ServerProcess = 0x1 << 5, //!< 可能であればサーバープロセスでエンコードします。
    EncodeFlag_GpuAuto       = 0x1 << 6  //!< GpuEncoding 指定時に GPU が非対応なら CPU でエンコードします。
};

//-----------------------------------------------------------------------------
//! @brief フォーマットに応じたデータサイズ（バイト数）を取得する関数の型です。
//!
//! @param[in] format 中間ファイルのフォーマット文字列です。
//! @param[in] imageW 画像の幅です。
//! @param[in] imageH 画像の高さです。
//! @param[in] imageD 画像の奥行きです。
//!
//! @return データサイズを返します。
//-----------------------------------------------------------------------------
size_t GetDataSize(
    const std::string& format,
    const int imageW,
    const int imageH,
    const int imageD
);

//-----------------------------------------------------------------------------
//! @brief 画像のフォーマットを変換する関数の型です。
//!
//! @param[out] pDst 変換後のデータを格納します。
//! @param[in] pSrc 変換前のデータです。
//! @param[in] dstFormat 変換後のフォーマット文字列です。
//! @param[in] srcFormat 変換前のフォーマット文字列です。
//!                      dstFormat、srcFormat のどちらかはプレーンな RGBA フォーマットである
//!                      必要があります。
//!                      （unorm_8_8_8_8 / snorm_8_8_8_8 / srgb_8_8_8_8 / float_32_32_32_32）
//! @param[in] quality エンコード品質文字列です。
//!                    0 / 1 / 2 / 3 の 4 段階で指定します（空文字なら 1 とみなします）。
//!                    1 が標準的な品質です。数値が大きいほど高品質となり、エンコード時間が増大します。
//!                    1_perceptual のように _perceptual を付加すると RGB に重み付けして圧縮します。
//!                    （bc1/2/3、etc1/2 のみ影響）
//!                    変換後が bc1/2/3 の場合、perceptual の有無のみ影響します。
//!                    変換後が bc7 の場合、0 / 1 の 2 段階のみ有効です。
//!                    （1 なら 3 分割パーティション使用。2 / 3 は 1 と同じ）
//!                    変換後が etc1/2 の場合、1 / 2 の 2 段階のみ有効です。
//!                    （0 は 1 と同じ。3 は 2 と同じ）
//!                    変換後が pvrtc1/2、astc の場合、0 / 1 / 2 / 3 の 4 段階が有効です。
//!                    それ以外の場合、この引数は無視されます（空文字指定で構いません）。
//! @param[in] encodeFlag エンコードフラグです。
//!                       ReverseRgba はデコードの際にも影響します。
//! @param[in] imageW 画像の幅です。
//! @param[in] imageH 画像の高さです。
//! @param[in] imageD 画像の奥行きです。
//!
//! @return 処理成功なら true を返します。
//-----------------------------------------------------------------------------
bool ConvertFormat(
    void* pDst,
    const void* pSrc,
    const std::string& dstFormat,
    const std::string& srcFormat,
    const std::string& quality,
    const int encodeFlag,
    const int imageW,
    const int imageH,
    const int imageD
);

// 変換をサポートしているかフォーマットかどうか簡易的にチェックします。
bool IsSupported(
    const std::string& format
);
} // texenc
} // tool
} // gfx
} // nn

