﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#ifdef NW_FOR_FTX

 // GetDataSize ConvertFormat IsGpuEncodingAvailable
 // ConvertLevelsFormat
 // UsesNvtt InitializeNvtt ConvertByNvtt

 //=============================================================================
 // include
 //=============================================================================
 //#include "ShellExtension_PCH.h"
#include "Encoder.h"
#include "SimpleFormat.h"
#include "BcFormat.h"
#include "AstcFormat.h"

using namespace std;
using namespace nn::gfx::tool::texenc;


//-----------------------------------------------------------------------------
//! @brief 構造化例外の変換関数です。
//-----------------------------------------------------------------------------
void TranslateSe(unsigned int code, struct _EXCEPTION_POINTERS* ep)
{
    throw ep; // 標準 C++ の例外を発生させます。
    ENC_UNUSED_VARIABLE(code);
}


//-----------------------------------------------------------------------------
//! @brief エンコード後のデータサイズ（バイト数）を取得します。
//!
//! @param[in] format 中間ファイルのフォーマット文字列です。
//! @param[in] imageW 画像の幅です。
//! @param[in] imageH 画像の高さです。
//! @param[in] imageD 画像の奥行きです。
//!
//! @return エンコード後のデータサイズを返します。
//-----------------------------------------------------------------------------
size_t nn::gfx::tool::texenc::GetDataSize(
    const std::string& format,
    const int imageW,
    const int imageH,
    const int imageD
)
{
    if (format.empty())
    {
        return 0;
    }
    const bool isAstc = IsAstcFormat(format);
    const size_t bpp = GetBitsPerPixel(format);
    int minW = 1;
    int minH = 1;
    int minD = 1;
    GetMinimumWhd(&minW, &minH, &minD, format);

    size_t dataSize = 0;
    const int levelW = imageW;
    const int levelH = imageH;
    const int levelD = imageD;
    dataSize += GetLevelDataSize(levelW, levelH, levelD, isAstc, bpp, minW, minH, minD);
    return dataSize;
}

//-----------------------------------------------------------------------------
//! @brief 画像のフォーマットを変換します。
//!
//! @param[out] pDst 変換後のデータを格納します。
//! @param[in] pSrc 変換前のデータです。
//! @param[in] dstFormat 変換後のフォーマット文字列です。
//! @param[in] srcFormat 変換前のフォーマット文字列です。
//! @param[in] quality エンコード品質文字列です。
//! @param[in] encodeFlag エンコードフラグです。
//! @param[in] imageW 画像の幅です。
//! @param[in] imageH 画像の高さです。
//! @param[in] imageD 画像の奥行きです。
//!
//! @return 処理成功なら true を返します。
//-----------------------------------------------------------------------------
bool nn::gfx::tool::texenc::ConvertFormat(
    void* pDst,
    const void* pSrc,
    const std::string& dstFormatStr,
    const std::string& srcFormatStr,
    const std::string& qualityStr,
    const int encodeFlag,
    const int imageW,
    const int imageH,
    const int imageD
)
{
    //-----------------------------------------------------------------------------
    // 引数をチェックします。
    if (pDst == nullptr ||
        pSrc == nullptr)
    {
        return false;
    }

    //-----------------------------------------------------------------------------
    // 変換前または変換後がプレーンな RGBA フォーマットでなければ変換しません。
    if (!IsPlainRgbaFormat(srcFormatStr) &&
        !IsPlainRgbaFormat(dstFormatStr))
    {
        return false;
    }

    //-----------------------------------------------------------------------------
    // フォーマットの情報を取得します。
    const size_t srcBpp = GetBitsPerPixel(srcFormatStr);
    const size_t dstBpp = GetBitsPerPixel(dstFormatStr);
    int srcMinW = 1;
    int srcMinH = 1;
    int srcMinD = 1;
    GetMinimumWhd(&srcMinW, &srcMinH, &srcMinD, srcFormatStr);
    int dstMinW = 1;
    int dstMinH = 1;
    int dstMinD = 1;
    GetMinimumWhd(&dstMinW, &dstMinH, &dstMinD, dstFormatStr);

    const bool isSrcBc = IsBcFormat(srcFormatStr);
    const bool isDstBc = IsBcFormat(dstFormatStr);
    const bool isSrcAstc = IsAstcFormat(srcFormatStr);
    const bool isDstAstc = IsAstcFormat(dstFormatStr);

    const bool usesDirectXTex = (isSrcBc || isDstBc);
    const bool usesAstcCodec = (isSrcAstc || isDstAstc);

    //EncTimeMeasure tm;
    bool isSucceeded = true;

    //-----------------------------------------------------------------------------
    const uint8_t* pLevelSrc = reinterpret_cast<const uint8_t*>(pSrc);
    uint8_t* pLevelDst = reinterpret_cast<uint8_t*>(pDst);

    //-----------------------------------------------------------------------------
    const int levelW = imageW;
    const int levelH = imageH;
    const int levelD = imageD;

    //-----------------------------------------------------------------------------
    // 各ライブラリでエンコードします。
    if (usesDirectXTex)
    {
        bool isOpenMpUsed = false;
        isSucceeded = ConvertByDirectXTex(&isOpenMpUsed,
            pLevelDst, pLevelSrc, dstFormatStr, srcFormatStr, qualityStr, encodeFlag,
            levelW, levelH, levelD);
    }
    else if (usesAstcCodec)
    {
        isSucceeded = ConvertByAstcEvaluationCodec(pLevelDst, pLevelSrc, dstFormatStr, srcFormatStr, qualityStr, encodeFlag,
            levelW, levelH, levelD);
    }
    else
    {
        isSucceeded = ConvertSimpleFormat(pLevelDst, pLevelSrc, dstFormatStr, srcFormatStr, encodeFlag,
            levelW, levelH, levelD);
    }

    if (!isSucceeded)
    {
        return false;
    }

    //cerr << "convert format: " << tm.GetMilliSec() << " ms" << endl;
    //NoteTrace("convert format: %fms", tm.GetMilliSec());

    return isSucceeded;
}

// 変換をサポートしているかフォーマットかどうか簡易的にチェックします。
bool nn::gfx::tool::texenc::IsSupported(
    const std::string& format
)
{
    if (IsEtcFormat(format) || IsEacFormat(format) || IsPvrtcFormat(format))
    {
        return false;
    }

    // 簡易的なチェックなので上記以外は true を返す。
    return true;
}
#endif


