﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

//=============================================================================
// include
//=============================================================================
#include <algorithm>
#include <cstdint>
#include <ctime>
#include <iostream>
#include <string>

#ifndef NOMINMAX
#define NOMINMAX
#endif
#include <windows.h>

//=============================================================================
// constants
//=============================================================================

const int EncRgbCount  = 3; //!< RGB の成分数です。
const int EncRgbaCount = 4; //!< RGBA の成分数です。

const int EncRgbBytes  = 3; //!< RGB 各 8 ビット整数のピクセルのバイト数です。
const int EncRgbaBytes = 4; //!< RGBA 各 8 ビット整数のピクセルのバイト数です。

const int EncRgbFloat16Bytes  = 3 * sizeof(uint16_t); //!< RGB 各 16 ビット浮動小数点数のピクセルのバイト数です。
const int EncRgbaFloat16Bytes = 4 * sizeof(uint16_t); //!< RGBA 各 16 ビット浮動小数点数のピクセルのバイト数です。

const int EncRgbFloatBytes  = 3 * sizeof(float); //!< RGB 各 32 ビット浮動小数点数のピクセルのバイト数です。
const int EncRgbaFloatBytes = 4 * sizeof(float); //!< RGBA 各 32 ビット浮動小数点数のピクセルのバイト数です。

const int EncOpaAlphaBorder = 0x80; //!< 1bit アルファフォーマットでアルファがこの値以上なら不透明とみなします。

//-----------------------------------------------------------------------------
// 未使用変数の警告を抑止するためのマクロです。
//-----------------------------------------------------------------------------
#define ENC_UNUSED_VARIABLE(Variable) (void)(&Variable);

//=============================================================================
//! @brief 時間計測のクラスです。
//=============================================================================
class EncTimeMeasure
{
protected:
    std::clock_t m_StartClock; //!< 計測開始時のクロックです。

public:
    //! @brief コンストラクタです。
    EncTimeMeasure()
    {
        Reset();
    }

    //! 計測開始時間を現在のクロックに設定します。
    void Reset()
    {
        m_StartClock = std::clock();
    }

    //! 計測開始時からの秒数を取得します。
    float GetSec() const
    {
        return static_cast<float>(std::clock() - m_StartClock) / CLOCKS_PER_SEC;
    }

    //! 計測開始時からのミリ秒数を取得します。
    float GetMilliSec() const
    {
        return GetSec() * 1000.0f;
    }
};

//-----------------------------------------------------------------------------
//! @brief 四捨五入した値を返します。
//-----------------------------------------------------------------------------
template <typename T>
inline int RoundValue(const T v)
{
    return (v >= static_cast<T>(0)) ?
        static_cast<int>(v + static_cast<T>(0.5)) :
        static_cast<int>(v - static_cast<T>(0.5));
}

//-----------------------------------------------------------------------------
//! @brief 指定した範囲にクランプした値を返します。
//-----------------------------------------------------------------------------
template <typename T>
inline T ClampValue(const T v, const T minVal, const T maxVal)
{
    return (v < minVal) ? minVal : ((v > maxVal) ? maxVal : v);
}

//-----------------------------------------------------------------------------
//! @brief 整列した値を返します。
//-----------------------------------------------------------------------------
template <typename T>
inline T AlignValue(T v, const int align)
{
    return static_cast<T>((v + align - 1) & (~(align - 1)));
}

//-----------------------------------------------------------------------------
//! @brief float 型の値を uint8_t 型に変換した値を返します。
//-----------------------------------------------------------------------------
inline uint8_t GetUint8Value(const float v)
{
    return static_cast<uint8_t>(ClampValue(RoundValue(v), 0x00, 0xff));
}

//-----------------------------------------------------------------------------
//! @brief float 型の値を int8_t 型に変換した値を返します。
//-----------------------------------------------------------------------------
inline int8_t GetInt8Value(const float v)
{
    return static_cast<int8_t>(ClampValue(RoundValue(v), -0x7f, 0x7f));
}

//-----------------------------------------------------------------------------
//! @brief float 型の値を uint16_t 型に変換した値を返します。
//-----------------------------------------------------------------------------
inline uint16_t GetUint16Value(const float v)
{
    return static_cast<uint16_t>(ClampValue(RoundValue(v), 0x0000, 0xffff));
}

//-----------------------------------------------------------------------------
//! @brief float 型の値を int16_t 型に変換した値を返します。
//-----------------------------------------------------------------------------
inline int16_t GetInt16Value(const float v)
{
    return static_cast<int16_t>(ClampValue(RoundValue(v), -0x7fff, 0x7fff));
}

//-----------------------------------------------------------------------------
//! @brief float 型の値を uint32_t 型に変換した値を返します。
//-----------------------------------------------------------------------------
inline uint32_t GetUint32Value(const float v)
{
    const uint32_t MaxU32 = 0xffffffff;
    const float MaxF32 = static_cast<float>(MaxU32) - 0.5f;
    return (v < 0.0f   ) ? static_cast<uint32_t>(0) :
           (v >= MaxF32) ? MaxU32                   :
           static_cast<uint32_t>(v + 0.5f);
}

//-----------------------------------------------------------------------------
//! @brief float 型の値を int32_t 型に変換した値を返します。
//-----------------------------------------------------------------------------
inline int32_t GetInt32Value(const float v)
{
    if (v >= 0.0f)
    {
        const int32_t MaxS32 = 0x7fffffff;
        const float MaxF32 = static_cast<float>(MaxS32) - 0.5f;
        return (v >= MaxF32) ? MaxS32 : static_cast<int32_t>(v + 0.5f);
    }
    else
    {
        const int32_t MinS32 = -0x7fffffff;
        const float MinF32 = static_cast<float>(MinS32) + 0.5f;
        return (v <= MinF32) ? MinS32 : static_cast<int32_t>(v - 0.5f);
    }
}

//-----------------------------------------------------------------------------
//! @brief 開いているメモ帳のウィンドウにメッセージを出力します（デバッグ用）。
//-----------------------------------------------------------------------------
void NoteTrace(const char* format, ...);

//-----------------------------------------------------------------------------
//! @brief プロセッサ数を取得します。
//-----------------------------------------------------------------------------
int GetProcessorCount();

//-----------------------------------------------------------------------------
//! @brief 指定した文字が Shift JIS の 2 バイト文字の 1 バイト目なら true を返します。
//-----------------------------------------------------------------------------
bool IsShiftJisFirstByte(const uint8_t val);

//-----------------------------------------------------------------------------
//! @brief ANSI 文字列からワイド文字列（Unicode）を取得します。
//-----------------------------------------------------------------------------
std::wstring GetUnicodeFromAnsi(const std::string& src);

//-----------------------------------------------------------------------------
//! @brief ワイド文字列（Unicode）から ANSI 文字列を取得します。
//-----------------------------------------------------------------------------
std::string GetAnsiFromUnicode(const std::wstring& src);

//-----------------------------------------------------------------------------
//! @brief ファイルパス文字列からフォルダのパス部分を抽出して返します。
//!        返り値の最後に / は付きません。
//-----------------------------------------------------------------------------
std::string GetFolderFromFilePath(const std::string& path);

//-----------------------------------------------------------------------------
//! @brief ファイルが存在するなら true を返します。
//-----------------------------------------------------------------------------
//bool FileExists(const std::string& path);

//-----------------------------------------------------------------------------
//! @brief 環境変数の値を取得します。
//!
//! @param[in] name 環境変数の名前です。
//!
//! @return 環境変数の値を返します。環境変数が定義されていない場合は空文字を返します。
//-----------------------------------------------------------------------------
std::string GetEnvVariable(const char* name);

//-----------------------------------------------------------------------------
//! @brief Windows のシステムメッセージ文字列を取得します。
//!
//! @param[in] messageId メッセージ ID です。
//!
//! @return Windows のシステムメッセージ文字列を返します。
//-----------------------------------------------------------------------------
std::string GetWindowsSystemMessage(const int messageId);

//-----------------------------------------------------------------------------
//! @brief Windows の最後のエラーのメッセージ文字列を取得します。
//!
//! @return Windows の最後のエラーのメッセージ文字列を返します。
//-----------------------------------------------------------------------------
std::string GetWindowsLastErrorMessage();

//-----------------------------------------------------------------------------
//! @brief プレーンな RGBA フォーマットなら true を返します。
//!
//! @param[in] formatStr 中間ファイルのフォーマット文字列です。
//!
//! @return プレーンな RGBA フォーマットなら true を返します。
//-----------------------------------------------------------------------------
bool IsPlainRgbaFormat(const std::string& formatStr);

//-----------------------------------------------------------------------------
//! @brief 浮動小数点数型フォーマットなら true を返します。
//!
//! @param[in] formatStr 中間ファイルのフォーマット文字列です。
//!
//! @return 浮動小数点数型フォーマットなら true を返します。
//-----------------------------------------------------------------------------
bool IsFloatFormat(const std::string& formatStr);

//-----------------------------------------------------------------------------
//! @brief 符号ありフォーマットなら true を返します。
//!
//! @param[in] formatStr 中間ファイルのフォーマット文字列です。
//!
//! @return 符号ありフォーマットなら true を返します。
//-----------------------------------------------------------------------------
bool IsSignedFormat(const std::string& formatStr);

//-----------------------------------------------------------------------------
//! @brief snorm フォーマットなら true を返します。
//!
//! @param[in] formatStr 中間ファイルのフォーマット文字列です。
//!
//! @return snorm フォーマットなら true を返します。
//-----------------------------------------------------------------------------
bool IsSnormFormat(const std::string& formatStr);

//-----------------------------------------------------------------------------
//! @brief snorm または sint フォーマットなら true を返します。
//!
//! @param[in] formatStr 中間ファイルのフォーマット文字列です。
//!
//! @return snorm または sint フォーマットなら true を返します。
//-----------------------------------------------------------------------------
bool IsSnormSintFormat(const std::string& formatStr);

//-----------------------------------------------------------------------------
//! @brief sRGB フェッチのフォーマットなら true を返します。
//!
//! @param[in] formatStr 中間ファイルのフォーマット文字列です。
//!
//! @return sRGB フェッチのフォーマットなら true を返します。
//-----------------------------------------------------------------------------
bool IsSrgbFetchFormat(const std::string& formatStr);

//-----------------------------------------------------------------------------
//! @brief BC1/BC2/BC3 フォーマットなら true を返します。
//!
//! @param[in] formatStr 中間ファイルのフォーマット文字列です。
//!
//! @return BC1/BC2/BC3 フォーマットなら true を返します。
//-----------------------------------------------------------------------------
bool IsBc123Format(const std::string& formatStr);

//-----------------------------------------------------------------------------
//! @brief BC フォーマットなら true を返します。
//!
//! @param[in] formatStr 中間ファイルのフォーマット文字列です。
//!
//! @return BC フォーマットなら true を返します。
//-----------------------------------------------------------------------------
bool IsBcFormat(const std::string& formatStr);

//-----------------------------------------------------------------------------
//! @brief ETC1 フォーマットなら true を返します。
//!
//! @param[in] formatStr 中間ファイルのフォーマット文字列です。
//!
//! @return ETC1 フォーマットなら true を返します。
//-----------------------------------------------------------------------------
bool IsEtc1Format(const std::string& formatStr);

//-----------------------------------------------------------------------------
//! @brief ETC2 フォーマットなら true を返します。
//!
//! @param[in] formatStr 中間ファイルのフォーマット文字列です。
//!
//! @return ETC2 フォーマットなら true を返します。
//-----------------------------------------------------------------------------
bool IsEtc2Format(const std::string& formatStr);

//-----------------------------------------------------------------------------
//! @brief ETC フォーマットなら true を返します。
//!
//! @param[in] formatStr 中間ファイルのフォーマット文字列です。
//!
//! @return ETC フォーマットなら true を返します。
//-----------------------------------------------------------------------------
bool IsEtcFormat(const std::string& formatStr);

//-----------------------------------------------------------------------------
//! @brief EAC フォーマットなら true を返します。
//!
//! @param[in] formatStr 中間ファイルのフォーマット文字列です。
//!
//! @return EAC フォーマットなら true を返します。
//-----------------------------------------------------------------------------
bool IsEacFormat(const std::string& formatStr);

//-----------------------------------------------------------------------------
//! @brief PVRTC1 フォーマットなら true を返します。
//!
//! @param[in] formatStr 中間ファイルのフォーマット文字列です。
//!
//! @return PVRTC1 フォーマットなら true を返します。
//-----------------------------------------------------------------------------
bool IsPvrtc1Format(const std::string& formatStr);

//-----------------------------------------------------------------------------
//! @brief PVRTC2 フォーマットなら true を返します。
//!
//! @param[in] formatStr 中間ファイルのフォーマット文字列です。
//!
//! @return PVRTC2 フォーマットなら true を返します。
//-----------------------------------------------------------------------------
bool IsPvrtc2Format(const std::string& formatStr);

//-----------------------------------------------------------------------------
//! @brief PVRTC フォーマットなら true を返します。
//!
//! @param[in] formatStr 中間ファイルのフォーマット文字列です。
//!
//! @return PVRTC フォーマットなら true を返します。
//-----------------------------------------------------------------------------
bool IsPvrtcFormat(const std::string& formatStr);

//-----------------------------------------------------------------------------
//! @brief ASTC フォーマットなら true を返します。
//!
//! @param[in] formatStr 中間ファイルのフォーマット文字列です。
//!
//! @return ASTC フォーマットなら true を返します。
//-----------------------------------------------------------------------------
bool IsAstcFormat(const std::string& formatStr);

//-----------------------------------------------------------------------------
//! @brief フォーマット文字列から ASTC ブロックのサイズを取得します。
//!
//! @param[out] pXdim ブロックの幅を格納します。
//! @param[out] pYdim ブロックの高さを格納します。
//! @param[out] pZdim ブロックの奥行きを格納します。
//! @param[in] formatStr 中間ファイルのフォーマット文字列です。
//!
//! @return 処理成功なら true を返します。
//-----------------------------------------------------------------------------
bool GetAstcBlockSize(int* pXdim, int* pYdim, int* pZdim, const std::string& formatStr);

//-----------------------------------------------------------------------------
//! @brief ASTC フォーマットにエンコードした際のデータサイズを取得します。
//!
//! @param[in] imageW 画像の幅です。
//! @param[in] imageH 画像の高さです。
//! @param[in] imageD 画像の奥行きです。
//! @param[in] xdim ブロックの幅です。
//! @param[in] ydim ブロックの高さです。
//! @param[in] zdim ブロックの奥行きです。
//!
//! @return データサイズを返します。
//-----------------------------------------------------------------------------
size_t GetAstcDataSize(
    const int imageW,
    const int imageH,
    const int imageD,
    const int xdim,
    const int ydim,
    const int zdim
);

//-----------------------------------------------------------------------------
//! @brief テクスチャフォーマットから成分数を取得します。
//!
//! @param[in] formatStr 中間ファイルのフォーマット文字列です。
//!
//! @return 成分数を返します。
//-----------------------------------------------------------------------------
int GetComponentCount(const std::string& formatStr);

//-----------------------------------------------------------------------------
//! @brief 1 ピクセル当たりのビット数を取得します。
//!
//! @param[in] formatStr 中間ファイルのフォーマット文字列です。
//!
//! @return 1 ピクセル当たりのビット数を返します。
//!         ASTC フォーマットの場合は 0 を返します。
//-----------------------------------------------------------------------------
size_t GetBitsPerPixel(const std::string& formatStr);

//-----------------------------------------------------------------------------
//! @brief フォーマットに対する最小の幅／高さ／奥行きを取得します。
//!
//! @param[out] pMinW 最小の幅を格納します。
//! @param[out] pMinH 最小の高さを格納します。
//! @param[out] pMinD 最小の奥行きを格納します。
//! @param[in] formatStr 中間ファイルのフォーマット文字列です。
//-----------------------------------------------------------------------------
void GetMinimumWhd(int* pMinW, int* pMinH, int* pMinD, const std::string& formatStr);

//-----------------------------------------------------------------------------
//! @brief 1 レベルのデータサイズ（バイト数）を取得します。
//!
//! @param[in] levelW 対象レベルの幅です。
//! @param[in] levelH 対象レベルの高さです。
//! @param[in] levelD 対象レベルの奥行きです。
//! @param[in] isAstc ASTC フォーマットなら true です。
//! @param[in] bpp 1 ピクセル当たりのビット数です。
//! @param[in] minW フォーマットに対する最小の幅です。
//! @param[in] minH フォーマットに対する最小の高さです。
//! @param[in] minD フォーマットに対する最小の奥行きです。
//!
//! @return 1 レベルのデータサイズを返します。
//-----------------------------------------------------------------------------
size_t GetLevelDataSize(
    const int levelW,
    const int levelH,
    const int levelD,
    const bool isAstc,
    const size_t bpp,
    const int minW,
    const int minH,
    const int minD
);

//-----------------------------------------------------------------------------
//! @brief float 型の値から float_16 の値を取得します。
//!
//! @param[in] valueF32 float 型の値です。
//!
//! @return float_16 の値を返します。
//-----------------------------------------------------------------------------
uint16_t GetFloat16Value(const float valueF32);

//-----------------------------------------------------------------------------
//! @brief float_16 の値から float 型の値を取得します。
//!
//! @param[in] valueF16 float_16 の値です。
//!
//! @return float 型の値を返します。
//-----------------------------------------------------------------------------
float GetFloat32Value(const uint16_t valueF16);

//-----------------------------------------------------------------------------
//! @brief エンコード品質文字列からエンコードレベルを取得します。
//!
//! @param[in] qualityStr エンコード品質文字列です。
//!
//! @return エンコードレベルを返します。
//-----------------------------------------------------------------------------
int GetQualityLevel(const std::string& qualityStr);

//-----------------------------------------------------------------------------
//! @brief RGB 成分にディザを使用するべき画像かチェックします。
//!
//! @param[in] pPixels ピクセルデータです。
//!                    1 ピクセルあたり 4 バイトで、RGBA の順に格納します。画像の左上が先頭です。
//! @param[in] imageW 画像の幅です。
//! @param[in] imageH 画像の高さです。
//! @param[in] imageD 画像の奥行きです。
//!
//! @return RGB 成分にディザを使用するべき画像なら true を返します。
//-----------------------------------------------------------------------------
bool CheckRgbDitherIsNeeded(
    const void* pPixels,
    const int imageW,
    const int imageH,
    const int imageD
);

