﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using Microsoft.Win32;

namespace NintendoWare.ExtensionManager
{
    public enum ShellExtensionService
    {
        None,
        IconService,
        ColumnService,
        PreviewService,
        InfoTipService,
        ContextMenuService,
        ShellCommandService,
        AllServices,
    }

    //==========================================================================
    /// <summary>
    /// Utility class
    /// </summary>
    //==========================================================================
    public sealed class Utility
    {
        public static readonly string[] IntermediateFileExtensions =
        {
            //"fmd",
            "fsk",
            "fsp",
            "fcl",
            "fts",
            "ftp",
            "fvb",
            "fvm",
            "fsh",
            "fsn",
            "fsc",
            "fsd",
            "fsv"
        };

        //----------------------------------------------------------------------
        /// <summary>
        /// Find XML element's child by name
        /// </summary>
        ///
        /// <param name="doc">Parent document</param>
        /// <param name="name">Name to search</param>
        ///
        /// <returns>First child node with matching name. null if not found</returns>
        //----------------------------------------------------------------------
        static public System.Xml.XmlElement XMLFindChildElement(System.Xml.XmlDocument doc,
                                                                String name)
        {
            if (doc == null)
                return null;

            foreach (System.Xml.XmlElement child in doc.ChildNodes.OfType<System.Xml.XmlElement>())
            {
                if (child == null)
                    continue;

                if (child.Name == name)
                    return child;
            }

            return null;
        }

        //----------------------------------------------------------------------
        /// <summary>
        /// Find XML element's child by name
        /// </summary>
        ///
        /// <param name="parent">Parent node</param>
        /// <param name="name">Name to search</param>
        ///
        /// <returns>First child node with matching name. null if not found</returns>
        //----------------------------------------------------------------------
        static public System.Xml.XmlElement XMLFindChildElement(System.Xml.XmlElement parent,
                                                                String name)
        {
            if (parent == null)
                return null;

            foreach (System.Xml.XmlElement child in parent.ChildNodes.OfType<System.Xml.XmlElement>())
            {
                if (child == null)
                    continue;

                if (child.Name == name)
                    return child;
            }

            return null;
        }

        //----------------------------------------------------------------------
        /// <summary>
        /// Convert string to integer
        /// </summary>
        ///
        /// <param name="valStr">String containing value</param>
        /// <param name="defaultVal">Default value in case string is in invalid format</param>
        ///
        /// <returns>Integer value</returns>
        //----------------------------------------------------------------------
        static public int ConvertStringToInt(String valStr, int defaultVal)
        {
            int val = defaultVal;

            try
            {
                System.Globalization.NumberStyles numStyle = System.Globalization.NumberStyles.Any;
                if (valStr.StartsWith("0x"))
                {
                    numStyle = System.Globalization.NumberStyles.HexNumber;
                    valStr = valStr.Remove(0, 2);
                    val = Int32.Parse(valStr, numStyle);
                    return val;
                }

                double doubleVal = Double.Parse(valStr, numStyle);
                val = (int)doubleVal;
            }

            catch (System.Exception e)
            {
                if (e.Message == null)
                    return val;

                return val;
            }

            return val;
        }

        //----------------------------------------------------------------------
        /// <summary>
        /// Convert string to float
        /// </summary>
        ///
        /// <param name="valStr">String containing value</param>
        /// <param name="defaultVal">Default value in case string is in invalid format</param>
        ///
        /// <returns>Float value</returns>
        //----------------------------------------------------------------------
        static public float ConvertStringToFloat(String valStr, float defaultVal)
        {
            float val = defaultVal;

            try
            {
                System.Globalization.NumberStyles numStyle = System.Globalization.NumberStyles.Any;
                if (valStr.StartsWith("0x"))
                {
                    numStyle = System.Globalization.NumberStyles.HexNumber;
                    valStr = valStr.Remove(0, 2);
                    val = Int32.Parse(valStr, numStyle);
                    return val;
                }

                double doubleVal = Double.Parse(valStr, numStyle);
                val = (float)doubleVal;
            }

            catch (System.Exception e)
            {
                if (e.Message == null)
                    return val;

                return val;
            }

            return val;
        }

        //----------------------------------------------------------------------
        /// <summary>
        /// Convert string to boolean
        /// </summary>
        ///
        /// <param name="valStr">String containing value</param>
        /// <param name="defaultVal">Default value in case string is in invalid format</param>
        ///
        /// <returns>Boolean value</returns>
        //----------------------------------------------------------------------
        static public bool ConvertStringToBool(String valStr, bool defaultVal)
        {
            bool val = defaultVal;

            if (String.Equals(valStr, "True", StringComparison.OrdinalIgnoreCase))
                val = true;
            else if (String.Equals(valStr, "False", StringComparison.OrdinalIgnoreCase))
                val = false;

            return val;
        }

        //----------------------------------------------------------------------
        /// <summary>
        /// Window Configuration
        /// </summary>
        //----------------------------------------------------------------------
        public class WindowConfig
        {
            public WindowConfig()
            {
                windowState = System.Windows.Forms.FormWindowState.Normal;
                x = -1;
                y = -1;
                width = -1;
                height = -1;
            }

            public System.Windows.Forms.FormWindowState windowState;
            public int x;
            public int y;
            public int width;
            public int height;
        };

        //----------------------------------------------------------------------
        /// <summary>
        /// Get window config from window
        /// </summary>
        //----------------------------------------------------------------------
        static public void GetWindowConfig(System.Windows.Forms.Control control, WindowConfig config)
        {
            config.windowState = System.Windows.Forms.FormWindowState.Normal;
            config.x = control.Location.X;
            config.y = control.Location.Y;
            config.width = control.Size.Width;
            config.height = control.Size.Height;

            System.Windows.Forms.Form form = control as System.Windows.Forms.Form;
            if (form != null)
            {
                config.windowState = form.WindowState;
            }
        }

        //----------------------------------------------------------------------
        /// <summary>
        /// Apply window config to window
        /// </summary>
        //----------------------------------------------------------------------
        static public void ApplyWindowConfig(System.Windows.Forms.Control control,
                                             WindowConfig config, bool bClipToScreen)
        {
            System.Drawing.Point loc = control.Location;
            System.Drawing.Size size = control.Size;

            if (config.x >= 0)
                loc.X = config.x;
            if (config.y >= 0)
                loc.Y = config.y;
            if (config.width >= 0)
                size.Width = config.width;
            if (config.height >= 0)
                size.Height = config.height;

            System.Drawing.Rectangle rect = new System.Drawing.Rectangle();
            rect.Location = loc;
            rect.Size = size;

            System.Windows.Forms.Form form = control as System.Windows.Forms.Form;
            if (form != null)
            {
                form.WindowState = config.windowState;

                if (bClipToScreen)
                {
                    if (!IsIntersectingWithAnyScreen(rect))
                    {
                        form.StartPosition = System.Windows.Forms.FormStartPosition.WindowsDefaultLocation;
                        form.Size = size;
                    }
                    else
                    {
                        form.StartPosition = System.Windows.Forms.FormStartPosition.Manual;
                        form.Location = loc;
                        form.Size = size;
                    }
                }
                else
                {
                    form.StartPosition = System.Windows.Forms.FormStartPosition.Manual;
                    form.Location = loc;
                    form.Size = size;
                }
            }
            else
            {
                control.Location = loc;
                control.Size = size;
            }
        }

        //----------------------------------------------------------------------
        /// <summary>
        /// Check if window is intersecting screen
        /// </summary>
        ///
        /// <returns>True if success</returns>
        //----------------------------------------------------------------------
        static public bool IsIntersectingWithAnyScreen(System.Drawing.Rectangle rect)
        {
            foreach (System.Windows.Forms.Screen screen in System.Windows.Forms.Screen.AllScreens)
            {
                if (rect.IntersectsWith(screen.Bounds))
                {
                    return true;
                }
            }

            return false;
        }

        //----------------------------------------------------------------------
        /// <summary>
        /// Loading window config from xml
        /// Override this to customize
        /// </summary>
        ///
        /// <param name="element">XML Element</param>
        ///
        /// <returns>True if success</returns>
        //----------------------------------------------------------------------
        static public bool LoadWindowConfig(System.Xml.XmlElement element, WindowConfig winConfig)
        {
            { // windowState
                String str = element.GetAttribute("WindowState");
                if (str.Length > 0)
                {
                    if (String.Equals(str, "Maximized", StringComparison.CurrentCultureIgnoreCase))
                    {
                        winConfig.windowState = System.Windows.Forms.FormWindowState.Maximized;
                    }// End if
                    else if (String.Equals(str, "Minimized", StringComparison.CurrentCultureIgnoreCase))
                    {
                        winConfig.windowState = System.Windows.Forms.FormWindowState.Minimized;
                    }// End else if
                    else
                    {
                        winConfig.windowState = System.Windows.Forms.FormWindowState.Normal;
                    }
                }
            }

            { // x
                String str = element.GetAttribute("x");
                if (str.Length > 0)
                    winConfig.x = Convert.ToInt32(str);
            }

            { // y
                String str = element.GetAttribute("y");
                if (str.Length > 0)
                    winConfig.y = Convert.ToInt32(str);
            }

            { // width
                String str = element.GetAttribute("Width");
                if (str.Length > 0)
                    winConfig.width = Convert.ToInt32(str);
            }

            { // height
                String str = element.GetAttribute("Height");
                if (str.Length > 0)
                    winConfig.height = Convert.ToInt32(str);
            }

            return true;
        }

        //----------------------------------------------------------------------
        /// <summary>
        /// Saving mainframe section
        /// Override this to customize
        /// </summary>
        ///
        /// <param name="element">XML Element</param>
        ///
        /// <returns>True if success</returns>
        //----------------------------------------------------------------------
        static public bool SaveWindowConfig(System.Xml.XmlElement element, WindowConfig winConfig)
        {
            if (winConfig.windowState == System.Windows.Forms.FormWindowState.Minimized)
            {
                element.SetAttribute("WindowState", "minimized");
            }
            else if (winConfig.windowState == System.Windows.Forms.FormWindowState.Maximized)
            {
                element.SetAttribute("WindowState", "maximized");
            }
            else
            {
                element.SetAttribute("WindowState", "normal");
            } // else

            element.SetAttribute("x", winConfig.x.ToString());
            element.SetAttribute("y", winConfig.y.ToString());
            element.SetAttribute("Width", winConfig.width.ToString());
            element.SetAttribute("Height", winConfig.height.ToString());

            return true;
        }
    }
}




