﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Threading;
using System.Threading.Tasks;
using System.Windows.Forms;

namespace NintendoWare.ExtensionManager
{
    public partial class ProgressDialog : Form
    {
        /// <summary>
        /// メソッドです。
        /// </summary>
        private Action method;

        /// <summary>
        /// ダイアログの表示が完了するまでスレッドを停止するためのフラグです。
        /// </summary>
        private AutoResetEvent activateFlag;

        /// <summary>
        /// 処理開始時間です。
        /// </summary>
        private DateTime startTime;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="message">メッセージ</param>
        /// <param name="caption">キャプション</param>
        /// <param name="method">メソッド</param>
        /// <param name="time">時間</param>
        public ProgressDialog(string message, string caption, Action method, float time)
        {
            this.InitializeComponent();

            this.Text = caption;
            this.lblMessage.Text = message;
            this.method = method;
            this.progressBar.Minimum = 0;
            this.progressBar.Maximum = (int)(time * 10.0f);

            this.activateFlag = new AutoResetEvent(false);
        }

        /// <summary>
        /// ダイアログを表示し、メソッドを行います。
        /// </summary>
        public void ShowAndRun(IWin32Window dialogOwner = null)
        {
            startTime = DateTime.Now;

            // ダイアログの表示を非同期で行った場合、ダイアログのモーダル化が複雑になるので
            // アップデート処理の方を非同期で実行する
            Thread thread = new Thread(new ThreadStart(this.Run));
            thread.IsBackground = true;
            thread.Start();

            // ダイアログを表示
            this.ShowDialog(dialogOwner);

            // 処理が完了したらダイアログは自動で閉じられる
        }

        /// <summary>
        /// ダイアログの表示が完了したときの処理を行います
        /// </summary>
        /// <param name="e">イベントの情報</param>
        protected override void OnActivated(EventArgs e)
        {
            base.OnActivated(e);

            // 表示完了フラグを立てる
            this.activateFlag.Set();
        }

        /// <summary>
        /// タイマー処理を行います。
        /// </summary>
        /// <param name="sender">イベントの発行元</param>
        /// <param name="e">イベント情報</param>
        private void OnTimerTick(object sender, EventArgs e)
        {
            // 経過時間を取得
            TimeSpan span = DateTime.Now - startTime;

            // プログレスバーを進める
            this.progressBar.Value = Math.Min((int)(span.TotalSeconds * 10.0f), this.progressBar.Maximum);
        }

        /// <summary>
        /// ダイアログ表示中に実行する処理です。
        /// </summary>
        private void Run()
        {
            // ダイアログの表示が完了するまで待機
            this.activateFlag.WaitOne();

            // メソッドを実行
            this.method();

            // ダイアログを閉じる
            this.Invoke(new MethodInvoker(this.Close));
        }
    }
}
