﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Diagnostics;
using System.Xml;
using System.Runtime.InteropServices;
using Microsoft.Win32;
using System.Globalization;
using System.Threading;
using static NintendoWare.ExtensionManager.ProcessUtility;

namespace NintendoWare.ExtensionManager
{
    //==========================================================================
    /// <summary>
    /// RegistrationManager
    /// </summary>
    //==========================================================================
    public class RegistrationManager : SubManager
    {
        public static readonly CultureInfo DefaultCulture = CultureInfo.GetCultureInfo("en-US");

        private System.String m_appdataDirectory = "";
        private bool m_bUseLog = false;
        private List<string> AdditionalExtensions { get; set; }
        private bool RemoveUnspecifiedAdditionalExtensionsRegistry { get; set; }

        //----------------------------------------------------------------------
        /// <summary>
        /// Constructor
        /// </summary>
        //----------------------------------------------------------------------
        public RegistrationManager(bool bUseLog, string appdataPath, List<string> additionalExtensions, bool removeUnspecifiedAdditionalExtensionsRegistry)
        {
            BuildDropMenuApplicationsList();

            m_appdataDirectory = appdataPath;
            m_bUseLog = bUseLog;
            AdditionalExtensions = additionalExtensions;
            RemoveUnspecifiedAdditionalExtensionsRegistry = removeUnspecifiedAdditionalExtensionsRegistry;
        }

        //----------------------------------------------------------------------
        /// <summary>
        /// Dispose
        /// </summary>
        //----------------------------------------------------------------------
        public override void Dispose()
        {
            base.Dispose();
        }

        #region Init / Deinit
        //----------------------------------------------------------------------
        /// <summary>
        /// Initialization. Override this to customize initialization.
        /// </summary>
        ///
        /// <returns>True if success</returns>
        //----------------------------------------------------------------------
        public override bool Init()
        {
            if (base.Init() == false)
                return false;

            // FTXA
            RegistrationInfo ftxaInfo = new RegistrationInfo("ftxa", "FTX");
            ftxaInfo.AvailableServices = new[]
            {
                ShellExtensionService.IconService,
                ShellExtensionService.ColumnService,
                ShellExtensionService.PreviewService,
                ShellExtensionService.InfoTipService,
                ShellExtensionService.ContextMenuService,   // すでにレジストリに登録されているものを削除するために使用
                ShellExtensionService.ShellCommandService,
            };
            m_registrationList.Add(ftxaInfo);

            // FTXB
            RegistrationInfo ftxbInfo = new RegistrationInfo("ftxb", "FTX");
            ftxbInfo.AvailableServices = new[]
            {
                ShellExtensionService.IconService,
                ShellExtensionService.ColumnService,
                ShellExtensionService.PreviewService,
                ShellExtensionService.InfoTipService,
                ShellExtensionService.ContextMenuService,   // すでにレジストリに登録されているものを削除するために使用
                ShellExtensionService.ShellCommandService,
            };
            m_registrationList.Add(ftxbInfo);

            // TGA
            RegistrationInfo tgaInfo = new RegistrationInfo("tga", "TGA");
            tgaInfo.AvailableServices = new[]
            {
                ShellExtensionService.IconService,
                ShellExtensionService.ColumnService,      // すでにレジストリに登録されているものを削除するために使用
                ShellExtensionService.PreviewService,
                ShellExtensionService.InfoTipService,
                ShellExtensionService.ShellCommandService,
            };
            m_registrationList.Add(tgaInfo);

            // PSD
            RegistrationInfo psdInfo = new RegistrationInfo("psd", "PSD");
            psdInfo.AvailableServices = new[]
            {
                ShellExtensionService.IconService,
                ShellExtensionService.ColumnService,       // すでにレジストリに登録されているものを削除するために使用
                ShellExtensionService.PreviewService,
                ShellExtensionService.InfoTipService,
                ShellExtensionService.ShellCommandService,
            };
            m_registrationList.Add(psdInfo);

            // AI
            RegistrationInfo aiInfo = new RegistrationInfo("ai", "AI");
            aiInfo.AvailableServices = new[]
            {
                ShellExtensionService.IconService,
                ShellExtensionService.ShellCommandService,
            };
            m_registrationList.Add(aiInfo);

            // EPS
            RegistrationInfo epsInfo = new RegistrationInfo("eps", "EPS");
            epsInfo.AvailableServices = new[]
            {
                ShellExtensionService.IconService,
                ShellExtensionService.ShellCommandService,
            };
            m_registrationList.Add(epsInfo);

            // 中間ファイル
            foreach (var item in new[] { "eset", "flyt", "fmda", "fmdb" })
            {
                var info = new RegistrationInfo(item, "THUMBS")
                {
                    AvailableServices = new[]
                    {
                        ShellExtensionService.IconService,
                        ShellExtensionService.PreviewService,
                        ShellExtensionService.ShellCommandService,
                    }
                };
                m_registrationList.Add(info);
            }

            var extensions = m_registrationList.Select(x => x.ExtensionName.ToLower());

            // ユーザ指定拡張子
            foreach (var item in AdditionalExtensions.Select(x => x.ToLower()).Distinct().Except(extensions.ToArray()))
            {
                var info = new RegistrationInfo(item, "THUMBS")
                {
                    AvailableServices = new[]
                    {
                        ShellExtensionService.IconService,
                        ShellExtensionService.PreviewService,
                        ShellExtensionService.ShellCommandService,
                    },
                    IsCustomExtension = true,
                };
                m_registrationList.Add(info);
            }

            var extentionsInThumbsRegistry = new List<string>();
            using (RegistryKey thumbsKey = Registry.ClassesRoot.OpenSubKey("NintendoWare.THUMBS.2"))
            {
                if (thumbsKey != null)
                {
                    foreach (var name in thumbsKey.GetValueNames())
                    {
                        try
                        {
                            var value = thumbsKey.GetValue(name) as string;
                            if (value == "ext")
                            {
                                extentionsInThumbsRegistry.Add(name.ToLower());
                            }
                        }
                        catch (Exception e)
                        {
                            MainApp.Instance.WriteToLog(e.Message);
                        }
                    }
                }
            }

            // レジストリに保存されているユーザ指定拡張子
            foreach (var item in extentionsInThumbsRegistry.Select(x => x.ToLower()).Distinct().Except(extensions.ToArray()))
            {
                var info = new RegistrationInfo(item, "THUMBS")
                {
                    AvailableServices = new[]
                    {
                        ShellExtensionService.IconService,
                        ShellExtensionService.PreviewService,
                        ShellExtensionService.ShellCommandService,
                    },
                    IsCustomExtension = true,
                    RemoveCustomExtension = RemoveUnspecifiedAdditionalExtensionsRegistry,
                };
                m_registrationList.Add(info);
            }

            return true;
        }

        //----------------------------------------------------------------------
        /// <summary>
        /// Cleanup code. Overridable
        /// </summary>
        //----------------------------------------------------------------------
        public override void Deinit()
        {
            base.Deinit();
        }
        #endregion

        #region Registration Info
        //----------------------------------------------------------------------
        /// <summary>
        /// List of registration
        /// </summary>
        //----------------------------------------------------------------------
        private List<RegistrationInfo> m_registrationList = new List<RegistrationInfo>();

        //----------------------------------------------------------------------
        /// <summary>
        /// Number of registration info
        /// </summary>
        //----------------------------------------------------------------------
        public int NumRegistrationInfos
        {
            get { return m_registrationList.Count; }
        }

        //----------------------------------------------------------------------
        /// <summary>
        /// Get registration by index
        /// </summary>
        //----------------------------------------------------------------------
        public RegistrationInfo GetRegistrationInfo(int index)
        {
            return m_registrationList[index];
        }

        public IEnumerable<RegistrationInfo> RegistrationInfo
        {
            get { return m_registrationList; }
        }

        //----------------------------------------------------------------------
        /// <summary>
        /// Find registration from extension name
        /// </summary>
        //----------------------------------------------------------------------
        public RegistrationInfo FindRegistrationInfo(String extensionName)
        {
            foreach (RegistrationInfo info in m_registrationList)
            {
                if (String.Equals(info.ExtensionName, extensionName, StringComparison.CurrentCultureIgnoreCase))
                {
                    return info;
                }
            }

            return null;
        }
        #endregion

        #region Active registarion info
        //----------------------------------------------------------------------
        /// <summary>
        /// Set Editing registration
        /// </summary>
        //----------------------------------------------------------------------
        public void SetEditingRegistration(String extensionName)
        {
            RegistrationInfo regInfo = FindRegistrationInfo(extensionName);
            if (m_editingRegistration != regInfo)
                m_editingRegistration = regInfo;
        }

        private RegistrationInfo m_editingRegistration = null;

        //----------------------------------------------------------------------
        /// <summary>
        /// Editing registration
        /// </summary>
        //----------------------------------------------------------------------
        public RegistrationInfo EditingRegistration
        {
            get { return m_editingRegistration; }
        }
        #endregion

        #region Load and Apply registration

        public bool LoadFromEasyUnInstall()
        {
            EasySetup[] easy = new EasySetup[6]{
                                //             icon,   tip,preiew, name,  path
                                new EasySetup(false, false, false, Properties.Resources.TEXT_RESET), // ftxa
                                new EasySetup(false, false, false, Properties.Resources.TEXT_RESET), // ftxb
                                new EasySetup(false, false, false, Properties.Resources.TEXT_RESET), // tga
                                new EasySetup(false, false, false, Properties.Resources.TEXT_RESET), // psd
                                new EasySetup(false, false, false, Properties.Resources.TEXT_RESET), // ai
                                new EasySetup(false, false, false, Properties.Resources.TEXT_RESET), // eps
            };

            int i = 0;
            foreach (RegistrationInfo info in m_registrationList)
            {
                if (i < easy.Length)
                {
                    info.LoadFromEasySetup(easy[i]);
                }
                else
                {
                    info.LoadFromEasySetup(new EasySetup(false, false, false, Properties.Resources.TEXT_RESET));

                }
                i++;
            }

            return true;
        }

        public bool LoadFromEasyInstall()
        {
            EasySetup[] easy = new EasySetup[6]{
                                //             icon,   tip,preiew,column, name,  path
                                new EasySetup( true,  true,  true, Properties.Resources.TEXT_CUSTOM), // ftxa
                                new EasySetup( true,  true,  true, Properties.Resources.TEXT_CUSTOM), // ftxb
                                new EasySetup( true,  true,  true, Properties.Resources.TEXT_CUSTOM), // tga
                                new EasySetup( true,  true,  true, Properties.Resources.TEXT_CUSTOM), // psd
                                new EasySetup( true, false, false, Properties.Resources.TEXT_CUSTOM), // ai
                                new EasySetup( true, false, false, Properties.Resources.TEXT_CUSTOM), // eps
            };

            int i = 0;
            foreach (RegistrationInfo info in m_registrationList)
            {
                if (i < easy.Length)
                {
                    info.LoadFromEasySetup(easy[i]);
                }
                else
                {
                    info.LoadFromEasySetup(new EasySetup(true, false, true, Properties.Resources.TEXT_CUSTOM));
                }
                i++;
            }

            return true;
        }

        //----------------------------------------------------------------------
        /// <summary>
        /// Load registration
        /// </summary>
        //----------------------------------------------------------------------
        public bool LoadFromRegistry()
        {
            foreach (RegistrationInfo info in m_registrationList)
            {
                info.LoadFromRegistry();
            }

            return true;
        }

        //----------------------------------------------------------------------
        /// <summary>
        /// Apply registration
        /// </summary>
        //----------------------------------------------------------------------
        public bool Apply(bool uninstall)
        {
            bool bSuccess = true;

            DateTime t1 = System.DateTime.Now;
            MainApp.Instance.WriteToLog(String.Format("Apply Registration -- Started {0}", t1.TimeOfDay));

            CleanupPropertyDesc();

            foreach (RegistrationInfo info in m_registrationList)
            {
                DateTime t1_info = System.DateTime.Now;
                MainApp.Instance.WriteToLog(String.Format("    Save to registry {0} -- Started {1}", info.ExtensionName, t1_info.TimeOfDay));

                // 拡張子ごとの設定
                if (info.SaveToRegistry(uninstall) == false)
                    bSuccess = false;

                DateTime t2_info = System.DateTime.Now;
                MainApp.Instance.WriteToLog(String.Format("    Save to registry {0} -- Ended   {1} ( {2} Sec )", info.ExtensionName, t2_info.TimeOfDay, (t2_info - t1_info).TotalSeconds));
            }

            var plugins = from info in m_registrationList
                          group info by info.PluginName;

            // dll 毎の設定
            foreach (var plugin in plugins)
            {
                SavePluginsToRegistry(plugin, uninstall);
            }

            DateTime t2 = System.DateTime.Now;
            MainApp.Instance.WriteToLog(String.Format("Apply Registration -- Ended   {0} ( {1} Sec )", t2.TimeOfDay, (t2 - t1).TotalSeconds));

            return bSuccess;
        }

        /// <summary>
        /// dll 毎の設定の保存
        /// </summary>
        public bool SavePluginsToRegistry(IEnumerable<RegistrationInfo> info, bool uninstall)
        {
            var result = true;
            var item = info.First();
            var plugin = MainApp.Instance.ShellPluginManager.FindPlugin(item.PluginName);

            // Icon
            {
                var service = ShellExtensionService.IconService;
                bool use = !uninstall && info.Any(x => x.IsServiceAvailable(service) && x.CurrentStatus.UseIconService);
                var description = string.Format("NintendoWare {0} Icon Shell Extension", item.PluginName);
                MainApp.Instance.WriteToLog(description + use);
                Guid guid;
                if (Define.IDs[service].TryGetValue(item.PluginName?.ToUpper(), out guid))
                {
                    result &= item.SavePluginToRegistry(plugin, use, guid, description, service);
                }
            }

            // InfoTip
            {
                var service = ShellExtensionService.InfoTipService;
                bool use = !uninstall && info.Any(x => x.IsServiceAvailable(service) && x.CurrentStatus.UseInfoTipService);
                var description = string.Format("NintendoWare {0} InfoTip Shell Extension", item.PluginName);
                MainApp.Instance.WriteToLog(description + use);
                Guid guid;
                if (Define.IDs[service].TryGetValue(item.PluginName?.ToUpper(), out guid))
                {
                    result &= item.SavePluginToRegistry(plugin, use, guid, description, service);
                }
            }

            // Preview
            {
                var service = ShellExtensionService.PreviewService;
                bool use = !uninstall && info.Any(x => x.IsServiceAvailable(service) && x.CurrentStatus.UsePreviewService);
                var description = string.Format("NintendoWare {0} Preview Shell Extension", item.PluginName);
                MainApp.Instance.WriteToLog(description + use);
                Guid guid;
                if (Define.IDs[service].TryGetValue(item.PluginName?.ToUpper(), out guid))
                {
                    result &=item.SavePluginToRegistry(plugin, use, guid, description, service);
                }
            }

            return result;
        }

        //----------------------------------------------------------------------
        /// <summary>
        /// Raw C functions
        /// </summary>
        //----------------------------------------------------------------------
        [DllImport("kernel32", CharSet = CharSet.Unicode, SetLastError = true)]
        private static extern IntPtr LoadLibrary(string lpFileName);

        [DllImport("kernel32", SetLastError = true)]
        private static extern bool FreeLibrary(IntPtr hModule);

        [DllImport("kernel32", CharSet = CharSet.Ansi, SetLastError = true, ExactSpelling = false)]
        private static extern IntPtr GetProcAddress(IntPtr hModule, string lpProcName);

        [UnmanagedFunctionPointer(CallingConvention.Cdecl, CharSet= CharSet.Unicode)]
        public delegate int PSRegisterPropertySchema( String path );

        [UnmanagedFunctionPointer(CallingConvention.Cdecl, CharSet= CharSet.Unicode)]
        public delegate int PSUnregisterPropertySchema( String path );

        //----------------------------------------------------------------------
        /// <summary>
        /// Cleanup PropertyDesc
        /// </summary>
        //----------------------------------------------------------------------
        private void CleanupPropertyDesc()
        {
            int i;

            IntPtr propSysSLL;

            DateTime t1 = System.DateTime.Now;
            MainApp.Instance.WriteToLog(String.Format("    CleanupPropertyDesc -- Started {0}", t1.TimeOfDay));

            {
                DateTime t1_1 = System.DateTime.Now;
                MainApp.Instance.WriteToLog(String.Format("        CleanupPropertyDesc -- Load propsys.dll Started {0}", t1_1.TimeOfDay));

                propSysSLL = LoadLibrary("propsys.dll");
                if (propSysSLL == IntPtr.Zero)
                    return;

                DateTime t1_2 = System.DateTime.Now;
                MainApp.Instance.WriteToLog(String.Format("        CleanupPropertyDesc -- Load propsys.dll Ended   {0} ( {1} Sec )", t1_2.TimeOfDay, (t1_2 - t1_1).TotalSeconds));
            }

            //------------------------------------------------------------------
            // Cleanup existing schema
            //------------------------------------------------------------------
            List<String> propDescFilesToUnregister = new List<String>();

            {
                DateTime t1_1 = System.DateTime.Now;
                MainApp.Instance.WriteToLog(String.Format("        CleanupPropertyDesc -- Checking existing schema Started {0}", t1_1.TimeOfDay));


                // Add to property store list
                RegistryKey localMachineKey = Registry.LocalMachine;
                RegistryKey propStoreListKey = localMachineKey.OpenSubKey("SOFTWARE\\Microsoft\\Windows\\CurrentVersion\\PropertySystem\\PropertySchema", false);

                if (propStoreListKey == null)
                {
                    DateTime t11 = System.DateTime.Now;
                    MainApp.Instance.WriteToLog(String.Format("        CleanupPropertyDesc -- propStoreListKey = null", t11.TimeOfDay));
                    return;
                }

                String[] subKeyNames = propStoreListKey.GetSubKeyNames();
                for (i = 0; i < propStoreListKey.SubKeyCount; i++)
                {
                    RegistryKey subKey = propStoreListKey.OpenSubKey(subKeyNames[i]);
                    String descFilePath = subKey.GetValue(null) as String;
                    if ((descFilePath == null) || (descFilePath.Length <= 0))
                        continue;

                    String fileName = System.IO.Path.GetFileNameWithoutExtension(descFilePath);
                    if (String.Equals(fileName, "ShellExtension", StringComparison.CurrentCultureIgnoreCase) ||
                        String.Equals(fileName, "NWShellExt", StringComparison.CurrentCultureIgnoreCase))
                    {
                        propDescFilesToUnregister.Add(descFilePath);
                    }
                }
                propStoreListKey.Close();

                DateTime t1_2 = System.DateTime.Now;
                MainApp.Instance.WriteToLog(String.Format("        CleanupPropertyDesc -- Checking existing schema Ended {0}", t1_2.TimeOfDay));
            }

            if (propDescFilesToUnregister.Count > 0)
            {
                DateTime t1_1 = System.DateTime.Now;
                MainApp.Instance.WriteToLog(String.Format("        CleanupPropertyDesc -- Unregistering old schema Started {0}", t1_1.TimeOfDay));

                PSUnregisterPropertySchema unregPropDescSchema = Marshal.GetDelegateForFunctionPointer(GetProcAddress(propSysSLL, "PSUnregisterPropertySchema"), typeof(PSUnregisterPropertySchema)) as PSUnregisterPropertySchema;
                if (unregPropDescSchema != null)
                {
                    for (i = 0; i < propDescFilesToUnregister.Count; i++)
                    {
                        unregPropDescSchema(propDescFilesToUnregister[i]);
                    }
                }

                DateTime t1_2 = System.DateTime.Now;
                MainApp.Instance.WriteToLog(String.Format("        CleanupPropertyDesc -- Unregistering old schema Ended   {0} ( {1} Sec )", t1_2.TimeOfDay, (t1_2 - t1_1).TotalSeconds));
            }

            propDescFilesToUnregister.Clear();

            DateTime t2 = System.DateTime.Now;
            MainApp.Instance.WriteToLog(String.Format("    CleanupPropertyDesc -- Ended {0}", t2.TimeOfDay));

        }

        #endregion

        #region Windows applications
        public class AppInfo
        {
            public AppInfo()
            {
                Name      = "";
                Extension = "";
                Path      = "";
            }

            public String Name { set; get; }
            public String Extension { set; get; }
            public String Path { set; get; }

            public bool IsExtensionSupported(String extensionName)
            {
                if (String.Compare(Extension, "All", StringComparison.CurrentCultureIgnoreCase) == 0)
                    return true;

                if (Extension.Contains(extensionName))
                    return true;

                return false;
            }
        };

        private List<AppInfo> m_applicationList = new List<AppInfo>();

        //----------------------------------------------------------------------
        /// <summary>
        /// Load default application list config
        /// </summary>
        //----------------------------------------------------------------------
        public void LoadSuggestedApplicationsListFromXML()
        {
            MainApp app = MainApp.Instance;
            Debug.Assert(app != null);

            String configPath = app.ExecutableDirectory + "\\session-application-list.xml";
            XmlTextReader reader = new XmlTextReader(configPath);
            XmlDocument doc = new XmlDocument();

            try
            {
                doc.Load(reader);
            }
            catch (Exception e)
            {
                if (e.Message.Length > 0) {} // To shutup compiler
                return;
            }

            XmlNode root = doc.DocumentElement;
            if (root == null)
                return;

            int i;
            for (i = 0; i < root.ChildNodes.Count; i++)
            {
                XmlElement child = root.ChildNodes[i] as XmlElement;
                if (child==null)
                    continue;

                if (String.Compare(child.Name, "application", StringComparison.CurrentCultureIgnoreCase) == 0)
                {
                    String appName = FindApplicationName(child);
                    String extName = child.GetAttribute("Extension");
                    String appPath = child.GetAttribute("Path");
                    String absPath = child.GetAttribute("AboslutePath");
                    bool bAbsPath = false;

                    if ( (absPath!=null) && (String.Compare(absPath, "True",StringComparison.CurrentCultureIgnoreCase)==0) )
                    {
                        bAbsPath = true;
                    }

                    if ( (appName!=null) && (appName.Length>0) )
                    {
                        AppInfo appInfo   = new AppInfo();
                        appInfo.Name      = appName;
                        appInfo.Extension = extName;

                        if ( (appPath != null) && (appPath.Length>0) )
                        {
                            if (bAbsPath == false)
                            {
                                appInfo.Path = app.ExecutableDirectory + "\\" + appPath;
                                appInfo.Path = System.IO.Path.GetFullPath(appInfo.Path);
                            }
                            else
                            {
                                appInfo.Path = appPath;
                            }
                        }

                        if (appInfo.Path.Length>0)
                            m_applicationList.Add(appInfo);
                    }
                }
            }

            // Special case for Photoshop
            CollectPhotoshopAppInfo();

            if (System.Environment.Is64BitOperatingSystem)
                CollectPhotoshopAppInfoWow6432();

            if (m_photoShopInfoList.Count > 0)
            {
                foreach (PhotoshopInfo csInfo in m_photoShopInfoList)
                {
                    AppInfo appInfo   = new AppInfo();
                    appInfo.Name      = csInfo.AppName;
                    appInfo.Path      = csInfo.ExePath;
                    appInfo.Extension = "tga;psd;ftxa;ftxb";

                    m_applicationList.Add(appInfo);
                }
            }
        } // LoadSuggestedApplicationsListFromXML

        private string FindApplicationName(XmlElement applicationNode)
        {
            if (applicationNode == null)
                return string.Empty;

            var namesNode = applicationNode.ChildNodes
                .Cast<XmlNode>()
                .FirstOrDefault(xn => String.Equals(xn.Name, "names", StringComparison.CurrentCultureIgnoreCase));

            if (namesNode != null)
            {
                foreach (XmlElement nameNode in namesNode.ChildNodes)
                {
                    string cultureAttribute = nameNode.GetAttribute("Culture");

                    CultureInfo culture = null;
                    if (cultureAttribute.Length > 0)
                    {
                        try
                        {
                            culture = CultureInfo.GetCultureInfo(cultureAttribute);
                        }
                        catch { }
                    }
                    else
                        culture = DefaultCulture;

                    if (culture != null && culture.ThreeLetterISOLanguageName == Thread.CurrentThread.CurrentCulture.ThreeLetterISOLanguageName)
                        return nameNode.GetAttribute("Title");
                }
            }

            return applicationNode.GetAttribute("Name"); // fallback value
        }

        //----------------------------------------------------------------------
        /// <summary>
        /// Photoshop info
        /// </summary>
        //----------------------------------------------------------------------
        private class PhotoshopInfo
        {
            public PhotoshopInfo(String appName, String exePath)
            {
                AppName = appName;
                ExePath = exePath;
            }

            public String AppName { set; get; }
            public String ExePath { set; get; }
        };

        private List<PhotoshopInfo> m_photoShopInfoList = new List<PhotoshopInfo>();

        //----------------------------------------------------------------------
        /// <summary>
        /// Find Photoshop path
        /// </summary>
        //----------------------------------------------------------------------
        private void CollectPhotoshopAppInfo()
        {
            MainApp.Instance.WriteToLog("Try to find Photoshop in system");

            String path = "";
            RegistryKey localMachineKey = Registry.LocalMachine;
            RegistryKey adobeKey        = localMachineKey.OpenSubKey("SOFTWARE\\Adobe\\Photoshop");
            if (adobeKey == null)
            {
                MainApp.Instance.WriteToLog("    Failed to open registry : " + "HKEY_LOCAL_MACHINE\\SOFTWARE\\Adobe\\Photoshop");
                return;
            }

            if (adobeKey.SubKeyCount <= 0)
            {
                MainApp.Instance.WriteToLog("    No subkeys in registry : " + "HKEY_LOCAL_MACHINE\\SOFTWARE\\Adobe\\Photoshop");
                return;
            }

            String[] subKeyNames = adobeKey.GetSubKeyNames();
            int i;
            for (i=0;i<subKeyNames.Length;i++)
            {
                MainApp.Instance.WriteToLog("    Find subkey : " + subKeyNames[i]);

                float versionNum = -1.0f;
                if (float.TryParse(subKeyNames[i],out versionNum))
                {
                    RegistryKey csAppKey = adobeKey.OpenSubKey(subKeyNames[i]);
                    if (csAppKey == null)
                    {
                        MainApp.Instance.WriteToLog("        Failed to open this key");
                        continue;
                    }

                    String dirName = csAppKey.GetValue("ApplicationPath") as String;
                    if ((dirName == null) || (dirName.Length <= 0))
                    {
                        MainApp.Instance.WriteToLog("        No ApplicationPath found");
                        continue;
                    }

                    path = dirName + "Photoshop.exe";

                    String appName = "Photoshop ";
                    int csVer = (int)versionNum;

                    // 新しいバージョンが追加されたら CollectPhotoshopAppInfoWow6432 も直すこと
                    switch (csVer)
                    {
                        case 8 :
                            appName += "CS 1";
                            break;

                        case 9 :
                            appName += "CS 2";
                            break;

                        case 10 :
                            appName += "CS 3";
                            break;

                        case 11 :
                            appName += "CS 4";
                            break;

                        case 12 :
                            appName += "CS 5";
                            break;

                        case 60:
                            appName += "CS 6";
                            break;

                        case 70:
                            appName += "CC";
                            break;

                        case 80:
                            appName += "CC 2014";
                            break;

                        default:
                            // 未対応バージョン
                            appName += string.Format("version {0}", versionNum);
                            break;
                    }

                    if (Environment.Is64BitOperatingSystem)
                    {
                        appName += " ( 64 Bit )";
                    }

                    MainApp.Instance.WriteToLog("    Photoshop found : " + appName);

                    PhotoshopInfo info = new PhotoshopInfo(appName, path);
                    m_photoShopInfoList.Add(info);
                }
            }
        }

        //----------------------------------------------------------------------
        /// <summary>
        /// Find Photoshop path for 32 bit app installed in 64 bit OS
        /// </summary>
        //----------------------------------------------------------------------
        private void CollectPhotoshopAppInfoWow6432()
        {
            MainApp.Instance.WriteToLog("Try to find Photoshop in system Wow6432");

            String path = "";
            RegistryKey localMachineKey = Registry.LocalMachine;
            RegistryKey adobeKey = localMachineKey.OpenSubKey("SOFTWARE\\Wow6432Node\\Adobe\\Photoshop");
            if (adobeKey == null)
            {
                MainApp.Instance.WriteToLog("    Failed to open registry : " + "HKEY_LOCAL_MACHINE\\Wow6432Node\\SOFTWARE\\Adobe\\Photoshop");
                return;
            }

            if (adobeKey.SubKeyCount <= 0)
            {
                MainApp.Instance.WriteToLog("    No subkeys in registry : " + "HKEY_LOCAL_MACHINE\\SOFTWARE\\Adobe\\Photoshop");
                return;
            }

            String[] subKeyNames = adobeKey.GetSubKeyNames();
            int i;
            for (i = 0; i < subKeyNames.Length; i++)
            {
                MainApp.Instance.WriteToLog("    Find subkey : " + subKeyNames[i]);

                float versionNum = -1.0f;
                if (float.TryParse(subKeyNames[i], out versionNum))
                {
                    RegistryKey csAppKey = adobeKey.OpenSubKey(subKeyNames[i]);
                    if (csAppKey == null)
                    {
                        MainApp.Instance.WriteToLog("        Failed to open this key");
                        continue;
                    }

                    String dirName = csAppKey.GetValue("ApplicationPath") as String;
                    if ((dirName == null) || (dirName.Length <= 0))
                    {
                        MainApp.Instance.WriteToLog("        No ApplicationPath found");
                        continue;
                    }

                    path = dirName + "Photoshop.exe";

                    String appName = "Photoshop ";
                    int csVer = (int)versionNum;

                    // 新しいバージョンが追加されたら CollectPhotoshopAppInfo も直すこと
                    switch (csVer)
                    {
                        case 8:
                            appName += "CS 1";
                            break;

                        case 9:
                            appName += "CS 2";
                            break;

                        case 10:
                            appName += "CS 3";
                            break;

                        case 11:
                            appName += "CS 4";
                            break;

                        case 12:
                            appName += "CS 5";
                            break;

                        case 60:
                            appName += "CS 6";
                            break;

                        case 70:
                            appName += "CC";
                            break;

                        case 80:
                            appName += "CC 2014";
                            break;

                        default:
                            // 未対応バージョン
                            appName += string.Format("version {0}", versionNum);
                            break;
                    }

                    appName += " ( 32 Bit )";

                    MainApp.Instance.WriteToLog("    Photoshop found : " + appName);

                    PhotoshopInfo info = new PhotoshopInfo(appName, path);
                    m_photoShopInfoList.Add(info);
                }
            }
        }

        //----------------------------------------------------------------------
        /// <summary>
        /// Build drop box for application list
        /// </summary>
        //----------------------------------------------------------------------
        public void BuildDropMenuApplicationsList()
        {
            m_applicationList.Clear();
            LoadSuggestedApplicationsListFromXML();
        }

        //----------------------------------------------------------------------
        /// <summary>
        /// Number of app list count
        /// </summary>
        //----------------------------------------------------------------------
        public int NumRegisteredApplications
        {
            get { return m_applicationList.Count; }
        }

        //----------------------------------------------------------------------
        /// <summary>
        /// Get registered app data
        /// </summary>
        //----------------------------------------------------------------------
        public AppInfo GetRegisteredApplicationInfo( int index )
        {
            return m_applicationList[index];
        }

        //----------------------------------------------------------------------
        /// <summary>
        /// Find registered app info from app name ( return -1 if not found )
        /// </summary>
        //----------------------------------------------------------------------
        public int FindRegisterAppName(String path)
        {
            int i;
            for (i = 0; i < m_applicationList.Count; i++)
            {
                AppInfo appInfo = m_applicationList[i];
                if (String.Compare(appInfo.Name,path,StringComparison.CurrentCultureIgnoreCase)==0)
                    return i;
            }

            return -1;
        }

        //----------------------------------------------------------------------
        /// <summary>
        /// Find registered app info from app path name ( return -1 if not found )
        /// </summary>
        //----------------------------------------------------------------------
        public int FindRegisterAppFromPath(String path)
        {
            int i;
            for (i = 0; i < m_applicationList.Count; i++)
            {
                AppInfo appInfo = m_applicationList[i];
                if (appInfo.Path.Contains(path))
                    return i;
            }

            return -1;
        }

        #endregion

        #region Check if saving required
        //----------------------------------------------------------------------
        /// <summary>
        /// IsSavingRequired
        /// </summary>
        //----------------------------------------------------------------------
        public bool IsSavingRequired()
        {
            foreach (RegistrationInfo info in m_registrationList)
            {
                if (info.IsSettingsChanged())
                    return true;
            }

            return false;
        }
        #endregion

        #region Check if we should warn about reset
        //----------------------------------------------------------------------
        /// <summary>
        /// Should we warn about resetting?
        /// </summary>
        //----------------------------------------------------------------------
        public bool ShouldWarnAboutResetting()
        {
            foreach (RegistrationInfo info in m_registrationList)
            {
                if (info.ShouldWarnAboutResetting())
                    return true;
            }

            return false;
        }
        #endregion

        #region Check if we should warn about reset
        //----------------------------------------------------------------------
        /// <summary>
        /// Turn of explorer services, if reset is selected
        /// </summary>
        //----------------------------------------------------------------------
        public void TurnOffServicesIfReset()
        {
            foreach (RegistrationInfo info in m_registrationList)
            {
                info.TurnOffServicesIfReset();
            }
        }
        #endregion

        //----------------------------------------------------------------------
        /// <summary>
        /// インストールしようとしているプラグインをロックしているアプリ
        /// </summary>
        //----------------------------------------------------------------------
        public IEnumerable<LockAppPair> LockApps
        {
            get
            {
                var processUtility = new ProcessUtility();

                var filePaths =
                    MainApp.Instance.ShellPluginManager.Plugins.SelectMany(x => new[] { x.ModulePath32Bit, x.ModulePath64Bit })
                    .ToDictionary(x => x.ToLowerInvariant());

                return processUtility.GetBlockingModulePaths(filePaths);
            }
        }
    }
}
