﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Diagnostics;
using Microsoft.Win32;
using System.Security.AccessControl;

namespace NintendoWare.ExtensionManager
{
    //----------------------------------------------------------------------
    /// <summary>
    /// Easy Setup Status
    /// </summary>
    //----------------------------------------------------------------------
    public struct EasySetup
    {
        public bool UseIconService;
        public bool UseInfoTipService;
        public bool UsePreviewService;
        public string ShellCommandName;
        public string ShellCommandPath;

        public EasySetup(bool icon, bool tip, bool preview, string name)
        {
            this.UseIconService = icon;
            this.UseInfoTipService = tip;
            this.UsePreviewService = preview;
            this.ShellCommandName = name;
            this.ShellCommandPath = "";
        }

    }

    //==========================================================================
    /// <summary>
    /// RegistrationInfo for given file extension
    /// </summary>
    //==========================================================================
    public class RegistrationInfo : IDisposable
    {
        //----------------------------------------------------------------------
        /// <summary>
        /// Previously registered application
        /// </summary>
        //----------------------------------------------------------------------
        public class PrevRegisteredAppInfo
        {
            //------------------------------------------------------------------
            /// <summary>
            /// Constructor
            /// </summary>
            //------------------------------------------------------------------
            public PrevRegisteredAppInfo()
            {
                ShellProgID        = "";
                UserChoiceCommand  = "";
                UserChoiceApplicationPath = "";
                UserChoiceHash = "";
            }

            //------------------------------------------------------------------
            /// <summary>
            /// Shell ProgId
            /// </summary>
            //------------------------------------------------------------------
            public String ShellProgID { get; set; }

            //------------------------------------------------------------------
            /// <summary>
            /// UserChoice Command
            /// </summary>
            //------------------------------------------------------------------
            public String UserChoiceCommand { get; set; }

            //------------------------------------------------------------------
            /// <summary>
            /// UserChoice Application Path
            /// </summary>
            //------------------------------------------------------------------
            public String UserChoiceApplicationPath { get; set; }

            //------------------------------------------------------------------
            /// <summary>
            /// UserChoice Hash key for Windows8
            /// </summary>
            //------------------------------------------------------------------
            public String UserChoiceHash { get; set; }
        };

        //----------------------------------------------------------------------
        /// <summary>
        /// Status
        /// </summary>
        //----------------------------------------------------------------------
        public class Status
        {
            //------------------------------------------------------------------
            /// <summary>
            /// Constructor
            /// </summary>
            //------------------------------------------------------------------
            public Status()
            {
                UseIconService        = false;
                UseInfoTipService     = false;
                UsePreviewService     = false;
                ShellCommandName      = Properties.Resources.TEXT_NONE;
                ShellCommandPath      = "";
            }

            #region PlugIn Service
            //------------------------------------------------------------------
            /// <summary>
            /// Icon Service
            /// </summary>
            //------------------------------------------------------------------
            public bool UseIconService
            {
                set;
                get;
            }

            //------------------------------------------------------------------
            /// <summary>
            /// Preview Service
            /// </summary>
            //------------------------------------------------------------------
            public bool UsePreviewService
            {
                set;
                get;
            }

            //------------------------------------------------------------------
            /// <summary>
            /// InfoTip Service
            /// </summary>
            //------------------------------------------------------------------
            public bool UseInfoTipService
            {
                set;
                get;
            }

            //------------------------------------------------------------------
            /// <summary>
            /// Column Service
            /// </summary>
            //------------------------------------------------------------------
            public bool UseColumnService
            {
                set;
                get;
            }

            //------------------------------------------------------------------
            /// <summary>
            /// Set using Service by Name
            /// </summary>
            //------------------------------------------------------------------
            public void SetUseService(ShellExtensionService service, bool bUse)
            {
                if (service == ShellExtensionService.IconService)
                {
                    UseIconService = bUse;
                }
                else if (service == ShellExtensionService.InfoTipService)
                {
                    UseInfoTipService = bUse;
                }
                else if (service == ShellExtensionService.PreviewService)
                {
                    UsePreviewService = bUse;
                }
            }

            //------------------------------------------------------------------
            /// <summary>
            /// Service check by Name
            /// </summary>
            //------------------------------------------------------------------
            public bool IsUsingService(ShellExtensionService service)
            {
                if (service == ShellExtensionService.IconService)
                {
                    return UseIconService;
                }
                else if (service == ShellExtensionService.InfoTipService)
                {
                    return UseInfoTipService;
                }
                else if (service == ShellExtensionService.PreviewService)
                {
                    return UsePreviewService;
                }
                else if (service == ShellExtensionService.ContextMenuService)
                {
                    return false;
                }

                return false;
            }

            //------------------------------------------------------------------
            /// <summary>
            /// ShellCommand Name ( from known app list, "Custom" (Properties.Resources.TEXT_CUSTOM) for user defined )
            /// </summary>
            //------------------------------------------------------------------
            public String ShellCommandName
            {
                set;
                get;
            }

            //------------------------------------------------------------------
            /// <summary>
            /// ShellCommand Service
            /// </summary>
            //------------------------------------------------------------------
            public string ShellCommandPath
            {
                set { m_shellCommandPath = value; }
                get { return m_shellCommandPath; }
            }

            private String m_shellCommandPath = "";
            #endregion

            #region Copy from source
            //------------------------------------------------------------------
            /// <summary>
            /// Copy from source
            /// </summary>
            //------------------------------------------------------------------
            public void CopyFrom( Status src )
            {
                this.UseIconService        = src.UseIconService;
                this.UseInfoTipService     = src.UseInfoTipService;
                this.UsePreviewService     = src.UsePreviewService;
                this.ShellCommandName      = src.ShellCommandName;
                this.ShellCommandPath      = src.ShellCommandPath;
            }

            //------------------------------------------------------------------
            /// <summary>
            /// Copy from source ( just given service )
            /// </summary>
            //------------------------------------------------------------------
            public void CopyFrom(Status src, ShellExtensionService service, RegistrationInfo regInfo)
            {
                if (service == ShellExtensionService.AllServices)
                {
                    if (regInfo.IsServiceAvailable(ShellExtensionService.IconService))
                    {
                        this.UseIconService = src.UseIconService;
                    }

                    if (regInfo.IsServiceAvailable(ShellExtensionService.InfoTipService))
                    {
                        this.UseInfoTipService = src.UseInfoTipService;
                    }

                    if (regInfo.IsServiceAvailable(ShellExtensionService.PreviewService))
                    {
                        this.UsePreviewService = src.UsePreviewService;
                    }
                }

                else if (service == ShellExtensionService.ShellCommandService)
                {
                    if (regInfo.IsServiceAvailable(service))
                    {
                        this.ShellCommandName = src.ShellCommandName;
                        this.ShellCommandPath = src.ShellCommandPath;
                    }
                }
                else if (regInfo.IsServiceAvailable(service))
                {
                    this.SetUseService(service, src.IsUsingService(service));
                }
            }
            #endregion

            #region Different?
            //------------------------------------------------------------------
            /// <summary>
            /// Copy from source
            /// </summary>
            //------------------------------------------------------------------
            public bool IsDifferentFrom(Status src)
            {
                if ( (this.UseIconService != src.UseIconService) ||
                     (this.UseInfoTipService != src.UseInfoTipService) ||
                     (this.UsePreviewService != src.UsePreviewService) ||
                     (this.ShellCommandName != src.ShellCommandName) ||
                     (this.ShellCommandPath != src.ShellCommandPath) )
                {
                    return true;
                }

                return false;
            }
            #endregion
        }


        //----------------------------------------------------------------------
        /// <summary>
        /// Constructor
        /// </summary>
        //----------------------------------------------------------------------
        public RegistrationInfo( String extension, String pluginName )
        {
            ExtensionName  = extension;
            PluginName     = pluginName;
        }

        //----------------------------------------------------------------------
        /// <summary>
        /// Dispose
        /// </summary>
        //----------------------------------------------------------------------
        virtual public void Dispose()
        {
        }

        #region Extension
        //----------------------------------------------------------------------
        /// <summary>
        /// Extension Name
        /// </summary>
        //----------------------------------------------------------------------
        public String ExtensionName
        {
            set;
            get;
        }

        /// <summary>
        /// ユーザ指定の拡張子かどうか
        /// </summary>
        public bool IsCustomExtension { get; set; } = false;

        /// <summary>
        /// ユーザ指定の拡張子を削除(アンインストール)するかどうか
        /// </summary>
        public bool RemoveCustomExtension { get; set; } = false;

        #endregion

        #region Plugin
        //----------------------------------------------------------------------
        /// <summary>
        /// Plugin Name
        /// </summary>
        //----------------------------------------------------------------------
        public String PluginName
        {
            set;
            get;
        }

        #endregion

        #region AvailableServices
        //----------------------------------------------------------------------
        /// <summary>
        /// Services that are available for this extension
        /// </summary>
        //----------------------------------------------------------------------
        public IEnumerable<ShellExtensionService> AvailableServices
        {
            set;
            get;
        }

        #endregion

        #region Current and Old Status
        //----------------------------------------------------------------------
        /// <summary>
        /// Current status
        /// </summary>
        //----------------------------------------------------------------------
        public Status CurrentStatus
        {
            get { return m_currentStatus; }
        }

        private Status m_currentStatus = new Status();

        //----------------------------------------------------------------------
        /// <summary>
        /// Old status
        /// </summary>
        //----------------------------------------------------------------------
        public Status OldStatus
        {
            get { return m_oldStatus; }
        }

        private Status m_oldStatus = new Status();

        //----------------------------------------------------------------------
        /// <summary>
        /// Is settings changed?
        /// </summary>
        //----------------------------------------------------------------------
        public bool IsSettingsChanged()
        {
            if (m_currentStatus.IsDifferentFrom(m_oldStatus))
                return true;

            return false;
        }

        //----------------------------------------------------------------------
        /// <summary>
        /// Should we warn about resetting?
        /// </summary>
        //----------------------------------------------------------------------
        public bool ShouldWarnAboutResetting()
        {
            // We command is to Reset and if any of services are set to Use, then we should warn user
            // that all services will be disabled
            if ((AvailableServices.Contains(ShellExtensionService.ShellCommandService) &&
                String.Compare(m_currentStatus.ShellCommandName, Properties.Resources.TEXT_RESET,
                               StringComparison.CurrentCultureIgnoreCase) == 0) &&
                ((AvailableServices.Contains(ShellExtensionService.IconService) && m_currentStatus.UseIconService) ||
                 (AvailableServices.Contains(ShellExtensionService.InfoTipService) && m_currentStatus.UseInfoTipService) ||
                 (AvailableServices.Contains(ShellExtensionService.PreviewService) && m_currentStatus.UsePreviewService))
                )
            {
                return true;
            }

            return false;
        }

        //----------------------------------------------------------------------
        /// <summary>
        /// Turn of explorer services, if reset is selected
        /// </summary>
        //----------------------------------------------------------------------
        public void TurnOffServicesIfReset()
        {
            // We command is to Reset and if any of services are set to Use, then we should warn user
            // that all services will be disabled
            if ((AvailableServices.Contains(ShellExtensionService.ShellCommandService) &&
                String.Compare(m_currentStatus.ShellCommandName, Properties.Resources.TEXT_RESET,
                               StringComparison.CurrentCultureIgnoreCase) == 0) &&
                ((AvailableServices.Contains(ShellExtensionService.IconService) && m_currentStatus.UseIconService) ||
                 (AvailableServices.Contains(ShellExtensionService.InfoTipService) && m_currentStatus.UseInfoTipService) ||
                 (AvailableServices.Contains(ShellExtensionService.PreviewService) && m_currentStatus.UsePreviewService))
                )
            {
                m_currentStatus.UseIconService        = false;
                m_currentStatus.UseInfoTipService     = false;
                m_currentStatus.UsePreviewService     = false;
            }
        }

        //----------------------------------------------------------------------
        /// <summary>
        /// If service is available
        /// </summary>
        //----------------------------------------------------------------------
        public bool IsServiceAvailable(ShellExtensionService service)
        {
            if (service == ShellExtensionService.None)
                return false;

            if (AvailableServices == null)
                return false;

            return AvailableServices.Contains(service);
        }
        #endregion

        #region Load from easy setup
        public bool LoadFromEasySetup(EasySetup setup)
        {
            m_currentStatus.UseIconService = setup.UseIconService;
            m_currentStatus.UseInfoTipService = setup.UseInfoTipService;
            m_currentStatus.UsePreviewService = setup.UsePreviewService;
            m_currentStatus.ShellCommandName = setup.ShellCommandName;
            m_currentStatus.ShellCommandPath = setup.ShellCommandPath;

            // ファイル関連付けがカスタムが指定されている場合は、レジストリの情報を取得する
            if (m_currentStatus.ShellCommandName == Properties.Resources.TEXT_CUSTOM)
            {
                MainApp app = MainApp.Instance;
                Debug.Assert(app != null);

                ShellPluginManager pluginManager = app.ShellPluginManager;
                Debug.Assert(pluginManager != null);

                // Find plugin
                ShellPlugin plugin = pluginManager.FindPlugin(PluginName);
                if (plugin == null)
                    return false;

                RegistrationManager registrationManager = app.RegistrationManager;
                Debug.Assert(registrationManager != null);

                // Try to load prev associated app info
                LoadPrevRegisteredAppInfo(); // Load previous information

                // レジストリからファイル関連付けの情報取得してセット
                LoadRegistryProgramID(plugin, m_currentStatus);
            }

            m_oldStatus.CopyFrom(m_currentStatus);

            return true;
        }
        #endregion

        #region Load from registry
        //----------------------------------------------------------------------
        /// <summary>
        /// Load from registry
        /// </summary>
        //----------------------------------------------------------------------
        public bool LoadFromRegistry()
        {
            m_currentStatus.UseIconService        = false;
            m_currentStatus.UseInfoTipService     = false;
            m_currentStatus.UsePreviewService     = false;
            m_currentStatus.ShellCommandName      = Properties.Resources.TEXT_NONE;
            m_currentStatus.ShellCommandPath      = "";

            MainApp app = MainApp.Instance;
            Debug.Assert( app != null );

            ShellPluginManager pluginManager = app.ShellPluginManager;
            Debug.Assert( pluginManager != null );

            // Find plugin
            ShellPlugin plugin = pluginManager.FindPlugin( PluginName );
            if (plugin == null)
                return false;

            // Try to load prev associated app info
            LoadPrevRegisteredAppInfo(); // Load previous information

            // Load registered
            LoadRegistryProgramID(plugin, m_currentStatus);

            m_oldStatus.CopyFrom( m_currentStatus );

            return true;
        }

        //----------------------------------------------------------------------
        /// <summary>
        /// Load registry for program id
        /// </summary>
        //----------------------------------------------------------------------
        private bool LoadRegistryProgramID(ShellPlugin plugin, Status status)
        {
            MainApp app = MainApp.Instance;
            Debug.Assert( app != null );

            RegistrationManager registrationManager = app.RegistrationManager;
            Debug.Assert( registrationManager != null );

            RegistryKey classRootkey = Registry.ClassesRoot;
            if (classRootkey == null)
                return false;

            // Open program ID key
            RegistryKey progIDkey = classRootkey.OpenSubKey("NintendoWare." + ExtensionName + ".1");
            if (progIDkey != null)
            {
                // Add extension handling registration
                String extKeyStr = ("." + ExtensionName).ToLower();
                RegistryKey extKey = classRootkey.OpenSubKey(extKeyStr);
                var progIDStr = extKey?.GetValue(null) as string;

                // オプションの引継ぎに対応していないバージョンのセットアップツールが残した情報は引き継げない。
                // 対応していないバージョンのセットアップツールでは、ResetFlag が削除されずに残る。
                if (progIDkey.GetValue("ResetFlag") == null &&
                    ((progIDStr != null && progIDStr.Equals("NintendoWare." + ExtensionName + ".1", StringComparison.CurrentCultureIgnoreCase)) ||
                     (progIDkey.GetValue("UninstallFlag") as int?) == 1)) // 直前でアンインストールした場合
                {
                    // ShellEx
                    if (progIDkey.OpenSubKey("ShellEx") != null)
                    {
                        bool manageIcon = false;
                        LoadRegistryIconService(plugin, progIDkey, ref manageIcon);
                        m_currentStatus.UseIconService = manageIcon;

                        bool manageInfoTip = false;
                        LoadRegistryInfoTipService(plugin, progIDkey, ref manageInfoTip);
                        m_currentStatus.UseInfoTipService = manageInfoTip;

                        bool managePreview = false;
                        LoadRegistryPreviewService(plugin, progIDkey, ref managePreview);
                        m_currentStatus.UsePreviewService = managePreview;

                        bool manageColumn = false;
                        if (Environment.OSVersion.Version.Major >= 6) // Vista or Windows 7
                        {
                            LoadRegistryPropertyStoreService(plugin, progIDkey, ref manageColumn);
                        }
                        else
                        {
                            LoadRegistryColumnProviderService(plugin, progIDkey, ref manageColumn);
                        }

                        m_currentStatus.UseColumnService = manageColumn;
                    }
                }
            }

            if (SelectShellCommand(plugin, registrationManager) == false)
                return false;

            return true;
        }
        #endregion

        #region Save to registry
        //----------------------------------------------------------------------
        /// <summary>
        /// Add to approved list
        /// </summary>
        //----------------------------------------------------------------------
        private void AddToApprovedList(RegistryKey approvedList, Guid guid, String serviceName)
        {
            if (guid == Guid.Empty)
                return;

            String GUIDStr = "{" + guid.ToString().ToUpper() + "}";
            approvedList.SetValue(GUIDStr, serviceName);
        }

        //----------------------------------------------------------------------
        /// <summary>
        /// Remove to approved list
        /// </summary>
        //----------------------------------------------------------------------
        private void RemoveToApprovedList(RegistryKey approvedList, Guid guid)
        {
            if (guid == Guid.Empty)
                return;

            String GUIDStr = "{" + guid.ToString().ToUpper() + "}";
            approvedList.DeleteValue(GUIDStr, false);
        }

        //----------------------------------------------------------------------
        /// <summary>
        /// Save to registry
        /// </summary>
        //----------------------------------------------------------------------
        public bool SaveToRegistry(bool uninstall)
        {
            MainApp app = MainApp.Instance;
            Debug.Assert(app!=null);

            ShellPluginManager pluginManager = app.ShellPluginManager;
            Debug.Assert(pluginManager!=null);

            // Find plugin
            ShellPlugin plugin = pluginManager.FindPlugin( PluginName );
            if (plugin == null)
                return false;

            // Save program ID
            RegistryKey progIDkey = SaveRegistryProgramID( plugin );
            if (progIDkey == null)
                return false;

            if (!uninstall)
            {
                if (AvailableServices.Contains(ShellExtensionService.IconService))
                    SaveRegistryIconService(plugin, progIDkey, m_currentStatus.UseIconService && !RemoveCustomExtension);

                if (AvailableServices.Contains(ShellExtensionService.InfoTipService))
                    SaveRegistryInfoTipService(plugin, progIDkey, m_currentStatus.UseInfoTipService && !RemoveCustomExtension);

                if (AvailableServices.Contains(ShellExtensionService.PreviewService))
                    SaveRegistryPreviewService(plugin, progIDkey, m_currentStatus.UsePreviewService && !RemoveCustomExtension);
            }

            if (AvailableServices.Contains(ShellExtensionService.ContextMenuService))
                SaveRegistryContextMenuService(plugin, progIDkey, false);

            if (AvailableServices.Contains(ShellExtensionService.ColumnService))
            {
                if (Environment.OSVersion.Version.Major >= 6) // Vista or Windows 7
                {
                    SaveRegistryPropertyStoreService(plugin, progIDkey, false);
                }
                else
                {
                    SaveRegistryColumnProviderService(plugin, progIDkey, false);
                }
            }

            if (AvailableServices.Contains(ShellExtensionService.ShellCommandService))
            {
                String commandPath = "";
                if ((m_currentStatus.ShellCommandName.Length == 0) || (m_currentStatus.ShellCommandName == Properties.Resources.TEXT_NONE) || RemoveCustomExtension)
                {
                    // If empty
                }
                else
                {
                    commandPath = m_currentStatus.ShellCommandPath;
                }

                // 簡易アンインストールのときは設定を残す
                if (!uninstall)
                {
                    SaveRegistryShellCommandService(plugin, progIDkey, commandPath);
                }

                // Save prev associated app info ( if this is 1st time install )
                SavePrevRegisteredAppInfo();
            }

            if (Environment.OSVersion.Version.Major == 6 && Environment.OSVersion.Version.Minor >= 2) // Windows 8
            {
                EditFileExtOpenWithProgid();
            }

            bool reset = false;
            // If we should reset, set rest flag before DLL get called
            if (
                ((!AvailableServices.Contains(ShellExtensionService.ShellCommandService) || String.Compare(m_currentStatus.ShellCommandName, Properties.Resources.TEXT_RESET, StringComparison.CurrentCultureIgnoreCase) == 0) &&
                 (!AvailableServices.Contains(ShellExtensionService.IconService) || m_currentStatus.UseIconService==false) &&
                 (!AvailableServices.Contains(ShellExtensionService.InfoTipService) || m_currentStatus.UseInfoTipService == false) &&
                 (!AvailableServices.Contains(ShellExtensionService.PreviewService) || m_currentStatus.UsePreviewService == false)) ||
                 RemoveCustomExtension
                )
            {
                reset = true;

                // 一旦削除する。RegEdit_Main で復元される場合もある。
                RemoveExtKey(plugin);
            }

            RemoveUserChoiceKey();

            if (reset)
            {
                RestorePrevRegisteredAppInfo(progIDkey);
            }

            if (uninstall)
            {
                // 次回のインストール時に設定が引き継がれるようにする
                progIDkey.SetValue("UninstallFlag", 1);
            }
            else
            {
                progIDkey.DeleteValue("UninstallFlag", false);

                // ユーザ指定拡張子の情報の更新
                if (IsCustomExtension)
                {
                    using (RegistryKey thumbsKey = Registry.ClassesRoot.CreateSubKey("NintendoWare.THUMBS.2"))
                    {
                        if (RemoveCustomExtension)
                        {
                            thumbsKey.DeleteValue(ExtensionName);
                        }
                        else
                        {
                            thumbsKey.SetValue(ExtensionName, "ext");
                        }
                    }
                }
            }

            // 現バージョンでは不要になっているレジストリの削除
            progIDkey.DeleteValue("ResetFlag", false);

            return true;
        }

        private void RemoveUserChoiceKey()
        {
            using (var userFileExtKey = Registry.CurrentUser.OpenSubKey(@"SOFTWARE\Microsoft\Windows\CurrentVersion\Explorer\FileExts\." + ExtensionName, true))
            {
                if (userFileExtKey != null)
                {
                    try
                    {
                        userFileExtKey.DeleteSubKey("UserChoice", false);
                    }
                    catch (Exception e)
                    {
                        MainApp.Instance.WriteToLog(e.Message);
                    }
                }
            }
        }

        private void RestorePrevRegisteredAppInfo(RegistryKey progIdKey)
        {
            using (var prevAppInfoKey = progIdKey.OpenSubKey("PrevAppInfo"))
            {
                using (var extKey = Registry.ClassesRoot.CreateSubKey("." + ExtensionName))
                {
                    var progID = prevAppInfoKey.GetValue("ShellProgID") as string;
                    if (!string.IsNullOrEmpty(progID))
                    {
                        // 関連付けを設定
                        try
                        {
                            if (extKey != null)
                            {
                                extKey.SetValue(string.Empty, progID);
                            }
                        }
                        catch (Exception e)
                        {
                            MainApp.Instance.WriteToLog(e.Message);
                        }
                    }
                    else
                    {
                        // 関連付けを破棄
                        try
                        {
                            extKey.DeleteValue(string.Empty, false);
                        }
                        catch (Exception e)
                        {
                            MainApp.Instance.WriteToLog(e.Message);
                        }
                    }
                }

                var userChoiceCmd = prevAppInfoKey.GetValue("UserChoice") as string;
                var userChoiceHash = prevAppInfoKey.GetValue("UserChoiceHash") as string;
                if (!string.IsNullOrEmpty(userChoiceCmd))
                {
                    using (var userFileExtKey = Registry.CurrentUser.OpenSubKey(@"SOFTWARE\Microsoft\Windows\CurrentVersion\Explorer\FileExts\." + ExtensionName, true))
                    {
                        if (userFileExtKey != null)
                        {
                            using (var userChoiceKeyTest = userFileExtKey.OpenSubKey("UserChoice"))
                            {
                                // UserChoice が設定されているということは、エクスプローラ拡張がインストールされている間に
                                // エクスプローラー拡張を使わずに設定されたと考えられるため、そのままにする。
                                if (userChoiceKeyTest == null)
                                {
                                    using (var userChoiceKey = userFileExtKey.CreateSubKey("UserChoice"))
                                    {
                                        if (userChoiceKey != null)
                                        {
                                            try
                                            {
                                                userChoiceKey.SetValue("Progid", userChoiceCmd);
                                                if (!string.IsNullOrEmpty(userChoiceHash))
                                                {
                                                    userChoiceKey.SetValue("Hash", userChoiceHash);
                                                }
                                            }
                                            catch (Exception e)
                                            {
                                                MainApp.Instance.WriteToLog(e.Message);
                                            }

                                        }
                                    }
                                }
                            }
                        }
                    }
                }
            }

        }

        private void RemoveExtKey(ShellPlugin plugin)
        {

            RegistryKey classRootkey = Registry.ClassesRoot;
            if (classRootkey == null)
                return;

            try
            {
                String extKeyStr = ("." + ExtensionName).ToLower();
                RegistryKey extKey = classRootkey.OpenSubKey(extKeyStr);
                if (extKey != null && extKey.ValueCount == 1 && extKey.SubKeyCount == 0)
                {
                    extKey.Close();
                    classRootkey.DeleteSubKey(extKeyStr);
                }
            }
            catch (System.Exception e)
            {
                MainApp.Instance.WriteToLog(e.Message);
            }
        }

        //----------------------------------------------------------------------
        /// <summary>
        /// Save registry for program id
        /// </summary>
        //----------------------------------------------------------------------
        private RegistryKey SaveRegistryProgramID( ShellPlugin plugin )
        {
            MainApp app = MainApp.Instance;
            Debug.Assert( app != null );

            RegistrationManager registrationManager = app.RegistrationManager;
            Debug.Assert( registrationManager != null );

            RegistryKey classRootkey = Registry.ClassesRoot;
            if (classRootkey == null)
                return null;

            RegistryKey progIDKey = null;

            try
            {
                // Add extension handling registration
                String extKeyStr = ("." + ExtensionName).ToLower();
                RegistryKey extKey = classRootkey.CreateSubKey( extKeyStr );

                // Set program ID to this extension
                String extNameUpper = ExtensionName.ToUpper();
                String progIDStr = "NintendoWare." + extNameUpper + ".1";
                extKey.SetValue( null, progIDStr );

                // Create program ID key
                progIDKey = classRootkey.CreateSubKey( progIDStr );

                progIDKey.CreateSubKey( "ShellEx" );

                //--------------------------------------------------------------
                // If we have user choice choice app associated with copy the shell command
                // then remove this user choice
                //--------------------------------------------------------------
                String userAppKeyStr = "SOFTWARE\\Microsoft\\Windows\\CurrentVersion\\Explorer\\FileExts\\." + ExtensionName;
                RegistryKey curUserKey = Registry.CurrentUser;
                if (curUserKey != null)
                {
                    RegistryKey userAppKey = curUserKey.OpenSubKey( userAppKeyStr );
                    if (userAppKey != null)
                    {
                        // To do
                    }
                }
            }

            catch (System.Exception e)
            {
                MainApp.Instance.WriteToLog( e.Message );
                return null;
            }

            return progIDKey;
        }

        //----------------------------------------------------------------------
        /// <summary>
        /// Save registry for shell plugin
        /// </summary>
        //----------------------------------------------------------------------
        private bool SavePluginToCLSID( Guid guid,
                                        ShellPlugin plugin,
                                        String pluginDescription,
                                        bool bUse)
        {
            MainApp app = MainApp.Instance;
            Debug.Assert( app != null );

            RegistrationManager registrationManager = app.RegistrationManager;
            Debug.Assert( registrationManager != null );

            try
            {
                RegistryKey classRootkey = Registry.ClassesRoot;

                RegistryKey CLSIDkey = classRootkey.OpenSubKey( "CLSID", true );

                String GUIDStr = "{" + guid.ToString().ToUpper() + "}";
                RegistryKey inprocServerKey = null;
                if (bUse)
                {
                    RegistryKey GUIDKey = CLSIDkey.CreateSubKey(GUIDStr);

                    GUIDKey.SetValue(null, pluginDescription);

                    inprocServerKey = GUIDKey.CreateSubKey("InprocServer32");
                }
                else
                {
                    CLSIDkey.DeleteSubKeyTree(GUIDStr, false);
                }

                if (Environment.Is64BitOperatingSystem)
                {
                    if (inprocServerKey != null)
                    {
                        inprocServerKey.SetValue(null, plugin.ModulePath64Bit);
                        inprocServerKey.SetValue("ThreadingModel", "Apartment");
                    }

                    // If 64 bit OS, we have to register 32 bit mode as well to override WOW64 automatic mirroring
                    RegistryKey classRootkey32 = RegistryKey.OpenBaseKey(Microsoft.Win32.RegistryHive.ClassesRoot, RegistryView.Registry32);
                    RegistryKey CLSIDkey32 = classRootkey32.OpenSubKey("CLSID", true);
                    RegistryKey inprocServerKey32 = null;
                    if (bUse)
                    {
                        RegistryKey GUIDKey32 = CLSIDkey32.CreateSubKey(GUIDStr);
                        GUIDKey32.SetValue(null, pluginDescription);

                        inprocServerKey32 = GUIDKey32.CreateSubKey("InprocServer32");
                    }
                    else
                    {
                        CLSIDkey32.DeleteSubKeyTree(GUIDStr, false);
                    }

                    if (inprocServerKey32 != null)
                    {
                        inprocServerKey32.SetValue(null, plugin.ModulePath32Bit);
                        inprocServerKey32.SetValue("ThreadingModel", "Apartment");
                    }
                }
                else
                {
                    if (inprocServerKey != null)
                    {
                        inprocServerKey.SetValue(null, plugin.ModulePath32Bit);
                        inprocServerKey.SetValue("ThreadingModel", "Apartment");
                    }
                }
            }

            catch (Exception e)
            {
                MainApp.Instance.WriteToLog( e.Message );
                return false;
            }

            return true;
        }

        #region Icon Service
        //----------------------------------------------------------------------
        /// <summary>
        /// Load registry for Icon Service
        /// </summary>
        //----------------------------------------------------------------------
        private bool LoadRegistryIconService(ShellPlugin plugin, RegistryKey progIDKey,
                                             ref bool bOut)
        {
            try
            {
                RegistryKey shellExtKey = progIDKey.OpenSubKey("ShellEx");
                RegistryKey iconHandlerKey = shellExtKey.OpenSubKey("IconHandler");
                bOut = (iconHandlerKey != null);
            }
            catch (Exception e)
            {
                MainApp.Instance.WriteToLog(e.Message);
                return false;
            }

            return true;
        }

        //----------------------------------------------------------------------
        /// <summary>
        /// Save registry for Icon Service
        /// </summary>
        //----------------------------------------------------------------------
        private bool SaveRegistryIconService( ShellPlugin plugin, RegistryKey progIDKey,
                                              bool bUse )
        {
            MainApp app = MainApp.Instance;
            Debug.Assert( app != null );

            RegistrationManager registrationManager = app.RegistrationManager;
            Debug.Assert( registrationManager != null );

            String serviceName = String.Format("NintendoWare {0} Icon Shell Extension", PluginName);

            Guid guid;
            if (!Define.IDs[ShellExtensionService.IconService].TryGetValue(PluginName?.ToUpper(), out guid))
            {
                return false;
            }

            try
            {
                RegistryKey shellExtKey       = progIDKey.CreateSubKey( "ShellEx" );
                if (bUse)
                {
                    RegistryKey serviceHandlerKey = shellExtKey.CreateSubKey("IconHandler");

                    String GUIDStr = "{" + guid.ToString().ToUpper() + "}";
                    serviceHandlerKey.SetValue(null, GUIDStr);
                }
                else
                {
                    shellExtKey.DeleteSubKeyTree("IconHandler", false);
                }
            }

            catch (Exception e)
            {
                MainApp.Instance.WriteToLog( e.Message );
                return false;
            }

            return true;
        }
        #endregion

        #region InfoTip Service
        // http://msdn.microsoft.com/en-us/library/windows/desktop/cc144067%28v=vs.85%29.aspx
        private const string infoTipHandler = "{00021500-0000-0000-C000-000000000046}";

        //----------------------------------------------------------------------
        /// <summary>
        /// Load registry for InfoTip Service
        /// </summary>
        //----------------------------------------------------------------------
        private bool LoadRegistryInfoTipService(ShellPlugin plugin, RegistryKey progIDKey,
                                                ref bool bOut)
        {
            try
            {
                RegistryKey shellExtKey = progIDKey.OpenSubKey("ShellEx");
                RegistryKey serviceHandlerKey = shellExtKey.OpenSubKey(infoTipHandler);
                bOut = (serviceHandlerKey != null);
            }
            catch (Exception e)
            {
                MainApp.Instance.WriteToLog(e.Message);
                return false;
            }

            return true;
        }

        //----------------------------------------------------------------------
        /// <summary>
        /// Save registry for InfoTip Service
        /// </summary>
        //----------------------------------------------------------------------
        private bool SaveRegistryInfoTipService( ShellPlugin plugin, RegistryKey progIDKey,
                                                 bool bUse )
        {
            MainApp app = MainApp.Instance;
            Debug.Assert( app != null );

            RegistrationManager registrationManager = app.RegistrationManager;
            Debug.Assert( registrationManager != null );

            String serviceName = String.Format("NintendoWare {0} InfoTip Shell Extension", PluginName);

            Guid guid = Guid.Empty;
            if (String.Compare(PluginName, "FTX", true) == 0)
                guid = Define.CLSID_ShellExtensionInfoTipFTX;
            else if (String.Compare(PluginName, "TGA", true) == 0)
                guid = Define.CLSID_ShellExtensionInfoTipTGA;
            else if (String.Compare(PluginName, "PSD", true) == 0)
                guid = Define.CLSID_ShellExtensionInfoTipPSD;
            else
                return false;

            try
            {
                RegistryKey shellExtKey = progIDKey.CreateSubKey( "ShellEx" );
                if (bUse)
                {
                    RegistryKey serviceHandlerKey = shellExtKey.CreateSubKey(infoTipHandler);

                    String GUIDStr = "{" + guid.ToString().ToUpper() + "}";
                    serviceHandlerKey.SetValue(null, GUIDStr);
                }
                else
                {
                    shellExtKey.DeleteSubKeyTree(infoTipHandler, false);
                }
            }

            catch (Exception e)
            {
                MainApp.Instance.WriteToLog( e.Message );
                return false;
            }

            return true;
        }
        #endregion


        #region Preview Service
        // http://msdn.microsoft.com/en-us/library/windows/desktop/cc144144%28v=vs.85%29.aspx
        private const string previewHandler = "{8895b1c6-b41f-4c1c-a562-0d564250836f}";
        private const string appIDHandler = "{6d2b5079-2f0b-48dd-ab7f-97cec514d30b}";

        //----------------------------------------------------------------------
        /// <summary>
        /// Load registry for Preview Service
        /// </summary>
        //----------------------------------------------------------------------
        private bool LoadRegistryPreviewService(ShellPlugin plugin, RegistryKey progIDKey,
                                                ref bool bOut)
        {
            try
            {
                RegistryKey shellExtKey = progIDKey.OpenSubKey("ShellEx");
                RegistryKey serviceHandlerKey = shellExtKey.OpenSubKey(previewHandler);
                bOut = (serviceHandlerKey != null);
            }
            catch (Exception e)
            {
                MainApp.Instance.WriteToLog(e.Message);
                return false;
            }

            return true;
        }

        //----------------------------------------------------------------------
        /// <summary>
        /// Save registry for Preview Service
        /// </summary>
        //----------------------------------------------------------------------
        private bool SaveRegistryPreviewService( ShellPlugin plugin, RegistryKey progIDKey,
                                                 bool bUse )
        {
            MainApp app = MainApp.Instance;
            Debug.Assert( app != null );

            RegistrationManager registrationManager = app.RegistrationManager;
            Debug.Assert( registrationManager != null );

            String serviceName = String.Format("NintendoWare {0} Preview Shell Extension", PluginName);

            Guid guid = Guid.Empty;
            if (!Define.IDs[ShellExtensionService.PreviewService].TryGetValue(PluginName?.ToUpper(), out guid))
            {
                return false;
            }

            String GUIDStr = "";

            try
            {
                RegistryKey shellExtKey = progIDKey.CreateSubKey( "ShellEx" );
                if (bUse)
                {
                    RegistryKey serviceHandlerKey = shellExtKey.CreateSubKey(previewHandler);

                    GUIDStr = "{" + guid.ToString().ToUpper() + "}";
                    serviceHandlerKey.SetValue(null, GUIDStr);
                }
                else
                {
                    shellExtKey.DeleteSubKeyTree(previewHandler, false);
                }
            }

            catch (Exception e)
            {
                MainApp.Instance.WriteToLog( e.Message );
                return false;
            }

            return true;
        }
        #endregion


        /// <summary>
        /// dll 毎のサービスの設定の保存
        /// </summary>
        public bool SavePluginToRegistry(ShellPlugin plugin, bool use, Guid guid, string description, ShellExtensionService service)
        {
            using (RegistryKey approvedListKey = Registry.LocalMachine.OpenSubKey("SOFTWARE\\Microsoft\\Windows\\CurrentVersion\\Shell Extensions\\Approved", true))
            {
                if (use)
                {
                    AddToApprovedList(approvedListKey, guid, description);
                }
                else
                {
                    RemoveToApprovedList(approvedListKey, guid);
                }
            }

            if (!SavePluginToCLSID(guid, plugin, description, use))
            {
                return false;
            }

            if (service == ShellExtensionService.PreviewService)
            {
                // プレビューは追加の設定が必要
                SaveAdditionalRegistryPreviewPlugin(plugin, use, guid, description);
            }

            return true;
        }

        /// <summary>
        /// dll 毎のプレビュー関連の設定の保存
        /// </summary>
        private bool SaveAdditionalRegistryPreviewPlugin(ShellPlugin plugin, bool use, Guid guid, string description)
        {
            var GUIDStr = "{" + guid.ToString().ToUpper() + "}";
            try
            {
                using (var CLSIDKey = Registry.ClassesRoot.OpenSubKey("CLSID", true))
                {
                    using (var GUIDKey = CLSIDKey.CreateSubKey(GUIDStr))
                    {
                        if (use)
                        {
                            GUIDKey.SetValue("AppId", appIDHandler);
                        }
                        else
                        {
                            GUIDKey.DeleteValue("AppId", false);
                        }
                    }
                }
            }
            catch (Exception e)
            {
                MainApp.Instance.WriteToLog(e.Message);
                return false;
            }

            try
            {
                using (var previewHandlersKey = Registry.LocalMachine.OpenSubKey("SOFTWARE\\Microsoft\\Windows\\CurrentVersion\\PreviewHandlers", true))
                {
                    previewHandlersKey.SetValue(GUIDStr, description);
                }
            }
            catch (Exception e)
            {
                MainApp.Instance.WriteToLog(e.Message);
                return false;
            }

            return true;
        }



        #region ColumnProvider Service
        //----------------------------------------------------------------------
        /// <summary>
        /// Load registry for ColumnProvider Service
        /// </summary>
        //----------------------------------------------------------------------
        private bool LoadRegistryColumnProviderService(ShellPlugin plugin, RegistryKey progIDKey,
                                                       ref bool bOut)
        {
            try
            {
                RegistryKey classRootkey = Registry.ClassesRoot;
                RegistryKey folderKey = classRootkey.OpenSubKey("Folder");
                RegistryKey shellExtKey = folderKey.OpenSubKey("ShellEx", true);
                RegistryKey serviceHandlerKey = shellExtKey.OpenSubKey("ColumnHandlers");

                Guid guid = Guid.Empty;
                if (String.Compare(PluginName, "FTX", true) == 0)
                    guid = Define.CLSID_ShellExtensionColumnProviderFTX;
                else if (String.Compare(PluginName, "TGA", true) == 0)
                    guid = Define.CLSID_ShellExtensionColumnProviderTGA;
                else if (String.Compare(PluginName, "PSD", true) == 0)
                    guid = Define.CLSID_ShellExtensionColumnProviderPSD;
                else
                    return false;

                String GUIDStr = "{" + guid.ToString().ToUpper() + "}";
                RegistryKey thisHandlerKey = serviceHandlerKey.OpenSubKey(GUIDStr);

                bOut = (thisHandlerKey != null);
            }
            catch (Exception e)
            {
                MainApp.Instance.WriteToLog(e.Message);
                return false;
            }

            return true;
        }

        //----------------------------------------------------------------------
        /// <summary>
        /// Save registry for ColumnProvider Service
        /// </summary>
        //----------------------------------------------------------------------
        private bool SaveRegistryColumnProviderService( ShellPlugin plugin, RegistryKey progIDKey,
                                                        bool bUse )
        {
            MainApp app = MainApp.Instance;
            Debug.Assert( app != null );

            RegistrationManager registrationManager = app.RegistrationManager;
            Debug.Assert( registrationManager != null );

            String serviceName = String.Format("NintendoWare {0} ColumnProvider Shell Extension", PluginName);

            Guid guid = Guid.Empty;
            if (String.Compare(PluginName, "FTX", true) == 0)
                guid = Define.CLSID_ShellExtensionColumnProviderFTX;
            else if (String.Compare(PluginName, "TGA", true) == 0)
                guid = Define.CLSID_ShellExtensionColumnProviderTGA;
            else if (String.Compare(PluginName, "PSD", true) == 0)
                guid = Define.CLSID_ShellExtensionColumnProviderPSD;
            else
                return false;

            if (SavePluginToCLSID(guid, plugin, serviceName, bUse) == false)
            {
                return false;
            }

            try
            {
                RegistryKey classRootkey = Registry.ClassesRoot;
                RegistryKey folderKey    = classRootkey.OpenSubKey( "Folder" );
                RegistryKey shellExtKey  = folderKey.OpenSubKey( "ShellEx", true );
                RegistryKey serviceHandlerkey = shellExtKey.CreateSubKey("ColumnHandlers");
                if (serviceHandlerkey == null)
                    return false;

                String GUIDStr = "{" + guid.ToString().ToUpper() + "}";

                if (bUse)
                {
                    RegistryKey thisHandlerKey = serviceHandlerkey.CreateSubKey(GUIDStr);
                    thisHandlerKey.SetValue(null, serviceName);
                }
                else
                {
                    serviceHandlerkey.DeleteSubKeyTree(GUIDStr, false);
                }
            }

            catch (Exception e)
            {
                MainApp.Instance.WriteToLog( e.Message );
                return false;
            }

            return true;
        }
        #endregion

        #region PropertyStore Service
        //----------------------------------------------------------------------
        /// <summary>
        /// Load registry for PropertyStore Service
        /// </summary>
        //----------------------------------------------------------------------
        private bool LoadRegistryPropertyStoreService(ShellPlugin plugin, RegistryKey progIDKey,
                                                      ref bool bOut)
        {
            try
            {
                RegistryKey localMachineKey = Registry.LocalMachine;
                RegistryKey propStoreListKey = localMachineKey.OpenSubKey("SOFTWARE\\Microsoft\\Windows\\CurrentVersion\\PropertySystem\\PropertyHandlers");
                String extKeyStr = ("." + ExtensionName).ToLower();
                RegistryKey thisPropStoreKey = propStoreListKey.OpenSubKey(extKeyStr);

                bOut = (thisPropStoreKey != null);
            }
            catch (Exception e)
            {
                MainApp.Instance.WriteToLog(e.Message);
                return false;
            }

            return true;
        }

        //----------------------------------------------------------------------
        /// <summary>
        /// Save registry for PropertyStore Service
        /// </summary>
        //----------------------------------------------------------------------
        private bool SaveRegistryPropertyStoreService( ShellPlugin plugin, RegistryKey progIDKey,
                                                       bool bUse )
        {
            MainApp app = MainApp.Instance;
            Debug.Assert( app != null );

            RegistrationManager registrationManager = app.RegistrationManager;
            Debug.Assert( registrationManager != null );

            String serviceName = String.Format("NintendoWare {0} PropertyStore Shell Extension", PluginName);

            Guid guid = Guid.Empty;
            if (String.Compare(PluginName, "FTX", true) == 0)
                guid = Define.CLSID_ShellExtensionPropertyStoreFTX;
            else if (String.Compare(PluginName, "TGA", true) == 0)
                guid = Define.CLSID_ShellExtensionPropertyStoreTGA;
            else if (String.Compare(PluginName, "PSD", true) == 0)
                guid = Define.CLSID_ShellExtensionPropertyStorePSD;
            else
                return false;

            if (SavePluginToCLSID(guid, plugin, serviceName, bUse) == false)
            {
                return false;
            }

            try
            {
                // Remove misspelled value
                progIDKey.DeleteValue("FullDetail", false); // <- false to make it not throw an exception if the node does not exist

                String detailStr = "prop:";
                detailStr += "Nintendo.NWTexture.Format;";
                detailStr += "Nintendo.NWTexture.Width;";
                detailStr += "Nintendo.NWTexture.Height;";
                detailStr += "Nintendo.NWTexture.Mipmap;";
                detailStr += "Nintendo.NWTexture.DataSize;";
                detailStr += "Nintendo.NWTexture.CompSel;";
                detailStr += "Nintendo.NWTexture.Linear;";
                detailStr += "Nintendo.NWTexture.Hint;";
                detailStr += "Nintendo.NWTexture.OutputPath;";
                detailStr += "Nintendo.NWTexture.TGAOutputPath;";
                detailStr += "Nintendo.NWTexture.Comment;";
                detailStr += "Nintendo.NWTexture.LayerCompsCount";



                // Add to property store list
                RegistryKey localMachineKey  = Registry.LocalMachine;
                RegistryKey propStoreListKey = localMachineKey.OpenSubKey("SOFTWARE\\Microsoft\\Windows\\CurrentVersion\\PropertySystem\\PropertyHandlers", true);

                String extKeyStr = ("." + ExtensionName).ToLower();
                if (bUse)
                {
                    RegistryKey thisPropStoreKey = propStoreListKey.CreateSubKey(extKeyStr);

                    String GUIDStr = "{" + guid.ToString().ToUpper() + "}";
                    thisPropStoreKey.SetValue(null, GUIDStr);

                    progIDKey.SetValue("FullDetails", detailStr);
                }
                else
                {
                    propStoreListKey.DeleteSubKeyTree(extKeyStr, false);

                    progIDKey.DeleteValue("FullDetails", false);
                }

                // If we are in 64 bit OS, we must register to Wow64 for 32 bit app running in 64 bit OS
                if (Environment.Is64BitOperatingSystem)
                {
                    RegistryKey localMachineKey32 = RegistryKey.OpenBaseKey(Microsoft.Win32.RegistryHive.LocalMachine,
                                                                            RegistryView.Registry32);

                    RegistryKey propStoreListKey32 = localMachineKey32.OpenSubKey("SOFTWARE\\Microsoft\\Windows\\CurrentVersion\\PropertySystem\\PropertyHandlers", true);
                    if (propStoreListKey32 != null)
                    {
                        String extKeyStr32 = ("." + ExtensionName).ToLower();
                        if (bUse)
                        {
                            RegistryKey thisPropStoreKey32 = propStoreListKey32.CreateSubKey(extKeyStr32);

                            String GUIDStr32 = "{" + guid.ToString().ToUpper() + "}";
                            thisPropStoreKey32.SetValue(null, GUIDStr32);
                        }
                        else
                        {
                            propStoreListKey32.DeleteSubKeyTree(extKeyStr32, false);
                        }
                    }
                }
            }

            catch (Exception e)
            {
                MainApp.Instance.WriteToLog( e.Message );
                return false;
            }

            return true;
        }
        #endregion

        #region ShellCommand Service
        //----------------------------------------------------------------------
        /// <summary>
        /// Load registry for ShellCommand Service
        /// </summary>
        //----------------------------------------------------------------------
        private bool LoadRegistryShellCommandService(ShellPlugin plugin, RegistryKey progIDKey,
                                                     ref string command)
        {
            try
            {
                RegistryKey shellKey = progIDKey.OpenSubKey("Shell");
                if (null == shellKey)
                    return true;

                RegistryKey openKey = shellKey.OpenSubKey("open");
                if (null == openKey)
                    return true;

                RegistryKey commandKey = openKey.OpenSubKey("command");
                if (null == commandKey)
                    return true;

                command = (string)commandKey.GetValue(null);
                if (command.EndsWith(" \"%1\"")) // Expected to be true
                {
                    command = (command.Substring(0, command.Length - "\"%1\"".Length)).Trim();
                }
            }

            catch (Exception e)
            {
                MainApp.Instance.WriteToLog(e.Message);
                return false;
            }

            return true;
        }

        //----------------------------------------------------------------------
        /// <summary>
        /// Select ShellCommand
        /// </summary>
        //----------------------------------------------------------------------
        private bool SelectShellCommand(ShellPlugin plugin, RegistrationManager registrationManager)
        {
            RegistryKey classRootkey = Registry.ClassesRoot;

            // Add extension handling registration
            String extKeyStr = ("." + ExtensionName).ToLower();
            RegistryKey extKey = classRootkey.OpenSubKey(extKeyStr);

            // Set program ID to this extension
            String progIDStr = extKey?.GetValue(null) as String;

            if (string.IsNullOrEmpty(progIDStr))
            {
                // 簡易アンインストール後は以前の設定が残っている。
                // 以前の設定が残っているならそれを使う。
                progIDStr = "NintendoWare." + ExtensionName + ".1";
            }

            if (progIDStr.Equals("NintendoWare." + ExtensionName + ".1", StringComparison.CurrentCultureIgnoreCase) == false)
            {
                // OS側のファイル関連付け機能を使用している場合は、カスタムでパスを選択する
                // 場合によってUserChoiceが存在しないことがある。progIDStrからPathを引っ張る。
                if ((m_prevAppInfo.UserChoiceCommand != null) && (m_prevAppInfo.UserChoiceCommand.Length > 0))
                {
                    m_currentStatus.ShellCommandName = Properties.Resources.TEXT_CUSTOM;
                    m_currentStatus.ShellCommandPath = m_prevAppInfo.UserChoiceApplicationPath;

                    return false;
                }
            }
            // Open program ID key
            RegistryKey progIDkey = classRootkey.OpenSubKey(progIDStr);
            if (progIDkey == null)
                return false;

            String shellCommand = null;
            LoadRegistryShellCommandService(plugin, progIDkey, ref shellCommand);

            // OS側のファイル関連付け使用している場合
            if (m_prevAppInfo.UserChoiceCommand.Length > 0)
                shellCommand = "";

            if ((shellCommand != null) && (shellCommand.Length > 0))
            {
                int regAppInfo = registrationManager.FindRegisterAppFromPath(shellCommand);
                if (regAppInfo >= 0)
                {
                    RegistrationManager.AppInfo appInfo = registrationManager.GetRegisteredApplicationInfo(regAppInfo);
                    m_currentStatus.ShellCommandName = appInfo.Name;
                    m_currentStatus.ShellCommandPath = appInfo.Path;
                }
                else
                {
                    m_currentStatus.ShellCommandName = Properties.Resources.TEXT_CUSTOM;
                    m_currentStatus.ShellCommandPath = shellCommand;
                }
            }
            else
            {
                if ((m_prevAppInfo.UserChoiceCommand != null) && (m_prevAppInfo.UserChoiceCommand.Length > 0))
                {
                    m_currentStatus.ShellCommandName = Properties.Resources.TEXT_CUSTOM;
                    m_currentStatus.ShellCommandPath = m_prevAppInfo.UserChoiceApplicationPath;
                }
                else
                {
                    m_currentStatus.ShellCommandName = Properties.Resources.TEXT_NONE;
                    m_currentStatus.ShellCommandPath = "";
                }
            }
            return true;
        }

        //----------------------------------------------------------------------
        /// <summary>
        /// Save registry for ShellCommand Service
        /// </summary>
        //----------------------------------------------------------------------
        private bool SaveRegistryShellCommandService(ShellPlugin plugin, RegistryKey progIDKey,
                                                     string command)
        {
            MainApp app = MainApp.Instance;
            Debug.Assert(app != null);

            RegistrationManager registrationManager = app.RegistrationManager;
            Debug.Assert(registrationManager != null);

            if ((command != null) &&
                (command.Length > 0) &&
                (String.Compare(command, Properties.Resources.TEXT_NONE, true) != 0))
            {
                try
                {
                    RegistryKey shellKey   = progIDKey.CreateSubKey("Shell");
                    RegistryKey openKey    = shellKey.CreateSubKey("open");
                    RegistryKey commandKey = openKey.CreateSubKey("command");
                    String appCmd = command + " \"%1\"";
                    commandKey.SetValue(null, appCmd);
                }

                catch (Exception e)
                {
                    MainApp.Instance.WriteToLog(e.Message);
                    return false;
                }
            }
            else
            {
                try
                {
                    progIDKey.DeleteSubKeyTree("Shell");
                }

                catch (System.Exception e)
                {
                    if (e.Message == "") {}
                    return false;
                }
            }

            return true;
        }
        #endregion

        #region ContextMenu Service

        //----------------------------------------------------------------------
        /// <summary>
        /// Load registry for ContextMenu Service
        /// </summary>
        //----------------------------------------------------------------------
        private bool LoadRegistryContextMenuService(ShellPlugin plugin, RegistryKey progIDKey, ref bool bOut)
        {
            try
            {
                RegistryKey shellExtKey = progIDKey.OpenSubKey("ShellEx");
                if (shellExtKey == null)
                    return false;

                RegistryKey serviceHandlerKey = shellExtKey.OpenSubKey("ContextMenuHandlers");
                if (serviceHandlerKey == null)
                    return false;

                RegistryKey commandHandlerKey = serviceHandlerKey.OpenSubKey("Command");
                if (commandHandlerKey == null)
                    return false;

                bOut = (commandHandlerKey != null);
            }
            catch (Exception e)
            {
                MainApp.Instance.WriteToLog(e.Message);
                return false;
            }

            return true;
        }

        //----------------------------------------------------------------------
        /// <summary>
        /// Save registry for ContextMenu Service
        /// </summary>
        //----------------------------------------------------------------------
        private bool SaveRegistryContextMenuService(ShellPlugin plugin, RegistryKey progIDKey, bool bUse)
        {
            MainApp app = MainApp.Instance;
            Debug.Assert(app != null);

            RegistrationManager registrationManager = app.RegistrationManager;
            Debug.Assert(registrationManager != null);

            String serviceName = String.Format("NintendoWare {0} ContextMenu Shell Extension", PluginName);

            Guid guid = Guid.Empty;
            if (String.Compare(PluginName, "FTX", true) == 0)
                guid = Define.CLSID_ShellExtensionContextMenuFTX;
            else
                return false;

            // ContextMenu機能は非対応のため、CLSIDの書き込みは行わない。
            // もしContextMenu機能を復活する場合は、以下のコメントを外す
            //if (SavePluginToCLSID(guid, plugin, serviceName) == false)
            //{
            //    return false;
            //}

            String GUIDStr = "";
            GUIDStr = "{" + guid.ToString().ToUpper() + "}";

            try
            {
                RegistryKey shellExtKey = progIDKey.CreateSubKey("ShellEx");
                RegistryKey serviceHandlerKey = shellExtKey.CreateSubKey("ContextMenuHandlers");
                if (bUse)
                {
                    RegistryKey commandHandlerKey = serviceHandlerKey.CreateSubKey("Command");
                    //GUIDStr = "{" + guid.ToString().ToUpper() + "}";
                    commandHandlerKey.SetValue(null, GUIDStr);

                }
                else
                {
                    shellExtKey.DeleteSubKeyTree("ContextMenuHandlers", false);
                }
            }

            catch (Exception e)
            {
                MainApp.Instance.WriteToLog(e.Message);
                return false;
            }

            // For Preview handler, we have to add extra below GUID key
            try
            {
                RegistryKey classRootkey = Registry.ClassesRoot;
                RegistryKey CLSIDkey = classRootkey.OpenSubKey("CLSID", true);
                RegistryKey GUIDKey = CLSIDkey.CreateSubKey(GUIDStr);
                if (bUse)
                {
                    GUIDKey.SetValue("AppId", appIDHandler);
                }
                else
                {
                    GUIDKey.DeleteValue("AppId", false);
                }
            }

            catch (Exception e)
            {
                MainApp.Instance.WriteToLog(e.Message);
                return false;
            }

            /* it seems this does not exist for context menus
            // Add to global preview list
            try
            {
                RegistryKey localMachineKey    = Registry.LocalMachine;
                RegistryKey previewHandlersKey = localMachineKey.OpenSubKey("SOFTWARE\\Microsoft\\Windows\\CurrentVersion\\PreviewHandlers", true);
                if (bUse)
                {
                    String valStr = String.Format("NintendoWare {0} Preview Shell Extension", PluginName);
                    previewHandlersKey.SetValue(GUIDStr, valStr);
                }
            }

            catch (Exception e)
            {
                MainApp.Instance.WriteToLog(e.Message);
                return false;
            }
            */

            return true;
        }
        #endregion

        #endregion

        #region Prev registered app for restore
        private PrevRegisteredAppInfo m_prevAppInfo = new PrevRegisteredAppInfo();

        //----------------------------------------------------------------------
        //
        //----------------------------------------------------------------------
        public PrevRegisteredAppInfo GetPrevRegsiteredAppInfo()
        {
            return m_prevAppInfo;
        }

        private string LoadApplicationPath(string command, RegistryKey registryKey)
        {
            if (registryKey == null)
                return "";

            String userClassesKeyPath = "SOFTWARE\\Classes\\";
            userClassesKeyPath += command + "\\Shell\\Open\\Command";

            RegistryKey applicationsKey = registryKey.OpenSubKey(userClassesKeyPath, false);
            if (applicationsKey == null)
                return "";

            string applicationPath = "";
            applicationPath = (string)applicationsKey.GetValue(null);
            if (applicationPath.EndsWith(" \"%1\"")) // Expected to be true
            {
                applicationPath = (applicationPath.Substring(0, applicationPath.Length - "\"%1\"".Length)).Trim();
            }
            return applicationPath;
        }



        //----------------------------------------------------------------------
        /// <summary>
        /// Load prev registered app info
        /// </summary>
        //----------------------------------------------------------------------
        private bool LoadPrevRegisteredAppInfo()
        {
            MainApp app = MainApp.Instance;
            Debug.Assert(app != null);

            RegistrationManager registrationManager = app.RegistrationManager;
            Debug.Assert(registrationManager != null);

            RegistryKey classRootkey = Registry.ClassesRoot;
            if (classRootkey == null)
                return false;

            // Add extension handling registration
            String extKeyStr = ("." + ExtensionName).ToLower();
            RegistryKey extKey = classRootkey.OpenSubKey(extKeyStr);
            if (extKey == null)
                return false;

            // Set program ID to this extension
            String progIDStr = extKey.GetValue(null) as String;
            if (progIDStr != null)
            {
                String thisProgId = "NintendoWare." + ExtensionName + ".1";
                if (String.Compare(thisProgId,progIDStr,StringComparison.CurrentCultureIgnoreCase)!=0)
                    m_prevAppInfo.ShellProgID = progIDStr;
            }

            // Get user choice
            RegistryKey curUserKey = Registry.CurrentUser;
            if (curUserKey == null)
                return false;

            String userChoiceKeyPath = "SOFTWARE\\Microsoft\\Windows\\CurrentVersion\\Explorer\\FileExts\\";
            userChoiceKeyPath += "." + ExtensionName + "\\UserChoice";

            RegistryKey userChoiceKey = curUserKey.OpenSubKey(userChoiceKeyPath, false);
            if (userChoiceKey == null)
                return false;

            String command = userChoiceKey.GetValue("Progid") as String;
            if ((command != null) && (command.Length > 0))
            {
                m_prevAppInfo.UserChoiceCommand = command;
                m_prevAppInfo.UserChoiceApplicationPath = "";

                // commandパスの頭にApplicationsの時は、別の場所からapplicationパスを取得する
                if (String.Compare(command, 0, "Applications", 0, 13, false) != 0)
                {
                    // まずはHKCUに登録されているアプリケーションパスを取得する
                    m_prevAppInfo.UserChoiceApplicationPath = LoadApplicationPath(command, Registry.CurrentUser);

                    if (m_prevAppInfo.UserChoiceApplicationPath == "")
                    {
                        // 次にHKLMに登録されているアプリケーションパスを取得する
                        m_prevAppInfo.UserChoiceApplicationPath = LoadApplicationPath(command, Registry.LocalMachine);
                    }

                }

            }

            // Windows8より追加された
            String hashKey = userChoiceKey.GetValue("Hash") as String;
            if ((hashKey != null) && (hashKey.Length > 0))
            {
                m_prevAppInfo.UserChoiceHash = hashKey;
            }

            return true;
        }

        //----------------------------------------------------------------------
        /// <summary>
        /// Save prev registered app info
        /// </summary>
        //----------------------------------------------------------------------
        private bool SavePrevRegisteredAppInfo()
        {
            RegistryKey classRootkey = Registry.ClassesRoot;
            if (classRootkey == null)
                return false;

            // Open program ID key
            String      progIDStr = "NintendoWare." + ExtensionName.ToUpper() + ".1";
            RegistryKey progIDkey = classRootkey.OpenSubKey(progIDStr, true);
            if (progIDkey == null)
                return false;

            // PrevAppInfo
            RegistryKey prevAppInfo = progIDkey.CreateSubKey("PrevAppInfo");
            if (prevAppInfo==null)
                return false;

            if ( (m_prevAppInfo.ShellProgID!=null) &&
                 (m_prevAppInfo.ShellProgID.Length > 0))
            {
                prevAppInfo.SetValue("ShellProgID", m_prevAppInfo.ShellProgID);
            }

            if ( (m_prevAppInfo.UserChoiceCommand!=null) &&
                 (m_prevAppInfo.UserChoiceCommand.Length > 0))
            {
                prevAppInfo.SetValue("UserChoice", m_prevAppInfo.UserChoiceCommand);
            }


            if ((m_prevAppInfo.UserChoiceHash != null) &&
                 (m_prevAppInfo.UserChoiceHash.Length > 0))
            {
                prevAppInfo.SetValue("UserChoiceHash", m_prevAppInfo.UserChoiceHash);
            }

            return true;
        }

        //----------------------------------------------------------------------
        /// <summary>
        /// Edit FileExt OpenWitdhProgid Registry.
        /// </summary>
        //----------------------------------------------------------------------
        private bool EditFileExtOpenWithProgid()
        {
            // Get user choice
            RegistryKey curUserKey = Registry.CurrentUser;
            if (curUserKey == null)
                return false;

            String userOpenWithProgidPath = "SOFTWARE\\Microsoft\\Windows\\CurrentVersion\\Explorer\\FileExts\\";
            userOpenWithProgidPath += "." + ExtensionName + "\\OpenWithProgids";

            RegistryKey userOpenWithProgidKey = curUserKey.OpenSubKey(userOpenWithProgidPath, true);
            if (userOpenWithProgidKey == null)
                return false;

            List<String> OpenWithProgidList = new List<String>();

            String[] valueNames = userOpenWithProgidKey.GetValueNames();
            String progIDStr = "NintendoWare." + ExtensionName + ".1";

            int i;
            for (i = 0; i < valueNames.Count(); i++)
            {
                if (!String.Equals(valueNames[i], progIDStr, StringComparison.CurrentCultureIgnoreCase))
                {
                    OpenWithProgidList.Add(valueNames[i]);
                }
            }

            if (OpenWithProgidList.Count > 0)
            {
                for (i = 0; i < OpenWithProgidList.Count; i++)
                {
                    userOpenWithProgidKey.DeleteValue(OpenWithProgidList[i], false);
                }
            }
            userOpenWithProgidKey.Close();

            return true;
        }

        #endregion
    }
}

