﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.IO;
using System.Windows.Forms;
using System.Diagnostics;
using System.Runtime.InteropServices; // Win32 APIを呼び出すために必要

namespace NintendoWare.ExtensionManager
{
    public enum ReturnCode : int
    {
        ErrorFailedToWriteLog = -2,
        Error = -1,
        Success = 0,
        Cancel = 1
    };

    //==========================================================================
    /// <summary>
    /// Main Program
    /// </summary>
    //==========================================================================
    public class Program : Microsoft.VisualBasic.ApplicationServices.WindowsFormsApplicationBase
    {

        /// <summary>
        /// Check .NET 4.5 is supported
        /// </summary>
        public static bool IsNet45OrNewer()
        {
            // Class "ReflectionContext" exists from .NET 4.5 onwards.
            return Type.GetType("System.Reflection.ReflectionContext", false) != null;
        }

        /// <summary>
        /// アプリケーションのメイン エントリ ポイントです。
        /// </summary>
        [STAThread]
        static void Main(string[] args)
        {

            if (!Environment.Is64BitOperatingSystem)
            {
                string errorMsg = NintendoWare.ExtensionManager.Properties.Resources.ERROR_32BITOS;

                MessageBox.Show(errorMsg, "ThumbnailExplorerExtensions Error", MessageBoxButtons.OK);
                return;
            }


            if (IsNet45OrNewer() == false)
            {
                string errorMsg = NintendoWare.ExtensionManager.Properties.Resources.ERROR_NET450;

                MessageBox.Show(errorMsg, "ThumbnailExplorerExtensions Error", MessageBoxButtons.OK);
                System.Environment.Exit((int)ReturnCode.Cancel);
                return;
            }

            Application.EnableVisualStyles();
            Application.SetCompatibleTextRenderingDefault(false);

            MainApp app = new MainApp();
            ReturnCode rtCode = ReturnCode.Success;
            try
            {

                Program program = new Program();
                if (program.ArgParse(args) == false)
                {
                    System.Environment.Exit((int)ReturnCode.Error);
                    return;
                }

                app.Init(program.UseUnInstallSetup, program.UseEasy || program.CheckAll, program.RemoveUnspecifiedAdditionalExtensionsRegistry, program.AdditionalExtensions);

                if (program.UseEasy)
                {
                    IWin32Window ownerWindow = null;

                    // 環境変数 NW_EM_SETUP_HWND にウィンドウハンドルが設定されていれば、そのウィンドウをダイアログのオーナーとして設定する
                    int hwnd = 0;
                    if (int.TryParse(Environment.GetEnvironmentVariable("NW_EM_SETUP_HWND"), out hwnd) && hwnd != 0)
                    {
                        ownerWindow = new Win32Window(hwnd);
                    }

                    // 簡易インストール/アンインストール時はGUIを表示せず実行
                    if (program.UseUnInstallSetup)
                    {
                        rtCode = app.SetupRegistraition(program.UseEasy, program.IsSilent, MainApp.SetupType.Uninstall, ownerWindow);
                    }
                    else
                    {
                        rtCode = app.SetupRegistraition(program.UseEasy, program.IsSilent, MainApp.SetupType.Install, ownerWindow);
                    }
                }
                else
                {
                    // GUI表示
                    program.Run(args);
                }

                app.Deinit();
            }
            catch (System.Exception e)
            {
                rtCode = ReturnCode.Error;
                try
                {
                    app.WriteToLog(e.ToString());
                }
                catch
                {
                    // WriteToLog に失敗したことが分かるようにエラーコードを設定
                    rtCode = ReturnCode.ErrorFailedToWriteLog;
                }
            }
            System.Environment.Exit((int)rtCode);
            return;
        }

        //----------------------------------------------------------------------
        /// <summary>
        /// Constructor
        /// </summary>
        //----------------------------------------------------------------------
        public Program() :
            base(Microsoft.VisualBasic.ApplicationServices.AuthenticationMode.Windows)
        {
            EnableVisualStyles = true;
        }

        //----------------------------------------------------------------------
        /// <summary>
        /// Override to create own app form
        /// </summary>
        //----------------------------------------------------------------------
        protected override void OnCreateMainForm()
        {
            MainApp app = MainApp.Instance;
            Debug.Assert(app != null);

            app.InitGUIApp();

            MainForm = app.UIManager.MainForm;
        }

        //----------------------------------------------------------------------
        /// <summary>
        /// Create splash screen
        /// </summary>
        //----------------------------------------------------------------------
        protected override void OnCreateSplashScreen()
        {
            //SplashScreen = new UI.SplashScreenPanel();
        }

        protected enum ParseMode
        {
            Normal,
            AdditionalArgs,
        }
        //----------------------------------------------------------------------
        /// <summary>
        /// Parse argument
        /// </summary>
        //----------------------------------------------------------------------
        protected bool ArgParse(string[] args)
        {
            UseEasy = false;                // 簡易インストール/アンインストール
            UseUnInstallSetup = false;      // アンインストール実行
            IsSilent = false;               // ダイアログ非表示


            var mode = ParseMode.Normal;
            bool showHelp = false;
            var invalidFileNameChars = new HashSet<char>((Path.GetInvalidFileNameChars()));
            foreach (string arg in args)
            {
                switch (mode)
                {
                    case ParseMode.Normal:
                        if (string.Compare(arg, "-u") == 0)
                        {
                            UseUnInstallSetup = true;
                        }
                        else if (string.Compare(arg, "-e") == 0)
                        {
                            UseEasy = true;
                        }
                        else if (string.Compare(arg, "-v") == 0)
                        {
                            // バージョン表示
                            System.Diagnostics.FileVersionInfo ver = System.Diagnostics.FileVersionInfo.GetVersionInfo(
                                                                        System.Reflection.Assembly.GetExecutingAssembly().Location);
                            Console.Out.WriteLine("Version number: " + ver.FileVersion);
                            return false;
                        }
                        else if (string.Compare(arg, "-s") == 0)
                        {
                            IsSilent = true;
                        }
                        else if (string.Compare(arg, "-c") == 0)
                        {
                            CheckAll = true;
                        }
                        else if (string.Compare(arg, "--custom") == 0)
                        {
                            mode = ParseMode.AdditionalArgs;
                        }
                        else if (string.Compare(arg, "--remove-custom") == 0)
                        {
                            // 指定されていない拡張子を削除する
                            RemoveUnspecifiedAdditionalExtensionsRegistry = true;
                        }
                        else
                        {
                            showHelp = true;
                        }
                        break;
                    case ParseMode.AdditionalArgs:
                        var splitted = arg.Split(";".ToArray(), StringSplitOptions.RemoveEmptyEntries);
                        foreach (var item in splitted)
                        {
                            if (item.All(x => !invalidFileNameChars.Contains(x)) && !arg.StartsWith("-") && arg != ".")
                            {
                                var ext = item.ToUpper();
                                if (ext.StartsWith("."))
                                {
                                    ext = ext.Substring(1);
                                }
                                AdditionalExtensions.Add(ext);
                            }
                            else
                            {
                                showHelp = true;
                            }
                        }
                        mode = ParseMode.Normal;
                        break;
                }
            }

            if (mode != ParseMode.Normal || showHelp)
            {
                // ヘルプ表示
                string helpMsg = NintendoWare.ExtensionManager.Properties.Resources.Help;
                Console.Out.WriteLine(helpMsg, Path.GetFileName(System.Reflection.Assembly.GetExecutingAssembly().Location));
                return false;
            }

            return true;
        }

        //------------------------------------------------------------------
        /// <summary>
        /// Use Easy Install/Uninstall
        /// </summary>
        //------------------------------------------------------------------
        public bool UseEasy
        {
            set;
            get;
        }

        //------------------------------------------------------    ------------
        /// <summary>
        /// Use UnInstall Setup
        /// </summary>
        //------------------------------------------------------------------
        public bool UseUnInstallSetup
        {
            set;
            get;
        }

        //------------------------------------------------------------------
        /// <summary>
        /// Is Silent Mode
        /// </summary>
        //------------------------------------------------------------------
        public bool IsSilent
        {
            set;
            get;
        }

        public bool CheckAll
        {
            set;
            get;
        }

        public bool RemoveUnspecifiedAdditionalExtensionsRegistry { get; set; } = false;
        public List<string> AdditionalExtensions { get; set; } = new List<string>();

        private class Win32Window : IWin32Window
        {
            public Win32Window(int handle)
            {
                Handle = new IntPtr(handle);
            }

            public IntPtr Handle { get; private set; }
        }

    }
}
