﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Diagnostics;
using System.ComponentModel;

namespace NintendoWare.ExtensionManager
{
    //==========================================================================
    /// <summary>
    /// Config manager
    /// </summary>
    //==========================================================================
    public class SessionConfig : IDisposable
    {
        //----------------------------------------------------------------------
        /// <summary>
        /// Constructor
        /// </summary>
        //----------------------------------------------------------------------
        public SessionConfig()
        {
            m_fileName = "session_config.xml";
        }

        //----------------------------------------------------------------------
        /// <summary>
        /// Dispose
        /// </summary>
        //----------------------------------------------------------------------
        virtual public void Dispose()
        {
        }

        #region FileName
        //----------------------------------------------------------------------
        /// <summary>
        /// Config filename
        /// </summary>
        //----------------------------------------------------------------------
        public String FileName
        {
            get { return m_fileName; }
        }

        protected String m_fileName = "";
        #endregion

        #region Localization
        //----------------------------------------------------------------------
        /// <summary>
        /// Localization Config
        /// </summary>
        //----------------------------------------------------------------------
        public class LocalizationConfig
        {
            public LocalizationConfig()
            {
                UICultureName = "";
            }

            public String UICultureName
            {
                set;
                get;
            }
        }

        public SessionConfig.LocalizationConfig AppLocalizationConfig
        {
            get { return m_localizationConfig; }
        }

        protected SessionConfig.LocalizationConfig m_localizationConfig = new SessionConfig.LocalizationConfig();
        #endregion

        #region MainForm
        //----------------------------------------------------------------------
        /// <summary>
        /// MainFrame Config
        /// </summary>
        //----------------------------------------------------------------------
        public Utility.WindowConfig MainFormConfig
        {
            get { return m_mainFrameConfig; }
        }

        protected Utility.WindowConfig m_mainFrameConfig = new Utility.WindowConfig();

        //----------------------------------------------------------------------
        /// <summary>
        /// Loading mainframe section
        /// Override this to customize
        /// </summary>
        ///
        /// <param name="element">XML Element</param>
        ///
        /// <returns>True if success</returns>
        //----------------------------------------------------------------------
        protected bool LoadMainFormSection(System.Xml.XmlElement element)
        {
            Utility.LoadWindowConfig(element, m_mainFrameConfig);

            return true;
        }

        //----------------------------------------------------------------------
        /// <summary>
        /// Saving mainframe section
        /// Override this to customize
        /// </summary>
        ///
        /// <param name="element">XML Element</param>
        ///
        /// <returns>True if success</returns>
        //----------------------------------------------------------------------
        protected bool SaveMainFormSection(System.Xml.XmlElement element)
        {
            MainApp app = MainApp.Instance;
            if (app == null)
                return false;

            if (app.UIManager == null)
                return false;

            MainForm mainForm = app.UIManager.MainForm;
            if (mainForm == null)
                return false;

            Utility.WindowConfig winConfig = new Utility.WindowConfig();
            Utility.GetWindowConfig(mainForm, winConfig);
            Utility.SaveWindowConfig(element, winConfig);

            return true;
        }
        #endregion

        #region Load
        //----------------------------------------------------------------------
        /// <summary>
        /// Load Config
        /// </summary>
        ///
        /// <returns>True on success</returns>
        //----------------------------------------------------------------------
        public bool Load(String configDir)
        {
            System.Xml.XmlDocument doc = new System.Xml.XmlDocument();

            String pathName = configDir + "/" + m_fileName;

            try
            {
                doc.Load(pathName);
            }

            catch (System.Exception e)
            {
                Debug.WriteLine(e.Message);
                return false;
            }

            PostLoad(doc);

            return true;
        }

        //----------------------------------------------------------------------
        /// <summary>
        /// Post process after config file is read.
        /// Override this to customize
        /// </summary>
        ///
        /// <param name="doc">XML Document</param>
        ///
        /// <returns>True if success</returns>
        //----------------------------------------------------------------------
        protected bool PostLoad(System.Xml.XmlDocument doc)
        {
            System.Xml.XmlElement root = Utility.XMLFindChildElement(doc, "SessionConfig");
            if (root == null)
                return false;

            { // Main From section
                System.Xml.XmlElement element = Utility.XMLFindChildElement(root, "MainForm");
                if (element != null)
                    LoadMainFormSection(element);
            }

            return true;
        }

        #endregion

        #region Save
        //----------------------------------------------------------------------
        /// <summary>
        /// Save configuration
        /// </summary>
        ///
        /// <returns>True if success</returns>
        //----------------------------------------------------------------------
        public bool Save(String configDir)
        {
            System.Xml.XmlDocument doc = new System.Xml.XmlDocument();
            System.Xml.XmlDeclaration decl = doc.CreateXmlDeclaration("1.0", "utf-8", null);

            doc.AppendChild(decl);

            System.Xml.XmlElement root = doc.CreateElement("SessionConfig");
            doc.AppendChild(root);

            PreSave(doc, root);

            String pathName = configDir + "/" + m_fileName;
            try
            {
                System.Xml.XmlWriterSettings writeSetts = new System.Xml.XmlWriterSettings();
                writeSetts.Indent = true;
                writeSetts.IndentChars = "    ";

                System.Xml.XmlWriter writer = System.Xml.XmlWriter.Create(pathName, writeSetts);

                doc.Save(writer);
            }

            catch (System.Exception e)
            {
                Debug.WriteLine(e.Message);
                return false;
            }

            return true;
        }

        //----------------------------------------------------------------------
        /// <summary>
        /// Preprocess before config file is saved.
        /// Override this to customize
        /// </summary>
        ///
        /// <param name="doc">XML Document</param>
        /// <param name="root">Root element</param>
        ///
        /// <returns>True if success</returns>
        //----------------------------------------------------------------------
        protected bool PreSave(System.Xml.XmlDocument doc, System.Xml.XmlElement root)
        {
            { // MainForm section
                System.Xml.XmlElement element = doc.CreateElement("MainForm");
                root.AppendChild(element);
                SaveMainFormSection(element);
            }

            return true;
        }
        #endregion

        #region Appy
        //----------------------------------------------------------------------
        /// <summary>
        /// Apply Config
        /// </summary>
        //----------------------------------------------------------------------
        public void ApplyConfig()
        {
        }

        #endregion
    }
}
