﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.ServiceModel;
using System.Text;
using System.Threading;

namespace nw.g3d.bifedit.IPC
{
    /// <summary>
    /// プロセス間通信(IPC)を管理するクラスです。
    /// </summary>
    public class CommunicationService
    {
        private readonly RemoteControllerService service = new RemoteControllerService();

        /// <summary>
        /// サービス停止イベント
        /// </summary>
        private readonly ManualResetEventSlim serviceInterupt = new ManualResetEventSlim(true);

        public event EventHandler ServiceFaulted;

        public event Action ShowUserInterfaceRequested
        {
            add { service.ShowUserInterfaceRequested += value; }
            remove { service.ShowUserInterfaceRequested -= value; }
        }

        public event Action<string> EditorProcessFileRequested
        {
            add { service.EditorProcessFileRequested += value; }
            remove { service.EditorProcessFileRequested -= value; }
        }

        public event Action<string[]> CompaleProcessFileRequested
        {
            add { service.CompaleProcessFileRequested += value; }
            remove { service.CompaleProcessFileRequested -= value; }
        }

        public bool IsRunning
        {
            get
            {
                // returns true when not signaled
                return serviceInterupt.Wait(0) == false;
            }
        }

        /// <summary>
        /// IPCサーバーを起動します。
        /// </summary>
        public void Start()
        {
            if (IsRunning)
                return;

            new Thread(() =>
            {
                using (var host = new ServiceHost(service))
                {
                    // サービスを設定、起動
                    host.Faulted += InternalServiceFaulted;
                    host.AddServiceEndpoint(typeof(IRemoteController),
                        new NetNamedPipeBinding(),
                        "net.pipe://localhost/NamedPipeArgumentContainer");
                    host.Open();

                    serviceInterupt.Reset(); // <- considered has running from this timing

                    // サービス停止イベントがセットされるまで待機
                    serviceInterupt.Wait();

                    host.Faulted -= InternalServiceFaulted;

                    // after investigating for a while, it seems that stopping the application via
                    // Visual Studio's "Stop Debugging" command, all the threads are interupted instead
                    // of being properly stopped, and that causes a delay in the service shutdown

                    host.Close(); // <- sometimes long delay when application is "killed" through Visual Studio "Stop Debugging"
                }
            })
            {
                Name = "IPC Server Thread",
                IsBackground = false,
            }.Start();
        }

        /// <summary>
        /// IPCサーバーを停止します。
        /// </summary>
        public void Stop()
        {
            // サービス停止イベントをセット
            serviceInterupt.Set(); // <- considered has NOT running from this timing
        }

        /// <summary>
        /// IPCクライアントを生成します。
        /// </summary>
        /// <returns>IPCクライアントを返します。</returns>
        public IRemoteController GenerateClientSideProxy()
        {
            // IPCクライアントを生成
            var channelFactory = new ChannelFactory<IRemoteController>(
                 new NetNamedPipeBinding(),
                 new EndpointAddress("net.pipe://localhost/NamedPipeArgumentContainer"));

            return channelFactory.CreateChannel();
        }

        private void InternalServiceFaulted(object sender, EventArgs e)
        {
            if (ServiceFaulted != null)
                ServiceFaulted(sender, e);
        }
    }
}
