﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using nw.g3d.iflib;
using nw.g3d.toollib;
using nw.g3d.nw4f_3dif;
using System.Windows.Forms;
using System.Diagnostics;
using System.IO;

namespace nw.g3d.bifedit
{
    /// <summary>
    /// Editor app information
    /// </summary>
    public class AppInfo
    {
        public AppInfo() { }

        public String Name { get; set; }
        public String Path { get; set; }
        public String CommandArguments { get; set; }
    }

    /// <summary>
    /// Config
    /// </summary>
    public class Config
    {
        /// <summary>
        /// Constructor
        /// </summary>
        public Config()
        {
            this.XSDFilePath = "";
            this.IsValidXSDPath = false;
            this.ApplicationList = new List<AppInfo>();
            this.SelectedAppName = "";
            this.CustomAppPath = "";
            this.CustomAppCommandArguments = "";
        }

        public string DiffToolFilename { get; set; }
        public string DiffToolCommandArguments { get; set; }

        /// <summary>
        /// XSD file to use
        /// </summary>
        public String XSDFilePath { get; private set; }

        /// <summary>
        /// Valid XSD file
        /// </summary>
        public bool IsValidXSDPath { get; private set; }

        /// <summary>
        /// List of application info
        /// </summary>
        public List<AppInfo> ApplicationList { get; private set; }

        /// <summary>
        /// Selected app name
        /// </summary>
        public String SelectedAppName { get; set; }

        /// <summary>
        /// Custom app path
        /// </summary>
        public String CustomAppPath { get; set; }

        /// <summary>
        /// Custom app path
        /// </summary>
        public String CustomAppCommandArguments { get; set; }

        /// <summary>
        /// Load config from file
        /// </summary>
        public bool LoadConfig(String appModulePath)
        {
            System.Xml.XmlDocument doc = new System.Xml.XmlDocument();

            String path = appModulePath + "\\3dBinaryIntermediateFileEdit.xml";

            try
            {
                doc.Load(path);
            } // End of try

            catch (System.Exception e)
            {
                Debug.WriteLine(e.Message);
                return false;
            } // End catch

            PostLoad(appModulePath, doc);

            LoadStatusConfig(appModulePath);

            return true;
        }

        /// <summary>
        /// Load config from file
        /// </summary>
        private bool LoadStatusConfig(String appModulePath)
        {
            System.Xml.XmlDocument doc = new System.Xml.XmlDocument();

            String path = appModulePath + "\\3dBinaryIntermediateFileEdit_status.xml";

            try
            {
                doc.Load(path);
            } // End of try

            catch (System.Exception e)
            {
                Debug.WriteLine(e.Message);
                return false;
            } // End catch

            PostLoadStatus(appModulePath, doc);

            return true;
        }

        /// <summary>
        /// Replace string with system variable
        /// </summary>
        /// <param name="pathName"></param>
        /// <param name="envName"></param>
        private void ReplaceSystemEnvironmentVariables(ref String pathName)
        {
            bool bDone    = false;

            int searchPos = 0;

            while (bDone == false)
            {
                if (searchPos>=pathName.Length)
                    break;

                int idx0 = pathName.IndexOf("%", searchPos);
                if (idx0 < 0)
                    break;

                if (idx0 >= pathName.Length)
                    break;

                int idx1 = pathName.IndexOf("%", idx0 + 1);
                if (idx1 < 0)
                    break;

                if (idx0 >= idx1)
                    break;

                String str1    = pathName.Remove(0, idx0 + 1);
                String envName = str1.Remove(idx1 - idx0 - 1);

                String sysEnvName = Environment.GetEnvironmentVariable(envName);
                if (sysEnvName == null)
                {
                    searchPos = idx1 + 1;
                    continue;
                }

                String newPathName = pathName.Remove(idx0,idx1-idx0+1);
                pathName = newPathName.Insert(idx0, sysEnvName);

                searchPos = idx0 + sysEnvName.Length;
            } // End while
        }

        /// <summary>
        /// xsd要素を読み込みます。
        /// </summary>
        /// <param name="modulePath">exeファイルパス</param>
        /// <param name="element">xsd要素</param>
        private void ReadXSDSection(String modulePath, System.Xml.XmlElement element)
        {
            // Path属性を取得
            String path = element.GetAttribute("Path");

            if (string.IsNullOrEmpty(path) == false)
            {
                // Pathが指定されていたとき環境変数をロードする
                ReplaceSystemEnvironmentVariables(ref path);
                this.XSDFilePath = path;
            }
            else
            {
                // Pathが指定されていないときデフォルト値を適用する
                // 旧 NW4F パッケージ用のミラーリング等は不要なはずなので、g3dxsd をサポートしない。
                const string PATH_FOR_NINTENDO_SDK = @"..\..\..\..\3dTools\3dIntermediateFileXsd\";
                const string PATH_FOR_UNIT_PACKAGE = @"..\CommandLineTools\3dIntermediateFileXsd\";

                string path_nsdk = Path.Combine(modulePath, PATH_FOR_NINTENDO_SDK);  // Nintendo SDKのパス
                string path_unit = Path.Combine(modulePath, PATH_FOR_UNIT_PACKAGE);  // 単体パッケージのパス

                if (Directory.Exists(path_nsdk))
                {
                    this.XSDFilePath = path_nsdk;
                }
                else if (Directory.Exists(path_unit))
                {
                    this.XSDFilePath = path_unit;
                }
            }

            // XSDFilePathにディレクトリがあるかチェック
            this.IsValidXSDPath = Directory.Exists(this.XSDFilePath);
        }

        /// <summary>
        /// Read Application List
        /// </summary>
        private void ReadApplicationList(System.Xml.XmlElement element)
        {
            foreach (System.Xml.XmlNode childNode in element.ChildNodes)
            {
                System.Xml.XmlElement childElement = childNode as System.Xml.XmlElement;
                if (childElement == null)
                    continue;

                if (String.Compare(childElement.Name, "application", StringComparison.CurrentCultureIgnoreCase) == 0)
                {
                    String appName = childElement.GetAttribute("Name");
                    String appPath = childElement.GetAttribute("Path");
                    String appCmdArgs = childElement.GetAttribute("CommandArguments");

                    if (string.IsNullOrWhiteSpace(appName) == false &&
                        string.IsNullOrWhiteSpace(appPath) == false)
                    {
                        ReplaceSystemEnvironmentVariables(ref appPath);

                        var appInfo = new AppInfo
                        {
                            Name = appName,
                            Path = appPath,
                            CommandArguments = appCmdArgs,
                        };

                        this.ApplicationList.Add(appInfo);
                    }
                }
            }
        }

        /// <summary>
        /// Post process after config file is read.
        /// Override this to customize
        /// </summary>
        ///
        /// <param name="doc">XML Document</param>
        ///
        /// <returns>True if success</returns>
        protected bool PostLoad(String modulePath, System.Xml.XmlDocument doc)
        {
            System.Xml.XmlElement root = Utility.XMLFindChildElement(doc, "appconfig");
            if (root == null)
                return false;

            foreach (System.Xml.XmlNode childNode in root.ChildNodes)
            {
                System.Xml.XmlElement childElement = childNode as System.Xml.XmlElement;
                if (childElement == null)
                    continue;

                if (String.Compare(childElement.Name, "xsd", StringComparison.CurrentCultureIgnoreCase) == 0)
                {
                    ReadXSDSection(modulePath, childElement);
                }

                else if (String.Compare(childElement.Name, "application-list", StringComparison.CurrentCultureIgnoreCase) == 0)
                {
                    ReadApplicationList(childElement);
                }
            }

            return true;
        }

        /// <summary>
        /// Post process after status config file is read.
        /// Override this to customize
        /// </summary>
        ///
        /// <param name="doc">XML Document</param>
        ///
        /// <returns>True if success</returns>
        protected bool PostLoadStatus(String modulePath, System.Xml.XmlDocument doc)
        {
            System.Xml.XmlElement root = Utility.XMLFindChildElement(doc, "appstatus");
            if (root == null)
                return false;

            foreach (System.Xml.XmlNode childNode in root.ChildNodes)
            {
                System.Xml.XmlElement childElement = childNode as System.Xml.XmlElement;
                if (childElement == null)
                    continue;

                if (String.Compare(childElement.Name, "SelectedApp", StringComparison.CurrentCultureIgnoreCase) == 0)
                {
                    this.CustomAppPath = childElement.GetAttribute("CustomPath");
                    if (string.IsNullOrEmpty(this.CustomAppPath))
                        this.SelectedAppName = childElement.GetAttribute("Name");
                    this.CustomAppCommandArguments = childElement.GetAttribute("CommandArguments");
                }
                else if (string.Equals(childElement.Name, "DiffTool", StringComparison.CurrentCultureIgnoreCase))
                {
                    var path = childElement.GetAttribute("Path");
                    if (string.IsNullOrWhiteSpace(path) == false)
                        DiffToolFilename = Environment.ExpandEnvironmentVariables(path.Trim());

                    path = childElement.GetAttribute("CommandArguments");
                    DiffToolCommandArguments = string.Empty;
                    if (string.IsNullOrWhiteSpace(path) == false)
                        DiffToolCommandArguments = path.Trim();
                }
            }

            return true;
        }

        /// <summary>
        /// Save status config to file
        /// </summary>
        public bool SaveStatusConfig(String appModulePath)
        {
            System.Xml.XmlDocument doc = new System.Xml.XmlDocument();

            System.Xml.XmlDeclaration decl = doc.CreateXmlDeclaration("1.0", "utf-8", null);

            doc.AppendChild(decl);

            System.Xml.XmlElement root = doc.CreateElement("appstatus");
            doc.AppendChild(root);

            PathManager pathManager = PathManager.Instance;

            if ( (pathManager.EditorApplication != null) && (pathManager.EditorApplication.Length>0) )
            {
                System.Xml.XmlElement selectedAppElement = doc.CreateElement("SelectedApp");
                root.AppendChild(selectedAppElement);

                if (string.IsNullOrWhiteSpace(this.SelectedAppName) == false)
                    selectedAppElement.SetAttribute("Name", this.SelectedAppName);
                if (string.IsNullOrWhiteSpace(this.CustomAppPath) == false)
                    selectedAppElement.SetAttribute("CustomPath", this.CustomAppPath);
                if (string.IsNullOrWhiteSpace(this.CustomAppCommandArguments) == false)
                    selectedAppElement.SetAttribute("CommandArguments", this.CustomAppCommandArguments);
            }

            if (string.IsNullOrWhiteSpace(DiffToolFilename) == false)
            {
                System.Xml.XmlElement difftoolElement = doc.CreateElement("DiffTool");
                root.AppendChild(difftoolElement);

                difftoolElement.SetAttribute("Path", DiffToolFilename);
                if (string.IsNullOrWhiteSpace(DiffToolCommandArguments) == false)
                    difftoolElement.SetAttribute("CommandArguments", DiffToolCommandArguments);
            }

            String path = appModulePath + "\\3dBinaryIntermediateFileEdit_status.xml";

            try
            {
                doc.Save(path);
            }
            catch (System.Exception e)
            {
                Debug.WriteLine(e.Message);
                CustomMessageBox.Show(e.Message, Properties.Resources.NW4F_SAVE_FILE_ERROR,
                    MessageBoxButtons.OK, MessageBoxIcon.Error);
                return false;
            }

            return true;
        }
    }
} // End of nw.g3d.bifedit
