﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using nw.g3d.bifedit.Properties;
using System.IO;
using nw.g3d.iflib;
using System.Diagnostics;
using System.Threading;
using System.Text.RegularExpressions;

namespace nw.g3d.bifedit
{
    public static class CompareProcess
    {
        public static bool IsComparasion(string[] args)
        {
            return args.Any(p => p == "-c" || p == "-compare");
        }

        public static bool Run(string[] args)
        {
            return ThreadPool.QueueUserWorkItem(_ => ProcessFile(args));
        }

        public static void ProcessFile(string[] args)
        {
            args = args.Except(new[] { "-c", "-compare" }).ToArray();

            if (args.Length < 2)
            {
                CustomMessageBox.Show(Resources.NW4F_COMPARE_INVALID_ARG_COUNT, Resources.NW4F_COMPARE_ERROR,
                    MessageBoxButtons.OK, MessageBoxIcon.Error);
                return;
            }

            var pathManager = PathManager.Instance;

            if (string.IsNullOrWhiteSpace(pathManager.Config.DiffToolFilename))
            {
                CustomMessageBox.Show(Resources.NW4F_COMPATE_NO_DIFF_TOOL,
                    Resources.NW4F_COMPARE_ERROR,
                    MessageBoxButtons.OK, MessageBoxIcon.Error);
                return;
            }

            bool hasError = false;
            var needToDeleted = new bool[args.Length];

            if (pathManager.IsXSDFilePathValid==false)
            {
                CustomMessageBox.Show(string.Format("{0} ({1})\r\n", Resources.NW4F_INVALID_XSD_PATH, pathManager.XSDFilePath),
                                      Resources.NW4F_LOAD_FILE_FILE_ERROR,
                                      MessageBoxButtons.OK, MessageBoxIcon.Error);
                hasError = true;
            }

            if (hasError == false)
            {
                for (int i = 0; i < args.Length; i++)
                {
                    if (nw4f_3dif.G3dPath.IsBinaryPath(args[i]) == false)
                        continue;

                    var temp = ConvertBinaryToText(args[i], pathManager);

                    if (temp == null)
                    {
                        /* Important Change - We will let process continue with ill formatted file as is
                        CustomMessageBox.Show(string.Format("{0}\r\n'{1}'", Resources.NW4F_LOAD_FILE_INVALID_BINARY_IF_FILE, args[i]),
                            Resources.NW4F_LOAD_FILE_FILE_ERROR,
                            MessageBoxButtons.OK, MessageBoxIcon.Error);
                        hasError = true;
                        break;
                         */
                        continue;
                    }

                    args[i] = temp;
                    needToDeleted[i] = true; // a temporary file has been generated, it must be deleted when it's done
                }
            }

            if (hasError == false)
            {
                var processedArguemnts = Regex.Replace(pathManager.Config.DiffToolCommandArguments, @"\%(?<n>\d+)",
                    m =>
                    {
                        var num = int.Parse(m.Groups["n"].Value);
                        if (num < 1 || num > args.Length)
                            return string.Format("%{0}", num);
                        return args[num - 1];
                    });

                var process = new Process
                {
                    StartInfo = new ProcessStartInfo(pathManager.Config.DiffToolFilename, processedArguemnts),
                };

                try
                {
                    process.Start();
                    process.WaitForExit();
                }
                catch (Exception ex)
                {
                    string message = string.Format("{0}\r\n'{1}'\r\n{2}", ex.Message, pathManager.Config.DiffToolFilename, processedArguemnts);
                    CustomMessageBox.Show(message, Resources.NW4F_COMPARE_ERROR, MessageBoxButtons.OK, MessageBoxIcon.Error);
                }
            }

            for (int i = 0; i < args.Length; i++)
            {
                if (needToDeleted[i])
                {
                    try
                    {
                        File.Delete(args[i]);
                    }
                    catch { }
                }
            }
        }

        private static string ConvertBinaryToText(string inputFullFilename, PathManager pathManager)
        {
            try
            {
                string outputPath = Path.GetDirectoryName(Path.GetTempPath());
                string inputFilename = Path.GetFileNameWithoutExtension(inputFullFilename);
                string inputExtension = Path.GetExtension(inputFullFilename);

                string outputFullFilename = string.Format("{0}\\{1}{2}", outputPath, inputFilename, inputExtension);

                if (File.Exists(outputFullFilename))
                {
                    int counter = 1;
                    do
                    {
                        outputFullFilename = string.Format("{0}\\{1}({3}){2}", outputPath, inputFilename, inputExtension, counter++);
                    } while (File.Exists(outputFullFilename));
                }

                var intermediateFile = IfBinaryReadUtility.Read(inputFullFilename, pathManager.XSDFilePath);
                IfTextWriteUtility.Write(intermediateFile, outputFullFilename, null);

                return outputFullFilename;
            }
            catch
            {
                return null;
            }
        }
    }
}
