﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// ビューアは、開発用の機能である、ホスト通信を利用するので、Release ビルド はできません。

#include <stdint.h>
#include <nn/util/util_Matrix.h>

#if defined(NN_BUILD_CONFIG_OS_WIN32)
#ifndef NOMINMAX
#define NOMINMAX
#endif
#include <nn/nn_windows.h>
#include <mmsystem.h>
#endif

#include <nn/fs.h>

#include <nn/os/os_Tick.h>

#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>

#include <nn/vi.h>
#include <nn/gfx/gfx_Variation-api.gl.h>
#include <nns/gfx/gfx_GraphicsFramework.h>

#include <nn/ui2d.h>
#include <nn/font.h>
#include <nn/ui2d/viewer/ui2d_Viewer.h>

#include <nn/ui2d/viewer/ui2d_Screen.h>
#include <nn/ui2d/viewer/ui2d_Viewer2.h>

#include <nns/hid.h>
#include <nns/nac/nac_Pad.h>
#include <nns/nac/nac_Pad.h>
#include <nns/nac/nac_Mouse.h>
#include <nns/nac/nac_File.h>

#include "SimpleAllocator.h"
#include "GlareEffect.h"

#include <nn/htcs.h>

#if !defined(NN_BUILD_CONFIG_OS_WIN32) && defined(NN_BUILD_CONFIG_SPEC_NX)
#if !defined(WIN_INTERFACE_CUSTOM)
#define WIN_INTERFACE_CUSTOM
#endif
#include <nn/init.h>
#endif

#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
#include <nv/nv_MemoryManagement.h>
#endif

#define NN_PERF_PROFILE_ENABLED
#include <nn/perf/perf_Profile.h>

#include <nns/gfx/gfx_ColorBuffer.h>
#include <nns/gfx/gfx_PrimitiveRenderer.h>
#include <nns/gfx/gfx_PrimitiveRendererMeterDrawer.h>


namespace {

//------------------------------------------------------------------------------
const size_t    AppHeapSize = 1024 * 1024 * 256;

const int       AppPathLengthMax = 260;
const uint32_t  CharMax = 1024;
const int       PadRepeatFrames = 30;

// Lifetime_Layout 用のキャプチャテクスチャ用メモリサイズ
const size_t    StaticCaptureTextureMemorySize = 256 * 1024 * 1024;
// Lifetime_OneFrame 用のキャプチャテクスチャ用メモリサイズ
const size_t    DynamicCaptureTextureMemorySize = 256 * 1024 * 1024;

//------------------------------------------------------------------------------
//  変数定義
//------------------------------------------------------------------------------
// グラフィックスフレームワーク
nns::gfx::GraphicsFramework                 g_GfxFramework;
ptrdiff_t                                   g_offsetToUi2dDrawMemoryPool;
ptrdiff_t                                   g_offsetToHudDrawMemoryPool;
ptrdiff_t                                   g_offsetToMeterMemoryPool;
ptrdiff_t                                   g_offsetToRenderTargetTextureMemoryPoolHead = NULL;
ptrdiff_t                                   g_offsetToRenderTargetTextureMemoryPoolStatic = NULL;
ptrdiff_t                                   g_offsetToRenderTargetTextureMemoryPoolDynamic = NULL;
ptrdiff_t                                   g_offsetToColorCopyTexture = NULL;

#if defined(NN_BUILD_CONFIG_OS_WIN32)
ptrdiff_t                                   g_offsetToScreenShotMemoryPool;
#endif

#if defined( NN_BUILD_CONFIG_OS_WIN32 )
nn::gfx::SwapChain                          g_RGBSwapChain;
// HDR カラーバッファキャプチャ用レンダーターゲット
nns::gfx::ColorBuffer                       g_CaptureColorBuffer;
#endif

GlareEffect                                 g_GlareEffect;
nn::gfx::TextureView                        g_ColorBufferTextureView;
nn::gfx::DescriptorSlot                     g_ColorBufferDescSlot;


// コンバイナエディタ
nn::gfx::Texture*                           g_pColorCopyTexture = NULL;
nn::gfx::ColorTargetView*                   g_pColorCopyTargetView = NULL;
nn::gfx::TextureView*                       g_pColorCopyTextureView = NULL;
nn::gfx::DescriptorSlot*                    g_pColorCopyDescSlot = NULL;

nn::gfx::ImageFormat                        g_ColorBufferFormat;

int                             g_Width = 1280;
int                             g_Height = 720;
#if defined( NN_BUILD_CONFIG_OS_WIN )
HWND                            g_Hwnd;
#endif

//------------------------------------------------------------------------------
// フォント用定義

nn::font::DispStringBuffer      g_DispStringBuf;
nn::font::ResFont               g_ResFont;
nn::font::RectDrawer            g_RectDrawer;
nn::util::MatrixT4x4fType       g_ProjectionFont;
void*                           g_pRectDrawerWorkBuf;
void*                           g_pFontResource;

// Hud フォント描画用バッファ関連
nn::font::GpuBuffer             g_HudFontConstantBuffer;
uint32_t                        g_HudFontFrameIndex = 0;
nn::font::TextWriter            g_Writer;

//------------------------------------------------------------------------------
nns::nac::Pad                   g_Pad;
int                             g_PadLeftRepeat  = 0;
int                             g_PadRightRepeat = 0;
nns::nac::Mouse                 g_Mouse;
nns::hid::ControllerManager     g_ControllerManager;


//------------------------------------------------------------------------------
//  PrimitiveRenderer 初期化処理
nns::gfx::PrimitiveRenderer::Renderer*      g_pPrimitiveRenderer;
nns::gfx::PrimitiveRenderer::MeterDrawer    g_MeterDrawer;
void*                                       g_pMeterMemory;

//------------------------------------------------------------------------------
float                             g_ScreenW = 1280.f;
float                             g_ScreenH = 720.f;
bool                              g_HDRWriteEnabled = false;

uint32_t                        g_FrameIndex = 0;
bool                            g_FirstUpdate = true;

bool                            g_Viewer2Enabled = false;

nn::ui2d::viewer::Viewer        g_LayoutViewer;

nn::ui2d::viewer::ViewerScreen  g_ViewerScreen;
nn::ui2d::viewer::ScreenManager g_ScreenManager;

nn::os::Tick                    g_LastUpdateTime;

// ui2d 描画用バッファ
nn::font::GpuBuffer             g_Ui2dConstantBuffer;

nn::ui2d::GraphicsResource      g_GraphicsResource;
int64_t                         g_GraphicsResourceHeapSize = 0;

SimpleAllocator*                g_pAllocator = NULL;

//------------------------------------------------------------------------------
// 注意：最低限動作させるための仮実装です。
// 本来は、マネージャーが samplerInfo をキーに、Sampler オブジェクトや slot を管理します。
int                                  g_SamplerSlotCount = 0;
nn::gfx::Sampler                     g_SamplerSlots[64];


#if defined( NN_BUILD_CONFIG_OS_WIN32 )

//------------------------------------------------------------------------------
static void InitializeGfxWin_(HWND hwnd, int width, int height )
{
    // 2015/4/1現在、gfxではウインドウサイズが設定出来ないようなので
    // Win32APIを呼び出して強制的にウインドウサイズを設定
    // 将来的には削除予定
    {
        RECT rw, rc;
        ::GetWindowRect(hwnd, &rw);
        ::GetClientRect(hwnd, &rc);

        int new_width = (rw.right - rw.left) - (rc.right - rc.left) + width;
        int new_height = (rw.bottom - rw.top) - (rc.bottom - rc.top) + height;

        SetWindowPos(hwnd, NULL, 0, 0, new_width, new_height, SWP_NOMOVE | SWP_NOZORDER);
        //
#if defined(NN_BUILD_CONFIG_SPEC_NX)
        SetWindowText(hwnd, L"LayoutViewer Gfx(NVN)");
#else
        SetWindowText(hwnd, L"LayoutViewer Gfx(OpenGL)");
#endif// NN_BUILD_CONFIG_SPEC_NX
    }
}
#endif // NN_BUILD_CONFIG_OS_WIN32

#if defined(NN_BUILD_CONFIG_OS_WIN)
// return always "0" because nn::os::SetThreadCoreMask sometimes doesn't work on windows
int GetFixedCoreNumber()
{
    return 0;
}
#endif

//------------------------------------------------------------------------------
//  nn::fs用メモリ取得用関数
//------------------------------------------------------------------------------
void* Allocate(size_t size)
{
    void* pMem = malloc(size);
    memset(pMem, 0xBE, size);
    return pMem;
}

//------------------------------------------------------------------------------
//  nn::fs用メモリ解放用関数
//------------------------------------------------------------------------------
void Deallocate(void* p, size_t size)
{
    NN_UNUSED(size);
    free(p);
}

//-----------------------------------------------------------------------------
void* AllocFunction_(size_t size, size_t alignment, void* pUserData)
{
    NN_UNUSED(pUserData);

    void* pMem = g_pAllocator->Alloc(size, alignment);
    memset(pMem, 0xBE, size);
    return pMem;
}

//-----------------------------------------------------------------------------
void FreeFunction_(void* ptr, void* pUserData)
{
    NN_UNUSED(pUserData);
    g_pAllocator->Free(ptr);
}

void* HtcsAllocate( size_t size )
{
    void* pMem = malloc( size );
    memset(pMem, 0xBE, size);
    return pMem;
}

void HtcsDeallocate( void* p, size_t size )
{
    NN_UNUSED( size );
    free(p);
}

//------------------------------------------------------------------------------
bool RegisterSlotForTexture_(nn::gfx::DescriptorSlot* pDstSlot, const nn::gfx::TextureView& textureView, void* pUserData)
{
    if (pUserData)
    {
        nns::gfx::GraphicsFramework* graphicsFramework = reinterpret_cast<nns::gfx::GraphicsFramework*>(pUserData);

        const int slotIndex = graphicsFramework->AllocateDescriptorSlot(nn::gfx::DescriptorPoolType::DescriptorPoolType_TextureView, 1);
        if (slotIndex == nn::gfx::util::DescriptorPoolAllocator::InvalidIndex)
        {
            return false;
        }

        nn::gfx::DescriptorPool* pPool = graphicsFramework->GetDescriptorPool(nn::gfx::DescriptorPoolType::DescriptorPoolType_TextureView);

        pPool->BeginUpdate();
        {
            pPool->SetTextureView(slotIndex, &textureView);
        }
        pPool->EndUpdate();
        pPool->GetDescriptorSlot(pDstSlot, slotIndex);

        return true;
    }

    return false;
}

//------------------------------------------------------------------------------
bool RegisterSlotForSampler_(nn::gfx::DescriptorSlot* pDstSlot, const nn::gfx::Sampler& sampler, void* pUserData)
{
    if (pUserData)
    {
        nns::gfx::GraphicsFramework* graphicsFramework = reinterpret_cast<nns::gfx::GraphicsFramework*>(pUserData);

        const int slotIndex = graphicsFramework->AllocateDescriptorSlot(nn::gfx::DescriptorPoolType::DescriptorPoolType_Sampler, 1);
        if (slotIndex == nn::gfx::util::DescriptorPoolAllocator::InvalidIndex)
        {
            return false;
        }

        nn::gfx::DescriptorPool* pPool = graphicsFramework->GetDescriptorPool(nn::gfx::DescriptorPoolType::DescriptorPoolType_Sampler);

        pPool->BeginUpdate();
        {
            pPool->SetSampler(slotIndex, &sampler);
        }
        pPool->EndUpdate();
        pPool->GetDescriptorSlot(pDstSlot, slotIndex);

        return true;
    }

    return false;
}

//------------------------------------------------------------------------------
bool AcquireSlotForSampler_(nn::gfx::DescriptorSlot* pDstSlot, const nn::gfx::SamplerInfo& samplerInfo, void* pUserData)
{
    if (pUserData)
    {
        // 注意：動作確認用の仮実装です。本来は、マネージャーが samplerInfo をキーに、Sampler オブジェクトや slot を管理します。
        NN_SDK_ASSERT(g_SamplerSlotCount < 64);
        nn::gfx::Sampler* pSampler = &g_SamplerSlots[g_SamplerSlotCount];

        nns::gfx::GraphicsFramework* graphicsFramework = reinterpret_cast<nns::gfx::GraphicsFramework*>(pUserData);
        pSampler->Initialize(graphicsFramework->GetDevice(), samplerInfo);

        if (!RegisterSlotForSampler_(pDstSlot, *pSampler, pUserData))
        {
            return false;
        }

        g_SamplerSlotCount++;
        return true;
    }

    return false;
}

//------------------------------------------------------------------------------
void UnregisterSlotForSampler_(nn::gfx::DescriptorSlot* pDstSlot, const nn::gfx::Sampler& sampler, void* pUserData)
{
    NN_UNUSED(sampler);
    if (pUserData)
    {
        nns::gfx::GraphicsFramework* graphicsFramework = reinterpret_cast<nns::gfx::GraphicsFramework*>(pUserData);
        nn::gfx::DescriptorPool* pPool = graphicsFramework->GetDescriptorPool(nn::gfx::DescriptorPoolType::DescriptorPoolType_Sampler);
        const int slotIndex = pPool->GetDescriptorSlotIndex(*pDstSlot);
        graphicsFramework->FreeDescriptorSlot(nn::gfx::DescriptorPoolType::DescriptorPoolType_Sampler, slotIndex);
    }
}

//------------------------------------------------------------------------------
void ReleaseSlotForSampler_(nn::gfx::DescriptorSlot* pDstSlot, const nn::gfx::SamplerInfo& samplerInfo, void* pUserData)
{
    NN_UNUSED(samplerInfo);
    if (pUserData)
    {
        UnregisterSlotForSampler_(pDstSlot, g_SamplerSlots[g_SamplerSlotCount], pUserData);

        nns::gfx::GraphicsFramework* graphicsFramework = reinterpret_cast<nns::gfx::GraphicsFramework*>(pUserData);

        // 注意：動作確認用の仮実装です。本来は、マネージャーが samplerInfo をキーに、Sampler オブジェクトや slot を管理します。
        NN_SDK_ASSERT(g_SamplerSlotCount >= 1);
        g_SamplerSlots[g_SamplerSlotCount - 1].Finalize(graphicsFramework->GetDevice());
        g_SamplerSlotCount--;
    }
}

//------------------------------------------------------------------------------
void UnregisterSlotForTexture_(nn::gfx::DescriptorSlot* pDstSlot, const nn::gfx::TextureView& textureView, void* pUserData)
{
    NN_UNUSED(textureView);
    if (pUserData)
    {
        nns::gfx::GraphicsFramework* graphicsFramework = reinterpret_cast<nns::gfx::GraphicsFramework*>(pUserData);
        nn::gfx::DescriptorPool* pPool = graphicsFramework->GetDescriptorPool(nn::gfx::DescriptorPoolType::DescriptorPoolType_TextureView);
        int slotIndex = pPool->GetDescriptorSlotIndex(*pDstSlot);
        graphicsFramework->FreeDescriptorSlot(nn::gfx::DescriptorPoolType::DescriptorPoolType_TextureView, slotIndex);
    }
}

//-----------------------------------------------------------------------------
void* ViewerLoad_(const char* pPath, int alignment)
{
    size_t      size;
    void*   pData;

    NN_LOG("---------------------------------------------------\n");
    NN_LOG("LayoutViewer:load_path=");
    NN_LOG("%s\n", pPath);
    NN_LOG("---------------------------------------------------\n");

    // nac::FileLoad が特殊処理をしているので 先頭の / を読み飛ばす（TODO:理由を確認して修正する）
    if(pPath != NULL && pPath[0] == '/')
    {
        pPath++;
    }

    if(!nns::nac::FileLoad( &pData, &size, pPath, ".", g_pAllocator, alignment ))
    {
        NN_LOG("Fail to load file ! (%s)\n", pPath);
        NN_ASSERT(false);
    }

    return pData;
}

//-----------------------------------------------------------------------------
void ViewerUnload_(void* pData)
{
    g_pAllocator->Free(pData);
}

//-----------------------------------------------------------------------------
void ViewerMemoryLeakChecking_()
{
    // メモリリークがあればassertで止める
    const int64_t occupiedSize = g_pAllocator->GetOccupiedSize();
    if(occupiedSize != 0)
    {
        g_pAllocator->PrintOccupiedMemory();
        NN_SDK_ASSERT(false, "Memory leak: %lld", occupiedSize);
    }
}

//-----------------------------------------------------------------------------
void InitializeCopyFrameBuffer()
{
    nn::gfx::Device* pDevice = g_GfxFramework.GetDevice();

    // カラーテクスチャ
    nn::gfx::Texture::InfoType info;
    info.SetDefault();
    info.SetWidth(g_GfxFramework.GetDisplayWidth());
    info.SetHeight(g_GfxFramework.GetDisplayHeight());
    info.SetGpuAccessFlags(nn::gfx::GpuAccess_ColorBuffer);
    info.SetImageStorageDimension(nn::gfx::ImageStorageDimension_2d);
    info.SetImageFormat(g_ColorBufferFormat);
    info.SetMipCount(1);
    size_t size = nn::gfx::Texture::CalculateMipDataSize(pDevice, info);
    size_t align = nn::gfx::Texture::CalculateMipDataAlignment(pDevice, info);
    g_offsetToColorCopyTexture = g_GfxFramework.AllocatePoolMemory(nns::gfx::GraphicsFramework::MemoryPoolType_RenderTarget, size, align);
    nn::gfx::MemoryPool* pMemoryPool = g_GfxFramework.GetMemoryPool(nns::gfx::GraphicsFramework::MemoryPoolType_RenderTarget);

    g_pColorCopyTexture = nn::ui2d::Layout::AllocateAndConstruct<nn::gfx::Texture>();
    NN_ASSERT_NOT_NULL(g_pColorCopyTexture);
    g_pColorCopyTexture->Initialize(pDevice, info, pMemoryPool, g_offsetToColorCopyTexture, size);

    // カラーターゲットビュー
    nn::gfx::ColorTargetView::InfoType targetViewInfo;
    targetViewInfo.SetDefault();
    targetViewInfo.SetImageDimension(nn::gfx::ImageDimension_2d);
    targetViewInfo.SetImageFormat(g_ColorBufferFormat);
    targetViewInfo.SetTexturePtr(g_pColorCopyTexture);
    g_pColorCopyTargetView = nn::ui2d::Layout::AllocateAndConstruct<nn::gfx::ColorTargetView>();
    NN_ASSERT_NOT_NULL(g_pColorCopyTargetView);
    g_pColorCopyTargetView->Initialize(pDevice, targetViewInfo);

    // カラーバッファのテクスチャビュー
    nn::gfx::TextureView::InfoType viewInfo;
    viewInfo.SetDefault();
    viewInfo.SetImageDimension(nn::gfx::ImageDimension_2d);
    viewInfo.SetImageFormat(g_ColorBufferFormat);
    viewInfo.SetTexturePtr(g_pColorCopyTexture);
    g_pColorCopyTextureView = nn::ui2d::Layout::AllocateAndConstruct<nn::gfx::TextureView>();
    NN_ASSERT_NOT_NULL(g_pColorCopyTextureView);
    g_pColorCopyTextureView->Initialize(pDevice, viewInfo);

    g_pColorCopyDescSlot = nn::ui2d::Layout::AllocateAndConstruct<nn::gfx::DescriptorSlot>();
    NN_ASSERT_NOT_NULL(g_pColorCopyDescSlot);
    // カラーバッファテクスチャのデスクリプタスロット
    RegisterSlotForTexture_(g_pColorCopyDescSlot, *g_pColorCopyTextureView, &g_GfxFramework);
}

void FinalizeCopyFrameBuffer()
{
    nn::gfx::Device* pDevice = g_GfxFramework.GetDevice();
    if (g_pColorCopyDescSlot)
    {
        UnregisterSlotForTexture_(g_pColorCopyDescSlot, *g_pColorCopyTextureView, &g_GfxFramework);
        nn::ui2d::Layout::DeleteObj(g_pColorCopyDescSlot);
        g_pColorCopyDescSlot = NULL;
    }

    if (g_pColorCopyTargetView)
    {
        g_pColorCopyTargetView->Finalize(pDevice);
        nn::ui2d::Layout::DeleteObj(g_pColorCopyTargetView);
        g_pColorCopyTargetView = NULL;
    }

    if (g_pColorCopyTextureView)
    {
        g_pColorCopyTextureView->Finalize(pDevice);
        nn::ui2d::Layout::DeleteObj(g_pColorCopyTextureView);
        g_pColorCopyTextureView = NULL;
    }

    if (g_pColorCopyTexture)
    {
        g_pColorCopyTexture->Finalize(pDevice);
        nn::ui2d::Layout::DeleteObj(g_pColorCopyTexture);
        g_pColorCopyTexture =  NULL;
    }

    if (g_offsetToColorCopyTexture != NULL)
    {
        g_GfxFramework.FreePoolMemory(nns::gfx::GraphicsFramework::MemoryPoolType_RenderTarget, g_offsetToColorCopyTexture);
        g_offsetToColorCopyTexture = 0;
    }
}

void CopyFrameBufferImage(nn::gfx::CommandBuffer* pCmdBuffer)
{
    if (g_pColorCopyTexture == NULL)
    {
        return;
    }

    nn::gfx::TextureSubresource dstSubResource;
    dstSubResource.SetDefault();
    dstSubResource.SetMipLevel( 0 );
    dstSubResource.SetArrayIndex( 0 );

    nn::gfx::TextureCopyRegion srcCopyRegion;
    srcCopyRegion.SetDefault();
    srcCopyRegion.SetWidth( g_GfxFramework.GetDisplayWidth() );
    srcCopyRegion.SetHeight( g_GfxFramework.GetDisplayHeight() );
    srcCopyRegion.EditSubresource().SetDefault();
    srcCopyRegion.EditSubresource().SetMipLevel( 0 );
    srcCopyRegion.EditSubresource().SetArrayIndex( 0 );

    pCmdBuffer->FlushMemory( nn::gfx::GpuAccess_ColorBuffer );

    pCmdBuffer->CopyImage( g_pColorCopyTexture, dstSubResource,
                           0, 0, 0,
                           g_GfxFramework.GetColorBuffer(), srcCopyRegion );
}

//------------------------------------------------------------------------------
//  レイアウトデータの読み込みまえコールバック
//------------------------------------------------------------------------------
void PreReloadLayoutDataCallback()
{
    nn::gfx::Device* pDevice = g_GfxFramework.GetDevice();

    // 使用中のバッファの終了処理
    if (g_Ui2dConstantBuffer.IsInitialized())
    {
        g_Ui2dConstantBuffer.Finalize(pDevice, FreeFunction_, NULL);
    }

    // 描画に使用したコンスタントバッファ領域をフレームワークに返却する
    if (g_offsetToUi2dDrawMemoryPool != NULL)
    {
        const nns::gfx::GraphicsFramework::MemoryPoolType memoryPoolType = nns::gfx::GraphicsFramework::MemoryPoolType_ConstantBuffer;
        g_GfxFramework.FreePoolMemory(memoryPoolType, g_offsetToUi2dDrawMemoryPool);
        g_offsetToUi2dDrawMemoryPool = NULL;
    }

    // RenderTarget 用の領域をフレームワークに返却する
    if (g_offsetToRenderTargetTextureMemoryPoolHead != NULL)
    {
        const nns::gfx::GraphicsFramework::MemoryPoolType memoryPoolType = nns::gfx::GraphicsFramework::MemoryPoolType_RenderTarget;
        g_GfxFramework.FreePoolMemory(memoryPoolType, g_offsetToRenderTargetTextureMemoryPoolHead);
        g_offsetToRenderTargetTextureMemoryPoolHead = NULL;
        g_offsetToRenderTargetTextureMemoryPoolStatic = NULL;
        g_offsetToRenderTargetTextureMemoryPoolDynamic = NULL;
    }

    // フレームバッファコピーの破棄をする。
    FinalizeCopyFrameBuffer();

    // グラフィックスリソースを開放する
    if (g_GraphicsResource.Initialized())
    {
        g_GraphicsResource.ReleaseCommonSamplerSlot(ReleaseSlotForSampler_, &g_GfxFramework);
        g_GraphicsResource.Finalize(pDevice);

        ViewerMemoryLeakChecking_();
    }

    // RenderTarget 用の領域をフレームワークから確保する
    {
        nn::gfx::MemoryPool::InfoType memorypoolInfo;

        memorypoolInfo.SetDefault();
        memorypoolInfo.SetMemoryPoolProperty(nn::gfx::MemoryPoolProperty_CpuInvisible | nn::gfx::MemoryPoolProperty_GpuCached | nn::gfx::MemoryPoolProperty_Compressible);

        size_t alignment = nn::gfx::MemoryPool::GetPoolMemoryAlignment(pDevice, memorypoolInfo);

        const nns::gfx::GraphicsFramework::MemoryPoolType memoryPoolType = nns::gfx::GraphicsFramework::MemoryPoolType_RenderTarget;
        g_offsetToRenderTargetTextureMemoryPoolHead = g_GfxFramework.AllocatePoolMemory(memoryPoolType, StaticCaptureTextureMemorySize + DynamicCaptureTextureMemorySize, alignment);
        g_offsetToRenderTargetTextureMemoryPoolStatic = g_offsetToRenderTargetTextureMemoryPoolHead;
        g_offsetToRenderTargetTextureMemoryPoolDynamic = g_offsetToRenderTargetTextureMemoryPoolHead + StaticCaptureTextureMemorySize;
    }
}

//------------------------------------------------------------------------------
//  レイアウトデータの読み込み後コールバック
//------------------------------------------------------------------------------
void PostReloadLayoutDataCallback(nn::ui2d::DrawInfo& drawInfo, nn::ui2d::BuildResultInformation& buildResultInformation)
{
    NN_SDK_ASSERT(!g_GraphicsResource.Initialized());

    nn::gfx::Device* pDevice = g_GfxFramework.GetDevice();

    g_GraphicsResource.Setup(pDevice, CharMax);
    g_GraphicsResource.AcquireCommonSamplerSlot(AcquireSlotForSampler_, &g_GfxFramework);

    // フレームバッファコピーの設定をする。
    const nn::ui2d::Layout* pLayout = g_Viewer2Enabled ? g_ViewerScreen.GetScreen()->GetLayout() : g_LayoutViewer.GetLayout();
    if (nn::ui2d::CheckFrameBufferTextureDescriptorSlotRequired(pLayout))
    {
        InitializeCopyFrameBuffer();
    }

    // バッファの再セットアップ
    // ui2d 描画に使用するコンスタントバッファのためのメモリプールを作成
    nn::gfx::MemoryPoolInfo memPoolInfo;
    memPoolInfo.SetDefault();
    memPoolInfo.SetMemoryPoolProperty(nn::gfx::MemoryPoolProperty_CpuUncached | nn::gfx::MemoryPoolProperty_GpuCached);

    // BuildWithName でレイアウトデータをセットアップした際に BuildResultInformation に収集した
    // コンスタントバッファのサイズを満たすメモリプールを作成する。

    nn::gfx::GpuAccess  gpuAccessFlags = static_cast<nn::gfx::GpuAccess>(nn::gfx::GpuAccess_ConstantBuffer | nn::gfx::GpuAccess_VertexBuffer | nn::gfx::GpuAccess_IndexBuffer);
    nn::gfx::Buffer::InfoType info;
    info.SetDefault();
    info.SetGpuAccessFlags(gpuAccessFlags);

    size_t  bufferAlignment = nn::gfx::Buffer::GetBufferAlignment(pDevice, info);

    // キャプチャペインの 0 フレーム強制キャプチャのため 1 フレームに 2 回描画されることがあるため、確保するコンスタントバッファの量を 2 倍にする。
    const size_t constantBufferTotalSize = nn::util::align_up((buildResultInformation.requiredUi2dConstantBufferSize + buildResultInformation.requiredFontConstantBufferSize) * 2, bufferAlignment);
    const size_t memoryPoolSize = nn::util::align_up(constantBufferTotalSize, nn::gfx::MemoryPool::GetPoolMemorySizeGranularity(pDevice, memPoolInfo));
    const size_t MemoryPoolAlignment = nn::gfx::MemoryPool::GetPoolMemoryAlignment(pDevice, memPoolInfo);

    if (memoryPoolSize > 0)
    {
        const nns::gfx::GraphicsFramework::MemoryPoolType memoryPoolType = nns::gfx::GraphicsFramework::MemoryPoolType_ConstantBuffer;

        nn::gfx::MemoryPool* pMemoryPool = g_GfxFramework.GetMemoryPool(memoryPoolType);
        g_offsetToUi2dDrawMemoryPool = g_GfxFramework.AllocatePoolMemory(memoryPoolType, memoryPoolSize * nn::ui2d::ConstantBufferCount, MemoryPoolAlignment);

        // コンスタントバッファを作成する。
        // レイアウトとレイアウト内で使用されるフォント用のコンスタントバッファを共有とする。
        if (constantBufferTotalSize > 0)
        {
            nn::font::GpuBuffer::InitializeArg  arg;

            arg.SetGpuAccessFlag(gpuAccessFlags);
                arg.SetBufferSize(constantBufferTotalSize);
            arg.SetBufferCount(nn::ui2d::ConstantBufferCount);
                arg.SetMemoryPool(pMemoryPool);
                arg.SetMemoryPoolOffset(g_offsetToUi2dDrawMemoryPool);
            arg.SetAllocator(AllocFunction_, NULL);
            arg.SetAllocateSyncFlag(false);

            // フォント描画のためのコンスタントバッファを作成
            g_Ui2dConstantBuffer.Initialize(pDevice, arg);

            drawInfo.SetUi2dConstantBuffer(&g_Ui2dConstantBuffer);
            drawInfo.SetFontConstantBuffer(&g_Ui2dConstantBuffer);
        }
        else
        {
            drawInfo.SetUi2dConstantBuffer(NULL);
            drawInfo.SetFontConstantBuffer(NULL);
        }

        // キャプチャペインでキャプチャされるテクスチャを設定する
        drawInfo.SetFramebufferTexture(g_GfxFramework.GetColorBuffer(), g_GfxFramework.GetDisplayWidth(), g_GfxFramework.GetDisplayHeight());

        // コンバイナユーザーシェーダで描画として利用する用途として、フレームバッファ用のテクスチャデスクリプタ、及びサンプラデスクリプタを設定します。
        if (g_pColorCopyDescSlot)
        {
            drawInfo.SetFramebufferTextureDescriptorSlot(g_pColorCopyDescSlot);
            drawInfo.SetFramebufferSamplerDescriptorSlot(&drawInfo.GetGraphicsResource()->GetSamplerDescriptorSlot(
                nn::ui2d::TexWrap::TexWrap_Repeat,
                nn::ui2d::TexWrap::TexWrap_Repeat,
                nn::ui2d::TexFilter::TexFilter_Linear,
                nn::ui2d::TexFilter::TexFilter_Linear));
        }

        // ビューポートとシザー設定のデフォルト値を設定する。
        nn::gfx::ViewportStateInfo  viewportInfo;
        viewportInfo.SetDefault();
        viewportInfo.SetWidth(static_cast<float>(g_GfxFramework.GetDisplayWidth()));
        viewportInfo.SetHeight(static_cast<float>(g_GfxFramework.GetDisplayHeight()));

        nn::gfx::ScissorStateInfo  scissorInfo;
        scissorInfo.SetDefault();
        scissorInfo.SetWidth(g_GfxFramework.GetDisplayWidth());
        scissorInfo.SetHeight(g_GfxFramework.GetDisplayHeight());
        drawInfo.SetDefaultViewportScissorInfo(viewportInfo, scissorInfo);

        drawInfo.SetDefaultColorTargetView(g_GfxFramework.GetColorTargetView());
    }

    if (g_Viewer2Enabled)
    {
        nn::ui2d::viewer::Screen* pScreen = g_ViewerScreen.GetScreen();
        g_ScreenManager.RegisterAndCompleteAllGlyphsForScalableFontTextureCache(pScreen);
    }
    else
    {
        g_LayoutViewer.RegisterAndCompleteAllGlyphsForScalableFontTextureCache();
    }

    g_FirstUpdate = true;
}

//------------------------------------------------------------------------------
nn::ui2d::RenderTargetTextureLifetime CreateRenderTargetTexture(nn::gfx::Texture** pTexture, nn::gfx::TextureView** pTextureView, nn::gfx::DescriptorSlot** pSlot, const nn::ui2d::Layout* pLayout, const nn::gfx::TextureInfo& info, void* pUserData, nn::ui2d::RenderTargetTextureLifetime lifetimeHint)
{
    NN_UNUSED(pLayout);

    NN_SDK_ASSERT_NOT_NULL(pTexture);
    NN_SDK_ASSERT_NOT_NULL(pTextureView);
    NN_SDK_ASSERT_NOT_NULL(pSlot);

    nns::gfx::GraphicsFramework* pFramework = static_cast<nns::gfx::GraphicsFramework*>(pUserData);
    nn::gfx::Device*    pDevice = pFramework->GetDevice();

    const size_t size = nn::gfx::Texture::CalculateMipDataSize(pDevice, info);
    const size_t mipDataAlignment = nn::gfx::Texture::CalculateMipDataAlignment(pDevice, info);

    *pTexture = static_cast<nn::gfx::Texture*>(g_pAllocator->Alloc(sizeof(nn::gfx::Texture)));
    new(*pTexture) nn::gfx::Texture();

    // テクスチャの初期化
    switch (lifetimeHint)
    {
    case nn::ui2d::RenderTargetTextureLifetime_Layout:
        // nn::gfx::ImageFormat_R8_Unorm のような 1 要素で中途半端なサイズのテクスチャを作成することがあるため、テクスチャ作成時のアライメントを調整する必要があります。
        g_offsetToRenderTargetTextureMemoryPoolStatic = nn::util::align_up(g_offsetToRenderTargetTextureMemoryPoolStatic, mipDataAlignment);
        (*pTexture)->Initialize(pDevice, info, pFramework->GetMemoryPool(nns::gfx::GraphicsFramework::MemoryPoolType_RenderTarget), g_offsetToRenderTargetTextureMemoryPoolStatic, size);
        g_offsetToRenderTargetTextureMemoryPoolStatic += size;
        // キャプチャテクスチャは2枚組でビューアーでは 1 フレームで２回キャプチャすることがあることから、確保サイズの 1/4 が実際にユーザーから見えるメモリ量となる
        NN_SDK_ASSERT(g_offsetToRenderTargetTextureMemoryPoolStatic - g_offsetToRenderTargetTextureMemoryPoolHead < StaticCaptureTextureMemorySize,
            "Failed to allocate the static capture texture memory. The LayoutViewer can't preview over %d MB for static capture texture memory.",
            StaticCaptureTextureMemorySize / (1024 * 1024) / 4);
        break;
    case nn::ui2d::RenderTargetTextureLifetime_OneFrame:
        // nn::gfx::ImageFormat_R8_Unorm のような 1 要素で中途半端なサイズのテクスチャを作成することがあるため、テクスチャ作成時のアライメントを調整する必要があります。
        g_offsetToRenderTargetTextureMemoryPoolDynamic = nn::util::align_up(g_offsetToRenderTargetTextureMemoryPoolDynamic, mipDataAlignment);
        (*pTexture)->Initialize(pDevice, info, pFramework->GetMemoryPool(nns::gfx::GraphicsFramework::MemoryPoolType_RenderTarget), g_offsetToRenderTargetTextureMemoryPoolDynamic, size);
        g_offsetToRenderTargetTextureMemoryPoolDynamic += size;
        // キャプチャテクスチャは2枚組でビューアーでは 1 フレームで２回キャプチャすることがあることから、確保サイズの 1/4 が実際にユーザーから見えるメモリ量となる
        NN_SDK_ASSERT(g_offsetToRenderTargetTextureMemoryPoolDynamic - g_offsetToRenderTargetTextureMemoryPoolHead < StaticCaptureTextureMemorySize + DynamicCaptureTextureMemorySize,
            "Failed to allocate the dynamic capture texture memory. The LayoutViewer can't preview over %d MB for static capture texture memory.",
            DynamicCaptureTextureMemorySize / (1024 * 1024) / 4);
        break;
    default:
        break;
    }

    // TextureView
    {
        nn::gfx::TextureView::InfoType textureViewInfo;
        textureViewInfo.SetDefault();
        textureViewInfo.SetImageDimension(nn::gfx::ImageDimension_2d);
        textureViewInfo.EditSubresourceRange().EditArrayRange().SetArrayLength(1);
        textureViewInfo.EditSubresourceRange().EditArrayRange().SetBaseArrayIndex(0);
        textureViewInfo.SetImageFormat(info.GetImageFormat());
        textureViewInfo.SetTexturePtr(*pTexture);

        *pTextureView = static_cast<nn::gfx::TextureView*>(g_pAllocator->Alloc(sizeof(nn::gfx::TextureView)));
        new(*pTextureView) nn::gfx::TextureView();

        (*pTextureView)->Initialize(pDevice, textureViewInfo);
    }

    *pSlot = static_cast<nn::gfx::DescriptorSlot*>(g_pAllocator->Alloc(sizeof(nn::gfx::DescriptorSlot)));
    new(*pSlot) nn::gfx::DescriptorSlot();

    RegisterSlotForTexture_(*pSlot, **pTextureView, pFramework);

    return lifetimeHint;
}

//------------------------------------------------------------------------------
void DestroyRenderTargetTexture(nn::gfx::Texture* pTexture, nn::gfx::TextureView* pTextureView, nn::gfx::DescriptorSlot* pSlot, const nn::ui2d::Layout* pLayout, void* pUserData, nn::ui2d::RenderTargetTextureLifetime lifetimeHint)
{
    NN_UNUSED(pLayout);
    NN_UNUSED(lifetimeHint);

    nns::gfx::GraphicsFramework* pFramework = static_cast<nns::gfx::GraphicsFramework*>(pUserData);
    nn::gfx::Device*    pDevice = pFramework->GetDevice();

    UnregisterSlotForTexture_(pSlot, *pTextureView, pUserData);
    pTextureView->Finalize(pDevice);
    pTexture->Finalize(pDevice);

    g_pAllocator->Free(pSlot);
    g_pAllocator->Free(pTextureView);
    g_pAllocator->Free(pTexture);
}


//------------------------------------------------------------------------------
static void WaitIfNeeded_(bool is30Fps)
{
#if defined( NN_BUILD_CONFIG_OS_WIN32 )
    // 30 FPS の場合は、 presentInterval が利用できない状況が多いようなので、フレーム待ちをせずに時間を計って待機しています。
    if(is30Fps)
    {
        nn::TimeSpan frameTime = nn::TimeSpan::FromMicroSeconds( static_cast<int64_t>( 1000000 / 30.f));
        for(;;)
        {
            nn::TimeSpan dt = (nn::os::GetSystemTick() - g_LastUpdateTime).ToTimeSpan();
            if(dt > frameTime)
            {
                break;
            }
            else if ( dt + nn::TimeSpan::FromMilliSeconds( 10 ) > frameTime )
            {
                // 残り10.0msec以下のときはSleep(0).
                // Sleep(1)でも1.5msecぐらいかかっているようなので……
                Sleep( 0 );
            }
            else
            {
                // 残り10.0msec以上ならSleep(1)で下位スレッドに処理を回す.
                timeBeginPeriod( 1 );
                Sleep( 1 );
                timeEndPeriod( 1 );
            }
        }
    }

    g_LastUpdateTime = nn::os::GetSystemTick();
#endif
}

//------------------------------------------------------------------------------
static int GetPresetInterval_(bool is60Fps)
{
#if defined( NN_BUILD_CONFIG_OS_WIN32 )
    NN_UNUSED(is60Fps);
    // 30 FPS の場合は、 presentInterval が利用できない状況が多いようなので、presentIntervalは利用しない（WaitIfNeeded_も参照）。
    return 1;
#else
    return is60Fps ? 1 : 2;
#endif
}

//-------------------------------------------------------------------------
static void InitFileDeviceManager_()
{
    // ファイルシステムのアロケータ設定
    nn::fs::SetAllocator(Allocate, Deallocate);

    nn::Result ret = nn::fs::MountHostRoot();
    if (ret.IsFailure())
    {
        NN_LOG("Failed to mount host root.\n");
    }
}

//-------------------------------------------------------------------------
static void FinalizeFileDeviceManager_()
{
    nn::fs::UnmountHostRoot();
}

//-----------------------------------------------------------------------------
static void InitLayoutViewer_(const char* nwRootPath, float screenW, float screenH, bool isLinearMode)
{
    NN_UNUSED(nwRootPath);

    nn::gfx::Device* pDevice = g_GfxFramework.GetDevice();

    nn::ui2d::Initialize(AllocFunction_, FreeFunction_, NULL);
    nn::ui2d::Layout::SetRenderTargetTextureResourceManagementCallback(CreateRenderTargetTexture, DestroyRenderTargetTexture, &g_GfxFramework);

    g_pAllocator->SetMemoryLeakCheckingEnabled(true);

    g_GraphicsResource.Setup(pDevice, CharMax);
    g_GraphicsResource.AcquireCommonSamplerSlot(AcquireSlotForSampler_, &g_GfxFramework);


    // プレビュー中のデータサイズを計算するために、リソースサイズを記憶しておきます。
    g_GraphicsResourceHeapSize = g_pAllocator->GetOccupiedSize();

    // ツールコネクタでメモリを確保する分はリークチェックに含めないようにする
    g_pAllocator->SetMemoryLeakCheckingEnabled(false);
    if (g_Viewer2Enabled)
    {
        nn::ui2d::viewer::ViewerScreen::InitializeArg arg;

        arg.SetDefault();
        arg.mainViewportSize.Set(screenW, screenH);
        arg.loadFunction = ViewerLoad_;
        arg.unloadFunction = ViewerUnload_;

        // ScreenManager の初期化
        {
            auto& screenManager = g_ScreenManager;

            screenManager.pGraphicsResource = &g_GraphicsResource;

            screenManager.m_RegisterTextureViewSlot = RegisterSlotForTexture_;
            screenManager.m_UnregisterTextureViewSlot = UnregisterSlotForTexture_;
            screenManager.m_pUserDataForDescriptorSlotAllocator = &g_GfxFramework;

            screenManager.m_ScalableFontManager.SetTextureCacheParamaters(
                4096,
                4096,
                1024 * 1024 * 4,
                1024 * 1024
            );
        }

        nn::ui2d::viewer::Screen* pScreen = g_ScreenManager.CreateScreen();
        NN_SDK_ASSERT_NOT_NULL(pScreen);
        g_ViewerScreen.Initialize(*pScreen, arg);

        //  データ読み込み後にコンスタントバッファを再作成するためにコールバックを設定する。
        g_ViewerScreen.SetPreReloadLayoutDataCallback(PreReloadLayoutDataCallback);
        g_ViewerScreen.SetPostReloadLayoutDataCallback(PostReloadLayoutDataCallback);
        g_ViewerScreen.SetSrgbWriteEnabled(isLinearMode);
    }
    else
    {
        nn::ui2d::viewer::Viewer::InitializeArg arg;

        arg.SetDefault();
        arg.pGraphicsResource = &g_GraphicsResource;
        arg.mainViewportSize.Set(screenW, screenH);
        arg.loadFunction = ViewerLoad_;
        arg.unloadFunction = ViewerUnload_;
        arg.registerTextureViewSlot = RegisterSlotForTexture_;
        arg.unregisterTextureViewSlot = UnregisterSlotForTexture_;
        arg.pUserDataForDescriptorSlotAllocator = &g_GfxFramework;
        arg.scalableFontTextureCacheHeight = 4096;
        arg.scalableFontTextureCacheWidth = 4096;
        arg.scalableFontWorkMemorySize = 1024 * 1024 * 4;
        arg.scalableFontNoPlotWorkMemorySize = 1024 * 1024;

        g_LayoutViewer.Initialize(arg);
        //  データ読み込み後にコンスタントバッファを再作成するためにコールバックを設定する。
        g_LayoutViewer.SetPreReloadLayoutDataCallback(PreReloadLayoutDataCallback);
        g_LayoutViewer.SetPostReloadLayoutDataCallback(PostReloadLayoutDataCallback);
        g_LayoutViewer.SetSrgbWriteEnabled(isLinearMode);
    }

    g_pAllocator->SetMemoryLeakCheckingEnabled(true);
}

//-----------------------------------------------------------------------------
static void FinalizeLayoutViewer_(nn::gfx::Device* pDevice)
{
    // 使用中のバッファの終了処理
    if (g_Ui2dConstantBuffer.IsInitialized())
    {
        g_Ui2dConstantBuffer.Finalize(pDevice, FreeFunction_, NULL);
    }

    // 描画に使用したコンスタントバッファ領域をフレームワークへ返却
    if (g_offsetToUi2dDrawMemoryPool != NULL)
    {
        const nns::gfx::GraphicsFramework::MemoryPoolType memoryPoolType = nns::gfx::GraphicsFramework::MemoryPoolType_ConstantBuffer;
        g_GfxFramework.FreePoolMemory(memoryPoolType, g_offsetToUi2dDrawMemoryPool);
        g_offsetToUi2dDrawMemoryPool = NULL;
    }

    g_GraphicsResource.ReleaseCommonSamplerSlot(ReleaseSlotForSampler_, &g_GfxFramework);
    g_GraphicsResource.Finalize(pDevice);

    if (g_Viewer2Enabled)
    {
        g_ViewerScreen.Finalize(pDevice);
    }
    else
    {
        g_LayoutViewer.Finalize(pDevice);
    }
}

//------------------------------------------------------------------------------
//      Font 初期化処理
//------------------------------------------------------------------------------
const void GetFontLoadPath_(char* pFontLoadPath)
{
    char rootPathBuff[AppPathLengthMax];

    // 注意：正式なファイルアクセス方法が整備されるまでのワークアラウンドです。
    // 第一引数から、実行ファイルの host 上での位置を取得します。
    char contentPath[260];

    char* pExeFilePath = nn::os::GetHostArgv()[ 0 ];

    NN_LOG("---------------------------------------------------\n");
    NN_LOG("LayoutViewer:exe_path=");
    NN_LOG("%s\n", pExeFilePath);
    NN_LOG("---------------------------------------------------\n");

    char* pEnd = strrchr( pExeFilePath, '\\' );
    NN_ASSERT_NOT_NULL( pEnd );

    // 実行ファイルからの相対パスで、リソースフォルダを定義します。
    size_t length = pEnd - pExeFilePath + 1;
    nn::util::Strlcpy( contentPath, pExeFilePath, static_cast<int>(length + 1) );
    strcpy( contentPath + length, "\\data" );

    nn::util::Strlcpy(rootPathBuff, contentPath, AppPathLengthMax);

    // 対象API によって利用するフォントを変更します。
    // #include <nn/gfx/gfx_Variation-api.gl.h> として、 nn::gfx::ApiTypeGl が常に参照できるようにしています。
    const char* pFontName = (nn::gfx::Device::Target::Type::value == nn::gfx::ApiTypeGl::value) ? "nintendo_NTLG-DB_002_Linear.bffnt" : "nintendo_NTLG-DB_002_NX.bffnt";

    sprintf( pFontLoadPath, "%s\\%s", rootPathBuff, pFontName );
}

//------------------------------------------------------------------------------
//      Font 初期化処理
//------------------------------------------------------------------------------
void InitializeFont_(const uint32_t  CharCountMax, SimpleAllocator* allocator)
{
    nn::gfx::Device* pDevice = g_GfxFramework.GetDevice();

    // 文字列表示用バッファを管理するクラスの初期化
    {
        nn::font::DispStringBuffer::InitializeArg arg;
        arg.SetCharCountMax(CharCountMax);
        arg.SetShadowEnabled( true );

        nn::gfx::MemoryPoolInfo memPoolInfo;
        memPoolInfo.SetDefault();
        memPoolInfo.SetMemoryPoolProperty( nn::gfx::MemoryPoolProperty_CpuUncached | nn::gfx::MemoryPoolProperty_GpuCached );

        const size_t constantBufferSize = nn::font::DispStringBuffer::GetRequiredConstantBufferSize(pDevice, arg);
         // 頂点バッファの後ろにコンスタントバッファを置くため
        // コンスタントバッファの先頭アドレスがアライメント制限にそろうようにバッファのサイズを調整する。
        nn::gfx::Buffer::InfoType info;
        info.SetDefault();
        info.SetGpuAccessFlags(nn::gfx::GpuAccess_ConstantBuffer);
        info.SetSize(constantBufferSize);
        const size_t memoryPoolSize = nn::util::align_up(constantBufferSize * nn::ui2d::ConstantBufferCount, nn::gfx::MemoryPool::GetPoolMemorySizeGranularity(pDevice, memPoolInfo));
        const size_t MemoryPoolAlignment = nn::gfx::MemoryPool::GetPoolMemoryAlignment(pDevice, memPoolInfo);

        const nns::gfx::GraphicsFramework::MemoryPoolType memoryPoolType = nns::gfx::GraphicsFramework::MemoryPoolType_ConstantBuffer;

        nn::gfx::MemoryPool* pMemoryPool = g_GfxFramework.GetMemoryPool(memoryPoolType);
        g_offsetToHudDrawMemoryPool = g_GfxFramework.AllocatePoolMemory(memoryPoolType, memoryPoolSize * nn::ui2d::ConstantBufferCount, MemoryPoolAlignment);



        nn::font::GpuBuffer::InitializeArg  gpuBufferArg;

        // フォント描画のためのコンスタントバッファを作成する。
        gpuBufferArg.SetGpuAccessFlag( nn::gfx::GpuAccess_ConstantBuffer );
        gpuBufferArg.SetBufferSize( constantBufferSize );
        gpuBufferArg.SetBufferCount( nn::ui2d::ConstantBufferCount );
        gpuBufferArg.SetMemoryPool(pMemoryPool);
        gpuBufferArg.SetMemoryPoolOffset(g_offsetToHudDrawMemoryPool);
        gpuBufferArg.SetAllocator( AllocFunction_, NULL );
        gpuBufferArg.SetAllocateSyncFlag( false );
        g_HudFontConstantBuffer.Initialize(pDevice, gpuBufferArg );

        const size_t  drawBufSize = nn::font::DispStringBuffer::GetRequiredDrawBufferSize( arg );
        if ( drawBufSize > 0 )
        {
            arg.SetDrawBuffer( allocator->Alloc( drawBufSize ) );
        }

        arg.SetConstantBuffer(&g_HudFontConstantBuffer);

        g_DispStringBuf.Initialize(pDevice, arg );
    }

    // フォント矩形描画クラスの初期化
    {
        const size_t  workSize   = nn::font::RectDrawer::GetWorkBufferSize(pDevice, CharMax );
        g_pRectDrawerWorkBuf = allocator->Alloc( workSize, nn::font::RectDrawer::GetWorkBufferAlignment() );
        g_RectDrawer.Initialize(pDevice, g_pRectDrawerWorkBuf, CharMax );
        g_RectDrawer.AcquireCommonSamplerSlot(AcquireSlotForSampler_, &g_GfxFramework);
    }

    // フォントの読み込み
    {
        char fontPath[ AppPathLengthMax ];
        GetFontLoadPath_(fontPath);

        size_t  fontSize = 0;
        if(!nns::nac::FileLoad( &g_pFontResource, &fontSize, fontPath, ".", allocator, nn::font::ResourceAlignment ))
        {
            NN_LOG("Fail to load font ! (%s)\n", fontPath);
            NN_ASSERT(false);
        }

        g_ResFont.SetResource(pDevice, g_pFontResource );
        g_ResFont.RegisterTextureViewToDescriptorPool( RegisterSlotForTexture_, &g_GfxFramework);
        g_ResFont.SetKerningEnabled(false);
    }

    // テキストライターの初期化
    {
        const float FontScale = 0.8f;
        const nn::util::Unorm8x4 white = { { std::numeric_limits<uint8_t>::max(), std::numeric_limits<uint8_t>::max(), std::numeric_limits<uint8_t>::max(), std::numeric_limits<uint8_t>::max() } };
        g_Writer.SetDispStringBuffer( &g_DispStringBuf );
        g_Writer.SetFont( &g_ResFont );
        g_Writer.SetScale( FontScale, FontScale );
        g_Writer.SetTextColor( white, white );
        g_Writer.EnableFixedWidth(true);
        g_Writer.SetFixedWidth(g_ResFont.GetWidth() * FontScale * 0.6f);
    }

    // プロジェクション行列行列の初期化
    nn::util::MatrixOrthographicOffCenterRightHanded(&g_ProjectionFont, 0, static_cast<float>( g_ScreenW ), static_cast<float>( g_ScreenH ), 0, 0.0f, 300.0f);
}

//------------------------------------------------------------------------------
//      Font 破棄処理
//------------------------------------------------------------------------------
void FinalizeFont_(SimpleAllocator* allocator)
{
    nn::gfx::Device* pDevice = g_GfxFramework.GetDevice();

    g_ResFont.UnregisterTextureViewFromDescriptorPool(UnregisterSlotForTexture_, &g_GfxFramework);
    void* memory = g_ResFont.RemoveResource(pDevice);
    allocator->Free( memory );

    void* drawBuffer = g_DispStringBuf.GetDrawBuffer();
    allocator->Free( drawBuffer );
    g_DispStringBuf.Finalize(pDevice);

    g_HudFontConstantBuffer.Finalize(pDevice, FreeFunction_, NULL);

    const nns::gfx::GraphicsFramework::MemoryPoolType memoryPoolType = nns::gfx::GraphicsFramework::MemoryPoolType_ConstantBuffer;
    g_GfxFramework.FreePoolMemory(memoryPoolType, g_offsetToHudDrawMemoryPool);

    g_RectDrawer.UnregisterSamplerFromDescriptorPool(UnregisterSlotForSampler_, &g_GfxFramework);
    g_RectDrawer.Finalize(pDevice);
}

//-----------------------------------------------------------------------------
//  PrimitiveRenderer 初期化処理
void InitializePrimitiveRenderer_( nn::gfx::Device* pDevice )
{
    NN_SDK_ASSERT_NOT_NULL( pDevice );

    nns::gfx::PrimitiveRenderer::RendererInfo info;
    info.SetDefault();
    info.SetAllocator(AllocFunction_, NULL);

    // ダブルバッファで運用する場合は、SetMultiBufferQuantity で 2 を指定し、
    // プリミティブレンダラ内部のバッファをダブルにしたうえで、
    // g_pPrimitiveRenderer->Update(); で利用するバッファを選択( 0 -> 1 -> 0 -> 1 )する
    info.SetMultiBufferQuantity( 1 );

    // PrimitiveRendererのインスタンス
    g_pPrimitiveRenderer = nns::gfx::PrimitiveRenderer::CreateRenderer( pDevice, info);

    g_pPrimitiveRenderer->SetScreenWidth(g_Width);
    g_pPrimitiveRenderer->SetScreenHeight(g_Height);
}

//------------------------------------------------------------------------------
//  PrimitiveRenderer 破棄処理
void FinalizePrimitiveRenderer_( nn::gfx::Device* pDevice)
{
    NN_SDK_ASSERT_NOT_NULL( pDevice );
    nns::gfx::PrimitiveRenderer::DestroyRenderer( g_pPrimitiveRenderer, pDevice, FreeFunction_, NULL);
}

static nn::ui2d::viewer::Viewer::InputDeviceState CalcForInput_()
{
    // 入力デバイスを更新します。
    {
        g_Pad.Update();
        g_Mouse.Update();
        g_ControllerManager.Update();
    }

    if (g_Pad.IsHold(nns::nac::Pad::MASK_LEFT))
    {
        if (g_PadLeftRepeat < PadRepeatFrames)
        {
            g_PadLeftRepeat += 1;
        }
    }
    else
    {
        g_PadLeftRepeat = 0;
    }

    if (g_Pad.IsHold(nns::nac::Pad::MASK_RIGHT))
    {
        if (g_PadRightRepeat < PadRepeatFrames)
        {
            g_PadRightRepeat += 1;
        }
    }
    else
    {
        g_PadRightRepeat = 0;
    }

    bool isLeftHold = (g_PadLeftRepeat >= PadRepeatFrames) || g_Pad.IsTrigger(nns::nac::Pad::MASK_LEFT);
    bool isRgihtHold = (g_PadRightRepeat >= PadRepeatFrames) || g_Pad.IsTrigger(nns::nac::Pad::MASK_RIGHT);

    nn::ui2d::viewer::Viewer::InputDeviceState inputDeviceState;
    memset(&inputDeviceState, 0, sizeof(nn::ui2d::viewer::Viewer::InputDeviceState));

#if defined(NN_BUILD_CONFIG_OS_WIN32)
    inputDeviceState.isMousePointerOn = g_Mouse.IsPointerOn();
    inputDeviceState.pointerX = g_Mouse.GetPointerX();
    inputDeviceState.pointerY = g_Mouse.GetPointerY();

    inputDeviceState.isTrigMouseLButton = g_Mouse.IsTrig(g_Mouse.MASK_LBUTTON);
    inputDeviceState.isReleaseMouseLButton = g_Mouse.IsRelease(g_Mouse.MASK_LBUTTON);
#else
    inputDeviceState.isMousePointerOn = false;

    nns::hid::TouchScreen* pTouchScreen = reinterpret_cast<nns::hid::TouchScreen*>(g_ControllerManager.GetController(nns::hid::ControllerId_TouchScreen, 0));
    const std::vector<nns::hid::TouchScreen::TouchState>& states = pTouchScreen->GetTouchStates();
    for (size_t i = 0; i < states.size(); i++)
    {
        // 複数のタッチがあった場合は最初の 1 つだけを処理する
        const nns::hid::TouchScreen::TouchState& state = states[i];
        inputDeviceState.pointerX = state.position.x * (g_ScreenW / 1280.0f);
        inputDeviceState.pointerY = state.position.y * (g_ScreenH / 720.0f);
        inputDeviceState.isMousePointerOn = true;

        inputDeviceState.isTrigMouseLButton = state.IsBegan();
        inputDeviceState.isReleaseMouseLButton = state.IsEnded();

        break;
    }

#endif // defined(NN_BUILD_CONFIG_OS_WIN32)

    inputDeviceState.isHoldLeft = isLeftHold;
    inputDeviceState.isHoldRight = isRgihtHold;

    inputDeviceState.isTrigUp = g_Pad.IsTrigger(nns::nac::Pad::MASK_UP);
    inputDeviceState.isTrigDown = g_Pad.IsTrigger(nns::nac::Pad::MASK_DOWN);

    inputDeviceState.isTrigA = g_Pad.IsTrigger(nns::nac::Pad::MASK_A);
    inputDeviceState.isTrigB = g_Pad.IsTrigger(nns::nac::Pad::MASK_B);
    inputDeviceState.isTrigX = g_Pad.IsTrigger(nns::nac::Pad::MASK_X);
    inputDeviceState.isTrigY = g_Pad.IsTrigger(nns::nac::Pad::MASK_Y);

    return inputDeviceState;
}

//-----------------------------------------------------------------------------
static void Calc_()
{
    nn::gfx::Device* pDevice = g_GfxFramework.GetDevice();
    nn::ui2d::viewer::Viewer::InputDeviceState inputDeviceState = CalcForInput_();

#if defined(NN_BUILD_CONFIG_OS_WIN32)
    if (g_LayoutViewer.IsScreenShotDone())
    {
        return;
    }

    g_LayoutViewer.UpdateSystem(pDevice);

    nn::gfx::MemoryPool* pMemoryPool = g_GfxFramework.GetMemoryPool(nns::gfx::GraphicsFramework::MemoryPoolType_ConstantBuffer);
    g_LayoutViewer.UpdateForScreenShot(
        g_FrameIndex,
        pDevice,
        pMemoryPool,
        &g_offsetToScreenShotMemoryPool,
        AllocFunction_, NULL);
#else
    g_LayoutViewer.UpdateSystem(pDevice);
#endif

    g_LayoutViewer.UpdateInputs(pDevice, inputDeviceState);
    g_LayoutViewer.UpdateMenu();

    // 初回フレーム更新は Draw の中で初回フレームキャプチャテクスチャの処理と一緒に実行します。
    if (!g_FirstUpdate)
    {
        g_LayoutViewer.UpdateLayout(g_FrameIndex);
        ++g_FrameIndex;
        g_FrameIndex %= nn::ui2d::ConstantBufferCount;
    }
}

//-----------------------------------------------------------------------------
static void Calc2_()
{
    nn::gfx::Device* pDevice = g_GfxFramework.GetDevice();

    g_ViewerScreen.UpdateSystem(pDevice);

    nn::ui2d::viewer::Viewer::InputDeviceState inputDeviceState = CalcForInput_();
    g_ViewerScreen.UpdateInputs(pDevice, inputDeviceState);
    g_ViewerScreen.UpdateMenu();

    g_ViewerScreen.GetScreen()->UpdateLayout(g_FrameIndex);

    ++g_FrameIndex;
    g_FrameIndex %= nn::ui2d::ConstantBufferCount;
}

//------------------------------------------------------------------------------
//      HUD 更新
void CalcViewerHud_()
{
    if(g_Viewer2Enabled ? g_ViewerScreen.IsHudVisible() : g_LayoutViewer.IsHudVisible())
    {
        g_Writer.SetCursor( 0, 0 );
        g_Writer.StartPrint();
        {
            if (g_Viewer2Enabled)
            {
                g_ViewerScreen.WriteHud(g_Writer);
            }
            else
            {
                g_LayoutViewer.WriteHud(g_Writer);
            }

            g_Writer.Printf("\n");
            g_Writer.Printf(" Memory: %.1f KB\n", (g_pAllocator->GetOccupiedSize() - g_GraphicsResourceHeapSize) / 1024.f);
            g_Writer.Printf(" HDR: %s\n", g_HDRWriteEnabled ? "Enabled" : "Disabled" );
        }
        g_Writer.EndPrint();
    }else{
        g_Writer.StartPrint();
        g_Writer.EndPrint();
    }
}

//-----------------------------------------------------------------------------

static void DrawViewerHud_(nn::gfx::CommandBuffer* pCmdBuffer)
{
    pCmdBuffer->SetBlendState(g_GfxFramework.GetBlendState(
        nns::gfx::GraphicsFramework::BlendStateType_Alpha));
    pCmdBuffer->SetRasterizerState(g_GfxFramework.GetRasterizerState(
        nns::gfx::GraphicsFramework::RasterizerStateType_FillSolid_CullNone));

    // 負荷メーター
    {
        //負荷メーターを描画
        nn::perf::CpuMeter* pFrameMeter = NN_PERF_GET_FRAME_METER();
        nn::util::Float2 pos = NN_UTIL_FLOAT_2_INITIALIZER(32.f, g_Height - g_MeterDrawer.GetHeight(pFrameMeter) - g_MeterDrawer.GetBarHeight());
        g_MeterDrawer.SetPosition(pos);
        g_MeterDrawer.SetBarHeight(16);
        g_MeterDrawer.SetWidth(g_Width - 64.f);
        g_MeterDrawer.Draw(pCmdBuffer, g_pPrimitiveRenderer, pFrameMeter);
        g_MeterDrawer.SetDepthCountMax(1);
        g_MeterDrawer.SetAdjustFrameCount(10);

        g_Writer.SetCursor(pos.x + 4.f, pos.y - 0.f);
        g_Writer.Printf("Calc(%6.3f msec)",
            NN_PERF_GET_ELAPSED_TIME_CPU(NULL, "Calc", 0).GetMicroSeconds() / 1000.f);

        g_Writer.SetCursor(pos.x + 4.f, pos.y - 0.f + 16.f);
        g_Writer.Printf("Draw(%6.3f msec)",
            NN_PERF_GET_ELAPSED_TIME_CPU(NULL, "Draw", 0).GetMicroSeconds() / 1000.f);

        g_Writer.SetCursor(pos.x + 4.f, pos.y - 0.f + 16.f + 16.f);
        g_Writer.Printf("Gpu (%6.3f msec)",
            NN_PERF_GET_ELAPSED_TIME_GPU("Draw", 0).GetMicroSeconds() / 1000.f);
    }

    // ビューアー HUD
    {
        nn::util::MatrixT4x3fType translate;
        nn::util::MatrixIdentity(&translate);
        const nn::util::Unorm8x4 white = { { std::numeric_limits<uint8_t>::max(), std::numeric_limits<uint8_t>::max(), std::numeric_limits<uint8_t>::max(), std::numeric_limits<uint8_t>::max() } };
        const nn::util::Unorm8x4 black = { { 0, 0, 0, std::numeric_limits<uint8_t>::max() } };
        nn::util::MatrixT4x3fType identity;
        nn::util::MatrixIdentity(&identity);

        // 背景色によって HUD が見えにくくなるため、影をつけて視認性をあげる。
        nn::font::ShadowParameter param;
        param.shadowUpperColor = black;
        param.shadowLowerColor = black;
        param.shadowOffset.x = 1.0f;
        param.shadowOffset.y = -1.0f;
        param.shadowScale.x = 1.0f;
        param.shadowScale.y = 1.0f;
        param.shadowItalicOffset = 0.0f;

        g_HudFontConstantBuffer.Map(g_HudFontFrameIndex);
        g_DispStringBuf.BuildConstantBuffer(
            g_ProjectionFont,
            &nn::font::ConstantBufferAdditionalContent()
            .SetViewMatrix(&identity)
            .SetLocalMatrix(&translate)
            .SetInterpolateWhite(white)
            .SetInterpolateAlpha(255)
            .SetInterpolateBlack(black)
            .SetShadowInterpolateBlack(black)
            .SetShadowInterpolateWhite(black)
            .SetShadowInterpolateAlpha(255)
            .SetShadowParam(&param)
        );

        g_HudFontConstantBuffer.Unmap();
        g_HudFontConstantBuffer.SetGpuAccessBufferIndex(g_HudFontFrameIndex);

        nn::util::Color4u8 colorDrawHUD;
        colorDrawHUD.SetLerp(nn::util::Color4u8::Gray(), nn::util::Color4u8::Magenta(), 0.1f);

        g_RectDrawer.Draw(*pCmdBuffer, g_DispStringBuf);
    }
}

//-----------------------------------------------------------------------------

static nn::gfx::SwapChain* GetSwapChain_(bool isSrgbWriteEnabled)
{
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
    // SwapChin を複数生成すると、後に生成したほうしか利用できない下位描画レイヤーの不具合があるため切り替え機能をOFFにし、強制的に リニアモードで動作します。
    return g_GfxFramework.GetSwapChain();
#else
    return isSrgbWriteEnabled ? g_GfxFramework.GetSwapChain() : &g_RGBSwapChain;
#endif
}

//-----------------------------------------------------------------------------

static void EndFrame_(nn::gfx::CommandBuffer* pCmdBuffer, const int cmdBufferIndex, nn::gfx::SwapChain* pSwapChain)
{
    // g_GfxFramework.EndFrame(cmdBufferIndex); 相当の処理
    // アプリケーションが用意したバッファをそのまま表示するにはフレームワークのカラーバッファへコピーする必要がある。
    // 無駄なコピーを避けるため EndFrame を呼ばず、EndFrame 内で行われているスキャンバッファへのコピーをアプリケーション側で実装する。
    // スキャンバッファへCopyToScanBufferのコピーコマンド生成
    {
        nn::gfx::ColorTargetView* pScanBufferView = pSwapChain->AcquireNextScanBufferView();
        // シェーダで裏のスキャンバッファにコピーする
        g_GlareEffect.CopyTexture(pCmdBuffer, g_pPrimitiveRenderer, pScanBufferView, g_ColorBufferDescSlot);
    }
    g_GfxFramework.GetRootCommandBuffer(cmdBufferIndex)->End();
}

//-----------------------------------------------------------------------------

static void Present_(nn::gfx::SwapChain* pSwapChain, int presetInterval)
{
    // g_GfxFramework.Present(); 相当の処理
    // Present に渡す SwapChain にフレームワークが持っているもの以外を渡したいため
    // GraphicsFramework::Present() 相当の処理をアプリケーション側で実装する。
    nn::gfx::Queue* queue = g_GfxFramework.GetQueue();
    queue->Present(pSwapChain, presetInterval);
    queue->Sync();
}

//-----------------------------------------------------------------------------
// コマンドを生成
static void Draw_()
{
    const int cmdBufferIndex = 0;

    g_GfxFramework.BeginFrame(cmdBufferIndex);
    NN_PERF_SET_COLOR_GPU( nn::util::Color4u8::Gray() );
    nn::gfx::Device*    pDevice = g_GfxFramework.GetDevice();
    nn::gfx::ColorTargetView* pTargetView = g_GfxFramework.GetColorTargetView();
    nn::gfx::DepthStencilView* pDepthStencilView = g_GfxFramework.GetDepthStencilView();
    nn::gfx::CommandBuffer* pCmdBuffer = g_GfxFramework.GetRootCommandBuffer(cmdBufferIndex);

    {
        NN_PERF_BEGIN_MEASURE_GPU( pCmdBuffer);

        {
            // フレームバッファのクリア
            {
                float bgColor[4];
                g_LayoutViewer.GetBackgroundColor(bgColor);

                pCmdBuffer->ClearColor(pTargetView, bgColor[0], bgColor[1], bgColor[2], bgColor[3], NULL);
                pCmdBuffer->ClearDepthStencil(pDepthStencilView, 1.0f, 0, nn::gfx::DepthStencilClearMode_DepthStencil, NULL);
            }

            // レンダーステートの設定
            pCmdBuffer->SetRenderTargets(1, &pTargetView, pDepthStencilView);

            // ビューポート・シザーの設定
            pCmdBuffer->SetViewportScissorState(g_GfxFramework.GetViewportScissorState());

            // ブレンドモード
            pCmdBuffer->SetBlendState(g_GfxFramework.GetBlendState(
                nns::gfx::GraphicsFramework::BlendStateType_Alpha));
        }

        g_pPrimitiveRenderer->Update(0);

        // 背景画像の描画
        if (g_LayoutViewer.IsBackgroundImageEnabled())
        {
            float   width = (float)g_LayoutViewer.GetBackgroundImageSize().width / 1280.0f * 2.0f;
            float   height = (float)g_LayoutViewer.GetBackgroundImageSize().height / 720.0f * 2.0f;
            nn::util::Vector3fType center = { -width * 0.5f, -height * 0.5f, 0.f };
            nn::util::Vector3fType size = { width, height, 0.0f };

            g_pPrimitiveRenderer->DrawQuad(
                pCmdBuffer,
                center,
                size,
                g_LayoutViewer.GetBackgroundImage(),
                g_GraphicsResource.GetSamplerDescriptorSlot(nn::ui2d::TexWrap_Clamp, nn::ui2d::TexWrap_Clamp, nn::ui2d::TexFilter_Linear, nn::ui2d::TexFilter_Linear));
        }

        pCmdBuffer->FlushMemory(nn::gfx::GpuAccess_ColorBuffer|nn::gfx::GpuAccess_Texture);
        g_offsetToRenderTargetTextureMemoryPoolDynamic = g_offsetToRenderTargetTextureMemoryPoolHead + StaticCaptureTextureMemorySize;

        // レイアウトビューアの描画
        // Draw(赤)
        // Gpu（マゼンダ）
        NN_PERF_SET_COLOR_INDEX(0, nn::util::Color4u8::Red() );
        NN_PERF_BEGIN_MEASURE_INDEX_NAME( 0, "Draw" );
        {
            NN_PERF_SET_COLOR_GPU( nn::util::Color4u8::Magenta() );
            NN_PERF_BEGIN_MEASURE_NAME_GPU( pCmdBuffer, "Draw" );
            {
                if (g_FirstUpdate)
                {
                    g_LayoutViewer.DrawFirstFrameCaptureTextureAndUpdateLayout(pDevice, *pCmdBuffer, g_FrameIndex);
                    g_FirstUpdate = false;
                    ++g_FrameIndex;
                    g_FrameIndex %= nn::ui2d::ConstantBufferCount;
                }

                CopyFrameBufferImage(pCmdBuffer);

                g_LayoutViewer.DrawCaptureTexture(g_GfxFramework.GetDevice(), *pCmdBuffer);

                // レンダーステートの設定
                pCmdBuffer->SetRenderTargets(1, &pTargetView, pDepthStencilView);
                pCmdBuffer->SetViewportScissorState(g_GfxFramework.GetViewportScissorState());
                pCmdBuffer->SetBlendState(g_GfxFramework.GetBlendState(
                    nns::gfx::GraphicsFramework::BlendStateType_Alpha));

                g_LayoutViewer.DrawLayout(*pCmdBuffer);
                g_LayoutViewer.DrawSystem(*pCmdBuffer);
            }
            NN_PERF_END_MEASURE_GPU(pCmdBuffer);
        }
        NN_PERF_END_MEASURE_INDEX(0);

        if (g_HDRWriteEnabled)
        {
            // ポストエフェクト(グレア)
            g_GlareEffect.Draw(pCmdBuffer, g_GfxFramework.GetColorTargetView(), g_pPrimitiveRenderer, &g_ColorBufferDescSlot);
        }

#if defined(NN_BUILD_CONFIG_OS_WIN32)
        // スクリーンショットを取らない状態で HDR -> LDR への変換は不必要な負荷になってしまうため
        // スクリーンショット機能の状態を確認して変換を行うかどうかを判断する。
        if (g_LayoutViewer.IsScreenShotReady())
        {
            if (g_HDRWriteEnabled)
            {
                // g_GlareEffect.Draw 内でステートが返されるためリセットする。
                pCmdBuffer->SetBlendState(g_GfxFramework.GetBlendState(
                    nns::gfx::GraphicsFramework::BlendStateType_Alpha));
                pCmdBuffer->SetRasterizerState(g_GfxFramework.GetRasterizerState(
                    nns::gfx::GraphicsFramework::RasterizerStateType_FillSolid_CullNone));

                // RGBA8 でキャプチャするためにキャプチャ用のカラーバッファへコピーする
                g_GlareEffect.CopyTexture(
                    pCmdBuffer,
                    g_pPrimitiveRenderer,
                    g_CaptureColorBuffer.GetColorTargetView(),
                    g_ColorBufferDescSlot);

                // 元のカラーバッファへレンダーターゲットを戻す
                pCmdBuffer->SetRenderTargets(1, &pTargetView, pDepthStencilView);

                // 内部でも State が Ready かどうか判断しているが、挙動を変更しないため PutCommand の外側での状態の判定に含めています。
                g_LayoutViewer.PutCommandScreenShot(pCmdBuffer, g_CaptureColorBuffer.GetTexture());
            }
            else
            {
                // 内部でも State が Ready かどうか判断しているが、挙動を変更しないため PutCommand の外側での状態の判定に含めています。
                g_LayoutViewer.PutCommandScreenShot(pCmdBuffer, g_GfxFramework.GetColorBuffer());
            }
        }
#endif

        if (g_LayoutViewer.IsHudVisible())
        {
            DrawViewerHud_(pCmdBuffer);
        }

        NN_PERF_END_MEASURE_GPU(pCmdBuffer);
    }

    nn::gfx::SwapChain* pSwapChain = GetSwapChain_(g_LayoutViewer.IsSrgbWriteEnabled());
    EndFrame_(pCmdBuffer, cmdBufferIndex, pSwapChain);

    // コマンドの実行
    g_GfxFramework.ExecuteCommand(cmdBufferIndex);

    // CPU 処理全体の計測終了
    {
        NN_PERF_END_MEASURE_INDEX(0);
        NN_PERF_END_MEASURE();
    }

    // 結果の表示
    Present_(pSwapChain, GetPresetInterval_(g_LayoutViewer.GetFps() == nn::ui2d::viewer::Viewer::Fps_60));

#ifdef NN_BUILD_CONFIG_OS_WIN32
    g_LayoutViewer.TakeScreenShotAfterCommandDone(FreeFunction_, NULL);
#endif
}// NOLINT(impl/function_size)


//-----------------------------------------------------------------------------
static void Draw2_()
{
    const int cmdBufferIndex = 0;

    g_GfxFramework.BeginFrame(cmdBufferIndex);
    NN_PERF_SET_COLOR_GPU(nn::util::Color4u8::Gray());
    nn::gfx::ColorTargetView* pTargetView = g_GfxFramework.GetColorTargetView();
    nn::gfx::DepthStencilView* pDepthStencilView = g_GfxFramework.GetDepthStencilView();
    nn::gfx::CommandBuffer* pCmdBuffer = g_GfxFramework.GetRootCommandBuffer(cmdBufferIndex);

    {
        NN_PERF_BEGIN_MEASURE_GPU(pCmdBuffer);

        {
            // フレームバッファのクリア
            {
                float bgColor[4];
                g_ViewerScreen.GetBackgroundColor(bgColor);

                pCmdBuffer->ClearColor(pTargetView, bgColor[0], bgColor[1], bgColor[2], bgColor[3], NULL);
                pCmdBuffer->ClearDepthStencil(pDepthStencilView, 1.0f, 0, nn::gfx::DepthStencilClearMode_DepthStencil, NULL);
            }

            // レンダーステート/ビューポート・シザー/ブレンドモード
            pCmdBuffer->SetRenderTargets(1, &pTargetView, pDepthStencilView);
            pCmdBuffer->SetViewportScissorState(g_GfxFramework.GetViewportScissorState());
            pCmdBuffer->SetBlendState(g_GfxFramework.GetBlendState(nns::gfx::GraphicsFramework::BlendStateType_Alpha));
        }

        g_pPrimitiveRenderer->Update(0);
        pCmdBuffer->FlushMemory(nn::gfx::GpuAccess_ColorBuffer | nn::gfx::GpuAccess_Texture);
        g_offsetToRenderTargetTextureMemoryPoolDynamic = g_offsetToRenderTargetTextureMemoryPoolHead + StaticCaptureTextureMemorySize;

        // レイアウトビューアの描画
        // Draw(赤)
        // Gpu（マゼンダ）
        NN_PERF_SET_COLOR_INDEX(0, nn::util::Color4u8::Red());
        NN_PERF_BEGIN_MEASURE_INDEX_NAME(0, "Draw");
        {
            NN_PERF_SET_COLOR_GPU(nn::util::Color4u8::Magenta());
            NN_PERF_BEGIN_MEASURE_NAME_GPU(pCmdBuffer, "Draw");

            auto* pScreen = g_ViewerScreen.GetScreen();
            if(pScreen != NULL)
            {
                pScreen->DrawCaptureTexture(g_GfxFramework.GetDevice(), *pCmdBuffer);

                // レンダーステートの設定
                pCmdBuffer->SetRenderTargets(1, &pTargetView, pDepthStencilView);
                pCmdBuffer->SetViewportScissorState(g_GfxFramework.GetViewportScissorState());
                pCmdBuffer->SetBlendState(g_GfxFramework.GetBlendState(nns::gfx::GraphicsFramework::BlendStateType_Alpha));


                pScreen->DrawLayout(*pCmdBuffer);
            }
            NN_PERF_END_MEASURE_GPU(pCmdBuffer);
        }
        NN_PERF_END_MEASURE_INDEX(0);

        if (g_HDRWriteEnabled)
        {
            // ポストエフェクト(グレア)
            g_GlareEffect.Draw(pCmdBuffer, g_GfxFramework.GetColorTargetView(), g_pPrimitiveRenderer, &g_ColorBufferDescSlot);
        }

        if (g_ViewerScreen.IsHudVisible())
        {
            DrawViewerHud_(pCmdBuffer);
        }

        NN_PERF_END_MEASURE_GPU(pCmdBuffer);
    }

    nn::gfx::SwapChain* pSwapChain = GetSwapChain_(g_ViewerScreen.IsSrgbWriteEnabled());
    EndFrame_(pCmdBuffer, cmdBufferIndex, pSwapChain);

    // コマンドの実行
    g_GfxFramework.ExecuteCommand(cmdBufferIndex);

    // CPU 処理全体の計測終了
    {
        NN_PERF_END_MEASURE_INDEX(0);
        NN_PERF_END_MEASURE();
    }

    // 結果の表示
    Present_(pSwapChain, GetPresetInterval_(g_ViewerScreen.GetFps() == nn::ui2d::viewer::ViewerScreen::Fps_60));
}

} // anonymous namespace

//-----------------------------------------------------------------------------

const char* GetOptionValue_(int argc, const char** argv, const char* name)
{
    for(int i = 0; i < argc; i++)
    {
        if(strcmp(argv[i], name) == 0)
        {
            if(i + 1 < argc)
            {
                return argv[i + 1];
            }else{
                return NULL;
            }
        }
    }

    return NULL;
}

//-----------------------------------------------------------------------------

const bool CheckOptionFlag_(int argc, const char** argv, const char* name)
{
    for(int i = 0; i < argc; i++)
    {
        if(strcmp(argv[i], name) == 0)
        {
            return true;
        }
    }

    return false;
}


//-----------------------------------------------------------------------------
void InitializeApp_(SimpleAllocator& allocator, int displayWidth, int displayHeight, bool isLinerMode)
{
    // アロケータの初期化
    // アプリケーションで使用するヒープ領域を初期化
    uint8_t*  pMem  = new uint8_t[AppHeapSize];
    allocator.Initialize(pMem, AppHeapSize, SimpleAllocator::DebugFill_Enabled);

    InitFileDeviceManager_();

    nns::hid::util::SetControllerManagerWithDefault(&g_ControllerManager);

    // Gfxの初期化
    {
        nns::gfx::GraphicsFramework::InitializeGraphicsSystem(1024 * 1024 * 32);
        nns::gfx::GraphicsFramework::FrameworkInfo fwInfo;
        fwInfo.SetDefault();
        fwInfo.SetDisplayWidth(displayWidth);
        fwInfo.SetDisplayHeight(displayHeight);

        // Float バッファの場合は SwapChain にコピー時、もしくはキャプチャ時に適切なガンマ設定のバッファへシェーダーでコピーされる
        if (g_HDRWriteEnabled)
        {
            fwInfo.SetColorBufferFormat(nn::gfx::ImageFormat_R16_G16_B16_A16_Float);
        }
        else if (!isLinerMode)
        {
            fwInfo.SetColorBufferFormat(nn::gfx::ImageFormat_R8_G8_B8_A8_Unorm);
        }

        fwInfo.SetMemoryPoolSize(nns::gfx::GraphicsFramework::MemoryPoolType_RenderTarget,
                                    128 * 1024 * 1024 + StaticCaptureTextureMemorySize + DynamicCaptureTextureMemorySize);

        g_GfxFramework.Initialize(fwInfo);

        g_ColorBufferFormat = fwInfo.GetColorBufferFormat();
    }

    g_Width = g_GfxFramework.GetDisplayWidth();
    g_Height = g_GfxFramework.GetDisplayHeight();
    g_ScreenW = static_cast<float>(g_Width);
    g_ScreenH = static_cast<float>(g_Height);

#if defined( NN_BUILD_CONFIG_OS_WIN32 )
    nn::vi::GetNativeWindow((nn::vi::NativeWindowHandle*)&g_Hwnd, g_GfxFramework.GetLayer());
    InitializeGfxWin_(g_Hwnd, static_cast<int>(g_ScreenW), static_cast<int>(g_ScreenH));
#endif

    nn::gfx::Device* pDevice = g_GfxFramework.GetDevice();

#if defined( NN_BUILD_CONFIG_OS_WIN32 )
    // RGB スワップチェインを作成
    {
        int swapChainBufferCount = 2;
        nn::gfx::SwapChain::InfoType swapchainInfo;
        swapchainInfo.SetDefault();
        swapchainInfo.SetLayer(g_GfxFramework.GetLayer());
        swapchainInfo.SetWidth(g_GfxFramework.GetDisplayWidth());
        swapchainInfo.SetHeight(g_GfxFramework.GetDisplayHeight());
        swapchainInfo.SetFormat(nn::gfx::ImageFormat_R8_G8_B8_A8_Unorm);
        swapchainInfo.SetBufferCount(swapChainBufferCount);
        {
            size_t size = g_RGBSwapChain.CalculateScanBufferSize(pDevice, swapchainInfo);
            size_t alignment = nn::gfx::SwapChain::GetScanBufferAlignment(pDevice, swapchainInfo);
            ptrdiff_t offset = g_GfxFramework.AllocatePoolMemory(nns::gfx::GraphicsFramework::MemoryPoolType_RenderTarget, size, alignment);
            NN_SDK_ASSERT(offset != nn::gfx::util::MemoryPoolAllocator::InvalidOffset);
            g_RGBSwapChain.Initialize(pDevice, swapchainInfo, g_GfxFramework.GetMemoryPool(nns::gfx::GraphicsFramework::MemoryPoolType_RenderTarget), offset, size);
        }
    }

    // HDR レンダリング時のキャプチャ用カラーバッファを作成
    {
        nn::gfx::Texture::InfoType info;
        info.SetDefault();
        info.SetWidth(g_GfxFramework.GetDisplayWidth());
        info.SetHeight(g_GfxFramework.GetDisplayHeight());
        info.SetGpuAccessFlags(nn::gfx::GpuAccess_ColorBuffer);
        info.SetImageStorageDimension(nn::gfx::ImageStorageDimension_2d);
        info.SetImageFormat(isLinerMode ? nn::gfx::ImageFormat_R8_G8_B8_A8_UnormSrgb : nn::gfx::ImageFormat_R8_G8_B8_A8_Unorm);
        info.SetMipCount(1);
        size_t size = nn::gfx::Texture::CalculateMipDataSize(pDevice, info);
        size_t align = nn::gfx::Texture::CalculateMipDataAlignment(pDevice, info);
        ptrdiff_t offset = g_GfxFramework.AllocatePoolMemory(nns::gfx::GraphicsFramework::MemoryPoolType_RenderTarget, size, align);
        nn::gfx::MemoryPool* pMemoryPool = g_GfxFramework.GetMemoryPool(nns::gfx::GraphicsFramework::MemoryPoolType_RenderTarget);
        g_CaptureColorBuffer.Initialize(pDevice, info, pMemoryPool, offset, size);
    }
#endif

    InitializePrimitiveRenderer_(pDevice);

    // グレア関連の初期化
    g_GlareEffect.Initialize(
        pDevice,
        AllocFunction_,
        FreeFunction_,
        NULL,
        RegisterSlotForTexture_,
        UnregisterSlotForTexture_,
        RegisterSlotForSampler_,
        UnregisterSlotForSampler_,
        &g_GfxFramework,
        g_Width, g_Height);

    // カラーバッファのテクスチャビューを作成する。
    nn::gfx::TextureView::InfoType viewInfo;
    viewInfo.SetDefault();
    viewInfo.SetImageDimension(nn::gfx::ImageDimension_2d);
    viewInfo.SetImageFormat(g_ColorBufferFormat);
    viewInfo.SetTexturePtr(g_GfxFramework.GetColorBuffer());
    g_ColorBufferTextureView.Initialize(pDevice, viewInfo);

    // カラーバッファテクスチャのデスクリプタスロット設定
    RegisterSlotForTexture_(&g_ColorBufferDescSlot, g_ColorBufferTextureView, &g_GfxFramework);

    //------------------------------------------
    // 処理メータ( nn::perf )初期化
    //------------------------------------------
    {
        nn::perf::LoadMeterCenterInfo info;
        info.SetDefault();
        info.SetCoreCount( 1 );
        info.SetUserMeterCount( 1 );
        info.SetCpuSectionCountMax( 64 );
        info.SetGpuSectionCountMax( 64 );
        info.SetGpuBufferCount( 2 );

        size_t meterMemorySize = NN_PERF_GET_BUFFER_SIZE(info);
        size_t meterMemoryAlignment = NN_PERF_GET_BUFFER_ALIGNMENT();

        size_t meterMemoryPoolSize = NN_PERF_GET_MEMORY_POOL_SIZE(pDevice, info );

        const nns::gfx::GraphicsFramework::MemoryPoolType memoryPoolType = nns::gfx::GraphicsFramework::MemoryPoolType_ConstantBuffer;

        nn::gfx::MemoryPool* pMemoryPool = g_GfxFramework.GetMemoryPool(memoryPoolType);
        g_offsetToMeterMemoryPool = g_GfxFramework.AllocatePoolMemory(memoryPoolType, meterMemoryPoolSize, NN_PERF_GET_MEMORY_POOL_ALIGNMENT(pDevice, info));
        g_pMeterMemory = AllocFunction_(meterMemorySize, meterMemoryAlignment, NULL);


        NN_PERF_INITIALIZE_METER(pDevice, info, g_pMeterMemory, meterMemorySize, pMemoryPool, g_offsetToMeterMemoryPool, meterMemoryPoolSize );

        // 現在のスレッド(メインスレッド)を 0 番コアに割り当て
        nn::os::SetThreadCoreMask(nn::os::GetCurrentThread(), 0, 1);

#if defined(NN_BUILD_CONFIG_OS_WIN)
        // Windows では GetCurrentCoreNumber() が正しい値を返さないことがあるため、
        // その不具合が修正されるまでは強制的にコア番号 0 を返すようにする
        NN_PERF_SET_GET_CORE_NUMBER_FUNCTION(GetFixedCoreNumber);
#endif
    }

    nn::ui2d::Initialize(AllocFunction_, FreeFunction_, NULL);

    // 入力インターフェースの初期化
    {
#if defined( NN_BUILD_CONFIG_OS_WIN )
        g_Mouse.Initialize( g_Hwnd );
        g_Pad.Initialize( g_Hwnd );
#else
        g_Pad.Initialize();
#endif
    }

    InitializeFont_(1024, &allocator);
    InitLayoutViewer_("", g_ScreenW, g_ScreenH, isLinerMode);

#if defined(NN_BUILD_CONFIG_OS_WIN32)
    g_offsetToScreenShotMemoryPool = g_GfxFramework.AllocatePoolMemory(nns::gfx::GraphicsFramework::MemoryPoolType_ConstantBuffer, g_LayoutViewer.GetScreenShotRequiredMemoryPoolSize(), g_LayoutViewer.GetScreenShotMemoryPoolAlignment(pDevice));
#endif
}// NOLINT(impl/function_size)


//-----------------------------------------------------------------------------
void FinalizeApp_(SimpleAllocator& allocator)
{
#if defined(NN_BUILD_CONFIG_OS_WIN32)
    const nns::gfx::GraphicsFramework::MemoryPoolType memoryPoolType = nns::gfx::GraphicsFramework::MemoryPoolType_ConstantBuffer;
    g_GfxFramework.FreePoolMemory(memoryPoolType, g_offsetToScreenShotMemoryPool);
#endif

    nn::gfx::Device* pDevice = g_GfxFramework.GetDevice();

    FinalizeLayoutViewer_(pDevice);
    FinalizeFont_(&allocator);

    // 処理メータ破棄
    {
        NN_PERF_FINALIZE_METER(pDevice);
        FreeFunction_(g_pMeterMemory, NULL);
    }

    // パッド
    {
        g_Mouse.Finalize();
        g_Pad.Finalize();
    }

    UnregisterSlotForTexture_(&g_ColorBufferDescSlot, g_ColorBufferTextureView, &g_GfxFramework);
    g_ColorBufferTextureView.Finalize(pDevice);

    // グレア関連の終了処理
    g_GlareEffect.Finalize(pDevice, AllocFunction_, FreeFunction_, NULL);

    FinalizePrimitiveRenderer_(pDevice);

#if defined( NN_BUILD_CONFIG_OS_WIN32 )
    g_CaptureColorBuffer.Finalize(pDevice);
    g_RGBSwapChain.Finalize(pDevice);
#endif

    g_GfxFramework.Finalize();

    FinalizeFileDeviceManager_();
}



//-----------------------------------------------------------------------------
//! @brief        メイン関数です。
//-----------------------------------------------------------------------------
extern "C" void nnMain()
{
    int argc = nn::os::GetHostArgc();
    const char** argv = const_cast<const char**>(nn::os::GetHostArgv());

    int previewArgc = argc;
    int displayWidth = 1280;
    int displayHeight = 720;
    const bool isLinerMode = CheckOptionFlag_(argc, argv, "-g");
    if(argc > 1)
    {
        const char* pValue = nullptr;
#ifdef NN_BUILD_CONFIG_OS_WIN32

        pValue = GetOptionValue_(argc, argv, "-w");
        if(pValue != nullptr)
        {
            g_ScreenW = std::stof(pValue);
        }

        pValue = GetOptionValue_(argc, argv, "-h");
        if(pValue != nullptr)
        {
            g_ScreenH = std::stof(pValue);
        }

#endif

        pValue = GetOptionValue_(argc, argv, "-dw");
        if(pValue != nullptr)
        {
            displayWidth = std::stoi(pValue);
            previewArgc -= 2;
        }

        pValue = GetOptionValue_(argc, argv, "-dh");
        if(pValue != nullptr)
        {
            displayHeight = std::stoi(pValue);
            previewArgc -= 2;
        }

        g_HDRWriteEnabled = CheckOptionFlag_(argc, argv, "-hdr_color_buffer");
    }

    nn::htcs::Initialize(HtcsAllocate, HtcsDeallocate);

    //------------------------------------------
    // 初期化
    SimpleAllocator allocator;
    g_pAllocator = &allocator;

    InitializeApp_(allocator, displayWidth, displayHeight, isLinerMode);

#ifdef NN_BUILD_CONFIG_OS_WIN32
    if (g_HDRWriteEnabled)
    {
        previewArgc -= 1;
    }

    // ビューアー初期化のためのオプション指定はプレビューに無関係の引数のため
    // プレビューに使用されている引数の数で起動時にプレビューを動かすかどうか判定する。
    if(previewArgc > 1)
    {
        g_LayoutViewer.PreviewByCommandLineOption(g_GfxFramework.GetDevice(), argc, argv);
    }
#endif



    //------------------------------------------
    // 描画ループ
    for(;;)
    {
        // フレームの開始
        NN_PERF_BEGIN_FRAME();

        // CPU 全体（グレー）開始
        {
            NN_PERF_SET_COLOR( nn::util::Color4u8::Gray() );
            NN_PERF_BEGIN_MEASURE();

            NN_PERF_SET_COLOR_INDEX(0, nn::util::Color4u8::Gray() );
            NN_PERF_BEGIN_MEASURE_INDEX(0);
        }

#if defined(NN_BUILD_CONFIG_OS_WIN32)
        // 終了するべきか
        if (g_LayoutViewer.IsScreenShotDone())
        {
            break;
        }

        // WindowsMessege の処理
        {
            MSG  msg;
            if ( PeekMessage( &msg, NULL, 0, 0, PM_REMOVE) )
            {
                TranslateMessage(&msg);

                if ( msg.message == WM_QUIT)
                {
                    break;
                }

                DispatchMessage(&msg);
            }
        }
#else
        if(g_Pad.IsHold(g_Pad.MASK_L) && g_Pad.IsHold(g_Pad.MASK_R) &&
           g_Pad.IsHold(g_Pad.MASK_X) && g_Pad.IsHold(g_Pad.MASK_Y) &&
           g_Pad.IsHold(g_Pad.MASK_A) && g_Pad.IsHold(g_Pad.MASK_B))
        {
            NN_LOG("\n");
            NN_LOG("****** Quit LayoutViewer(Hold XYABLR) ******\n");
            NN_LOG("\n");
            break;
        }
#endif

        // Calc 開始（ミドリ）
        NN_PERF_SET_COLOR( nn::util::Color4u8::Green() );
        NN_PERF_BEGIN_MEASURE_NAME( "Calc" );
        {
            if (g_Viewer2Enabled)
            {
                Calc2_();
            }
            else
            {
                Calc_();
            }
        }
        NN_PERF_END_MEASURE();

        CalcViewerHud_();

        if (g_Viewer2Enabled)
        {
            Draw2_();
        }
        else
        {
            Draw_();
        }

        ++g_HudFontFrameIndex;
        g_HudFontFrameIndex %= nn::ui2d::ConstantBufferCount;

        WaitIfNeeded_(g_Viewer2Enabled ?
            g_ViewerScreen.GetFps() == nn::ui2d::viewer::ViewerScreen::Fps_30 :
            g_LayoutViewer.GetFps() == nn::ui2d::viewer::Viewer::Fps_30);

        // フレームの終了
        NN_PERF_END_FRAME();
    }

    //------------------------------------------
    // 終了
    {
        FinalizeApp_(allocator);
        allocator.Finalize();

        nn::htcs::Finalize();
    }
}
