﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows.Forms;
using System.Xml.Serialization;
using System.Xml;
using System.IO;
using NW4F.LayoutBinaryConverter.Schema.Flyt;

namespace SearchInfluenceAlphaChild
{
    public partial class frmMain : Form
    {
        public frmMain()
        {
            InitializeComponent();
        }

        private void frmMain_Load(object sender, EventArgs e)
        {
            string[] args = Environment.GetCommandLineArgs();
            if (args.Length >= 2)
            {
                this.Show();
                SearchInfluenceAlpha(args[1]);
            }
        }

        private void btnOpenFolder_Click(object sender, EventArgs e)
        {
            if (dlgOpenFolder.ShowDialog(this) != DialogResult.Cancel)
            {
                SearchInfluenceAlpha(dlgOpenFolder.SelectedPath);
            }
        }

        private void btnSaveFile_Click(object sender, EventArgs e)
        {
            dlgSaveFile.Filter = "テキストファイル|*.txt|すべてのファイル(*.*)|*.*";
            if (dlgSaveFile.ShowDialog(this) != DialogResult.Cancel)
            {
                StreamWriter file = new StreamWriter(dlgSaveFile.FileName);
                file.Write(txtOutput.Text);
                file.Close();
                MessageBox.Show(this, "保存しました。");
            }
        }

        Dictionary<string, bool> Result; // 結果格納用

        // 「子階層に透明度の影響を与える」が無効に設定された部品ペインを検索する
        void SearchInfluenceAlpha(string path)
        {
            txtPath.Text = path;
            txtOutput.Text = "";
            int count = CountFlytFiles(path);
            barProgress.Maximum = count;
            barProgress.Value = 0;
            Application.DoEvents();
            Result = new Dictionary<string, bool>();
            SearchFlytFiles(path, path, 0);
            foreach (KeyValuePair<string, bool> result in Result)
            {
                txtOutput.Text += result.Key + "\r\n";
            }
            MessageBox.Show(this, "完了しました。");
        }

        // flyt ファイルの数を数える
        int CountFlytFiles(string path)
        {
            int count = 0;
            foreach (string dir in Directory.GetDirectories(path))
            {
                count += CountFlytFiles(dir);
            }
            foreach (string file in Directory.GetFiles(path))
            {
                if (Path.GetExtension(file).ToLower() == ".flyt")
                {
                    count++;
                }
            }
            return count;
        }

        // flyt ファイルを再帰的に洗い出す
        int SearchFlytFiles(string root, string path, int count)
        {
            foreach (string dir in Directory.GetDirectories(path))
            {
                count = SearchFlytFiles(root, dir, count);
            }
            foreach (string file in Directory.GetFiles(path))
            {
                if (Path.GetExtension(file).ToLower() != ".flyt")
                {
                    continue;
                }
                try
                {
                    CheckFlytFile(root, file);
                    count++;
                    barProgress.Value = count;
                    Application.DoEvents();
                }
                catch
                {
                    txtOutput.Text += "[Error] " + path + "\r\n";
                }
            }
            return count;
        }

        // flyt ファイル内をチェック
        void CheckFlytFile(string root, string path)
        {
            XmlSerializer xml = new XmlSerializer(typeof(NW4F.LayoutBinaryConverter.Schema.Flyt.Document));
            XmlReader reader = XmlReader.Create(new StreamReader(path));
            var flyt = (NW4F.LayoutBinaryConverter.Schema.Flyt.Document)xml.Deserialize(reader);
            reader.Close();
            var hierarchy = flyt?.body?.lyt?.paneHierarchy;
            var paneSet = flyt?.body?.lyt?.paneSet;
            if (hierarchy == null || paneSet == null)
            {
                return;
            }

            // 親ペインを洗い出してチェック
            PaneTree[] panes = hierarchy?.paneTree?.paneTree;
            if (panes != null)
            {
                foreach (var pane in panes)
                {
                    const bool defaultContextInfluenceAlpha = false;
                    CheckChildPane(root, defaultContextInfluenceAlpha, pane, path, paneSet);
                }
            }
        }

        // 子ペインをチェック
        // 引数の wrongCase とは、influenceAlpha が 親ペインが true、子ペインが false のケースに該当する。
        void CheckChildPane(string root, bool contextInfluenceAlpha, PaneTree child, string path, Pane[] paneSet)
        {
            string childName = child?.name;
            if (childName == null)
            {
                return;
            }

            // ペインを検索
            Pane childPane = null;
            foreach (var pane in paneSet)
            {
                if (pane?.name == childName)
                {
                    childPane = pane;
                    break;
                }
            }
            if (childPane == null)
            {
                return;
            }

            PaneTree[] grandchildren = child?.paneTree;

            // 部品ペインで influenceAlpha が true の場合は、
            // 部品のルートにぶらさがるペインが false の場合に描画に影響が出る。
            if (childPane.kind == PaneKind.Parts && childPane.influencedAlpha)
            {
                Parts parts = childPane.Item as Parts;
                if (parts != null)
                {
                    string partsFlytName = parts?.path;
                    if (partsFlytName != null)
                    {
                        string partsPath = FindPartsPath(root, partsFlytName, root);
                        if (partsPath == null)
                        {
                            txtOutput.Text += "[Not Found] " + partsFlytName + "\r\n";
                        }
                        else
                        {
                            // 部品ルートにぶらさがるペインの influenceAlpha が false かどうかチェック
                            CheckInfluenceAlphaOfPartsTop(partsPath);
                        }
                    }
                }
            }

            // influenceAlpha が、親ペイン(context)が true、子ペインが false、孫ペインが存在するケースをチェック。
            // それ以外のケースでは描画に影響は出ない。
            // 子が部品ペインの場合も孫ペインには問題が出ない。
            if (contextInfluenceAlpha && !childPane.influencedAlpha && grandchildren != null && grandchildren.Length > 0)
            {
                Result[path + " - " + childName] = true;
            }

            // ペインの influenceAlpha が true の場合に、context の influenceAlpha は true に更新される。
            // それ以外の場合は、context の influenceAlpha はそのまま引き継がれる。
            if (childPane.influencedAlpha)
            {
                contextInfluenceAlpha = true;
            }

            // 孫ペインも再帰的にチェック
            if (grandchildren != null)
            {
                foreach (var grandchild in grandchildren)
                {
                    CheckChildPane(root, contextInfluenceAlpha, grandchild, path, paneSet);
                }
            }
        }

        // 部品のファイル名から絶対パスの取得
        string FindPartsPath(string root, string parts, string path)
        {
            foreach (string dir in Directory.GetDirectories(path))
            {
                string result = FindPartsPath(root, parts, dir);
                if (result != null)
                {
                    return result;
                }
            }
            foreach (string file in Directory.GetFiles(path))
            {
                if (Path.GetFileName(file).ToLower() == parts.ToLower())
                {
                    return file;
                }
            }
            return null;
        }

        // 部品ルートにぶらさがるペインの influenceAlpha が false かどうかをチェックする
        void CheckInfluenceAlphaOfPartsTop(string path)
        {
            XmlSerializer xml = new XmlSerializer(typeof(NW4F.LayoutBinaryConverter.Schema.Flyt.Document));
            XmlReader reader = XmlReader.Create(new StreamReader(path));
            var flyt = (NW4F.LayoutBinaryConverter.Schema.Flyt.Document)xml.Deserialize(reader);
            reader.Close();
            var hierarchy = flyt?.body?.lyt?.paneHierarchy;
            var paneSet = flyt?.body?.lyt?.paneSet;
            if (hierarchy == null || paneSet == null)
            {
                throw new Exception();
            }

            PaneTree[] panes = hierarchy?.paneTree?.paneTree;
            if (panes == null)
            {
                return;
            }

            // 部品ルートに直接ぶらさがる各ペインを走査
            {
                foreach (var child in panes)
                {
                    string childName = child?.name;
                    if (childName == null)
                    {
                        throw new Exception();
                    }

                    Pane childPane = null;
                    foreach (var pane in paneSet)
                    {
                        if (pane?.name == childName)
                        {
                            childPane = pane;
                            break;
                        }
                    }
                    if (childPane == null)
                    {
                        throw new Exception();
                    }

                    // 1 つでも influenceAlpha が false で孫ペインが存在するなら、false を返す
                    if (!childPane.influencedAlpha)
                    {
                        PaneTree[] grandchildren = child?.paneTree;
                        if (grandchildren != null && grandchildren.Length > 0)
                        {
                            Result[path + " - " + childName] = true;
                        }
                    }
                }
            }
        }
    }
}
