﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Diagnostics;
using System.Drawing;
using System.IO;
using System.Windows.Forms;
using NW4C.Tga;

namespace TgaTest
{
    /// <summary>
    /// メインフォームクラス。
    /// </summary>
    public sealed partial class MainForm : Form
    {
        // 処理計測用ストップウォッチ１
        private readonly Stopwatch _stopwatch1 = new Stopwatch();
        // 処理計測用ストップウォッチ２
        private readonly Stopwatch _stopwatch2 = new Stopwatch();
        // イメージページ配列
        private readonly TabPage[] _imagePages = new TabPage[11];
        // イメージパネル配列
        private readonly ImagePanel[] _imagePanels = new ImagePanel[11];

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public MainForm()
        {
            InitializeComponent();

            // イメージページ格納
            _imagePages[ 0] = tpgLevel01;
            _imagePages[ 1] = tpgLevel02;
            _imagePages[ 2] = tpgLevel03;
            _imagePages[ 3] = tpgLevel04;
            _imagePages[ 4] = tpgLevel05;
            _imagePages[ 5] = tpgLevel06;
            _imagePages[ 6] = tpgLevel07;
            _imagePages[ 7] = tpgLevel08;
            _imagePages[ 8] = tpgLevel09;
            _imagePages[ 9] = tpgLevel10;
            _imagePages[10] = tpgLevel11;

            // イメージパネル格納
            _imagePanels[ 0] = pnlImage01;
            _imagePanels[ 1] = pnlImage02;
            _imagePanels[ 2] = pnlImage03;
            _imagePanels[ 3] = pnlImage04;
            _imagePanels[ 4] = pnlImage05;
            _imagePanels[ 5] = pnlImage06;
            _imagePanels[ 6] = pnlImage07;
            _imagePanels[ 7] = pnlImage08;
            _imagePanels[ 8] = pnlImage09;
            _imagePanels[ 9] = pnlImage10;
            _imagePanels[10] = pnlImage11;

            // 各レベルタブは非表示
            for (int i = 0; i < _imagePages.Length; i++)
            {
                tabImages.TabPages.Remove(_imagePages[i]);
            }
        }

        /// <summary>
        /// アプリケーションのメインエントリ。
        /// </summary>
        [STAThread]
        public static void Main()
        {
            Application.Run(new MainForm());
        }

        /// <summary>
        /// ファイルの読み込み。
        /// </summary>
        private void ReadFile(string filePath)
        {
            //-----------------------------------------------------------------
            // 読み込み処理
            //-----------------------------------------------------------------
            TgaReader reader = new TgaReader();
            using (FileStream stream = new FileStream(filePath, FileMode.Open))
            {
                // 読み込み開始
                try
                {
                    _stopwatch1.Reset();
                    _stopwatch1.Start();
                    reader.Read(stream);
                    _stopwatch1.Stop();
                }
                catch (TgaNoAdditionalInformationException)
                {
                    ShowMessageBox("付加情報がありません。", MessageBoxIcon.Warning);
                    return;
                }
                catch (TgaInvalidAdditionalInformationException)
                {
                    ShowMessageBox("不正な付加情報です。", MessageBoxIcon.Error);
                    return;
                }
                catch (Exception ex)
                {
                    ShowMessageBox(string.Format("エラーが発生しました。\r\n\r\n{0}:\r\n{1}", ex.Message, ex.StackTrace), MessageBoxIcon.Error);
                    return;
                }
                finally
                {
                    if (_stopwatch1.IsRunning)
                    {
                        _stopwatch1.Stop();
                    }
                }
            }

            //-----------------------------------------------------------------
            // 表示更新
            //-----------------------------------------------------------------
            Bitmap[] colorBitmaps = reader.ColorBitmaps;
            Bitmap[] alphaBitmaps = reader.AlphaBitmaps;

            // 結合イメージ作成
            Bitmap colorCombined = null;
            Bitmap alphaCombined = null;
            {
                _stopwatch2.Reset();
                _stopwatch2.Start();
                {
                    colorCombined = ImageBitmapCreator.CombineMipmapBitmaps(colorBitmaps);
                    alphaCombined = ImageBitmapCreator.CombineMipmapBitmaps(alphaBitmaps);
                }
                _stopwatch2.Stop();
            }

            // 全体イメージページ
            pnlImageEntire.Clear();
            pnlImageEntire.Set(colorCombined, alphaCombined);

            // 各レベルイメージページ
            for (int i = 0; i < _imagePages.Length; i++)
            {
                TabPage page = _imagePages[i];
                if (i < colorBitmaps.Length)
                {
                    // 非表示→表示
                    if (!tabImages.TabPages.Contains(page))
                    {
                        tabImages.TabPages.Add(page);
                    }

                    // データ更新
                    _imagePanels[i].Clear();
                    _imagePanels[i].Set(colorBitmaps[i], alphaBitmaps[i]);
                }
                else
                {
                    // 表示→非表示
                    if (tabImages.TabPages.Contains(page))
                    {
                        tabImages.TabPages.Remove(page);
                    }

                    // データクリア
                    _imagePanels[i].Clear();
                }
            }
            tabImages.SelectedIndex = 0;

            // テクスチャフォーマット
            lblTextureFormat.Text = reader.TextureFormat.ToString();

            // テクスチャサイズ
            lblTextureSize.Text = string.Format("{0} x {1}", reader.Size.Width, reader.Size.Height);

            // ミップマップレベル
            lblMipmapLevel.Text = reader.MipmapLevel.ToString();

            // テクスチャデータサイズ
            lblTextureDataSize.Text = string.Format("{0} KB", reader.TextureDataSize / 1024);

            // アルファ成分の有無
            lblHasAlpha.Text = reader.HasAlpha.ToString();

#if false
            // パレット情報
            if (reader.TextureFormat == TextureFormat.C4 ||
                reader.TextureFormat == TextureFormat.C8 ||
                reader.TextureFormat == TextureFormat.C14)
            {
                grpPalette.Enabled = true;

                // パレット名
                lblPaletteName.Text = reader.PaletteName;

                // パレットフォーマット
                lblPaletteFormat.Text = reader.PaletteFormat.ToString();

                // パレットの色数
                lblPaletteEntries.Text = reader.PaletteEntries.ToString();

                // パレットデータサイズ
                lblPaletteDataSize.Text = string.Format("{0} KB", reader.PaletteDataSize / 1024);
            }
            else
            {
                grpPalette.Enabled = false;
                lblPaletteName.Text = string.Empty;
                lblPaletteFormat.Text = string.Empty;
                lblPaletteEntries.Text = string.Empty;
                lblPaletteDataSize.Text = string.Empty;
            }
#endif

            // 処理時間
            tspStatus.Text = string.Format("Bitmap変換 : {0} [ms] - {1} [ticks] Bitmap結合 : {2} [ms] - {3} [ticks]",
                _stopwatch1.ElapsedMilliseconds, _stopwatch1.ElapsedTicks,
                _stopwatch2.ElapsedMilliseconds, _stopwatch2.ElapsedTicks
            );
        }

        /// <summary>
        /// メッセージボックスを表示。
        /// </summary>
        private void ShowMessageBox(string text, MessageBoxIcon icon)
        {
            MessageBox.Show(this, text, this.Text, MessageBoxButtons.OK, icon);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnDragEnter(DragEventArgs e)
        {
            base.OnDragEnter(e);

            // ファイルドラッグ時
            if (e.Data.GetDataPresent(DataFormats.FileDrop))
            {
                // 非モーダル時
                if (!this.Modal)
                {
                    // TGAファイル（１ファイル）のみ
                    string[] files = (string[])e.Data.GetData(DataFormats.FileDrop);
                    if (files != null)
                    {
                        if (files.Length == 1 && Path.GetExtension(files[0]) == ".tga")
                        {
                            e.Effect = DragDropEffects.Copy;
                            return;
                        }
                    }
                }
            }
            e.Effect = DragDropEffects.None;
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnDragDrop(DragEventArgs e)
        {
            // アクティブ化しておく
            Activate();

            // ドロップ処理
            string[] files = (string[])e.Data.GetData(DataFormats.FileDrop, false);
            if (files != null)
            {
                string filePath = files[0];
                using (WaitCursor wait = new WaitCursor())
                {
                    ReadFile(filePath);
                }
            }
            base.OnDragDrop(e);
        }

        //---------------------------------------------------------------------
        // ファイルを開く
        private void tsiFileOpen_Click(object sender, EventArgs e)
        {
            if (openFileDialog.ShowDialog(this) == DialogResult.OK)
            {
                Update();
                using (WaitCursor wait = new WaitCursor())
                {
                    ReadFile(openFileDialog.FileName);
                }
            }
        }

        //---------------------------------------------------------------------
        // 終了
        private void tsiFileExit_Click(object sender, EventArgs e)
        {
            Close();
        }
    }

    /// <summary>
    /// 待機カーソルクラス。
    /// </summary>
    public sealed class WaitCursor : IDisposable
    {
        // カレントカーソル
        private readonly Cursor _cursor = Cursor.Current;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public WaitCursor()
        {
            // 待機カーソルに変更
            Cursor.Current = Cursors.WaitCursor;
        }

        /// <summary>
        /// リセット。
        /// </summary>
        public void Reset()
        {
            Cursor.Current = Cursors.WaitCursor;
        }

        /// <summary>
        /// インタフェース実装。
        /// </summary>
        public void Dispose()
        {
            // カーソルを元に戻す
            Cursor.Current = _cursor;
        }
    }
}
