﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Drawing;
using System.IO;
using System.Linq;
using System.Text;

namespace NW4C.Tga
{
    //-------------------------------------------------------------------------
    // TGA読み込みクラス
    //-------------------------------------------------------------------------
    #region TgaReader
    /// <summary>
    /// 付加情報付き TGA ファイル読み込みクラス。
    /// </summary>
    public sealed class TgaReader
    {
        public const int MaxWidth = 4096;
        public const int MaxHeight = 4096;

        //---------------------------------------------------------------------
        // 公開インタフェース
        //---------------------------------------------------------------------
        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public TgaReader()
        {
        }

        #region アトリビュート
        //---------------------------------------------------------------------
        // テクスチャ情報
        //---------------------------------------------------------------------
        /// <summary>
        /// テクスチャフォーマットを取得します。
        /// </summary>
        public TextureFormat TextureFormat { get { return _textureFormat; }}

        /// <summary>
        /// テクスチャサイズ（幅×高さ）を取得します。
        /// </summary>
        public Size Size { get { return _textureSize; }}

        /// <summary>
        /// テクスチャデータサイズ（bytes）を取得します。
        /// </summary>
        public int TextureDataSize { get { return _textureDataSize; }}

        /// <summary>
        /// ミップマップレベル（ミップマップしない場合は１）を取得します。
        /// </summary>
        public int MipmapLevel { get { return _mipmapLevel; }}

        /// <summary>
        /// アルファ成分を持つかどうか。
        /// </summary>
        public bool HasAlpha
        {
            get
            {
                switch (_textureFormat)
                {
        case TextureFormat.A4:
        case TextureFormat.A8:
        case TextureFormat.La4:
        case TextureFormat.La8:
        case TextureFormat.Rgb5_a1:
        case TextureFormat.Rgba4:
        case TextureFormat.Rgba8:
        case TextureFormat.Etc1_a4:
                        return true;

        default:
                        break;
                }
                return false;
            }
        }

        //---------------------------------------------------------------------
        // ビットマップ画像
        //---------------------------------------------------------------------
        /// <summary>
        /// トップレベル部のカラーイメージビットマップを取得します。
        /// </summary>
        public Bitmap ColorBitmap { get { return _colorBitmaps != null ? _colorBitmaps[0] : null; }}

        /// <summary>
        /// トップレベル部のアルファイメージビットマップを取得します。
        /// </summary>
        public Bitmap AlphaBitmap { get { return _alphaBitmaps != null ? _alphaBitmaps[0] : null; }}

        /// <summary>
        /// ミップマップ部を含むカラーイメージビットマップ配列を取得します。
        /// </summary>
        public Bitmap[] ColorBitmaps { get { return _colorBitmaps; }}

        /// <summary>
        /// ミップマップ部を含むアルファイメージビットマップ配列を取得します。
        /// </summary>
        public Bitmap[] AlphaBitmaps { get { return _alphaBitmaps; }}
        #endregion

        #region 読み込み
        /// <summary>
        /// TGAファイルストリームからデータを読み込みます。
        /// 付加情報がない場合や、不正な内容だった場合は対応する例外が送出されます。
        /// </summary>
        public void Read(FileStream stream)
        {
            // 各要素を初期化
            _textureFormat   = TextureFormat.Unknown;
            _textureSize     = Size.Empty;
            _textureDataSize = 0;
            _mipmapLevel     = 1;

            // 付加情報取得
            TgaAdditionalInformation ai = GetAdditionalInforamtion(stream);
            if (ai == null)
            {
                throw new TgaNoAdditionalInformationException();
            }

            //-----------------------------------------------------------------
            // テクスチャ情報取得
            if (ai.Nw4rTxd != null)
            {
                _textureFormat   = TgaAdditionalInformation.GetTextureFormat(ai.Nw4rTfm);
                _textureSize     = ai.ImageSize;
                _textureActualSize = TgaHelper.ActualImageSize( _textureFormat, ai.ImageSize);
                _textureDataSize = ai.Nw4rTxd.Length;
                _mipmapLevel     = ai.Nw4rMpl;

                // エラーチェック
                if (_textureFormat == TextureFormat.Unknown)
                {
                    throw new TgaInvalidAdditionalInformationException();
                }
                if (!(
                    _textureSize.Width >= 0 && _textureSize.Width <= MaxWidth &&
                    _textureSize.Height >= 0 && _textureSize.Height <= MaxHeight))
                {
                    throw new TgaInvalidAdditionalInformationException();
                }
                if (!(_textureDataSize > 0))
                {
                    throw new TgaInvalidAdditionalInformationException();
                }
                if (!(_mipmapLevel >= 1 && _mipmapLevel <= 11))
                {
                    throw new TgaInvalidAdditionalInformationException();
                }
            }
            // テクスチャデータがない
            else
            {
                throw new TgaInvalidAdditionalInformationException();
            }

            //-----------------------------------------------------------------
            // ビットマップ作成
            ImageBitmapPair bitmapPair = null;
      Size size = _textureActualSize;
            switch (_textureFormat)
            {
      case TextureFormat.Etc1_a4:
          bitmapPair = ImageBitmapCreator.CreateBitmap_Etc1(size, ai.Nw4rTxd, _mipmapLevel, true);
                    break;

      case TextureFormat.Etc1:
                    bitmapPair = ImageBitmapCreator.CreateBitmap_Etc1(size, ai.Nw4rTxd, _mipmapLevel, false);
                    break;

      case TextureFormat.Hilo8:
                    bitmapPair = ImageBitmapCreator.CreateBitmap_Hilo8(size, ai.Nw4rTxd, _mipmapLevel);
                    break;

      case TextureFormat.L4:
                    bitmapPair = ImageBitmapCreator.CreateBitmap_L4(size, ai.Nw4rTxd, _mipmapLevel);
                    break;

      case TextureFormat.A4:
                    bitmapPair = ImageBitmapCreator.CreateBitmap_A4(size, ai.Nw4rTxd, _mipmapLevel);
                    break;

      case TextureFormat.L8:
                    bitmapPair = ImageBitmapCreator.CreateBitmap_L8(size, ai.Nw4rTxd, _mipmapLevel);
                    break;

      case TextureFormat.A8:
                    bitmapPair = ImageBitmapCreator.CreateBitmap_A8(size, ai.Nw4rTxd, _mipmapLevel);
                    break;

      case TextureFormat.La4:
                    bitmapPair = ImageBitmapCreator.CreateBitmap_La4(size, ai.Nw4rTxd, _mipmapLevel);
                    break;

      case TextureFormat.La8:
                    bitmapPair = ImageBitmapCreator.CreateBitmap_La8(size, ai.Nw4rTxd, _mipmapLevel);
                    break;

      case TextureFormat.Rgb5_a1:
                    bitmapPair = ImageBitmapCreator.CreateBitmap_Rgb5a1(size, ai.Nw4rTxd, _mipmapLevel);
                    break;

      case TextureFormat.Rgba4:
                    bitmapPair = ImageBitmapCreator.CreateBitmap_Rgba4(size, ai.Nw4rTxd, _mipmapLevel);
                    break;

      case TextureFormat.Rgba8:
                    bitmapPair = ImageBitmapCreator.CreateBitmap_Rgba8(size, ai.Nw4rTxd, _mipmapLevel);
                    break;

      case TextureFormat.Rgb8:
                    bitmapPair = ImageBitmapCreator.CreateBitmap_Rgb8(size, ai.Nw4rTxd, _mipmapLevel);
                    break;

      case TextureFormat.Rgb565:
                    bitmapPair = ImageBitmapCreator.CreateBitmap_Rgb565(size, ai.Nw4rTxd, _mipmapLevel);
                    break;

                default:
                    break;
            }
            _colorBitmaps = bitmapPair.ColorImages;
            _alphaBitmaps = bitmapPair.AlphaImages;
        }
        #endregion

        #region 付加情報の取得
        /// <summary>
        /// 付加情報を取得します。
        /// 付加情報が無い場合は NULL が返ります。
        /// </summary>
        public static TgaAdditionalInformation GetAdditionalInforamtion(FileStream stream)
        {
            // 付加情報の有無を確認
            stream.Seek(0, SeekOrigin.Begin);
            int offset = GetAdditionalInformationOffset(stream);
            if (offset == -1)
            {
                return null;
            }

            // 付加情報を準備
            TgaAdditionalInformation ai = new TgaAdditionalInformation();

            // テクスチャサイズ
            // この情報は付加情報ではなくTGAヘッダから参照する
            {
                byte[] bytes = new byte[2];

                // 幅
                const int widthOffset = 12;
                stream.Seek(widthOffset, SeekOrigin.Begin);
                stream.Read(bytes, 0, 2);
                int width = (int)BitConverter.ToUInt16(bytes, 0);

                // 高さ
                const int heightOffset = 14;
                stream.Seek(heightOffset, SeekOrigin.Begin);
                stream.Read(bytes, 0, 2);
                int height = (int)BitConverter.ToUInt16(bytes, 0);

                // 設定
                ai.ImageSize = new Size(width, height);
            }

            // データブロックヘッダの要素サイズ
            const int cBlockHeaderTagBytes  = 8;
            const int cBlockHeaderSizeBytes = 4;
            const int cBlockHeaderBytes     = cBlockHeaderTagBytes + cBlockHeaderSizeBytes;

            // 付加情報部に移動してデータブロックを順に解析する
            long current = stream.Seek(offset, SeekOrigin.Begin);
            while (current < stream.Length)
            {
                // ブロックヘッダ
                byte[] blockHeader = new byte[cBlockHeaderBytes];
                if (stream.Read(blockHeader, 0, cBlockHeaderBytes) < cBlockHeaderBytes)
                {
                    break;
                }

                // ブロックサイズ
                //int blockSize = (int)ChangeEndian(BitConverter.ToUInt32(blockHeader, cBlockHeaderTagBytes));
                int blockSize = (int)BitConverter.ToUInt32(blockHeader, cBlockHeaderTagBytes);
                if (blockSize > cBlockHeaderBytes)
                {
                    // ブロックデータ
                    int blockDataSize = blockSize - cBlockHeaderBytes;
                    byte[] blockData = new byte[blockDataSize];
                    if (stream.Read(blockData, 0, blockDataSize) < blockDataSize)
                    {
                        return null;
                    }

                    // ブロック識別子
                    string tag = _encoding.GetString(blockHeader, 0, cBlockHeaderTagBytes);
                    switch (tag)
                    {
          case "nw4c_tfm":
                            ai.Nw4rTfm = _encoding.GetString(blockData, 0, blockDataSize);
                            break;

          case "nw4c_cub":
              //未実装
              break;

          case "nw4c_mps":
              ai.Nw4rMpl = int.Parse(_encoding.GetString(blockData, 0, blockDataSize));
              break;

          case "nw4c_txd":
                            ai.Nw4rTxd = blockData;
                            break;

          case "nw4c_udt":
              //ユーザーデータ
              break;

          case "nw4c_mpi":
              //未実装
              break;

          case "nw4c_gnm":
              //テクスチャファイルを出力したツールの名前
              break;

          case "nw4c_gvr":
              //テクスチャファイルを出力したツールのバージョン
              break;

          case "nw4c_psh":
              //Adobe Photoshop専用データ
              break;

          default:
                            break;
                    }
                }
                // ブロックデータなし
                else
                {
                    // 終端なら強制終了
                    string tag = _encoding.GetString(blockHeader, 0, cBlockHeaderTagBytes);
                    if (tag == "nw4c_end")
                    {
                        break;
                    }
                }

                // 次のブロックへ
                offset += blockSize;
                current = stream.Seek(offset, SeekOrigin.Begin);
            }

            return ai;
        }
        #endregion

        #region 付加情報の確認
        /// <summary>
        /// 付加情報の有無を確認します。
        /// 付加情報があれば true が返ります。
        /// </summary>
        public static bool CheckAdditionalInformation(FileStream stream)
        {
            return GetAdditionalInformationOffset(stream) != -1;
        }
        #endregion

        //---------------------------------------------------------------------
        // 内部処理用
        //---------------------------------------------------------------------
        #region フィールド
        // テクスチャフォーマット
        private TextureFormat _textureFormat = TextureFormat.Unknown;
        // テクスチャサイズ
        private Size _textureSize = Size.Empty;
        // テクスチャサイズ
        private Size _textureActualSize = Size.Empty;
        // テクスチャデータサイズ（bytes）
        private int _textureDataSize = 0;
        // ミップマップレベル（ミップマップしない場合は１）
        private int _mipmapLevel = 1;
        // カラーイメージビットマップ配列（ミップマップレベル分）
        private Bitmap[] _colorBitmaps = null;
        // アルファイメージビットマップ配列（ミップマップレベル分）
        private Bitmap[] _alphaBitmaps = null;

        // 文字列取得用エンコーディング
        private static readonly Encoding _encoding = Encoding.GetEncoding("Shift_JIS");
        #endregion

        #region 付加情報解析
        /// <summary>
        /// 付加情報へのオフセットを取得。
        /// 付加情報がなければ -1 が返ります。
        /// </summary>
        private static int GetAdditionalInformationOffset(FileStream stream)
        {
            // +--------------------+
            // | 1| IDフィールド長  |
            // |17| TGAヘッダ       |
            // +--+-----------------+
            // |  IDフィールド      |
            // |（付加情報専用タグ）|
            // +--------------------+
            // |                    |

            // IDフィールドサイズ
            int idFieldSize = stream.ReadByte();
            if (idFieldSize == 0)
            {
                return -1;
            }

            // IDフィールドに移動
            const int idFieldOffset = 18;
            stream.Seek(idFieldOffset, SeekOrigin.Begin);

            // IDフィールドを取得
            byte[] idFieldBytes = new byte[idFieldSize];
            stream.Read(idFieldBytes, 0, idFieldSize);

            // "NW4C_Tga Ver..."のバージョン部を除いた先頭８文字をチェック
            string keyword = _encoding.GetString(idFieldBytes, 0, 8);
            if (keyword != "NW4C_Tga")
            {
                return -1;
            }

            // 付加情報へのオフセット
            int offset =
                (idFieldBytes[idFieldSize - 4] <<  0) |
                (idFieldBytes[idFieldSize - 3] <<  8) |
                (idFieldBytes[idFieldSize - 2] << 16) |
                 idFieldBytes[idFieldSize - 1] << 24;

            // オフセット部にデータがあるかどうか
            long position = stream.Seek(offset, SeekOrigin.Begin);
            if (position != offset)
            {
                return -1;
            }

            // "nw4c_tfm"ブロックが先頭にあるかどうか
            byte[] blockTagBytes = new byte[8];
            position = stream.Read(blockTagBytes, 0, 8);
            keyword = _encoding.GetString(blockTagBytes, 0, 8);
            if (position == 0 || keyword != "nw4c_tfm")
            {
                return -1;
            }

            return offset;
        }
        #endregion

        #region エンディアン変換
        /// <summary>
        /// エンディアン変換（ushort型）。
        /// </summary>
        private static ushort ChangeEndian(ushort value)
        {
            return (ushort)(
                ((value & 0x00ff) << 8) |
                ((value & 0xff00) >> 8)
            );
        }

        /// <summary>
        /// エンディアン変換（uint型）。
        /// </summary>
        private static uint ChangeEndian(uint value)
        {
            return (uint)(
                ((value & 0x000000ff) << 24) |
                ((value & 0x0000ff00) <<  8) |
                ((value & 0x00ff0000) >>  8) |
                ((value & 0xff000000) >> 24)
            );
        }
        #endregion
    }
    #endregion

    #region TgaAdditionalInformation
    /// <summary>
    /// TGA付加情報クラス。
    /// </summary>
    public sealed class TgaAdditionalInformation
    {
        /// <summary>
        /// 画像サイズ。
        /// </summary>
        public Size ImageSize = Size.Empty;

        /// <summary>
        /// nw4r_tfm データブロック内容（テクスチャフォーマット）。
        /// </summary>
        public string Nw4rTfm = string.Empty;

        /// <summary>
        /// nw4r_mpl データブロック内容（ミップマップレベル）。
        /// </summary>
        public int Nw4rMpl = 1;

        /// <summary>
        /// nw4r_txd データブロック内容（テクスチャデータ）。
        /// </summary>
        public byte[] Nw4rTxd = null;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public TgaAdditionalInformation()
        {
        }

        /// <summary>
        /// テクスチャフォーマットを取得。
        /// </summary>
        public static TextureFormat GetTextureFormat(string nw4r_tfm)
        {
            switch (nw4r_tfm)
            {
                case "L4":      return TextureFormat.L4;
                case "A4":      return TextureFormat.A4;
                case "L8":      return TextureFormat.L8;
                case "A8":      return TextureFormat.A8;
                case "La4":     return TextureFormat.La4;
                case "La8":     return TextureFormat.La8;
                case "Hilo8":   return TextureFormat.Hilo8;
                case "Rgb565":  return TextureFormat.Rgb565;
                case "Rgb8":    return TextureFormat.Rgb8;
                case "Rgb5_a1": return TextureFormat.Rgb5_a1;
                case "Rgba4":   return TextureFormat.Rgba4;
                case "Rgba8":   return TextureFormat.Rgba8;
                case "Etc1":    return TextureFormat.Etc1;
                case "Etc1_a4": return TextureFormat.Etc1_a4;
                default: break;
            }
            return TextureFormat.Unknown;
        }
    }
    #endregion

    //-------------------------------------------------------------------------
    // 例外クラス
    //-------------------------------------------------------------------------
    #region TgaReaderException
    /// <summary>
    /// 付加情報付きTGAファイル読み込み時の基本例外クラス。
    /// </summary>
    public class TgaReaderException : ApplicationException
    {
        /// <summary>
        /// コンストラクタ。
        /// </summary>
        protected TgaReaderException(string message)
            : base(message)
        {
        }
    }
    #endregion

    #region TgaNoAdditionalInformationException
    /// <summary>
    /// 付加情報付きTGAファイルではない場合の例外クラス。
    /// </summary>
    public sealed class TgaNoAdditionalInformationException : TgaReaderException
    {
        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public TgaNoAdditionalInformationException()
            : base("TgaNoAdditionalInformationException")
        {
        }
    }
    #endregion

    #region TgaInvalidAdditionalInformationException
    /// <summary>
    /// 不正な付加情報が見つかった場合の例外クラス。
    /// </summary>
    public sealed class TgaInvalidAdditionalInformationException : TgaReaderException
    {
        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public TgaInvalidAdditionalInformationException()
            : base("TgaInvalidAdditionalInformationException")
        {
        }
    }
    #endregion

    //-------------------------------------------------------------------------
    // 列挙型定義
    //-------------------------------------------------------------------------
    #region TextureFormat
    /// <summary>
    /// テクスチャフォーマット。
    /// </summary>
    public enum TextureFormat
    {
        /// <summary>不明。</summary>
        Unknown = -1,
        /// <summary>L4。</summary>
        L4 = 0,
        /// <summary>A4。</summary>
        A4,
        /// <summary>L8。</summary>
        L8,
        /// <summary>A8。</summary>
        A8,
        /// <summary>La4。</summary>
        La4,
        /// <summary>La8。</summary>
        La8,
        /// <summary>Hilo8。</summary>
        Hilo8,
        /// <summary>Rgb565。</summary>
        Rgb565,
        /// <summary>Rgb8。</summary>
        Rgb8,
        /// <summary>Rgb5_a1。</summary>
        Rgb5_a1,
        /// <summary>Rgba4。</summary>
        Rgba4,
        /// <summary>Rgba8。</summary>
        Rgba8,
        /// <summary>Etc1。</summary>
        Etc1,
        /// <summary>Etc1_a4。</summary>
        Etc1_a4,
    };
    #endregion

    //-------------------------------------------------------------------------
    //
    //-------------------------------------------------------------------------
    #region TextureFormat
    /// <summary>
    ///
    /// </summary>
    public class TgaHelper
    {
        /// <summary>
        ///
        /// </summary>
        public static Size ActualImageSize(TextureFormat format, Size imageSize)
        {
            Size size = new Size(SmallestPow2(imageSize.Width), SmallestPow2(imageSize.Height));
            int minHeight = format == TextureFormat.Etc1 ? 16 : 8;
            const int minWidth = 8;

            if (size.Height < minHeight)
            {
                size.Height = minHeight;
            }

            if (size.Width < minWidth)
            {
                size.Width = minWidth;
            }

            return size;
        }

        /// <summary>
        ///
        /// </summary>
        private static int SmallestPow2(int targetValue)
        {
            int exp = 0;
            int value = 0;

            while (targetValue > (value = (int)Math.Pow(2, exp)))
            {
                exp++;
            }
            return value;
        }

        /// <summary>
        /// 編集距離を計算します。
        /// </summary>
        static int ComputeLevenshteinDistance_(string lhs, string rhs)
        {
            int cntL = lhs.Length;
            int cntR = rhs.Length;

            int[,] distances = new int[cntL + 1, cntR + 1];

            if (cntL == 0)
            {
                return cntR;
            }

            if (cntR == 0)
            {
                return cntL;
            }

            for (int i = 0; i <= cntL; distances[i, 0] = i++) ;
            for (int j = 0; j <= cntR; distances[0, j] = j++) ;

            for (int i = 1; i <= cntL; i++)
            {
                for (int j = 1; j <= cntR; j++)
                {
                    int cost = (rhs[j - 1] == lhs[i - 1]) ? 0 : 1;

                    int minDist = Math.Min(distances[i - 1, j] + 1, distances[i, j - 1] + 1);
                    minDist = Math.Min(minDist, distances[i - 1, j - 1] + cost);

                    distances[i, j] = minDist;
                }
            }

            return distances[cntL, cntR];
        }

        /// <summary>
        /// tga ファイルのソースファイルを探す。
        /// </summary>
        public static string SearchSourceFileForTga(string searchRootDir, string resFilePath)
        {
            if (!Directory.Exists(searchRootDir))
            {
                return resFilePath;
            }

            // 部分一致するものを広めに列挙する
            string resFileName = Path.GetFileNameWithoutExtension(resFilePath);
            var searchResults = Directory.EnumerateFiles(
                    searchRootDir, "*.psd", SearchOption.AllDirectories).
                    Where((psdFile) => psdFile.Contains(resFileName));

            // フォルダ名で絞りながら数を減らしていく。
            string subResFilePath = resFilePath;
            while (!string.IsNullOrEmpty(subResFilePath) && searchResults.Count() > 1)
            {
                // ファイル名もしくは、フォルダ名を含むかどうかで絞る
                string subName = Path.GetFileNameWithoutExtension(subResFilePath);
                var filtered = searchResults.Where((searchRes) => searchRes.Contains(subName));

                // 絞った結果ゼロになった場合はその結果は採用しない。
                if (filtered.Any())
                {
                    searchResults = filtered;
                }

                // 一つ上の階層に
                subResFilePath = Path.GetDirectoryName(subResFilePath);
            }

            // 見つからなかった場合は、ソースパスを返す。
            if(!searchResults.Any())
            {
                return string.Empty;
            }

            // 絞り切れなかった場合、編集距離がもっとも近いものを返す。
            var sortedResult = searchResults.OrderBy((filePath) => ComputeLevenshteinDistance_(Path.GetFileNameWithoutExtension(filePath), resFileName));
            return sortedResult.FirstOrDefault();
        }
    }
    #endregion
}
