﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Drawing;
using System.Drawing.Imaging;

namespace NW4C.Tga
{
    /// <summary>
    /// テクスチャ画像ビットマップ作成クラス。
    /// 実テクスチャデータからカラー/アルファそれぞれの Bitmap イメージを作成します。
    /// </summary>
    public static class ImageBitmapCreator
    {
        private static int[] _ixs = new int[] {
            0, 1, 0, 1,
            2, 3, 2, 3,
            0, 1, 0, 1,
            2, 3, 2, 3,

            4, 5, 4, 5,
            6, 7, 6, 7,
            4, 5, 4, 5,
            6, 7, 6, 7,

            0, 1, 0, 1,
            2, 3, 2, 3,
            0, 1, 0, 1,
            2, 3, 2, 3,

            4, 5, 4, 5,
            6, 7, 6, 7,
            4, 5, 4, 5,
            6, 7, 6, 7,
        };

        private static int[] _iys = new int[] {
            0, 0, 1, 1,
            0, 0, 1, 1,
            2, 2, 3, 3,
            2, 2, 3, 3,

            0, 0, 1, 1,
            0, 0, 1, 1,
            2, 2, 3, 3,
            2, 2, 3, 3,

            4, 4, 5, 5,
            4, 4, 5, 5,
            6, 6, 7, 7,
            6, 6, 7, 7,

            4, 4, 5, 5,
            4, 4, 5, 5,
            6, 6, 7, 7,
            6, 6, 7, 7,
        };

        ///
        private static int[,]   _compressParams = new int[16,4] {
            {   -8,   -2,   2,    8 },
            {   -8,   -2,   2,    8 },
            {  -17,   -5,   5,   17 },
            {  -17,   -5,   5,   17 },
            {  -29,   -9,   9,   29 },
            {  -29,   -9,   9,   29 },
            {  -42,  -13,  13,   42 },
            {  -42,  -13,  13,   42 },
            {  -60,  -18,  18,   60 },
            {  -60,  -18,  18,   60 },
            {  -80,  -24,  24,   80 },
            {  -80,  -24,  24,   80 },
            { -106,  -33,  33,  106 },
            { -106,  -33,  33,  106 },
            { -183,  -47,  47,  183 },
            { -183,  -47,  47,  183 },
        };

        private static int[]    _unscramble = new int[4] { 2, 3, 1, 0 };

        /// <summary>
        /// ビット値の取得(32〜63ビットが対象)
        /// </summary>
        private static int GetBitsHigh( uint source, int size, int startpos)
        {
            int shift = ( startpos - 32 ) - size + 1;
            uint mask = (uint)( 1 << size ) - 1;
            return (int)(( source >> shift ) & mask );
        }

        /// <summary>
        /// ビット値の取得(0〜31ビットが対象)
        /// </summary>
        private static uint GetBits( uint source, int size, int startpos)
        {
            int shift = startpos - size + 1;
            uint mask = (uint)( 1 << size ) - 1;
            return (uint)( source >> shift ) & mask;
        }

        /// <summary>
        /// 値のクランプ
        /// </summary>
        private static int Clamp( int ll, int x, int ul)
        {
            if( x < ll ) { return ll; }
            if( x > ul ) { return ul; }
            return x;
        }

        /// <summary>
        ///
        /// </summary>
        private static byte[] DecompressBlockDiffFlip(uint block_part1, uint block_part2, int width, int height, int ox, int oy)
        {
            byte[] img = new byte[4 * 4 * 3];

            byte[] avg_color = new byte[3];
            byte[] enc_color1 = new byte[3];
            byte[] enc_color2 = new byte[3];

            sbyte[] diff = new sbyte[3];

            int table;
            int i, shift;
            int r, g, b;
            int diffbit;
            int flipbit;

            diffbit = (GetBitsHigh(block_part1, 1, 33));
            flipbit = (GetBitsHigh(block_part1, 1, 32));

            if (diffbit == 0)
            {
                // まず、ブロックの左側をデコードします。
                avg_color[0] = (byte)GetBitsHigh(block_part1, 4, 63);
                avg_color[1] = (byte)GetBitsHigh(block_part1, 4, 55);
                avg_color[2] = (byte)GetBitsHigh(block_part1, 4, 47);

                avg_color[0] *= 17;
                avg_color[1] *= 17;
                avg_color[2] *= 17;

                uint pixelIndicesMSB, pixelIndicesLSB;

                table = GetBitsHigh(block_part1, 3, 39) << 1;
                pixelIndicesMSB = GetBits(block_part2, 16, 31);
                pixelIndicesLSB = GetBits(block_part2, 16, 15);

                // フリップするかどうか
                if ((flipbit) == 0)
                {
                    shift = 0;
                    for (int x = ox; x < ox + 2; x++)
                    {
                        for (int y = oy; y < oy + 4; y++)
                        {
                            i = (int)((pixelIndicesMSB >> shift) & 1) << 1;
                            i |= (int)((pixelIndicesLSB >> shift) & 1);
                            shift++;
                            i = _unscramble[i];

                            r = img[3 * (y * width + x) + 0] = (byte)Clamp(0, avg_color[0] + _compressParams[table, i], 255);
                            g = img[3 * (y * width + x) + 1] = (byte)Clamp(0, avg_color[1] + _compressParams[table, i], 255);
                            b = img[3 * (y * width + x) + 2] = (byte)Clamp(0, avg_color[2] + _compressParams[table, i], 255);
                        }
                    }
                }
                else
                {
                    shift = 0;
                    for (int x = ox; x < ox + 4; x++)
                    {
                        for (int y = oy; y < oy + 2; y++)
                        {
                            i = (int)((pixelIndicesMSB >> shift) & 1) << 1;
                            i |= (int)((pixelIndicesLSB >> shift) & 1);
                            shift++;
                            i = _unscramble[i];

                            r = img[3 * (y * width + x) + 0] = (byte)Clamp(0, avg_color[0] + _compressParams[table, i], 255);
                            g = img[3 * (y * width + x) + 1] = (byte)Clamp(0, avg_color[1] + _compressParams[table, i], 255);
                            b = img[3 * (y * width + x) + 2] = (byte)Clamp(0, avg_color[2] + _compressParams[table, i], 255);
                        }
                        shift += 2;
                    }
                }

                // ブロックのほかの部分についても、デコードします。
                avg_color[0] = (byte)GetBitsHigh(block_part1, 4, 59);
                avg_color[1] = (byte)GetBitsHigh(block_part1, 4, 51);
                avg_color[2] = (byte)GetBitsHigh(block_part1, 4, 43);

                avg_color[0] *= 17;
                avg_color[1] *= 17;
                avg_color[2] *= 17;

                table = GetBitsHigh(block_part1, 3, 36) << 1;
                pixelIndicesMSB = GetBits(block_part2, 16, 31);
                pixelIndicesLSB = GetBits(block_part2, 16, 15);

                // フリップするかどうか
                if ((flipbit) == 0)
                {
                    shift = 8;
                    for (int x = ox + 2; x < ox + 4; x++)
                    {
                        for (int y = oy; y < oy + 4; y++)
                        {
                            i = (int)((pixelIndicesMSB >> shift) & 1) << 1;
                            i |= (int)((pixelIndicesLSB >> shift) & 1);
                            shift++;
                            i = _unscramble[i];

                            r = img[3 * (y * width + x) + 0] = (byte)Clamp(0, avg_color[0] + _compressParams[table, i], 255);
                            g = img[3 * (y * width + x) + 1] = (byte)Clamp(0, avg_color[1] + _compressParams[table, i], 255);
                            b = img[3 * (y * width + x) + 2] = (byte)Clamp(0, avg_color[2] + _compressParams[table, i], 255);
                        }
                    }
                }
                else
                {
                    shift = 2;
                    for (int x = ox; x < ox + 4; x++)
                    {
                        for (int y = oy + 2; y < oy + 4; y++)
                        {
                            i = (int)((pixelIndicesMSB >> shift) & 1) << 1;
                            i |= (int)((pixelIndicesLSB >> shift) & 1);
                            shift++;
                            i = _unscramble[i];

                            r = img[3 * (y * width + x) + 0] = (byte)Clamp(0, avg_color[0] + _compressParams[table, i], 255);
                            g = img[3 * (y * width + x) + 1] = (byte)Clamp(0, avg_color[1] + _compressParams[table, i], 255);
                            b = img[3 * (y * width + x) + 2] = (byte)Clamp(0, avg_color[2] + _compressParams[table, i], 255);
                        }
                        shift += 2;
                    }
                }

            }
            else
            {
                // diffbit = 1.

                //      63 62 61 60 59 58 57 56 55 54 53 52 51 50 49 48 47 46 45 44 43 42 41 40 39 38 37 36 35 34  33  32
                //      ---------------------------------------------------------------------------------------------------
                //     | base col1    | dcol 2 | base col1    | dcol 2 | base col 1   | dcol 2 | table  | table  |diff|flip|
                //     | R1' (5 bits) | dR2    | G1' (5 bits) | dG2    | B1' (5 bits) | dB2    | cw 1   | cw 2   |bit |bit |
                //      ---------------------------------------------------------------------------------------------------
                //
                //
                //
                //      31 30 29 28 27 26 25 24 23 22 21 20 19 18 17 16 15 14 13 12 11 10  9  8  7  6  5  4  3   2   1  0
                //      --------------------------------------------------------------------------------------------------
                //     |       most significant pixel index bits       |         least significant pixel index bits       |
                //     | p| o| n| m| l| k| j| i| h| g| f| e| d| c| b| a| p| o| n| m| l| k| j| i| h| g| f| e| d| c | b | a |
                //      --------------------------------------------------------------------------------------------------


                // まず、ブロックの左側をデコードします。
                enc_color1[0] = (byte)GetBitsHigh(block_part1, 5, 63);
                enc_color1[1] = (byte)GetBitsHigh(block_part1, 5, 55);
                enc_color1[2] = (byte)GetBitsHigh(block_part1, 5, 47);


                // 5ビットから 8 ビットに拡張します。
                avg_color[0] = (byte)((enc_color1[0] << 3) | (enc_color1[0] >> 2));
                avg_color[1] = (byte)((enc_color1[1] << 3) | (enc_color1[1] >> 2));
                avg_color[2] = (byte)((enc_color1[2] << 3) | (enc_color1[2] >> 2));

                uint pixel_indices_MSB, pixel_indices_LSB;

                table = GetBitsHigh(block_part1, 3, 39) << 1;
                pixel_indices_MSB = GetBits(block_part2, 16, 31);
                pixel_indices_LSB = GetBits(block_part2, 16, 15);

                // フリップするかどうか
                if ((flipbit) == 0)
                {
                    shift = 0;
                    for (int x = ox; x < ox + 2; x++)
                    {
                        for (int y = oy; y < oy + 4; y++)
                        {
                            i = (int)((pixel_indices_MSB >> shift) & 1) << 1;
                            i |= (int)((pixel_indices_LSB >> shift) & 1);
                            shift++;
                            i = _unscramble[i];

                            r = img[3 * (y * width + x) + 0] = (byte)Clamp(0, avg_color[0] + _compressParams[table, i], 255);
                            g = img[3 * (y * width + x) + 1] = (byte)Clamp(0, avg_color[1] + _compressParams[table, i], 255);
                            b = img[3 * (y * width + x) + 2] = (byte)Clamp(0, avg_color[2] + _compressParams[table, i], 255);
                        }
                    }
                }
                else
                {
                    shift = 0;
                    for (int x = ox; x < ox + 4; x++)
                    {
                        for (int y = oy; y < oy + 2; y++)
                        {
                            i = (int)((pixel_indices_MSB >> shift) & 1) << 1;
                            i |= (int)((pixel_indices_LSB >> shift) & 1);
                            shift++;
                            i = _unscramble[i];

                            r = img[3 * (y * width + x) + 0] = (byte)Clamp(0, avg_color[0] + _compressParams[table, i], 255);
                            g = img[3 * (y * width + x) + 1] = (byte)Clamp(0, avg_color[1] + _compressParams[table, i], 255);
                            b = img[3 * (y * width + x) + 2] = (byte)Clamp(0, avg_color[2] + _compressParams[table, i], 255);
                        }
                        shift += 2;
                    }
                }

                // ブロックの右部分をデコードします。
                diff[0] = (sbyte)GetBitsHigh(block_part1, 3, 58);
                diff[1] = (sbyte)GetBitsHigh(block_part1, 3, 50);
                diff[2] = (sbyte)GetBitsHigh(block_part1, 3, 42);

                enc_color2[0] = (byte)(enc_color1[0] + diff[0]);
                enc_color2[1] = (byte)(enc_color1[1] + diff[1]);
                enc_color2[2] = (byte)(enc_color1[2] + diff[2]);

                // 符号ビットをビット全体に
                diff[0] = (sbyte)(diff[0] << 5);
                diff[1] = (sbyte)(diff[1] << 5);
                diff[2] = (sbyte)(diff[2] << 5);
                diff[0] = (sbyte)(diff[0] >> 5);
                diff[1] = (sbyte)(diff[1] >> 5);
                diff[2] = (sbyte)(diff[2] >> 5);

                // 第2カラーを計算します。
                enc_color2[0] = (byte)(enc_color1[0] + diff[0]);
                enc_color2[1] = (byte)(enc_color1[1] + diff[1]);
                enc_color2[2] = (byte)(enc_color1[2] + diff[2]);

                // 5ビットから 8 ビットに拡張します。
                avg_color[0] = (byte)((enc_color2[0] << 3) | (enc_color2[0] >> 2));
                avg_color[1] = (byte)((enc_color2[1] << 3) | (enc_color2[1] >> 2));
                avg_color[2] = (byte)((enc_color2[2] << 3) | (enc_color2[2] >> 2));

                table = GetBitsHigh(block_part1, 3, 36) << 1;
                pixel_indices_MSB = GetBits(block_part2, 16, 31);
                pixel_indices_LSB = GetBits(block_part2, 16, 15);

                // フリップするかどうか
                if ((flipbit) == 0)
                {
                    shift = 8;
                    for (int x = ox + 2; x < ox + 4; x++)
                    {
                        for (int y = oy; y < oy + 4; y++)
                        {
                            i = (int)((pixel_indices_MSB >> shift) & 1) << 1;
                            i |= (int)((pixel_indices_LSB >> shift) & 1);
                            shift++;
                            i = _unscramble[i];

                            r = img[3 * (y * width + x) + 0] = (byte)Clamp(0, avg_color[0] + _compressParams[table, i], 255);
                            g = img[3 * (y * width + x) + 1] = (byte)Clamp(0, avg_color[1] + _compressParams[table, i], 255);
                            b = img[3 * (y * width + x) + 2] = (byte)Clamp(0, avg_color[2] + _compressParams[table, i], 255);
                        }
                    }
                }
                else
                {
                    shift = 2;
                    for (int x = ox; x < ox + 4; x++)
                    {
                        for (int y = oy + 2; y < oy + 4; y++)
                        {
                            i = (int)((pixel_indices_MSB >> shift) & 1) << 1;
                            i |= (int)((pixel_indices_LSB >> shift) & 1);
                            shift++;
                            i = _unscramble[i];

                            r = img[3 * (y * width + x) + 0] = (byte)Clamp(0, avg_color[0] + _compressParams[table, i], 255);
                            g = img[3 * (y * width + x) + 1] = (byte)Clamp(0, avg_color[1] + _compressParams[table, i], 255);
                            b = img[3 * (y * width + x) + 2] = (byte)Clamp(0, avg_color[2] + _compressParams[table, i], 255);
                        }
                        shift += 2;
                    }
                }
            }

            return img;
        }

        private const int _EtcBlockWidth  = 4;  //1ブロックの横幅
        private const int _EtcBlockHeight = 4;  //1ブロックの縦幅
        private const int _EtcBlockColorSize = 8; //圧縮されたカラーの1ブロックのデータサイズ
        private const int _EtcBlockAlphaSize = 8; //圧縮されたアルファの1ブロックのデータサイズ
        private const int _EtcDecodeBlockCount = 4; //DecodeEtcBlockが一度に解凍するブロックの数
        private const int _RgbaSize = 4; //Rgba要素のサイズ(各1バイト)

        /// <summary>
        /// ETC1ブロックの解凍
        /// 4*4ドット*4ブロックの64ドット分を解凍します。
        ///
        /// 1ブロック内のデータの並び
        /// +--+--+--+--+
        /// |12|13|14|15|
        /// +--+--+--+--+
        /// | 8| 9|10|11|
        /// +--+--+--+--+
        /// | 4| 5| 6| 7|
        /// +--+--+--+--+
        /// | 0| 1| 2| 3|
        /// +--+--+--+--+
        ///
        /// ブロックの並び
        /// +--+--+--+--+
        /// |B2|B3|B6|B7|
        /// +--+--+--+--+
        /// |B0|B1|B4|B5|
        /// +--+--+--+--+
        /// </summary>
        private static byte[] DecodeEtcBlock( byte[] image, int index, bool alpha)
        {
            byte[] decodeImage = new byte[ _EtcBlockWidth * _EtcBlockHeight * _EtcDecodeBlockCount * _RgbaSize ]; //16 dot * block * rgba
            byte[] tempImage = null;
            uint blockPart1 = 0;
            uint blockPart2 = 0;
            int decodeImageIndex = 0;

            for( int block = 0; block < _EtcDecodeBlockCount; block++ )
            {
                // アルファの展開
                if( alpha != false ) {
                    for( int y = 0; y < _EtcBlockHeight; y++ )
                    {
                        byte vh0 = (byte)( image[ index + 0 ] >> 4 );
                        byte vl0 = (byte)( image[ index + 0 ] & 0x0F );
                        byte vh1 = (byte)( image[ index + 1 ] >> 4 );
                        byte vl1 = (byte)( image[ index + 1 ] & 0x0F );

                        const int lo = _EtcBlockWidth * _RgbaSize;
                        decodeImage[decodeImageIndex+(1*lo)+y*4+3] = (byte)(( vh0 << 4 ) | vh0 );
                        decodeImage[decodeImageIndex+(0*lo)+y*4+3] = (byte)(( vl0 << 4 ) | vl0 );
                        decodeImage[decodeImageIndex+(3*lo)+y*4+3] = (byte)(( vh1 << 4 ) | vh1 );
                        decodeImage[decodeImageIndex+(2*lo)+y*4+3] = (byte)(( vl1 << 4 ) | vl1 );

                        //読み取ったアルファの圧縮データサイズ分だけ進める
                        index += _EtcBlockAlphaSize / _EtcBlockHeight;
                    }
                }
                else
                {
                    for( int i = 0; i < _EtcBlockWidth * _EtcBlockHeight; i++ )
                    {
                        decodeImage[decodeImageIndex+i*4+3] = 0xFF;
                    }
                }

                //カラーの展開
                blockPart1 = (uint)( image[index+4] <<  0 |
                                     image[index+5] <<  8 |
                                     image[index+6] << 16 |
                                     image[index+7] << 24 );

                blockPart2 = (uint)( image[index+0] <<  0 |
                                     image[index+1] <<  8 |
                                     image[index+2] << 16 |
                                     image[index+3] << 24 );

                tempImage = DecompressBlockDiffFlip( blockPart1, blockPart2, 4, 4, 0, 0);
                for( int i = 0; i < _EtcBlockWidth * _EtcBlockHeight; i++ )
                {
                    decodeImage[decodeImageIndex+i*4+0] = tempImage[i*3+0];
                    decodeImage[decodeImageIndex+i*4+1] = tempImage[i*3+1];
                    decodeImage[decodeImageIndex+i*4+2] = tempImage[i*3+2];
                }

                //読み取ったカラーの圧縮データサイズ分だけ進める
                index += _EtcBlockColorSize;

                //展開先のバッファのインデックスを次の位置に進める
                decodeImageIndex += _EtcBlockWidth * _EtcBlockHeight * _RgbaSize;
            }

            return decodeImage;
        }

        #region Etc1
        /// <summary>
        /// ビットマップ作成（ＥＴＣ１＆ＥＴＣ１Ａ４）。
        /// </summary>
        public static ImageBitmapPair CreateBitmap_Etc1(Size imageSize, byte[] imageData, int mipmapLevel, bool alpha)
        {
            ImageSizeDesc isd = new ImageSizeDesc(imageSize, new Size(8, 8));
            List<Bitmap> bitmapsC = new List<Bitmap>();
            List<Bitmap> bitmapsA = new List<Bitmap>();
            int dataNo = 0;

            for (int i = 0; i < mipmapLevel; i++)
            {
                Bitmap bitmapC = new Bitmap(isd.ImageW, isd.ImageH, PixelFormat.Format24bppRgb);
                Bitmap bitmapA = CreateGrayScaleIndexedBitmap(isd.ImageW, isd.ImageH, 256);

                using (BitmapProcess bpC = new BitmapProcess(bitmapC))
                {
                    using (BitmapProcess bpA = new BitmapProcess(bitmapA))
                    {
                        unsafe
                        {
                            byte* bmpPtrC = (byte*)(void*)bpC.BitmapData.Scan0;
                            byte* bmpPtrA = (byte*)(void*)bpA.BitmapData.Scan0;
                            int strideC = bpC.BitmapData.Stride;
                            int strideA = bpA.BitmapData.Stride;
                            int pixelX = 0;
                            int pixelY = 0;
                            byte[] decodeImageData = null;

                            for (int tcy = 0; tcy < isd.TileCountY; tcy++)
                            {
                                for (int tcx = 0; tcx < isd.TileCountX; tcx++)
                                {
                                    decodeImageData = DecodeEtcBlock( imageData, dataNo, alpha);
                                    dataNo += _EtcBlockColorSize * _EtcDecodeBlockCount;

                                    if( alpha != false )
                                    {
                                        dataNo += _EtcBlockAlphaSize * _EtcDecodeBlockCount;
                                    }

                                    // タイル単位
                                    int dataIndex = 0;
                                    for( int inblock = 0; inblock < 64; inblock++ )
                                    {
                                        pixelX = tcx * isd.TileW + (( inblock / 16 ) % 2 ) * 4 + ( inblock % 4 );
                                        pixelY = tcy * isd.TileH + ( inblock / 32 ) * 4 + ( inblock / 4 ) % 4;

                                        if (pixelX < isd.ImageW && pixelY < isd.ImageH)
                                        {
                                            bmpPtrA[strideA * pixelY + pixelX] = decodeImageData[ dataIndex + 3 ];

                                            bmpPtrC[strideC * pixelY + pixelX * 3 + 0] = decodeImageData[ dataIndex + 2 ];
                                            bmpPtrC[strideC * pixelY + pixelX * 3 + 1] = decodeImageData[ dataIndex + 1 ];
                                            bmpPtrC[strideC * pixelY + pixelX * 3 + 2] = decodeImageData[ dataIndex + 0 ];
                                        }

                                        dataIndex += _RgbaSize;
                                    }
                                }
                            }
                        }
                    }
                }

                // 次のレベルへ
                bitmapsC.Add(bitmapC);
                bitmapsA.Add(bitmapA);
                isd.NextLevel();
            }

            ImageBitmapPair pair = new ImageBitmapPair();
            pair.ColorImages = bitmapsC.ToArray();
            pair.AlphaImages = bitmapsA.ToArray();
            return pair;
        }
        #endregion

        #region Hilo8
        /// <summary>
        /// ビットマップ作成（ＨＩＬＯ８）。
        /// </summary>
        public static ImageBitmapPair CreateBitmap_Hilo8(Size imageSize, byte[] imageData, int mipmapLevel)
        {
            ImageSizeDesc isd = new ImageSizeDesc(imageSize, new Size(8, 8));
            List<Bitmap> bitmapsC = new List<Bitmap>();
            List<Bitmap> bitmapsA = new List<Bitmap>();
            int dataNo = 0;

            for (int i = 0; i < mipmapLevel; i++)
            {
                Bitmap bitmapC = new Bitmap(isd.ImageW, isd.ImageH, PixelFormat.Format24bppRgb);
                Bitmap bitmapA = CreateGrayScaleIndexedBitmap(isd.ImageW, isd.ImageH, 256);

                using (BitmapProcess bpC = new BitmapProcess(bitmapC))
                    using (BitmapProcess bpA = new BitmapProcess(bitmapA))
                    {
                        unsafe
                        {
                            byte* bmpPtrC = (byte*)(void*)bpC.BitmapData.Scan0;
                            byte* bmpPtrA = (byte*)(void*)bpA.BitmapData.Scan0;
                            int strideC = bpC.BitmapData.Stride;
                            int strideA = bpA.BitmapData.Stride;
                            int pixelX = 0;
                            int pixelY = 0;
                            byte r = 0;

                            // タイル単位
                            for (int tcy = 0; tcy < isd.TileCountY; tcy++)
                            {
                                for (int tcx = 0; tcx < isd.TileCountX; tcx++)
                                {
                                    for( int inblock = 0; inblock < 64; inblock++ )
                                    {
                                        pixelX = tcx * isd.TileW + _ixs[ inblock ];
                                        pixelY = tcy * isd.TileH + _iys[ inblock ];

                                        if (pixelX < isd.ImageW && pixelY < isd.ImageH)
                                        {
                                            r = imageData[dataNo + 1];
                                            bmpPtrA[strideA * pixelY + pixelX] = r;

                                            bmpPtrC[strideC * pixelY + pixelX * 3 + 0] = r;
                                            bmpPtrC[strideC * pixelY + pixelX * 3 + 1] =
                                                imageData[dataNo + 0];
                                            bmpPtrC[strideC * pixelY + pixelX * 3 + 2] = r;
                                        }

                                        dataNo += 2;
                                    }
                                }
                            }
                        }
                    }

                // 次のレベルへ
                bitmapsC.Add(bitmapC);
                bitmapsA.Add(bitmapA);
                isd.NextLevel();
            }

            ImageBitmapPair pair = new ImageBitmapPair();
            pair.ColorImages = bitmapsC.ToArray();
            pair.AlphaImages = bitmapsA.ToArray();
            return pair;
        }
        #endregion

        #region A8
        /// <summary>
        /// ビットマップ作成（Ａ８）。
        /// </summary>
        public static ImageBitmapPair CreateBitmap_A8(Size imageSize, byte[] imageData, int mipmapLevel)
        {
            ImageSizeDesc isd = new ImageSizeDesc(imageSize, new Size(8, 8));
            List<Bitmap> bitmapsC = new List<Bitmap>();
            List<Bitmap> bitmapsA = new List<Bitmap>();
            int dataNo = 0;

            for (int i = 0; i < mipmapLevel; i++)
            {
                Bitmap bitmapA = CreateGrayScaleIndexedBitmap(isd.ImageW, isd.ImageH, 256);

                using (BitmapProcess bp = new BitmapProcess(bitmapA))
                {
                    unsafe
                    {
                        byte* bmpPtr = (byte*)(void*)bp.BitmapData.Scan0;
                        int stride = bp.BitmapData.Stride;
                        int pixelX = 0;
                        int pixelY = 0;

                        // タイル単位
                        for (int tcy = 0; tcy < isd.TileCountY; tcy++)
                        {
                            for (int tcx = 0; tcx < isd.TileCountX; tcx++)
                            {
                                for( int inblock = 0; inblock < 64; inblock++ )
                                {
                                    pixelX = tcx * isd.TileW + _ixs[ inblock ];
                                    pixelY = tcy * isd.TileH + _iys[ inblock ];

                                    if (pixelX < isd.ImageW && pixelY < isd.ImageH)
                                    {
                                        bmpPtr[stride * pixelY + pixelX] = imageData[dataNo];
                                    }

                                    dataNo++;
                                }
                            }
                        }
                    }
                }

                // カラー
                Bitmap bitmapC = CreateBlackBitmap(isd.ImageW, isd.ImageH);

                // 次のレベルへ
                bitmapsC.Add(bitmapC);
                bitmapsA.Add(bitmapA);
                isd.NextLevel();
            }

            ImageBitmapPair pair = new ImageBitmapPair();
            pair.ColorImages = bitmapsC.ToArray();
            pair.AlphaImages = bitmapsA.ToArray();
            return pair;
        }
        #endregion

        #region L8
        /// <summary>
        /// ビットマップ作成（Ｌ８）。
        /// </summary>
        public static ImageBitmapPair CreateBitmap_L8(Size imageSize, byte[] imageData, int mipmapLevel)
        {
            ImageSizeDesc isd = new ImageSizeDesc(imageSize, new Size(8, 8));
            List<Bitmap> bitmapsC = new List<Bitmap>();
            List<Bitmap> bitmapsA = new List<Bitmap>();
            int dataNo = 0;

            for (int i = 0; i < mipmapLevel; i++)
            {
                Bitmap bitmapC = CreateGrayScaleIndexedBitmap(isd.ImageW, isd.ImageH, 256);

                using (BitmapProcess bp = new BitmapProcess(bitmapC))
                {
                    unsafe
                    {
                        byte* bmpPtr = (byte*)(void*)bp.BitmapData.Scan0;
                        int stride = bp.BitmapData.Stride;
                        int pixelX = 0;
                        int pixelY = 0;

                        // タイル単位
                        for (int tcy = 0; tcy < isd.TileCountY; tcy++)
                        {
                            for (int tcx = 0; tcx < isd.TileCountX; tcx++)
                            {
                                for( int inblock = 0; inblock < 64; inblock++ )
                                {

                                    pixelX = tcx * isd.TileW + _ixs[ inblock ];
                                    pixelY = tcy * isd.TileH + _iys[ inblock ];

                                    if (pixelX < isd.ImageW && pixelY < isd.ImageH)
                                    {
                                        bmpPtr[stride * pixelY + pixelX] = imageData[dataNo];
                                    }

                                    dataNo++;
                                }
                            }
                        }
                    }
                }

                // アルファ
                Bitmap bitmapA = CreateWhiteBitmap(isd.ImageW, isd.ImageH);

                // 次のレベルへ
                bitmapsC.Add(bitmapC);
                bitmapsA.Add(bitmapA);
                isd.NextLevel();
            }

            ImageBitmapPair pair = new ImageBitmapPair();
            pair.ColorImages = bitmapsC.ToArray();
            pair.AlphaImages = bitmapsA.ToArray();
            return pair;
        }
        #endregion

        #region LA4
        /// <summary>
        /// ビットマップ作成（ＬＡ４）。
        /// </summary>
        public static ImageBitmapPair CreateBitmap_La4(Size imageSize, byte[] imageData, int mipmapLevel)
        {
            ImageSizeDesc isd = new ImageSizeDesc(imageSize, new Size(8, 8));
            List<Bitmap> bitmapsC = new List<Bitmap>();
            List<Bitmap> bitmapsA = new List<Bitmap>();
            int dataNo = 0;

            for (int i = 0; i < mipmapLevel; i++)
            {
                Bitmap bitmapC = CreateGrayScaleIndexedBitmap(isd.ImageW, isd.ImageH, 16);
                Bitmap bitmapA = CreateGrayScaleIndexedBitmap(isd.ImageW, isd.ImageH, 16);

                using (BitmapProcess bpC = new BitmapProcess(bitmapC))
                    using (BitmapProcess bpA = new BitmapProcess(bitmapA))
                    {
                        unsafe
                        {
                            byte* bmpPtrC = (byte*)(void*)bpC.BitmapData.Scan0;
                            byte* bmpPtrA = (byte*)(void*)bpA.BitmapData.Scan0;
                            int strideC = bpC.BitmapData.Stride;
                            int strideA = bpA.BitmapData.Stride;
                            int pixelX = 0;
                            int pixelY = 0;

                            // タイル単位
                            for (int tcy = 0; tcy < isd.TileCountY; tcy++)
                            {
                                for (int tcx = 0; tcx < isd.TileCountX; tcx++)
                                {
                                    for( int inblock = 0; inblock < 64; inblock++ )
                                    {

                                        pixelX = tcx * isd.TileW + _ixs[ inblock ];
                                        pixelY = tcy * isd.TileH + _iys[ inblock ];

                                        if (pixelX < isd.ImageW && pixelY < isd.ImageH)
                                        {
                                            bmpPtrA[strideA * pixelY + pixelX] =
                                                (byte)(( imageData[dataNo] >> 0 ) & 0x0F );

                                            bmpPtrC[strideC * pixelY + pixelX] =
                                                (byte)(( imageData[dataNo] >> 4 ) & 0x0F );
                                                }

                                        dataNo++;
                                    }
                                }
                            }
                        }
                    }

                // 次のレベルへ
                bitmapsC.Add(bitmapC);
                bitmapsA.Add(bitmapA);
                isd.NextLevel();
            }

            ImageBitmapPair pair = new ImageBitmapPair();
            pair.ColorImages = bitmapsC.ToArray();
            pair.AlphaImages = bitmapsA.ToArray();
            return pair;
        }
        #endregion

        #region LA8
        /// <summary>
        /// ビットマップ作成（ＬＡ８）。
        /// </summary>
        public static ImageBitmapPair CreateBitmap_La8(Size imageSize, byte[] imageData, int mipmapLevel)
        {
            ImageSizeDesc isd = new ImageSizeDesc(imageSize, new Size(8, 8));
            List<Bitmap> bitmapsC = new List<Bitmap>();
            List<Bitmap> bitmapsA = new List<Bitmap>();
            int dataNo = 0;

            for (int i = 0; i < mipmapLevel; i++)
            {
                Bitmap bitmapC = CreateGrayScaleIndexedBitmap(isd.ImageW, isd.ImageH, 256);
                Bitmap bitmapA = CreateGrayScaleIndexedBitmap(isd.ImageW, isd.ImageH, 256);

                using (BitmapProcess bpC = new BitmapProcess(bitmapC))
                    using (BitmapProcess bpA = new BitmapProcess(bitmapA))
                    {
                        unsafe
                        {
                            byte* bmpPtrC = (byte*)(void*)bpC.BitmapData.Scan0;
                            byte* bmpPtrA = (byte*)(void*)bpA.BitmapData.Scan0;
                            int strideC = bpC.BitmapData.Stride;
                            int strideA = bpA.BitmapData.Stride;
                            int pixelX = 0;
                            int pixelY = 0;

                            // タイル単位
                            for (int tcy = 0; tcy < isd.TileCountY; tcy++)
                            {
                                for (int tcx = 0; tcx < isd.TileCountX; tcx++)
                                {
                                    for( int inblock = 0; inblock < 64; inblock++ )
                                    {

                                        pixelX = tcx * isd.TileW + _ixs[ inblock ];
                                        pixelY = tcy * isd.TileH + _iys[ inblock ];

                                        if (pixelX < isd.ImageW && pixelY < isd.ImageH)
                                        {
                                            bmpPtrA[strideA * pixelY + pixelX] =
                                                (byte)imageData[dataNo + 0];

                                            bmpPtrC[strideC * pixelY + pixelX] =
                                                (byte)imageData[dataNo + 1];
                                        }

                                        dataNo += 2;
                                    }
                                }
                            }
                        }
                    }

                // 次のレベルへ
                bitmapsC.Add(bitmapC);
                bitmapsA.Add(bitmapA);
                isd.NextLevel();
            }

            ImageBitmapPair pair = new ImageBitmapPair();
            pair.ColorImages = bitmapsC.ToArray();
            pair.AlphaImages = bitmapsA.ToArray();
            return pair;
        }
        #endregion

        #region L4
        /// <summary>
        /// ビットマップ作成（Ｌ４）。
        /// </summary>
        public static ImageBitmapPair CreateBitmap_L4(Size imageSize, byte[] imageData, int mipmapLevel)
        {
            ImageSizeDesc isd = new ImageSizeDesc(imageSize, new Size(8, 8));
            List<Bitmap> bitmapsC = new List<Bitmap>();
            List<Bitmap> bitmapsA = new List<Bitmap>();
            int dataNo = 0;

            for (int i = 0; i < mipmapLevel; i++)
            {
                Bitmap bitmapC = CreateGrayScaleIndexedBitmap(isd.ImageW, isd.ImageH, 16);

                using (BitmapProcess bpC = new BitmapProcess(bitmapC))
                    unsafe
                    {
                        byte* bmpPtrC = (byte*)(void*)bpC.BitmapData.Scan0;
                        int strideC = bpC.BitmapData.Stride;
                        int pixelX = 0;
                        int pixelY = 0;

                        // タイル単位
                        for (int tcy = 0; tcy < isd.TileCountY; tcy++)
                        {
                            for (int tcx = 0; tcx < isd.TileCountX; tcx++)
                            {
                                for( int inblock = 0; inblock < 64; inblock++ )
                                {
                                    pixelX = tcx * isd.TileW + _ixs[ inblock ];
                                    pixelY = tcy * isd.TileH + _iys[ inblock ];

                                    if (pixelX < isd.ImageW && pixelY < isd.ImageH)
                                    {
                                        byte colorL = (byte)((imageData[dataNo] >> 0 ) & 0x0F);
                                        byte colorH = (byte)((imageData[dataNo] >> 4 ) & 0x0F);

                                        bmpPtrC[strideC * pixelY + pixelX] =
                                            ( pixelX & 1 ) != 0 ? colorH : colorL;
                                    }

                                    dataNo += ( pixelX & 1 ) != 0 ? 1 : 0;
                                }
                            }
                        }
                    }

                // アルファ
                Bitmap bitmapA = CreateWhiteBitmap(isd.ImageW, isd.ImageH);

                // 次のレベルへ
                bitmapsC.Add(bitmapC);
                bitmapsA.Add(bitmapA);
                isd.NextLevel();
            }

            ImageBitmapPair pair = new ImageBitmapPair();
            pair.ColorImages = bitmapsC.ToArray();
            pair.AlphaImages = bitmapsA.ToArray();
            return pair;
        }
        #endregion

        #region A4
        /// <summary>
        /// ビットマップ作成（Ａ４）。
        /// </summary>
        public static ImageBitmapPair CreateBitmap_A4(Size imageSize, byte[] imageData, int mipmapLevel)
        {
            ImageSizeDesc isd = new ImageSizeDesc(imageSize, new Size(8, 8));
            List<Bitmap> bitmapsC = new List<Bitmap>();
            List<Bitmap> bitmapsA = new List<Bitmap>();
            int dataNo = 0;

            for (int i = 0; i < mipmapLevel; i++)
            {
                Bitmap bitmapA = CreateGrayScaleIndexedBitmap(isd.ImageW, isd.ImageH, 16);

                using (BitmapProcess bp = new BitmapProcess(bitmapA))
                {
                    unsafe
                    {
                        byte* bmpPtrA = (byte*)(void*)bp.BitmapData.Scan0;
                        int strideA = bp.BitmapData.Stride;
                        int pixelX = 0;
                        int pixelY = 0;

                        // タイル単位
                        for (int tcy = 0; tcy < isd.TileCountY; tcy++)
                        {
                            for (int tcx = 0; tcx < isd.TileCountX; tcx++)
                            {
                                for( int inblock = 0; inblock < 64; inblock++ )
                                {
                                    pixelX = tcx * isd.TileW + _ixs[ inblock ];
                                    pixelY = tcy * isd.TileH + _iys[ inblock ];

                                    if (pixelX < isd.ImageW && pixelY < isd.ImageH)
                                    {
                                        byte colorL = (byte)((imageData[dataNo] >> 0 ) & 0x0F);
                                        byte colorH = (byte)((imageData[dataNo] >> 4 ) & 0x0F);

                                        bmpPtrA[strideA * pixelY + pixelX] =
                                            ( pixelX & 1 ) != 0 ? colorH : colorL;
                                    }

                                    dataNo += ( pixelX & 1 ) != 0 ? 1 : 0;
                                }
                            }
                        }
                    }
                }

                // カラー
                Bitmap bitmapC = CreateBlackBitmap(isd.ImageW, isd.ImageH);

                // 次のレベルへ
                bitmapsC.Add(bitmapC);
                bitmapsA.Add(bitmapA);
                isd.NextLevel();
            }

            ImageBitmapPair pair = new ImageBitmapPair();
            pair.ColorImages = bitmapsC.ToArray();
            pair.AlphaImages = bitmapsA.ToArray();
            return pair;
        }
        #endregion

        #region RGB5A1
        /// <summary>
        /// ビットマップ作成（ＲＧＢ５Ａ１）。
        /// </summary>
        public static ImageBitmapPair CreateBitmap_Rgb5a1(Size imageSize, byte[] imageData, int mipmapLevel)
        {
            ImageSizeDesc isd = new ImageSizeDesc(imageSize, new Size(8, 8));
            List<Bitmap> bitmapsC = new List<Bitmap>();
            List<Bitmap> bitmapsA = new List<Bitmap>();
            int dataNo = 0;

            for (int i = 0; i < mipmapLevel; i++)
            {
                Bitmap bitmapC = new Bitmap(isd.ImageW, isd.ImageH, PixelFormat.Format16bppRgb555);
                Bitmap bitmapA = CreateGrayScaleIndexedBitmap(isd.ImageW, isd.ImageH, 2);

                using (BitmapProcess bpC = new BitmapProcess(bitmapC))
                    using (BitmapProcess bpA = new BitmapProcess(bitmapA))
                    {
                        unsafe
                        {
                            ushort* bmpPtrC = (ushort*)(void*)bpC.BitmapData.Scan0;
                            byte* bmpPtrA = (byte*)(void*)bpA.BitmapData.Scan0;
                            int strideC = bpC.BitmapData.Stride;
                            int strideA = bpA.BitmapData.Stride;
                            int pixelX = 0;
                            int pixelY = 0;
                            ushort data = 0;

                            // タイル単位
                            for (int tcy = 0; tcy < isd.TileCountY; tcy++)
                            {
                                for (int tcx = 0; tcx < isd.TileCountX; tcx++)
                                {
                                    for( int inblock = 0; inblock < 64; inblock++ )
                                    {

                                        pixelX = tcx * isd.TileW + _ixs[ inblock ];
                                        pixelY = tcy * isd.TileH + _iys[ inblock ];

                                        if (pixelX < isd.ImageW && pixelY < isd.ImageH)
                                        {
                                            data = (ushort)( imageData[dataNo + 0] |
                                                             imageData[dataNo + 1] << 8);

                                            bmpPtrA[strideA * pixelY + pixelX] =
                                                (byte)( data & 1 );

                                            data = (ushort)( data >> 1 );
                                            bmpPtrC[(strideC / 2) * pixelY + pixelX] = data;
                                        }

                                        dataNo += 2;
                                    }
                                }
                            }
                        }
                    }

                // 次のレベルへ
                bitmapsC.Add(bitmapC);
                bitmapsA.Add(bitmapA);
                isd.NextLevel();
            }

            ImageBitmapPair pair = new ImageBitmapPair();
            pair.ColorImages = bitmapsC.ToArray();
            pair.AlphaImages = bitmapsA.ToArray();
            return pair;
        }
        #endregion

        #region RGBA4
        /// <summary>
        /// ビットマップ作成（ＲＧＢＡ４）。
        /// </summary>
        public static ImageBitmapPair CreateBitmap_Rgba4(Size imageSize, byte[] imageData, int mipmapLevel)
        {
            ImageSizeDesc isd = new ImageSizeDesc(imageSize, new Size(8, 8));
            List<Bitmap> bitmapsC = new List<Bitmap>();
            List<Bitmap> bitmapsA = new List<Bitmap>();
            int dataNo = 0;

            for (int i = 0; i < mipmapLevel; i++)
            {
                Bitmap bitmapC = new Bitmap(isd.ImageW, isd.ImageH, PixelFormat.Format24bppRgb);
                Bitmap bitmapA = CreateGrayScaleIndexedBitmap(isd.ImageW, isd.ImageH, 16);

                using (BitmapProcess bpC = new BitmapProcess(bitmapC))
                    using (BitmapProcess bpA = new BitmapProcess(bitmapA))
                    {
                        unsafe
                        {
                            byte* bmpPtrC = (byte*)(void*)bpC.BitmapData.Scan0;
                            byte* bmpPtrA = (byte*)(void*)bpA.BitmapData.Scan0;
                            int strideC = bpC.BitmapData.Stride;
                            int strideA = bpA.BitmapData.Stride;
                            int pixelX = 0;
                            int pixelY = 0;
                            byte data = 0;

                            // タイル単位
                            for (int tcy = 0; tcy < isd.TileCountY; tcy++)
                            {
                                for (int tcx = 0; tcx < isd.TileCountX; tcx++)
                                {
                                    for( int inblock = 0; inblock < 64; inblock++ )
                                    {

                                        pixelX = tcx * isd.TileW + _ixs[ inblock ];
                                        pixelY = tcy * isd.TileH + _iys[ inblock ];

                                        if (pixelX < isd.ImageW && pixelY < isd.ImageH)
                                        {
                                            data = imageData[dataNo + 0];
                                            bmpPtrA[strideA * pixelY + pixelX] =
                                                (byte)(data & 0x0F);

                                            int r = (( data & 0xF0 ) >> 4 );
                                            data = imageData[dataNo + 1];
                                            int g = (( data & 0x0F )      );
                                            int b = (( data & 0xF0 ) >> 4 );
                                            float v = 255.0F / 15.0F;

                                            bmpPtrC[strideC * pixelY + pixelX * 3 + 0] =
                                                (byte)((float)r * v );
                                            bmpPtrC[strideC * pixelY + pixelX * 3 + 1] =
                                                (byte)((float)g * v );
                                            bmpPtrC[strideC * pixelY + pixelX * 3 + 2] =
                                                (byte)((float)b * v );
                                        }

                                        dataNo += 2;
                                    }
                                }
                            }
                        }
                    }

                // 次のレベルへ
                bitmapsC.Add(bitmapC);
                bitmapsA.Add(bitmapA);
                isd.NextLevel();
            }

            ImageBitmapPair pair = new ImageBitmapPair();
            pair.ColorImages = bitmapsC.ToArray();
            pair.AlphaImages = bitmapsA.ToArray();
            return pair;
        }
        #endregion

        #region RGBA8
        /// <summary>
        /// ビットマップ作成（ＲＧＢＡ８）。
        /// </summary>
        public static ImageBitmapPair CreateBitmap_Rgba8(Size imageSize, byte[] imageData, int mipmapLevel)
        {
            ImageSizeDesc isd = new ImageSizeDesc(imageSize, new Size(8, 8));
            List<Bitmap> bitmapsC = new List<Bitmap>();
            List<Bitmap> bitmapsA = new List<Bitmap>();
            int dataNo = 0;

            for (int i = 0; i < mipmapLevel; i++)
            {
                Bitmap bitmapC = new Bitmap(isd.ImageW, isd.ImageH, PixelFormat.Format24bppRgb);
                Bitmap bitmapA = CreateGrayScaleIndexedBitmap(isd.ImageW, isd.ImageH, 256);

                using (BitmapProcess bpC = new BitmapProcess(bitmapC))
                    using (BitmapProcess bpA = new BitmapProcess(bitmapA))
                    {
                        unsafe
                        {
                            byte* bmpPtrC = (byte*)(void*)bpC.BitmapData.Scan0;
                            byte* bmpPtrA = (byte*)(void*)bpA.BitmapData.Scan0;
                            int strideC = bpC.BitmapData.Stride;
                            int strideA = bpA.BitmapData.Stride;
                            int pixelX = 0;
                            int pixelY = 0;

                            // タイル単位
                            for (int tcy = 0; tcy < isd.TileCountY; tcy++)
                            {
                                for (int tcx = 0; tcx < isd.TileCountX; tcx++)
                                {
                                    for( int inblock = 0; inblock < 64; inblock++ )
                                    {

                                        pixelX = tcx * isd.TileW + _ixs[ inblock ];
                                        pixelY = tcy * isd.TileH + _iys[ inblock ];

                                        if (pixelX < isd.ImageW && pixelY < isd.ImageH)
                                        {
                                            bmpPtrA[strideA * pixelY + pixelX] =
                                                (byte)imageData[dataNo + 0];

                                            bmpPtrC[strideC * pixelY + pixelX * 3 + 0] =
                                                (byte)imageData[dataNo + 1];
                                            bmpPtrC[strideC * pixelY + pixelX * 3 + 1] =
                                                (byte)imageData[dataNo + 2];
                                            bmpPtrC[strideC * pixelY + pixelX * 3 + 2] =
                                                (byte)imageData[dataNo + 3];
                                        }

                                        dataNo += 4;
                                    }
                                }
                            }
                        }
                    }

                // 次のレベルへ
                bitmapsC.Add(bitmapC);
                bitmapsA.Add(bitmapA);
                isd.NextLevel();
            }

            ImageBitmapPair pair = new ImageBitmapPair();
            pair.ColorImages = bitmapsC.ToArray();
            pair.AlphaImages = bitmapsA.ToArray();
            return pair;
        }
        #endregion

        #region RGB8
        /// <summary>
        /// ビットマップ作成（ＲＧＢ８）。
        /// </summary>
        public static ImageBitmapPair CreateBitmap_Rgb8(Size imageSize, byte[] imageData, int mipmapLevel)
        {
            ImageSizeDesc isd = new ImageSizeDesc(imageSize, new Size(8, 8));
            List<Bitmap> bitmapsC = new List<Bitmap>();
            List<Bitmap> bitmapsA = new List<Bitmap>();
            int dataNo = 0;

            for (int i = 0; i < mipmapLevel; i++)
            {
                Bitmap bitmapC = new Bitmap(isd.ImageW, isd.ImageH, PixelFormat.Format24bppRgb);

                using (BitmapProcess bp = new BitmapProcess(bitmapC))
                {
                    unsafe
                    {
                        byte* bmpPtr = (byte*)(void*)bp.BitmapData.Scan0;
                        int stride = bp.BitmapData.Stride;
                        int pixelX = 0;
                        int pixelY = 0;

                        // タイル単位
                        for (int tcy = 0; tcy < isd.TileCountY; tcy++)
                        {
                            for (int tcx = 0; tcx < isd.TileCountX; tcx++)
                            {
                                for( int inblock = 0; inblock < 64; inblock++ )
                                {

                                    pixelX = tcx * isd.TileW + _ixs[ inblock ];
                                    pixelY = tcy * isd.TileH + _iys[ inblock ];

                                    if (pixelX < isd.ImageW && pixelY < isd.ImageH)
                                    {
                                        bmpPtr[stride * pixelY + pixelX * 3 + 0] =
                                            (byte)imageData[dataNo + 0];
                                        bmpPtr[stride * pixelY + pixelX * 3 + 1] =
                                            (byte)imageData[dataNo + 1];
                                        bmpPtr[stride * pixelY + pixelX * 3 + 2] =
                                            (byte)imageData[dataNo + 2];
                                    }

                                    dataNo += 3;
                                }
                            }
                        }
                    }
                }

                // アルファ
                Bitmap bitmapA = CreateWhiteBitmap(isd.ImageW, isd.ImageH);

                // 次のレベルへ
                bitmapsC.Add(bitmapC);
                bitmapsA.Add(bitmapA);
                isd.NextLevel();
            }

            ImageBitmapPair pair = new ImageBitmapPair();
            pair.ColorImages = bitmapsC.ToArray();
            pair.AlphaImages = bitmapsA.ToArray();
            return pair;
        }
        #endregion

        #region RGB565
        /// <summary>
        /// ビットマップ作成（ＲＧＢ５６５）。
        /// </summary>
        public static ImageBitmapPair CreateBitmap_Rgb565(Size imageSize, byte[] imageData, int mipmapLevel)
        {
            ImageSizeDesc isd = new ImageSizeDesc(imageSize, new Size(8, 8));
            List<Bitmap> bitmapsC = new List<Bitmap>();
            List<Bitmap> bitmapsA = new List<Bitmap>();
            int dataNo = 0;

            for (int i = 0; i < mipmapLevel; i++)
            {
                // カラー
                Bitmap bitmapC = new Bitmap(isd.ImageW, isd.ImageH, PixelFormat.Format16bppRgb565);
                using (BitmapProcess bp = new BitmapProcess(bitmapC))
                {
                    unsafe
                    {
                        ushort* bmpPtr = (ushort*)(void*)bp.BitmapData.Scan0;
                        int stride = bp.BitmapData.Stride;
                        int pixelX = 0;
                        int pixelY = 0;

                        // タイル単位
                        for (int tcy = 0; tcy < isd.TileCountY; tcy++)
                        {
                            for (int tcx = 0; tcx < isd.TileCountX; tcx++)
                            {
                                for( int inblock = 0; inblock < 64; inblock++ )
                                {

                                    pixelX = tcx * isd.TileW + _ixs[ inblock ];
                                    pixelY = tcy * isd.TileH + _iys[ inblock ];

                                    if (pixelX < isd.ImageW && pixelY < isd.ImageH)
                                    {
                                        bmpPtr[(stride / 2) * pixelY + pixelX] =
                                            (ushort)((imageData[dataNo + 0] << 0) |
                                                     (imageData[dataNo + 1] << 8));
                                    }

                                    dataNo += 2;
                                }
                            }
                        }
                    }
                }

                // アルファ
                Bitmap bitmapA = CreateWhiteBitmap(isd.ImageW, isd.ImageH);

                // 次のレベルへ
                bitmapsC.Add(bitmapC);
                bitmapsA.Add(bitmapA);
                isd.NextLevel();
            }

            ImageBitmapPair pair = new ImageBitmapPair();
            pair.ColorImages = bitmapsC.ToArray();
            pair.AlphaImages = bitmapsA.ToArray();
            return pair;
        }
        #endregion


        #region ImageSizeDesc
        /// <summary>
        /// 画像サイズ記述子クラス。
        /// </summary>
        private sealed class ImageSizeDesc
        {
            // タイルサイズ
            private readonly Size _tileSize;
            // 画像サイズ
            private Size _imageSize;
            // タイル補正画像サイズ
            private Size _tiledImageSize;
            // タイルの個数
            private Size _tileCountSize;

            /// <summary>
            /// コンストラクタ。
            /// </summary>
            public ImageSizeDesc(Size imageSize, Size tileSize)
            {
                _tileSize       = tileSize;
                _imageSize      = imageSize;
                _tiledImageSize = GetTiledSize(imageSize, tileSize);
                _tileCountSize  = new Size(_tiledImageSize.Width / tileSize.Width, _tiledImageSize.Height / tileSize.Height);
            }

            /// <summary>
            /// タイル幅。
            /// </summary>
            public int TileW { get { return _tileSize.Width; } }

            /// <summary>
            /// タイル高さ。
            /// </summary>
            public int TileH { get { return _tileSize.Height; } }

            /// <summary>
            /// イメージ幅。
            /// </summary>
            public int ImageW { get { return _imageSize.Width; } }

            /// <summary>
            /// イメージ高さ。
            /// </summary>
            public int ImageH { get { return _imageSize.Height; } }

            /// <summary>
            /// タイル補正イメージ幅。
            /// </summary>
            public int TiledImageW { get { return _tiledImageSize.Width; } }

            /// <summary>
            /// タイル補正イメージ高さ。
            /// </summary>
            public int TiledImageH { get { return _tiledImageSize.Height; } }

            /// <summary>
            /// 横方向のタイル個数。
            /// </summary>
            public int TileCountX { get { return _tileCountSize.Width; } }

            /// <summary>
            /// 縦方向のタイル個数。
            /// </summary>
            public int TileCountY { get { return _tileCountSize.Height; } }

            /// <summary>
            /// 次のレベルに変更。
            /// </summary>
            public void NextLevel()
            {
                // 全てのサイズを半分にする
                _imageSize.Width  = _imageSize.Width  / 2;
                _imageSize.Height = _imageSize.Height / 2;

                _tiledImageSize.Width  = _tiledImageSize.Width  / 2;
                _tiledImageSize.Height = _tiledImageSize.Height / 2;

                _tileCountSize.Width  = _tileCountSize.Width  / 2;
                _tileCountSize.Height = _tileCountSize.Height / 2;

                // タイルの個数は必ず１以上
                if (_tileCountSize.Width  == 0) { _tileCountSize.Width  = 1; }
                if (_tileCountSize.Height == 0) { _tileCountSize.Height = 1; }
            }

            /// <summary>
            /// タイルサイズに補正した画像サイズを取得。
            /// </summary>
            private static Size GetTiledSize(Size imageSize, Size tileSize)
            {
                Size size = new Size();
                size.Width  = (imageSize.Width  + (tileSize.Width  - 1)) / tileSize.Width  * tileSize.Width;
                size.Height = (imageSize.Height + (tileSize.Height - 1)) / tileSize.Height * tileSize.Height;
                return size;
            }
        }
        #endregion

        #region BitmapProcess
        /// <summary>
        /// ビットマップ加工処理クラス。
        /// </summary>
        private sealed class BitmapProcess : IDisposable
        {
            // ビットマップ
            private readonly Bitmap _bitmap;
            // ビットマップデータ
            private readonly BitmapData _bitmapData;

            /// <summary>
            /// コンストラクタ。
            /// </summary>
            public BitmapProcess(Bitmap bitmap)
            {
                _bitmap = bitmap;

                // ロック開始
                _bitmapData = bitmap.LockBits(new Rectangle(0, 0, bitmap.Width, bitmap.Height), ImageLockMode.ReadWrite, bitmap.PixelFormat);
            }

            /// <summary>
            /// ビットマップデータ。
            /// </summary>
            public BitmapData BitmapData
            {
                get { return _bitmapData; }
            }

            /// <summary>
            /// インタフェース実装。
            /// </summary>
            public void Dispose()
            {
                // ロック解除
                _bitmap.UnlockBits(_bitmapData);
            }
        }
        #endregion

        #region PaletteEntriesPair
        /// <summary>
        /// カラーパレット配列ペアクラス。
        /// </summary>
        private sealed class PaletteEntriesPair
        {
            /// <summary>
            /// エントリ数。
            /// </summary>
            public readonly int EntryCount;

            /// <summary>
            /// カラー配列。
            /// </summary>
            public readonly Color[] ColorEntries;

            /// <summary>
            /// アルファ配列。
            /// </summary>
            public readonly Color[] AlphaEntries;

            /// <summary>
            /// コンストラクタ。
            /// </summary>
            public PaletteEntriesPair(int entryCount)
            {
                this.EntryCount = entryCount;
                this.ColorEntries = new Color[entryCount];
                this.AlphaEntries = new Color[entryCount];
            }
        }
        #endregion

        //---------------------------------------------------------------------
        // 内部処理用
        //---------------------------------------------------------------------
        #region 非公開部
        // タイルあたりのデータサイズ
        private const int cTileBytes = 32;

        /// <summary>
        /// グレースケールインデクス方式ビットマップ作成。
        /// </summary>
        private static Bitmap CreateGrayScaleIndexedBitmap(int width, int height, int entries)
        {
            Debug.Assert(entries <= 256);

            // インデクス部設定を容易にするため、常に８ビット形式で作成
            Bitmap bitmap = new Bitmap(width, height, PixelFormat.Format8bppIndexed);
            ColorPalette palette = bitmap.Palette;
            SetGrayScaleColorPaleteEntries(palette, entries);

            // bitmap.Palette はコピーを返すので設定し直す必要あり
            bitmap.Palette = palette;
            return bitmap;
        }

        /// <summary>
        /// グレースケールカラーパレット配列設定。
        /// </summary>
        private static void SetGrayScaleColorPaleteEntries(ColorPalette palette, int entries)
        {
            int    value = 0;
            double ratio = 255.0 / (entries - 1);

            // グレースケール設定
            for (int i = 0; i < entries - 1; i++)
            {
                value = (int)Math.Round(i * ratio);
                palette.Entries[i] = Color.FromArgb(value, value, value);
            }

            // 末尾まで白で埋める
            for (int i = entries - 1; i < palette.Entries.Length; i++)
            {
                palette.Entries[i] = Color.White;
            }
        }

        /// <summary>
        /// 白単色ビットマップ作成。
        /// </summary>
        private static Bitmap CreateWhiteBitmap(int width, int height)
        {
            // ビットマップ結合を容易にするため、常に８ビット形式で作成
            Bitmap bitmap = new Bitmap(width, height, PixelFormat.Format8bppIndexed);
            ColorPalette palette = bitmap.Palette;
            {
                // インデクス部は既定で０番指定なのでパレットのみ設定する
                palette.Entries[0] = Color.White;
            }

            // bitmap.Palette はコピーを返すので設定し直す必要あり
            bitmap.Palette = palette;
            return bitmap;
        }

        /// <summary>
        /// 黒単色ビットマップ作成。
        /// </summary>
        private static Bitmap CreateBlackBitmap(int width, int height)
        {
            // ビットマップ結合を容易にするため、常に８ビット形式で作成
            Bitmap bitmap = new Bitmap(width, height, PixelFormat.Format8bppIndexed);
            ColorPalette palette = bitmap.Palette;
            {
                // インデクス部は既定で０番指定なのでパレットのみ設定する
                palette.Entries[0] = Color.Black;
            }

            // bitmap.Palette はコピーを返すので設定し直す必要あり
            bitmap.Palette = palette;
            return bitmap;
        }
        #endregion

        #region Bitmap 結合
        /// <summary>
        /// ミップマップ用 Bitmap 配列を１つの Bitmap に結合します。
        /// 下位レベルの画像は左下方向に向かって連続的に配置されます。
        /// 結合条件を満たさない場合は NULL が返ります。
        /// </summary>
        public static Bitmap CombineMipmapBitmaps(Bitmap[] bitmaps)
        {
            //-----------------------------------------------------------------
            // 準備
            //-----------------------------------------------------------------
            // ２レベル以上必要
            if (bitmaps.Length == 1)
            {
                return (Bitmap)bitmaps[0].Clone();
            }

            // フォーマット確認
            PixelFormat format = bitmaps[0].PixelFormat;
            switch (format)
            {
                // 対応するフォーマットのみ
            case PixelFormat.Format8bppIndexed:
            case PixelFormat.Format16bppRgb555:
            case PixelFormat.Format16bppRgb565:
            case PixelFormat.Format24bppRgb:
                break;
                // 対応外のフォーマット
            default:
                return null;
            }

            // サイズ確認
            Size topLvSize = bitmaps[0].Size;
            Size totalSize = topLvSize;
            Size checkSize = topLvSize;
            for (int i = 1; i < bitmaps.Length; i++)
            {
                // 下位レベルが半分の大きさか
                checkSize = new Size(checkSize.Width / 2, checkSize.Height / 2);
                if (bitmaps[i].Size != checkSize)
                {
                    return null;
                }
                // 下位レベルがトップレベルと同じフォーマットか
                if (bitmaps[i].PixelFormat != format)
                {
                    return null;
                }
                // 総合サイズ計算（横幅は変わらない）
                totalSize.Height += checkSize.Height;
            }

            //-----------------------------------------------------------------
            // 作成
            //-----------------------------------------------------------------
            Bitmap bitmap = new Bitmap(totalSize.Width, totalSize.Height, PixelFormat.Format32bppArgb);
            switch (format)
            {
            case PixelFormat.Format8bppIndexed: CombineBitmaps_8bppIndexed(bitmap, bitmaps); break;
            case PixelFormat.Format16bppRgb555: CombineBitmaps_16bppRgb555(bitmap, bitmaps); break;
            case PixelFormat.Format16bppRgb565: CombineBitmaps_16bppRgb565(bitmap, bitmaps); break;
            case PixelFormat.Format24bppRgb:    CombineBitmaps_24bppRgb(bitmap, bitmaps);    break;
            }
            return bitmap;
        }

        #region 8bppIndexed
        /// <summary>
        /// ビットマップ結合（8bppIndexed）
        /// </summary>
        private static void CombineBitmaps_8bppIndexed(Bitmap dstBitmap, Bitmap[] srcBitmaps)
        {
            int pixelY = 0;
            unsafe
                {
                    // 先画像に
                    using (BitmapProcess dstBp = new BitmapProcess(dstBitmap))
                        {
                            byte* dstBmpPtr = (byte*)(void*)dstBp.BitmapData.Scan0;
                            int   dstStride = dstBp.BitmapData.Stride;

                            // 元画像をレベル順にコピー
                            for (int i = 0; i < srcBitmaps.Length; i++)
                            {
                                Bitmap  srcBitmap  = srcBitmaps[i];
                                Color[] srcEntries = srcBitmap.Palette.Entries;
                                using (BitmapProcess srcBp = new BitmapProcess(srcBitmap))
                                    {
                                        byte* srcBmpPtr = (byte*)(void*)srcBp.BitmapData.Scan0;
                                        int   srcStride = srcBp.BitmapData.Stride;

                                        byte index = 0;
                                        for (int y = 0; y < srcBitmap.Height; y++)
                                        {
                                            for (int x = 0; x < srcBitmap.Width; x++)
                                            {
                                                index = srcBmpPtr[srcStride * y + x];

                                                dstBmpPtr[dstStride * (pixelY + y) + x * 4 + 0] = srcEntries[index].B;
                                                dstBmpPtr[dstStride * (pixelY + y) + x * 4 + 1] = srcEntries[index].G;
                                                dstBmpPtr[dstStride * (pixelY + y) + x * 4 + 2] = srcEntries[index].R;
                                                dstBmpPtr[dstStride * (pixelY + y) + x * 4 + 3] = 255;
                                            }

                                            // 余白領域（透過色）
                                            for (int px = srcBitmap.Width; px < dstBitmap.Width; px++)
                                            {
                                                dstBmpPtr[dstStride * (pixelY + y) + px * 4 + 0] = 0;
                                                dstBmpPtr[dstStride * (pixelY + y) + px * 4 + 1] = 0;
                                                dstBmpPtr[dstStride * (pixelY + y) + px * 4 + 2] = 0;
                                                dstBmpPtr[dstStride * (pixelY + y) + px * 4 + 3] = 0;
                                            }
                                        }
                                    }
                                pixelY += srcBitmap.Height;
                            }
                        }
                }
        }
        #endregion

        #region 16bppRgb555
        /// <summary>
        /// ビットマップ結合（16bppRgb555）
        /// </summary>
        private static void CombineBitmaps_16bppRgb555(Bitmap dstBitmap, Bitmap[] srcBitmaps)
        {
            int pixelY = 0;
            unsafe
                {
                    // 先画像に
                    using (BitmapProcess dstBp = new BitmapProcess(dstBitmap))
                        {
                            byte* dstBmpPtr = (byte*)(void*)dstBp.BitmapData.Scan0;
                            int   dstStride = dstBp.BitmapData.Stride;

                            // 元画像をレベル順にコピー
                            for (int i = 0; i < srcBitmaps.Length; i++)
                            {
                                Bitmap srcBitmap = srcBitmaps[i];
                                using (BitmapProcess srcBp = new BitmapProcess(srcBitmap))
                                    {
                                        ushort* srcBmpPtr = (ushort*)(void*)srcBp.BitmapData.Scan0;
                                        int     srcStride = srcBp.BitmapData.Stride;

                                        ushort data = 0;
                                        for (int y = 0; y < srcBitmap.Height; y++)
                                        {
                                            for (int x = 0; x < srcBitmap.Width; x++)
                                            {
                                                data = srcBmpPtr[(srcStride / 2) * y + x];

                                                dstBmpPtr[dstStride * (pixelY + y) + x * 4 + 0] = (byte)(((data & 0x001f) << 3) | ((data & 0x001c) >>  2));
                                                dstBmpPtr[dstStride * (pixelY + y) + x * 4 + 1] = (byte)(((data & 0x03e0) >> 2) | ((data & 0x0380) >>  7));
                                                dstBmpPtr[dstStride * (pixelY + y) + x * 4 + 2] = (byte)(((data & 0x7c00) >> 7) | ((data & 0x7000) >> 12));
                                                dstBmpPtr[dstStride * (pixelY + y) + x * 4 + 3] = 255;
                                            }

                                            // 余白領域（透過色）
                                            for (int px = srcBitmap.Width; px < dstBitmap.Width; px++)
                                            {
                                                dstBmpPtr[dstStride * (pixelY + y) + px * 4 + 0] = 0;
                                                dstBmpPtr[dstStride * (pixelY + y) + px * 4 + 1] = 0;
                                                dstBmpPtr[dstStride * (pixelY + y) + px * 4 + 2] = 0;
                                                dstBmpPtr[dstStride * (pixelY + y) + px * 4 + 3] = 0;
                                            }
                                        }
                                    }
                                pixelY += srcBitmap.Height;
                            }
                        }
                }
        }
        #endregion

        #region 16bppRgb565
        /// <summary>
        /// ビットマップ結合（16bppRgb565）
        /// </summary>
        private static void CombineBitmaps_16bppRgb565(Bitmap dstBitmap, Bitmap[] srcBitmaps)
        {
            int pixelY = 0;
            unsafe
                {
                    // 先画像に
                    using (BitmapProcess dstBp = new BitmapProcess(dstBitmap))
                        {
                            byte* dstBmpPtr = (byte*)(void*)dstBp.BitmapData.Scan0;
                            int   dstStride = dstBp.BitmapData.Stride;

                            // 元画像をレベル順にコピー
                            for (int i = 0; i < srcBitmaps.Length; i++)
                            {
                                Bitmap srcBitmap = srcBitmaps[i];
                                using (BitmapProcess srcBp = new BitmapProcess(srcBitmap))
                                    {
                                        ushort* srcBmpPtr = (ushort*)(void*)srcBp.BitmapData.Scan0;
                                        int     srcStride = srcBp.BitmapData.Stride;

                                        ushort data = 0;
                                        for (int y = 0; y < srcBitmap.Height; y++)
                                        {
                                            for (int x = 0; x < srcBitmap.Width; x++)
                                            {
                                                data = srcBmpPtr[(srcStride / 2) * y + x];

                                                dstBmpPtr[dstStride * (pixelY + y) + x * 4 + 0] = (byte)(((data & 0x001f) << 3) | ((data & 0x001c) >>  2));
                                                dstBmpPtr[dstStride * (pixelY + y) + x * 4 + 1] = (byte)(((data & 0x07e0) >> 3) | ((data & 0x0600) >>  9));
                                                dstBmpPtr[dstStride * (pixelY + y) + x * 4 + 2] = (byte)(((data & 0xf800) >> 8) | ((data & 0xe000) >> 13));
                                                dstBmpPtr[dstStride * (pixelY + y) + x * 4 + 3] = 255;
                                            }

                                            // 余白領域（透過色）
                                            for (int px = srcBitmap.Width; px < dstBitmap.Width; px++)
                                            {
                                                dstBmpPtr[dstStride * (pixelY + y) + px * 4 + 0] = 0;
                                                dstBmpPtr[dstStride * (pixelY + y) + px * 4 + 1] = 0;
                                                dstBmpPtr[dstStride * (pixelY + y) + px * 4 + 2] = 0;
                                                dstBmpPtr[dstStride * (pixelY + y) + px * 4 + 3] = 0;
                                            }
                                        }
                                    }
                                pixelY += srcBitmap.Height;
                            }
                        }
                }
        }
        #endregion

        #region 24bppRgb
        /// <summary>
        /// ビットマップ結合（24bppRgb）
        /// </summary>
        private static void CombineBitmaps_24bppRgb(Bitmap dstBitmap, Bitmap[] srcBitmaps)
        {
            int pixelY = 0;
            unsafe
                {
                    // 先画像に
                    using (BitmapProcess dstBp = new BitmapProcess(dstBitmap))
                        {
                            byte* dstBmpPtr = (byte*)(void*)dstBp.BitmapData.Scan0;
                            int   dstStride = dstBp.BitmapData.Stride;

                            // 元画像をレベル順にコピー
                            for (int i = 0; i < srcBitmaps.Length; i++)
                            {
                                Bitmap srcBitmap = srcBitmaps[i];
                                using (BitmapProcess srcBp = new BitmapProcess(srcBitmap))
                                    {
                                        byte* srcBmpPtr = (byte*)(void*)srcBp.BitmapData.Scan0;
                                        int   srcStride = srcBp.BitmapData.Stride;

                                        for (int y = 0; y < srcBitmap.Height; y++)
                                        {
                                            for (int x = 0; x < srcBitmap.Width; x++)
                                            {
                                                dstBmpPtr[dstStride * (pixelY + y) + x * 4 + 0] = srcBmpPtr[srcStride * y + x * 3 + 0];
                                                dstBmpPtr[dstStride * (pixelY + y) + x * 4 + 1] = srcBmpPtr[srcStride * y + x * 3 + 1];
                                                dstBmpPtr[dstStride * (pixelY + y) + x * 4 + 2] = srcBmpPtr[srcStride * y + x * 3 + 2];
                                                dstBmpPtr[dstStride * (pixelY + y) + x * 4 + 3] = 255;
                                            }

                                            // 余白領域（透過色）
                                            for (int px = srcBitmap.Width; px < dstBitmap.Width; px++)
                                            {
                                                dstBmpPtr[dstStride * (pixelY + y) + px * 4 + 0] = 0;
                                                dstBmpPtr[dstStride * (pixelY + y) + px * 4 + 1] = 0;
                                                dstBmpPtr[dstStride * (pixelY + y) + px * 4 + 2] = 0;
                                                dstBmpPtr[dstStride * (pixelY + y) + px * 4 + 3] = 0;
                                            }
                                        }
                                    }
                                pixelY += srcBitmap.Height;
                            }
                        }
                }
        }
        #endregion
        #endregion
    }

    /// <summary>
    /// テクスチャ画像ビットマップペアクラス。
    /// カラー/アルファそれぞれの Bitmap 配列（ミップマップレベル分）を保持します。
    /// </summary>
    public sealed class ImageBitmapPair
    {
        /// <summary>
        /// カラーイメージ配列（ミップマップレベル分）。
        /// </summary>
        public Bitmap[] ColorImages = null;

        /// <summary>
        /// アルファイメージ配列（ミップマップレベル分）。
        /// </summary>
        public Bitmap[] AlphaImages = null;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public ImageBitmapPair()
        {
        }
    }
}
