﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Text;
using System.Diagnostics;

namespace NW4F.LayoutBinaryConverter.TexConv
{
    public enum TexelFormat
    {

        /// <remarks/>
        L4,

        /// <remarks/>
        A4,

        /// <remarks/>
        L8,

        /// <remarks/>
        A8,

        /// <remarks/>
        LA4,

        /// <remarks/>
        LA8,

        /// <remarks/>
        HILO8,

        /// <remarks/>
        RGB565,

        /// <remarks/>
        RGB8,

        /// <remarks/>
        RGB5A1,

        /// <remarks/>
        RGBA4,

        /// <remarks/>
        RGBA8,

        /// <remarks/>
        ETC1,

        /// <remarks/>
        ETC1A4,

        /// <remarks/>
        BC1,

        /// <remarks/>
        BC2,

        /// <remarks/>
        BC3,

        /// <remarks/>
        BC4L,

        /// <remarks/>
        BC4A,

        /// <remarks/>
        BC5,

        /// <remarks/>
        BC7,

        /// <remarks/>
        RGB565_INDIRECT,

        /// <remarks/>
        ASTC4x4,

        /// <remarks/>
        ASTC5x4,

        /// <remarks/>
        ASTC5x5,

        /// <remarks/>
        ASTC6x5,

        /// <remarks/>
        ASTC6x6,

        /// <remarks/>
        ASTC8x5,

        /// <remarks/>
        ASTC8x6,

        /// <remarks/>
        ASTC8x8,

        /// <remarks/>
        ASTC10x5,

        /// <remarks/>
        ASTC10x6,

        /// <remarks/>
        ASTC10x8,

        /// <remarks/>
        ASTC10x10,

        /// <remarks/>
        ASTC12x10,

        /// <remarks/>
        ASTC12x12,
    }

    public enum LayerType
    {
        COLOR_RGB24,
        COLOR_CI16,
        ALPHA_A8,
        COLOR_CMP
    }

    public class Layer
    {
        LayerType _type;
        int _width;
        int _height;
        byte[] _data;

        public Layer(LayerType type, int width, int height)
            : this(type, width, height, CalcBufferSize(type, width, height))
        {}


        public void AddCacheKey(CacheKeyGenerator gen)
        {
            gen.AddCacheKey((int)_type);
            gen.AddCacheKey(_width);
            gen.AddCacheKey(_height);
            gen.AddCacheKey(_data);
        }

        public Layer(LayerType type, int width, int height, int bufSize)
        {
            Debug.Assert((width <= 8192), "TCSetLayerAttributes: layer width exceeds 8192 texels");
            Debug.Assert((height <= 8192), "TCSetLayerAttributes: layer width exceeds 8192 texels");

            switch (type)
            {
            case LayerType.COLOR_RGB24:      // fall through
            case LayerType.COLOR_CI16:
            case LayerType.ALPHA_A8:
            case LayerType.COLOR_CMP:
                break;

            default:
                Debug.Assert(false, "TCSetLayerAttributes: invalid layer type");
                break;
            }

            Debug.Assert(!((width == 0) || (width > 8192)), "TCSetLayerDimensions: layer width is out of range");

            Debug.Assert(!((height == 0) || (height > 8192)), "TCSetLayerDimensions: layer height is out of range\n");

            _type = type;
            _width = width;
            _height = height;

            _data = new byte[bufSize];
        }

        public LayerType Type { get { return _type; } }

        public int Width
        {
            get { return _width; }
            set { _width = value; }
        }

        public int Height
        {
            get { return _height; }
            set { _height = value; }
        }

        public byte[] Data { get { return _data; } }

        static int CalcBufferSize(LayerType type, int width, int height)
        {
            // compute total memory size of base image
            // and any mipmaps included in this layer
            int size = 0;
            switch (type)
            {
            case LayerType.COLOR_RGB24:
                size = (width * height * 3);
                break;

            case LayerType.COLOR_CI16:
                size = (width * height * 2);
                break;

            case LayerType.ALPHA_A8:
                size = (width * height);
                break;

            case LayerType.COLOR_CMP:
                // 4 bits per texel
                // 4x4 texel block minimum

                //       ( width      x height      x 4 bpp / 8bits per byte
                size = ((width * height * 4) >> 3);
                if (size < 8)
                {
                    size = 8;
                }
                break;

            default:                     // in case this.type wasn't set
                Debug.Assert(false, "TCSetLayerBuffer: unknown layer type");
                break;
            }

            return size;
        }

        /*>*******************************(*)*******************************<*/
        // set an individual pixel value based on an x,y location
        // actual internal data representation depends on the buffer format:
        // rgb color is stored as rgb triples -> byte[3] = { R, G, B }
        //
        // 'ria' is an unsigned short to accomodate 14-bit color index indices
        //
        // NOTE: assume (x,y) coordinates start at (0,0)
        /*>*******************************(*)*******************************<*/
        public void SetValue(int x, int y, ushort ria, byte g, byte b)
        {
            Debug.Assert(this._width != 0, "SetValue: layer width is not set\n");
            Debug.Assert(this._data != null, "SetValue: layer data ptr is not set\n");

            switch (this._type)
            {
                case LayerType.COLOR_RGB24:
                    {
                        int offset = ((y * this._width) + x) * 3;

                        this._data[offset] = (byte)ria;
                        this._data[offset + 1] = g;
                        this._data[offset + 2] = b;
                    }
                    break;
                case LayerType.COLOR_CI16:  // this is the only 16-bit format
                    {
                        int offset = ((y * this._width) + x) * 2;  // offset in bytes
                        Array.Copy(BitConverter.GetBytes(ria), 0, this._data, offset, 2);
                    }
                    break;
                case LayerType.ALPHA_A8:
                    {
                        int offset = (y * this._width) + x;
                        this._data[offset] = (byte)ria;
                    }
                    break;

                case LayerType.COLOR_CMP:     // cant return a value from compressed texture yet
                    Debug.Assert(false, "TCsetLayerValue: can't 'set' a texel value from a CMPR layer");
                    return;

                default:                 // in case format was not set
                    Debug.Assert(false, "SetValue: unknown layer type");
                    return;
            }  // end switch
        }

        /*>*******************************(*)*******************************<*/
        // retrieve color information from the given layer depending on its
        // format
        /*>*******************************(*)*******************************<*/
        public void
        GetValue(int x, int y, ref ushort riaPtr, ref byte gPtr, ref byte bPtr)
        {
            Debug.Assert(this._width != 0, "GetValue: layer width is not set\n");
            Debug.Assert(this._data != null, "GetValue: layer data ptr is not set\n");

            int offset;

            switch (this._type)
            {
            case LayerType.COLOR_RGB24:
                offset = (int)(((y * this._width) + x) * 3);
                riaPtr = this._data[offset];

                gPtr = this._data[offset + 1];
                bPtr = this._data[offset + 2];
                break;

            case LayerType.COLOR_CI16:  // this is the only 16-bit format
                offset = (int)(((y * this._width) + x) * 2);  // offset in bytes
                riaPtr = BitConverter.ToUInt16(this._data, (int)offset);
                break;

            case LayerType.ALPHA_A8:
                offset = (int)((y * this._width) + x);
                riaPtr = this._data[offset];
                break;

            case LayerType.COLOR_CMP:  // cant set a compressed pixel yet
                Debug.Assert(false, "SetValue: can't 'set' a CMPR texel value");
                return;

            default:                 // in case type wasn't set
                Debug.Assert(false, "SetValue: unknown layer type- can't set value");
                return;
            }

        }

        public static Layer CreateSameSize(Layer other)
        {
            Debug.Assert(other != null);
            return new Layer(other._type, other._width, other._height);
        }

        /*>*******************************(*)*******************************<*/
        // convert an image CI layer to an rgb24 layer
        // leave the image alpha layer as is.
        /*>*******************************(*)*******************************<*/
        public static Layer ConvertCI_To_RGB(SrcImage file, Layer srcLayer)
        {
            Debug.Assert(file.palPtr != null, "TCLayer::ConvertCI_To_RGB: TCFile %s has no palette\n", file.name);
            Debug.Assert(srcLayer._type == LayerType.COLOR_CI16, "TCLayer::ConvertCI_To_RGB: color layer type is not color-index");

            Layer newLayer = new Layer(LayerType.COLOR_RGB24, srcLayer._width, srcLayer._height);

            // perform color-table lookups
            for (int row = 0; row < srcLayer._height; row++)
            {
                for (int col = 0; col < srcLayer._width; col++)
                {
                    ushort index = 0;
                    byte dmyG = 0, dmyB = 0;
                    // fetch the index, perform the color look-up, convert to rgb
                    srcLayer.GetValue(col, row, ref index, ref dmyG, ref dmyB);

                    // ignore 'a' value; leave image alpha layer as is.
                    byte r, g, b, dummyAlpha;
                    file.palPtr.Get(index, out r, out g, out b, out dummyAlpha);

                    newLayer.SetValue(col, row, (ushort)r, g, b);
                }
            }

            return newLayer;
        }
    }
}
