﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// これは メイン DLL ファイルです。

#include "stdafx.h"
#include "LayoutTextureGeneric.h"
#include <windows.h>
#include <iostream>
#include <msclr/marshal_cppstd.h>

// seadでのnullptrのdefineで方のエラーが起こるのでundef
#undef nullptr

using namespace System;
using namespace msclr::interop;
using namespace System::Runtime::InteropServices;

namespace NW4F
{
namespace LayoutBinaryConverter
{
    //-----------------------------------------------------------------------------
    //! @biref 初期化します。
    //!
    //! @param[in] graphicsToolsPath エンコーダの DLL ファイルを探す GraphicsTools のパスです。
    //!                    例えば、\Tools\Graphics\GraphicsTools\ を指定してください。
    //!                    NULL なら SetDllPath で設定したパスを使用します。
    //!
    //!                    以下のようなファイルを探して利用します。
    //!                    GraphicsTools の並びに、3dTools フォルダが存在すると想定して読み込みを行います。
    //!
    //!                       \Tools\Graphics\GraphicsTools\TextureConverter.dll
    //!                       \Tools\Graphics\3dTools\3dTextureConverter.dll
    //!
    //! @return 処理結果を返します。
    //!
    int REncoder::Initialize(const wchar_t* graphicsToolsPath)
    {
        std::wstring gfxToolsPath = std::wstring(graphicsToolsPath);
        std::wstring gfxDirName = std::wstring(L"GraphicsTools");

        size_t pos = gfxToolsPath.rfind(gfxDirName);
        if (pos == std::string::npos)
        {
            std::wcerr << L"Can not find [GraphicsTools] folder. path=" << gfxToolsPath.c_str() << std::endl;
            std::wcerr << L"Please specified the folder of GraphicsTools.(e.g. SDKRoot/Tools/Graphics/GraphicsTools)" << std::endl;
            Finalize();
            return 1;
        }

        std::wstring g3dToolsPath = std::wstring(gfxToolsPath);
        g3dToolsPath.replace(pos, gfxDirName.length(), L"3dTools");

        // TextureConverter.dll
        const std::wstring converterPath = g3dToolsPath + L"\\" + L"3dTextureConverter.dll";
        m_hDll = LoadLibraryEx(converterPath.c_str(), NULL, LOAD_WITH_ALTERED_SEARCH_PATH);
        if (m_hDll == nullptr)
        {
            std::wcerr << L"Failure in LoadLibraryEx path=" << converterPath.c_str() << std::endl;
            Finalize();
            return 1;
        }

        // 関数アドレスを取得します。
        *reinterpret_cast<void**>(&SetOptions) = GetProcAddress(m_hDll, "SetOptions");
        *reinterpret_cast<void**>(&Clear) = GetProcAddress(m_hDll, "Clear");
        *reinterpret_cast<void**>(&ReadBitmapData) = GetProcAddress(m_hDll, "ReadBitmapData");
        *reinterpret_cast<void**>(&ConvertToData) = GetProcAddress(m_hDll, "ConvertToData");
        *reinterpret_cast<void**>(&GetErrorString) = GetProcAddress(m_hDll, "GetErrorString");

        if (SetOptions == nullptr || Clear == nullptr || ReadBitmapData == nullptr || ConvertToData == nullptr || GetErrorString == nullptr)
        {
            std::wcerr << L"Can't load TextureConverter functions." << std::endl;
            Finalize();
            return 1;
        }

        // TextureConverterEncoder
        const std::wstring encoderPath = gfxToolsPath + L"\\" + L"TextureConverterEncoder.dll";
        m_hDllEncoder = LoadLibraryEx(encoderPath.c_str(), NULL, LOAD_WITH_ALTERED_SEARCH_PATH);
        if (m_hDllEncoder == nullptr)
        {
            std::wcerr << L"Failure in LoadLibraryEx path=" << encoderPath.c_str() << std::endl;
            Finalize();
            return 1;
        }

        // 関数アドレスを取得します。
        *reinterpret_cast<void**>(&GetDataSize) = GetProcAddress(m_hDllEncoder, "GetDataSize");
        *reinterpret_cast<void**>(&ConvertFormat) = GetProcAddress(m_hDllEncoder, "ConvertFormat");
        *reinterpret_cast<void**>(&IsGpuEncodingAvailable) = GetProcAddress(m_hDllEncoder, "IsGpuEncodingAvailable");

        if (GetDataSize == nullptr || ConvertFormat == nullptr || IsGpuEncodingAvailable == nullptr)
        {
            std::wcerr << L"Can't load TextureConverterEncoder functions." << std::endl;
            Finalize();
            return 1;
        }

        m_TextureConverterExePath = g3dToolsPath + L"\\..\\" + L"3dTextureConverter.exe";

        // NXDLL は実際に利用する直前にロードを試行します。
        // Generic 動作時に、XN 用 DLL をロードしようとして、予期せぬエラーを起こす可能性を想定しての対処です。
        m_TextureConverterNXPath = gfxToolsPath + L"\\" + L"TextureConverterNX.dll";

        return 0;
    }

    //-----------------------------------------------------------------------------
    //! @biref NX用タイリング API を初期化します。
    int REncoder::InitializeNXTilingApi()
    {
        // TextureConverterNX
        m_hDllNX = LoadLibraryEx(m_TextureConverterNXPath.c_str(), NULL, LOAD_WITH_ALTERED_SEARCH_PATH);
        if (m_hDllNX == nullptr)
        {
            return 1;
        }

        // 関数アドレスを取得します。
        *reinterpret_cast<void**>(&ConvertTilingNX) = GetProcAddress(m_hDllNX, "ConvertTiling");
        *reinterpret_cast<void**>(&ReleaseTextureNX) = GetProcAddress(m_hDllNX, "ReleaseTexture");
        *reinterpret_cast<void**>(&GetDataSizeNX) = GetProcAddress(m_hDllNX, "GetDataSizeNX");

        if (ConvertTilingNX == nullptr || ReleaseTextureNX == nullptr || GetDataSizeNX == nullptr)
        {
            FreeLibrary(m_hDllNX);
            m_hDllNX = nullptr;
            return 1;
        }

        return 0;
    }

    //-----------------------------------------------------------------------------
    //! @biref 終了します。
    void REncoder::Finalize(void)
    {
        FreeLibrary(m_hDll);
        FreeLibrary(m_hDllEncoder);
        FreeLibrary(m_hDllNX);

        SetOptions = nullptr;
        Clear = nullptr;
        ReadBitmapData = nullptr;
        ConvertToData = nullptr;
        GetErrorString = nullptr;

        GetDataSize = nullptr;
        ConvertFormat = nullptr;

        m_hDll = nullptr;
        m_hDllEncoder = nullptr;
        m_hDllNX = nullptr;
    }

    //-----------------------------------------------------------------------------
    //! @biref フォーマット文字列を取得します。
    wchar_t* GetFormatString_(TexConv::TexelFormat format, bool degamma)
    {
        // 現状実装は nvn に合わせて、フォーマットを適当に変換している。Generic 実装としては適当でないと思うので、将来的に動作を切り替えられるように拡張が必要。
        switch (format)
        {
        case TexConv::TexelFormat::A4: return L"unorm_8";
        case TexConv::TexelFormat::A8: return L"unorm_8";
        case TexConv::TexelFormat::BC1: return degamma ? L"srgb_bc1" : L"unorm_bc1";
        case TexConv::TexelFormat::BC2: return degamma ? L"srgb_bc2" : L"unorm_bc2";
        case TexConv::TexelFormat::BC3: return degamma ? L"srgb_bc3" : L"unorm_bc3";
        case TexConv::TexelFormat::BC7: return degamma ? L"srgb_bc7" : L"unorm_bc7";
        case TexConv::TexelFormat::BC4A: return L"unorm_bc4";
        case TexConv::TexelFormat::BC4L: return L"unorm_bc4";
        case TexConv::TexelFormat::BC5: return L"unorm_bc5";
        case TexConv::TexelFormat::L4: return L"unorm_8";
        case TexConv::TexelFormat::L8: return L"unorm_8";
        case TexConv::TexelFormat::LA4: return L"unorm_bc5";
        case TexConv::TexelFormat::LA8: return L"unorm_8_8";
        case TexConv::TexelFormat::RGB565: return L"unorm_5_6_5";
        case TexConv::TexelFormat::RGB565_INDIRECT: return L"unorm_5_6_5";
        case TexConv::TexelFormat::RGB8: return degamma ? L"srgb_bc1" : L"unorm_bc1";
        case TexConv::TexelFormat::RGB5A1: return L"unorm_5_5_5_1";
        case TexConv::TexelFormat::RGBA4: return L"unorm_4_4_4_4";
        case TexConv::TexelFormat::RGBA8: return degamma ? L"srgb_8_8_8_8" : L"unorm_8_8_8_8";

        case TexConv::TexelFormat::ASTC4x4:  return degamma ? L"srgb_astc_4x4" : L"unorm_astc_4x4";
        case TexConv::TexelFormat::ASTC5x4:  return degamma ? L"srgb_astc_5x4" : L"unorm_astc_5x4";
        case TexConv::TexelFormat::ASTC5x5:  return degamma ? L"srgb_astc_5x5" : L"unorm_astc_5x5";
        case TexConv::TexelFormat::ASTC6x5:  return degamma ? L"srgb_astc_6x5" : L"unorm_astc_6x5";
        case TexConv::TexelFormat::ASTC6x6:  return degamma ? L"srgb_astc_6x6" : L"unorm_astc_6x6";
        case TexConv::TexelFormat::ASTC8x5:  return degamma ? L"srgb_astc_8x5" : L"unorm_astc_8x5";
        case TexConv::TexelFormat::ASTC8x6:  return degamma ? L"srgb_astc_8x6" : L"unorm_astc_8x6";
        case TexConv::TexelFormat::ASTC8x8:  return degamma ? L"srgb_astc_8x8" : L"unorm_astc_8x8";
        case TexConv::TexelFormat::ASTC10x5: return degamma ? L"srgb_astc_10x5" : L"unorm_astc_10x5";
        case TexConv::TexelFormat::ASTC10x6: return degamma ? L"srgb_astc_10x6" : L"unorm_astc_10x6";
        case TexConv::TexelFormat::ASTC10x8: return degamma ? L"srgb_astc_10x8" : L"unorm_astc_10x8";
        case TexConv::TexelFormat::ASTC10x10: return degamma ? L"srgb_astc_10x10" : L"unorm_astc_10x10";
        case TexConv::TexelFormat::ASTC12x10: return degamma ? L"srgb_astc_12x10" : L"unorm_astc_12x10";
        case TexConv::TexelFormat::ASTC12x12: return degamma ? L"srgb_astc_12x12" : L"unorm_astc_12x12";

        default: return nullptr;
        }
    }

    //-----------------------------------------------------------------------------
    bool Is2ChannelFormat_(TexConv::TexelFormat texFormat)
    {
        return
            texFormat == TexConv::TexelFormat::BC5 ||
            texFormat == TexConv::TexelFormat::LA4 ||
            texFormat == TexConv::TexelFormat::LA8 ||
            texFormat == TexConv::TexelFormat::HILO8;
    }

    //-----------------------------------------------------------------------------
    bool IsAlphaFormat_(TexConv::TexelFormat texFormat)
    {
        return
            texFormat == TexConv::TexelFormat::A4 ||
            texFormat == TexConv::TexelFormat::A8 ||
            texFormat == TexConv::TexelFormat::BC4A;
    }

    //-----------------------------------------------------------------------------
    // ガンマを sRGB からリニアに変換する処理を行います。
    // 0.00129465 -> (1 / Math::Pow(255.0, 2.2)) * 255.0
    unsigned char inline Degamma_(unsigned char srgbColor)
    {
        const double srgbGammaConst = 2.2;
        const double degammaConst = 0.00129465; // 0.00129465 -> (1 / Math::Pow(255.0, 2.2)) * 255.0

        return static_cast<unsigned char>(std::pow(srgbColor, srgbGammaConst) * degammaConst);
    }

    //-----------------------------------------------------------------------------
    void ToEncoderSrcFor2ChannelFormatForEncoder_(unsigned char* pDstImg, TexConv::SrcImage^ texImage, bool degamma)
    {
        if (degamma)
        {
            for (int y = 0; y < texImage->lyColor->Height; y++)
            {
                int yId = y * texImage->lyColor->Width;
                int yIdDst = yId * 4;
                int yIdSrcCol = yId * 3;
                for (int x = 0; x < texImage->lyColor->Width; x++)
                {
                    int xIdDst = x * 4;
                    int xIdSrc = x * 3;

                    pDstImg[yIdDst + xIdDst + 0] = Degamma_(texImage->lyColor->Data[yIdSrcCol + xIdSrc + 0]);
                    pDstImg[yIdDst + xIdDst + 1] = texImage->lyAlpha->Data[yId + x];
                    pDstImg[yIdDst + xIdDst + 2] = 0;
                    pDstImg[yIdDst + xIdDst + 3] = 0;
                }
            }
        }
        else
        {
            for (int y = 0; y < texImage->lyColor->Height; y++)
            {
                int yId = y * texImage->lyColor->Width;
                int yIdDst = yId * 4;
                int yIdSrcCol = yId * 3;
                for (int x = 0; x < texImage->lyColor->Width; x++)
                {
                    int xIdDst = x * 4;
                    int xIdSrc = x * 3;

                    pDstImg[yIdDst + xIdDst + 0] = texImage->lyColor->Data[yIdSrcCol + xIdSrc + 0];
                    pDstImg[yIdDst + xIdDst + 1] = texImage->lyAlpha->Data[yId + x];
                    pDstImg[yIdDst + xIdDst + 2] = 0;
                    pDstImg[yIdDst + xIdDst + 3] = 0;
                }
            }
        }
    }

    //-----------------------------------------------------------------------------
    void ToEncoderSrcFor2ChannelFormat_(unsigned char* pDstImg, TexConv::Layer^ colorLayer, TexConv::Layer^ alphaLayer, int alphaElementCount, bool degamma)
    {
        if (degamma)
        {
            for (int y = 0; y < colorLayer->Height; y++)
            {
                int yId = y * colorLayer->Width;
                int yIdDst = yId * 4;
                int yIdSrcCol = yId * 3;

                for (int x = 0; x < colorLayer->Width; x++)
                {
                    int xIdDst = x * 4;
                    int xIdSrc = x * 3;
                    int xIdSrcAlp = (yId + x) * alphaElementCount;

                    const unsigned char color = Degamma_(colorLayer->Data[yIdSrcCol + xIdSrc + 0]);

                    pDstImg[yIdDst + xIdDst + 0] = color;
                    pDstImg[yIdDst + xIdDst + 1] = alphaLayer->Data[xIdSrcAlp];
                    pDstImg[yIdDst + xIdDst + 2] = 0;
                    pDstImg[yIdDst + xIdDst + 3] = alphaLayer->Data[xIdSrcAlp];
                }
            }
        }
        else
        {
            for (int y = 0; y < colorLayer->Height; y++)
            {
                int yId = y * colorLayer->Width;
                int yIdDst = yId * 4;
                int yIdSrcCol = yId * 3;

                for (int x = 0; x < colorLayer->Width; x++)
                {
                    int xIdDst = x * 4;
                    int xIdSrc = x * 3;
                    int xIdSrcAlp = (yId + x) * alphaElementCount;

                    const unsigned char color = colorLayer->Data[yIdSrcCol + xIdSrc + 0];

                    pDstImg[yIdDst + xIdDst + 0] = color;
                    pDstImg[yIdDst + xIdDst + 1] = alphaLayer->Data[xIdSrcAlp];
                    pDstImg[yIdDst + xIdDst + 2] = 0;
                    pDstImg[yIdDst + xIdDst + 3] = alphaLayer->Data[xIdSrcAlp];
                }
            }
        }
    }

    //-----------------------------------------------------------------------------
    void ToEncoderSrcForRGB565Indirect_(unsigned char* pDstImg, TexConv::SrcImage^ texImage)
    {
        // RGB565_INDIRECTはインダイレクト用なので、degamma対応が必要ない。
        //
        // TexConv::TexelFormat::RGB565_INDIRECT
        // R->R A->G G->Bという詰め方をする
        for (int y = 0; y < texImage->lyColor->Height; y++)
        {
            int yId = y * texImage->lyColor->Width;
            int yIdDst = yId * 4;
            int yIdSrcCol = yId * 3;
            for (int x = 0; x < texImage->lyColor->Width; x++)
            {
                int xIdDst = x * 4;
                int xIdSrc = x * 3;

                pDstImg[yIdDst + xIdDst + 0] = texImage->lyColor->Data[yIdSrcCol + xIdSrc + 0];
                pDstImg[yIdDst + xIdDst + 1] = texImage->lyAlpha->Data[yId + x];
                pDstImg[yIdDst + xIdDst + 2] = texImage->lyColor->Data[yIdSrcCol + xIdSrc + 1];
                pDstImg[yIdDst + xIdDst + 3] = 255;
            }
        }
    }

    //-----------------------------------------------------------------------------
    void ToEncoderSrcForAlphaFormat_(unsigned char* pDstImg, TexConv::SrcImage^ texImage)
    {
        // アルファ用なので、degamma対応が必要ない。
        for (int y = 0; y < texImage->lyColor->Height; y++)
        {
            int yId = y * texImage->lyColor->Width;
            int yIdDst = yId * 4;
            int yIdSrcCol = yId * 3;
            for (int x = 0; x < texImage->lyColor->Width; x++)
            {
                int xIdDst = x * 4;

                pDstImg[yIdDst + xIdDst + 0] = texImage->lyAlpha->Data[yId + x];
                pDstImg[yIdDst + xIdDst + 1] = 0;
                pDstImg[yIdDst + xIdDst + 2] = 0;
                pDstImg[yIdDst + xIdDst + 3] = 0;
            }
        }
    }

    //-----------------------------------------------------------------------------
    void ToEncoderSrc_(unsigned char* pDstImg, TexConv::SrcImage^ texImage, bool degamma)
    {
        if (degamma)
        {
            for (int y = 0; y < texImage->lyColor->Height; y++)
            {
                int yId = y * texImage->lyColor->Width;
                int yIdDst = yId * 4;
                int yIdSrcCol = yId * 3;
                for (int x = 0; x < texImage->lyColor->Width; x++)
                {
                    int xIdDst = x * 4;
                    int xIdSrc = x * 3;

                    pDstImg[yIdDst + xIdDst + 0] = Degamma_(texImage->lyColor->Data[yIdSrcCol + xIdSrc + 0]);
                    pDstImg[yIdDst + xIdDst + 1] = Degamma_(texImage->lyColor->Data[yIdSrcCol + xIdSrc + 1]);
                    pDstImg[yIdDst + xIdDst + 2] = Degamma_(texImage->lyColor->Data[yIdSrcCol + xIdSrc + 2]);
                    pDstImg[yIdDst + xIdDst + 3] = texImage->lyAlpha->Data[yId + x];
                }
            }
        }
        else
        {
            for (int y = 0; y < texImage->lyColor->Height; y++)
            {
                int yId = y * texImage->lyColor->Width;
                int yIdDst = yId * 4;
                int yIdSrcCol = yId * 3;
                for (int x = 0; x < texImage->lyColor->Width; x++)
                {
                    int xIdDst = x * 4;
                    int xIdSrc = x * 3;

                    pDstImg[yIdDst + xIdDst + 0] = texImage->lyColor->Data[yIdSrcCol + xIdSrc + 0];
                    pDstImg[yIdDst + xIdDst + 1] = texImage->lyColor->Data[yIdSrcCol + xIdSrc + 1];
                    pDstImg[yIdDst + xIdDst + 2] = texImage->lyColor->Data[yIdSrcCol + xIdSrc + 2];
                    pDstImg[yIdDst + xIdDst + 3] = texImage->lyAlpha->Data[yId + x];
                }
            }
        }
    }

    //-----------------------------------------------------------------------------
    void ToEncoderSrcWithoutAlphaSource_(unsigned char* pDstImg, TexConv::SrcImage^ texImage, bool degamma)
    {
        if (degamma)
        {
            for (int y = 0; y < texImage->lyColor->Height; y++)
            {
                int yId = y * texImage->lyColor->Width;
                int yIdDst = yId * 4;
                int yIdSrcCol = yId * 3;
                for (int x = 0; x < texImage->lyColor->Width; x++)
                {
                    int xIdDst = x * 4;
                    int xIdSrc = x * 3;

                    pDstImg[yIdDst + xIdDst + 0] = Degamma_(texImage->lyColor->Data[yIdSrcCol + xIdSrc + 0]);
                    pDstImg[yIdDst + xIdDst + 1] = Degamma_(texImage->lyColor->Data[yIdSrcCol + xIdSrc + 1]);
                    pDstImg[yIdDst + xIdDst + 2] = Degamma_(texImage->lyColor->Data[yIdSrcCol + xIdSrc + 2]);
                    pDstImg[yIdDst + xIdDst + 3] = 255;
                }
            }
        }
        else
        {
            for (int y = 0; y < texImage->lyColor->Height; y++)
            {
                int yId = y * texImage->lyColor->Width;
                int yIdDst = yId * 4;
                int yIdSrcCol = yId * 3;
                for (int x = 0; x < texImage->lyColor->Width; x++)
                {
                    int xIdDst = x * 4;
                    int xIdSrc = x * 3;

                    pDstImg[yIdDst + xIdDst + 0] = texImage->lyColor->Data[yIdSrcCol + xIdSrc + 0];
                    pDstImg[yIdDst + xIdDst + 1] = texImage->lyColor->Data[yIdSrcCol + xIdSrc + 1];
                    pDstImg[yIdDst + xIdDst + 2] = texImage->lyColor->Data[yIdSrcCol + xIdSrc + 2];
                    pDstImg[yIdDst + xIdDst + 3] = 255;
                }
            }
        }
    }

    //-----------------------------------------------------------------------------
    array<System::Byte>^ ConvertToByteArray_(void* pImage, size_t imageSize)
    {
        array<System::Byte>^ dst_buffer = gcnew array<System::Byte>((int)imageSize);
        System::IntPtr ptr(pImage);
        System::Runtime::InteropServices::Marshal::Copy(ptr, dst_buffer, 0, (int)imageSize);

        return dst_buffer;
    }

    //-----------------------------------------------------------------------------
    // エンコーダーの初期化を行います。
    bool GenericTextureWriter::InitializeEncoder(System::String^ encoderPath)
    {
        if (m_pEncoder != nullptr)
        {
            return 1;
        }

        m_pEncoder = new REncoder();

        std::wstring stdPath = marshal_as<std::wstring>(encoderPath);

        return m_pEncoder->Initialize(stdPath.c_str()) == 0;
    }

    //-----------------------------------------------------------------------------
    // エンコーダーの終了処理を行います。
    void GenericTextureWriter::FinalizeEncoder()
    {
        delete m_pEncoder;
        m_pEncoder = nullptr;
    }

    //-----------------------------------------------------------------------------
    // Bntx のアライメントを読みます。
    // PC ツールから、nn ライブラリを利用する場合、nact で依存関係が設定できない制限があるので
    // 自前で解釈してアライメントを読み取ります。
    size_t GetBntxAlignmentShift_(const void* dstImageBuffer)
    {
        //struct nn::util::BinaryFileHeader
        //{
        //	//! @brief ファイルの種類を表す 8 つの 1 バイト値です。
        //	BinFileSignature signature;

        //	//! @brief バージョンを表す 4 つの 1 バイト値です。
        //	BinVersion version;

        //	uint16_t _byteOrderMark;        // エンディアンを識別するバイトオーダーマークです。
        //	uint8_t _alignmentShift;        // ファイルを配置する際に揃える必要があるアライメントの 2 を底とする指数部です。
        //	uint8_t _targetAddressSize;     // リソースのアドレスサイズです。
        //	uint32_t _offsetToFileName;     // ファイルの先頭からファイル名へのオフセットです。
        //	uint16_t _flag;                 // フラグです。
        //	uint16_t _offsetToFirstBlock;   // ファイルの先頭から最初のデータブロックへのオフセットです。
        //	uint32_t _offsetToRelTable;     // ファイルの先頭からリロケーションテーブルへのオフセットです。
        //	uint32_t _fileSize;             // ファイルサイズです。
        //};

        const uint8_t* pAlignmentShift = reinterpret_cast<const uint8_t*>(dstImageBuffer) + 8 + 4 + 2;
        return *pAlignmentShift;
    }

    //-----------------------------------------------------------------------------
    // Bntx のアライメントを読みます。
    // PC ツールから、nn ライブラリを利用する場合、nact で依存関係が設定できない制限があるので
    // 自前で解釈してアライメントを読み取ります。
    size_t GetBntxAlignmentShift_(array<byte>^ dstImageBuffer)
    {
        //struct nn::util::BinaryFileHeader
        //{
        //	//! @brief ファイルの種類を表す 8 つの 1 バイト値です。
        //	BinFileSignature signature;

        //	//! @brief バージョンを表す 4 つの 1 バイト値です。
        //	BinVersion version;

        //	uint16_t _byteOrderMark;        // エンディアンを識別するバイトオーダーマークです。
        //	uint8_t _alignmentShift;        // ファイルを配置する際に揃える必要があるアライメントの 2 を底とする指数部です。
        //	uint8_t _targetAddressSize;     // リソースのアドレスサイズです。
        //	uint32_t _offsetToFileName;     // ファイルの先頭からファイル名へのオフセットです。
        //	uint16_t _flag;                 // フラグです。
        //	uint16_t _offsetToFirstBlock;   // ファイルの先頭から最初のデータブロックへのオフセットです。
        //	uint32_t _offsetToRelTable;     // ファイルの先頭からリロケーションテーブルへのオフセットです。
        //	uint32_t _fileSize;             // ファイルサイズです。
        //};

        return dstImageBuffer[8 + 4 + 2];
    }

    //-----------------------------------------------------------------------------
    //
    void ReadConvertSourceImage_(unsigned char* pEncorderSrcImg, TexConv::SrcImage^ texImage, bool degamma, bool useBntx)
    {
        TexConv::TexelFormat texFormat = texImage->dstFormat;

        if (texImage->lyAlpha != nullptr)
        {
            if (Is2ChannelFormat_(texFormat))
            {
                if (useBntx)
                {
                    ToEncoderSrcFor2ChannelFormat_(pEncorderSrcImg, texImage->lyColor, texImage->lyAlpha, 1, degamma);
                }
                else{
                    ToEncoderSrcFor2ChannelFormatForEncoder_(pEncorderSrcImg, texImage, degamma);
                }
            }
            else if (texFormat == TexConv::TexelFormat::RGB565_INDIRECT)
            {
                ToEncoderSrcForRGB565Indirect_(pEncorderSrcImg, texImage);
            }
            else if (IsAlphaFormat_(texFormat))
            {
                ToEncoderSrcForAlphaFormat_(pEncorderSrcImg, texImage);
            }
            else
            {
                ToEncoderSrc_(pEncorderSrcImg, texImage, degamma);
            }
        }
        else
        {
            if (Is2ChannelFormat_(texFormat) && useBntx)
            {
                ToEncoderSrcFor2ChannelFormat_(pEncorderSrcImg, texImage->lyColor, texImage->lyColor, 3, degamma);
            }
            else
            {
                bool isDegammaEnalbed = !IsAlphaFormat_(texFormat) && degamma;
                ToEncoderSrcWithoutAlphaSource_(pEncorderSrcImg, texImage, isDegammaEnalbed);
            }
        }
    }

    //-----------------------------------------------------------------------------
    // \Programs\Iris\Sources\Libraries\ui2d\ui2d_Util.cpp で行っていたコンポーネント選択を行っています。
    const wchar_t* GetCompselStr_(TexConv::SrcImage^ texImage)
    {
        if(texImage->isIndirectTexture)
        {
            switch (texImage->dstFormat)
            {
            // 2チャンネルフォーマット
            case TexConv::TexelFormat::LA4:
            case TexConv::TexelFormat::LA8:
            case TexConv::TexelFormat::BC5:
                return L"rg01";
            // 3チャンネルフォーマット
            case TexConv::TexelFormat::RGB8:
            case TexConv::TexelFormat::RGB565:
            case TexConv::TexelFormat::RGB565_INDIRECT:
            case TexConv::TexelFormat::BC1:
                return L"rg0b";
            default:
                break;// 後続のコードで決定します。
            }
        }

        switch (texImage->dstFormat)
        {
        case TexConv::TexelFormat::A4: return L"111r";
        case TexConv::TexelFormat::A8: return L"111r";
        case TexConv::TexelFormat::BC1: return L"rgba";
        case TexConv::TexelFormat::BC2: return L"rgba";
        case TexConv::TexelFormat::BC3: return L"rgba";
        case TexConv::TexelFormat::BC7: return L"rgba";
        case TexConv::TexelFormat::BC4A: return L"111r";
        case TexConv::TexelFormat::BC4L: return L"rrr1";
        case TexConv::TexelFormat::BC5: return L"rrrg";
        case TexConv::TexelFormat::L4: return L"rrr1";
        case TexConv::TexelFormat::L8: return L"rrr1";
        case TexConv::TexelFormat::LA4: return L"rrrg";
        case TexConv::TexelFormat::LA8: return L"rrrg";
        case TexConv::TexelFormat::RGB565: return L"rgb1";
        case TexConv::TexelFormat::RGB565_INDIRECT: return L"rgb1";
        case TexConv::TexelFormat::RGB8: return L"rgb1";
        case TexConv::TexelFormat::RGBA4: return L"rgba";
        case TexConv::TexelFormat::RGBA8: return L"rgba";
        case TexConv::TexelFormat::ASTC4x4 : return L"rgba";
        case TexConv::TexelFormat::ASTC5x4 : return L"rgba";
        case TexConv::TexelFormat::ASTC5x5 : return L"rgba";
        case TexConv::TexelFormat::ASTC6x5 : return L"rgba";
        case TexConv::TexelFormat::ASTC6x6 : return L"rgba";
        case TexConv::TexelFormat::ASTC8x5 : return L"rgba";
        case TexConv::TexelFormat::ASTC8x6 : return L"rgba";
        case TexConv::TexelFormat::ASTC8x8 : return L"rgba";
        case TexConv::TexelFormat::ASTC10x5 : return L"rgba";
        case TexConv::TexelFormat::ASTC10x6 : return L"rgba";
        case TexConv::TexelFormat::ASTC10x8 : return L"rgba";
        case TexConv::TexelFormat::ASTC10x10: return L"rgba";
        case TexConv::TexelFormat::ASTC12x10: return L"rgba";
        case TexConv::TexelFormat::ASTC12x12: return L"rgba";

        default:
            return L"rgba";
        }
    }

    //-----------------------------------------------------------------------------
    bool ReadBitmapData_(REncoder* pEncoder, void* srcImageBuffer, TexConv::SrcImage^ texImage, bool isUseSrgbFormat, String^ tileMode, int depth, wchar_t* pDstExt, bool isUse2DArray, bool isLogSilent, String^ isGpuEncodigEnabled)
    {
        // 2DArray が指定されていないにもかかわらず、depth に1より大きな値が指定されている場合はエラーにします。
        if (!isUse2DArray && depth > 1)
        {
            std::wcerr << "Invalid texture depth was specified for 2D-Texture." << std::endl;
            return false;
        }

        const int TempBufferSize = 128;

        TexConv::TexelFormat texFormat = texImage->dstFormat;

        const wchar_t* pSrcFormatString = GetFormatString_(TexConv::TexelFormat::RGBA8, false);
        const wchar_t* pDstFormatString = GetFormatString_(texFormat, isUseSrgbFormat);

        marshal_context context;

        System::String^ imageName = texImage->name;
        std::wstring srcTexName = context.marshal_as<std::wstring>(imageName);

        const wchar_t* paths[] =
        {
            srcTexName.c_str(),
            NULL
        };

        const wchar_t* origPaths[] =
        {
            srcTexName.c_str(),
            NULL
        };

        // ログ出力オプションを指定します。
        {
            const wchar_t* option[] =
            {
                isLogSilent ? L"--silent" : NULL,
                NULL
            };
            pEncoder->SetOptions(option);
        }

        wchar_t bufferFmt[TempBufferSize];
        std::swprintf(bufferFmt, sizeof(bufferFmt), L"-f%s", pDstFormatString);

        wchar_t bufferDim[16];
        std::swprintf(bufferDim, sizeof(bufferDim), L"-d%s", isUse2DArray ? L"2d_array" : L"2d");

        wchar_t bufferDepth[TempBufferSize];
        std::swprintf(bufferDepth, sizeof(bufferDepth), L"-t%d", depth);

        wchar_t output[TempBufferSize];
        std::swprintf(output, sizeof(output), L"--output=%s.%s", origPaths[0], pDstExt);

        wchar_t compsel[TempBufferSize];
        std::swprintf(compsel, sizeof(compsel), L"--comp-sel=%s", GetCompselStr_(texImage));

        wchar_t tilemode[TempBufferSize];
        std::wstring tileModeString = context.marshal_as<std::wstring>(tileMode);
        std::swprintf(tilemode, sizeof(tilemode), L"--tile-mode=%s", tileModeString.c_str());

        wchar_t gpuencoding[TempBufferSize];
        std::wstring gpuEncodingString = context.marshal_as<std::wstring>(isGpuEncodigEnabled);
        std::swprintf(gpuencoding, sizeof(gpuencoding), L"--gpu-encoding=%s", gpuEncodingString.c_str());

        const wchar_t* formatOptions[] =
        {
            bufferFmt,
            L"-i1",
            gpuencoding,
            output,
            bufferDim,
            compsel,
            tilemode,
            L"--disable-file-info",
            depth > 1  ? bufferDepth : NULL,
            NULL
        };

        const int srcW = texImage->lyColor->Width;
        const int srcH = texImage->lyColor->Height;

        return pEncoder->ReadBitmapData(srcImageBuffer, srcW, srcH, depth, texImage->lyAlpha != nullptr, false, paths, origPaths, formatOptions);
    }

    //-----------------------------------------------------------------------------
    bool WriteDataToFile_(void* pSrcBuffer, size_t size, String^ outPath)
    {
        System::IO::FileStream^ fs = gcnew System::IO::FileStream(outPath, System::IO::FileMode::Create);
        System::IO::BinaryWriter^ w = gcnew System::IO::BinaryWriter(fs);

        try
        {
            unsigned char* pSrc = reinterpret_cast<unsigned char*>(pSrcBuffer);
            for (size_t i=0; i < size; i++)
            {
                w->Write(pSrc[i]);
            }
        }
        catch (Exception^ e)
        {
            std::wcerr << marshal_as<std::wstring>(e->ToString()) << std::endl;

            fs->Close();
            return false;
        }

        fs->Close();

        return true;
    }

    //-----------------------------------------------------------------------------
    //
    TexConv::ConvertedImage^ ConvertBntx2DArray(
        REncoder* pEncoder, array<TexConv::SrcImage^>^ texImage, String^ tileMode, bool isLogSilent, String^ isGpuEncodigEnabled)
    {
        const int srcW = texImage[0]->lyColor->Width;
        const int srcH = texImage[0]->lyColor->Height;
        const size_t srcImageSize = srcW * srcH * 4;

        bool useBntx = true;

        std::unique_ptr<unsigned char> pSrcImageBuffer(new unsigned char[srcImageSize * texImage->Length ]);
        unsigned char* srcImageBuffer = pSrcImageBuffer.get();

        // 複数画像をまとめてソースバッファに読み込んで変換します。
        for (int i = 0; i < texImage->Length; i++)
        {
            unsigned char* pEncorderSrcImg = static_cast<unsigned char*>(srcImageBuffer) + srcImageSize * i;
            ReadConvertSourceImage_(pEncorderSrcImg, texImage[i], texImage[i]->isDegamma, useBntx);
        }

        const bool isUse2DArray = true;
        bool readResult = ReadBitmapData_(pEncoder, srcImageBuffer, texImage[0], texImage[0]->isSrgbFormat, tileMode, texImage->Length, L"bntx", isUse2DArray, isLogSilent, isGpuEncodigEnabled);
        if (!readResult)
        {
            std::wcerr << pEncoder->GetErrorString() << std::endl;
            pEncoder->Clear();
            return nullptr;
        }

        void* dstImageBuffer;
        size_t dstImageSize;
        bool result = pEncoder->ConvertToData((const void**)&dstImageBuffer, &dstImageSize);

        if (!result)
        {
            std::wcerr << pEncoder->GetErrorString() << std::endl;
            pEncoder->Clear();
            return nullptr;
        }

        // 変換結果を、マネージド形式にコピーして返す。
        TexConv::ConvertedImage^ convertedImage = gcnew TexConv::ConvertedImage;
        convertedImage->image = ConvertToByteArray_(dstImageBuffer, dstImageSize);

        // 注：以下は 共通テクスチャフォーマットが整備されるまでの暫定処理です。
        convertedImage->neededAlignment = (int)GetBntxAlignmentShift_(dstImageBuffer);
        convertedImage->tileMode = 0x00000000; // 現在は、利用していません。
        pEncoder->Clear();

        return convertedImage;
    }

    //-----------------------------------------------------------------------------
    //
    bool GenericTextureWriter::ExportSrcImageToFtxbFile(TexConv::SrcImage^ texImage, String^ apiTypeName, String^ outputDir, String^ cacheDir)
    {
        if (m_pEncoder == nullptr)
        {
            return false;
        }

        String^ ftxPath = System::IO::Path::Combine(outputDir, texImage->name + gcnew System::String(L".ftxb"));

        String^ cacheHashValue = nullptr;
        String^ cacheFilePath = nullptr;
        if (!System::String::IsNullOrEmpty(cacheDir))
        {
            TexConv::CacheKeyGenerator^ gen = gcnew TexConv::CacheKeyGenerator();
            try
            {
                // テクスチャの情報をキーに含める
                texImage->AddCacheKey(gen);

                // 動作に影響のある環境変数をキーに含める
                String^ variable = "NINTENDO_TEXTURE_CONVERTER_NVTT_BC123";
                gen->AddCacheKey(variable);
                String^ value = Environment::GetEnvironmentVariable(variable);
                gen->AddCacheKey(value != nullptr ? value : String::Empty);
                // m_isGpuEncodigEnabled をキーに含める
                gen->AddCacheKey("GpuEncodigEnabled");
                gen->AddCacheKey(m_isGpuEncodigEnabled);

                cacheHashValue = gen->GetCacheKeyHashValue();
                cacheFilePath = System::IO::Path::Combine(cacheDir, cacheHashValue + gcnew System::String(L".ftxb"));
            }
            finally
            {
                delete gen;
            }
        }


        System::Threading::Mutex^ cacheFileMutex = nullptr;
        try
        {
            if (!System::String::IsNullOrEmpty(cacheHashValue))
            {
                cacheFileMutex = gcnew System::Threading::Mutex(false, gcnew System::String("NW_FontConverter_") + cacheHashValue);
                cacheFileMutex->WaitOne();

                // キャッシュを使用する
                if (System::IO::File::Exists(cacheFilePath))
                {
                    System::IO::File::Copy(cacheFilePath, ftxPath, true);

                    // ファイル属性から読み取り専用を削除
                    System::IO::FileAttributes fas = System::IO::File::GetAttributes(ftxPath) & ~System::IO::FileAttributes::ReadOnly;
                    System::IO::File::SetAttributes(ftxPath, fas);

                    DateTime now = DateTime::Now;
                    // 更新日時に関係する問題が起きるのを避けるため各種時間を更新しておく
                    System::IO::File::SetCreationTime(ftxPath, now);
                    System::IO::File::SetLastWriteTime(ftxPath, now);
                    System::IO::File::SetLastAccessTime(ftxPath, now);

                    return true;
                }
            }

            bool useBntx = true;

            // 一旦すべての画像を ftx に出力します。（変更のないファイルは変換をスキップするように修正します）
            const int srcW = texImage->lyColor->Width;
            const int srcH = texImage->lyColor->Height;
            const size_t srcImageSize = srcW * srcH * 4;

            std::unique_ptr<unsigned char> pSrcImageBuffer(new unsigned char[srcImageSize]);
            unsigned char* srcImageBuffer = pSrcImageBuffer.get();

            ReadConvertSourceImage_(srcImageBuffer, texImage, texImage->isDegamma, useBntx);

            // 不必要なタイリング処理を避けるために、ftxb 出力時に指定するフォーマットは linear です。
            String^ linearTileMode = gcnew String("linear");
            const bool isUse2DArray = false; // ui2d など通常用のテクスチャでは、2dArray 形式は利用しません。
            bool readResult = ReadBitmapData_(m_pEncoder, srcImageBuffer, texImage, texImage->isSrgbFormat, linearTileMode, 1, L"ftxb", isUse2DArray, m_isLogSilent, m_isGpuEncodigEnabled);
            if (!readResult)
            {
                std::wcerr << m_pEncoder->GetErrorString() << std::endl;
                m_pEncoder->Clear();
                return false;
            }

            void* dstImageBuffer;
            size_t dstImageSize;
            bool result = m_pEncoder->ConvertToData((const void**)&dstImageBuffer, &dstImageSize);

            if (!result)
            {
                std::wcerr << m_pEncoder->GetErrorString() << std::endl;
                m_pEncoder->Clear();
                return false;
            }

            // ftxファイル書き出しします。
            WriteDataToFile_(dstImageBuffer, dstImageSize, ftxPath);
            m_pEncoder->Clear();

            // キャッシュに保存する
            if (!System::String::IsNullOrEmpty(cacheHashValue))
            {
                System::IO::File::Copy(ftxPath, cacheFilePath, true);
            }

            return true;
        }
        finally
        {
            if (cacheFileMutex != nullptr)
            {
                // WaitOne 分を開放する
                cacheFileMutex->ReleaseMutex();

                delete cacheFileMutex;
            }
        }
    }

    //-----------------------------------------------------------------------------
    //
    TexConv::ConvertedImage^
        GenericTextureWriter::ConvertSrcImageToNative2DArrayFormat(array<TexConv::SrcImage^>^ texImage, System::String^ tileMode, bool useBntx)
    {
        if (m_pEncoder == nullptr)
        {
            return nullptr;
        }

        if(useBntx)
        {
            return ConvertBntx2DArray(m_pEncoder, texImage, tileMode, m_isLogSilent, m_isGpuEncodigEnabled);

        }else{

            std::wcerr << "error : Unexpected case !!(ConvertSrcImageToNative2DArrayFormat(bntx=false) is not implemented.)" << std::endl;
            return nullptr;
        }
    }

    //-----------------------------------------------------------------------------
    //
    System::IntPtr GenericTextureWriter::ConvertTiling( bool isTiling, System::IntPtr pSrcData, System::UInt64 srcDataSize, System::String^ dimension, System::String^ format, int imageW, int imageH, int imageD, int arrayLength, int mipCount, System::String^ tileMode, System::UInt64% dstDataSize, System::IntPtr pTextureLayout, bool isSparseTiled )
    {
        System::String^ tileModeNX = "NX";
        if(String::Compare(tileMode, tileModeNX, System::StringComparison::InvariantCultureIgnoreCase) == 0)
        {
            if(m_pEncoder != nullptr)
            {
                // 未初期化なら初期化を試行します。
                if(!m_pEncoder->IsNXInitialized())
                {
                    if(m_pEncoder->InitializeNXTilingApi() != 0)
                    {
                        std::wcerr << L"Can't load TextureConverterNX tiling functions." << std::endl;
                    }
                }

                if(m_pEncoder->IsNXInitialized())
                {
                    std::wstring dimantionStr = marshal_as<std::wstring>(dimension);
                    std::wstring formatStr = marshal_as<std::wstring>(format);
                    nn::gfx::tool::texcvtr::NXTexture texture;

                    bool ret = m_pEncoder->ConvertTilingNX(&texture,
                        isTiling,
                        (void*)pSrcData,
                        (size_t)srcDataSize,
                        dimantionStr.c_str(),
                        formatStr.c_str(),
                        imageW,
                        imageH,
                        imageD,
                        arrayLength,
                        mipCount,
                        (void*)pTextureLayout,
                        isSparseTiled ? nn::gfx::tool::texcvtr::NXTilingFlag_Sparse : nn::gfx::tool::texcvtr::NXTilingFlag_Default,
                        0);

                    if(ret)
                    {
                        // DLLが確保されたメモリからコピー
                        dstDataSize = texture.dataSize;
                        auto mem = new char[texture.dataSize];
                        memcpy(mem, texture.pData, texture.dataSize);
                        m_pEncoder->ReleaseTextureNX(&texture);

                        return (System::IntPtr)mem;
                    }
                }
            }
        }

        //タイリング処理が行われなかった場合
        dstDataSize = 0;
        return (System::IntPtr)nullptr;
    }

    //-----------------------------------------------------------------------------
    //
    int GenericTextureWriter::GetNativeDataSize(TexConv::TexelFormat format, int imageW, int imageH)
    {
        if (m_pEncoder->GetDataSize == nullptr)
        {
            std::wcerr << L"Can't load TextureConverterNX GetDataSize functions." << std::endl;
            return 0;
        }

        const wchar_t* pDstFormatString = GetFormatString_(format, false);
        const size_t size = m_pEncoder->GetDataSize(
            pDstFormatString,
            nn::gfx::tool::texenc::ImageDimension_2d,
            imageW,
            imageH,
            1,
            1
        );

        return static_cast<int>(size);
    }

    int GenericTextureWriter::GetNativeDataSizeNX(TexConv::TexelFormat format, int imageW, int imageH, uint32_t tilingFlags, int sizeThreshold)
    {
        if (InitializeNXApi() == false)
        {
            return 0;
        }

        if (m_pEncoder->GetDataSizeNX == nullptr)
        {
            std::wcerr << L"Can't load TextureConverterNX GetDataSizeNX functions." << std::endl;
            return 0;
        }

        const wchar_t* pDstFormatString = GetFormatString_(format, false);
        const size_t size = m_pEncoder->GetDataSizeNX(
            L"2d",
            pDstFormatString,
            imageW,
            imageH,
            1,
            1,
            1,
            tilingFlags,
            sizeThreshold
        );

        return static_cast<int>(size);
    }

    bool GenericTextureWriter::InitializeNXApi()
    {
        if (!m_pEncoder->IsNXInitialized())
        {
            if (m_pEncoder->InitializeNXTilingApi() != 0)
            {
                std::wcerr << L"Can't load TextureConverterNX tiling functions." << std::endl;
                return false;
            }
        }

        return true;
    }

    bool GenericTextureWriter::IsGpuEncodingAvailable()
    {
        return m_pEncoder->IsGpuEncodingAvailable();
    }

}   // namespace LayoutBinaryConverter
}   // namespace NW4F



