﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------


// このソースコードは ImageFitMesh として独立した DLL で使われていたものですが
// ツールのビルド後イベントでワイルドカードでの dll コピーを行った際に nact でエラーが発生することから
// 暫定対応として sub プロジェクトへソースコードで取り込んでいます。
// 根本対処は SIGLO-65243 で行う予定です。

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Drawing;

namespace ImageFitMesh
{
    /// <summary>
    /// 2次元の頂点データ
    /// </summary>
    public struct Vertex2D
    {
        public double   x;
        public double   y;

        public Vertex2D(double x_, double y_)
        {
            x = x_;
            y = y_;
        }

        public void Set(double x_, double y_)
        {
            x = x_;
            y = y_;
        }

        public double Length()
        {
            return Math.Sqrt(x * x + y * y);
        }

        public void Normalize()
        {
            double fInvLength = 1.0 / Length();
            x *= fInvLength;
            y *= fInvLength;
        }

        public static Vertex2D operator +(Vertex2D lhs, Vertex2D rhs)
        {
            return new ImageFitMesh.Vertex2D(lhs.x + rhs.x, lhs.y + rhs.y);
        }

        public static Vertex2D operator -(Vertex2D lhs, Vertex2D rhs)
        {
            return new ImageFitMesh.Vertex2D(lhs.x - rhs.x, lhs.y - rhs.y);
        }

        public static Vertex2D operator *(Vertex2D lhs, double scale)
        {
            return new ImageFitMesh.Vertex2D(lhs.x * scale, lhs.y * scale);
        }

        public static double Dot(Vertex2D lhs, Vertex2D rhs)
        {
            return lhs.x * rhs.x + rhs.y * rhs.y;
        }

        public static double Cross(Vertex2D lhs, Vertex2D rhs)
        {
            return lhs.x * rhs.y - lhs.y * rhs.x;
        }
    };

    /// <summary>
    /// 最適化タイプ。
    /// </summary>
    public enum OptimizationType
    {
        None,
        AABB,
        OBB,
        Max
    }

    /// <summary>
    /// メッシュ生成の結果
    /// </summary>
    public enum CreateMeshResult
    {
        TransparentImage, // 完全に透明な画像
        OpaqueEdgeRepeat, // 不透明エッジが引き伸ばされる
        Succeeded,
        Max
    };

    public class Generator
    {
        /// <summary>
        /// アルファチャンネルの不透明領域にフィットする AABB を作成します。
        /// </summary>
        /// <param name="vertexList"></param>
        /// <param name="optimizedWidth"></param>
        /// <param name="optimizedHeight"></param>
        /// <param name="srcAlphaImage"></param>
        /// <param name="width"></param>
        /// <param name="height"></param>
        public static CreateMeshResult CreateAABBMesh(out Vertex2D[] vertexList, out int optimizedWidth, out int optimizedHeight, Byte[] srcAlphaImage, int width, int height, int margin)
        {
            vertexList = new Vertex2D[4];

            var verticalSum = new int[width];
            var total = 0;

            // 縦横それぞれのピクセルのアルファ値の合計から透明な行と列を決定する。

            // 列のアルファ値の合計を計算。
            for (int x = 0; x < width; ++x)
            {
                verticalSum[x] = 0;
                for (int y = 0; y < height; ++y)
                {
                    verticalSum[x] += srcAlphaImage[y * width + x];
                }
                total += verticalSum[x];
            }

            //  全てのピクセルが透明。
            //  最適化のしようがないのでそのままの矩形を返す。
            if (total == 0)
            {
                optimizedWidth = width;
                optimizedHeight = height;
                return CreateMeshResult.TransparentImage;
            }

            var horizontalSum = new int[height];

            // 行のアルファ値の合計を計算。
            for (int y = 0; y < height; ++y)
            {
                horizontalSum[y] = 0;
                for (int x = 0; x < width; ++x)
                {
                    horizontalSum[y] += srcAlphaImage[y * width + x];
                }
            }

            int left = 0;
            int right = width - 1;

            for (int x = 0; x < width; ++x)
            {
                left = x;
                if (verticalSum[x] != 0)
                {
                    break;
                }
            }

            for (int x = width-1; x >= 0; --x)
            {
                right = x;
                if (verticalSum[x] != 0)
                {
                    break;
                }
            }

            int top = 0;
            int bottom = height - 1;

            for (int y = 0; y < height; ++y)
            {
                top = y;
                if (horizontalSum[y] != 0)
                {
                    break;
                }
            }

            for (int y = height-1; y >= 0; --y)
            {
                bottom = y;
                if (horizontalSum[y] != 0)
                {
                    break;
                }
            }

            left -= margin;
            if (left < 0)
            {
                left = 0;
            }

            right += margin;
            if (right >= width)
            {
                right = width - 1;
            }

            top -= margin;
            if (top < 0)
            {
                top = 0;
            }

            bottom += margin;
            if (bottom >= height)
            {
                bottom= height - 1;
            }

            double fLeft = (double)left / (double)width;
            double fRight = (double)(right + 1) / (double)width;
            double fTop = (double)top / (double)height;
            double fBottom = (double)(bottom + 1) / (double)height;

            vertexList[0] = new Vertex2D(fLeft, fTop);
            vertexList[1] = new Vertex2D(fRight, fTop);
            vertexList[2] = new Vertex2D(fLeft, fBottom);
            vertexList[3] = new Vertex2D(fRight, fBottom);

            optimizedWidth = (right + 1) - left;
            optimizedHeight = (bottom + 1) - top;

            return CreateMeshResult.Succeeded;
        }

        //-----------------------------------------------------------------------------
        //! @brief 有向バウンディングボックスの jocobi 計算です。
        //!        NUMERICAL RECIPES in C より
        //!        http://www.library.cornell.edu/nr/bookcpdf/c11-1.pdf
        //-----------------------------------------------------------------------------
        static void JacobiRotate(ref double[,] a, double s, double tau, int i, int j, int k, int l)
        {
            double h, g;
            g = a[i, j];
            h = a[k, l];
            a[i, j] = g - s * (h + g * tau);
            a[k, l] = h + s * (g - h * tau);
        }

        const double R_SAME_TOLERANCE_F = 1.0e-6;

        /// <summary>
        /// 浮動小数点数の誤差許容付きの同一チェック
        /// </summary>
        /// <param name="a"></param>
        /// <param name="b"></param>
        /// <returns></returns>
        static bool RIsSame(double a, double b)
        {
            if (a == b)
            {
                return true;
            }
            double c = a - b;
            return (-R_SAME_TOLERANCE_F < c && c < R_SAME_TOLERANCE_F);
        }

        // return true is solved
        /// <summary>
        /// Jacobi 法による固有値と固有ベクトルの計算。
        /// </summary>
        /// <param name="a"></param>
        /// <param name="v"></param>
        /// <param name="d"></param>
        /// <returns></returns>
        static bool JacobiCalc(double[,] a, ref double[,] v, ref double[] d)
        {
            int n = 2;
            int i, j, iq, ip;
            double tresh, theta, tau, t, sm, s, h, g, c;
            double[] b = new double[2];
            double[] z = new double[2];

            for (ip = 0; ip < n; ip++)
            {
                for (iq = 0; iq < n; iq++) v[ip, iq] = 0.0;
                v[ip, ip] = 1.0;
            }
            for (ip = 0; ip < n; ip++)
            {
                b[ip] = d[ip] = a[ip, ip];
                z[ip] = 0.0;
            }
            for (i = 0; i < 50; i++)
            {
                sm = 0.0;
                for (ip = 0; ip < n - 1; ip++)
                {
                    for (iq = ip + 1; iq < n; iq++) sm += Math.Abs(a[ip, iq]);
                }

                if (RIsSame(sm, 0.0)) return true;
                if (i < 3) tresh = 0.2 * sm / (n * n);
                else tresh = 0.0;
                for (ip = 0; ip < n - 1; ip++)
                {
                    for (iq = ip + 1; iq < n; iq++)
                    {
                        g = 100.0 * Math.Abs(a[ip, iq]);
                        if (i > 3 && RIsSame((Math.Abs(d[ip]) + g), Math.Abs(d[ip]))
                            && RIsSame((Math.Abs(d[iq]) + g), Math.Abs(d[iq]))) a[ip, iq] = 0.0;
                        else if (Math.Abs(a[ip, iq]) > tresh)
                        {
                            h = d[iq] - d[ip];
                            if (RIsSame((Math.Abs(h) + g), Math.Abs(h))) t = a[ip, iq] / h;
                            else
                            {
                                theta = 0.5 * h / a[ip, iq];
                                t = 1.0 / (Math.Abs(theta) + Math.Sqrt(1.0 + theta * theta));
                                if (theta < 0.0) t = -t;
                            }
                            c = 1.0 / Math.Sqrt(1 + t * t);
                            s = t * c;
                            tau = s / (1.0 + c);
                            h = t * a[ip, iq];
                            z[ip] -= h;
                            z[iq] += h;
                            d[ip] -= h;
                            d[iq] += h;
                            a[ip, iq] = 0.0;

                            for (j = 0; j < ip; j++) JacobiRotate(ref a, s, tau, j, ip, j, iq);
                            for (j = ip + 1; j < iq; j++) JacobiRotate(ref a, s, tau, ip, j, j, iq);
                            for (j = iq + 1; j < n; j++) JacobiRotate(ref a, s, tau, ip, j, iq, j);
                            for (j = 0; j < n; j++) JacobiRotate(ref v, s, tau, j, ip, j, iq);
                        }
                    }
                }
                for (ip = 0; ip < n; ip++)
                {
                    b[ip] += z[ip];
                    d[ip] = b[ip];
                    z[ip] = 0.0;
                }
            }
            return false;
        }

        struct EIGEN_SORT
        {
            public int ID;
            public double Value;

            public EIGEN_SORT(int id_, double value_)
            {
                ID = id_;
                Value = value_;
            }
        }

        /// <summary>
        /// OBB で作成されたポリゴンのエッジ不透明チェック。
        /// </summary>
        /// <param name="vertexList">頂点リスト</param>
        /// <param name="pointConnectIndex">頂点リストの接続順序</param>
        /// <param name="srcAlphaImage">アルファイメージ</param>
        /// <param name="width">アルファイメージの幅</param>
        /// <param name="height">アルファイメージの高さ</param>
        /// <returns></returns>
        static bool RepeqtEdgeOpaqueTest(Vertex2D[] vertexList, int[] pointConnectIndex, Byte[] srcAlphaImage, int width, int height)
        {
            List<Vertex2D>[] intersectPoints = new List<Vertex2D>[vertexList.Length];
            Vertex2D[,] edgeLineList = new Vertex2D[,]
            {
                { new Vertex2D(0.0, 0.0), new Vertex2D(1.0, 0.0) }, // 上
                { new Vertex2D(0.0, 1.0), new Vertex2D(1.0, 0.0) }, // 下
                { new Vertex2D(1.0, 0.0), new Vertex2D(0.0, 1.0) }, // 右
                { new Vertex2D(0.0, 0.0), new Vertex2D(0.0, 1.0) }  // 左
            };

            for (int i = 0; i < 4; ++i)
            {
                intersectPoints[i] = new List<Vertex2D>();

                Vertex2D edgeStart = edgeLineList[i, 0];
                Vertex2D edgeVec = edgeLineList[i, 1];

                for (int j = 0; j < vertexList.Length; ++j)
                {
                    Vertex2D targetStart = vertexList[pointConnectIndex[j]];
                    Vertex2D targetVec = vertexList[(pointConnectIndex[j] + 1) % 4] - targetStart;

                    if (Vertex2D.Cross(edgeVec, targetVec) == 0.0f)
                    {
                        // 平行
                    }
                    else
                    {
                        Vertex2D toStart = targetStart - edgeStart;
                        double t = Vertex2D.Cross(toStart, edgeVec) / Vertex2D.Cross(edgeVec, targetVec);

                        // 線分の範囲内での交点。
                        if (t >= 0.0f && t < 1.0)
                        {
                            intersectPoints[i].Add(targetStart + (targetVec * t));
                        }
                    }
                }
            }

            for(int i = 0; i < vertexList.Length;++i)
            {
                // 一つのライン上で 2 点交差していればその範囲はポリゴンの内部になるため
                // エッジ上に不透明ピクセルが存在しないかチェックする。
                if (intersectPoints[i].Count == 2)
                {
                    // edgeLine の方向からチェックする方向を決定する
                    if (i < 2)
                    {
                        int startX = (int)Math.Round(intersectPoints[i][0].x * (width-1));
                        int endX = (int)Math.Round(intersectPoints[i][1].x * (width-1));
                        int y = (int)Math.Round(intersectPoints[i][0].y * (height-1));

                        for (int x = startX; x < endX; ++x)
                        {
                            if (srcAlphaImage[width * y + x] != 0)
                            {
                                return false;
                            }
                        }
                    }
                    else
                    {
                        int startY = (int)Math.Round(intersectPoints[i][0].y * (height-1));
                        int endY = (int)Math.Round(intersectPoints[i][1].y * (height-1));
                        int x = (int)Math.Round(intersectPoints[i][0].x * (width-1));

                        for (int y = startY; y < endY; ++y)
                        {
                            if (srcAlphaImage[width * y + x] != 0)
                            {
                                return false;
                            }
                        }
                    }
                }
            }


            return true;
        }

        /// <summary>
        /// 4 頂点 OBB メッシュを作成します。
        /// </summary>
        /// <param name="vertexList">出力頂点リスト</param>
        /// <param name="axis">OBB メッシュの軸</param>
        /// <param name="optimizedWidth">最適化結果の幅</param>
        /// <param name="optimizedHeight">最適化結果の高さ</param>
        /// <param name="srcAlphaImage">OBB 作成のためのアルファイメージ</param>
        /// <param name="width">アルファイメージ幅</param>
        /// <param name="height">アルファイメージの高さ</param>
        /// <param name="margin">固定マージン</param>
        public static CreateMeshResult CreateOBBMesh(out Vertex2D[] vertexList, out Vertex2D[] axis, out int optimizedWidth, out int optimizedHeight, Byte[] srcAlphaImage, int width, int height, int margin)
        {
            vertexList = new Vertex2D[4];
            axis = new Vertex2D[2];
            optimizedWidth = width;
            optimizedHeight = height;

            var pixels = new Vertex2D[width * height * 4];
            int pixelCount = 0;
            for (int y = 0; y < height; ++y)
            {
                for (int x = 0; x < width; ++x)
                {
                    if (srcAlphaImage[y * width + x] != 0)
                    {
                        pixels[pixelCount++] = new Vertex2D((double)x       , (double)y);
                        pixels[pixelCount++] = new Vertex2D((double)x + 1.0, (double)y);
                        pixels[pixelCount++] = new Vertex2D((double)x       , (double)y + 1.0);
                        pixels[pixelCount++] = new Vertex2D((double)x + 1.0, (double)y + 1.0);
                    }
                }
            }

            if (pixelCount == 0)
            {
                return CreateMeshResult.TransparentImage;
            }

            // 平均位置 m を計算
            var m = new Vertex2D(0.0, 0.0);
            foreach (var pixel in pixels)
            {
                m.x += pixel.x;
                m.y += pixel.y;
            }
            double invPosCount = 1.0 / pixelCount;
            m.x *= invPosCount;
            m.y *= invPosCount;

            // 共分散行列を作成
            double C11 = 0.0, C22 = 0.0, C12 = 0.0;
            for (int i = 0; i < pixelCount;++i)
            {
                var pixel = pixels[i];
                C11 += (pixel.x - m.x) * (pixel.x - m.x);
                C22 += (pixel.y - m.y) * (pixel.y - m.y);
                C12 += (pixel.x - m.x) * (pixel.y - m.y);
            }
            C11 *= invPosCount;
            C22 *= invPosCount;
            C12 *= invPosCount;

            double[,] cvMtx = new double[2,2]
            {
                { C11, C12 },
                { C12, C22 }
            };

            // jacobi 法で固有値 & 固有ベクトルを算出
            double[,]    EigenVectors = new double[2, 2];
            double[]     EigenValue = new double[2];
            bool solved = JacobiCalc(cvMtx, ref EigenVectors, ref EigenValue);
            if (solved)
            {
                // 固有値を降順でソート
                EIGEN_SORT[] Sort = new EIGEN_SORT[2]
                {
                    new EIGEN_SORT(0, EigenValue[0]),
                    new EIGEN_SORT(1, EigenValue[1])
                };

                if (Sort[0].Value < Sort[1].Value)
                {
                    EIGEN_SORT a = Sort[1];
                    Sort[1] = Sort[0];
                    Sort[0] = a;
                }


                for (int i = 0; i< 2; ++i)
                {
                    axis[i].x = EigenVectors[0, Sort[i].ID];
                    axis[i].y = EigenVectors[1, Sort[i].ID];
                }
            }
            else
            {
                // 解がなければ AABB に
                axis[0].Set(1.0, 0.0);
                axis[1].Set(0.0, 1.0);
            }

            // 境界ボックスを算出
            double[] XfMin = new double[2];
            double[] XfMax = new double[2];

            XfMax[0] = XfMin[0] = axis[0].x * pixels[0].x + axis[0].y * pixels[0].y;
            XfMax[1] = XfMin[1] = axis[1].x * pixels[1].x + axis[1].y * pixels[1].y;

            for (int ipos = 1; ipos<pixelCount; ++ipos)
            {
                double XfTemp;
                XfTemp = axis[0].x * pixels[ipos].x + axis[0].y * pixels[ipos].y;
                XfMin[0] = Math.Min(XfMin[0], XfTemp);
                XfMax[0] = Math.Max(XfMax[0], XfTemp);

                XfTemp = axis[1].x * pixels[ipos].x + axis[1].y * pixels[ipos].y;
                XfMin[1] = Math.Min(XfMin[1], XfTemp);
                XfMax[1] = Math.Max(XfMax[1], XfTemp);
            }

            var centerPos = new Vertex2D();
            centerPos.x =
                axis[0].x * ((XfMax[0] + XfMin[0]) * 0.5) +
                axis[1].x * ((XfMax[1] + XfMin[1]) * 0.5);
            centerPos.y =
                axis[0].y * ((XfMax[0] + XfMin[0]) * 0.5) +
                axis[1].y * ((XfMax[1] + XfMin[1]) * 0.5);

            // UV に合わせて 0.0 - 1.0 に変換。
            centerPos.x /= width;
            centerPos.y /= height;

            double[] size = new double[2];
            size[0] = XfMax[0] - XfMin[0] + (margin * 2);
            size[1] = XfMax[1] - XfMin[1] + (margin * 2);

            // 頂点座標を決定。
            var axisX = new Vertex2D(axis[1].x * size[1] * 0.5 / (double)width, axis[1].y * size[1] * 0.5 / (double)height);
            var axisY = new Vertex2D(axis[0].x * size[0] * 0.5 / (double)width, axis[0].y * size[0] * 0.5 / (double)height);
            vertexList[0] = new Vertex2D(centerPos.x - axisX.x - axisY.x, centerPos.y - axisX.y - axisY.y);
            vertexList[1] = new Vertex2D(centerPos.x + axisX.x - axisY.x, centerPos.y + axisX.y - axisY.y);
            vertexList[2] = new Vertex2D(centerPos.x - axisX.x + axisY.x, centerPos.y - axisX.y + axisY.y);
            vertexList[3] = new Vertex2D(centerPos.x + axisX.x + axisY.x, centerPos.y + axisX.y + axisY.y);

            // ui2d ランタイムでは頂点の並びが左から右、上から下のポリゴンしか描画できないため
            // その仕様に適合するように頂点を並べ替える。
            if (vertexList[0].x > vertexList[1].x)
            {
                var tmp = vertexList[0];
                vertexList[0] = vertexList[1];
                vertexList[1] = tmp;

                tmp = vertexList[2];
                vertexList[2] = vertexList[3];
                vertexList[3] = tmp;

                axis[1].x *= -1.0;
                axis[1].y *= -1.0;
            }

            if (vertexList[0].y > vertexList[2].y)
            {
                var tmp = vertexList[0];
                vertexList[0] = vertexList[2];
                vertexList[2] = tmp;

                tmp = vertexList[1];
                vertexList[1] = vertexList[3];
                vertexList[3] = tmp;

                axis[0].x *= -1.0;
                axis[0].y *= -1.0;
            }

            // エッジの不透明チェック
            // OBB の場合、元の形状からはみ出した矩形が生成されることがあるが
            // このときはみ出す領域と交差する元テクスチャのエッジが不透明だと
            // 引き伸ばされてアーティファクトが発生する。
            // はみ出す領域のエッジが不透明な場合は OBB での最適化はかけないようにする。

            int[] pointConnectIndex = new int[4]
            {
                0, 1, 3, 2
            };

            if (!RepeqtEdgeOpaqueTest(vertexList, pointConnectIndex, srcAlphaImage, width, height))
            {
                optimizedWidth = width;
                optimizedHeight = height;

                return CreateMeshResult.OpaqueEdgeRepeat;
            }

            optimizedWidth = (int)Math.Round(size[0]) + 1;
            optimizedHeight = (int)Math.Round(size[1]) + 1;

            return CreateMeshResult.Succeeded;
        }
    }
}
