﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Text;
using System.Diagnostics;

namespace NW4F.LayoutBinaryConverter.TexConv
{
    public class TgaReader
    {
        // 簡易的な .tga の定義です。
        class SimpleTga
        {
            public int imageType;
            public int width;
            public int height;
            public int pixelDepth;
            public int abpp;

            public int colorMapType;
            public int cMapLength;
            public int cMapDepth;

            public byte[] imageData;

            public byte[] paletteData;
        }

        // tga ファイルを DecodedFile に読み込みます。
        public static int Read( int rawSize, byte[] rawBits, SrcImage srcImg )
        {
            Debug.Assert(rawBits != null );

            Debug.Assert( rawSize  >    0 );
            Debug.Assert( srcImg   != null );

            SimpleTga tga = new SimpleTga();

            tga.imageType    =  rawBits[0x02];
            tga.width        =  BitConverter.ToUInt16(rawBits, 0x0C);
            tga.height       =  BitConverter.ToUInt16(rawBits, 0x0E);
            tga.pixelDepth   =  rawBits[0x10];              // bits
            tga.abpp         =  rawBits[0x11] & 0x0F;       // bits 0 to 3

            tga.colorMapType =  rawBits[0x01];
            tga.cMapLength   =  BitConverter.ToUInt16(rawBits, 0x05);
            tga.cMapDepth    =  rawBits[0x07];              // bits

            ushort cMapStart  =  BitConverter.ToUInt16(rawBits, 0x03);
            byte imageDesc    =  rawBits[0x11];


            // 不正な imageType のチェック(image type == 0 なら image も palette data も存在しない)
            if( tga.imageType == 0 )
            {
                throw new TexConvException(string.Format("TgaReader.Read() : file {0} contains no image or palette data", srcImg.name));
            }

            // パレットエントリーサイズと、ビット深度 をビット数からバイト数に変換します。
            // 15bit フォーマットはチェックしますが、その他の 奇数ビット深度はサポートしません。
            {
                // palette エントリーサイズ
                switch (tga.cMapDepth)
                {
                    case 0x00:  // パレット無し
                        tga.cMapDepth = 0;
                        break;

                    case 0x10:  // 16-bits
                    case 0x18:  // 24 bits
                    case 0x20:  // 32 bits
                        tga.cMapDepth >>= 3;
                        break;

                    case 0x0F:  // 15 bit エントリ 5551 フォーマットは 16 bits と同様に扱われる
                        tga.cMapDepth = 2;
                        break;

                    default:    // サポートされないパレットサイズ
                        throw new TexConvException(string.Format("TgaReader.Read() : unsupported palette entry size in file {0}", srcImg.name));
                }

                // ビット深度
                switch (tga.pixelDepth)
                {
                    case 0x08:  //  8-bits
                    case 0x10:  // 16-bits
                    case 0x18:  // 24 bits
                    case 0x20:  // 32 bits
                        tga.pixelDepth >>= 3;
                        break;

                    case 0x0F:  // 15 bit エントリ 5551 フォーマットは 16 bits と同様に扱われる
                        tga.pixelDepth = 2;
                        break;

                    default:    // サポートされないビット深度
                        throw new TexConvException(string.Format("TgaReader.Read(): unsupported palette bit depth in file {0}", srcImg.name));
                }
            }

            // .tga ヘッダサイズ(18 bytes + IDLength bytes (.tga header の 先頭バイト))
            int rawHeaderSize = rawBits[0] + 18;

            int rawPaletteSize = 0;
            // ヘッダ以降にあるパレット（あれば）
            if( tga.colorMapType == 0 )
            {
            }
            else
            {
                rawPaletteSize = tga.cMapDepth * tga.cMapLength;

                // 使われていなくてもパレットだけ存在することがある。
                // その場合は、パレットサイズだけ考慮し、パレットデーターは保存しない。
                if( (tga.imageType == 1) || (tga.imageType == 9) )
                {
                    tga.paletteData = new byte[rawPaletteSize];
                    Buffer.BlockCopy(rawBits, rawHeaderSize, tga.paletteData, 0, tga.paletteData.Length);
                }
            }

            // 画像（あれば）
            if( (tga.width == 0) || (tga.height == 0) || (tga.pixelDepth == 0) )
            {
                // 何もしません。
            }
            else
            {
                tga.imageData = new byte[tga.width * tga.height * tga.pixelDepth];

                // 圧縮フォーマットなら (RLE), tga.imageData に展開する
                UncompressTgaImage(rawBits, rawHeaderSize + rawPaletteSize, tga );

                // パレット と cMapStart を含むなら、画像を調整する（オフセットを適用する）
                AdjustTgaForPaletteOffset( tga, cMapStart );

                // 左上原点となるように、変換を行います。
                tga.imageData = OrientUncompressedTgaData(tga.imageData, tga.width, tga.height,
                                           tga.pixelDepth, imageDesc                );

            }

            // カラー、アルファ、パレットと、レイヤーを作成します。
            CreateTgaColorLayer( srcImg, tga );
            CreateTgaAlphaLayer( srcImg, tga );
            CreateTgaPalTable(   srcImg, tga );

            return 1;
        }

        //-----------------------------------------------------------------------
        // カラーレイヤーを作成します。
        static void CreateTgaColorLayer(SrcImage srcImg, SimpleTga tga)
        {
            LayerType type = (LayerType)(0);
            switch (tga.imageType)
            {
                case 1: type = LayerType.COLOR_CI16; break; // color-index
                case 2: type = LayerType.COLOR_RGB24; break;// truecolor
                case 3: type = LayerType.COLOR_RGB24; break;// monochrome
            }

            Layer newLayer = new Layer(type, tga.width, tga.height);
            srcImg.lyColor = newLayer;

            int pixelIdx = 0;

            for (int row = 0; row < tga.height; row++)
            {
                for (int col = 0; col < tga.width; col++)
                {
                    byte g = 0, b = 0;
                    ushort r = 0, u16Tmp;

                    switch (tga.imageType)
                    {
                    case 1:
                        // colormapped
                        switch (tga.pixelDepth)
                        {
                        case 1: r = tga.imageData[pixelIdx]; break;

                        case 2: r = BitConverter.ToUInt16(tga.imageData, pixelIdx); break;

                        default:
                            throw new TexConvException(string.Format("CreateTgaColorLayer(): unknown pixel depth for file {0} colormapped data.", srcImg.name));
                        }
                        break;

                    case 2:
                        // truecolor
                        switch (tga.pixelDepth)
                        {
                        case 2:
                            u16Tmp = BitConverter.ToUInt16(tga.imageData, pixelIdx);

                            r = (ushort)(((u16Tmp & 0x7C00) >> 7) | ((u16Tmp & 0x7000) >> 12));
                            g = (byte)(((u16Tmp & 0x03E0) >> 2) | ((u16Tmp & 0x0380) >> 7));
                            b = (byte)(((u16Tmp & 0x001F) << 3) | ((u16Tmp & 0x001C) >> 2));
                            break;

                        case 3:
                        case 4:
                            // alpha 値を無視する(pixelIdx + 3)
                            r = tga.imageData[pixelIdx + 2];
                            g = tga.imageData[pixelIdx + 1];
                            b = tga.imageData[pixelIdx + 0];
                            break;

                        default:
                            throw new TexConvException(string.Format("CreateTgaColorLayer(): unknown pixel depth for file {0} data.", srcImg.name));
                        }

                        break;

                    case 3:
                        // monochrome
                        // pixel depth > 1 の場合は RGB の平均をとる

                        switch (tga.pixelDepth)
                        {
                        case 1:

                            r = tga.imageData[pixelIdx];
                            g = (byte)r;
                            b = (byte)r;
                            break;

                        case 2:

                            u16Tmp = BitConverter.ToUInt16(tga.imageData, pixelIdx);

                            r = (ushort)(((u16Tmp & 0x7C00) >> 7) | ((u16Tmp & 0x7000) >> 12));
                            r += (ushort)(((u16Tmp & 0x03E0) >> 2) | ((u16Tmp & 0x0380) >> 7));
                            r += (ushort)(((u16Tmp & 0x001F) << 3) | ((u16Tmp & 0x001C) >> 2));
                            r /= 3;

                            g = (byte)r;
                            b = (byte)r;
                            break;

                        case 3:
                        case 4:
                            // alpha 値を無視する(pixelIdx + 3)
                            r = tga.imageData[pixelIdx + 2];
                            r += tga.imageData[pixelIdx + 1];
                            r += tga.imageData[pixelIdx + 0];
                            r /= 3;

                            g = (byte)r;
                            b = (byte)r;
                            break;

                        default:
                            throw new TexConvException(string.Format("CreateTgaColorLayer(): unknown pixel depth for file {0} intensity data.", srcImg.name));
                        }

                        break;

                    }

                    // 下記はホットスポットとなっている。
                    // ほとんどの場合、LayerType.COLOR_RGB24 と考えられるので
                    // SetValue() の中身をインラインで記述すれば効率は改善するが、
                    // 対応TGA入力データに仕様変更が生じるため手をつけていない。
                    newLayer.SetValue(col, row, r, g, b);
                    pixelIdx += tga.pixelDepth;

                }

            }
        }


        //------------------------------------------------------------------------------------
        // アルファレイヤーを作成します。
        static void CreateTgaAlphaLayer(SrcImage srcImg, SimpleTga tga)
        {
            // color-indexed なら NG
            if (tga.imageType == 1)
            {
                return;
            }

            // true color なら
            if (tga.imageType == 2)
            {
                // rgba フォーマットのみ許容
                if ((tga.pixelDepth != 2) && (tga.pixelDepth != 4))
                {
                    return;
                }
            }

            Layer newLayer = new Layer(LayerType.ALPHA_A8, tga.width, tga.height);
            srcImg.lyAlpha = newLayer;

            int pixelIdx = 0;
            for (int row = 0; row < tga.height; row++)
            {
                for (int col = 0; col < tga.width; col++)
                {
                    ushort a = 0, u16Tmp;

                    switch (tga.imageType)
                    {

                    case 2:
                        // truecolor

                        switch (tga.pixelDepth)
                        {

                        case 2:
                            // 1-bit アルファ
                            u16Tmp = BitConverter.ToUInt16(tga.imageData, pixelIdx);

                            a = 0;

                            // alpha bit が有効か
                            if ((u16Tmp & 0x8000) == 0x8000)
                            {
                                a = 0x00FF;
                            }
                            break;

                        case 4:
                            // 8-bit alpha
                            a = tga.imageData[pixelIdx + 3];
                            break;
                        }

                        break;

                    case 3:
                        // monochrome

                        switch (tga.pixelDepth)
                        {
                        case 1:
                            // 8-bit モノクロ
                            a = tga.imageData[pixelIdx];
                            break;

                        case 2:
                            u16Tmp = BitConverter.ToUInt16(tga.imageData, pixelIdx);
                            // 16-bit モノクロ

                            a = 0;

                            // アルファビットが有効か
                            if ((u16Tmp & 0x8000) == 0x8000)
                            {
                                a = 0x00FF;
                            }
                            break;

                        case 3:
                            // 24-bit モノクロ
                            // RGB の平均を採用

                            a = tga.imageData[pixelIdx + 2];    // red
                            a += tga.imageData[pixelIdx + 1];   // green
                            a += tga.imageData[pixelIdx + 0];   // blue
                            a /= 3;
                            break;

                        case 4:
                            // 32-bit モノクロ
                            // アルファチャンネルをそのまま使う。

                            a = tga.imageData[pixelIdx + 3];
                            break;

                        default:
                            throw new TexConvException(string.Format("CreateImageAlphaLayerFromTga(): intensity file {0} has unknown pixel depth.", srcImg.name));
                        }

                        break;

                    }

                    newLayer.SetValue(col, row, a, 0, 0);
                    pixelIdx += tga.pixelDepth;
                }
            }
        }

        //------------------------------------------------------------------------------------
        // カラーパレットレイヤーを作成します。
        static void CreateTgaPalTable(SrcImage dfPtr, SimpleTga tga)
        {
            if (tga.paletteData == null)
            {
                return;
            }

            dfPtr.palPtr = new PaletteTable(tga.cMapLength);

            int rawIdx = 0;

            for (int i = 0; i < tga.cMapLength; i++)
            {
                byte r = 0, g = 0, b = 0, a = 0;
                ushort u16Tmp;

                switch (tga.cMapDepth)
                {

                case 2:
                    // 15 or 16-bit
                    u16Tmp = BitConverter.ToUInt16(tga.paletteData, rawIdx);

                    r = (byte)(((u16Tmp & 0x7C00) >> 7) | ((u16Tmp & 0x7000) >> 12));
                    g = (byte)(((u16Tmp & 0x03E0) >> 2) | ((u16Tmp & 0x0380) >> 7));
                    b = (byte)(((u16Tmp & 0x001F) << 3) | ((u16Tmp & 0x001C) >> 2));
                    a = 0;
                    if ((u16Tmp & 0x8000) == 0x0000)
                    {
                        a = 0xFF;
                    }

                    rawIdx += 2;
                    break;

                case 3:
                    // rgb24

                    r = tga.paletteData[rawIdx + 2];
                    g = tga.paletteData[rawIdx + 1];
                    b = tga.paletteData[rawIdx + 0];
                    a = 0xFF;

                    rawIdx += 3;
                    break;

                case 4:
                    // rgba

                    r = tga.paletteData[rawIdx + 2];
                    g = tga.paletteData[rawIdx + 1];
                    b = tga.paletteData[rawIdx + 0];
                    a = tga.paletteData[rawIdx + 3];

                    rawIdx += 4;
                    break;
                }

                dfPtr.palPtr.Set(i, r, g, b, a);
            }
        }

        //------------------------------------------------------------------------------------
        // RLE 圧縮イメージを展開します。
        static void UnencodeTgaImage(byte[] srcBuf, int imageIdx, byte[] dstBuf, int width, int height, int pixelDepth)
        {
            int srcOffset = 0;
            int dstOffset = 0;
            int totalCount = 0;
            int maxCount = width * height;

            while (totalCount < maxCount)
            {
                // カウントバイトをよむ
                byte countByte = srcBuf[imageIdx + srcOffset];

                int runLength = (countByte & 0x7F) + 1;   // ランレングス = 下位 7 bits + 1 (1 to 128)
                srcOffset += 1;

                // 最上位ビットをみて、 RLE か否かを判定する
                if ((countByte & 0x80) == 0x80)
                {
                    // RLE
                    for (int i = 0; i < runLength; i++)
                    {
                        for (int j = 0; j < pixelDepth; j++)
                        {
                            dstBuf[dstOffset] = srcBuf[imageIdx + srcOffset + j];
                            dstOffset++;
                        }
                    }
                    totalCount += runLength;
                    srcOffset += pixelDepth;
                }
                else
                {
                    for (int i = 0; i < runLength; i++)
                    {
                        for (int j = 0; j < pixelDepth; j++)
                        {
                            dstBuf[dstOffset] = srcBuf[imageIdx + srcOffset];
                            dstOffset++;
                            srcOffset++;
                        }
                    }
                    totalCount += runLength;
                }
            }
        }

        // RLE 圧縮イメージを展開します。
        static void UncompressTgaImage(byte[] rawBits, int imageIdx, SimpleTga tga)
        {
            int size = tga.width * tga.height * tga.pixelDepth;

            switch (tga.imageType)
            {
                case 0:
                case 1:
                case 2:
                case 3:
                    // 非圧縮タイプはそのまま
                    Buffer.BlockCopy(rawBits, imageIdx, tga.imageData, 0, size);
                    break;
                case 9:
                case 10:
                case 11:
                    // RLE 圧縮
                    // 展開された画像のサイズを計算します。
                    UnencodeTgaImage(rawBits, imageIdx, tga.imageData, tga.width, tga.height, tga.pixelDepth);
                    break;
            }

            // 画像タイプ非圧縮形式に調整します。
            switch (tga.imageType)
            {
                case 9: tga.imageType = 1; break;
                case 10: tga.imageType = 2; break;
                case 11: tga.imageType = 3; break;
            }
        }


        //------------------------------------------------------------------------------------
        // 圧縮されていない、TGA を左上原点に整列します。
        static byte[] OrientUncompressedTgaData(byte[] srcBuff, int width, int height, int pixelDepth, int imageDesc)
        {
            // bits 4 , 5 が原点の情報
            int origin = ((imageDesc & 0x30) >> 4);

            switch (origin)
            {
                case 0x00:     // 左下
                    {
                        int srcIdx = 0;
                        int dstIdx = width * pixelDepth * (height - 1);
                        int dstXstep = pixelDepth;
                        int dstYstep = -1 * (width * pixelDepth);
                        int srcYStep = width * pixelDepth;

                        // プロファイリングの結果、下記ブロックがホットスポットとなっています。
                        // できるだけ効率のよい画像コピー処理をカスタムで記述しています。
                        // (なお、ほとんどのデータが case 0x00 のようです。)
                        byte[] dstBuff = new byte[width * height * pixelDepth];
                        for (int i = 0; i < height; i++)
                        {
                            Buffer.BlockCopy(srcBuff, srcIdx, dstBuff, dstIdx, width * pixelDepth);

                            srcIdx += srcYStep;
                            dstIdx += dstYstep;
                        }

                        return dstBuff;
                    }
                case 0x01:     // 右下
                    {
                        int srcIdx = 0;
                        int dstIdx = (width * pixelDepth * height) - pixelDepth;
                        int dstXstep = -pixelDepth;
                        int dstYstep = 0;

                        byte[] dstBuff = new byte[width * height * pixelDepth];
                        ReorderImage_(height, width, pixelDepth, dstBuff, dstIdx, srcBuff, srcIdx, dstXstep, dstYstep);

                        return dstBuff;
                    }
                case 0x02:     // 左上
                    // 原点が一致しているので変換の必要がありません。
                    return srcBuff;

                case 0x03:     // 右上
                    {
                        int srcIdx = 0;
                        int dstIdx = (width * pixelDepth) - pixelDepth;
                        int dstXstep = -pixelDepth;
                        int dstYstep = (width * pixelDepth) * 2;

                        byte[] dstBuff = new byte[width * height * pixelDepth];
                        ReorderImage_(height, width, pixelDepth, dstBuff, dstIdx, srcBuff, srcIdx, dstXstep, dstYstep);

                        return dstBuff;
                    }
                default:
                    Debug.Assert(false);
                    return srcBuff;
            }

        }

        /// <summary>
        /// 画像を左右上限反転を行いながらコピーします。
        /// </summary>
        private static void ReorderImage_(int height, int width, int pixelDepth, byte[] dstBuff, int dstIdx, byte[] srcBuff, int srcIdx, int dstXstep, int dstYstep)
        {
            for (int i = 0; i < height; i++)
            {
                for (int j = 0; j < width; j++)
                {
                    for (int k = 0; k < pixelDepth; k++)
                    {
                        dstBuff[dstIdx + k] = srcBuff[srcIdx++];
                    }

                    dstIdx += dstXstep;
                }

                dstIdx += dstYstep;
            }
        }

        //------------------------------------------------------------------------------------
        //   8 or 16 BitConverter カラーマップ形式で、cMapStart がゼロでないときだけ利用します。
        static void AdjustTgaForPaletteOffset(SimpleTga tga, ushort cMapStart)
        {
            // パレットのオフセット cMapStart がある場合, 実際の 画素値 は realValue - cMapStart となります。
            // 画素値 + cMapStart となるように値を調整します。

            if (cMapStart == 0)
            {
                return;
            }

            // color-indexed 画像か
            if ((tga.imageType != 1) &&
                (tga.imageType != 9))
            {
                return;
            }

            // 8 or 16 bit インデックスだけを許容する
            if ((tga.pixelDepth != 1) &&
                (tga.pixelDepth != 2))
            {
                throw new TexConvException("AdjustTgaForPaletteOffset(): unsupported pixel depth\n");
            }

            int vIdx = 0;
            if (tga.pixelDepth == 1)
            {
                for (int i = 0; i < tga.height; i++)
                {
                    for (int j = 0; j < tga.width; j++)
                    {
                        tga.imageData[vIdx] = (byte)(tga.imageData[vIdx] + cMapStart);
                        ++vIdx;
                    }
                }
            }
            else if (tga.pixelDepth == 2)
            {
                for (int i = 0; i < tga.height; i++)
                {
                    for (int j = 0; j < tga.width; j++)
                    {
                        ushort tmpVal = (ushort)(BitConverter.ToUInt16(tga.imageData, vIdx) + cMapStart);
                        BitConverter.GetBytes(tmpVal).CopyTo(tga.imageData, vIdx);
                        vIdx += 2;
                    }
                }
            }
        }
    }
}


