﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Text;

namespace NW4F.LayoutBinaryConverter
{
    using Schema.Flyt;

    class MaterialInfoHelper
    {
        public static MaterialInfo Create(Pane pane, Material matSmp, Material_CTR matCTR, bool bOutTexture, bool bDetailSetting)
        {
            MaterialInfo matInfo = new MaterialInfo(pane, matSmp, matCTR, bOutTexture, bDetailSetting);
            if (bDetailSetting)
            {
                matInfo.name = matCTR.name;

                // 詳細コンバイナ設定、及びコンバイナユーザーシェーダ設定では、通常マテリアルと同じ白黒補間カラーに加えて
                // コンスタントカラーが使われます。
                // 従来の仕様との混乱をさけるため<tevConstColor>を新設しています。
                if (matCTR != null && (matCTR.useLowLevelCombinerSettings || matCTR.useCombinerUserShaderSettings))
                {
                    // 詳細コンバイナ時はステージ数のチェックを行う
                    if (matCTR.useLowLevelCombinerSettings && matCTR.tevStageNum <= 0)
                    {
                        throw new LayoutDataException(string.Format(Properties.Resources.ErrorDetailedCombinerStageNum, pane.name));
                    }

                    if (matCTR.tevConstColor != null)
                    {
                        matInfo.konstColors = new Color4f[matCTR.tevConstColor.Length + 1];
                        for (int i = 0; i < matCTR.tevConstColor.Length; i++)
                        {
                            matInfo.konstColors[i + 1] = new Color4f(matCTR.tevConstColor[i]);
                        }
                    }
                    else
                    {
                        matInfo.konstColors = new Color4f[1];
                    }

                    // 白黒補間カラー
                    if (matSmp.blackColorFloat != null)
                    {
                        matInfo.color = new Color4f(matSmp.blackColorFloat);
                    }
                    else
                    {
                        matInfo.color = new Color4f(matSmp.blackColor);
                    }
                    if (matSmp.whiteColorFloat != null)
                    {
                        matInfo.konstColors[0] = new Color4f(matSmp.whiteColorFloat);
                    }
                    else
                    {
                        matInfo.konstColors[0] = new Color4f(matSmp.whiteColor);
                    }
                }
                else
                {
                    if (matCTR.tevConstRegFloat != null)
                    {
                        matInfo.color = new Color4f(matCTR.tevColRegFloat);
                    }
                    else
                    {
                        matInfo.color = new Color4f(matCTR.tevColReg);
                    }
                    if (matCTR.tevConstRegFloat != null)
                    {
                        matInfo.konstColors = new Color4f[matCTR.tevConstRegFloat.Length];
                        for (int i = 0; i < matCTR.tevConstRegFloat.Length; i++)
                        {
                            matInfo.konstColors[i] = new Color4f(matCTR.tevConstRegFloat[i]);
                        }
                    }
                    else
                    {
                        if (matCTR.tevConstReg != null)
                        {
                            matInfo.konstColors = new Color4f[matCTR.tevConstReg.Length];
                            for (int i = 0; i < matCTR.tevConstReg.Length; i++)
                            {
                                matInfo.konstColors[i] = new Color4f(matCTR.tevConstReg[i]);
                            }
                        }
                    }
                }

                matInfo.texMap = matCTR.texMap;
                matInfo.texMatrix = matCTR.texMatrix;
                matInfo.texCoordGen = matCTR.texCoordGen;

                // デフォルト設定の場合はアルファテスト設定を出力しない
                if (matCTR.useDefaultAlphaTestSettings)
                {
                    matInfo.alphaCompare = null;
                }
                else
                {
                    matInfo.alphaCompare = matCTR.alphaCompare;
                }

                // デフォルト設定の場合はブレンド設定を出力しない
                if (matCTR.useDefaultBlendSettings)
                {
                    matInfo.blendMode = null;
                    matInfo.blendModeAlpha = null;
                }
                else
                {
                    matInfo.blendMode = matCTR.blendMode;
                    matInfo.blendModeAlpha = matCTR.blendModeAlpha;
                }

                if (bOutTexture)
                {
                    if (matCTR.tevStage != null && matCTR.tevStageNum > 0)
                    {
                        matInfo.tevStage = matCTR.tevStage;
                        DataUtil.ResizeArray(
                            ref matInfo.tevStage,
                            Math.Min((int)matCTR.tevStageNum, BinaryLytWriter.CapOfTevStage));
                    }
                    else
                    {
                        DataUtil.ResizeArray(ref matInfo.tevStage, 0);
                    }
                }
                else
                {
                    DataUtil.ResizeArray(ref matInfo.tevStage, 0);
                }
            }
            else    // 詳細設定モードOFF
            {
                matInfo.name = matSmp.name;

                // テクスチャ混合比率
                // KColor のお尻から混合比率の値をセットする
                TexBlendRatio[] texBlendRatio = matSmp.texBlendRatio;
                DataUtil.ResizeArray(ref texBlendRatio, BinaryLytWriter.CapOfTexMap);

                matInfo.konstColors = new Color4f[BinaryLytWriter.CapOfTevKColor];
                // 白黒補間カラー
                if (matSmp.blackColorFloat != null)
                {
                    matInfo.color = new Color4f(matSmp.blackColorFloat);
                }
                else
                {
                    matInfo.color = new Color4f(matSmp.blackColor);
                }
                if (matSmp.whiteColorFloat != null)
                {
                    matInfo.konstColors[0] = new Color4f(matSmp.whiteColorFloat);
                }
                else
                {
                    matInfo.konstColors[0] = new Color4f(matSmp.whiteColor);
                }
                for (int i = 1; i < matInfo.konstColors.Length; ++i)
                {
                    matInfo.konstColors[i] = new Color4f(1.0f, 1.0f, 1.0f, 1.0f);
                }

                for (int i = 0; i < texBlendRatio.Length; ++i)
                {
                    int dIdx = matInfo.konstColors.Length - 1 - i / 4;

                    switch (i % 4)
                    {
                    case 3: matInfo.konstColors[dIdx].r = texBlendRatio[i].color; break;
                    case 2: matInfo.konstColors[dIdx].g = texBlendRatio[i].color; break;
                    case 1: matInfo.konstColors[dIdx].b = texBlendRatio[i].color; break;
                    case 0: matInfo.konstColors[dIdx].a = texBlendRatio[i].color; break;
                    }
                }

                matInfo.texMap = matSmp.texMap;
                matInfo.texMatrix = matSmp.texMatrix;
                matInfo.texCoordGen = matSmp.texCoordGen;

                DataUtil.ResizeArray(ref matInfo.tevStage, 0);
            }

            if (matSmp.hsbAdjustment != null && (matSmp.hsbAdjustment.hOffset != 0.0 || matSmp.hsbAdjustment.sScale != 1.0 || matSmp.hsbAdjustment.bScale != 1.0))
            {
                matInfo.hsbAdjustment = matSmp.hsbAdjustment;
            }

            if (bOutTexture)
            {
                DataUtil.ResizeArray(ref matInfo.texMap, BinaryLytWriter.CapOfTexMap);
                DataUtil.ResizeArray(ref matInfo.texMatrix, BinaryLytWriter.CapOfTexMtx);
                DataUtil.ResizeArray(ref matInfo.texCoordGen, BinaryLytWriter.CapOfTexCoordGen);
            }
            else
            {
                DataUtil.ResizeArray(ref matInfo.texMap, 0);
                DataUtil.ResizeArray(ref matInfo.texMatrix, 0);
                DataUtil.ResizeArray(ref matInfo.texCoordGen, 0);
            }

            matInfo.UseThresholdingAlphaInterpolation = matSmp.isThresholdingAlphaInterpolationEnabled;

            VerifyName(matInfo);
            return matInfo;
        }

        /// <summary>
        /// マテリアル名のチェックを行う。
        /// 問題がある場合は、LayoutDataException例外がスローされる。
        /// </summary>
        /// <param name="material">チェックするマテリアル。</param>
        static void VerifyName(MaterialInfo material)
        {
            if (string.IsNullOrEmpty(material.name) || material.name.Length > BinaryLytWriter.MaterialNameStrMax)
            {
                throw new LayoutDataException(string.Format(Properties.Resources.ErrorInvalidMaterialName, material.name));
            }
        }

    }
}
