﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Text;
using System.IO;
using System.Runtime.InteropServices;
using System.Diagnostics;

namespace NW4F.LayoutBinaryConverter
{
    using Lan = Schema.Flan;
    using Lyt = Schema.Flyt;

    class LanWriter
    {
        /// <summary>
        /// アニメーション区間タグを使用するオプションが有効のときは真
        /// </summary>
        readonly bool _bUseAnimSecTagOpt;

        /// <summary>
        /// アニメーション区間タグを使用すると共にアニメーション全体も出力するオプションが有効のときは真
        /// </summary>
        readonly bool _bTagSplitAndWholeOpt;

        /// <summary>
        /// タグブロックを出力するときは真。
        /// </summary>
        readonly bool _bOutTagBlock;

        readonly bool _bOmitSameKey;
        readonly bool _bOmitSameKeyAll;
        readonly bool _bOmitNoKeyAll;
        readonly bool _bBakeInfinityAreaKey;
        readonly bool _bConvertAllAnimContent;
        readonly bool _bAllowedNoGroupAnimTag;
        readonly bool _bDegammaTexture;
        readonly bool _bDegammaParameter;
        readonly bool _bDegammaParameterOldBug;
        readonly bool _bBannerFormat;
        readonly string _ftxbCacheDirectory;
        readonly bool _bCompatible_0_12_x;

        public LanWriter(bool bTagSplitOpt, bool bTagSplitAndWholeOpt, bool bNoOutTagBlock, bool bOmitSameKey, bool bOmitSameKeyAll, bool bOmitNoKeyAll, bool bBakeInfinityAreaKey, bool bConvertAllAnimContent, bool bAllowedNoGroupAnimTag, bool bDegammaTexture, bool bDegammaParameter, bool bDegammaParameterOldBug, bool bBannerFormat, string ftxbCacheDirectory, bool bCompatible_0_12_x)
        {
            _bUseAnimSecTagOpt = bTagSplitOpt || bTagSplitAndWholeOpt;
            _bTagSplitAndWholeOpt = bTagSplitAndWholeOpt;

            // タグ分割するときに有効なオプション
            _bOutTagBlock = false;
            _bOmitSameKey = false;
            _bOmitSameKeyAll = false;
            _bOmitNoKeyAll = false;
            if (_bUseAnimSecTagOpt)
            {
                if (! bBannerFormat)  // バナー用フォーマットモードでない
                {
                    _bOutTagBlock = ! bNoOutTagBlock;
                }

                _bOmitSameKey = bOmitSameKey;
                _bOmitSameKeyAll = bOmitSameKeyAll;
                _bOmitNoKeyAll = bOmitNoKeyAll;

                if (1 < (bOmitSameKey? 1 : 0) + (bOmitSameKeyAll? 1 : 0) + (bOmitNoKeyAll? 1: 0))
                {
                    throw new ArgumentException("bOmitSameKey, bOmitSameKeyAll, and bOmitNoKeyAll are exclusive.");
                }
            }

            _bBakeInfinityAreaKey = bBakeInfinityAreaKey;
            _bConvertAllAnimContent = bConvertAllAnimContent;
            _bAllowedNoGroupAnimTag = bAllowedNoGroupAnimTag;
            _bDegammaTexture = bDegammaTexture;
            _bDegammaParameter = bDegammaParameter;
            _bDegammaParameterOldBug = bDegammaParameterOldBug;
            _bBannerFormat = bBannerFormat;
            _ftxbCacheDirectory = ftxbCacheDirectory;
            _bCompatible_0_12_x = bCompatible_0_12_x;
        }

        public Dictionary<string, Lan::RefRes> Write(string outDirName, string inFileName, LanInfo lanInfo, int outputAnimTagIndexOffset)
        {
            Dictionary<string, Lan::RefRes> cvtRefTexDic = new Dictionary<string, Lan::RefRes>(StringComparer.InvariantCultureIgnoreCase);

            if (lanInfo.LanArray == null)
            {
                return cvtRefTexDic;
            }

            DirectoryInfo animDir = new DirectoryInfo(Path.Combine(outDirName, DataUtil.GetResourceTypeString(BinaryLanWriter.TypeOfAnimationResource)));
            if (!animDir.Exists)
            {
                animDir.Create();
            }

            // 現状のアニメーション共有設定は全てのアニメーション区間において共通である。
            // アニメーション共有設定が無い場合は、DocumentBody.animShare は null であり、
            // アニメーション共有設定がある場合は、1つだけ存在する。
            Lan::AnimShare animShare = lanInfo.AnimShare != null ? lanInfo.AnimShare[0] : null;

            // タグによるバイナリ分割が有効で、タグ情報があるとき
            if (_bUseAnimSecTagOpt && lanInfo.AnimTagArray != null)
            {
                Lan::AnimTag[] animTags = lanInfo.AnimTagArray;

                int outputAnimTagIdx = outputAnimTagIndexOffset;
                foreach (Lan::AnimTag animTag in animTags)
                {
                    this.WriteSection(
                        inFileName,
                        lanInfo,
                        cvtRefTexDic,
                        animDir,
                        animShare,
                        outputAnimTagIdx,
                        animTag,
                        _bUseAnimSecTagOpt,
                        _bOmitSameKey,
                        _bOmitSameKeyAll,
                        _bOmitNoKeyAll,
                        _bOutTagBlock);

                    ++outputAnimTagIdx;
                }

                if (_bTagSplitAndWholeOpt)
                {
                    // アニメーション全体を含んだバイナリの出力。
                    Lan::AnimTag animTag = lanInfo.CreateAnimTagFromConvertFrameRange();
                    this.WriteSection(
                        inFileName,
                        lanInfo,
                        cvtRefTexDic,
                        animDir,
                        animShare,
                        0 /* outputAnimTagIdx */,
                        animTag,
                        false /* bUseAnimSecTag */,
                        false /* bOmitSameKey */,
                        false /* bOmitSameKeyAll */,
                        false /* bOmitNoKeyAll */,
                        false /* bOutTagBlock */);
                }
            }
            else
            {
                // タグによる分割をしない、またはタグ情報がないときは、rlanの値を使用する
                Lan::AnimTag animTag = lanInfo.CreateAnimTagFromConvertFrameRange();
                this.WriteSection(
                    inFileName,
                    lanInfo,
                    cvtRefTexDic,
                    animDir,
                    animShare,
                    0 /* outputAnimTagIdx */,
                    animTag,
                    false /* bUseAnimSecTag */,
                    _bOmitSameKey,
                    _bOmitSameKeyAll,
                    _bOmitNoKeyAll,
                    _bOutTagBlock);
            }

            return cvtRefTexDic;
        }

        private void WriteSection(
            string inFileName,
            LanInfo lanInfo,
            Dictionary<string, Lan::RefRes> cvtRefTexDic,
            DirectoryInfo animDir,
            Lan::AnimShare animShare,
            int outputAnimTagIdx,
            Lan::AnimTag animTag,
            bool bUseAnimSecTag,
            bool bOmitSameKey,
            bool bOmitSameKeyAll,
            bool bOmitNoKeyAll,
            bool bOutTagBlock)
        {
            if (!animTag.outputEnabled)   // 出力しないタグは飛ばす
            {
                return;
            }

            AnimSection animSec = new AnimSection(animTag, outputAnimTagIdx);

            MergeAnimData mergeAnimData = new MergeAnimData(
                lanInfo,
                animSec,
                lanInfo.GetLytInfo().GroupSet,
                bUseAnimSecTag,
                bOmitSameKey,
                bOmitSameKeyAll,
                bOmitNoKeyAll,
                _bBakeInfinityAreaKey,
                _bConvertAllAnimContent,
                _bAllowedNoGroupAnimTag);

            // 出力するAnimContentが1つも無い場合もそのファイルは出力しない。
            if (mergeAnimData.AnimContentAry == null)
            {
                return;
            }

            string outFileName = FileUtil.MakeOutFileNameStr(animDir.FullName, null, inFileName, animTag.GetFileName());

            using (FileStream fs = new FileStream(outFileName, FileMode.Create, FileAccess.Write, FileShare.None))
            {
                SortedList<string, Lan::RefRes> refTexList = mergeAnimData.MakeRefTextureList(animTag, lanInfo.GetLytInfo().MaterialList);
                foreach (KeyValuePair<string, Lan::RefRes> refTexPair in refTexList)
                {
                    if (!cvtRefTexDic.ContainsKey(refTexPair.Key))
                    {
                        cvtRefTexDic.Add(refTexPair.Key, refTexPair.Value);
                    }

                    // レイアウトが持っていないテクスチャを参照している場合はエラーで止める。
                    if (!lanInfo.GetLytInfo().TextureFileList.ContainsKey(refTexPair.Key))
                    {
                        throw new LayoutDataException(string.Format(Properties.Resources.ErrorTexFileNotExistForAnimTag,
                            animTag.GetFileName(), refTexPair.Key), lanInfo.GetLytInfo().FilePath);
                    }

                    // フォーマット情報を入れておく
                    Lyt::TextureFile texFile = lanInfo.GetLytInfo().TextureFileList[refTexPair.Key];
                    refTexPair.Value.format = texFile.format;
                    // TextureFileの方にもフィードバックする
                    if (refTexPair.Value.isIndirect) {
                        texFile.IsIndirect = true;
                    } else {
                        texFile.IsNoIndirect = true;
                    }
                }

                AnimTagWriter animTagWriter = new AnimTagWriter(new BinaryLanWriter(fs), animSec, refTexList, _bDegammaParameter, _bDegammaParameterOldBug, _bBannerFormat, _bCompatible_0_12_x, _bDegammaParameter);
                animTagWriter.Write(
                    animShare,
                    mergeAnimData,
                    bOutTagBlock,
                    outputAnimTagIdx);
            }
        }

        public bool GenTexPatternAnimTexFiles(string outDirName, ICollection<string> refTexDic, LytInfo rlytInfo)
        {
            List<Lyt::TextureFile> texFileList = new List<Lyt::TextureFile>();
            // 参照しているテクスチャをコピー
            foreach (string fileName in refTexDic)
            {
                Lyt::TextureFile texFile = rlytInfo.TextureFileList[fileName];
                if (texFile != null)
                {
                    texFileList.Add(texFile);
                }
            }

            var srcFiles = TgaUtil.LoadTextureSourceFiles(
                rlytInfo.FilePath,
                texFileList,
                _bDegammaTexture);

            // textureファイルの変換
            // レイアウトの変換よりも先に行います。(テクスチャ書き出し時に取得した情報を利用するため)
            return TgaUtil.GenTextureFiles(
                srcFiles,
                outDirName,
                _ftxbCacheDirectory,
                _bDegammaTexture);
        }
    }

    class AnimTagWriter
    {
        readonly BinaryLanWriter _binWriter;
        readonly AnimSection _animSec;
        readonly SortedList<string, Lan::RefRes> _fileNameList;
        readonly bool _bDegamma;
        readonly bool _bDegammaOldBug;
        readonly bool _bBannerFormat;
        readonly bool _bCompatible_0_12_x;
        readonly bool _bDegammaParameter;

        Stream _stream { get { return _binWriter.Stream; } }

        public AnimTagWriter(BinaryLanWriter binWriter, AnimSection animSec, SortedList<string, Lan::RefRes> fileNameList, bool bDegamma, bool bDegammaOldBug, bool bBannerFormat, bool bCompatible_0_12_x, bool bDegammaParameter)
        {
            _binWriter = binWriter;
            _animSec = animSec;
            _fileNameList = fileNameList;
            _bDegamma = bDegamma;
            _bDegammaOldBug = bDegammaOldBug;
            _bBannerFormat = bBannerFormat;
            _bCompatible_0_12_x = bCompatible_0_12_x;
            _bDegammaParameter = bDegammaParameter;
        }

        public void Write(Lan::AnimShare animShare, MergeAnimData mergeAnimData, bool bOutTagBlock, int tagOrder)
        {
            if (_bBannerFormat)
            {
                // 安全のため、バナーモードオプションが指定されて、タグにグループ名が含まれているときは、
                // 変換を中止する。
                //
                // (バナーモードではなく、
                //  明示的にタグを出力しないオプションによってタグブロックを出力しないケースのときは、
                //  エラーとしない。)
                if (_animSec.Tag.group.Length > 0)
                {
                    throw new BannerConvertException(string.Format(Properties.Resources.ErrorIncludeGroupName, _animSec.Tag.name));
                }

                // 安全のため、バナーモードオプションが指定されて、アニメーション共有が含まれているときは、
                // 変換を中止する。
                if (animShare != null)
                {
                    throw new BannerConvertException(Properties.Resources.ErrorCantUserAnimationShare);
                }
            }

            _binWriter.WriteBinaryFileHeader(0, 0, false, _bCompatible_0_12_x);      // バイナリファイルヘッダ

            ushort dataBlockNum = 0;

            if (bOutTagBlock)
            {
                WriteAnimTagBlock(tagOrder);
                ++dataBlockNum;
            }

            if (animShare != null)
            {
                WriteAnimShareBlock(animShare);
                ++dataBlockNum;
            }

            WriteAnimBlock(mergeAnimData);
            ++dataBlockNum;

            // この時点でファイル書き込みが全て終了したので、バイナリファイルヘッダを更新
            long fileSize = _stream.Position;   // ファイルポインタの現在位置がファイルサイズ
            _stream.Position = 0;
            _binWriter.WriteBinaryFileHeader((uint)fileSize, dataBlockNum, _bBannerFormat, _bCompatible_0_12_x);
        }

        /// <summary>
        /// ユーザー情報を out パラメーターに読み込みます。
        /// </summary>
        private static void ReadUserDataElement_(object userDataObj, out string nameStr, out string valueStr, out UserDataSubWriter.ExUserDataType type)
        {
            if (userDataObj is Lan::UserDataString)
            {
                var src = userDataObj as Lan::UserDataString;
                nameStr = src.name;
                valueStr = src.Value;
                type = UserDataSubWriter.ExUserDataType.String;
            }
            else if (userDataObj is Lan::UserDataIntList)
            {
                var src = userDataObj as Lan::UserDataIntList;
                nameStr = src.name;
                valueStr = src.Value;
                type = UserDataSubWriter.ExUserDataType.IntList;
            }
            else if (userDataObj is Lan::UserDataFloatList)
            {
                var src = userDataObj as Lan::UserDataFloatList;
                nameStr = src.name;
                valueStr = src.Value;
                type = UserDataSubWriter.ExUserDataType.FloatList;
            }
            else if (null != (userDataObj as Lan::UserDataNone))
            {
                nameStr = null;
                valueStr = null;
                type = UserDataSubWriter.ExUserDataType.None;
            }
            else
            {
                throw new LayoutDataException(string.Format(Properties.Resources.ErrorUnknownUserDataType, userDataObj.GetType()));
            }
        }

        // ResAnimationTagBlock
        void WriteAnimTagBlock(int tagOrder)
        {
            long blockHeadFilePos = _stream.Position;

            _binWriter.WriteBlock(_animSec, 0, 0, 0, 0);

            long nameOffset = _stream.Position - blockHeadFilePos;

            // タグ名文字列の書き出し
            byte[] byteTagName = Encoding.ASCII.GetBytes(_animSec.Tag.name + "\0");
            _stream.Write(byteTagName, 0, byteTagName.Length);
            FileUtil.RoundUpFileSize(_stream, 4);

            long groupsOffset = _stream.Position - blockHeadFilePos;
            if (_animSec.Tag.group.Length > 0)
            {
                foreach (Lan::GroupRef groupRef in _animSec.Tag.group)
                {
                    _binWriter.WriteStruct(groupRef);
                }

                FileUtil.RoundUpFileSize(_stream, 4);
            }

            // ResExtUserDataList の書き込み
            long userDataListOffset = _stream.Position - blockHeadFilePos;
            if (_animSec.Tag.userData != null && _animSec.Tag.userData.Length > 0)
            {
                UserDataSubWriter.WriteUserDataBlock(_binWriter, _animSec.Tag.name, _animSec.Tag.userData, 0, ReadUserDataElement_, _bDegammaParameter);
            }
            else
            {
                userDataListOffset = 0;
            }

            // データブロックの再書き込み
            FileUtil.WriteDataBlock(
                _stream,
                delegate(uint blockSize)
                {
                    _binWriter.WriteBlock(_animSec, blockSize, (uint)nameOffset, (uint)groupsOffset, (uint)userDataListOffset);
                },
                blockHeadFilePos
                );
        }

        void WriteAnimShareBlock(Lan::AnimShare animShare)
        {
            if (animShare.animShareInfo == null)
            {
                throw new ApplicationException("animShare.animShareInfo must be not null.");
            }

            long blockHeadFilePos = _stream.Position;

            _binWriter.WriteBlock(animShare, 0, 0);

            long animShareInfoOffset = _stream.Position - blockHeadFilePos;

            foreach (Lan::AnimShareInfo animShareInfo in animShare.animShareInfo)
            {
                _binWriter.WriteStruct(animShareInfo);
            }

            // データブロックの再書き込み
            FileUtil.WriteDataBlock(
                _stream,
                delegate(uint blockSize)
                {
                    _binWriter.WriteBlock(animShare, blockSize, (uint)animShareInfoOffset);
                },
                blockHeadFilePos
                );
        }

        static IEnumerable<string> GetGroupRefEnumerable(Lan::GroupRef[] groupRefs)
        {
            foreach (Lan::GroupRef groupRef in groupRefs)
            {
                yield return groupRef.name;
            }
        }

        void WriteAnimBlock(MergeAnimData mergeAnimData)
        {
            long blockHeadFilePos = _stream.Position;

            _binWriter.WriteBlock(mergeAnimData.AnimContentAry, _fileNameList, _animSec.Tag, 0, 0);

            if (_fileNameList.Count > 0)
            {
                NameTable nameTable = DataUtil.GenNameTable(
                    GetFileNameEnumerable(_fileNameList),
                    Encoding.ASCII,
                    (uint)Marshal.SizeOf(typeof(uint)),
                    0);

                FileUtil.WriteOffsetTable(_stream, nameTable.offsets);  // オフセットテーブルの書き込み
                FileUtil.WriteStringPool(_stream, nameTable);           // 文字列プールの書き込み
            }

            long animContOffsetsOffset = _stream.Position - blockHeadFilePos;

            FileUtil.WriteArray(
                WriteAnimContent,
                _stream,
                (ICollection<MergeAnimContent>)mergeAnimData.AnimContentAry,
                blockHeadFilePos);

            // データブロックの再書き込み
            FileUtil.WriteDataBlock(
                _stream,
                delegate(uint blockSize)
                {
                    _binWriter.WriteBlock(mergeAnimData.AnimContentAry, _fileNameList, _animSec.Tag, blockSize, (uint)animContOffsetsOffset);
                },
                blockHeadFilePos
                );
        }

        static IEnumerable<string> GetFileNameEnumerable(SortedList<string, Lan::RefRes> fileNameList)
        {
            foreach (Lan::RefRes refRes in fileNameList.Values)
            {
                yield return refRes.GetFileName(true);
            }
        }

        void WriteAnimContent(MergeAnimContent content)
        {
            Debug.Assert(content.TargetLists != null);

            Debug.WriteLine(string.Format("AnimPane {0}", content.Name));
            long animContentFilePos = _stream.Position;

            List<AnimTargetListInfo> tgtListInfoList = new List<AnimTargetListInfo>(content.TargetLists.Length);

            int animTypeIdx = 0;
            foreach (List<Lan::AnimTarget> targetList in content.TargetLists)
            {
                if (targetList != null)
                {
                    tgtListInfoList.Add(
                        new AnimTargetListInfo(
                            MergeAnimData.GetAnimationType(content.ContentType, animTypeIdx),
                            targetList)
                            );
                }

                ++animTypeIdx;
            }

            _binWriter.WriteBlock(content, (byte)tgtListInfoList.Count);

            if (content.ContentType == AnimContentType.ExtUserData)
            {
                // ExtUserData がアニメーション対象に指定されたときの特殊処理。
                // 対象の ExtUserData の名前をファイルに書き出したいが AnimContent に書き出す枠が存在しないため
                // AnimContent の後ろに StringTable として書き出す。
                // AnimContent の内容も StringTable も可変長のためそれぞれオフセットの指定が必要となるため
                // offsets として 0 番目に AnimContent のデータへのオフセット、1 番目に StringTable へのオフセットを書き出す。
                uint[] offsets = new uint[2];

                offsets[0] = 0;
                offsets[1] = 0;
                long tableOffsetPos = _stream.Position;

                FileUtil.WriteOffsetTable(_stream, offsets);  // オフセットテーブルの書き込み

                // AnimInfo へのオフセットを計算
                offsets[0] = (uint)(_stream.Position - animContentFilePos);

                FileUtil.WriteArray(
                    WriteAnimInfo,
                    _stream,
                    tgtListInfoList,
                    animContentFilePos);

                // ExtUserDataTarget テキストテーブルへのオフセットを計算
                offsets[1] = (uint)(_stream.Position - animContentFilePos);
                List<string> extUserDataTargetName = new List<string>();
                extUserDataTargetName.Add(content.extUserDataTargetName);

                NameTable nameTable = DataUtil.GenNameTable(
                                        extUserDataTargetName,
                                        Encoding.ASCII,
                                        (uint)Marshal.SizeOf(typeof(uint)),
                                        0);


                //  名前テーブルを書き出す
                FileUtil.WriteOffsetTable(_stream, nameTable.offsets);
                FileUtil.WriteStringPool(_stream, nameTable);

                long finishPos = _stream.Position;

                //  AnimInfo の位置にシークして kind のオフセット値を更新する
                _stream.Seek(tableOffsetPos, SeekOrigin.Begin);

                FileUtil.WriteOffsetTable(_stream, offsets);  // オフセットテーブルの書き込み

                //  最終書き込み位置に戻す
                _stream.Seek(finishPos, SeekOrigin.Begin);
            }
            else
            {
                FileUtil.WriteArray(
                    WriteAnimInfo,
                    _stream,
                    tgtListInfoList,
                    animContentFilePos);
            }
        }

        void WriteAnimInfo(AnimTargetListInfo tgtListInfo)
        {
            long animInfoFilePos = _stream.Position;

            //  通常の AnimationInfo 書き込み処理
            _binWriter.WriteBlock(tgtListInfo.Type, tgtListInfo.List);

            FileUtil.WriteArray(
                delegate(Lan::AnimTarget animTarget)
                {
                    WriteAnimTarget(animTarget, tgtListInfo.Type);
                },
                _stream,
                tgtListInfo.List,
                animInfoFilePos);
        }

        /// <summary>
        /// キーにガンマ補正処理を施す必要があるか判定します。
        /// </summary>
        bool IsKeyNeededDegamma(bool isDegammaSpecified, Lan::AnimTarget animTarget)
        {
            string animTargetName = Enum.GetName(typeof(Lan::AnimTargetType), animTarget.target);

            // FloatColor の場合はリニアワークフロー前提のためデガンマをかけない。
            return isDegammaSpecified &&
                IsTargetTypeColor(animTarget, animTargetName) &&
                !IsTargetTypeColorAlpha(animTarget, animTargetName) &&
                !IsTargetTypeFloatColor(animTarget, animTargetName);
        }

        void WriteAnimTarget(Lan::AnimTarget animTarget, Lan::AnimationType animType)
        {
            Lan::Hermite[] keys = animTarget.key;

            Debug.Assert(keys != null || animTarget.parameterizedAnimParameter != null);
            if (animTarget.parameterizedAnimParameter == null)
            {
                Debug.WriteLine(string.Format("  AnimTarget {0}, Hermite frame num {1}", animTarget.target, keys.Length));
            }
            else
            {
                Debug.WriteLine(string.Format("  AnimTarget {0}, ParameterizedAnim animation", animTarget.target));
            }

            int startFrame = _animSec.Tag.startFrame;

            _binWriter.WriteBlock(animTarget, animType);

            if (animTarget.parameterizedAnimParameter != null && animTarget.parameterizedAnimParameter.Length > 0)
            {
                // パラメタライズドアニメーションに複数のカーブが存在するときは、時間に対して昇順になるようにソートする
                Array.Sort(animTarget.parameterizedAnimParameter, (x, y) => x.offset.CompareTo(y.offset));

                long parameterFilePos = _stream.Position;
                _binWriter.WriteParameterizedAnim(animTarget.parameterizedAnimParameter);

                uint[] offsets = new uint[animTarget.parameterizedAnimParameter.Length];
                for (int i = 0; i < animTarget.parameterizedAnimParameter.Length; i++)
                {
                    offsets[i] = 0;
                }
                long tableOffsetPos = _stream.Position;
                FileUtil.WriteOffsetTable(_stream, offsets);

                for (int i = 0; i < animTarget.parameterizedAnimParameter.Length; i++)
                {
                    offsets[i] = (uint)(_stream.Position - parameterFilePos);
                    _binWriter.WriteParameterizedAnimParameter(animTarget.parameterizedAnimParameter[i], startFrame);
                }

                long finishPos = _stream.Position;
                _stream.Seek(tableOffsetPos, SeekOrigin.Begin);
                FileUtil.WriteOffsetTable(_stream, offsets);
                _stream.Seek(finishPos, SeekOrigin.Begin);

                return;
            }

            switch (animType)
            {
                case Lan::AnimationType.PaneSRT: _binWriter.WriteKeys(keys, startFrame, false); break;
            case Lan::AnimationType.Visibility:
                {
                    Lan::AnimVisibilityTarget viTarget = (Lan::AnimVisibilityTarget)animTarget;
                    try
                    {
                        _binWriter.WriteVisibilityKey(viTarget, startFrame);
                    }
                    catch (InvalidCastException)    // キーの型が正しくない(StepU16にキャストできない)
                    {
                        throw new LayoutDataException(Properties.Resources.ErrorInvalidVisibilityKeyType);
                    }
                }
                break;
            case Lan::AnimationType.VertexColor:
                    _binWriter.WriteKeys(keys, startFrame, IsKeyNeededDegamma(_bDegamma, animTarget));
                    break;
            case Lan::AnimationType.MaterialColor:
                    _binWriter.WriteKeys(keys, startFrame, IsKeyNeededDegamma(_bDegamma, animTarget));
                    break;
            case Lan::AnimationType.TextureSRT: _binWriter.WriteKeys(keys, startFrame, false); break;
            case Lan::AnimationType.TexturePattern:
                _binWriter.WriteTexturePatternKey(
                    (Lan::AnimTexPatternTarget)animTarget,
                    _fileNameList,
                    startFrame);
                break;
            case Lan::AnimationType.IndTextureSRT: _binWriter.WriteKeys(keys, startFrame, false); break;
            case Lan::AnimationType.AlphaTest: _binWriter.WriteKeys(keys, startFrame, false); break;
            case Lan::AnimationType.FontShadow:
                    _binWriter.WriteKeys(keys, startFrame, IsKeyNeededDegamma(_bDegamma, animTarget));
                    break;
            case Lan::AnimationType.PerCharacterTransform: _binWriter.WriteKeys(keys, startFrame, false); break;
            case Lan::AnimationType.PerCharacterTransformCurve:
                    _binWriter.WriteKeys(keys, startFrame, IsKeyNeededDegamma(_bDegamma, animTarget));
                    break;
            case Lan::AnimationType.Window: _binWriter.WriteKeys(keys, startFrame, false); break;
            case Lan::AnimationType.ExtUserData:
                _binWriter.WriteKeys(keys, startFrame, IsKeyNeededDegamma(_bDegamma, animTarget));
                break;
            case Lan::AnimationType.MaskTextureSRT:  _binWriter.WriteKeys(keys, startFrame, false); break;
            case Lan::AnimationType.DropShadow:
                    _binWriter.WriteKeys(keys, startFrame, IsKeyNeededDegamma(_bDegamma, animTarget));
                    break;
            case Lan::AnimationType.ProceduralShape:
                    _binWriter.WriteKeys(keys, startFrame, IsKeyNeededDegamma(_bDegamma, animTarget));
                    break;
            default: Debug.Assert(false); break;
            }

        }

        /// <summary>
        /// 対象がカラーアニメーションかどうか
        /// </summary>
        bool IsTargetTypeColor(Lan::AnimTarget animTarget, string animTargetName)
        {
            if(animTargetName.EndsWith("_r") ||
               animTargetName.EndsWith("_g") ||
               animTargetName.EndsWith("_b") ||
               animTargetName.EndsWith("_a"))
            {
                // カラーアニメーションは、_r _g _b _a という名称で終端するという想定です。
                return true;
            }
            else if(animTarget.target == Lan.AnimTargetType.PaneAlpha)
            {
                // ペイン透明度はカラーアニメーションとして扱います。
                return true;
            }

            return false;
        }

        bool IsTargetTypeColorAlpha(Lan::AnimTarget animTarget, string animTargetName)
        {
            if (animTargetName.EndsWith("_a"))
            {
                // カラーアニメーションは、_r _g _b _a という名称で終端するという想定です。
                return true;
            }
            else if (animTarget.target == Lan.AnimTargetType.PaneAlpha)
            {
                // ペイン透明度はカラーアニメーションとして扱います。
                if (_bDegammaOldBug)
                {
                    return false;
                }
                else
                {
                    return true;
                }
            }

            return false;
        }

        bool IsTargetTypeFloatColor(Lan::AnimTarget animTarget, string animTargetName)
        {
            // カラーアニメーションは、Float_r Float_g Float_b Float_a という名称で終端するという想定です。
            return (
                animTargetName.EndsWith("Float_r") ||
                animTargetName.EndsWith("Float_g") ||
                animTargetName.EndsWith("Float_b") ||
                animTargetName.EndsWith("Float_a"));
        }

        struct AnimTargetListInfo
        {
            public readonly Lan::AnimationType Type;
            public readonly List<Lan::AnimTarget> List;

            public AnimTargetListInfo(Lan::AnimationType animType, List<Lan::AnimTarget> list)
            {
                this.Type = animType;
                this.List = list;
            }

        }
    }
}
