﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.IO;
using System.Xml;
using System.Xml.Serialization;
using System.Xml.Schema;
using System.Diagnostics;
using System.Collections.Generic;
using System.Text;
using System.Text.RegularExpressions;
using System.Net;

namespace NW4F.LayoutBinaryConverter
{
    /// <summary>
    /// Class1 の概要の説明です。
    /// </summary>
    public class DataUtil
    {
        const int MajorVer = 1;
        const int MinorVer = 5;

        public static void VerifyDocumentVersion(string version)
        {
            // バージョン文字列は
            //   Major.Minor.Micro
            // で構成される。
            string[] verStrs = version.Split('.');                  // '.'で文字列を分割

            try
            {
                if (verStrs.Length == 3)
                {
                    int fileMajorVer = int.Parse(verStrs[0]);
                    int fileMinorVer = int.Parse(verStrs[1]);

                    // メジャーバージョンが同じで、マイナーが同じか下位バージョンである場合
                    if (fileMajorVer == MajorVer && fileMinorVer <= MinorVer)
                    {
                        return;
                    }
                }
            }
            // int.Parseの例外
            catch (FormatException) { }
            catch (OverflowException) { }

            throw new LayoutDataException(
                string.Format(
                    Properties.Resources.ErrorFileVersion,
                    string.Format("{0}.{1}.*", MajorVer, MinorVer),
                    version
                )
            );
        }

        public static string GetResourceTypeString(uint resType)
        {
            byte[] resTypeByteAry = BitConverter.GetBytes(System.Net.IPAddress.HostToNetworkOrder((int)resType));
            return Encoding.ASCII.GetString(resTypeByteAry);
        }

        public static NameTable GenNameTable(IEnumerable<string> strs, Encoding enc, uint stride, uint offsetBase)
        {
            List<byte[]> nameAry = new List<byte[]>();
            int totalBytes = 0;
            foreach (string str in strs)
            {
                byte[] nameBytes = enc.GetBytes(str + "\0");    // Cの終端文字を付加しておく
                nameAry.Add(nameBytes);
                totalBytes += nameBytes.Length;
            }

            NameTable nameTable = new NameTable();
            nameTable.offsets = new uint[nameAry.Count];

            totalBytes = (totalBytes + (4 - 1)) & ~(4 - 1);
            nameTable.namePool = new byte[totalBytes];

            nameTable.dataSize = (int)stride * nameTable.offsets.Length + nameTable.namePool.Length;

            uint offsetAdd = offsetBase + (uint)(stride * nameTable.offsets.Length);
            int dstIdx = 0;
            for (int i = 0; i < nameTable.offsets.Length; ++i)
            {
#if NW
                nameTable.offsets[i] = HostToTarget(offsetAdd + (uint)dstIdx);
#else
                nameTable.offsets[i] = (uint)System.Net.IPAddress.HostToNetworkOrder((int)(offsetAdd + (uint)dstIdx));
#endif
                Array.Copy(nameAry[i], 0, nameTable.namePool, dstIdx, nameAry[i].Length);
                dstIdx += nameAry[i].Length;
            }

            return nameTable;
        }

        /// <summary>
        /// タグ文字列をタグコードに変換する
        /// </summary>
        /// <param name="str"></param>
        /// <returns></returns>
        public static string ConvertTagString(string str)
        {
            StringBuilder sb = new StringBuilder(str.Length);

            Regex regEx = new Regex("[01][0-9a-fA-F]"); // ^ + 16新数で2文字 00 - 1F
            int startIdx = 0;
            int findIdx = -1;
            while (-1 != (findIdx = str.IndexOf('^', startIdx)))
            {
                Match match = null;

                int preStrLen = findIdx - startIdx;
                if (preStrLen > 0)
                {
                    sb.Append(str.Substring(startIdx, preStrLen));  // マッチした文字列以前の文字列を出力先にコピー
                }

                if (findIdx + 2 <= str.Length && str[findIdx + 1] == '^')     // ^^ -> ^
                {
                    sb.Append('^');
                    startIdx = findIdx + 2;
                }
                else if (findIdx + 3 <= str.Length && (match = regEx.Match(str.Substring(findIdx + 1, 2))).Success) // ^ + 16新数で2文字
                {
                    int tagCode = int.Parse(match.Value, System.Globalization.NumberStyles.HexNumber);
                    char tagChar = (char)tagCode;
                    sb.Append(tagChar);
                    startIdx = findIdx + 3;
                }
                else
                {
                    Report.Err.WriteLine(string.Format("invalid ^ character. - {0}", str));
                    startIdx = findIdx + 1;
                }
            }

            if (startIdx < str.Length)
            {
                sb.Append(str.Substring(startIdx)); // 残った文字列を出力先にコピー
            }
            return sb.ToString();
        }

        public static void ResizeArray<T>(ref T[] array, int lenMax)
        {
            if (array == null)
            {
                array = new T[0];
                return;
            }

            if (array.Length > lenMax)
            {
                Array.Resize(ref array, lenMax);
            }
        }

        public static short HostToTarget(short val)
        {
#if NW
            // TODO: バイトオーダーを正しく扱う
            return val;
#else
            return IPAddress.HostToNetworkOrder(val);
#endif
        }

        public static ushort HostToTarget(ushort val)
        {
#if NW
            // TODO: バイトオーダーを正しく扱う
            return val;
#else
            return (ushort)HostToTarget((short)val);
#endif
        }

        public static int HostToTarget(int val)
        {
#if NW
            // TODO: バイトオーダーを正しく扱う
            return val;
#else
            return IPAddress.HostToNetworkOrder(val);
#endif
        }

        public static uint HostToTarget(uint val)
        {
            return (uint)HostToTarget((int)val);
        }
    }

    public class NameTable
    {
        public uint[] offsets;
        public byte[] namePool;
        public int dataSize;
    }


}
