﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "StdAfx.h"
#include "WriteCommon.h"
#include "BinaryWriterBase.h"

using namespace System;
using namespace System::IO;

namespace tg = nn::ui2d;

namespace NW4F
{
namespace LayoutBinaryConverter
{

void
BinaryWriterBase::InitBinaryBlockHeader(
    nn::font::detail::BinaryBlockHeader*  pHeader,
    uint32_t             sig,
    uint32_t                         blockSize
)
{
    pHeader->kind = sig;
    pHeader->size = blockSize;
}

void
BinaryWriterBase::WriteBlockHeader(uint32_t sig)
{
    nn::font::detail::BinaryBlockHeader blockHead;
    InitBinaryBlockHeader(&blockHead, sig, sizeof(blockHead));
    WriteStructData(blockHead);
}

void
BinaryWriterBase::WriteBinaryFileHeader(
    uint32_t fileSig,
    uint32_t     fileSize,
    uint16_t     dataBlocks,
    bool    bBannerVersion,
    bool    compatible_0_12_x
)
{

    nn::font::detail::BinaryFileHeader binFileHeader = { 0 };

    binFileHeader.signature = fileSig;
    binFileHeader.byteOrder = nn::font::detail::ByteOrderMark;

    // バナー用バイナリのバージョン
    uint32_t BannerBinaryFileFormatVersion = NN_DETAIL_FONT_MAKE_VERSION(0, 8, 0, 0);

    // NintendoSDK 0.12.x でのリソースバージョン
    uint32_t Compatible_0_12_x_Version = NN_DETAIL_FONT_MAKE_VERSION(8, 0, 0, 0);

    if (compatible_0_12_x)
    {
        binFileHeader.version = Compatible_0_12_x_Version; // NintendoSDK 0.12.x 用互換バージョン
    }
    else if (bBannerVersion)
    {
        binFileHeader.version = BannerBinaryFileFormatVersion; // バナー用バージョン
    }
    else
    {
        binFileHeader.version = tg::BinaryFileFormatVersion; // ライブラリが想定するバージョン
    }

    binFileHeader.fileSize     = fileSize;
    binFileHeader.headerSize   = sizeof(binFileHeader);
    binFileHeader.dataBlocks   = dataBlocks;

    WriteStructData(binFileHeader);
}

void
BinaryWriterBase::WriteBinaryCpxFileHeader(
    uint32_t fileSig,
    uint32_t     fileSize,
    uint16_t     dataBlocks
)
{

    nn::font::detail::BinaryFileHeader binFileHeader = { 0 };

    binFileHeader.signature = fileSig;
    binFileHeader.byteOrder = nn::font::detail::ByteOrderMark;

    binFileHeader.version      = tg::BinaryCpxFileFormatVersion;

    binFileHeader.fileSize     = fileSize;
    binFileHeader.headerSize   = sizeof(binFileHeader);
    binFileHeader.dataBlocks   = dataBlocks;

    WriteStructData(binFileHeader);
}

void
BinaryWriterBase::WriteU8(uint8_t data)
{
    if(!swapEndian_)
    {
        WriteStructData(data);
    }else{
        uint8_t dst = static_cast<uint8_t>(data);
        WriteStructData(dst);
    }
}

void
BinaryWriterBase::WriteU16(uint16_t data)
{
    if(!swapEndian_)
    {
        WriteStructData(data);
    }else{
        uint16_t dst = static_cast<uint16_t>(data);
        WriteStructData(dst);
    }
}

void
BinaryWriterBase::WriteS16(int16_t data)
{
    if(!swapEndian_)
    {
        WriteStructData(data);
    }else{
        int16_t dst = static_cast<int16_t>(data);
        WriteStructData(dst);
    }
}

void
BinaryWriterBase::WriteU32(uint32_t data)
{
    if(!swapEndian_)
    {
        WriteStructData(data);
    }else{
        uint32_t dst = static_cast<uint32_t>(data);
        WriteStructData(dst);
    }
}

void
BinaryWriterBase::WriteS32(int32_t data)
{
    if(!swapEndian_)
    {
        WriteStructData(data);
    }else{
        int32_t dst = static_cast<int32_t>(data);
        WriteStructData(dst);
    }
}

void
BinaryWriterBase::WriteS64(int64_t data)
{
    if (!swapEndian_)
    {
        WriteStructData(data);
    }
    else {
        int64_t dst = static_cast<int64_t>(data);
        WriteStructData(dst);
    }
}

void
BinaryWriterBase::WriteF32(float data)
{
    if(!swapEndian_)
    {
        WriteStructData(data);
    }else{
        float dst = static_cast<float>(data);
        WriteStructData(dst);
    }
}

namespace
{

tg::ExtUserDataType
GetExtUserDataType(UserDataType type)
{
    switch (type)
    {
    case UserDataType::String:      return tg::ExtUserDataType_String;
    default:
    case UserDataType::IntList:     return tg::ExtUserDataType_Int;
    case UserDataType::FloatList:   return tg::ExtUserDataType_Float;
    case UserDataType::SystemData:   return tg::ExtUserDataType_SystemData;
    }
}

} // namespace

void
BinaryWriterBase::WriteStruct(UserDataHead^ userData)
{
    tg::ResExtUserData dst(
        userData->nameStrOffset,
        userData->dataOffset,
        userData->num,
        GetExtUserDataType(userData->type));
    WriteStructData(dst);
}

void
BinaryWriterBase::WriteUserDataList(uint16_t num, uint32_t blockSize)
{
    tg::ResExtUserDataList dst;
    std::memset(&dst, 0, sizeof(dst));
    dst.count = num;
    WriteBlockHeader(&dst, tg::DataBlockKindUserDataList, blockSize);
}

float
BinaryWriterBase::DegammaValue(float value)
{
    // 0.00129465f -> (1 / Math::Pow(255.f, 2.2f)) * 255.f
    return static_cast<float>(Math::Pow(value, 2.2) * 0.00129465);
}

}   // namespace LayoutBinaryConverter
}   // namespace NW4F
