﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "stdafx.h"

#include "WriteCommon.h"
#include "BinaryLanWriter.h"
#include <nn/ui2d/ui2d_Resources.h>

using namespace System;
using namespace System::Diagnostics;
using namespace System::IO;
using namespace System::Collections::Generic;
using namespace System::Text;

namespace sch = NW4F::LayoutBinaryConverter::Schema::Flan;
namespace tg = nn::ui2d;

namespace NW4F
{
namespace LayoutBinaryConverter
{

namespace
{

uint8_t
GetAnimTargetValue(sch::AnimTargetType targetType, sch::AnimationType animType)
{
    if (animType == sch::AnimationType::IndTextureSRT)
    {
        switch (targetType)
        {
        case sch::AnimTargetType::Rotate:       return tg::AnimTargetIndirectSrt_Rotate;    break;
        case sch::AnimTargetType::ScaleS:       return tg::AnimTargetIndirectSrt_ScaleS;    break;
        case sch::AnimTargetType::ScaleT:       return tg::AnimTargetIndirectSrt_ScaleT;    break;

        default:
           Debug::Assert(false);
           return 0;
        }
    }
    else
    {
        switch (targetType)
        {
        case sch::AnimTargetType::TranslateX:   return tg::AnimTargetPane_TranslateX;       break;
        case sch::AnimTargetType::TranslateY:   return tg::AnimTargetPane_TranslateY;       break;
        case sch::AnimTargetType::TranslateZ:   return tg::AnimTargetPane_TranslateZ;       break;
        case sch::AnimTargetType::RotateX:      return tg::AnimTargetPane_RotateX;          break;
        case sch::AnimTargetType::RotateY:      return tg::AnimTargetPane_RotateY;          break;
        case sch::AnimTargetType::RotateZ:      return tg::AnimTargetPane_RotateZ;          break;
        case sch::AnimTargetType::ScaleX:       return tg::AnimTargetPane_ScaleX;           break;
        case sch::AnimTargetType::ScaleY:       return tg::AnimTargetPane_ScaleY;           break;
        case sch::AnimTargetType::SizeW:        return tg::AnimTargetPane_SizeWidth;        break;
        case sch::AnimTargetType::SizeH:        return tg::AnimTargetPane_SizeHeight;       break;

        case sch::AnimTargetType::Visibility:   return tg::AnimTargetVisibility_Visibility; break;

        case sch::AnimTargetType::LT_r:         return tg::AnimTargetPaneColor_VertexLtR;      break;
        case sch::AnimTargetType::LT_g:         return tg::AnimTargetPaneColor_VertexLtG;      break;
        case sch::AnimTargetType::LT_b:         return tg::AnimTargetPaneColor_VertexLtB;      break;
        case sch::AnimTargetType::LT_a:         return tg::AnimTargetPaneColor_VertexLtA;      break;
        case sch::AnimTargetType::RT_r:         return tg::AnimTargetPaneColor_VertexRtR;      break;
        case sch::AnimTargetType::RT_g:         return tg::AnimTargetPaneColor_VertexRtG;      break;
        case sch::AnimTargetType::RT_b:         return tg::AnimTargetPaneColor_VertexRtB;      break;
        case sch::AnimTargetType::RT_a:         return tg::AnimTargetPaneColor_VertexRtA;      break;
        case sch::AnimTargetType::LB_r:         return tg::AnimTargetPaneColor_VertexLbR;      break;
        case sch::AnimTargetType::LB_g:         return tg::AnimTargetPaneColor_VertexLbG;      break;
        case sch::AnimTargetType::LB_b:         return tg::AnimTargetPaneColor_VertexLbB;      break;
        case sch::AnimTargetType::LB_a:         return tg::AnimTargetPaneColor_VertexLbA;      break;
        case sch::AnimTargetType::RB_r:         return tg::AnimTargetPaneColor_VertexRbR;      break;
        case sch::AnimTargetType::RB_g:         return tg::AnimTargetPaneColor_VertexRbG;      break;
        case sch::AnimTargetType::RB_b:         return tg::AnimTargetPaneColor_VertexRbB;      break;
        case sch::AnimTargetType::RB_a:         return tg::AnimTargetPaneColor_VertexRbA;      break;
        case sch::AnimTargetType::PaneAlpha:    return tg::AnimTargetPaneColor_PaneAlpha;      break;

        case sch::AnimTargetType::BlackColor_r:  return tg::AnimTargetMatColor_BufferR;  break;
        case sch::AnimTargetType::BlackColor_g:  return tg::AnimTargetMatColor_BufferG;  break;
        case sch::AnimTargetType::BlackColor_b:  return tg::AnimTargetMatColor_BufferB;  break;
        case sch::AnimTargetType::BlackColor_a:  return tg::AnimTargetMatColor_BufferA;  break;
        case sch::AnimTargetType::WhiteColor_r:  return tg::AnimTargetMatColor_Konst0R;  break;
        case sch::AnimTargetType::WhiteColor_g:  return tg::AnimTargetMatColor_Konst0G;  break;
        case sch::AnimTargetType::WhiteColor_b:  return tg::AnimTargetMatColor_Konst0B;  break;
        case sch::AnimTargetType::WhiteColor_a:  return tg::AnimTargetMatColor_Konst0A;  break;

        case sch::AnimTargetType::TevConstColor0_r: return tg::AnimTargetMatColor_C0_R; break;
        case sch::AnimTargetType::TevConstColor0_g: return tg::AnimTargetMatColor_C0_G; break;
        case sch::AnimTargetType::TevConstColor0_b: return tg::AnimTargetMatColor_C0_B; break;
        case sch::AnimTargetType::TevConstColor0_a: return tg::AnimTargetMatColor_C0_A; break;
        case sch::AnimTargetType::TevConstColor1_r: return tg::AnimTargetMatColor_C1_R; break;
        case sch::AnimTargetType::TevConstColor1_g: return tg::AnimTargetMatColor_C1_G; break;
        case sch::AnimTargetType::TevConstColor1_b: return tg::AnimTargetMatColor_C1_B; break;
        case sch::AnimTargetType::TevConstColor1_a: return tg::AnimTargetMatColor_C1_A; break;
        case sch::AnimTargetType::TevConstColor2_r: return tg::AnimTargetMatColor_C2_R; break;
        case sch::AnimTargetType::TevConstColor2_g: return tg::AnimTargetMatColor_C2_G; break;
        case sch::AnimTargetType::TevConstColor2_b: return tg::AnimTargetMatColor_C2_B; break;
        case sch::AnimTargetType::TevConstColor2_a: return tg::AnimTargetMatColor_C2_A; break;
        case sch::AnimTargetType::TevConstColor3_r: return tg::AnimTargetMatColor_C3_R; break;
        case sch::AnimTargetType::TevConstColor3_g: return tg::AnimTargetMatColor_C3_G; break;
        case sch::AnimTargetType::TevConstColor3_b: return tg::AnimTargetMatColor_C3_B; break;
        case sch::AnimTargetType::TevConstColor3_a: return tg::AnimTargetMatColor_C3_A; break;
        case sch::AnimTargetType::TevConstColor4_r: return tg::AnimTargetMatColor_C4_R; break;
        case sch::AnimTargetType::TevConstColor4_g: return tg::AnimTargetMatColor_C4_G; break;
        case sch::AnimTargetType::TevConstColor4_b: return tg::AnimTargetMatColor_C4_B; break;
        case sch::AnimTargetType::TevConstColor4_a: return tg::AnimTargetMatColor_C4_A; break;

        case sch::AnimTargetType::BlackColorFloat_r:  return tg::AnimTargetMatColorFloat_BufferR;  break;
        case sch::AnimTargetType::BlackColorFloat_g:  return tg::AnimTargetMatColorFloat_BufferG;  break;
        case sch::AnimTargetType::BlackColorFloat_b:  return tg::AnimTargetMatColorFloat_BufferB;  break;
        case sch::AnimTargetType::BlackColorFloat_a:  return tg::AnimTargetMatColorFloat_BufferA;  break;
        case sch::AnimTargetType::WhiteColorFloat_r:  return tg::AnimTargetMatColorFloat_Konst0R;  break;
        case sch::AnimTargetType::WhiteColorFloat_g:  return tg::AnimTargetMatColorFloat_Konst0G;  break;
        case sch::AnimTargetType::WhiteColorFloat_b:  return tg::AnimTargetMatColorFloat_Konst0B;  break;
        case sch::AnimTargetType::WhiteColorFloat_a:  return tg::AnimTargetMatColorFloat_Konst0A;  break;

        case sch::AnimTargetType::TevConstColor0Float_r: return tg::AnimTargetMatColorFloat_C0_R; break;
        case sch::AnimTargetType::TevConstColor0Float_g: return tg::AnimTargetMatColorFloat_C0_G; break;
        case sch::AnimTargetType::TevConstColor0Float_b: return tg::AnimTargetMatColorFloat_C0_B; break;
        case sch::AnimTargetType::TevConstColor0Float_a: return tg::AnimTargetMatColorFloat_C0_A; break;
        case sch::AnimTargetType::TevConstColor1Float_r: return tg::AnimTargetMatColorFloat_C1_R; break;
        case sch::AnimTargetType::TevConstColor1Float_g: return tg::AnimTargetMatColorFloat_C1_G; break;
        case sch::AnimTargetType::TevConstColor1Float_b: return tg::AnimTargetMatColorFloat_C1_B; break;
        case sch::AnimTargetType::TevConstColor1Float_a: return tg::AnimTargetMatColorFloat_C1_A; break;
        case sch::AnimTargetType::TevConstColor2Float_r: return tg::AnimTargetMatColorFloat_C2_R; break;
        case sch::AnimTargetType::TevConstColor2Float_g: return tg::AnimTargetMatColorFloat_C2_G; break;
        case sch::AnimTargetType::TevConstColor2Float_b: return tg::AnimTargetMatColorFloat_C2_B; break;
        case sch::AnimTargetType::TevConstColor2Float_a: return tg::AnimTargetMatColorFloat_C2_A; break;
        case sch::AnimTargetType::TevConstColor3Float_r: return tg::AnimTargetMatColorFloat_C3_R; break;
        case sch::AnimTargetType::TevConstColor3Float_g: return tg::AnimTargetMatColorFloat_C3_G; break;
        case sch::AnimTargetType::TevConstColor3Float_b: return tg::AnimTargetMatColorFloat_C3_B; break;
        case sch::AnimTargetType::TevConstColor3Float_a: return tg::AnimTargetMatColorFloat_C3_A; break;
        case sch::AnimTargetType::TevConstColor4Float_r: return tg::AnimTargetMatColorFloat_C4_R; break;
        case sch::AnimTargetType::TevConstColor4Float_g: return tg::AnimTargetMatColorFloat_C4_G; break;
        case sch::AnimTargetType::TevConstColor4Float_b: return tg::AnimTargetMatColorFloat_C4_B; break;
        case sch::AnimTargetType::TevConstColor4Float_a: return tg::AnimTargetMatColorFloat_C4_A; break;

        case sch::AnimTargetType::AlphaTest   :  return tg::AnimTargetAlphaCompare_Ref;  break;
        case sch::AnimTargetType::FontShadowBlackColor_r:  return tg::AnimTargetFontShadow_BlackInterporateColorR;  break;
        case sch::AnimTargetType::FontShadowBlackColor_g:  return tg::AnimTargetFontShadow_BlackInterporateColorG;  break;
        case sch::AnimTargetType::FontShadowBlackColor_b:  return tg::AnimTargetFontShadow_BlackInterporateColorB;  break;
        case sch::AnimTargetType::FontShadowWhiteColor_r:  return tg::AnimTargetFontShadow_WhiteInterporateColorR;  break;
        case sch::AnimTargetType::FontShadowWhiteColor_g:  return tg::AnimTargetFontShadow_WhiteInterporateColorG;  break;
        case sch::AnimTargetType::FontShadowWhiteColor_b:  return tg::AnimTargetFontShadow_WhiteInterporateColorB;  break;
        case sch::AnimTargetType::FontShadowWhiteColor_a:  return tg::AnimTargetFontShadow_WhiteInterporateColorA;  break;

        case sch::AnimTargetType::PerCharacterTransformEvalTimeOffset: return tg::AnimTargetPerCharacterTransform_EvalTypeOffset; break;
        case sch::AnimTargetType::PerCharacterTransformEvalTimeWidth: return tg::AnimTargetPerCharacterTransform_EvalTypeWidth; break;
        case sch::AnimTargetType::PerCharacterTransformScaleX: return tg::AnimTargetPerCharacterTransformCurve_ScaleX; break;
        case sch::AnimTargetType::PerCharacterTransformScaleY: return tg::AnimTargetPerCharacterTransformCurve_ScaleY; break;
        case sch::AnimTargetType::PerCharacterTransformTranslateX: return tg::AnimTargetPerCharacterTransformCurve_TranslateX; break;
        case sch::AnimTargetType::PerCharacterTransformTranslateY: return tg::AnimTargetPerCharacterTransformCurve_TranslateY; break;
        case sch::AnimTargetType::PerCharacterTransformTranslateZ: return tg::AnimTargetPerCharacterTransformCurve_TranslateZ; break;
        case sch::AnimTargetType::PerCharacterTransformRotateX: return tg::AnimTargetPerCharacterTransformCurve_RotateX; break;
        case sch::AnimTargetType::PerCharacterTransformRotateY: return tg::AnimTargetPerCharacterTransformCurve_RotateY; break;
        case sch::AnimTargetType::PerCharacterTransformRotateZ: return tg::AnimTargetPerCharacterTransformCurve_RotateZ; break;
        case sch::AnimTargetType::PerCharacterTransformLT_r: return tg::AnimTargetPerCharacterTransformCurve_LeftTopR; break;
        case sch::AnimTargetType::PerCharacterTransformLT_g: return tg::AnimTargetPerCharacterTransformCurve_LeftTopG; break;
        case sch::AnimTargetType::PerCharacterTransformLT_b: return tg::AnimTargetPerCharacterTransformCurve_LeftTopB; break;
        case sch::AnimTargetType::PerCharacterTransformLT_a: return tg::AnimTargetPerCharacterTransformCurve_LeftTopA; break;
        case sch::AnimTargetType::PerCharacterTransformLB_r: return tg::AnimTargetPerCharacterTransformCurve_LeftBottomR; break;
        case sch::AnimTargetType::PerCharacterTransformLB_g: return tg::AnimTargetPerCharacterTransformCurve_LeftBottomG; break;
        case sch::AnimTargetType::PerCharacterTransformLB_b: return tg::AnimTargetPerCharacterTransformCurve_LeftBottomB; break;
        case sch::AnimTargetType::PerCharacterTransformLB_a: return tg::AnimTargetPerCharacterTransformCurve_LeftBottomA; break;

        case sch::AnimTargetType::WindowFrameTop: return tg::AnimTargetWindow_FrameTop; break;
        case sch::AnimTargetType::WindowFrameBottom: return tg::AnimTargetWindow_FrameBottom ; break;
        case sch::AnimTargetType::WindowFrameLeft: return tg::AnimTargetWindow_FrameLeft; break;
        case sch::AnimTargetType::WindowFrameRight: return tg::AnimTargetWindow_FrameRight; break;

        case sch::AnimTargetType::TranslateS:   return tg::AnimTargetTexSrt_TranslateS;     break;
        case sch::AnimTargetType::TranslateT:   return tg::AnimTargetTexSrt_TranslateT;     break;
        case sch::AnimTargetType::Rotate:       return tg::AnimTargetTexSrt_Rotate;         break;
        case sch::AnimTargetType::ScaleS:       return tg::AnimTargetTexSrt_ScaleS;         break;
        case sch::AnimTargetType::ScaleT:       return tg::AnimTargetTexSrt_ScaleT;         break;

        case sch::AnimTargetType::Image:        return tg::AnimTargetTexPattern_Image;      break;

        case sch::AnimTargetType::ExUserDataDegammaByteColor_r:   return tg::AnimTargetPane_TranslateX;       break;
        case sch::AnimTargetType::ExUserDataDegammaByteColor_g:   return tg::AnimTargetPane_TranslateX;       break;
        case sch::AnimTargetType::ExUserDataDegammaByteColor_b:   return tg::AnimTargetPane_TranslateX;       break;
        case sch::AnimTargetType::ExUserDataDegammaByteColor_a:   return tg::AnimTargetPane_TranslateX;       break;

        case sch::AnimTargetType::MaskMtxTransX: return tg::AnimTargetMaskTexSrt_TranslateX;  break;
        case sch::AnimTargetType::MaskMtxScaleX: return tg::AnimTargetMaskTexSrt_ScaleX;  break;
        case sch::AnimTargetType::MaskMtxTransY: return tg::AnimTargetMaskTexSrt_TranslateY;  break;
        case sch::AnimTargetType::MaskMtxScaleY: return tg::AnimTargetMaskTexSrt_ScaleY;  break;
        case sch::AnimTargetType::MaskMtxRotate: return tg::AnimTargetMaskTexSrt_Rotate;  break;

        case sch::AnimTargetType::StrokeSize: return tg::AnimTargetDropShadow_StrokeSize; break;
        case sch::AnimTargetType::StrokeColor_r: return tg::AnimTargetDropShadow_StrokeColor_r; break;
        case sch::AnimTargetType::StrokeColor_g: return tg::AnimTargetDropShadow_StrokeColor_g; break;
        case sch::AnimTargetType::StrokeColor_b: return tg::AnimTargetDropShadow_StrokeColor_b; break;
        case sch::AnimTargetType::StrokeColor_a: return tg::AnimTargetDropShadow_StrokeColor_a; break;
        case sch::AnimTargetType::OuterGlowColor_r: return tg::AnimTargetDropShadow_OuterGlowColor_r; break;
        case sch::AnimTargetType::OuterGlowColor_g: return tg::AnimTargetDropShadow_OuterGlowColor_g; break;
        case sch::AnimTargetType::OuterGlowColor_b: return tg::AnimTargetDropShadow_OuterGlowColor_b; break;
        case sch::AnimTargetType::OuterGlowColor_a: return tg::AnimTargetDropShadow_OuterGlowColor_a; break;
        case sch::AnimTargetType::OuterGlowSpread: return tg::AnimTargetDropShadow_OuterGlowSpread; break;
        case sch::AnimTargetType::OuterGlowSize: return tg::AnimTargetDropShadow_OuterGlowSize; break;
        case sch::AnimTargetType::DropShadowColor_r: return tg::AnimTargetDropShadow_DropShadowColor_r; break;
        case sch::AnimTargetType::DropShadowColor_g: return tg::AnimTargetDropShadow_DropShadowColor_g; break;
        case sch::AnimTargetType::DropShadowColor_b: return tg::AnimTargetDropShadow_DropShadowColor_b; break;
        case sch::AnimTargetType::DropShadowColor_a: return tg::AnimTargetDropShadow_DropShadowColor_a; break;
        case sch::AnimTargetType::DropShadowAngle: return tg::AnimTargetDropShadow_DropShadowAngle; break;
        case sch::AnimTargetType::DropShadowDistance: return tg::AnimTargetDropShadow_DropShadowDistance; break;
        case sch::AnimTargetType::DropShadowSpread: return tg::AnimTargetDropShadow_DropShadowSpread; break;
        case sch::AnimTargetType::DropShadowSize: return tg::AnimTargetDropShadow_DropShadowSize; break;

        case sch::AnimTargetType::ProceduralShapeExp: return tg::AnimTargetProceduralShape_Exp; break;
        case sch::AnimTargetType::ProceduralShapeRadius: return tg::AnimTargetProceduralShape_Radius; break;
        case sch::AnimTargetType::ProceduralShapeInnerStrokeSize: return tg::AnimTargetProceduralShape_InnerStrokeSize; break;
        case sch::AnimTargetType::ProceduralShapeInnerStrokeColor_r: return tg::AnimTargetProceduralShape_InnerStrokeColor_r; break;
        case sch::AnimTargetType::ProceduralShapeInnerStrokeColor_g: return tg::AnimTargetProceduralShape_InnerStrokeColor_g; break;
        case sch::AnimTargetType::ProceduralShapeInnerStrokeColor_b: return tg::AnimTargetProceduralShape_InnerStrokeColor_b; break;
        case sch::AnimTargetType::ProceduralShapeInnerStrokeColor_a: return tg::AnimTargetProceduralShape_InnerStrokeColor_a; break;
        case sch::AnimTargetType::ProceduralShapeInnerShadowColor_r: return tg::AnimTargetProceduralShape_InnerShadowColor_r; break;
        case sch::AnimTargetType::ProceduralShapeInnerShadowColor_g: return tg::AnimTargetProceduralShape_InnerShadowColor_g; break;
        case sch::AnimTargetType::ProceduralShapeInnerShadowColor_b: return tg::AnimTargetProceduralShape_InnerShadowColor_b; break;
        case sch::AnimTargetType::ProceduralShapeInnerShadowColor_a: return tg::AnimTargetProceduralShape_InnerShadowColor_a; break;
        case sch::AnimTargetType::ProceduralShapeInnerShadowAngle: return tg::AnimTargetProceduralShape_InnerShadowAngle; break;
        case sch::AnimTargetType::ProceduralShapeInnerShadowDistance: return tg::AnimTargetProceduralShape_InnerShadowDistance; break;
        case sch::AnimTargetType::ProceduralShapeInnerShadowSize: return tg::AnimTargetProceduralShape_InnerShadowSize; break;
        case sch::AnimTargetType::ProceduralShapeColorOverlayColor_r: return tg::AnimTargetProceduralShape_ColorOverlayColor_r; break;
        case sch::AnimTargetType::ProceduralShapeColorOverlayColor_g: return tg::AnimTargetProceduralShape_ColorOverlayColor_g; break;
        case sch::AnimTargetType::ProceduralShapeColorOverlayColor_b: return tg::AnimTargetProceduralShape_ColorOverlayColor_b; break;
        case sch::AnimTargetType::ProceduralShapeColorOverlayColor_a: return tg::AnimTargetProceduralShape_ColorOverlayColor_a; break;
        case sch::AnimTargetType::ProceduralShapeGradationOverlayControl0: return tg::AnimTargetProceduralShape_GradationOverlayControl0; break;
        case sch::AnimTargetType::ProceduralShapeGradationOverlayControl1: return tg::AnimTargetProceduralShape_GradationOverlayControl1; break;
        case sch::AnimTargetType::ProceduralShapeGradationOverlayControl2: return tg::AnimTargetProceduralShape_GradationOverlayControl2; break;
        case sch::AnimTargetType::ProceduralShapeGradationOverlayControl3: return tg::AnimTargetProceduralShape_GradationOverlayControl3; break;
        case sch::AnimTargetType::ProceduralShapeGradationOverlayColor0_r: return tg::AnimTargetProceduralShape_GradationOverlayColor0_r; break;
        case sch::AnimTargetType::ProceduralShapeGradationOverlayColor0_g: return tg::AnimTargetProceduralShape_GradationOverlayColor0_g; break;
        case sch::AnimTargetType::ProceduralShapeGradationOverlayColor0_b: return tg::AnimTargetProceduralShape_GradationOverlayColor0_b; break;
        case sch::AnimTargetType::ProceduralShapeGradationOverlayColor0_a: return tg::AnimTargetProceduralShape_GradationOverlayColor0_a; break;
        case sch::AnimTargetType::ProceduralShapeGradationOverlayColor1_r: return tg::AnimTargetProceduralShape_GradationOverlayColor1_r; break;
        case sch::AnimTargetType::ProceduralShapeGradationOverlayColor1_g: return tg::AnimTargetProceduralShape_GradationOverlayColor1_g; break;
        case sch::AnimTargetType::ProceduralShapeGradationOverlayColor1_b: return tg::AnimTargetProceduralShape_GradationOverlayColor1_b; break;
        case sch::AnimTargetType::ProceduralShapeGradationOverlayColor1_a: return tg::AnimTargetProceduralShape_GradationOverlayColor1_a; break;
        case sch::AnimTargetType::ProceduralShapeGradationOverlayColor2_r: return tg::AnimTargetProceduralShape_GradationOverlayColor2_r; break;
        case sch::AnimTargetType::ProceduralShapeGradationOverlayColor2_g: return tg::AnimTargetProceduralShape_GradationOverlayColor2_g; break;
        case sch::AnimTargetType::ProceduralShapeGradationOverlayColor2_b: return tg::AnimTargetProceduralShape_GradationOverlayColor2_b; break;
        case sch::AnimTargetType::ProceduralShapeGradationOverlayColor2_a: return tg::AnimTargetProceduralShape_GradationOverlayColor2_a; break;
        case sch::AnimTargetType::ProceduralShapeGradationOverlayColor3_r: return tg::AnimTargetProceduralShape_GradationOverlayColor3_r; break;
        case sch::AnimTargetType::ProceduralShapeGradationOverlayColor3_g: return tg::AnimTargetProceduralShape_GradationOverlayColor3_g; break;
        case sch::AnimTargetType::ProceduralShapeGradationOverlayColor3_b: return tg::AnimTargetProceduralShape_GradationOverlayColor3_b; break;
        case sch::AnimTargetType::ProceduralShapeGradationOverlayColor3_a: return tg::AnimTargetProceduralShape_GradationOverlayColor3_a; break;
        case sch::AnimTargetType::ProceduralShapeGradationOverlayAngle: return tg::AnimTargetProceduralShape_GradationOverlayAngle; break;
        case sch::AnimTargetType::ProceduralShapeOuterShadowColor_r: return tg::AnimTargetProceduralShape_OuterShadowColor_r; break;
        case sch::AnimTargetType::ProceduralShapeOuterShadowColor_g: return tg::AnimTargetProceduralShape_OuterShadowColor_g; break;
        case sch::AnimTargetType::ProceduralShapeOuterShadowColor_b: return tg::AnimTargetProceduralShape_OuterShadowColor_b; break;
        case sch::AnimTargetType::ProceduralShapeOuterShadowColor_a: return tg::AnimTargetProceduralShape_OuterShadowColor_a; break;
        case sch::AnimTargetType::ProceduralShapeOuterShadowAngle: return tg::AnimTargetProceduralShape_OuterShadowAngle; break;

        default:
           Debug::Assert(false);
           return 0;
        }
    }
}

uint32_t
GetAnimationKind(sch::AnimationType type)
{
    switch (type)
    {
    case sch::AnimationType::PaneSRT:           return tg::AnimationTypePaneSrt;
    case sch::AnimationType::Visibility:        return tg::AnimationTypeVisibility;
    case sch::AnimationType::VertexColor:       return tg::AnimationTypeVertexColor;
    case sch::AnimationType::MaterialColor:     return tg::AnimationTypeMaterialColor;
    case sch::AnimationType::TextureSRT:        return tg::AnimationTypeTextureSrt;
    case sch::AnimationType::TexturePattern:    return tg::AnimationTypeTexturePattern;
    case sch::AnimationType::IndTextureSRT:     return tg::AnimationTypeIndirectSrt;
    case sch::AnimationType::AlphaTest:         return tg::AnimationTypeAlphaCompare;
    case sch::AnimationType::FontShadow:        return tg::AnimationTypeFontShadow;
    case sch::AnimationType::PerCharacterTransform: return tg::AnimationTypePerCharacterTransform;
    case sch::AnimationType::PerCharacterTransformCurve: return tg::AnimationTypePerCharacterTransformCurve;
    case sch::AnimationType::Window:            return tg::AnimationTypeWindow;
    case sch::AnimationType::ExtUserData:       return tg::AnimationTypeExtUserData;
    case sch::AnimationType::MaskTextureSRT:    return tg::AnimationTypeMaskTexSrt;
    case sch::AnimationType::DropShadow:        return tg::AnimationTypeDropShadow;
    case sch::AnimationType::ProceduralShape:   return tg::AnimationTypeProceduralShape;
    default:
        throw gcnew ArgumentOutOfRangeException();
    }
}

uint8_t
GetAnimationContentType(AnimContentType type)
{
    switch (type)
    {
    case AnimContentType ::Pane:        return tg::AnimContentType_Pane;
    case AnimContentType ::Material:    return tg::AnimContentType_Material;
    case AnimContentType ::ExtUserData:    return tg::AnimContentType_ExtUserData;

    default:
        throw gcnew ArgumentOutOfRangeException();
    }
}

}   // namespace

void
BinaryLanWriter::WriteBinaryFileHeader(uint32_t fileSize, uint16_t dataBlocks, bool bBannerVersion, bool bCompatible_0_12_x)
{
    BinaryWriterBase::WriteBinaryFileHeader(tg::FileSignatureFlan, fileSize, dataBlocks, bBannerVersion, bCompatible_0_12_x);
}

void
BinaryLanWriter::WriteBlock(AnimSection^ src, uint32_t blockSize, uint32_t nameOffset, uint32_t groupsOffset, uint32_t userDataListOffset)
{
    // ブロック
    tg::ResAnimationTagBlock dst;
    std::memset(&dst, 0, sizeof(dst));

    dst.tagOrder = uint16_t(src->TagOrder);
    dst.groupCount = uint16_t(src->Tag->group->Length);
    dst.nameOffset = nameOffset;
    dst.groupsOffset = groupsOffset;
    dst.userDataListOffset = userDataListOffset;
    dst.startFrame = Math::Min(Math::Max(src->Tag->startFrame, int32_t(int16_t::MinValue)), int32_t(int16_t::MaxValue));
    dst.endFrame =   Math::Min(Math::Max(src->Tag->endFrame,   int32_t(int16_t::MinValue)), int32_t(int16_t::MaxValue));
    uint8_t flag = 0;
    tg::detail::SetBit(&flag, tg::AnimTagFlag_DescendingBind, src->Tag->descendingBind);
    dst.flag = flag;
    WriteBlockHeader(&dst, tg::DataBlockKindPaneAnimTag, blockSize);
}

void
BinaryLanWriter::WriteStruct(sch::GroupRef^ src)
{
    tg::ResAnimationGroupRef dst;
    dst.SetDefault();

    CopyNameStrToCStr(dst.name, src->name, sizeof(dst.name));
    dst.flag = 0;

    WriteStructData(dst);
}

void
BinaryLanWriter::WriteBlock(sch::AnimShare^ src, uint32_t blockSize, uint32_t animShareInfoOffset)
{
    // ブロック
    tg::ResAnimationShareBlock dst;
    std::memset(&dst, 0, sizeof(dst));

    dst.shareCount = src->animShareInfo->Length;
    dst.animShareInfoOffset = animShareInfoOffset;

    WriteBlockHeader(&dst, tg::DataBlockKindPaneAnimShare, blockSize);
}

void
BinaryLanWriter::WriteStruct(Schema::Flan::AnimShareInfo^ src)
{
    tg::ResAnimationShareInfo dst;
    dst.SetDefault();

    CopyNameStrToCStr(dst.srcPaneName, src->srcPaneName, sizeof(dst.srcPaneName));
    CopyNameStrToCStr(dst.targetGroupName, src->targetGroupName, sizeof(dst.targetGroupName));

    WriteStructData(dst);
}

void
BinaryLanWriter::WriteBlock(array<MergeAnimContent^>^ animContentAry, SortedList<String^, sch::RefRes^>^ fileNameList, sch::AnimTag^ animTag, uint32_t blockSize, uint32_t animContOffsetsOffset)
{
    // ブロック
    tg::ResAnimationBlock animBlock;
    std::memset(&animBlock, 0, sizeof(animBlock));

    animBlock.frameSize = animTag->FrameSize;
    animBlock.loop = animTag->animLoop == sch::AnimLoopType::Loop ? 1: 0;
    animBlock.fileCount = fileNameList->Count;
    animBlock.animContCount = animContentAry->Length;
    animBlock.animContOffsetsOffset = animContOffsetsOffset;
    WriteBlockHeader(&animBlock, tg::DataBlockKindPaneAnimInfo, blockSize);
}

void
BinaryLanWriter::WriteBlock(MergeAnimContent^ content, uint8_t animContNum)
{
    tg::ResAnimationContent binAnimCont = { 0 };
    CopyNameStr(binAnimCont.name, content->Name, sizeof(binAnimCont.name));
    binAnimCont.count = animContNum;
    binAnimCont.type = GetAnimationContentType(content->ContentType);
    WriteStructData(binAnimCont);
}

void
BinaryLanWriter::WriteBlock(
    sch::AnimationType      animType,
    List<sch::AnimTarget^>^ list
)
{
    tg::ResAnimationInfo binAnimPane = { 0 };
    binAnimPane.kind = GetAnimationKind(animType);
    binAnimPane.count = list->Count;
    WriteStructData(binAnimPane);
}

void
BinaryLanWriter::WriteBlock(sch::AnimTarget^ animTarget, sch::AnimationType animType)
{
    tg::ResAnimationTarget binAnimTarget = { 0 };
    binAnimTarget.id = animTarget->id;
    binAnimTarget.target = GetAnimTargetValue(animTarget->target, animType);

    switch (animType)
    {
    case sch::AnimationType::PaneSRT:           binAnimTarget.curveType = nn::ui2d::AnimCurve_Hermite; break;
    case sch::AnimationType::Visibility:        binAnimTarget.curveType = nn::ui2d::AnimCurve_Step;    break;
    case sch::AnimationType::VertexColor:       binAnimTarget.curveType = nn::ui2d::AnimCurve_Hermite; break;
    case sch::AnimationType::MaterialColor:     binAnimTarget.curveType = nn::ui2d::AnimCurve_Hermite; break;
    case sch::AnimationType::TextureSRT:        binAnimTarget.curveType = nn::ui2d::AnimCurve_Hermite; break;
    case sch::AnimationType::TexturePattern:    binAnimTarget.curveType = nn::ui2d::AnimCurve_Step;    break;
    case sch::AnimationType::IndTextureSRT:     binAnimTarget.curveType = nn::ui2d::AnimCurve_Hermite; break;
    case sch::AnimationType::AlphaTest:         binAnimTarget.curveType = nn::ui2d::AnimCurve_Hermite; break;
    case sch::AnimationType::FontShadow:        binAnimTarget.curveType = nn::ui2d::AnimCurve_Hermite; break;
    case sch::AnimationType::PerCharacterTransform: binAnimTarget.curveType = nn::ui2d::AnimCurve_Hermite; break;
    case sch::AnimationType::PerCharacterTransformCurve: binAnimTarget.curveType = nn::ui2d::AnimCurve_Hermite; break;
    case sch::AnimationType::Window:            binAnimTarget.curveType = nn::ui2d::AnimCurve_Hermite; break;
    case sch::AnimationType::ExtUserData:       binAnimTarget.curveType = nn::ui2d::AnimCurve_Hermite; break;
    case sch::AnimationType::MaskTextureSRT:    binAnimTarget.curveType = nn::ui2d::AnimCurve_Hermite; break;
    case sch::AnimationType::DropShadow:        binAnimTarget.curveType = nn::ui2d::AnimCurve_Hermite; break;
    case sch::AnimationType::ProceduralShape:   binAnimTarget.curveType = nn::ui2d::AnimCurve_Hermite; break;
    default:
        throw gcnew ArgumentOutOfRangeException();
    }
    if (animTarget->parameterizedAnimParameter != nullptr)
    {
        binAnimTarget.curveType = nn::ui2d::AnimCurve_ParameterizedAnim;
    }

    binAnimTarget.keyCount = animTarget->key == nullptr ? 0 : animTarget->key->Length;
    binAnimTarget.keysOffset = sizeof(binAnimTarget);
    WriteStructData(binAnimTarget);

}

void BinaryLanWriter::WriteParameterizedAnim(array<sch::ParameterizedAnimParameter^>^ parameters)
{
    tg::ResParameterizedAnim block;
    std::memset(&block, 0, sizeof(block));
    block.parameterizedAnimCount = parameters->Length;
    WriteStructData(block);
}

void
BinaryLanWriter::WriteKeys(array<sch::Hermite^>^ keys, int32_t startFrame, bool _bDegamma)
{
    for (int i = 0; i < keys->Length; ++i)
    {
        sch::Hermite^ hermite = keys[i];

        tg::ResHermiteKey binHermite;
        binHermite.frame = hermite->frame - startFrame;
        if (_bDegamma)
        {
            // ガンマ特性を与える、0～255になっていることに注意が必要
            binHermite.value = static_cast<float>(DegammaValue(hermite->value));
        }
        else
        {
            binHermite.value = hermite->value;
        }
        binHermite.slope = hermite->slope;

        WriteStructData(binHermite);
    }
}

void
BinaryLanWriter::WriteTexturePatternKey(sch::AnimTexPatternTarget^ tpTarget, SortedList<String^, sch::RefRes^>^ fileNameList, int32_t startFrame)
{
    for each (sch::StepU16^ step in tpTarget->key)
    {
        Debug::WriteLine(String::Format("    frame {0}, value {1}", step->frame, step->value));

        tg::ResStepKey binStep;
        std::memset(&binStep, 0, sizeof(binStep));
        binStep.frame = step->frame - startFrame;
        int stepValue = (int)step->value;
        int index = fileNameList->IndexOfKey(tpTarget->refRes[stepValue]->name);
        Debug::Assert(index != -1);
        binStep.value = index;

        WriteStructData(binStep);
    }
}

void
BinaryLanWriter::WriteVisibilityKey(sch::AnimVisibilityTarget^ viTarget, int32_t startFrame)
{
    for each (sch::StepBool^ step in viTarget->key)
    {
        Debug::WriteLine(String::Format(L"    frame {0}, value {1}", step->frame, step->value));

        tg::ResStepKey binStep;
        std::memset(&binStep, 0, sizeof(binStep));
        binStep.frame = step->frame - startFrame;
        binStep.value = uint16_t(step->value);

        WriteStructData(binStep);
    }
}

void BinaryLanWriter::WriteParameterizedAnimParameter(Schema::Flan::ParameterizedAnimParameter^ parameter, int32_t startFrame)
{
    tg::ResParameterizedAnimParameter binParameter;
    std::memset(&binParameter, 0, sizeof(binParameter));
    binParameter.parameterizedAnimType = static_cast<uint8_t>(parameter->parameterizedAnimType);
    binParameter.value.startValue = parameter->startValue;
    binParameter.value.targetValue = parameter->targetValue;
    binParameter.offset = parameter->offset - startFrame;
    binParameter.duration = parameter->duration;
    WriteStructData(binParameter);
}

}   // namespace LayoutBinaryConverter
}   // namespace NW4F
