﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <vector>
#include <string>

#ifndef WIN32_LEAN_AND_MEAN
#define WIN32_LEAN_AND_MEAN
#endif
#ifndef NOMINMAX
#define NOMINMAX
#endif
#include <nn/nn_Windows.h>

namespace sharc {

typedef std::string PathString;

//------------------------------------------------------------------------------

// ハッシュを計算する
uint32_t calcHash32( const std::string& str, uint32_t key );

static const int cDefaultAlignment = 16;         ///< アライメントの規定値

// 文字列をコピーしたインスタンスを生成する
std::string* makeCopiedString(const std::string& str);

// multNumber の倍数に切り上げ
uint32_t roundUp(uint32_t val, uint32_t multNumber);

// エンディアン
enum EndianTypes
{
    cBig = 0,
    cLittle = 1
};

// エンディアンを変換する関数
uint16_t toHostU16(EndianTypes from, uint16_t x);
uint32_t toHostU32(EndianTypes from, uint32_t x);
uint16_t fromHostU16(EndianTypes to, uint16_t x);
uint32_t fromHostU32(EndianTypes to, uint32_t x);

// エンディアンマーク(Byte Order Mark)からエンディアンを判定する関数
EndianTypes markToEndian(uint16_t mark);

// 指定したエンディアンを表わすエンディアンマーク(Byte Order Mark)を返す関数
uint16_t endianToMark(EndianTypes endian);

// 拡張子を含んでいるかを判定
bool checkExistenceExt(const std::string& path);

// ディレクトリ名を取得
bool getDirectoryName(PathString& dst, const std::string& src);

// ファイル名を取得
bool getFileName(PathString& dst, const std::string& src);

// ディレクトリを作成
bool makeDirectory(const std::string& path);

// ファイルが存在するか調べる
bool tryIsExistFile(bool* is_exist, const std::string& path);

// ディレクトリが存在するか調べる
bool tryIsExistDirectory(bool* is_exist, const std::string& path);

// 親ディレクトリを含めてディレクトリを作成
bool makeDirectoryWithParent(const std::string& path);

// ロングネームとショートネームの対応
struct OptionName
{
    const char* pLongName;
    char shortName;
};

// コマンドライン引数を解釈するクラス
class GetOpt
{
public:
    // オプションがもう無いことを示す定数
    static const char END = static_cast<char>(-1);

    // コンストラクタ
    GetOpt(int argc, const char** argv, const char* opts, int longOptc, const OptionName* longOpts)
        : mArgc(argc), mArgv(argv), mOpts(opts), mLongOptc(longOptc), mLongOpts(longOpts)
    {
        mOptind = 1;
        mOptopt = 0;
        mOptarg = nullptr;
    }

    // 解釈を進める
    char next();

    // オプションの引数を返す
    const char* arg()
    {
        return mOptarg;
    }

    // オプションではない引数の個数を返す
    int getNonOptionArgNum() const
    {
        return static_cast<int>(mNonOptionArgs.size());
    }

    // index個目のオプションではない引数を返す
    const char* getNonOptionArg(int index) const
    {
        if (0 <= index && index < getNonOptionArgNum())
        {
            return mNonOptionArgs[index];
        }
        return nullptr;
    }

private:
    int mArgc;
    const char** mArgv;
    std::string mOpts;
    int mLongOptc;
    const OptionName* mLongOpts;

    int mOptind;
    char mOptopt;
    const char* mOptarg;
    std::vector<const char*> mNonOptionArgs;
};

//------------------------------------------------------------------------------
} // namespace sharc
