﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <string>
#include <vector>
#include "sharcArchive.h"

namespace sharc
{

/**
 *  アーカイブ内の情報を参照します
 */
class ArchiveReader
{
public:
    /// コンストラクタ
    ArchiveReader();
    /// デストラクタ
    virtual ~ArchiveReader();
    /**
     *  アーカイブをロードして、ファイル一覧を表示します
     *
     *  @param path アーカイブパス
     *
     *  @return 成功したらtrue
     */
    bool extractFileList(const std::string& path);

    /**
     *  アーカイブをロードするだけです
     *
     *  @param path アーカイブパス
     *
     *  @return 成功したらtrue
     */
    bool extractCommon(const std::string& path);

    /**
     *  アーカイブを解凍します
     *
     *  - アーカイブ名と同名のディレクトリを作業フォルダに作成し、ファイルを書き出します
     *  - ハッシュの衝突があるアーカイブはエラーです(ハッシュキーの大きいもの優先で上書きされます)
     *
     *  @param work 作業ディレクトリ
     *  @param path アーカイブパス
     *
     *  @return 成功したらtrue
     */
    bool resolve(const std::string& work, const std::string& path);

    /**
     *  アーカイブを比較します
     *
     *  - 自分自身と比較対象のアーカイブの両方のextractCommonを実行してから呼び出してください
     *  - hashを基に順番にサイズの一致を確認し、最終的にはファイルイメージブロックをバイナリ比較します
     *  - 一致しない場合、サイズが一致しないファイルがある or ファイルイメージブロックが異なる等のエラーを表示します
     *
     *  @param target 比較対象のアーカイブをロードしたArchiveReader
     *
     *  @return 一致したらtrue
     */
    bool compare( const ArchiveReader& target );

    /// ログ出力の抑止を設定します
    void setLogSilent(bool isLogSilent) { mIsLogSilent = isLogSilent; }
    /// ログ出力の抑止を取得します
    bool getLogSilent() const { return mIsLogSilent; }

private:
    void extractBntxList(void* pResTexFile);
    void OutLogMessage(char* pMsg, ...);

    const ArchiveBlockHeader*           mArchiveBlockHeader;    ///< ArchiveBlockHeader(nullptrのとき未initialize).
    const FATBlockHeader*               mFATBlockHeader;        ///< FATBlockHeader.
    const char*                         mFNTBlock;              ///< FNTBlockの先頭アドレス.
    std::vector<const FATEntry*>  mFATEntrys;             ///< ファイルアロケーションテーブルのエントリ配列
    FATEntry**                          mFATEntryPtrArray;      ///< FATEntry の実体。
    const uint8_t*                           mDataBlock;             ///< データブロックのポインタ
    bool                                mIsExtracted;
    bool                                mIsLogSilent;           ///< ログを休止しているかどうか
};

} // namespace sharc
