﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <string>
#include <vector>
#include "sharcArchive.h"
#include "sharcUtil.h"

namespace sharc {

//------------------------------------------------------------------------------
/**
 *  アーカイバの設定を管理します
 *
 */
class ArchiveConfig
{
public:
    /// コンストラクタ
    explicit ArchiveConfig();
    /// デストラクタ
    virtual ~ArchiveConfig();

    static const int cMaxScriptNum = 0xff;          ///< アーカイブに指定できるファイルの上限数
    static const int cMaxExceptNum = 0xf;           ///< アーカイブ時に除外できるNGディレクトリの上限数
    static const int cHashKeyCandidateNum = 16;     ///< 用意済みのハッシュキーの候補数

    /// アーカイブの動作モードです
    enum Mode{
        cMode_Create,       ///< 新規作成
        cMode_List,         ///< アーカイブ内のパスを列挙する
        cMode_Resolve,      ///< アーカイブを解凍してファイルを個別に取り出します
    };

    /// ArchiveBuilder::addEntry用のエントリ
    class PathEntry{
    public:
        uint32_t                    align;   ///< アライメント
        std::string path_win;   ///< Windows上でのファイルパス
        std::string path_arc;   ///< アーカイブ内でのファイルパス
        PathEntry()
            : align(cDefaultAlignment)
            , path_win()
            , path_arc(){}
        /// デストラクタ
        ~PathEntry(){
        };
    };

    /// Script, Except用のエントリ
    class CommonEntry{
    public:
        std::string path;       ///< パス
        CommonEntry() : path(){}
        /// デストラクタ
        ~CommonEntry(){
        };
    };

    typedef std::vector<PathEntry> PathList;
    typedef std::vector<CommonEntry> ScriptList;
    typedef std::vector<CommonEntry> ExceptList;

public:
    /**
     *  アーカイブに含めるパスをリストに追加します
     *
     *  @param  path    ディレクトリ or ファイル
     *  @param  align   アライメント
     */
    void addArchivePath( const std::string& path, uint32_t align );

    /// スクリプトのパスをリストに追加します
    void addScript( const std::string& script );

    /// 除外ディレクトリ名をリストに追加します
    void addExcept( const std::string& dir );

    /// 比較対象のアーカイブファイル名を記憶します
    void storeArcName( const std::string& arc ){ mArcPath = arc; }

    /// 比較対象のアーカイブファイル名を参照します
    const PathString& getArcName() const { return mArcPath; }

    /// 設定が正しいか検証します. false を返すときは設定に誤りがあります.
    bool initialize();

    /**
     *  ワークディレクトリを設定します
     *
     *  D:/temp/ui がワークディレクトリとして設定されているとき
     *  D:/temp/ui/style/title.bctex は style/title.bctex としてアーカイブに登録されます
     *
     *  このメソッドで設定されないときは GetWorkingDirectory で取得されるディレクトリへのパスが自動で設定されます
     *  ハッシュ値の計算は、ワークディレクトリ分を除いたパス( 上記の例では style/title.bctex 部分 )で行います
     *
     *  @param  workdir     ワークディレクトリのパス
     */
    void setWorkDir( const std::string& workdir );

    /// アーカイブオプションの設定
    const uint16_t& getOption() const { return mOption; }
    /// アーカイブモードの設定
    void setMode(Mode mode) { mMode = mode; }
    /// アーカイブオプションの参照
    void setOption(uint16_t opt) { mOption = opt; }
    /// アーカイブモードの参照
    const Mode& getMode() const { return mMode; }
    /// アーカイブされるファイルの情報を表示するフラグの設定
    void enableDisplayFileInfo() { mIsDisplayFileInfo = true; }
    /// アーカイブされるファイルの情報を表示するフラグの参照
    bool isDisplayFileInfo() const { return mIsDisplayFileInfo; }

    /// プラットフォーム名を設定する
    void setPlatformsName(const char* pName) { mPlatformName = pName; }
    void setTileMode(const char* pName) { mTileMode = pName; }
    const std::string& getPlatformsName() { return mPlatformName; }
    const std::string& getTileMode() { return mTileMode; }

    /// ターゲットエンディアンの設定
    void setEndianType(EndianTypes type) { mEndian = type; }
    /// ターゲットエンディアンの参照
    EndianTypes getEndianType() const { return mEndian; }

    /// ワークディレクトリパスの参照
    const std::string& getWorkDirString() const { return mWorkDir; }
    /// アーカイブするファイルパスリストを取得
    const PathList& getPathList() const { return mPathList; }
    /// アーカイブの設定時にエラーが起こったか
    bool hasError() const { return mHasError; }

    /**
     *  アライメントの設定
     *
     *  - アーカイブ全体に指定するデフォルトアライメントを設定します.
     *  - スクリプトファイルで個別にアライメントを指定した場合は、そちらのアライメントが優先されます.
     *  - 不正なアライメントが指定されたときは 自動的に 4 が設定されたことになります.
     *
     *  @param align アライメント
     */
    void setAlignment(int32_t align) { mAlignment = checkAlign_(align); }
    /// アーカイブ全体に適用されるデフォルトアライメントの参照
    const uint32_t getAlignment() const { return mAlignment; }

    /**
     *  ハッシュ値が衝突しないハッシュキーを取得します
     *
     *  - 最大16パターンの素数で衝突しないか確認し、衝突しないキーを返そうと試みます
     *  - いずれの素数もハッシュ衝突が発生するときは、最も衝突が少ないキーを返します
     *
     *  @return 探索後のハッシュキーを返す. 既にキー指定があれば、そのキーを返します
     */
    uint32_t getHashKey();
    /// ハッシュキーを設定します
    void setHashKey(uint32_t key) { mHashKey = key; }

    /// ログ出力の抑止を設定します
    void setLogSilent(bool isLogSilent) { mIsLogSilent = isLogSilent; }
    /// ログ出力の抑止を取得します
    bool getLogSilent() const { return mIsLogSilent; }

    /// Ftxb を残すかどうかを設定します
    void setKeepFtxb(bool isKeepFtxb) { mIsKeepFtxb = isKeepFtxb; }
    /// Ftxb を残すかどうかを取得します
    bool getKeepFtxb() const { return mIsKeepFtxb; }

    /// ShaderVariationXml を残すかどうかを設定します
    void setKeepShaderVariationXml(bool isKeepShaderVariationXml) { mIsKeepShaderVariationXml = isKeepShaderVariationXml; }
    /// ShaderVariationXml を残すかどうかを取得します
    bool getKeepShaderVariationXml() const { return mIsKeepShaderVariationXml; }
    /// シェーダー変換時のキャッシュパス の設定。
    void SetShaderConvertCachePath(const char* pPath) { pShaderConvertCachePath = pPath; }
    /// シェーダー変換時のキャッシュパス です。
    const char* GetShaderConvertCachePath() const { return pShaderConvertCachePath; }


    /// 指定されたアライメントの最大値を取得します
    uint32_t getMaxAlignment() const { return mMaxAlignment; }

    /// ApiTypeName の設定
    void SetApiTypeName(const char* pName)
    {
        pApiTypeName = pName;
    }
    /// ApiTypeName の取得
    const char* GetApiTypeName() const
    {
        return pApiTypeName;
    }
    /// CodeTypeName の設定
    void SetCodeTypeName(const char* pName)
    {
        pCodeTypeName = pName;
    }
    /// CodeTypeName の取得
    const char* GetCodeTypeName() const
    {
        return pCodeTypeName;
    }

    /// タイル最適化の設定
    void setTileOptimize(const char* pName) { mTileOptimize = pName; }

    /// タイル最適化取得
    const std::string& getTileOptimize() { return mTileOptimize; }

    /// タイル最適化閾値の設定
    void setTileSizeThreshold(const char* pValue) { mTileSizeThreshold = pValue; }

    /// タイル最適化閾値取得
    const std::string& getTileSizeThreshold() { return mTileSizeThreshold; }

private:
    /// 与えられたパスのディレクトリに除外キーワードが入っていないかを判定する
    bool isIncludeExceptDirName_(const std::string& path);

    /**
     *  アーカイブに含めるパスをスクリプトファイルからリストに追加します
     *
     *  - スクリプトファイルは1行1ファイルを [ alignment, filepath ] と並べてください
     *  - 自動的にディレクトリかファイルかの判定が行われ、存在しないパスはスキップされます
     *  - スクリプト内の filepath にはワイルドカード(*, ?)を制限つきで使用できます
     *  - ワイルドカードはパスのファイル部分に書いたものだけ有効です.
     *  - サブディレクトリ以下のファイルもワイルドカード指定の対象となります. 例えば, test/*.txt ならば testディレクトリ以下の全てのテキストファイルを対象になります
     *
     *  @param  path    スクリプトファイルのパス
     */
    void addArchivePathWithScript_( const std::string& script );

    /**
     *  アーカイブに含めるファイルパスをリストに追加します]
     *
     *  path_arc に空白文字列が指定されたときは、path_winと一致するように変更します
     *
     *  @param  path_win    Windows上でのファイルパス
     *  @param  path_arc    アーカイブ内での仮想パス
     *  @param  align       アライメント
     */
    void addArchiveFilePath_( const std::string& path_win, const std::string& path_arc, uint32_t align );

    /**
     *  ディレクトリを再帰的に巡回してアーカイブに含めるパスをリストに追加します
     *
     *  @param  dir     ディレクトリパス
     *  @param  align   アライメント
     */
    void addArchiveDirPath_( const std::string& dir, uint32_t align );

    /**
     *  アーカイブに含めるパスを拡張子を限定して検索し、リストに追加します
     *
     *  @param  path    ディレクトリ
     *  @param  card    拡張子のワイルドカード, *.bctex のように指定します
     *  @param  align   アライメント
     */
    void addArchivePathWithExtension_( const std::string& path, const std::string& card, uint32_t align );

    /// 不正なアライメントのときはデフォルトアライメントを返す
    uint32_t checkAlign_(int32_t align);

    /// ハッシュ値が衝突しないキーを探索します
    uint32_t findHashKey_();

    // bflim のアライメントを読みます。
    uint32_t ReadBflimAlignment_(const char* filePath);

    // Bnsh のアライメントを読みます。
    size_t ReadBnshAlignment_(const char* filePath);

    /// ログを出力します。
    void OutLogMessage(char* pMsg, ...);

private:
    uint16_t                             mOption;    ///< アーカイブのヘッダに書き込むオプション
    uint32_t                             mAlignment; ///< 個別指定しないときのアライメント
    uint32_t                             mMaxAlignment;  ///< 指定されたアライメントのうち、最大のもの。
    uint32_t                             mHashKey;   ///< ハッシュキー
    Mode                            mMode;      ///< アーカイブのモード
    bool                            mIsDisplayFileInfo;///< ハッシュリストを生成時に表示するか
    bool                            mHasError;  ///< 設定情報でエラーがあったか、なんらかのエラーが起こるとtrueになる
    EndianTypes                     mEndian;    ///< 作成するアーカイブのターゲットエンディアンタイプ
    PathString                      mWorkDir;   ///< 作業用ディレクトリ
    PathString                      mArcPath;   ///< 比較対象のアーカイブパス
    PathList                        mPathList;  ///< アーカイブに含めるパス
    ScriptList                      mScriptList;///< スクリプトファイルリスト
    ExceptList                      mExceptList;///< アーカイブから除くディレクトリ名リスト
    std::string                     mPlatformName;///< プラットフォーム名
    std::string                     mTileMode;    ///< タイルモード
    bool                            mIsLogSilent; ///< ログが抑制されているか
    bool                            mIsKeepFtxb;  ///< Bntx を生成した際に、Ftxb を残すかどうかです。
    bool                            mIsKeepShaderVariationXml;   ///< ShaderVariationXml を残すかどうかです。
    const char*                     pShaderConvertCachePath; ///< シェーダー変換時のキャッシュパス です。
    const char*                     pApiTypeName; ///< ApiTypeName です。
    const char*                     pCodeTypeName; ///< CodeTypeName です。
    std::string                     mTileOptimize; ///< タイル最適化
    std::string                     mTileSizeThreshold; ///< タイル最適化閾値

};

//------------------------------------------------------------------------------
} // namespace sharc
