﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "sharcArchiveConfig.h"
#include "sharcUtil.h"
#include <algorithm>

namespace {
    // ドライブを表すアルファベットを小文字にします
    void exchangeDrive2Lower(std::string* ptr)
    {
        if( ptr->size() > 2 ){
            // 先頭にドライブ名が付いているとき
            if( (*ptr)[1] == ':' ){
                // ドライブ名を小文字にする
                (*ptr)[0] = static_cast<char>(tolower(static_cast<int>((*ptr)[0])));
            }
        }
    }
    // バックスラッシュを スラッシュに置換します
    void exchangeBackSlash2Slash(std::string* dst)
    {
        int length = dst->size();
        exchangeDrive2Lower(dst);
        for( int index = 0; index < length; ++ index ){
            if( (*dst)[index] == '\\' ){
                (*dst)[index] = '/';
            }
        }
    }

    // リトライに使用する素数テーブル
    uint32_t sPrimeFactor[sharc::ArchiveConfig::cHashKeyCandidateNum] =
        { 101, 103, 107, 109, 113, 127, 131, 137, 139, 149, 151, 157, 163, 167, 173, 179 };

    inline bool isUnprintableChar_(char c)
    {
        if (c <= 0x20 || c == 0x7f)
        {
            return true;
        }
        else
        {
            return false;
        }
    }

    int rstripUnprintableAsciiChars(char str[])
    {
        int len = strlen(str);
        int count = 0;
        for (int i = len - 1; i >= 0; i--)
        {
            if (isUnprintableChar_(str[i]))
            {
                str[i] = 0;
                count++;
            }
            else
            {
                break;
            }
        }
        return count;
    }


}

namespace sharc {

//------------------------------------------------------------------------------

ArchiveConfig::ArchiveConfig()
: mOption(0)
, mAlignment(cDefaultAlignment)
, mMaxAlignment(cDefaultAlignment)
, mHashKey(0)
, mMode(cMode_Create)
, mIsDisplayFileInfo(false)
, mHasError(false)
, mEndian(EndianTypes::cLittle)
, mPathList()
, mScriptList()
, mExceptList()
, mIsLogSilent(false)
, mIsKeepFtxb(false)
, mIsKeepShaderVariationXml(false)
, pShaderConvertCachePath("")
, pApiTypeName("Gl")
, pCodeTypeName("Source")
, mTileOptimize("")
, mTileSizeThreshold("")
{
    setWorkDir(".");    // カレントディレクトリを取得しておく
}

//------------------------------------------------------------------------------

ArchiveConfig::~ArchiveConfig()
{
}

//------------------------------------------------------------------------------

uint32_t ArchiveConfig::findHashKey_()
{
    uint32_t size = mPathList.size();
    if( !size )
    {
        fprintf(stderr, "no file to include in archive.\n");
        mHasError = true;
        return mHashKey;
    }
    std::vector<uint32_t> hash_table;
    std::vector<uint32_t> hash_table2;

    uint32_t clash = 0x7fffffff;
    uint32_t key = sPrimeFactor[0];
    for( int i = 0; i < cHashKeyCandidateNum; i++ )
    {
        hash_table.clear();
        const PathList& list = mPathList;
        PathList::const_iterator end = list.end();
        for( PathList::const_iterator itr = list.begin(); itr != end; ++itr )
        {
            uint32_t val = calcHash32( itr->path_arc, sPrimeFactor[i] );

            // 要素の重複にかかわらず追加
            hash_table.push_back(val);

            // 要素が重複していなければ追加
            bool flag = false;
            const std::vector<uint32_t>::const_iterator end2 = hash_table2.end();
            for (std::vector<uint32_t>::const_iterator itr2 = hash_table2.begin(); itr2 != end2; ++itr2)
            {
                if (*itr2 == val)
                {
                    flag = true;
                    break;
                }
            }
            if (!flag)
            {
                hash_table2.push_back(val);
            }
        }
        const uint32_t before_size = hash_table.size();
        const uint32_t after_size = hash_table2.size();

        // ハッシュの重複がないときは、そのキーを返す
        if( before_size == after_size ){
            key = sPrimeFactor[i];
            break;
        }
        // ハッシュが衝突しているときは、衝突数がすくないキーを保存する
        else
        {
            uint32_t diff = before_size - after_size;
            if( clash > diff )
            {
                clash = diff;
                key = sPrimeFactor[i];
            }
        }
    }

    return key;
}

//------------------------------------------------------------------------------

uint32_t ArchiveConfig::getHashKey()
{
    // ハッシュキーが既に設定されているときはそのまま
    if( mHashKey )
    {
        return mHashKey;
    }

    // ハッシュキーが未セットなら探索
    mHashKey = findHashKey_();
    return mHashKey;
}

//------------------------------------------------------------------------------

void ArchiveConfig::setWorkDir( const std::string& workdir )
{
    // ワークディレクトリを変更します
    if( ! SetCurrentDirectoryA( workdir.c_str() ) )
    {
        fprintf(stderr, "not found. [ %s ]\n", workdir.c_str());
        mHasError = true;
        return;
    }

    // ワークディレクトリのフルパスを取得します
    char buffer[MAX_PATH + 1];
    if( ! GetCurrentDirectoryA( MAX_PATH+1, buffer ) )
    {
        fprintf(stderr, "can't get current directory.\n");
        mHasError = true;
        return;
    }

    mWorkDir = buffer;

    // ディレクトリの最終文字が / であることを保証します
    exchangeBackSlash2Slash( &mWorkDir );
    if( mWorkDir.back() != '/' ){
        mWorkDir.append("/");
    }
}

//------------------------------------------------------------------------------

void ArchiveConfig::addScript( const std::string& script )
{
    if( script.size() > MAX_PATH )
    {
        fprintf(stderr, "%s is too long path.\n", script.c_str());
        mHasError = true;
        return;
    }

    mScriptList.push_back(CommonEntry());
    CommonEntry* entry = &mScriptList.back();
    entry->path = script;
}

//------------------------------------------------------------------------------

void ArchiveConfig::addExcept( const std::string& dir )
{
    if( dir.size() > MAX_PATH )
    {
        fprintf(stderr, "%s is too long path.\n", dir.c_str());
        mHasError = true;
        return;
    }

    mExceptList.push_back(CommonEntry());
    CommonEntry* entry = &mExceptList.back();
    entry->path = dir;
}

//------------------------------------------------------------------------------

bool ArchiveConfig::initialize()
{
    // 内部エラーが既にあったらエラー
    if( mHasError ) return false;

    // スクリプトの検証
    const ScriptList& list = mScriptList;
    ScriptList::const_iterator end = list.end();
    for( ScriptList::const_iterator itr = list.begin(); itr != end; ++ itr )
    {
        addArchivePathWithScript_( itr->path );
    }

    // ハッシュキーの正当性検証
    getHashKey();

    return ( !mHasError );
}

//------------------------------------------------------------------------------

void ArchiveConfig::addArchivePathWithScript_( const std::string& script )
{
// このメソッド内だけで有効なマクロ
#define ERR_FINALIZE { mHasError = true; fclose(fp); return; }

    FILE* fp = nullptr;
    std::string path_win;
    std::string path_arc;
    {
        std::string file(script);

        // スクリプトファイルを開く
        errno_t res_code = fopen_s(&fp, file.c_str(), "r");
        // 正常に開けたときは 0 が返ってくる
        if (res_code != 0) {
            // スクリプトファイルが開けなければエラー
            fprintf(stderr, "not exist. [ %s ]\n", file.c_str());
            mHasError = true;
            return;
        }
    }

    const size_t bufferSize = MAX_PATH * 2;
    char line[bufferSize] = {};
    std::string         alignment;

    uint32_t line_num = 0, align = 0;
    while( fgets(line, bufferSize, fp ) != nullptr )
    {
        line_num ++;
        // コメントの解析
        int length = strlen(line);
        int comment_index = std::find(line, line + length, ';') - line;
        if( comment_index < length )    // コメントの記号が見つかった！
        {
            if( line[0] == ';' ){   // ラインの冒頭に ; がある場合は一行全てをコメントとみなす
                continue;
            }
            // コメント記号以降をラインから除く
            line[comment_index] = 0;
        }
        rstripUnprintableAsciiChars(line);

        // 区切り文字 , の数で書式確認
        length = strlen(line);
        int index1 = -1;    // 1つ目の区切り文字
        int index2 = -1;    // 2つ目の区切り文字
        int delimiter = 0;
        const char* line_ptr = line;
        for(int idx = 0; idx < length; ++idx ){
            if( line_ptr[idx] == ',' ){
                if( index1 < 0 ) { index1 = idx; }
                else if( index2 < 0 ) {
                    index2 = idx - index1;
                }
                delimiter ++;
            }
        }
        if( delimiter > 2 || delimiter == 0 ){
            fprintf(stderr, "caution, illegal format.\nline at %u => [ %s ]\n", line_num, line);
            ERR_FINALIZE
        }

        // アライメントの切り出し
        alignment.copy( line, index1 );
        // 先頭にある空白とタブを読み飛ばす処理
        std::string alignment_tmp;
        for(int idx = 0; idx < index1; ++ idx ){
            if( alignment[idx] == ' ' || alignment[idx] == '\t' ) continue;
            alignment_tmp = alignment.substr(idx);
            break;
        }

        {
            char* end_ptr;
            _set_errno(0);
            align = strtoul(alignment_tmp.c_str(), &end_ptr, 10);
            if (*end_ptr != '\0' || errno == ERANGE || errno == EINVAL)
            {
                // 正しく数値に変換できなかったが、現時点では特に何も対処しない
            }
        }

        // アライメントの正当性を検証
        align = checkAlign_( align );

        // パス部分の切り出し
        rstripUnprintableAsciiChars(line + index1 + 1);
        path_win = line + index1 + 1;
        // パス内の区切りをスラッシュに保証する
        exchangeBackSlash2Slash(&path_win);

        // 仮想パスが存在するとき
        if( index2 > 0 ){
            // 仮想パスが空でないか確認
            if( index2 == (length-index1-1) ){
                fprintf(stderr, "caution, virtual path is empty.\nline at %u => [ %s ]\n", line_num, line);
                ERR_FINALIZE
            }
            // 仮想パス部分を切り取り
            path_arc = path_win.substr(index2+1);
            // 仮想パス部分を削除
            int path_win_length = index2 - 1;
            path_win.resize(path_win_length);

            // 仮想パスがファイルを指すパスか拡張子の有無で確認
            if( !checkExistenceExt(path_arc) ){
                fprintf(stderr, "caution, virtual path has no extension.\nline at %u => [ %s ]\n", line_num, line);
                ERR_FINALIZE
            }

            // 実在パスにワイルドカードが含まれていないか
            if( path_win.find('*') != std::string::npos || path_win.find('?') != std::string::npos ){
                fprintf(stderr, "caution, illegal wildcard is including.\nline at %u => [ %s ]\n", line_num, line);
                ERR_FINALIZE
            }

            // 先頭にある空白とタブを読み飛ばす処理
            std::string path_twin;
            for(int idx = 0; idx < path_win_length; ++ idx ){
                if( path_win[idx] == ' ' || path_win[idx] == '\t' ) continue;
                path_twin = path_win.substr(idx);
                break;
            }

            // 実在パスがディレクトリパスならばエラー
            bool is_exist = false;
            if( !tryIsExistDirectory(&is_exist, path_twin) || is_exist  ){
                fprintf(stderr, "caution, directory path is not able to use as virtual path.\nline at %u => [ %s ]\n", line_num, line);
                ERR_FINALIZE
            }

            // 先頭にある空白とタブを読み飛ばす処理
            std::string path_tarc;
            int path_arc_length = length - index1 - index2;
            for(int idx = 0; idx < path_arc_length; ++ idx ){
                if( path_arc[idx] == ' ' || path_arc[idx] == '\t' ) continue;
                path_tarc = path_arc.substr(idx);
                break;
            }

            addArchiveFilePath_( path_twin, path_tarc, align );
        }
        // 仮想パスが存在しないとき
        else {
            // 仮想パスを空にしておく
            path_arc = "";

            // 先頭にある空白とタブを読み飛ばす処理
            std::string path_twin;
            int path_win_length = length - index1 - 1;
            for(int idx = 0; idx < path_win_length; ++ idx ){
                if( path_win[idx] == ' ' || path_win[idx] == '\t' ) continue;
                path_twin = path_win.substr(idx);
                break;
            }

            // ファイル部分を除いたパスを取得してワイルドカードが含まれていないか確認する
            std::string card;
            if( path_twin.find('/') != std::string::npos )    {
                // ↑カレントディレクトリ直下にあるファイルへの対策
                size_t rindex = path_twin.rfind('/');
                if( rindex < 1 || rindex == std::string::npos )
                {
                    fprintf(stderr, "caution, illegal path. not accept root directory.\nline at %u => [ %s ]\n", line_num, line);
                    ERR_FINALIZE
                }
                card = path_win.substr(0, rindex);
                if( card.find('*') != std::string::npos || card.find('?') != std::string::npos )
                {
                    fprintf(stderr, "caution, illegal path. not accept wildcard in a part of directory.\nline at %u => [ %s ]\n", line_num, line);
                    ERR_FINALIZE
                }
            }
            // ファイルパスを取得する
            std::string file;
            std::string path(path_twin);
            if( getFileName( file, path_twin ) ){
                if( file.find('*') != std::string::npos || file.find('?') != std::string::npos){
                    size_t file_path_length = file.size();
                    if( file_path_length != path_twin.size() ){
                        path.resize(std::max(ptrdiff_t(0), ptrdiff_t(path.size() - (file_path_length+1))));
                        addArchivePathWithExtension_( path, file, align );
                    }
                    else{
                        addArchivePathWithExtension_( mWorkDir, file, align );
                    }
                    continue;
                }
            }
            addArchivePath(path, align);
        }
    }

    fclose(fp);

#undef ERR_FINALIZE
}

//------------------------------------------------------------------------------

void ArchiveConfig::addArchivePath( const std::string& path, uint32_t align )
{
    if( path.size() > MAX_PATH )
    {
        fprintf(stderr, "too long path to add entry.\n[ %s ]\n", path.c_str());
        return;
    }
    std::string path_impl(path);
    if( path_impl.compare(0, 2, "./") == 0 )
    {
        path_impl = path.substr(2);
    }
    exchangeBackSlash2Slash(&path_impl);

    if(!path_impl.empty() && (path_impl.back() == '\\' || path_impl.back() == '/' ))
    {
        std::string dir_name(path_impl);
        dir_name.pop_back();
        // ディレクトリとして解釈する
        bool is_dir = false;
        if( tryIsExistDirectory(&is_dir, dir_name) && is_dir ){
            addArchiveDirPath_( dir_name, align );
        }
        // 存在しないパスなので何もしない
        else{
            OutLogMessage("not found. [ %s ]\n", path_impl.c_str());
        }
    }
    else{
        // ファイルとして認識
        bool is_exist = false;
        if( tryIsExistFile(&is_exist, path_impl) && is_exist ){
            addArchiveFilePath_( path_impl, "", align );
        }
        // ディレクトリとして認識
        else if( tryIsExistDirectory(&is_exist, path_impl) && is_exist ){
            addArchiveDirPath_( path_impl, align );
        }
        // 存在しないパスなので何もしない
        else{
            OutLogMessage("not found. [ %s ]\n", path_impl.c_str());
        }
    }
}

//------------------------------------------------------------------------------

bool IsBntx_(FILE* fp)
{
    fseek(fp, 0, SEEK_SET);
    char signature[4];
    fread(signature, sizeof(signature), 1, fp);

    return signature[0] == 'B' && signature[1] == 'N' && signature[2] == 'T' && signature[3] == 'X';
}

//-----------------------------------------------------------------------------
// 共通バイナリフォーマット のアライメントを読みます。
// PC ツールから、nn ライブラリを利用する場合、nact で依存関係が設定できない制限があるので
// nn::util::BinaryFileHeader を利用せず、自前で解釈してアライメントを読み取ります。
size_t ReadBinaryFileHeaderAlignment_(FILE* fp)
{
    //struct nn::util::BinaryFileHeader
    //{
    //	//! @brief ファイルの種類を表す 8 つの 1 バイト値です。
    //	BinFileSignature signature;

    //	//! @brief バージョンを表す 4 つの 1 バイト値です。
    //	BinVersion version;

    //	uint16_t _byteOrderMark;        // エンディアンを識別するバイトオーダーマークです。
    //	uint8_t _alignmentShift;        // ファイルを配置する際に揃える必要があるアライメントの 2 を底とする指数部です。
    //	uint8_t _targetAddressSize;     // リソースのアドレスサイズです。
    //	uint32_t _offsetToFileName;     // ファイルの先頭からファイル名へのオフセットです。
    //	uint16_t _flag;                 // フラグです。
    //	uint16_t _offsetToFirstBlock;   // ファイルの先頭から最初のデータブロックへのオフセットです。
    //	uint32_t _offsetToRelTable;     // ファイルの先頭からリロケーションテーブルへのオフセットです。
    //	uint32_t _fileSize;             // ファイルサイズです。
    //};

    fseek(fp, 8 + 4 + 2, SEEK_SET);
    uint8_t aligmentShift = 0;
    fread(&aligmentShift, sizeof(aligmentShift), 1, fp);

    return static_cast<size_t>(1 << aligmentShift);
}

//-----------------------------------------------------------------------------

uint32_t ArchiveConfig::ReadBflimAlignment_(const char* filePath)
{
    FILE* fp = nullptr;
    errno_t res_code = fopen_s(&fp, filePath, "rb");
    if( res_code !=0 ){
        OutLogMessage("not exist. [ %s ]\n", filePath);
        return 0;
    }

    // テクスチャ形式を調べる
    const bool isBNTX = IsBntx_(fp);
    if(isBNTX)
    {
        size_t aligment = ReadBinaryFileHeaderAlignment_(fp);
        fclose(fp);

        return static_cast<uint32_t>(aligment);
    }else{

        // ヘッダ位置を調べる
        // ※ファイルの最後尾４バイトにイメージサイズが入っています。
        fseek(fp, -4, SEEK_END);
        uint32_t imageSize = 0;
        fread(&imageSize, sizeof(uint32_t), 1, fp);
        imageSize = toHostU32(EndianTypes::cBig, imageSize);

        // テクスチャ必要アライメント情報を取得

        const uint32_t IDENTIFIER_SIZE = 4;

        //struct BinaryFileHeader
        //{
        //  union { uint8_t signature[4]; uint32_t sigWord; };
        //  union { uint8_t version[4]; uint32_t verWord; };
        //  uint16_t byteOrder;
        //  uint16_t headerSize;
        //  uint32_t fileSize;
        //};
        const uint32_t FILE_HEADER_SIZE = 16;

        //struct Image
        //{
        //  ut::BinaryBlockHeader blockHeader;
        //  ut::ResU16 width;
        //  ut::ResU16 height;
        //  ut::ResU16 neededAlignment;
        //  ut::ResU8 format;
        //  ut::ResU8 padding[1];
        //};
        const uint32_t BLOCK_HEADER_SIZE = 8;
        const uint32_t IMAGE_WIDTH_HEIGHT_SIZE = 4;

        const uint32_t ALIGNMENT_OFFSET = IDENTIFIER_SIZE + FILE_HEADER_SIZE + BLOCK_HEADER_SIZE + IMAGE_WIDTH_HEIGHT_SIZE;

        fseek(fp, imageSize + ALIGNMENT_OFFSET, SEEK_SET);
        uint16_t textureAlign = 0;
        fread(&textureAlign, sizeof(uint16_t), 1, fp);
        textureAlign = toHostU16(EndianTypes::cBig, textureAlign);
        fclose(fp);

        // BNTX ならシフト量が格納されています。
        return textureAlign;
    }
}

//-----------------------------------------------------------------------------
// Bnsh のアライメントを読みます。
// PC ツールから、nn ライブラリを利用する場合、nact で依存関係が設定できない制限があるので
// nn::util::BinaryFileHeader を利用せず、自前で解釈してアライメントを読み取ります。
size_t ArchiveConfig::ReadBnshAlignment_(const char* filePath)
{
    FILE* fp = nullptr;
    errno_t res_code = fopen_s(&fp, filePath, "rb");
    if( res_code !=0 ){
        OutLogMessage("not exist. [ %s ]\n", filePath);
        return 0;
    }

    size_t aligment = ReadBinaryFileHeaderAlignment_(fp);

    fclose(fp);

    return aligment;
}

//------------------------------------------------------------------------------

void ArchiveConfig::addArchiveFilePath_( const std::string& path_win, const std::string& path_arc, uint32_t align )
{
    if( path_win.size() > MAX_PATH ||
        path_arc.size() > MAX_PATH )
    {
        fprintf(stderr, "too long path to add entry.\nreal    :[ %s ]\nvirtual : [ %s ]\n", path_win.c_str(), path_arc.c_str());
        mHasError = true;
        return;
    }

    // 先頭が作業ディレクトリと一致するのであれば、切り取って相対パスにする
    std::string path_impl;
    // ./から始まるときはそれを削除したパスにする
    if( path_win.compare(0, 2, "./") == 0 ){
        path_impl = path_win.substr(2);
    }
    else{
        path_impl = path_win;
    }
    // 念のためスラッシュ区切りのパスであることを保証する
    exchangeBackSlash2Slash(&path_impl);

    // 実在パスがNGディレクトリ名を含んでいたらキャンセル
    if( isIncludeExceptDirName_(path_impl) ){
        return;
    }

    // 絶対パスで指定されているかを確認
    int work_length = mWorkDir.size();
    if( path_impl[1] == ':' &&  path_impl[2] == '/' ){
        // ワーキングディレクトリ以下にあるディレクトリか判断
        if( path_impl.compare(0, work_length, mWorkDir) == 0 ){
            std::string path_tmp(path_impl);
            path_impl = path_tmp.substr( work_length );
        }
        else{
            fprintf(stderr, "incorrect absolute path.\ndirectory : [ %s ]\n  current : [ %s ] ", mWorkDir.c_str(), path_impl.c_str());
            mHasError = true;   // ワーキングディレクトリの管理外のディレクトリはエラー
            return;
        }
    }

    std::string temp_arc;
    if( path_arc.empty() ){
        temp_arc = path_impl;
    }
    else{
        temp_arc = path_arc;
        // 仮想パスが存在するとき、実在パスが不正ならば注意して登録をキャンセルする
        bool is_file = false;
        if( !tryIsExistFile( &is_file, path_impl ) || !is_file ){
            fprintf(stderr, "not found. [ %s ]\n", path_impl.c_str());
            return;
        }
    }

    // ./から始まるときはそれを削除したパスにする
    if( temp_arc.compare(0, 2, "./") == 0 ){
        temp_arc = temp_arc.substr(2);
    }

    // 同一のパスが既に登録されていないか確認
    const PathList& list = mPathList;
    PathList::const_iterator end = list.end();
    for( PathList::const_iterator itr = list.begin(); itr != end; ++itr )
    {
        const std::string& win = itr->path_win;  // 実在パス
        const std::string& arc = itr->path_arc;  // 仮想パス

        // 重複する仮想パスがある
        if( temp_arc == arc ){
            // 異なる実在パスに同名の仮想パスを付与するのは許容できないのでエラー
            if( ! (path_impl == win) ){
                fprintf(stderr, "caution, virtual path is already assigned.\nconfirm : [ %s ]\n", arc.c_str());
                mHasError = true;
                return;
            }
            // 登録しようとしているエントリの実在パスと仮想パスが一致すれば、
            // 単純な上書きの試行なので、最初に登録した方を優先して新規登録をキャンセル
            else{
                return;
            }
        }
    }

    // TODO: find (旧 include) でいいのか？
    // フォントとテクスチャとシェーダのアライメント
    if (path_impl.find(".bffnt") != std::string::npos)
    {
        // TODO: ほかのリソースと同じように、ファイルから必要なアライメントを読み取るように修正する。
        align = 1024 * 4;
    }
    else if (path_impl.find(".bflim") != std::string::npos || path_impl.find(".bntx") != std::string::npos)
    {
        align = ReadBflimAlignment_(path_impl.c_str());
        if(align == 0)
        {
            mHasError = true;
        }
    }
    else if (path_impl.find(".bnsh") != std::string::npos)
    {
        align = static_cast<uint32_t>(ReadBnshAlignment_(path_impl.c_str()));
    }

    // 未登録だったので新規登録
    mPathList.push_back(PathEntry());
    PathEntry* entry = &mPathList.back();
    entry->align = align;
    if (mMaxAlignment < align) { mMaxAlignment = align; }

    entry->path_win = path_impl;
    entry->path_arc = temp_arc;
}

//------------------------------------------------------------------------------

void ArchiveConfig::addArchiveDirPath_( const std::string& path, uint32_t align )
{
    // Win32 API でディレクトリを参照する
    WIN32_FIND_DATAA find_file_data;
    HANDLE hFind;

    if( path.size() > MAX_PATH )
    {
        fprintf(stderr, "too long path to add entry.\n[ %s ]\n", path.c_str());
        mHasError = true;
        return;
    }

    std::string find_path(path);
    std::string file_path;
    std::string sub_dir;

    exchangeBackSlash2Slash(&find_path);
    if( find_path.compare(0, 2, "./") == 0 ){
        find_path = path.substr(2);
    }
    find_path.append("/*.*");

    //  サブディレクトリとファイルの検索
    hFind = FindFirstFileA( find_path.c_str(), &find_file_data );
    if( hFind != INVALID_HANDLE_VALUE )
    {
        do{
            // エントリがディレクトリのとき
            if( ( find_file_data.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY ) )
            {
                //  親ディレクトリと自分自身は無視する
                std::string name = find_file_data.cFileName;
                if( !(name == "..") && !(name == ".") )
                {
                    sub_dir = path + "/" + name;
                    addArchiveDirPath_(sub_dir, align); //  サブディレクトリの中を検索
                }
            }
            // エントリがファイルのとき
            else{
                file_path = path + "/" + find_file_data.cFileName;
                exchangeBackSlash2Slash( &file_path );
                addArchiveFilePath_( file_path, "", align );
            }
        }while( FindNextFileA( hFind, &find_file_data) );
    }
    FindClose(hFind);
}

//------------------------------------------------------------------------------

void ArchiveConfig::addArchivePathWithExtension_( const std::string& path, const std::string& card, uint32_t align )
{
    if( path.size() > MAX_PATH )
    {
        fprintf(stderr, "%s is too long path to add entry.\n", path.c_str());
        mHasError = true;
        return;
    }
    // Win32 API でディレクトリを参照する
    WIN32_FIND_DATAA find_file_data;
    HANDLE hFind;

    std::string find_path(path);
    std::string file_path;
    std::string sub_dir;

    exchangeBackSlash2Slash(&find_path);
    if( find_path.compare(0, 2, "./") == 0 ){
        find_path = path.substr(2);
    }

    if( !find_path.empty() && find_path.back() == '/' ){
        find_path.append("*.*");
    } else {
        find_path.append("/*.*");
    }

    //  サブディレクトリの検索
    hFind = FindFirstFileA( find_path.c_str(), &find_file_data );
    if( hFind != INVALID_HANDLE_VALUE )
    {
        do{
            // エントリがディレクトリのとき
            if( ( find_file_data.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY ) )
            {
                //  親ディレクトリと自分自身は無視する
                std::string name = find_file_data.cFileName;
                if( !(name == "..") && !(name == ".") )
                {
                    if( !path.empty() && path.back() == '/' ){
                        sub_dir = path + name;
                    } else {
                        sub_dir = path + "/" + name;
                    }
                    addArchivePathWithExtension_(sub_dir, card, align); //  サブディレクトリの中を検索
                }
            }
        }while( FindNextFileA( hFind, &find_file_data) );
    }
    FindClose(hFind);

    //  ファイルの検索
    find_path = path + "/" + card;
    exchangeBackSlash2Slash(&find_path);
    hFind = FindFirstFileA( find_path.c_str(), &find_file_data );
    if( hFind != INVALID_HANDLE_VALUE )
    {
        do{
            // エントリがファイルのとき
            if( ! ( find_file_data.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY ) )
            {
                if( !path.empty() && path.back() == '/' ){
                    file_path = path + find_file_data.cFileName;
                } else {
                    file_path = path + "/" + find_file_data.cFileName;
                }
                exchangeBackSlash2Slash(&file_path);
                addArchiveFilePath_( file_path, "", align );
            }
        }while( FindNextFileA( hFind, &find_file_data) );
    }
    FindClose(hFind);
}

//------------------------------------------------------------------------------

bool ArchiveConfig::isIncludeExceptDirName_(const std::string& path)
{
    // 除外リストが空
    if( mExceptList.empty() )   {
        return false;
    }

    std::string path_temp(path);
    std::string temp;

    ExceptList::iterator itr = mExceptList.begin();
    ExceptList::iterator end = mExceptList.end();

    for( ; itr != end; ++ itr ) {
        std::string& keyword = (itr)->path;
        int key_length = keyword.size();

        // 元のファイルパスをコピー
        temp = path_temp;

        // パス後方から探す
        size_t index = temp.rfind( keyword );
        while( index != std::string::npos ){
            // 範囲アサートを迂回するため
            const char* const temp_top = temp.c_str();
            // パスの一部にNGディレクトリ名と同じ部分を発見した
            if( index == 0 ){
                // 先頭がそのまま一致 => 直後が区切り文字になっているかを確認する
                if( temp_top[key_length] == '/' ){
                    return true;
                }
            }
            else {
                // パスの途中で一致したので, 前後が区切り文字になっているかを確認する
                if( temp_top[index+key_length] == '/' &&
                    temp_top[index-1] == '/' ){
                    return true;
                }
            }
            // まだ途中に潜んでいるかもしれないので確認済みの末尾を切り落とす
            temp.resize(index);

            // 再度パス後方から探す
            index = temp.rfind( keyword );
        }
    }

    return false;
}

//------------------------------------------------------------------------------

uint32_t ArchiveConfig::checkAlign_(int32_t align)
{
    // 負や4未満のアライメントはありえない
    if( align >= 4 ){
        uint32_t alignment = static_cast<uint32_t>(align);

        // 2の階乗はビット列でみるとビットが1つだけ立っていること
        // ctr用に192バイトの例外を考慮する
        if( alignment == 192 || (  ( alignment & ( alignment - 1 ) ) == 0 ) ){
            mAlignment = alignment;
            if( mMaxAlignment < alignment ){
                mMaxAlignment = alignment;
            }
        }
    }
    return mAlignment;
}


//------------------------------------------------------------------------------

void ArchiveConfig::OutLogMessage(char* pMsg, ...)
{
    if(getLogSilent())
    {
        return;
    }

    va_list va;
    va_start(va, pMsg);

    printf(pMsg, va);

    va_end(va);
}

}
