﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "sharcArchive.h"
#include "sharcArchiveConfig.h"
#include "sharcArchiveBuilder.h"
#include "sharcDebug.h"
namespace sharc {
//------------------------------------------------------------------------------

ArchiveBuilder::ArchiveBuilder()
: mEntryNum( 0 )
, mEntryList()
, mEntryItems(nullptr)
{
}

//------------------------------------------------------------------------------

ArchiveBuilder::~ArchiveBuilder()
{
    if (mEntryItems != nullptr)
    {
        delete[] mEntryItems;
    }
}

//------------------------------------------------------------------------------

void ArchiveBuilder::initialize(uint32_t entry_num )
{
    mEntryNum = entry_num;
    mEntryItems = new Entry[entry_num];
    mEntryList.reserve(entry_num);
}

//------------------------------------------------------------------------------

void ArchiveBuilder::addEntry( const std::string &path_win, const std::string &path_arc, uint32_t align )
{
    Entry* entry = mEntryItems + mEntryList.size();
    entry->path_win = path_win;
    {
        // path_arcについては、./で始まっていたらそれを取り除く
        std::string path;
        if (path_arc.compare(0, 2, "./") == 0){
            path = path_arc.substr(2);
        }
        else{
            path = path_arc;
        }
        entry->path_arc = path;
    }
    entry->align = align;
    mEntryList.push_back(entry);
}

//------------------------------------------------------------------------------
bool ArchiveBuilder::build( FILE* dst, const ArchiveConfig& config, uint32_t key )
{
    if ( !dst )
    {
        NW_ERR( "dst must not be nullptr." );
        fprintf(stderr, "----\nfailed.\narchive was not builded. because not found file handle. \n");
        return false;
    }
    // ターゲットエンディアンの取得
    EndianTypes endian = config.getEndianType();

    // ハッシュ生成.
    const std::string& workdir = config.getWorkDirString();
    calcHash_(key);

    // データの書き込み.
    // アーカイブヘッダ.
    ArchiveBlockHeader abh;
    memcpy( abh.signature, "SARC", 4 );
    abh.header_size = fromHostU16( endian, sizeof( ArchiveBlockHeader ) );
    abh.byte_order = endianToMark( endian );
    abh.file_size = 0;          // 仮.
    abh.data_block_offset = 0;  // 仮.
    abh.version = fromHostU16( endian, SHARC_ARCHIVE_VERSION );
    abh.reserved = 0;
    fseek( dst, sizeof( ArchiveBlockHeader ), SEEK_CUR );
    // FATブロックヘッダ.
    FATBlockHeader fatbh;
    memcpy( fatbh.signature, "SFAT", 4 );
    fatbh.header_size = fromHostU16( endian, sizeof( FATBlockHeader ) );
    fatbh.file_num = fromHostU16( endian, static_cast<uint16_t>(mEntryList.size()) );
    fatbh.hash_key = fromHostU32( endian, key );
    fwrite( &fatbh, sizeof( FATBlockHeader ), 1, dst );
    // FATブロック.
    // スペース空けておく.
    fseek( dst, sizeof( FATEntry ) * mEntryList.size(), SEEK_CUR );
    // FNTブロック.
    FNTBlockHeader fntbh;
    memcpy( fntbh.signature, "SFNT", 4 );
    fntbh.header_size = fromHostU16( endian, sizeof( FNTBlockHeader ) );
    fntbh.reserved = 0;
    fwrite( &fntbh, sizeof( FNTBlockHeader ), 1, dst);
    // 名前の書き込み.
    {
        uint32_t offset = 0;
        uint32_t old_offset = offset;
        EntryList::iterator end = mEntryList.end();
        for( EntryList::iterator it = mEntryList.begin(); it != end; ++it )
        {
            // unsafe オプションが有効であればハッシュが重複した場合だけ書き出す
            if( config.getOption() & cArchiveOption_Unsafe ){
                if ( (*it)->same_id == 0 )
                {
                    (*it)->name_offset = 0;
                    continue;
                }
            }
            else{
                // べリファイモード(not unsafe)では全て1つ重複している扱いにして、パスを全て保持する
                if( (*it)->same_id == 0 ){
                    (*it)->same_id = 1;
                }
            }

            old_offset = offset;
            // 長さ.
            uint32_t length = (*it)->path_arc.size();
            if ( length >= MAX_PATH )
            {
                // 長すぎる.
                NW_ERR( "path too long(%s).", (*it)->path_arc.c_str() );
                fprintf(stderr, "----\nfailed.\narchive building was not complete. because path too long.\n%s\n", (*it)->path_arc.c_str());
                return false;
            }
            uint32_t padding = ( ( SHARC_FNT_ALIGN - ( offset % SHARC_FNT_ALIGN ) ) % SHARC_FNT_ALIGN );
            if ( padding > 0 )
            {
                uint32_t dummy = 0;
                fwrite( &dummy, padding, 1, dst );
            }
            offset += padding;
            if ( offset / SHARC_FNT_ALIGN > 0x00ffffff )
            {
                NW_ERR( "total path length too long." );
                fprintf(stderr, "----\nfailed.\narchive building was not complete. because total path too long.\n");
                return false;
            }
            fwrite( (*it)->path_arc.c_str(), length, 1, dst );
            uint8_t value = 0;
            fwrite( &value, 1, 1, dst );
            (*it)->name_offset = ( (*it)->same_id << 24 ) | ( ( offset / SHARC_FNT_ALIGN ) & 0x00ffffff );
            offset += length + 1;
            if ( old_offset >= offset )
            {
                // オーバーフロー.
                NW_ERR( "total path length too long." );
                fprintf(stderr, "----\nfailed.\narchive building was not complete. because total path too long.\n");
                return false;
            }
        }
    }
    // データの書き込み.
    uint32_t data_total_size = 0;
    uint32_t data_offset = static_cast<uint32_t>(ftell(dst));
    // 192バイトのアライメントもあるので２の階乗にはできない
    data_offset = roundUp( data_offset, config.getMaxAlignment() );
    fseek( dst, data_offset, SEEK_SET );
    {
        uint32_t old_data_total_size = 0;
        EntryList::iterator end = mEntryList.end();
        std::string path_win;
        for( EntryList::iterator it = mEntryList.begin(); it != end; ++it )
        {
            path_win = workdir + (*it)->path_win;
            old_data_total_size = data_total_size;
            data_total_size = roundUp( data_total_size, (*it)->align );
            if( old_data_total_size > data_total_size )
            {
                NW_ERR( "total data size too long." );
                fprintf(stderr, "----\nfailed.\narchive building was not complete. because total data size too long.\n");
                return false;
            }
            (*it)->data_offset_start = data_total_size;

            old_data_total_size = data_offset + data_total_size;
            if( old_data_total_size < data_total_size )
            {
                NW_ERR( "total data size too long." );
                fprintf(stderr, "----\nfailed.\narchive building was not complete. because total data size too long.\n");
                return false;
            }
            fseek( dst, data_offset + data_total_size, SEEK_SET );
            FILE* handle;
            errno_t res_code = fopen_s( &handle, path_win.c_str(), "rb" );
            if (res_code != 0)
            {
                fprintf(stderr, "");
            }
            uint8_t buffer[ 1024 ];
            uint32_t size = 0;
            old_data_total_size = data_total_size;
            for (; ; )
            {
                size = static_cast<uint32_t>(fread( buffer, 1, 1024, handle ));
                if (ferror(handle))
                {
                    NW_ERR("read error.");
                    fprintf(stderr, "----\nfailed.\nread error.\n");
                    return false;
                }
                if (size <= 0)
                {
                    break;
                }
                fwrite( buffer, size, 1, dst );
                data_total_size += size;
            }
            if ( old_data_total_size > data_total_size )
            {
                NW_ERR( "total data size too long." );
                fprintf(stderr, "----\nfailed.\narchive building was not complete. because total data size too long.\n");
                fclose(handle);
                return false;
            }
            (*it)->data_offset_end = data_total_size;
            fclose(handle);
        }
    }
    // 更新したヘッダを書き込み.
    abh.file_size = fromHostU32( endian, (data_offset + data_total_size) );
    abh.data_block_offset = fromHostU32( endian, data_offset );
    fseek( dst, 0, SEEK_SET );
    fwrite( &abh, sizeof( ArchiveBlockHeader ), 1, dst );
    // 更新したFATを書き込み.
    fseek( dst, sizeof( FATBlockHeader ), SEEK_CUR );
    {
        EntryList::iterator end = mEntryList.end();
        for( EntryList::iterator it = mEntryList.begin(); it != end; ++it )
        {
            FATEntry entry;
            entry.hash = fromHostU32( endian, (*it)->hash );
            entry.name_offset = fromHostU32( endian, (*it)->name_offset );
            entry.data_start_offset = fromHostU32( endian, (*it)->data_offset_start );
            entry.data_end_offset = fromHostU32( endian, (*it)->data_offset_end );
            fwrite( &entry, sizeof( FATEntry ), 1, dst );
        }
    }

    if( config.isDisplayFileInfo() && !config.getLogSilent() ){
        displayFileList_();
    }

    return true;
}

//------------------------------------------------------------------------------

void ArchiveBuilder::displayFileList_()
{
    EntryList::iterator end = mEntryList.end();
    printf("hash       align    size       name(real | virtual) \n");
    for( EntryList::iterator it = mEntryList.begin(); it != end; ++it )
    {
        uint32_t size = (*it)->data_offset_end - (*it)->data_offset_start;
        printf("[%8x] [%6d] [%8d] %s | %s\n", (*it)->hash, (*it)->align, size, (*it)->path_win.c_str(), (*it)->path_arc.c_str());
    }
    printf("%d files.\n", mEntryList.size());
}

//------------------------------------------------------------------------------

void ArchiveBuilder::calcHash_(uint32_t key)
{
    bool clashed = false;
    // 「アーカイブ内での仮想パス」でハッシュを計算します
    {
        EntryList::iterator end = mEntryList.end();
        for ( EntryList::iterator it = mEntryList.begin(); it != end; ++it)
        {
            (*it)->hash = calcHash32((*it)->path_arc, key);
        }

        // mEntryList をヒープソートする
        {
            int i, k, t;
            int n = mEntryList.size();
            Entry* x;
            Entry** ptrs = &mEntryList.front();

            for (i = n / 2; i >= 1; i--) {
                t = i;
                x = ptrs[t - 1];
                while ((k = 2 * t) <= n) {
                    if (k < n && ArchiveBuilder::entryCompare_(ptrs[k - 1], ptrs[k]) < 0) {
                        k++;
                    }
                    if (ArchiveBuilder::entryCompare_(x, ptrs[k - 1]) >= 0) {
                        break;
                    }
                    ptrs[t - 1] = ptrs[k - 1];
                    t = k;
                }
                ptrs[t - 1] = x;
            }
            while (n > 1) {
                x = ptrs[n - 1];
                ptrs[n - 1] = ptrs[0];
                n--;
                t = 1;
                while ((k = 2 * t) <= n) {
                    if (k < n && ArchiveBuilder::entryCompare_(ptrs[k - 1], ptrs[k]) < 0) {
                        k++;
                    }
                    if (ArchiveBuilder::entryCompare_(x, ptrs[k - 1]) >= 0) {
                        break;
                    }
                    ptrs[t - 1] = ptrs[k - 1];
                    t = k;
                }
                ptrs[t - 1] = x;
            }
        }
    }

    int clash = 0;
    // 同じハッシュのエントリーが無い場合、フラグを立てておく.
    {
        EntryList::iterator end = mEntryList.end();
        uint32_t current_hash = 0;
        uint32_t same_hash_num = 0;
        for( EntryList::iterator it = mEntryList.begin(); it != end; ++it )
        {
            EntryList::iterator next = it;
            ++next;
            if ( current_hash != (*it)->hash && ((next == end) || (*it)->hash != (*next)->hash) )
            {
                same_hash_num = 0;
            }
            else
            {
                clashed = true;
                fprintf(stderr,  "%s\n", (*it)->path_arc.c_str() );
                same_hash_num++;
                clash ++;
            }
            (*it)->same_id = same_hash_num;
            current_hash = (*it)->hash;
        }
    }
    if( clashed ){
        fprintf(stderr, "Hash clash occured. above %d files, ( hash key : %u )\n", clash, key);
    }
}

//------------------------------------------------------------------------------

int ArchiveBuilder::entryCompare_( const Entry* lhs, const Entry* rhs )
{
    if ( lhs->hash < rhs->hash )
    {
        return -1;
    }
    else if ( lhs->hash > rhs->hash )
    {
        return 1;
    }
    else
    {
        return 0;
    }
}

//------------------------------------------------------------------------------
} // namespace sherbet
