﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows.Forms;
using System.Xml.Serialization;

namespace FcpxMaker.Xml
{
    using DataModel;
    using static Type;

    using FcpxComplexFontDescription = NW4F.LayoutBinaryConverter.Schema.Fcpx.ComplexFontDescription;
    using FcpxSubFont = NW4F.LayoutBinaryConverter.Schema.Fcpx.SubFont;
    using FcpxCharCodeRange = NW4F.LayoutBinaryConverter.Schema.Fcpx.CharCodeRange;
    using FcpxScalableFontDescription = NW4F.LayoutBinaryConverter.Schema.Fcpx.ScalableFontDescription;
    using FcpxBitmapFont = NW4F.LayoutBinaryConverter.Schema.Fcpx.BitmapFont;
    using FcpxScalableFont = NW4F.LayoutBinaryConverter.Schema.Fcpx.ScalableFont;
    using FcpxMultiScalableFont = NW4F.LayoutBinaryConverter.Schema.Fcpx.MultiScalableFont;
    using FcpxPairFont = NW4F.LayoutBinaryConverter.Schema.Fcpx.PairFont;

    /// <summary>
    /// XMLとデータモデルの変換を行うクラスです。
    /// </summary>
    public class XmlConverter
    {
        #region 入力

        /// <summary>
        /// XMLをデータモデルに変換します。
        /// </summary>
        static public DataModelBase ReadXml(string filePath)
        {
            // デシリアライズ
            StreamReader reader = null;
            FcpxComplexFontDescription cfd = null;
            DataModelBase dataModel = null;
            try
            {
                XmlSerializer serializer = new XmlSerializer(typeof(FcpxComplexFontDescription));
                Encoding enc = new UTF8Encoding(false); // UTF-8(BOMなし)
                reader = new StreamReader(filePath, enc);
                cfd = serializer.Deserialize(reader) as FcpxComplexFontDescription;

                // データモデルにセット
                dataModel = ReadComplexFontDescription(cfd);
            }
            catch (Exception e)
            {
                MessageBox.Show(e.Message, Properties.Resources.IO_FCPX_READ_ERROR);
                dataModel = null;
            }
            finally
            {
                if (reader != null)
                {
                    reader.Close();
                }
            }

            return dataModel;
        }

        /// <summary>
        /// ComplexFontDescriptionを読み込みます。
        /// </summary>
        static private DataModelBase ReadComplexFontDescription(FcpxComplexFontDescription input)
        {
            if (input == null) return null;

            RootDataModel ret = new RootDataModel();
            ret.Version = input.version;
            ret.SubFont = ReadSubFontSerializeData(input.subFont?.Item);

            return ret;
        }

        /// <summary>
        /// CharCodeRangeを読み込みます。
        /// </summary>
        static private CharCodeRange ReadCharCodeRangeSerializeData(FcpxCharCodeRange input)
        {
            if (input == null) return null;

            CharCodeRange ret = new CharCodeRange();
            ret.First = input.first;
            ret.Last = input.last;

            return ret;
        }

        /// <summary>
        /// ScalableFontDescriptionを読み込みます。
        /// </summary>
        static private DataModelBase ReadScalableFontDescriptionSerializeData(FcpxScalableFontDescription input)
        {
            if (input == null) return null;

            ScalableFontDescriptionDataModel ret = new ScalableFontDescriptionDataModel();
            ret.FontDataModel = ReadFontSerializeData(input.path, FontFileType.Scalable);
            ret.CharCodeRangeDataModel = SetCharCodeRangeSet(input.charCodeRangeSet);
            ret.BoldWeight = input.boldWeight;
            ret.BorderWidth = input.borderWidth;
            ret.TtcFontIndex = input.ttcFontIndex;
            ret.ScaleWidth = input.scaleWidth;
            ret.ScaleHeight = input.scaleHeight;
            ret.IgnorePalt = input.ignorePalt;
            ret.DeleteBearingX = input.deleteBearingX;
            ret.BearingOffsetX = input.bearingOffsetX;
            ret.ForceMonospacedEnabled = input.forceMonospacedEnabled;
            ret.ForceMonospacedWidth = input.forceMonospacedWidth;
            ret.BaseLineOffset = input.baselineOffset;

            return ret;
        }

        /// <summary>
        /// BitmapFontを読み込みます。
        /// </summary>
        static private DataModelBase ReadBitmapFontSerializeData(FcpxBitmapFont input)
        {
            if (input == null) return null;

            BitmapFontDataModel ret = new BitmapFontDataModel();
            ret.FontDataModel = ReadFontSerializeData(input.path, FontFileType.Bitmap);
            ret.CharCodeRangeDataModel = SetCharCodeRangeSet(input.charCodeRangeSet);

            return ret;
        }

        /// <summary>
        /// ScalableFontを読み込みます。
        /// </summary>
        static private DataModelBase ReadScalableFontSerializeData(FcpxScalableFont input)
        {
            if (input == null) return null;

            ScalableFontDataModel ret = new ScalableFontDataModel();
            ret.FontSize = input.size;
            ret.AlternateChar = input.alternateChar;
            ret.LineFeedOffset = input.lineFeedOffset;
            ret.ScalableFontDescription = ReadScalableFontDescriptionSerializeData(input.scalableFontDescription);

            return ret;
        }

        /// <summary>
        /// MultiScalableFontを読み込みます。
        /// </summary>
        static private DataModelBase ReadMultiScalableFontSerializeData(FcpxMultiScalableFont input)
        {
            if (input == null) return null;

            MultiScalableFontDataModel ret = new MultiScalableFontDataModel();
            ret.FontSize = input.size;
            ret.AlternateChar = input.alternateChar;
            ret.LineFeedOffset = input.lineFeedOffset;
            ret.ScalableFontDescriptionSet.AddRange(SetScalableFontDescriptionSet(input.scalableFontDescriptionSet));

            return ret;
        }

        /// <summary>
        /// PairFontを読み込みます。
        /// </summary>
        static private DataModelBase ReadPairFontSerializeData(FcpxPairFont input)
        {
            if (input == null) return null;

            PairFontDataModel ret = new PairFontDataModel();
            ret.FirstFont = ReadSubFontSerializeData(input.firstFont?.Item);
            ret.SecondFont = ReadSubFontSerializeData(input.secondFont?.Item);

            return ret;
        }

        /// <summary>
        /// Fontを読み込みます。
        /// </summary>
        static private DataModelBase ReadFontSerializeData(string path, FontFileType type)
        {
            // フォントの指定が無い場合もデータモデルを生成する
            //if (String.IsNullOrEmpty(path)) return null;

            FontDataModel ret = new FontDataModel();
            ret.AbsoluteFontFilePath = StringUtility.GetAbsolutePath(path, FileIOManager.FcpxOpenFilePath);
            ret.FontType = type;

            return ret;
        }

        /// <summary>
        /// SubFontを読み込みます。
        /// </summary>
        static private DataModelBase ReadSubFontSerializeData(object input)
        {
            if (input == null) return null;

            if (input is FcpxPairFont)
            {
                return ReadPairFontSerializeData(input as FcpxPairFont);
            }
            else if (input is FcpxBitmapFont)
            {
                return ReadBitmapFontSerializeData(input as FcpxBitmapFont);
            }
            else if (input is FcpxScalableFont)
            {
                return ReadScalableFontSerializeData(input as FcpxScalableFont);
            }
            else if (input is FcpxMultiScalableFont)
            {
                return ReadMultiScalableFontSerializeData(input as FcpxMultiScalableFont);
            }

            return null;
        }

        /// <summary>
        /// CharCodeRangeSetを読み込みます。
        /// </summary>
        static DataModelBase SetCharCodeRangeSet(IEnumerable<FcpxCharCodeRange> ccrSet)
        {
            CharCodeRangeDataModel ret = new CharCodeRangeDataModel();
            for (int i = 0; i < ccrSet.Count(); i++)
            {
                var target = ret.GetFontRange(i);
                var input = ccrSet.ElementAt(i);
                target.First = input.first;
                target.Last = input.last;
                target.Checked = true;
            }

            return ret;
        }

        /// <summary>
        /// ScalableFontDescriptionSetを読み込みます。
        /// </summary>
        static IEnumerable<DataModelBase> SetScalableFontDescriptionSet(IEnumerable<FcpxScalableFontDescription> sfdSet)
        {
            List<DataModelBase> ret = new List<DataModelBase>();
            foreach (var sfd in sfdSet)
            {
                var data = ReadScalableFontDescriptionSerializeData(sfd);
                if (data != null)
                {
                    ret.Add(data);
                }
            }

            return ret;
        }

        #endregion

        #region 出力

        /// <summary>
        /// ノードツリーをXMLに変換し、出力します。
        /// </summary>
        static public void WriteXml(string filePath, DataModelBase dataModel)
        {
            // スキーマ生成
            FcpxComplexFontDescription complexFontDesc = WriteComplexFontDescription(dataModel);

            // ファイルに出力
            StreamWriter writer = null;
            try
            {
                XmlSerializer serializer = new XmlSerializer(typeof(FcpxComplexFontDescription));
                Encoding enc = new UTF8Encoding(false); // UTF-8(BOMなし)
                writer = new StreamWriter(filePath, false, enc);
                serializer.Serialize(writer, complexFontDesc);
            }
            catch (Exception e)
            {
                MessageBox.Show(e.Message, Properties.Resources.IO_FCPX_WRITE_ERROR);
            }
            finally
            {
                if (writer != null)
                {
                    writer.Close();
                }
            }
        }

        /// <summary>
        /// ComplexFontDescriptionを生成します。
        /// </summary>
        static private FcpxComplexFontDescription WriteComplexFontDescription(DataModelBase dataModel)
        {
            RootDataModel dm = dataModel as RootDataModel;
            if (dm == null) return null;

            FcpxComplexFontDescription ret = new FcpxComplexFontDescription();
            ret.subFont = WriteSubFontSerializeData(dm.SubFont);

            return ret;
        }

        /// <summary>
        /// CharCodeRangeを生成します。
        /// </summary>
        static private FcpxCharCodeRange WriteCharCodeRangeSerializeData(CharCodeRange input)
        {
            FcpxCharCodeRange ret = new FcpxCharCodeRange();
            ret.first = input.First;
            ret.last = input.Last;

            return ret;
        }

        /// <summary>
        /// ScalableFontDescriptionを生成します。
        /// </summary>
        static private FcpxScalableFontDescription WriteScalableFontDescriptionSerializeData(DataModelBase dataModel)
        {
            ScalableFontDescriptionDataModel dm = dataModel as ScalableFontDescriptionDataModel;
            if (dm == null) return null;

            FcpxScalableFontDescription ret = new FcpxScalableFontDescription();
            ret.path = dm.FontFilePath;
            ret.charCodeRangeSet = GetCharCodeRangeSet((dm.CharCodeRangeDataModel as CharCodeRangeDataModel).CharCodeRangeList).ToArray();
            ret.boldWeight = dm.BoldWeight;
            ret.borderWidth = dm.BorderWidth;
            ret.ttcFontIndex = dm.TtcFontIndex;
            ret.scaleWidth = dm.ScaleWidth;
            ret.scaleHeight = dm.ScaleHeight;
            ret.ignorePalt = dm.IgnorePalt;
            ret.deleteBearingX = dm.DeleteBearingX;
            ret.bearingOffsetX = dm.BearingOffsetX;
            ret.forceMonospacedEnabled = dm.ForceMonospacedEnabled;
            ret.forceMonospacedWidth = dm.ForceMonospacedWidth;
            ret.baselineOffset = dm.BaseLineOffset;

            return ret;
        }

        /// <summary>
        /// BitmapFontを生成します。
        /// </summary>
        static private FcpxBitmapFont WriteBitmapFontSerializeData(DataModelBase dataModel)
        {
            BitmapFontDataModel dm = dataModel as BitmapFontDataModel;
            if (dm == null) return null;

            FcpxBitmapFont ret = new FcpxBitmapFont();
            ret.path = dm.FontFilePath;
            ret.charCodeRangeSet = GetCharCodeRangeSet((dm.CharCodeRangeDataModel as CharCodeRangeDataModel).CharCodeRangeList).ToArray();

            return ret;
        }

        /// <summary>
        /// ScalableFontを生成します。
        /// </summary>
        static private FcpxScalableFont WriteScalableFontSerializeData(DataModelBase dataModel)
        {
            ScalableFontDataModel dm = dataModel as ScalableFontDataModel;
            if (dm == null) return null;

            FcpxScalableFont ret = new FcpxScalableFont();
            ret.scalableFontDescription = WriteScalableFontDescriptionSerializeData(dm.ScalableFontDescription);
            ret.size = dm.FontSize;
            ret.alternateChar = dm.AlternateChar;
            ret.lineFeedOffset = dm.LineFeedOffset;

            return ret;
        }

        /// <summary>
        /// MultiScalableFontを生成します。
        /// </summary>
        static private FcpxMultiScalableFont WriteMultiScalableFontSerializeData(DataModelBase dataModel)
        {
            MultiScalableFontDataModel dm = dataModel as MultiScalableFontDataModel;
            if (dm == null) return null;

            FcpxMultiScalableFont ret = new FcpxMultiScalableFont();
            ret.scalableFontDescriptionSet = GetScalableFontDescriptionSet(dm.ScalableFontDescriptionSet).ToArray();
            ret.size = dm.FontSize;
            ret.alternateChar = dm.AlternateChar;
            ret.lineFeedOffset = dm.LineFeedOffset;

            return ret;
        }

        /// <summary>
        /// PairFontを生成します。
        /// </summary>
        static private FcpxPairFont WritePairFontSerializeData(DataModelBase dataModel)
        {
            PairFontDataModel dm = dataModel as PairFontDataModel;
            if (dm == null) return null;

            FcpxPairFont ret = new FcpxPairFont();
            ret.firstFont = WriteSubFontSerializeData(dm.FirstFont);
            ret.secondFont = WriteSubFontSerializeData(dm.SecondFont);

            return ret;
        }

        /// <summary>
        /// SubFontを生成します。
        /// </summary>
        static private FcpxSubFont WriteSubFontSerializeData(DataModelBase dataModel)
        {
            if (dataModel == null) return null;

            FcpxSubFont ret = new FcpxSubFont();
            switch (dataModel.DataType)
            {
                case DataType.PairFont:
                    {
                        PairFontDataModel dm = dataModel as PairFontDataModel;
                        ret.Item = WritePairFontSerializeData(dm);
                    }
                    break;
                case DataType.BitmapFont:
                    {
                        BitmapFontDataModel dm = dataModel as BitmapFontDataModel;
                        ret.Item = WriteBitmapFontSerializeData(dm);
                    }
                    break;
                case DataType.ScalableFont:
                    {
                        ScalableFontDataModel dm = dataModel as ScalableFontDataModel;
                        ret.Item = WriteScalableFontSerializeData(dm);
                    }
                    break;
                case DataType.MultiScalableFont:
                    {
                        MultiScalableFontDataModel dm = dataModel as MultiScalableFontDataModel;
                        ret.Item = WriteMultiScalableFontSerializeData(dm);
                    }
                    break;
            }

            return ret;
        }

        /// <summary>
        /// CharCodeRangeSetを生成します。
        /// </summary>
        static IEnumerable<FcpxCharCodeRange> GetCharCodeRangeSet(IEnumerable<CharCodeRange> ccrSet)
        {
            var list = ccrSet.Where(item => item.Checked == true);
            List<FcpxCharCodeRange> ccrList = new List<FcpxCharCodeRange>();
            foreach (var item in list)
            {
                ccrList.Add(WriteCharCodeRangeSerializeData(item));
            }

            return ccrList.ToArray();
        }

        /// <summary>
        /// ScalableFontDescriptionSetを生成します。
        /// </summary>
        static IEnumerable<FcpxScalableFontDescription> GetScalableFontDescriptionSet(IEnumerable<DataModelBase> sfdSet)
        {
            List<FcpxScalableFontDescription> sfdList = new List<FcpxScalableFontDescription>();
            foreach (var item in sfdSet)
            {
                sfdList.Add(WriteScalableFontDescriptionSerializeData(item));
            }

            return sfdList.ToArray();
        }

        #endregion
    }
}
