﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Data;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows.Forms;

namespace FcpxMaker.Panel.Property
{
    using FcpxMaker.DataModel;
    using Node;
    using static Type;

    public partial class ScalableFontDescriptionPropertyPanel : PropertyPanelBase
        , IEventListener
    {
        private int _boldWeightDefaultDecimalPlace;
        private int _fontScaleWidthDefaultDecimalPlace;
        private int _fontScaleHeightDefaultDecimalPlace;

        /// <summary>
        /// 操作対象のデータモデルです。
        /// </summary>
        private ScalableFontDescriptionDataModel TargetDataModel
        {
            get
            {
                return Target as ScalableFontDescriptionDataModel;
            }
        }

        /// <summary>
        /// UIを更新します。
        /// </summary>
        override public void UpdateProperty()
        {
            if (TargetDataModel != null)
            {
                _nudBoldWeight.Value = new decimal(TargetDataModel.BoldWeight);
                _nudBorderWidth.Value = TargetDataModel.BorderWidth;
                _nudTtcFontIndex.Value = TargetDataModel.TtcFontIndex;
                _nudScaleWidth.Value = new decimal(TargetDataModel.ScaleWidth);
                _nudScaleHeight.Value = new decimal(TargetDataModel.ScaleHeight);
                _nudBaseLineOffset.Value = TargetDataModel.BaseLineOffset;
                _chkIgnorePalt.Checked = TargetDataModel.IgnorePalt;
                _chkDeleteBearingX.Checked = TargetDataModel.DeleteBearingX;
                _nudBearingOffsetX.Value = TargetDataModel.BearingOffsetX;
                _chkForceMonospacedEnabled.Checked = TargetDataModel.ForceMonospacedEnabled;
                _nudForceMonospacedWidth.Value = TargetDataModel.ForceMonospacedWidth;
            }
        }

        /// <summary>
        /// データモデルを関連付けます。
        /// </summary>
        override public void BindTarget(DataModelBase target)
        {
            this.Target = target;
            _pnlFontRangePropertyPanel.BindTarget(TargetDataModel.CharCodeRangeDataModel);
        }

        /// <summary>
        ///コンストラクタです。
        /// </summary>
        public ScalableFontDescriptionPropertyPanel()
        {
            InitializeComponent();

            _boldWeightDefaultDecimalPlace = _nudBoldWeight.DecimalPlaces;
            _fontScaleWidthDefaultDecimalPlace = _nudScaleWidth.DecimalPlaces;
            _fontScaleHeightDefaultDecimalPlace = _nudScaleHeight.DecimalPlaces;
        }

        /// <summary>
        /// 小数点位置を算出します。
        /// </summary>
        public int CalcDecimalPlace(int initialPlace, decimal val)
        {
            int decimalPlaces = initialPlace;
            while ((decimal.Parse(val.ToString("F" + decimalPlaces)) != val) && decimalPlaces < 7)
            {
                decimalPlaces++;
            }

            return decimalPlaces;
        }

        /// <summary>
        /// イベントを通知します。
        /// </summary>
        public void dispatch(object sender, EventArgs arg)
        {
            EventParam param = arg as EventParam;

            switch (param.Type)
            {
                case EventType.PasteParams:
                    {
                        UpdateProperty();
                    }
                    break;
            }
        }

        /// <summary>
        /// 文字の太さが変更された時のイベントハンドラです。
        /// </summary>
        private void Event_BoldWeightValueChanged(object sender, EventArgs e)
        {
            TargetDataModel.BoldWeight = (float)_nudBoldWeight.Value;

            // 小数点位置を調整
            int decimalPlace = CalcDecimalPlace(_boldWeightDefaultDecimalPlace, new decimal(TargetDataModel.BoldWeight));
            _nudBoldWeight.DecimalPlaces = decimalPlace;

            // データの変更を通知します
            NotifyDataModified();
        }

        /// <summary>
        /// 縁取りの太さが変更された時のイベントハンドラです。
        /// </summary>
        private void Event_BorderWidthValueChanged(object sender, EventArgs e)
        {
            TargetDataModel.BorderWidth = Decimal.ToInt32(_nudBorderWidth.Value);

            // データの変更を通知します
            NotifyDataModified();
        }

        /// <summary>
        /// TTCフォントのインデックスが変更された時のイベントハンドラです。
        /// </summary>
        private void Event_TtcFontIndexValueChanged(object sender, EventArgs e)
        {
            TargetDataModel.TtcFontIndex = Decimal.ToInt32(_nudTtcFontIndex.Value);

            // データの変更を通知します
            NotifyDataModified();
        }

        /// <summary>
        /// 文字のスケール(幅)が変更された時のイベントハンドラです。
        /// </summary>
        private void Event_ScaleWidthValueChanged(object sender, EventArgs e)
        {
            TargetDataModel.ScaleWidth = (float)_nudScaleWidth.Value;

            // 小数点位置を調整
            int decimalPlace = CalcDecimalPlace(_fontScaleWidthDefaultDecimalPlace, new decimal(TargetDataModel.ScaleWidth));
            _nudScaleWidth.DecimalPlaces = decimalPlace;

            // データの変更を通知します
            NotifyDataModified();
        }

        /// <summary>
        /// 文字のスケール(高さ)が変更された時のイベントハンドラです。
        /// </summary>
        private void Event_ScaleHeightValueChanged(object sender, EventArgs e)
        {
            TargetDataModel.ScaleHeight = (float)_nudScaleHeight.Value;

            // 小数点位置を調整
            int decimalPlace = CalcDecimalPlace(_fontScaleHeightDefaultDecimalPlace, new decimal(TargetDataModel.ScaleHeight));
            _nudScaleHeight.DecimalPlaces = decimalPlace;

            // データの変更を通知します
            NotifyDataModified();
        }

        /// <summary>
        /// ベースラインのオフセットが変更された時のイベントハンドラです。
        /// </summary>
        private void Event_BaseLineOffsetValueChanged(object sender, EventArgs e)
        {
            TargetDataModel.BaseLineOffset = Decimal.ToInt32(_nudBaseLineOffset.Value);

            // データの変更を通知します
            NotifyDataModified();
        }

        /// <summary>
        /// IgnorePaltが変更された時のイベントハンドラです。
        /// </summary>
        private void Event_IgnorePaltCheckedChanged(object sender, EventArgs e)
        {
            TargetDataModel.IgnorePalt = _chkIgnorePalt.Checked;

            // データの変更を通知します
            NotifyDataModified();
        }

        /// <summary>
        /// DeleteBearingX が変更された時のイベントハンドラです。
        /// </summary>
        private void Event_ChkDeleteBearingXCheckedChanged(object sender, EventArgs e)
        {
            TargetDataModel.DeleteBearingX = _chkDeleteBearingX.Checked;

            // データの変更を通知します
            NotifyDataModified();
        }

        /// <summary>
        /// BearingOffsetX が変更された時のイベントハンドラです。
        /// </summary>
        private void Event_NudBearingOffsetXValueChanged(object sender, EventArgs e)
        {
            TargetDataModel.BearingOffsetX = Decimal.ToInt32(_nudBearingOffsetX.Value);

            // データの変更を通知します
            NotifyDataModified();
        }

        /// <summary>
        /// ForceMonospacedEnabled が変更された時のイベントハンドラです。
        /// </summary>
        private void Event_ChkForceMonospacedEnabledCheckedChanged(object sender, EventArgs e)
        {
            TargetDataModel.ForceMonospacedEnabled = _chkForceMonospacedEnabled.Checked;

            // データの変更を通知します
            NotifyDataModified();
        }

        /// <summary>
        /// ForceMonospacedWidth が変更された時のイベントハンドラです。
        /// </summary>
        private void Event_NudForceMonospacedWidthValueChanged(object sender, EventArgs e)
        {
            TargetDataModel.ForceMonospacedWidth = Decimal.ToInt32(_nudForceMonospacedWidth.Value);

            // データの変更を通知します
            NotifyDataModified();
        }
    }
}
