﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Data;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows.Forms;

namespace FcpxMaker.Panel.Property
{
    using FcpxMaker.DataModel;
    using Node;
    using static Type;

    public partial class MultiScalableFontPropertyPanel : PropertyPanelBase
        , IEventListener
    {
        private bool _isSystemCall = false;
        private int _keepIndex = -1;

        /// <summary>
        /// 操作対象のデータモデルです。
        /// </summary>
        private MultiScalableFontDataModel TargetDataModel
        {
            get
            {
                return Target as MultiScalableFontDataModel;
            }
        }

        /// <summary>
        /// データモデルを関連付けます。
        /// </summary>
        override public void BindTarget(DataModelBase target)
        {
            this.Target = target;
        }

        /// <summary>
        /// UIを更新します。
        /// </summary>
        override public void UpdateProperty()
        {
            if (TargetDataModel != null)
            {
                _nudFontSize.Value = new decimal(TargetDataModel.FontSize);
                _tbxAlternateChar.Text = Convert.ToChar(TargetDataModel.AlternateChar).ToString();
                _nudLineFeedOffset.Value = TargetDataModel.LineFeedOffset;

                // リストビューを再構築
                RebuildListView();

                // ボタンを更新
                UpdateButton();
            }
        }

        /// <summary>
        /// ボタンを更新します。
        /// </summary>
        private void UpdateButton()
        {
            bool bUp = false;
            bool bDown = false;

            ListViewItem item = _lstScalableFontDescription.SelectedItems.Count > 0 ?
                _lstScalableFontDescription.SelectedItems[0] : null;
            if (item != null)
            {
                bUp = item.Index > 0;
                bDown = (item.Index + 1) < _lstScalableFontDescription.Items.Count;
            }

            _btnUp.Enabled = bUp;
            _btnDown.Enabled = bDown;
        }

        /// <summary>
        /// リストビューを再構築します。
        /// </summary>
        private void RebuildListView()
        {
            _lstScalableFontDescription.Items.Clear();
            foreach (ScalableFontDescriptionDataModel dataModel in TargetDataModel.ScalableFontDescriptionSet)
            {
                FontDataModel fontDataModel = (dataModel.FontDataModel as FontDataModel);
                string type = (fontDataModel != null) ? fontDataModel.ToString() : "None";
                string path = (fontDataModel != null) ? fontDataModel.RelativeFontFilePath : string.Empty;

                ListViewItem item = new ListViewItem(type);
                item.SubItems.Add(path);
                item.Tag = dataModel;
                _lstScalableFontDescription.Items.Add(item);
            }
        }

        /// <summary>
        /// ListViewItemの選択状態を変更します。
        /// このメソッドを利用するとSelectedChangedイベントをスルーします。
        /// </summary>
        private void ListViewItemSelectedChange(ListViewItem item, bool val)
        {
            _isSystemCall = true;
            item.Selected = val;
            _isSystemCall = false;
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public MultiScalableFontPropertyPanel()
        {
            InitializeComponent();
        }

        /// <summary>
        /// イベントを通知します。
        /// </summary>
        public void dispatch(object sender, EventArgs arg)
        {
            EventParam param = arg as EventParam;

            switch (param.Type)
            {
                case EventType.AddFont:
                case EventType.Replace:
                    RebuildListView();
                    UpdateButton();
                    break;
                case EventType.AfterReplace:
                    {
                        _lstScalableFontDescription.Focus();
                        ListViewItemSelectedChange(_lstScalableFontDescription.Items[_keepIndex], true);
                        UpdateButton();
                    }
                    break;
                case EventType.PasteParams:
                    UpdateProperty();
                    break;
            }
        }

        /// <summary>
        /// フォントサイズが変更された時のイベントハンドラです。
        /// </summary>
        private void Event_FontSizeValueChanged(object sender, EventArgs e)
        {
            TargetDataModel.FontSize = (float)_nudFontSize.Value;

            // データの変更を通知します
            NotifyDataModified();
        }

        /// <summary>
        /// 代替文字が変更された時のイベントハンドラです。
        /// </summary>
        private void Event_AlternateCharTextChanged(object sender, EventArgs e)
        {
            TargetDataModel.AlternateChar = _tbxAlternateChar.Text.FirstOrDefault();

            // ツリービューの更新要求を通知します
            NotifyUpdateTreeView();

            // データの変更を通知します
            NotifyDataModified();
        }

        /// <summary>
        /// 行間オフセットが変更された時のイベントハンドラです。
        /// </summary>
        private void Event_LineFeedOffsetValueChanged(object sender, EventArgs e)
        {
            TargetDataModel.LineFeedOffset = Decimal.ToInt32(_nudLineFeedOffset.Value);

            // データの変更を通知します
            NotifyDataModified();
        }

        /// <summary>
        /// ↑ボタンが押された時のイベントハンドラです。
        /// </summary>
        private void Event_UpClick(object sender, EventArgs e)
        {
            // リストの入れ替え
            ListViewItem item1 = _lstScalableFontDescription.SelectedItems[0];
            int index = item1.Index;
            _keepIndex = index - 1;
            ListViewItem item2 = _lstScalableFontDescription.Items[_keepIndex];

            ListViewItemSelectedChange(item1, false);

            _lstScalableFontDescription.Items.RemoveAt(index);
            _lstScalableFontDescription.Items.RemoveAt(_keepIndex);

            _lstScalableFontDescription.Items.Insert(_keepIndex, item1);
            _lstScalableFontDescription.Items.Insert(index, item2);

            // データモデルの入れ替え
            this.TargetDataModel.ScalableFontDescriptionSet.Clear();
            foreach (ListViewItem item in _lstScalableFontDescription.Items)
            {
                this.TargetDataModel.ScalableFontDescriptionSet.Add(item.Tag as DataModelBase);
            }

            // 入れ替えイベントを発行
            EventParam args = new EventParam(EventType.Replace, null);
            EventManager.entry(sender, args);

            // データの変更を通知します
            NotifyDataModified();
        }

        /// <summary>
        /// ↓ボタンが押された時のイベントハンドラです。
        /// </summary>
        private void Event_DownClick(object sender, EventArgs e)
        {
            // リストの入れ替え
            ListViewItem item1 = _lstScalableFontDescription.SelectedItems[0];
            int index = item1.Index;
            _keepIndex = index + 1;
            ListViewItem item2 = _lstScalableFontDescription.Items[_keepIndex];

            ListViewItemSelectedChange(item1, false);

            _lstScalableFontDescription.Items.RemoveAt(_keepIndex);
            _lstScalableFontDescription.Items.RemoveAt(index);

            _lstScalableFontDescription.Items.Insert(index, item2);
            _lstScalableFontDescription.Items.Insert(_keepIndex, item1);

            // データモデルの入れ替え
            this.TargetDataModel.ScalableFontDescriptionSet.Clear();
            foreach (ListViewItem item in _lstScalableFontDescription.Items)
            {
                this.TargetDataModel.ScalableFontDescriptionSet.Add(item.Tag as DataModelBase);
            }

            // 入れ替えイベントを発行
            EventParam args = new EventParam(EventType.Replace, null);
            EventManager.entry(sender, args);

            // データの変更を通知します
            NotifyDataModified();
        }

        /// <summary>
        /// リスト項目の選択が変更された時のイベントハンドラです。
        /// </summary>
        private void Event_SelectedIndexChanged(object sender, EventArgs e)
        {
            if (_isSystemCall) return;

            if (_lstScalableFontDescription.SelectedItems.Count == 0)
            {
                _keepIndex = -1;
            }

            UpdateButton();
        }
    }
}
