﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Diagnostics;
using System.Linq;
using System.IO;
using System.Reflection;
using System.Text;
using System.Threading.Tasks;
using System.Windows.Forms;
using System.Xml.Serialization;

namespace FcpxMaker
{
    using DataModel;
    using Dialog;
    using Node;
    using Panel.Property;
    using SubForm;
    using static Type;

    public partial class MainForm : Form
        , IEventListener
    {
        static private MainForm _instance = null;
        private UserNodePanel _userNodePanel = null;
        private bool _isSystemCalled = false;

        /// <summary>
        /// MainFormのインスタンスです。
        /// </summary>
        static public MainForm Instance
        {
            get
            {
                if (_instance == null)
                {
                    _instance = new MainForm();
                }

                return _instance;
            }
        }

        /// <summary>
        /// コマンドライン引数です。
        /// </summary>
        public string[] CommandLineArgs
        {
            get;
            set;
        }

        /// <summary>
        /// ファイルを開いているか。
        /// </summary>
        private bool IsFileOpened
        {
            get
            {
                return _userNodePanel.HasNode;
            }
        }

        /// <summary>
        /// ファイルに変更があったか。
        /// </summary>
        private bool IsModified
        {
            get
            {
                return _userNodePanel.IsModified;
            }
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        private MainForm()
        {
            InitializeComponent();

            // 左パネルは固定なので初期化時にセットしておく
            _userNodePanel = new UserNodePanel();
            _userNodePanel.Dock = DockStyle.Fill;
            this._splitContainer.Panel1.Controls.Add(_userNodePanel);

            // メニューの表示を更新
            UpdateMenu();
        }

        /// <summary>
        /// イベントを通知します。
        /// </summary>
        public void dispatch(object sender, EventArgs arg)
        {
            EventParam param = arg as EventParam;

            switch (param.Type)
            {
                case EventType.New:
                    {
                        FileIOManager.ClearCaches();
                        SetPropertyPanel(null); // プロパティパネルをクリア
                        UpdateMenu();
                    }
                    break;
                case EventType.Open:
                    {
                        SetPropertyPanel(null); // プロパティパネルをクリア
                        UpdateMenu();
                    }
                    break;
                case EventType.SelectedNode:
                    {
                        DataNode node = param.Option as DataNode;

                        // 選択されたノードに対応するプロパティパネルを生成する
                        PropertyPanelBase propertyPanel = CreatePropertyPanel(node.DataType);
                        if (propertyPanel != null)
                        {
                            // ノードに紐付くデータモデルを関連付け
                            propertyPanel.BindTarget(node.DataModel);
                        }
                        SetPropertyPanel(propertyPanel);
                    }
                    break;
                case EventType.Save:
                case EventType.SaveAs:
                    {
                        UpdateMenu();
                    }
                    break;
                case EventType.Exit:
                    {
                        this.Close();
                    }
                    break;
                case EventType.DataModified:
                case EventType.PasteParams:
                    {
                        UpdateMenu();
                    }
                    break;
            }
        }

        /// <summary>
        /// データ種類からプロパティパネルを生成します。
        /// </summary>
        private PropertyPanelBase CreatePropertyPanel(DataType type)
        {
            switch (type)
            {
                case DataType.Root: return new RootPropertyPanel();
                case DataType.PairFont: return new PairFontPropertyPanel();
                case DataType.BitmapFont: return new BitmapFontPropertyPanel();
                case DataType.ScalableFont: return new ScalableFontPropertyPanel();
                case DataType.MultiScalableFont: return new MultiScalableFontPropertyPanel();
                case DataType.ScalableFontDescription: return new ScalableFontDescriptionPropertyPanel();
                case DataType.Font: return new FontPropertyPanel();
            }

            return null;
        }

        /// <summary>
        /// プロパティパネルをセットします。
        /// </summary>
        private void SetPropertyPanel(PropertyPanelBase panel)
        {
            this._splitContainer.Panel2.Controls.Clear();

            if (panel == null)
            {
                return;
            }

            this._splitContainer.Panel2.Controls.Add(panel);
        }

        /// <summary>
        /// メニューを更新します。
        /// </summary>
        private void UpdateMenu()
        {
            _tsmiSave.Enabled = this.IsFileOpened;
            _tsmiSaveAs.Enabled = this.IsFileOpened;

            Assembly assembly = Assembly.GetExecutingAssembly();
            FileVersionInfo info = FileVersionInfo.GetVersionInfo(assembly.Location);
            this.Text = info.FileDescription;
            if (!String.IsNullOrEmpty(FileIOManager.FcpxOpenFilePath))
            {
                string modify = string.Empty;
                if (this.IsModified)
                {
                    modify = "*";
                }

                this.Text += " - " + modify + StringUtility.GetFileName(FileIOManager.FcpxOpenFilePath);
            }
        }

        /// <summary>
        /// AppSettingを保存します。
        /// </summary>
        private void SaveAppSetting()
        {
            StreamWriter writer = null;
            try
            {
                XmlSerializer serializer = new XmlSerializer(typeof(AppSetting));
                Encoding enc = new UTF8Encoding(); // UTF-8(BOMあり)
                if (!Directory.Exists(AppSetting.AppSettingDirectory))
                {
                    Directory.CreateDirectory(AppSetting.AppSettingDirectory);
                }
                writer = new StreamWriter(AppSetting.AppSettingFilePath, false, enc);
                serializer.Serialize(writer, AppSetting.Inst);
            }
            catch (Exception)
            {
                // nope
            }
            finally
            {
                if (writer != null)
                {
                    writer.Close();
                }
            }
        }

        /// <summary>
        /// AppSettingを読み込みます。
        /// </summary>
        private void LoadAppSetting()
        {
            // デシリアライズ
            StreamReader reader = null;
            AppSetting setting = null;
            try
            {
                XmlSerializer serializer = new XmlSerializer(typeof(AppSetting));
                Encoding enc = new UTF8Encoding(false); // UTF-8(BOMなし)
                reader = new StreamReader(AppSetting.AppSettingFilePath, enc);
                setting = serializer.Deserialize(reader) as AppSetting;
            }
            catch (Exception)
            {
                setting = new AppSetting();
            }
            finally
            {
                if (reader != null)
                {
                    reader.Close();
                }
            }

            AppSetting.Inst.CopyParam(setting);
        }

        /// <summary>
        /// AppSettingをアプリケーションに反映させます。
        /// </summary>
        private void StoreAppSettingToApplication(AppSetting setting)
        {
            FileIOManager.FcpxOpenFileDirectory = setting.FcpxOpenDialogDefaultDirectory;
            FileIOManager.FcpxOpenFontDirectory = setting.FontOpenDialogDefaultDirectory;
            FileIOManager.FcpxSaveFileDirectory = setting.FcpxSaveDialogDefaultDirectory;
            this.Size = new Size(setting.MainWindowWidth, setting.MainWindowHeight);
            this._splitContainer.SplitterDistance = setting.SplitterDistance;
        }

        /// <summary>
        /// アプリケーションをAppSettingに反映させます。
        /// </summary>
        private void StoreApplicationToAppSetting(AppSetting setting)
        {
            setting.FcpxOpenDialogDefaultDirectory = FileIOManager.FcpxOpenFileDirectory;
            setting.FontOpenDialogDefaultDirectory = FileIOManager.FcpxOpenFontDirectory;
            setting.FcpxSaveDialogDefaultDirectory = FileIOManager.FcpxSaveFileDirectory;
            setting.MainWindowWidth = this.Size.Width;
            setting.MainWindowHeight = this.Size.Height;
            setting.SplitterDistance = this._splitContainer.SplitterDistance;
        }

        /// <summary>
        /// メインフォームが読み込まれた時に発生するイベントハンドラです。
        /// </summary>
        private void MainForm_Load(object sender, EventArgs e)
        {
            // AppSettingを読み込みます
            LoadAppSetting();
            StoreAppSettingToApplication(AppSetting.Inst);

            if (this.CommandLineArgs != null && this.CommandLineArgs.Count() > 0)
            {
                // コマンドライン引数がある場合
                FileIOManager.FcpxOpenFilePath = this.CommandLineArgs[0];
                EventParam args = new EventParam(EventType.Open, FileIOManager.FcpxOpenFilePath);
                EventManager.entry(this, args);
            }
            else
            {
                // 起動直後に新規作成状態にする
                EventParam args = new EventParam(EventType.New, null);
                EventManager.entry(this, args);
            }
        }

        /// <summary>
        /// メインフォームが閉じられた時に発生するイベントハンドラです。
        /// </summary>
        private void MainForm_FormClosed(object sender, FormClosedEventArgs e)
        {
            // AppSettingを保存します
            StoreApplicationToAppSetting(AppSetting.Inst);
            SaveAppSetting();
        }

        /// <summary>
        /// 新規作成がクリックされた時に発生するイベントハンドラです。
        /// </summary>
        private void Event_NewClick(object sender, EventArgs e)
        {
            Action action = () =>
            {
                EventParam args = new EventParam(EventType.New, null);
                EventManager.entry(sender, args);
            };

            if (this.IsModified)
            {
                // ダイアログを表示
                WarningDialog dialog = new WarningDialog();
                dialog.SaveAsPostCallback = action;
                dialog.Appear(WarningDialogType.FileNew);

                if (dialog.GetResult() == DialogResult.OK)
                {
                    action();
                }
            }
            else
            {
                action();
            }
        }

        /// <summary>
        /// 開くがクリックされた時に発生するイベントハンドラです。
        /// </summary>
        private void Event_OpenClick(object sender, EventArgs e)
        {
            Action action = () =>
            {
                // ファイルを開く
                if (FileIOManager.ShowFcpxOpenDialog() == DialogResult.OK)
                {
                    EventParam args = new EventParam(EventType.Open, FileIOManager.FcpxOpenFilePath);
                    EventManager.entry(sender, args);
                }
            };

            if (this.IsModified)
            {
                // ダイアログを表示
                WarningDialog dialog = new WarningDialog();
                dialog.SaveAsPostCallback = action;
                dialog.Appear(WarningDialogType.FileNew);

                if (dialog.GetResult() == DialogResult.OK)
                {
                    action();
                }
            }
            else
            {
                action();
            }
        }

        /// <summary>
        /// 保存がクリックされた時に発生するイベントハンドラです。
        /// </summary>
        private void Event_SaveClick(object sender, EventArgs e)
        {
            if (!File.Exists(FileIOManager.FcpxOpenFilePath) || String.IsNullOrEmpty(FileIOManager.FcpxOpenFilePath))
            {
                Event_SaveAsClick(sender, e);
            }
            else
            {
                EventParam args = new EventParam(EventType.Save, FileIOManager.FcpxOpenFilePath);
                EventManager.entry(sender, args);
            }
        }

        /// <summary>
        /// 名前を付けて保存がクリックされた時に発生するイベントハンドラです。
        /// </summary>
        private void Event_SaveAsClick(object sender, EventArgs e)
        {
            // ファイルを選択する
            if (FileIOManager.ShowFcpxSaveDialog() == DialogResult.OK)
            {
                EventParam args = new EventParam(EventType.SaveAs, FileIOManager.FcpxSaveFilePath);
                EventManager.entry(sender, args);
            }
        }

        /// <summary>
        /// 終了がクリックされた時に発生するイベントハンドラです。
        /// </summary>
        private void Event_ExitClick(object sender, EventArgs e)
        {
            Action action = () =>
            {
                _isSystemCalled = true;
                EventParam args = new EventParam(EventType.Exit, null);
                EventManager.entry(sender, args);
            };

            if (this.IsModified)
            {
                // ダイアログを表示
                WarningDialog dialog = new WarningDialog();
                dialog.SaveAsPostCallback = action;
                dialog.Appear(WarningDialogType.AppClose);

                if (dialog.GetResult() == DialogResult.OK)
                {
                    action();
                }
            }
            else
            {
                action();
            }
        }

        /// <summary>
        /// ドキュメントがクリックされた時に発生するイベントハンドラです。
        /// </summary>
        private void Event_DocumentClick(object sender, EventArgs e)
        {
            string executingFilePath = Assembly.GetExecutingAssembly().Location;
            string executingFolder = Path.GetDirectoryName(executingFilePath);

            // ページIDを含めたページパスを取得
            string pagePath = string.IsNullOrEmpty(executingFolder) ?
                string.Empty : Path.Combine(executingFolder, "../../../Documents/Package/contents/Pages/Page_293025225.html");

            if (File.Exists(pagePath))
            {
                var fullPath = Path.GetFullPath(pagePath);
                Process.Start(string.Format("file:///{0}", fullPath.Replace("\\", "/")));
            }
            else
            {
                WarningDialog dialog = new WarningDialog();
                dialog.Appear(WarningDialogType.DocumentNotFound);
            }
        }

        /// <summary>
        /// バージョンがクリックされた時に発生するイベントハンドラです。
        /// </summary>
        private void Event_VersionClick(object sender, EventArgs e)
        {
            VersionDialog dialog = new VersionDialog();
            dialog.StartPosition = FormStartPosition.CenterParent;
            dialog.ShowDialog(this);
        }

        /// <summary>
        /// プレビューがクリックされた時に発生するイベントハンドラです。
        /// </summary>
        private void Event_PreviewClick(object sender, EventArgs e)
        {
            PreviewForm previewForm = new PreviewForm();
            previewForm.Show(this);
        }

        /// <summary>
        /// 右上の閉じるボタンが押された時のイベントハンドラです。
        /// </summary>
        private void MainForm_FormClosing(object sender, FormClosingEventArgs e)
        {
            if (_isSystemCalled) return;

            Action action = () =>
            {
                _isSystemCalled = true;
                EventParam args = new EventParam(EventType.Exit, null);
                EventManager.entry(sender, args);
            };

            if (this.IsModified)
            {
                e.Cancel = true;

                // ダイアログを表示
                WarningDialog dialog = new WarningDialog();
                dialog.SaveAsPostCallback = action;
                dialog.Appear(WarningDialogType.AppClose);

                if (dialog.GetResult() == DialogResult.OK)
                {
                    action();
                }
            }
        }

        /// <summary>
        /// ファイルがドラッグされた時のイベントハンドラです。
        /// </summary>
        private void MainForm_DragEnter(object sender, DragEventArgs e)
        {
            e.Effect = DragDropEffects.Copy;
        }

        /// <summary>
        /// ファイルがドロップされた時のイベントハンドラです。
        /// </summary>
        private void MainForm_DragDrop(object sender, DragEventArgs e)
        {
            // ファイルが渡されていなければ何もしない
            if (!e.Data.GetDataPresent(DataFormats.FileDrop))
            {
                return;
            }

            // 渡されたファイルに対して処理を行う
            foreach (var filePath in (string[])e.Data.GetData(DataFormats.FileDrop))
            {
                FileIOManager.FcpxOpenFilePath = filePath;

                // 先頭のファイルのみ処理する
                break;
            }

            // 拡張子チェック
            if (!FileIOManager.IsValidFile(FileIOManager.FcpxOpenFilePath, FontFileType.Fcpx))
            {
                WarningDialog dialog = new WarningDialog();
                dialog.Appear(WarningDialogType.FileTypeError);
                return;
            }

            Action action = () =>
            {
                // ファイルを開く
                EventParam args = new EventParam(EventType.Open, FileIOManager.FcpxOpenFilePath);
                EventManager.entry(this, args);
            };

            if (this.IsModified)
            {
                // ダイアログを表示
                WarningDialog dialog = new WarningDialog();
                dialog.SaveAsPostCallback = action;
                dialog.Appear(WarningDialogType.FileNew);

                if (dialog.GetResult() == DialogResult.OK)
                {
                    action();
                }
            }
            else
            {
                action();
            }
        }
    }
}
