﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows.Forms;

namespace FcpxMaker
{
    using Dialog;
    using static FcpxMaker.Type;

    public class FileIOManager
    {
        static private string _defaultFcpxFileName = "NewFcpxFile";

        static private OpenFileDialog _fcpxOpenDialog = new OpenFileDialog();
        static private OpenFileDialog _fontOpenDialog = new OpenFileDialog();
        static private SaveFileDialog _fcpxSaveDialog = new SaveFileDialog();

        /// <summary>
        /// ファイル選択ダイアログの拡張子指定テーブルです。
        /// </summary>
        readonly private static Dictionary<FontFileType, string> OpenDialogExtensionStrings = new Dictionary<FontFileType, string>()
        {
            { FontFileType.All, Properties.Resources.DIALOG_EXTENSION_ALL},
            { FontFileType.Bitmap, Properties.Resources.DIALOG_EXTENSION_BITMAP },
            { FontFileType.Scalable, Properties.Resources.DIALOG_EXTENSION_SCALABLE },
            { FontFileType.Fcpx, Properties.Resources.DIALOG_EXTENSION_FCPX },
        };

        /// <summary>
        /// 開いたFcpxのファイルパスです。
        /// </summary>
        static public string FcpxOpenFilePath
        {
            get
            {
                return _fcpxOpenDialog.FileName;
            }
            set
            {
                _fcpxOpenDialog.FileName = value;
            }
        }

        /// <summary>
        /// 開いたFontのファイルパスです。
        /// </summary>
        static public string FontOpenFilePath
        {
            get
            {
                return _fontOpenDialog.FileName;
            }
        }

        /// <summary>
        /// 保存するFcpxのファイルパスです。
        /// </summary>
        static public string FcpxSaveFilePath
        {
            get
            {
                return _fcpxSaveDialog.FileName;
            }
        }

        /// <summary>
        /// 開いたFcpxのディレクトリです。
        /// </summary>
        static public string FcpxOpenFileDirectory { get; set; }

        /// <summary>
        /// 開いたFontのディレクトリです。
        /// </summary>
        static public string FcpxOpenFontDirectory { get; set; }

        /// <summary>
        /// 保存したFcpxのディレクトリです。
        /// </summary>
        static public string FcpxSaveFileDirectory { get; set; }

        /// <summary>
        /// Fcpxファイルを開きます。
        /// </summary>
        static public DialogResult ShowFcpxOpenDialog()
        {
            _fcpxOpenDialog.InitialDirectory = FcpxOpenFileDirectory;
            _fcpxOpenDialog.FileName = string.Empty;
            _fcpxOpenDialog.Filter = OpenDialogExtensionStrings[FontFileType.Fcpx];
            DialogResult ret = _fcpxOpenDialog.ShowDialog();
            FcpxOpenFileDirectory = StringUtility.GetDir(FcpxOpenFilePath);

            if (ret == DialogResult.OK && !IsValidFile(FcpxOpenFilePath, FontFileType.Fcpx))
            {
                WarningDialog dialog = new WarningDialog();
                dialog.Appear(WarningDialogType.FileTypeError);
                _fcpxOpenDialog.FileName = string.Empty;
                ret = DialogResult.Abort;
            }

            // ダイアログが閉じられたイベントを発行します
            EventParam args = new EventParam(EventType.DialogClose, null);
            EventManager.entry(null, args);

            return ret;
        }

        /// <summary>
        /// Fontファイルを開きます。
        /// </summary>
        static public DialogResult ShowFontOpenDialog(FontFileType type)
        {
            _fontOpenDialog.InitialDirectory = FcpxOpenFontDirectory;
            _fontOpenDialog.FileName = string.Empty;
            _fontOpenDialog.Filter = OpenDialogExtensionStrings[type];
            DialogResult ret = _fontOpenDialog.ShowDialog();
            FcpxOpenFontDirectory = StringUtility.GetDir(FontOpenFilePath);

            if (ret == DialogResult.OK && !IsValidFile(FontOpenFilePath, type))
            {
                WarningDialog dialog = new WarningDialog();
                dialog.Appear(WarningDialogType.FileTypeError);
                _fontOpenDialog.FileName = string.Empty;
                ret = DialogResult.Abort;
            }

            // ダイアログが閉じられたイベントを発行します
            EventParam args = new EventParam(EventType.DialogClose, null);
            EventManager.entry(null, args);

            return ret;
        }

        /// <summary>
        /// Fcpxファイルを保存します。
        /// </summary>
        static public DialogResult ShowFcpxSaveDialog()
        {
            _fcpxSaveDialog.FileName = _defaultFcpxFileName;
            _fcpxSaveDialog.InitialDirectory = FcpxSaveFileDirectory;
            _fcpxSaveDialog.Filter = OpenDialogExtensionStrings[FontFileType.Fcpx];
            DialogResult ret = _fcpxSaveDialog.ShowDialog();
            FcpxSaveFileDirectory = StringUtility.GetDir(FcpxSaveFilePath);

            // ダイアログが閉じられたイベントを発行します
            EventParam args = new EventParam(EventType.DialogClose, null);
            EventManager.entry(null, args);

            return ret;
        }

        /// <summary>
        /// キャッシュをクリアします。
        /// </summary>
        static public void ClearCaches()
        {
            _fcpxOpenDialog.FileName = string.Empty;
            _fontOpenDialog.FileName = string.Empty;
            _fcpxSaveDialog.FileName = string.Empty;
        }

        /// <summary>
        /// ファイルが有効かチェックします。
        /// </summary>
        static public bool IsValidFile(string fileName , FontFileType type)
        {
            string ext = System.IO.Path.GetExtension(fileName);
            if (type == FontFileType.Scalable)
            {
                return (ext == ".ttf" || ext == ".ttc" || ext == ".otf" || ext == ".bfttf" || ext == ".bfotf");
            }
            else if (type == FontFileType.Bitmap)
            {
                return (ext == ".ffnt" || ext == ".bffnt");
            }
            else if (type == FontFileType.Fcpx)
            {
                return (ext == ".fcpx");
            }

            return true;
        }
    }
}
