﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace FcpxMaker.DataModel
{
    using static FcpxMaker.Type;

    /// <summary>
    /// ペアフォントのデータモデルです。
    /// </summary>
    [Serializable]
    class PairFontDataModel : DataModelBase
    {
        private DataModelBase _firstFont = null;
        private DataModelBase _secondFont = null;

        /// <summary>
        /// フォント1です。
        /// </summary>
        public DataModelBase FirstFont
        {
            get { return _firstFont; }
            set { _firstFont = value; }
        }

        /// <summary>
        /// フォント2です。
        /// </summary>
        public DataModelBase SecondFont
        {
            get { return _secondFont; }
            set { _secondFont = value; }
        }

        /// <summary>
        /// フォント1の種類です。
        /// </summary>
        public DataType Font1Type
        {
            get
            {
                return (_firstFont != null) ? _firstFont.DataType : DataType.None;
            }
        }

        /// <summary>
        /// フォント2の種類です。
        /// </summary>
        public DataType Font2Type
        {
            get
            {
                return (_secondFont != null) ? _secondFont.DataType : DataType.None;
            }
        }

        /// <summary>
        /// フォント1が未設定でフォント2が設定されている状態か調べます。
        /// </summary>
        public bool IsFirstFontEmptyAndSecondFontUsed
        {
            get
            {
                return (_firstFont == null) && (_secondFont != null);
            }
        }

        /// <summary>
        /// フォントを追加します。
        /// </summary>
        public void Add(DataModelBase dataModel)
        {
            if (_firstFont == null)
            {
                _firstFont = dataModel;
            }
            else if (_secondFont == null)
            {
                _secondFont = dataModel;
            }
        }

        /// <summary>
        /// フォントを削除します。
        /// </summary>
        public void Remove(DataModelBase dataModel)
        {
            if (_firstFont == dataModel)
            {
                _firstFont = null;
            }
            else if (_secondFont == dataModel)
            {
                _secondFont = null;
            }
        }

        /// <summary>
        /// データの種類です。
        /// </summary>
        override public DataType DataType
        {
            get { return DataType.PairFont; }
        }

        /// <summary>
        /// データの検証結果です。
        /// </summary>
        override public bool Validate
        {
            get
            {
                // FirstとSecondの両方にフォントが設定されているか
                return (_firstFont != null) && (_secondFont != null);
            }
        }

        /// <summary>
        /// データモデルを比較します。
        /// </summary>
        override public bool Equals(DataModelBase dataModel)
        {
            PairFontDataModel dm = dataModel as PairFontDataModel;
            if (dm == null) return false;

            if (this.FirstFont == null && dm.FirstFont != null)
            {
                return false;
            }

            if (this.SecondFont == null && dm.SecondFont != null)
            {
                return false;
            }

            if ((this.FirstFont != null && !this.FirstFont.Equals(dm.FirstFont)) ||
                (this.SecondFont != null && !this.SecondFont.Equals(dm.SecondFont)))
            {
                return false;
            }

            return true;
        }

        /// <summary>
        /// データモデルを複製します。
        /// </summary>
        override public DataModelBase Clone()
        {
            PairFontDataModel newDataModel = new PairFontDataModel();
            newDataModel.FirstFont = this.FirstFont?.Clone();
            newDataModel.SecondFont = this.SecondFont?.Clone();

            return newDataModel;
        }

        /// <summary>
        /// データモデルをコピーします。
        /// </summary>
        override public bool CopyParam(DataModelBase dataModel, bool copyChild)
        {
            PairFontDataModel dm = dataModel as PairFontDataModel;
            if (dm == null) return false;

            if (copyChild)
            {
                this.FirstFont = dm.FirstFont;
                this.SecondFont = dm.SecondFont;
            }

            return true;
        }

        /// <summary>
        /// 子ノードとして追加可能かどうかを返します。
        /// </summary>
        override public bool CanAddChild(DataModelBase dataModel)
        {
            if (dataModel is PairFontDataModel ||
                dataModel is BitmapFontDataModel ||
                dataModel is ScalableFontDataModel ||
                dataModel is MultiScalableFontDataModel)
            {
                if (_firstFont == null || _secondFont == null)
                {
                    return true;
                }
            }

            return false;
        }

        /// <summary>
        /// 表示名です。
        /// </summary>
        override public string ToString()
        {
            return DataType.ToString();
        }
    }
}
