﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace BinaryToHeader
{
    class Program
    {
        static void Main(string[] args)
        {
            /*
             * args[0] 出力ヘッダパス
             * args[1] 入力バイナリパス
             * args[2] 変数名
             * args[3] アライメント(省略可)
             */

            if (args.Length != 3 && args.Length != 4)
            {
                Console.WriteLine("引数の数が不正です。");
                Console.WriteLine();
                Console.WriteLine("args[0] 出力ヘッダパス");
                Console.WriteLine("args[1] 入力バイナリパス");
                Console.WriteLine("args[2] 変数名");
                Console.WriteLine("args[3] アライメント(省略可)");
                return;
            }

            byte[] binary = ReadBinary(args[1]);
            if (binary == null)
            {
                Console.WriteLine("ファイルが読み込めませんでした。");
                return;
            }

            int alignment = 0;
            if (args.Length == 4)
            {
                if (!int.TryParse(args[3], out alignment))
                {
                    Console.WriteLine("アライメントの値が不正です。");
                    return;
                }
            }

            if (!WriteHeader(args[0], binary, args[2], alignment))
            {
                Console.WriteLine("ファイルの書き込みに失敗しました。");
                return;
            }
        }

        // バイナリの読み込み
        static byte[] ReadBinary(string inputPath)
        {
            byte[] binary;
            try
            {
                binary = System.IO.File.ReadAllBytes(inputPath);
            }
            catch
            {
                return null;
            }
            return binary;
        }

        // ヘッダの書き込み
        static bool WriteHeader(string outputPath, byte[] binary, string variableName, int alignment)
        {
            System.IO.StreamWriter fileHandle;
            try
            {
                fileHandle = new System.IO.StreamWriter(outputPath, false, new System.Text.UTF8Encoding(true)); // UTF-8 BOM あり
            }
            catch
            {
                return false;
            }

            WriteCopyright(fileHandle);
            WriteData(fileHandle, binary, variableName, alignment);

            fileHandle.Close();
            return true;
        }

        // コピーライトの書き込み
        static void WriteCopyright(System.IO.StreamWriter fileHandle)
        {
            fileHandle.WriteLine("/*--------------------------------------------------------------------------------*");
            fileHandle.WriteLine("  Copyright (C)Nintendo All rights reserved.");
            fileHandle.WriteLine();
            fileHandle.WriteLine("  These coded instructions, statements, and computer programs contain proprietary");
            fileHandle.WriteLine("  information of Nintendo and/or its licensed developers and are protected by");
            fileHandle.WriteLine("  national and international copyright laws. They may not be disclosed to third");
            fileHandle.WriteLine("  parties or copied or duplicated in any form, in whole or in part, without the");
            fileHandle.WriteLine("  prior written consent of Nintendo.");
            fileHandle.WriteLine();
            fileHandle.WriteLine("  The content herein is highly confidential and should be handled accordingly.");
            fileHandle.WriteLine(" *--------------------------------------------------------------------------------*/");
            fileHandle.WriteLine();
        }

        // データの書き込み
        static void WriteData(System.IO.StreamWriter fileHandle, byte[] binary, string variableName, int alignment)
        {
            fileHandle.WriteLine("#pragma once");
            fileHandle.WriteLine();
            if (alignment != 0)
            {
                fileHandle.Write("NN_ALIGNAS(" + alignment.ToString() + ") ");
            }
            fileHandle.WriteLine("static uint8_t " + variableName + "[" + binary.Length.ToString() + "] =");
            fileHandle.WriteLine("{");

            for (int i = 0; i < binary.Length; i++)
            {
                int mod = i % 16;

                if (mod == 0)
                {
                    fileHandle.Write("    ");
                }
                else
                {
                    fileHandle.Write(" ");
                }
                fileHandle.Write("0x" + binary[i].ToString("x2"));
                if (i != binary.Length - 1)
                {
                    fileHandle.Write(",");
                }
                if (mod == 15)
                {
                    if (i / 16 % 16 == 0)
                    {
                        // 0x0100 ごとにコメントを挿入する
                        fileHandle.Write("    // 0x" + (i / 16 * 16).ToString("x4"));
                    }
                    fileHandle.WriteLine();
                }

                if (i == binary.Length - 1 && mod != 15)
                {
                    // 中途半端なところでデータが終わった場合に改行しておく
                    fileHandle.WriteLine();
                }
            }

            fileHandle.WriteLine("};");
        }
    }
}
